/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/* ---------------------------------------------------------------------------

	File	:	sissb.c
	Author	:	Dean Kopesky
	Company	:	Minnesota Supercomputer Center. Inc.
	Date	:	August 1991

	sissb reads an plain SIS dataset and inserts simple blocked (4)
	control words.

--------------------------------------------------------------------------- */


#include <stdio.h>

static	char	helptext[] = "\
\n\
sissb reads a plain SIS file (with no control words), adds control words,\n\
and writes to stdout a simple blocked (4) file.\n\
\n\
Usage:\n\
\n\
	sissb [ -H ] [ plain ]\n\
\n\
	-H		Prints this help listing.\n\
	plain		The plain SIS input file (default is stdin).\n\
\n\
";

#define	BFRLEN	200000

#define	GETINT(dest,source,size) \
{ \
	(dest) = 0; \
	memcpy( (char *) &(dest) + sizeof(dest) - (size), (source), (size) ); \
}


int	main( argc, argv )

	int	argc;
	char *	argv[];

{
	FILE *	infile;
	char	buffer[ BFRLEN ];
	int	hlhsize, format, samples;
	int	trcsize;
	int	nread;

	/* Print help if -H specified. */

	if ( argc > 1 && strcmp( argv[1], "-H" ) == 0 )
	{
		fputs( helptext, stderr );
		return 0;
	}

	/* Get input file from command line. */

	if ( argc == 1 )
		infile = stdin;
	else if ( argc == 2 )
		infile = fopen( argv[ 1 ], "r" );
	else
	{
		fprintf( stderr, "usage:  sissb [ infile ]\n" );
		return 1;
	}

	if ( infile == NULL )
	{
		fprintf( stderr, "sissb: cannot open input file (%s).\n",
			argv[ 1 ] );
		perror( "sissb" );
		return 1;
	}

	/* Read fixed part of line header and get info about dataset. */

	if ( fread( buffer, 1004, 1, infile ) != 1 )
	{
		fprintf( stderr, "sissb: line header too short.\n" );
		return 1;
	}

	GETINT( hlhsize, buffer+1002, 2 );
	hlhsize -= 4;

	GETINT( format,  buffer+  64, 2 );
	GETINT( samples, buffer+  60, 4 );

	if ( format == 1 )
		trcsize = 256 + 2 * samples;
	else if ( format == 3 )
		trcsize = 256 + 4 * samples;
	else if ( format == 5 )
		trcsize = 356;
	else
	{
		fprintf( stderr, "sissb: invalid format code (%d).\n", format );
		return 1;
	}

	/* Read historical line header */

	if ( fread( buffer + 1004, hlhsize, 1, infile ) != 1 )
	{
		fprintf( stderr, "sissb: historical line header too short.\n" );
		return 1;
	}

	/* Write out line header. */

	if ( fwritesb( stdout, buffer, 1004+hlhsize ) != 1004+hlhsize )
	{
		fprintf( stderr, "sissb: error writing to output file.\n" );
		perror( "sissb" );
		return 1;
	}

	/* Now, read and write traces until end-of-file. */

	while ( ( nread = fread( buffer, 1, trcsize, infile ) ) == trcsize )
		if ( fwritesb( stdout, buffer, trcsize ) != trcsize )
		{
		fprintf( stderr, "sissb: error writing to output file.\n" );
			perror( "sissb" );
			return 1;
		}

	if ( nread != 0 )
	{
		fprintf( stderr, "sissb: short trace in input file.\n" );
		return 1;
	}

	return 0;
}


/* ---------------------------------------------------------------------------
	fwritesb()
	Writes a simple blocked file to output.
	Returns the number of bytes written, 0 on error.
--------------------------------------------------------------------------- */

int	fwritesb( file, buffer, nbytes )

	FILE *	file;		/* input  - output file */
	char *	buffer;		/* input  - buffer */
	int	nbytes;		/* input  - number of bytes to write */

{
	long	reclen;

	reclen = nbytes;
	if ( fwrite( (char *) &reclen + sizeof(reclen) - 4, 4, 1, file ) != 1 )
		return 0;

	return ( fwrite( buffer, 1, (unsigned) nbytes, file ) );
}
