/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/* ---------------------------------------------------------------------------

	Package	:	sisio
	Module	:	skipt.c
	Author	:	Dean Kopesky
	Company	:	Minnesota Supercomputer Center, Inc.


	SKIPT skips the specified number of records (line header and traces).
	A negative record count is an error, as is attempting to skip past EOF.
	This subroutine does not work on format L.

	FORTRAN:

	CALL SKIPT( unit, records, dummy )

	unit	input	integer	Unit to skip on.
	records	input	integer	Number of records to skip.
	dummy	dummy	any	Dummy parameter for compatibilty with an
				older library.

	C:

	skipt( unit, records );

	unit	input	int	Unit to skip on.
	records	input	int	Number of records to skip.


	$Header: /m/s1/dmk/xlate/sisio/RCS/skipt.c,v 4.3 91/09/11 09:02:11 dmk Exp $

	$Log:	skipt.c,v $
	Revision 4.3  91/09/11  09:02:11  dmk
	MODSET: 4.3
	Changed entry point names to macros.  Changed rtape() call.
	
	Revision 4.1  90/07/13  09:29:27  dmk
	MODSET: 4.1
	Added check for format L.  Changed SUN to sun.
	
	Revision 3.1  89/09/27  09:59:29  dmk
	MODSET: 3.1
	Added C entry point.  No dummy parameter on that one.  Now uses
	rtape() instead of RTAPE().
	
	Revision 2.4  88/09/26  14:39:44  dmk
	MODSET: 2.4
	Fixed a bug that was preventing SKIPT from working when it was the
	first operation on a dataset (it was flushing the buffer too soon).
	
	Revision 2.1  88/07/12  15:13:30  dmk
	MODSET: 2.1
	Added lower case entry point for SUN.  Added SUN block:  rtape_ instead
	of RTAPE.  Eliminated use of SISERROR.
	
	Revision 1.2  88/06/27  09:41:12  dmk
	MODSET: 1.2
	Will now flush the buffer if large buffers are enabled, rather than
	flagging an error.  Will now use _sisfit[ unit ].cwlen rather than
	recalculating cwlen.
	
	Revision 1.1  88/06/24  12:35:31  dmk
	Initial revision
	
--------------------------------------------------------------------------- */


#include	"sisio.h"

#define	MODULE	"SKIPT"


void	F_SKIPT( unit, records, dummy )

	int *	unit;
	int *	records;
	char *	dummy;

{
	C_SKIPT( *unit, *records );
}


void	C_SKIPT( unit, records )

	int	unit;
	int	records;

{
	char	buffer[ SMBUFSIZE ];
	int	ferror;
	int	length;
	int	pos;
	int	toskip;

	/*
		Initialize iff necessary, check unit for range, check for
		outstanding errors, check to see if unit is open.  Check for
		format L.
	*/

	INITIALIZE;
	CHECKRANGE(MODULE,unit);
	CHECKERROR(MODULE,unit);
	CHECKOPEN(MODULE,unit);

	if ( _sisfit[ unit ].format == 'L' )
	{
		_error( MODULE, unit, E_NOTONL );
		return;
	}

	/*
		Check record count:  if zero, return; if less than zero, error.
	*/

	if ( records == 0 ) return;

	if ( records < 0 )
	{
		_error( MODULE, unit, E_NEGREC );
		return;
	}

	toskip = records;

	/*
		Skip the line header iff necessary.  It will be necessary
		to read the line header if we don't know how long it is.
	*/

	if ( _sisfit[ unit ].pointer == 0 )
	{
		if ( _sisfit[ unit ].hdrlen == 0 )
		{
			C_RTAPE( unit, buffer, &length );
			if ( _sisfit[ unit ].error != 0 ) return;
		}
		else
		{
			pos = lseek( _sisfit[ unit ].filedes, ( long )
				( _sisfit[ unit ].hdrlen +
				_sisfit[ unit ].cwlen ), 0 );

			if ( pos < 0 )
			{
				_error( MODULE, unit, E_LSEEK );
				return;
			}

			if ( pos != _sisfit[ unit ].hdrlen +
				_sisfit[ unit ].cwlen )
			{
				_error( MODULE, unit, E_PASTEOF );
				return;
			}

			_sisfit[ unit ].pointer = 1;
		}

		toskip -= 1;
	}

	/*
		Skip traces if necessary.  Read a trace if we don't know
		how long they are.
	*/

	if ( toskip > 0 )
	{
		if ( _sisfit[ unit ].trclen == 0 )
		{
			C_RTAPE( unit, buffer, &length );
			if ( _sisfit[ unit ].error != 0 ) return;
			toskip -= 1;
		}

		pos = lseek( _sisfit[ unit ].filedes, 0L, 1 );
		if ( pos < 0 )
		{
			_error( MODULE, unit, E_LSEEK );
			return;
		}
		pos += toskip * ( _sisfit[ unit ].trclen +
			_sisfit[ unit ].cwlen );

		if ( lseek( _sisfit[ unit ].filedes, ( long ) ( toskip *
			( _sisfit[ unit ].trclen + _sisfit[ unit ].cwlen ) ),
			1 ) != pos )
		{
			_error( MODULE, unit, E_PASTEOF );
			return;
		}

		_sisfit[ unit ].pointer += toskip;
	}

	/*
		Check for large buffering.  If so, flush the buffer.
	*/

	if ( _sisfit[ unit ].largebuf )
	{
		ferror = _flushit( unit );
		if ( ferror < 0 )
		{
			_error( MODULE, unit, -ferror );
			return;
		}
	}
}
