C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C                                                                      *
C  ROUTINE:       FFT2DF                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C      2D DATA IS TRANSFORMED FROM THE T-X DOMAIN TO THE F-K DOMAIN AND*
C      FILTERED, TRANSPOSED, AND PUT IN NATURAL ORDER TO PRODUCE AN NK *
C      BY NW COMPLEX OUTPUT MATRIX.                                    *
C                                                                      *
C  ENTRY POINTS:                                                       *
C      FFT2DF  (MT,NT,NX,NK,NW,IW1,FILTER,DATAIN,DATAOUT)              *
C  ARGUMENTS:                                                          *
C      MT      INTEGER  ??IOU*       -                                 *
C      NT      INTEGER  ??IOU*       -                                 *
C      NX      INTEGER  ??IOU*       -                                 *
C      NK      INTEGER  ??IOU*       -                                 *
C      NW      INTEGER  ??IOU*       -                                 *
C      IW1     INTEGER  ??IOU*       -                                 *
C      FILTER  REAL     ??IOU*  (19) -                                 *
C      DATAIN  REAL     ??IOU*  (*)  -                                 *
C      DATAOU  COMPLEX  ??IOU*  (*)  -                                 *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 92/07/22  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 97/02/18  *
C       FORTRAN 77                                                     *
C                                                                      *
C  HISTORY:                                                            *
C       ORIGINAL                NOV 91          R.D. COLEMAN, CETech   *
C                                                                      *
C  CALLING FORMAT:                                                     *
C       CALL FFT2DF (MT, NT, NX, NK, NW, IW1, FILTER, DATAIN, DATAOUT) *
C                                                                      *
C  PARAMETERS:                                                         *
C       MT      INTEGER INPUT SCALAR                                   *
C               Row dimension of matrix DATAIN (MT >= NT+2).           *
C                                                                      *
C       NT      INTEGER INPUT SCALAR                                   *
C               Number of T's (times).                                 *
C                                                                      *
C       NX      INTEGER INPUT SCALAR                                   *
C               Number of X's.                                         *
C                                                                      *
C       NK      INTEGER INPUT SCALAR                                   *
C               Number of K's (wave numbers) (NK >= NX).               *
C                                                                      *
C       NW      INTEGER INPUT SCALAR                                   *
C               Number of omega's (frequencies) to keep (NW <= NT/2).  *
C                                                                      *
C       IW1     INTEGER INPUT SCALAR                                   *
C               Index of first frequency to keep.                      *
C                                                                      *
C       FILTER  REAL INPUT VECTOR OF LENGTH NW                         *
C               Contains the filter coefficients.                      *
C                                                                      *
C       DATAIN  REAL INPUT/SCRATCH MATRIX OF DIMENSION MT BY NK        *
C               Contains the source data in the NT by NX principle     *
C               submatix.                                              *
C                                                                      *
C       DATAOUT COMPLEX OUTPUT MATRIX OF DIMENSION MAX( NK*NW, NWORK ) *
C               Result matrix (NK by NW complex elements).  Note:      *
C               DATAOUT is also used for work space during processing. *
C               The amount of work space required, NWORK = (MWORK+1)/2,*
C               is system dependent.  For Cray systems, MWORK =        *
C               MAX( 2*NX*NT+2*NT, 4*NW*NK+2*NK ) real elements.  For  *
C               non-Cray systems, MWORK = MAX( 9*NT/2+18*NX+41,        *
C               6*NK+18*NW+41) real elements.                          *
C                                                                      *
C  DESCRIPTION:                                                        *
C       FFT2DF performs the following:                                 *
C          (1) Clears the last NK-NX elements of each row of the input *
C              matrix.                                                 *
C          (2) Performs a real to complex FFT of length NT on the first*
C              NX columns of DATAIN.                                   *
C          (3) Keeps the data for the NW frequencies beginning with the*
C              frequency whose index is IW1 and discards the rest.     *
C          (4) Performs a complex to complex FFT of length NK on the   *
C              NW complex rows.                                        *
C          (5) Scales the results.                                     *
C          (6) Filters the data; i.e., multiplies every element of the *
C              i-th complex row by real scalar FILTER(i).              *
C          (7) Rearranges the data into natural order as opposed to FFT*
C              order.                                                  *
C          (8) Transposes the data to form the NK by NW complex output *
C              matrix.                                                 *
C                                                                      *
C       The FFT's are performed in place, thus, the contents of the    *
C       input matrix are destroyed and two additional real elements are*
C       required at the bottom of each column (for the Nyquist fre-    *
C       quency).  In addition to these two extra rows in the input     *
C       matrix, an additional row should be added to make the row      *
C       dimension odd in order to optimize performance on some systems.*
C       Since NT must be an even number, MT should be set to           *
C       MT = NT + 3 + 2*k where k is a non-negative integer.           *
C                                                                      *
C  SUBPROGRAMS CALLED:                                                 *
C       FFTFAX, CFTFAX, RFFTMLT, CFFTMLT, CVSMUL                       *
C                                                                      *
C  ERROR CONDITIONS:                                                   *
C       None.                                                          *
C                                                                      *
C--------------------------------------------------------------------- *
C                                                                      *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      VCLR   -                                                        *
C      FFTFAX -                                                        *
C      RFFTML -                                                        *
C      CFTFAX -                                                        *
C      CFFTML -                                                        *
C      CVSMUL -                                                        *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      FLOAT   REAL -                                                  *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C*****************  ITEMS DELETED FROM DOCPACK  ************************
C  =============================== DATE: 92/07/22 ==================   *
C NAME: FFT2DF    FFT - 2D, FORWARD                    REV 1.0  NOV 91 *
C********************  END OF DELETED ITEMS  ***************************
C***********************************************************************
      SUBROUTINE FFT2DF (MT, NT, NX, NK, NW, IW1, FILTER, DATAIN,
     &                   DATAOUT)
C
C  SUBROUTINE ARGUMENTS
C
      INTEGER MT, NT, NX, NK, NW, IW1
      REAL    FILTER(19), DATAIN(*)
      COMPLEX DATAOUT(*)
C
C  LOCAL VARIABLES
C
      INTEGER IFAC(19), IWRK, JD1, JD2, JI, JR, JR1, JR2, JW, NK1, NK2
      REAL    RNK, SCALE
C
C-----------------------------------------------------------------------
C
      IF( NK .GT. NX ) CALL VCLR( DATAIN(MT*NX+1), 1, MT*(NK-NX) )
C
C===  TRANSFORM FROM T TO OMEGA
C===  NOTE: DATAOUT IS USED FOR WORK SPACE AND TRIG TABLES
C
      IWRK = 1 + NT
      CALL FFTFAX (NT, IFAC, DATAOUT)
      CALL RFFTMLT (DATAIN, DATAOUT(IWRK), DATAOUT, IFAC, 1,
     &              MT, NT, NX, -1)
C
C===  TRANSFORM FROM X TO K
C===  NOTE: DATAOUT IS USED FOR WORK SPACE AND TRIG TABLES
C
      IWRK = 1 + NK
      JR   = 2 * IW1 - 1
      JI   = JR + 1
      CALL CFTFAX (NK, IFAC, DATAOUT)
      CALL CFFTMLT (DATAIN(JR), DATAIN(JI), DATAOUT(IWRK), DATAOUT,
     &              IFAC, MT, 2, NK, NW, -1)
C
C===  SCALE, FILTER, TRANSPOSE, AND PUT IN NATUAL ORDER
C
      RNK = 1.0 / FLOAT( NK )
      NK1 = NK / 2 + 1
      NK2 = NK - NK1
      JR1 = JR
      JR2 = JR1 + NK1 * MT
      JD2 = 1
      JD1 = JD2 + NK2
      DO 110 JW = 1, NW
         SCALE = RNK * FILTER(JW)
         CALL CVSMUL (DATAIN(JR1), MT, SCALE, DATAOUT(JD1), 2, NK1)
         CALL CVSMUL (DATAIN(JR2), MT, SCALE, DATAOUT(JD2), 2, NK2)
         JR1 = JR1 + 2
         JR2 = JR2 + 2
         JD1 = JD1 + NK
         JD2 = JD2 + NK
  110 CONTINUE
C
      RETURN
      END
