C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C                                                                      *
C  ROUTINE:       GTFLTR                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C       GTFLTR generates a trapezoidal filter for filtering an FFT     *
C       output.  Also generates a vector of frequencies that correspond*
C       to the filter.                                                 *
C                                                                      *
C  ENTRY POINTS:                                                       *
C      GTFLTR  (F1,F2,F3,F4,NT,DT,MAXNW,NW,IW1,OMEGA,FILTR,IERR)       *
C  ARGUMENTS:                                                          *
C      F1      REAL     ??IOU*          -                              *
C      F2      REAL     ??IOU*          -                              *
C      F3      REAL     ??IOU*          -                              *
C      F4      REAL     ??IOU*          -                              *
C      NT      INTEGER  ??IOU*          -                              *
C      DT      REAL     ??IOU*          -                              *
C      MAXNW   INTEGER  ??IOU*          -                              *
C      NW      INTEGER  ??IOU*          -                              *
C      IW1     INTEGER  ??IOU*          -                              *
C      OMEGA   REAL     ??IOU*  (MAXNW) -                              *
C      FILTR   REAL     ??IOU*  (MAXNW) -                              *
C      IERR    INTEGER  ??IOU*          -                              *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 97/02/13  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 97/02/18  *
C       FORTRAN 77                                                     *
C                                                                      *
C  HISTORY:                                                            *
C       ORIGINAL                NOV 91          R.D. COLEMAN, CETech   *
C       Rev 1.1                 APR 93          R.D. COLEMAN, CETech   *
C               Fixed round-off bug by using an epsilon when doing     *
C               compares.                                              *
C                                                                      *
C  CALLING FORMAT:                                                     *
C       CALL GTFLTR( F1, F2, F3, F4, NT, DT, MAXNW,                    *
C      &             NW, IW1, OMEGA, FILTR, IERR )                     *
C                                                                      *
C  PARAMETERS:                                                         *
C       F1      REAL INPUT SCALAR                                      *
C               Low cut off frequency (in Hertz).                      *
C                                                                      *
C       F2      REAL INPUT SCALAR                                      *
C               Low corner frequency (in Hertz).                       *
C                                                                      *
C       F3      REAL INPUT SCALAR                                      *
C               High corner frequency (in Hertz).                      *
C                                                                      *
C       F4      REAL INPUT SCALAR                                      *
C               High cut off frequency (in Hertz).                     *
C                                                                      *
C       NT      INTEGER INPUT SCALAR                                   *
C               Number of time samples used in the corresponding FFT.  *
C                                                                      *
C       DT      REAL INPUT SCALAR                                      *
C               Sample interval (in seconds).                          *
C                                                                      *
C       MAXNW   INTEGER INPUT SCALAR                                   *
C               Maximum number of output frequencies allowed; i.e.,    *
C               the physical dimension of the output vectors.          *
C                                                                      *
C       NW      INTEGER OUTPUT SCALAR                                  *
C               Number of output frequencies (omegas).                 *
C                                                                      *
C       IW1     INTEGER OUTPUT SCALAR                                  *
C               FFT index corresponding to first output frequency.     *
C                                                                      *
C       OMEGA   REAL OUTPUT VECTOR OF LENGTH NW                        *
C               Frequency vector (radians per second).                 *
C                                                                      *
C       FILTR   REAL OUTPUT VECTOR OF LENGTH NW                        *
C               Filter coefficients.                                   *
C                                                                      *
C       IERR    INTEGER OUTPUT SCALAR                                  *
C               Completion code.  IERR = 0 for normal completion       *
C                                      = 1 for invalid input parameters*
C                                      = 2 for empty output vectors    *
C                                      = 3 for NW > MAXNW              *
C                                                                      *
C  DESCRIPTION:                                                        *
C       GTFLTR generates a trapezoidal filter for application to an    *
C       FFT output.  The filter is zero for frequencies less than F1 or*
C       greater than F4, a linear ramp from zero to one for frequencies*
C       between F1 and F2, one between F2 and F3, and a linear ramp fro*
C       one to zero between F3 and F4.  The non-zero filter coefficient*
C       are output in FILTR.  IW1 is the FFT frequency index that      *
C       corresponds to the coefficient in FILTR(1), where 1 is the FFT *
C       frequency index of the DC component and NT/2+1 is the index of *
C       the Nyquist frequency.  NW is the number of filter coefficients*
C       that are output.  The angular frequency that correponds to each*
C       coefficient in FILTR is output in the corresponding element of *
C       OMEGA; i.e., OMEGA(i) = (i+IW1-2)*DW where DW = 2*pi/(NT*DT).  *
C                                                                      *
C       The input parameters must obey the following relationships:    *
C       F4 >= F3 >= F2 >= F1 >= 0.0, NT > 0, and DT > 0.0.             *
C                                                                      *
C  SUBPROGRAMS CALLED:                                                 *
C       None.                                                          *
C                                                                      *
C  ERROR CONDITIONS:                                                   *
C       If an error is detected (see above), NW is set to zero, IERR is*
C       set to the appropriate value, and the routine is aborted.      *
C                                                                      *
C--------------------------------------------------------------------- *
C                                                                      *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      ABS     GENERIC -                                               *
C      NINT    INTEGER -                                               *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C*****************  ITEMS DELETED FROM DOCPACK  ************************
C  =============================== DATE: 97/02/13 ==================   *
C NAME: GTFLTR    GENERATE TRAPEZOIDAL FREQ. FILTER    REV 1.1  APR 93 *
C********************  END OF DELETED ITEMS  ***************************
C***********************************************************************
      SUBROUTINE GTFLTR( F1, F2, F3, F4, NT, DT, MAXNW, NW, IW1,
     &                   OMEGA, FILTR, IERR )
C
      REAL PI, EPS
      PARAMETER( PI = 3.141592653589793, EPS = 1.0E-6 )
C
C  SUBROUTINE ARGUMENTS
C
      INTEGER NT, MAXNW, NW, IW1, IERR
      REAL    F1, F2, F3, F4, DT, OMEGA(MAXNW), FILTR(MAXNW)
C
C  LOCAL VARIABLES
C
      INTEGER IW2, I
      REAL    DW, W1, W2, W3, W4
C
C-----------------------------------------------------------------------
C
C  DO VALIDITY CHECKS
C
      IF( F1.LT.0.0 .OR. F2.LT.F1 .OR. F3.LT.F2 .OR. F4.LT.F3 .OR.
     &    DT.LE.0.0 .OR. NT.LE.0 ) THEN
         NW   = 0
         IW1  = 1
         IERR = 1
         RETURN
      ENDIF
C
C  CONVERT TO ANGULAR FREQUENCIES
C
      W1 = 2.0 * PI * F1
      W2 = 2.0 * PI * F2
      W3 = 2.0 * PI * F3
      W4 = 2.0 * PI * F4
C
C  COMPUTED INDEX OF HIGH AND LOW FREQUENCY
C
      DW  = 2.0 * PI / (NT * DT)
      IW1 = NINT( W1 / DW ) + 1
      IW2 = NINT( W4 / DW ) + 1
      IF( DW*(IW1-1) .LT. W1-EPS ) IW1 = IW1 + 1
      IF( DW*(IW2-1) .GT. W4+EPS ) IW2 = IW2 - 1
C
C  ELIMINATED POSSIBLE ZERO COEFICIENTS ON THE ENDS
C
      IF( W2 .GT. W1 .AND. DW*(IW1-1) .EQ. W1 ) IW1 = IW1 + 1
      IF( W4 .GT. W3 .AND. DW*(IW2-1) .EQ. W4 ) IW2 = IW2 - 1
      IF( W2 .GT. W1 .AND. ABS(W1-DW*(IW1-1)) .LE. EPS ) IW1 = IW1 + 1
      IF( W4 .GT. W3 .AND. ABS(W4-DW*(IW2-1)) .LE. EPS ) IW2 = IW2 - 1
C
C  IW2 MUST INDEX A FREQUENCY THAT IS LESS THAN NYQUIST
C
      IF( IW2 .GT. NT/2 ) IW2 = NT/2
C
C  COMPUTE FILTER LENGTH AND INSURE THAT THE FILTER IS NOT EMPTY
C
      NW  = IW2 - IW1 + 1
C
      IF( NW .LE. 0 ) THEN
         NW   = 0
         IERR = 2
         RETURN
      ENDIF
C
C  MAKE SURE THAT THE FILTER WILL FIT IN THE SPACE ALLOCATED
C
      IF (NW .GT. MAXNW) THEN
         NW   = 0
         IERR = 3
         RETURN
      ENDIF
C
C  COMPUTE OMEGA AND FILTR
C
      DO 110 I = 1, NW
         OMEGA(I) = DW * (IW1 + I - 2)
         IF      (OMEGA(I) .LT. W2-EPS) THEN
            FILTR(I) = (OMEGA(I) - W1) / (W2 - W1)
         ELSE IF (OMEGA(I) .GT. W3+EPS) THEN
            FILTR(I) = (W4 - OMEGA(I)) / (W4 - W3)
         ELSE
            FILTR(I) = 1.0
         ENDIF
  110 CONTINUE
C
      IERR = 0
      RETURN
      END
