c***********************************************************************
c                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c***********************************************************************
c NAME: ZKK2ZXY    2D-FFT W/ RESAMPLE           REV 1.0  MAR 95 ********
c***********************************************************************
c
c  HISTORY:       MAR 95          REV 1.0         J. Cooperstein, CETech
c  Revised: December 12, 1995     REV 1.3              Mary Ann Thornton
C     Changed "stop" to call exitfu(0) so script can capture and test
C     the status upon exit of the program.
c  Revised: April 3, 1996         REV 1.4              Mary Ann Thornton
c     A copy of rdrec.F (from ~mbs/src/lib/libmbs) was moved into this
c     directory. The 30000 in the StaCor trace header word is set to
c     zero. Trace headers that were previously attached to incoming
c     zero traces, were being attached to outgoing non-zero traces and
c     causing these traces to get zeroed out when the data was read in
c     in the next step (cmmvz3d). This version of rdrec
c     (called rdrecn.F) will set the StaCor to 0 on dead input traces.
c
c***********************************************************************
      program zkk2zxy
c***********************************************************************
c     zkk2zxy - reads a DDS data dictionary and its associated usp data 
c     file in (z',kx,ky) and zeta file, performs a 2D fft, extracts the
c     real components, resamples z' to z (regular z grid), and
c     outputs a usp file in (z,x,y).
c     J. Cooperstein, CETech      Version 1.1                May 8, 1995
c     check for errors  in zeta file, print warning, do not abort
c     see zkrzeta.F
c     J. Cooperstein, CETech      Version 1.2               May 30, 1995
c     fixed bug so padding is removed in final output file
c  Revised: Mary Ann Thornton     Version 1.3          December 13, 1995      
C     Changed "stop" to call exitfu(0) so script can capture and test
C     the status upon exit of the program.
c***********************************************************************
      implicit none
c
#include     <save_defs.h>
#include     <f77/iounit.h>
#include     <f77/lhdrsz.h>
#include     <f77/sisdef.h>

c
c parameters:
c
      real      PI              ! PI
      parameter ( PI = 3.141592653589793 )
c
c variables:
c
c     real time0, timefft, timersmp ! timing variables
      real timefft, timersmp ! timing variables
c
      integer   debug           ! debug output flag
      logical   depth           ! true if output file is in depth
      integer   dheadr(SZLNHD)  ! length of input file line header (words)
      real      dkx             ! delta kx
      real      dky             ! delta ky
      real      dtdds           ! delta t from dds input file
      real      dt              ! delta t
      real      dx              ! delta x
      real      dy              ! delta y
      real      dz              ! delta z
      real      dzdds           ! delta z from dds input file
c     integer   ic              ! character index in title construction
      integer   idir            ! direction flag -1(+1), forward(inverse) FFT
      integer   idt             ! Sampling interval (milliseconds)
      integer   idt0            ! Sampling interval from line header
      integer   ierr            ! error flag
      integer   ifac(19)        ! factors from cftfax
c     integer   jc              ! character index in title construction
      integer   jrec            ! index of current input record
      real      kx0             ! first kx
      real      ky0             ! first ky
      integer   luinp           ! logical unit of input dataset
      integer   luout           ! logical unit of output dataset
      integer   mbytes          ! MBytes for scratch (default=32, 64-cray)
      integer   nbytes          ! length of input file line header (bytes)
      integer   nkx             ! number of kx_s
      integer   nky             ! number of ky_s
      integer   nrec            ! number of records in input dataset
      integer   nrec0           ! number of records, orig. data
      integer   nrec2           ! number of records in output dataset
      integer   nsmp            ! number of samples per trace in input
      integer   nsmp0           ! number of samples per trace, orig. data
      integer   nsmp2           ! number of samples per trace in output
      integer   ntoff           ! number of zero samples at start of trace
      integer   ntrc            ! number of traces per record in input
      integer   ntrc0           ! number of traces per record, orig. data
      integer   ntrc2           ! number of traces per record in output
      character ntap*128        ! name of input file
      integer   ntin            ! number of times, input
      integer   ntout           ! number of times, output
      integer   nx              ! number of traces to use
      integer   ny              ! number of records to use
      integer   nzin            ! number of z_s, input
      integer   nzout           ! number of z_s, output
      character otap*128        ! name of output file
      character ppname*7        ! program name
      real      t0              ! first time , output file
      logical   timel           ! true if output file is in time
      real      tmax            ! maximum t (msecs)
      character title*66        ! title for printout banner
      real      tmaxdds         ! maximum t from dds input file
      logical   verbos          ! verbose output flag
      character version*4       ! version number
      real      x0              ! first x
      real      y0              ! first y
      real      z0              ! first z , output file
      real      zmax            ! maximum z
      real      zmaxdds         ! maximum z from dds input file
      character ztap*128        ! name of input zeta file
      character outfil*54       ! output data dictionary
      character datpth*64       ! output data file path
      character datsfx*10       ! output data file suffix
c
c dynamically allocated arrays:
c
      integer*2 headr(1)        ! trace headers, current record
      real      rec(1)          ! scratch space for input record
      real      recout(1)       ! scratch space for resampled input record
      real      timin(1)        ! array of input times
      real      timout(1)       ! array of output times
      real      trace(1)        ! scratch space, trace (including header)
      real      trig(1)         ! table for FFTs
      real      work(1)         ! scratch space for output rec, cfftlmtu
      real      zin(1)          ! array of input z_s
      real      zout(1)         ! array of output z_s
c
c variables for length (in words) of dynamically allocated arrays:
c
      integer   lheadr, lrec, lrecout, ltrace, ltrig, lwork
cdan
c     integer*8 lwork_8
      integer   lwork_8
c
c functions:
c
      integer   argis           ! is argument present function
      integer   ncfft5          ! computes integer for use in cfftmlt
cmat  real      second          ! returns elapsed CPU time (seconds)

      integer   ifmt_TrcNum,l_TrcNum,ln_TrcNum
      integer   ifmt_RecNum,l_RecNum,ln_RecNum
      integer   ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc
      integer   ifmt_RecInd,l_RecInd,ln_RecInd
      integer   ifmt_DphInd,l_DphInd,ln_DphInd
      integer   ifmt_DstSgn,l_DstSgn,ln_DstSgn
      integer   ifmt_DstUsg,l_DstUsg,ln_DstUsg
      integer   ifmt_StaCor,l_StaCor,ln_StaCor
      integer   ifmt_SoPtNm,l_SoPtNm,ln_SoPtNm

c
c pointers:
c
      pointer ( pheadr,  headr  )
      pointer ( prec,    rec    )
      pointer ( precout, recout )
      pointer ( ptimin,  timin  )
      pointer ( ptimout, timout )
      pointer ( ptrace,  trace  )
      pointer ( ptrig,   trig   )
      pointer ( pwork,   work   )
      pointer ( pzin,    zin    )
      pointer ( pzout,   zout   )
c
c data initialization:
c
      data ierr   / 0  /
      data luinp  / -1 /
      data luout  / -1 /
      data ntoff  / 0  /
      data ppname / 'ZKK2ZXY' /
      data title   /                                                   '
     &                        ZKK2ZXY - 2D FFT, RESAMPLE
     &' /
      data verbos  /.false./
      data version /' 1.3' /
      data x0, y0, zmax, tmax   / 0.0, 0.0, 0.0, 0.0 /
c
c***********************************************************************
c
c format statements:
c
  900 format(
     &'***************************************************************'/
     &'  zkk2zxy - reads a DDS data dictionary and its associated'/
     &'  usp data file in (z^prime,kx,ky) and zeta file, performs a 2D'/
     &'  fft, extracts the real components, resamples z^prime to z '/
     &'  (regular z grid), and outputs a usp file in (z,x,y).'//
     &'  NOTE: Either dtms or dz may be specified but not both.'//
     &' Users enter the following parameters, or use default values:'/)
  901 format(
     &' dtms=<dtout>          - output delta t (ms) (see note) '
     &,' (no default)'/
     &' dz=<dzout>            - output delta z      (see note) '
     &,' (no default)'/
     &' zmax=<zmax>           - maximum z                      '
     &,' (no default)'/
     &' tmax=<tmax>           - maximum t (msecs)              '
     &,' (no default)'/)
  903 format(
     &' M=<mbytes>            - number of MBytes for scratch   '
     &', (default=64)'/)
  904 format(
     &' M=<mbytes>            - number of MBytes for scratch   '
     &,' (default=32)'/)
  905 format(
     &' out=<ddout>           - output data dictionary         '
     &', (no default)'/
     &' data_path=<path>      - output data/zeta file path     '
     &', (default=".")'/
     &' data_suffix=<dsuffix> - output data file suffix        '
     &', (default=".usp")'/
     &' flags=<flag_str>      - flags'/
     &' V                     - Verbose printout'/)
  902 format(//
     &' usage: zkk2zxy dtms=<dtout> dz=<dzout> zmax=<zmax> tmax=<tmax>'/
     &'        M=<mbytes>'/
     &'        out=<ddout> data_path=<path> data_suffix=<dsuffix>'/
     &'        flags=<flag_str> (V)'/
     &'***************************************************************')
  910 format( /' ', ' INPUT DATASET    = '       / a128,
     &        /' ', ' OUTPUT DATASET   = '       / a128,
     &        /' ', ' ZETA FILE        = '       / a128 )
  912 format(/10x,' *********** PERFORMING 3D TRANSFORM ************'/ )
  913 format( /' ', 'Input file line header values:'/
     1         ' ', '   Number of records       =', i6/
     2         ' ', '   Number of traces/record =', i6/
     3         ' ', '   Number of samples/trace =', i6)
  914 format( /' ', 'Output file line header values:'/
     1         ' ', '   Number of records       =', i6/
     2         ' ', '   Number of traces/record =', i6/
     3         ' ', '   Number of samples/trace =', i6)
  920 format( /' ', 'Command line arguments (adjusted):'/
     &         '    idt      =', i6/
     &         '    dt(msec) =', f13.6/
     &         '    dx       =', f13.6/
     &         '    dy       =', f13.6/
     &         '    dz       =', f13.6/
     &         '    mbytes   =', i6/
     &         '    zmax     =', f13.6/
     &         '    tmax     =', f13.6)
  922 format(  '    nkx      =', i6/
     &         '    nky      =', i6/
     &         '    nx       =', i6/
     &         '    ny       =', i6/
     &         '    ntin     =', i6/
     &         '    ntout    =', i6/
     &         '    nz       =', i6/
     &         '    nzout    =', i6)
  923 format(/' Dynamically allocated arrays: '/
     &        '    lheadr   =', i10/
     &        '    ltrace   =', i10/
     &        '    lrec     =', i10/
     &        '    lrecout  =', i10/
     &        '    lwork    =', i10/
     &        '    ltrig    =', i10)
9998  format( /' ', '***** NORMAL COMPLETION *****'/ )
9999  format( /' ', '***** ABNORMAL COMPLETION CODE = ', i4, ' *****'/ )
c
c***********************************************************************
c
c===  check for help
c
      if( argis( '-h' ) .gt. 0 .or. argis( '-?' ) .gt. 0 ) then
         write( LER, 900 )
         write( LER, 901 )
#ifdef CRAY
         write( LER, 903 )
#else
         write( LER, 904 )
#endif
         write( LER, 905 )
         write( LER, 902 )
         call exitfu(0)
      endif
c
c===  open printout file
c
      call openpr( LUPPRT, LUPRT, ppname, ierr)
c
      if( ierr .ne. 0 ) then
         write( LER,* ) '***** ERROR: ierr = ',ierr
     &       ,' reported by openpr *****'
         go to 800
      endif
c
c===  create print banner
c
cinclude <f77/mbsdate.h>
c
      call gamoco( title, 1, LUPRT )

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('SoPtNm',ifmt_SoPtNm,l_SoPtNm,ln_SoPtNm,TRACEHEADER)

c
c===  read program parameters from command line
c
      call zkgcmdln( dt, dz, tmax, zmax, mbytes, outfil, datsfx
     &                   , datpth, verbos, debug)
c
c===  construct full output file name, make sure a / is at the end
c
      call mkname( datpth, outfil, datsfx, otap )
c
c===  read dds file
c
      call zkrdds( ntap, ztap, nsmp0, ntrc0, nrec0, nzin, nkx, nky
     &                 , kx0, ky0, dkx, dky
     &                 , zmaxdds, tmaxdds, dzdds, dtdds
     &                 , verbos )
c
c===  does command line override dt, dz, zmax, tmax values?
c
      if( dt .le. 0.0 .and. dz .le. 0 ) then
         dt = dtdds
         dz = dzdds
      endif
c
      if( zmax .le. 0.0 ) zmax = zmaxdds
      if (tmax .le. 0.0 ) tmax = tmaxdds
c
c===  correct for the array running from 0 to nzin
c
      nzin = nzin - 1
      ntin = nzin

      write( LUPRT,  910 ) ntap, otap, ztap
c
      if( ntap .eq. ' ' .or. otap .eq. ' ' .or. ztap .eq. ' ') then
         write( LER, * ) ' ***** ERROR: input, output and zeta files'
     &                 , ' must be specified *****'
         ierr = 1
         go to 800
      endif
c
c===  open usp data files:
c
      call lbopen( luinp, ntap, 'r+' )
      call lbopen( luout, otap, 'w')
c
c===  allocate space for irregular z and t grids
c
      call galloc( pzin,   ISZBYT *(nzin+1), ierr, 'ABORT' )
      call galloc( ptimin, ISZBYT *(ntin+1), ierr, 'ABORT' )
c
c===  read zeta file
c
      call zkrzeta( ztap, nzin, zin, timin, ierr )
      idt = nint( dt )
c
      if ( ierr .ne. 0 ) then
         write( LER, * )' ***** ERROR reading zeta file *****'
         go to 800
      endif

c
c===  time or depth output?
c
      depth = ( dz .gt. 0.0 )
      timel = ( dt .gt. 0.0 )
      if( timel .and. depth ) then
         write( LER, * )' ***** ERROR: cant have depth and time *****'
         ierr = 1
         go to 800
      endif
c
c===  set up resampling grids
c
      if( depth ) then
         if( zmax .gt. 0.0 ) then
            zmax = min (zmax, zin(nzin+1) )
         else
            zmax = zin(nzin+1)
         endif
         nzout = nint( zmax / dz )
         call galloc( pzout, ISZBYT *(nzout+1), ierr, 'ABORT' )
         call vramp( zin(1), dz, zout, 1, nzout + 1 )
         z0 = zout(1)
         ntout = nzout
      endif
c
      if( timel ) then
         if( tmax .gt. 0.0 ) then
            tmax = min (tmax, timin(ntin+1) )
         else
            tmax = timin(ntin+1)
         endif
         ntout = nint( tmax / dt )
         call galloc( ptimout, ISZBYT *(ntout+1), ierr, 'ABORT' )
         call vramp( timin(1), dt, timout, 1, ntout + 1 )
         t0 = timout(1)
         nzout = ntout
      endif
c
c===  read data line header
c
      nbytes = 0
      call rtape( luinp, dheadr, nbytes )
      if( nbytes .eq. 0 ) then
         write( LER,* ) ' ***** ERROR - DATA LINE HEADER READ ERROR ***'
         ierr = 1
         go to 800
      endif
c
c===  get parameters from data line header
c
      call saver( dheadr, 'SmpInt', idt0, LINHED )
      call saver( dheadr, 'NumSmp', nsmp, LINHED )
      call saver( dheadr, 'NumTrc', ntrc, LINHED )
      call saver( dheadr, 'NumRec', nrec, LINHED )
c
c===  set some values
c
      if( idt .le. 0 ) idt   = idt0
      nx = ncfft5( nkx )
      ny = ncfft5( nky )
      nsmp2 = (nzout+1)
      ntrc2 = nx
      nrec2 = ny
c
      if( nrec2 .gt. nrec ) then
         write( LER, * ) '***** ERROR: nrec2 > nrec', nrec2, nrec
         ierr = 1
         go to 800
      endif

      if( ntrc2 .gt. ntrc ) then
         write( LER, * ) '***** ERROR: ntrc2 > ntrc', ntrc2, ntrc
         ierr = 1
         go to 800
      endif
c
c===  update and output line header
c
      call hlhprt( dheadr, nbytes, PPNAME, len(PPNAME), LUPRT )
      call savew( dheadr, 'SmpInt', idt    , LINHED )
      call savew( dheadr, 'NumSmp', nsmp2  , LINHED )
c
c===  restore original trace and record numbers to data
c
      call savew( dheadr, 'NumTrc', ntrc0  , LINHED )
      call savew( dheadr, 'NumRec', nrec0  , LINHED )
c
      call wrtape( luout, dheadr, nbytes )
c
      write( LUPRT, 912 )
      write( LUPRT, 913 ) nrec , ntrc , nsmp
      write( LUPRT, 914 ) nrec0, ntrc0, nsmp2
c
c===  output dds data file
c
      dx = ( 2.0*PI ) / ( nkx*dkx )
      dy = ( 2.0*PI ) / ( nky*dky )
c
      call zkwdds( outfil, otap, nzout+1, ntout+1, nx, ny, t0, z0, x0
     &            , y0, dz, dt, dx, dy, nsmp0, ntrc0, nrec0, verbos )
c
c===  allocate space for dynamic arrays
c
      lheadr  = ITRWRD * max0( ntrc, ntrc2 )
      ltrace  = ITRWRD + max0( nsmp, nsmp2 )
      lrec    = max0( nsmp, nsmp2 ) * max0( ntrc, ntrc2 )
      lrecout = lrec
      ltrig   = 2*nx
c
#ifdef CRAY
cdan
      lwork_8  = 2*nrec2 + 2*nsmp2*nrec2
     &                    + (ITRWRD+nsmp2)*nrec2*ntrc2
#else
cdan
      lwork_8  = nrec2*ntrc2
      lwork_8  = (ITRWRD+nsmp2)*lwork_8
     &         +  2*nrec2 + 4*nrec2+9*nsmp2+41 
#endif
c
cdan
      lwork_8 = max0( lwork_8, 1 + nzin + 6*(nzout+1) )
c
      lwork_8 = min0( lwork_8, 1048576 * mbytes / ISZBYT )
      lwork   = lwork_8
      write(0,*) 'lwork,lwork_8 = ',lwork,lwork_8
c
      call galloc( pheadr,   ISZBYT * lheadr,   ierr, 'ABORT' )
      call galloc( ptrace,   ISZBYT * ltrace,   ierr, 'ABORT' )
      call galloc( prec,     ISZBYT * lrec,     ierr, 'ABORT' )
      call galloc( precout,  ISZBYT * lrecout,  ierr, 'ABORT' )
      call galloc( pwork,    ISZBYT * lwork,    ierr, 'ABORT' )
      call galloc( ptrig,    ISZBYT * ltrig,    ierr, 'ABORT' )
c
c===  verbose output
c
      if( verbos ) then
         write( LUPRT , 920 ) idt, dt, dx, dy, dz, mbytes, zmax, tmax
         write( LUPRT , 922 ) nkx, nky, nx, ny, ntin, ntout, nzin, nzout
         write( LUPRT , 923 ) lheadr, ltrace, lrec, lrecout, lwork
     &                      , ltrig
      endif
c
c===  generate fft table for kx transform
c
      call cftfax( nx, ifac, trig )
c
c***********************************************************************
c
c===  begin the calculation
c
c***********************************************************************
c***********************************************************************
c
c===  perform complex fft across slowest dimension (records) ( ky -> y )
c===  NOTE : luinp will be obliterated!
c
cmat  time0 = second()
      idir = 1
c
      call cfftmltu( luinp, idir, nsmp, ntrc, nrec2, lwork, work
     &              , ierr )
c
cmat  timefft = second() - time0
cmat  write( LUPRT, * )' time for cfftmltu (ky->y) = ', timefft
cmat  write( LER,   * )' time for cfftmltu (ky->y) = ', timefft
c
      if( ierr .ne. 0 ) then
        write( LER, * )' ***** ERROR in CFFTMLTU ****** '
        go to 800
      endif
c***********************************************************************
c
c===  do second CFFT (kx->x), resample (zin->zout, or timin->timout)

      call rwd( luinp )
      call sisseek( luinp, 1 )
      timefft = 0.0
      timersmp = 0.0
c
c===  begin loop over records
c
c===  restore original dimensions, nrec0, ntrc0
       do jrec = 1, nrec0
c
c======  read record data
c======  rec will contain input data in { (z,x) or (t,x) }
c======  with the complex axis being the samples
c
         call rdrecn( luinp, LUPRT, jrec, ntrc, 1, ntrc, nsmp, 1, nsmp
     &          , ntoff, nsmp, 1, ntrc2, nsmp, trace, headr, rec, ierr
     &          , ISZBYT, ITRWRD, ITHWP1
     &          , ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER )

         if( ierr .ne. 0 ) then
            write( LER, * ) ' ***** ERROR reading record ***** '
            go to 800
         endif
c
c======  inverse FFT in the kx dimension (traces), complex axis is samples
c
cmat     time0 = second()
         call cfftmltn( rec(1), rec(2), work, trig, ifac, nsmp, 2
     &                , ntrc2, nsmp/2, 1 )
cmat     timefft = timefft + ( second() - time0 )
c
c======  rec now contains the image in ( (z,t), x )
c
c======  resample along depth or time and keep only real components
c
cmat     time0 = second()
c
         if( depth ) then
            call resamp(ntrc0, nsmp, nsmp2, nzin+1, 2, nzout+1, 1, zin
     &                , zout , work, rec, recout )
         endif
c
         if ( timel ) then
            call resamp(ntrc0, nsmp, nsmp2, ntin+1, 2, ntout+1, 1, timin
     &                , timout , work, rec, recout )
         endif
c
cmat     timersmp = timersmp + ( second() - time0 )
c
c======  recout now contains the resampled image in ( (z,t), kx )
c======  write output record
c
         call wrrec( luout, LUPRT, jrec, nsmp2, 1, ntrc0, nsmp2,
     &               trace, headr, recout, ierr )
         if( ierr .ne. 0 ) then
            write( LER, * ) ' ***** ERROR writing record ',jrec,' *****'
            go to 800
         endif
c
         if ( verbos ) then
            write( LUPRT,'('' Finished Record '',i5)')jrec
            write( LER,'('' Finished Record '',i5)')jrec
         endif
c
      enddo
c===  end loop over records
c
cmat  write( LUPRT, * ) ' TOTALS:  timefft = ', timefft, ' timersmp = '
cmat &                  , timersmp
cmat  write( LER , *  ) ' TOTALS:  timefft = ', timefft, ' timersmp = '
cmat &                  , timersmp
c
c***********************************************************************
c***********************************************************************
c
c===  calculation finished !
c
      ierr = 0
c
c===  close files, clean-up, & exit
c
800   continue
c
      if( luinp .gt. 1 ) call lbclos (luinp)
      if( luout .gt. 1 ) call lbclos (luout)
c
      if( ierr .le. 0 ) then
         if( LUPRT .gt. 0) write( LUPRT, 9998 )
         call exitfu(0)
      else
         if( LUPRT .gt. 0) write( LUPRT, 9999 ) ierr
         call exitfu(1)
      endif
c***********************************************************************
      end
c***********************************************************************
