      SUBROUTINE PRONY ( METHOD, N, IP, X, H, Z, XP, CFOUR, X5, ISTAT )

C
C   Solves for the exponential model parameters by the Prony method
C
C   Input parameters:
C
C     METHOD - Indicates desired method: 1 for regular Prony or 2
C              for modified Prony (integer)
C     N      - Number of data samples (integer)
C     IP     - Order of exponential model (integer); must be even
C              if METHOD=2
C     X      - Array of complex data samples X(1) through X(N)
C
C   Output parameters:
C
C     H      - Array of exponential model complex amplitudes
C     Z      - Array of exponential model complex exponents
C     ISTAT  - Integer status indicator at time of exit
C              0 - normal exit with no errors
C              1 - error exists in routine COVAR
C              2 - error exists in routine SYMCOVAR
C              3 - error exists in routine CPOLY
C              4 - error exists in routine CHOLESKY
C     XP     - Array of complex values predicting the original
C              sample values X(1), X(N)
C     CFOUR  - Array containing the Fourier transform (spectrum)
C              of the complex array XP
C
C   Notes:
C
C     External arrays H,Z must be dimensioned  .GE. IP  and array  X
C     must be dimensioned .GE. N in calling program.  Internal array
C     B must be dimensioned .GE. IP(IP+1)/2 ; arrays A, ROOTR, ROOTI
C     must be dimensioned .GE. IP ; arrays PR,PI must be dimensioned
C     .GE. IP+1 .  Array G must be dimensioned .GE. IP/2 .
C     Arrays XP and CFOUR must be dimensioned .GE. N in calling
C     program. The dimension for the complex arrays X, XP, H, Z,
C     CFOUR is set to 20000 in this subroutine
C
C     Subroutines COVAR (Appendix 8.B), SYMCOVAR (Appendix 14.A),
C     CPOLY (Appendix 14.B), and CHOLESKY (Appendix 3.A) are required
C

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
      
c variables passed from calling routine

      integer N, method, ip, istat

      complex X(N), H(N), Z(N), XP(N), CFOUR(N), X5(N)

c local variables

      INTEGER ORDFFT

      COMPLEX A(SZLNHD), B(SZLNHD), G(SZLNHD)
      COMPLEX X4(2*SZLNHD), W(2*SZLNHD)

      REAL PF,PB,PS,EPS
      REAL AMP(SZLNHD),DAMP(SZLNHD),FREQ(SZLNHD),PHASE(SZLNHD)

      REAL*8 PR(SZLNHD),PI(SZLNHD),ROOTR(SZLNHD),ROOTI(SZLNHD)
      REAL*8 C1,C2,C3,C4,C5,C6,SUMR,SUMI,SUM

      LOGICAL*4 FAIL

c initialize variables

      ISTAT=0
      EPS=1.E-15

C  ******************************************************************
C  *    FIRST STEP:  Covariance or symmetric covariance method      *
C  ******************************************************************

      nu=ordfft(n) + 1
      nnn=2 ** nu
      DO I=1,N
         X4(I)=X(I)
      END DO
      do i=1,nnn
         x5(i)=(0.0,0.0)
      end do

      CALL BURG ( N, IP, X, PF, A, JSTAT )

      IF (JSTAT .EQ. 0)  GO TO 5
      ISTAT=1
      RETURN

 5    PR(1)=1.D0
      PI(1)=0.D0
      X5(1)=(1.0,0.0)
      
      DO 10 K=1,IP
         X5(K+1)=A(K)
         PR(K+1)=REAL(A(K))
 10   PI(K+1)=AIMAG(A(K))
      
      IF(METHOD.EQ.2) GO TO 50
      CALL PREFFT(NNN,0,nu,W)
      CALL FFT(NNN,0,1.,nu,W,X5)
      GO TO 950
C     
 20   CALL SYMCOVAR (N,IP,X,PS,G,JSTAT)
      IF (JSTAT .EQ. 0)  GO TO 30
      ISTAT=2
      RETURN

 30   MHALF=IP/2
      M1=MHALF+1
      PR(M1)=1.D0
      PI(M1)=0.D0
      DO 40 K=1,MHALF
         PR(M1+K)=REAL(G(K))
         PI(M1+K)=-AIMAG(G(K))
         PR(M1-K)=REAL(G(K))
 40   PI(M1-K)=AIMAG(G(K))

C   *****************************************************************
C   *   SECOND STEP:  Polynomial rooting for complex exponential    *
C   *                 parameters                                    *
C   *****************************************************************

 50   CALL CPOLY (PR,PI,IP,ROOTR,ROOTI,FAIL)

      IF (.NOT. FAIL)  GO TO 55
      ISTAT=3
      RETURN

 55   DO 60 K=1,IP
 60   Z(K)=CMPLX(SNGL(ROOTR(K)),SNGL(ROOTI(K)))       

C   *****************************************************************
C   *   THIRD STEP:  Complex amplitude parameter estimates          *
C   *****************************************************************

      I=0
      DO 100 K=1,IP
         C1=ROOTR(K)
         C2=ROOTI(K)

         DO 80 J=1,K
            SUMR=C1*ROOTR(J)+C2*ROOTI(J)
            SUMI=C2*ROOTR(J)-C1*ROOTI(J)
            I=I+1
            C3=SUMR*SUMR+SUMI*SUMI
            SUM=C3-2.D0*SUMR+1.D0
            IF (SUM .EQ. 0.D0)  GO TO 70
            C3=C3**N
            C3=DSQRT(C3)
            C4=DATAN2(SUMI,SUMR)*N
            C5=C3*DCOS(C4)-1.D0
            C6=C3*DSIN(C4)
            SUMR=SUMR-1.D0
            SUMI=-SUMI
            C3=(SUMR*C5-SUMI*C6)/SUM
            C4=(SUMR*C6+SUMI*C5)/SUM
            B(I)=CMPLX(SNGL(C3),SNGL(C4))
            GO TO 80
 70         B(I)=CMPLX(FLOAT(N),0.)
 80      CONTINUE

         SUMR=REAL(X(1))
         SUMI=AIMAG(X(1))
         C2=-C2
         C3=1.D0
         C4=0.D0
         DO 90 J=2,N
            SUM=C3
            C3=SUM*C1-C4*C2
            C4=SUM*C2+C4*C1
            SUMR=SUMR+C3*REAL(X(J))-C4*AIMAG(X(J))
 90      SUMI=SUMI+C4*REAL(X(J))+C3*AIMAG(X(J))
 100  H(K)=CMPLX(SNGL(SUMR),SNGL(SUMI))

      CALL CHOLESKY (IP,EPS,B,H,JSTAT)

      IF (JSTAT .NE. 0)  ISTAT=4

C  *****************************************************************
C     COMPUTE HERE THE REAL AMPLITUDE, DAMPING, FREQUENCY AND PHASE
C     FROM THE COMPLEX AMPLITUDE AND EXPONENTIAL PARAMETER ARRAYS
C  *****************************************************************
C
C  *****************************************************************
C     COMPUTE HERE THE PREDICTED DATA SEQUENCE XP BASED ON THE PRONY
C     METHOD
C  *****************************************************************
C

      RMSD=0.0

      DO MM=1,N
         FI=FLOAT(MM-1)
         XP(MM)=(0.0,0.0)

         DO MM1=1,IP
            XP(MM)=XP(MM)+H(MM1)*(Z(MM1)**FI)
         END DO
         RMSD=RMSD+(X4(I)-XP(I))**2
      END DO

      RMSD=SQRT(RMSD/FLOAT(N))
      T=1.
      METHOD1=2
      CALL EXPARAMS(IP,T,H,Z,AMP,DAMP,FREQ,PHASE,N)

C  *****************************************************************
C     CALL FOR COMPUTING THE TWO-SIDED ESD AND THE ASSOCIATED PHASE
C     SPECTRUM OF THE SEQUENCE
C  *****************************************************************

      CALL ESD(METHOD1,IP,T,H,Z,N,CFOUR,SD)

C  *****************************************************************
C  *    Computation of residue could go here                       *
C  *****************************************************************

 950  continue

      RETURN
      END
      
      SUBROUTINE COVAR (N,IP,X,PF,AF,PB,AB,ISTAT)

#include <f77/lhdrsz.h>
C
C   Fast algorithm for the solution of the covariance least squares
C   normal equations.
C
C   Input Parameters:
C
C     N    - Number of data samples (integer)
C     IP   - Order of linear prediction model (integer)
C     X    - Array of complex data samples X(1) to X(N)
C
C   Output Parameters:
C
C     PF   - Real forward linear prediction variance at order IP
C     AF   - Array of complex forward linear prediction coefficients
C     PB   - Real backward linear prediction variance at order IP
C     AB    - Array of complex backward linear prediction coefficients
C     ISTAT - Integer indicating status of algorithm at time of exit:
C             0 for normal exit if no numerical ill-conditioning
C             1 if PF and PB are not positive values
C             2 if DELTA'' and GAMMA' do not lie in the range 0 to 1
C             3 if PF' and PB'' are not positive values
C             4 if DELTA and GAMMA do not lie in the range 0 to 1
C
C   Notes:
C
C     External arrays AF,AB must be dimensioned .GE. IP and array X must
C     be dimensioned .GE. N in the calling program.  Internal array  R
C     must be dimensioned .GE. IP and arrays  C,D  must be dimensioned
C     .GE. IP+1.  Program elements C(k+1) and D(k+1) correspond to text
C     elements c(k) and d(k), for k=0 to IP .
C
      integer N
      COMPLEX X(N),AF(SZLNHD),AB(SZLNHD)
      COMPLEX C(SZLNHD),D(SZLNHD),R(SZLNHD),EF,EB,THETA,TEMP,SAVE
      complex C1,C2,C3,C4
C
C   Initialization
C
      R1=0.
      DO 10 K=2,N-1
10      R1=R1+REAL(X(K))**2+AIMAG(X(K))**2
      R2=REAL(X(1))**2+AIMAG(X(1))**2
      R3=REAL(X(N))**2+AIMAG(X(N))**2
      PF=R1+R3
      PB=R1+R2
      R4=R1+R2+R3
      R5=1./R4
      DELTA=1.-R2*R5
      GAMMA=1.-R3*R5
      C(1)=CONJG(X(N))*R5
      D(1)=CONJG(X(1))*R5
      ISTAT=0
      M=0
      IF (IP .NE. 0)  GO TO 1000
      PF=R4/FLOAT(N)
      PB=PF
      RETURN
C
C   MAIN LOOP
C
1000  M=M+1
      R1=1./PF
      R2=1./PB
      R3=1./DELTA
      R4=1./GAMMA
C
C   Order update: AF and AB vectors ; time update: C and D vectors
C
      TEMP=(0.,0.)
      DO 20 K=M+1,N
20      TEMP=TEMP+X(K)*CONJG(X(K-M))
      R(M)=CONJG(TEMP)
      THETA=X(1)*C(M)
      IF (M .EQ. 1)  GO TO 40
      DO 30 K=1,M-1
        THETA=THETA+X(M+1-K)*C(K)              ! Eq. (8.C.39)
        R(K)=R(K)-X(N+1-M)*CONJG(X(N+1-M+K))   ! Eq. (8.C.32)
30      TEMP=TEMP+CONJG(R(K))*AF(M-K)
40    C1=-TEMP*R2
      C2=-CONJG(TEMP)*R1
      C3=THETA*R3
      C4=CONJG(THETA)*R4
      AF(M)=C1                                 ! Eq. (8.C.19)
      AB(M)=C2                                 ! Eq. (8.C.22)
      SAVE=C(M)
      C(M)=SAVE+C3*D(M)
      D(M)=D(M)+C4*SAVE
      IF (M .EQ. 1)  GO TO 60
      DO 50 K=1,M-1
        SAVE=AF(K)
        AF(K)=SAVE+C1*AB(M-K)                  ! Eq. (8.C.18)
        AB(M-K)=AB(M-K)+C2*SAVE                ! Eq. (8.C.21)
        SAVE=C(K)
        C(K)=SAVE+C3*D(K)                      ! Eq. (8.C.37)
50      D(K)=D(K)+C4*SAVE                      ! Eq. (8.C.38)
60    R5=REAL(TEMP)**2+AIMAG(TEMP)**2
      PF=PF-R5*R2                              ! Eq. (8.C.20)
      PB=PB-R5*R1                              ! Eq. (8.C.23)
      R5=REAL(THETA)**2+AIMAG(THETA)**2
      DELTA=DELTA-R5*R4                        ! Eq. (8.C.39)
      GAMMA=GAMMA-R5*R3                        ! Eq. (8.C.40)
      IF (M .NE. IP)  GO TO 65
      PF=PF/FLOAT(N-M)
      PB=PB/FLOAT(N-M)
      RETURN
65    IF (PF .GT. 0. .AND. PB .GT. 0.)  GO TO 70
      ISTAT=1
      RETURN
70    IF (DELTA .GT. 0. .AND. DELTA .LE. 1. .AND. GAMMA .GT. 0.
     *    .AND. GAMMA .LE. 1)  GO TO 80
      ISTAT=2
      RETURN
C
C   Time update:  AF and AB vectors; order update:  C and D vectors
C
80    R1=1./PF
      R2=1./PB
      R3=1./DELTA
      R4=1./GAMMA
      EF=X(M+1)
      EB=X(N-M)
      DO 90 K=1,M
        EF=EF+AF(K)*X(M+1-K)                   ! Eq. (8.C.1)
90      EB=EB+AB(K)*X(N-M+K)                   ! Eq. (8.C.2)
      C1=EF*R3
      C2=EB*R4
      C3=CONJG(EB)*R2
      C4=CONJG(EF)*R1
      DO 100 K=M,1,-1
        SAVE=AF(K)
        AF(K)=SAVE+C1*D(K)                     ! Eq. (8.C.33)
        D(K+1)=D(K)+C4*SAVE                    ! Eq. (8.C.25)
        SAVE=AB(K)
        AB(K)=SAVE+C2*C(M+1-K)                 ! Eq. (8.C.35)
100     C(M+1-K)=C(M+1-K)+C3*SAVE              ! Eq. (8.C.24)
      C(M+1)=C3
      D(1)=C4
      R5=REAL(EF)**2+AIMAG(EF)**2
      PF=PF-R5*R3                              ! Eq. (8.C.34)
      DELTA=DELTA-R5*R1                        ! Eq. (8.C.30)
      R5=REAL(EB)**2+AIMAG(EB)**2
      PB=PB-R5*R4                              ! Eq. (8.C.36)
      GAMMA=GAMMA-R5*R2                        ! Eq. (8.C.31)
      IF (PF .GT. 0. .AND. PB .GT. 0.)  GO TO 110
      ISTAT=3
      RETURN
110   IF (DELTA .GT. 0. .AND. DELTA .LE. 1. .AND. GAMMA .GT. 0.
     *    .AND. GAMMA .LE. 1.)  GO TO 1000
      ISTAT=4
      RETURN
      END

      SUBROUTINE SYMCOVAR ( N, IP, X, PS, G, ISTAT )

#include <f77/lhdrsz.h>

C
C   Fast algorithm for the solution of the Hermitian symmetric covariance
C   least squares normal equations.
C 
C   Input Parameters:
C
C     N     - Number of data samples (integer)
C     IP    - Order of linear prediction model (must be an even integer)
C     X     - Array of complex data samples X(1) through X(N)
C
C   Output Parameters:
C
C     PS    - Real linear smoothing squared error at order IP; note that PF
C             must be scaled by 1/(N-IP) to make it a variance estimate
C     G     - Array of IP/2 complex linear smoothing coefficients
C     ISTAT - Integer status indicator at time of exit
C             0 for normal exit (no numerical ill-conditioning)
C             1 if P is not a positive value
C             2 if DELTA' and GAMMA' are not in the range 0 to 1
C             3 if P' is not a positive value
C             4 if DELTA and GAMMA are not in the range 0 to 1
C             5 if IP is not an even number
C             6 if PS is not a positive value
C             7 if PS" is not a positive value
C             8 if error in subroutine CHOLESKY
C
C   Notes:
C
C     External array G must be dimensioned .GE. IP/2 .  External array X
C     must be dimensioned .GE. N and array A must be dimensioned .GE. IP
C     in the calling program.   Internal array R must be dimensioned .GE.
C     IP  and arrays C,D must be dimensioned .GE. IP+1.  Internal arrays
C     AA and BB are fixed at the indicated dimensions.
C

      integer N, IP, ISTAT

      COMPLEX X(N),AF(SZLNHD),C(SZLNHD),D(SZLNHD),R(SZLNHD),
     & LAMBDA,THETA,PSI,XI
      COMPLEX EF,EB,C1,C2,C3,C4,SAVE1,SAVE2,SAVE3,SAVE4
      COMPLEX G(SZLNHD),AA(10),BB(4),ESP,ESN
      REAL PS,DELTA,GAMMA,P,R1,R2,R3,R4,R5,EPS
C
C   Initialization
C
      EPS=1.E-15
      IF (MOD(IP,2) .EQ. 0)  GO TO 5
      ISTAT=5
      RETURN
5     R1=0.
      DO 10 K=2,N-1
10      R1=R1+REAL(X(K))**2+AIMAG(X(K))**2
      R2=REAL(X(1))**2+AIMAG(X(1))**2
      R3=REAL(X(N))**2+AIMAG(X(N))**2
      R4=1./(2.*(R1+R2+R3))
      P=2.*R1+R2+R3
      PS=R1                                       ! Eq. (11.A.20)
      DELTA=1.-R2*R4
      GAMMA=1.-R3*R4
      LAMBDA=CONJG(X(1)*X(N))*R4
      C(1)=X(N)*R4
      D(1)=CONJG(X(1))*R4
      ISTAT=0
      M=0
      IF (IP .EQ. 0)  RETURN
C
C   Main loop
C
1000  M=M+1
      SAVE1=(0.,0.)
      DO 20 K=M+1,N
20      SAVE1=SAVE1+X(K)*CONJG(X(K-M))
      SAVE1=2.*SAVE1
      R(M)=CONJG(SAVE1)
      THETA=X(N)*D(1)
      PSI=X(N)*C(1)
      XI=CONJG(X(1))*D(1)
      IF (M .EQ. 1)  GO TO 40
      DO 30 K=1,M-1
        THETA=THETA+X(N-K)*D(K+1)                  ! Eq. (8.D.45)
        PSI=PSI+X(N-K)*C(K+1)                      ! Eq. (8.D.45)
        XI=XI+CONJG(X(K+1))*D(K+1)                 ! Eq. (8.D.45)
        R(K)=R(K)-X(N+1-M)*CONJG(X(N+1-M+K))
     *           -CONJG(X(M))*X(M-K)               ! Eq. (8.D.37)
30      SAVE1=SAVE1+CONJG(R(K))*AF(M-K)            ! Eq. (8.D.24)
C
C   Order update of AF vector
C
40    C1=-SAVE1/P
      AF(M)=C1                                     ! Eq. (8.D.23)
      P=P*(1.-REAL(C1)**2-AIMAG(C1)**2)            ! Eq. (8.D.25)
      IF (M .EQ. 1)  GO TO 60
C
      IF (MOD(M,2) .EQ. 0)  GO TO 46
      MH=(M-1)/2
      MH1=MH+1
      BB(1)=ESP
      BB(2)=CONJG(ESP)
      BB(3)=CONJG(ESN)
      BB(4)=ESN
      AA(1)=CMPLX(DELTA,0.)
      AA(2)=-CONJG(XI)
      AA(3)=CMPLX(DELTA,0.)
      AA(4)=-CONJG(LAMBDA)
      AA(5)=-THETA
      AA(6)=CMPLX(GAMMA,0.)
      AA(7)=-CONJG(THETA)
      AA(8)=-LAMBDA
      AA(9)=-CONJG(PSI)
      AA(10)=CMPLX(GAMMA,0.)
      CALL CHOLESKY (4,EPS,AA,BB,JSTAT)            ! Eq. (11.A.17)
      IF (JSTAT .EQ. 0)  GO TO 42
      ISTAT=8
      RETURN
42    R5=1./(1.+2.*REAL(BB(3)*C(MH1))+2.*REAL(BB(1)*D(MH1))) !(11.A.18)
      DO 44 K=1,MH
44      G(K)=R5*(G(K)+BB(4)*CONJG(C(MH1+K))+BB(3)*C(MH1-K) ! (11.A.16)
     *       +BB(2)*CONJG(D(MH1+K))+BB(1)*D(MH1-K))  ! Eq. (11.A.19)
      PS=R5*PS
      IF (PS .GT. 0)  GO TO 46
      ISTAT=7
      RETURN
C
46    DO 50 K=1,M/2
        MK=M-K
        SAVE1=AF(K)
        AF(K)=SAVE1+C1*CONJG(AF(MK))               ! Eq. (8.D.22)
        IF (K .EQ. MK)  GO TO 50
        AF(MK)=AF(MK)+C1*CONJG(SAVE1)              ! Eq. (8.D.22)
50      CONTINUE
C
      IF (MOD(M,2) .NE. 0)  GO TO 60
      MH=M/2
      MH1=MH+1
      SAVE1=R(MH)
      DO 54 K=1,MH-1
54      SAVE1=SAVE1+CONJG(G(K))*R(MH+K)+G(K)*R(MH-K)
      C1=-SAVE1/P                                  ! Eq. (11.A.13)
      R5=1./(1.+2.*REAL(C1*AF(MH)))                ! Eq. (11.A.14)
      G(MH)=R5*(C1+CONJG(C1*AF(M)))
      ESP=X(MH1)+G(MH)*X(M+1)+CONJG(G(MH))*X(1)
      ESN=X(N-MH)+G(MH)*X(N)+CONJG(G(MH))*X(N-M)
      DO 56 K=1,MH-1
        G(K)=R5*(G(K)+C1*AF(MH-K)+CONJG(C1*AF(MH+K))) ! Eq. (11.A.12)
        ESP=ESP+G(K)*X(MH1+K)+CONJG(G(K))*X(MH1-K) ! Eq. (11.A.1)
56      ESN=ESN+G(K)*X(N-MH+K)+CONJG(G(K))*X(N-MH-K) ! Eq. (11.A.1)
      PS=R5*PS                                     ! Eq. (11.A.15)
      IF (PS .GT. 0.)  GO TO 60
      ISTAT=6
      RETURN
C
60    IF (M .EQ. IP)  RETURN
C
C   Time update of C,D vectors and GAMMA,DELTA,LAMBDA scalars
C
65    R1=1./(DELTA*GAMMA-REAL(LAMBDA)**2-AIMAG(LAMBDA)**2)
      C1=(THETA*CONJG(LAMBDA)+PSI*DELTA)*R1
      C2=(PSI*LAMBDA+THETA*GAMMA)*R1
      C3=(XI*CONJG(LAMBDA)+THETA*DELTA)*R1
      C4=(THETA*LAMBDA+XI*GAMMA)*R1
      DO 70 K=1,(M-1)/2+1
        MK=M+1-K
        SAVE1=CONJG(C(K))
        SAVE2=CONJG(D(K))
        SAVE3=CONJG(C(MK))
        SAVE4=CONJG(D(MK))
        C(K)=C(K)+C1*SAVE3+C2*SAVE4                ! Eq. (8.D.43)
        D(K)=D(K)+C3*SAVE3+C4*SAVE4                ! Eq. (8.D.44)
        IF (K .EQ. MK)  GO TO 70
        C(MK)=C(MK)+C1*SAVE1+C2*SAVE2              ! Eq. (8.D.43)
        D(MK)=D(MK)+C3*SAVE1+C4*SAVE2              ! Eq. (8.D.44)
70      CONTINUE
      R2=REAL(PSI)**2+AIMAG(PSI)**2
      R3=REAL(THETA)**2+AIMAG(THETA)**2
      R4=REAL(XI)**2+AIMAG(XI)**2
      R5=GAMMA-(R2*DELTA+R3*GAMMA+2.*REAL(PSI*LAMBDA*CONJG(THETA)))*R1
      R2=DELTA-(R3*DELTA+R4*GAMMA+2.*REAL(THETA*LAMBDA*CONJG(XI)))*R1
      GAMMA=R5                                     ! Eq. (8.D.46)
      DELTA=R2                                     ! Eq. (8.D.47)
      LAMBDA=LAMBDA+C3*CONJG(PSI)+C4*CONJG(THETA)  ! Eq. (8.D.48)
      IF (P .GT. 0.)  GO TO 80
      ISTAT=1
      RETURN
80    IF (DELTA .GT. 0. .AND. DELTA .LT. 1. .AND. GAMMA .GT. 0.
     *    .AND. GAMMA .LT. 1.)  GO TO 90
      ISTAT=2
      RETURN
C
C   Time update of AF vector; order updates of C,D vectors and GAMMA,
C   DELTA,LAMBDA scalars
C
90    R1=1./P
      R2=1./(DELTA*GAMMA-REAL(LAMBDA)**2-AIMAG(LAMBDA)**2) ! Eq. (8.D.41)
      EF=X(M+1)
      EB=X(N-M)
      DO 100 K=1,M
        EF=EF+AF(K)*X(M+1-K)                       ! Eq. (8.D.1)
100     EB=EB+CONJG(AF(K))*X(N-M+K)                ! Eq. (8.D.2)
      C1=EB*R1                                     ! Eq. (8.D.28)
      C2=CONJG(EF)*R1                              ! Eq. (8.D.29)
      C3=(CONJG(EB)*DELTA+EF*LAMBDA)*R2
      C4=(EF*GAMMA+CONJG(EB*LAMBDA))*R2
      DO 110 K=M,1,-1
        SAVE1=AF(K)
        AF(K)=SAVE1+C3*C(K)+C4*D(K)                ! Eq. (8.D.38)
        C(K+1)=C(K)+C1*SAVE1                       ! Eq. (8.D.26)
110     D(K+1)=D(K)+C2*SAVE1                       ! Eq. (8.D.27)
      C(1)=C1
      D(1)=C2
      R3=REAL(EB)**2+AIMAG(EB)**2
      R4=REAL(EF)**2+AIMAG(EF)**2
      P=P-(R3*DELTA+R4*GAMMA+2.*REAL(EF*EB*LAMBDA))*R2  ! Eq. (8.D.42)
      DELTA=DELTA-R4*R1                            ! Eq. (8.D.32)
      GAMMA=GAMMA-R3*R1                            ! Eq. (8.D.33)
      LAMBDA=LAMBDA+CONJG(EF*EB)*R1                ! Eq. (8.D.35)
      IF (P .GT. 0.)  GO TO 120
      ISTAT=3
      RETURN
120   IF (DELTA .GT. 0. .AND. DELTA .LT. 1. .AND. GAMMA .GT. 0.
     *    .AND. GAMMA .LT. 1.)  GO TO 1000
      ISTAT=4
      RETURN
      END


      SUBROUTINE CPOLY (OPR,OPI,DEGREE,ZEROR,ZEROI,FAIL)

#include <f77/lhdrsz.h>

C
C   Copyright 1972, Association for Computing Machinery.  Reprinted
C   with permission.
C
C   This program from M.A. Jenkins and J.F. Traub, "Zeros of a Complex
C   Polynomial," Communications of the ACM, vol.15, pp. 97-99, Feb. 1972
C
C   Note that subroutine MCON sets up four variables for the precision
C   and floating-point arithmetic range of the VAX 11/780.
C
C   Finds the zeros of a complex polynomial.
C
C   OPR,OPI     - Double-precision vectors of real and imaginary parts
C                 of the coefficients in order of decreasing powers
C   DEGREE      - Integer degree of the polynomial
C   ZEROR,ZEROI - Output double-precision vectors of real and imaginary
C                 parts of the zeros
C   FAIL        - Output logical parameter; .TRUE. only if leading
C                 coefficient is zero or if CPOLY has found fewer than
C                 DEGREE zeros
C
C   The program has been written to reduce the chance of overflow
C   occurring.  If it does occur, there is still a possibility that the
C   zerofinder will work provided the overflowed quantity is replaced
C   by a large number.
C
      COMMON /GLOBAL/PR,PI,HR,HI,QPR,QPI,QHR,QHI,SHR,SHI,SR,SI,TR,TI,
     *               PVR,PVI,ARE,MRE,ETA,INFIN,NN

      integer ip
      REAL*8 SR,SI,TR,TI,PVR,PVI,ARE,MRE,ETA,INFIN
      REAL*8 PR(SZLNHD), PI(SZLNHD),HR(1000),HI(1000)
      real*8 QPR(1000),QPI(1000)
      REAL*8 QHR(1000),QHI(1000),SHR(1000),SHI(1000)
C   To change the size of the polynomials which can be solved, replace
C   the dimension of the array in the COMMON area.
      REAL*8 XX,YY,COSR,SINR,SMALNO,BASE,XXX,ZR,ZI,BND
c     REAL*8 OPR(1),OPI(1),ZEROR(1),ZEROI(1),CMOD,SCALE,CAUCHY,DSQRT
      REAL*8 OPR(1),OPI(1),ZEROR(1),ZEROI(1),CMOD,CAUCHY,DSQRT
c
c - this routine was renamed to avoid conflict with system
c   intrinsics by the same name
c
      REAL*8 FUNCTION SKALE
      LOGICAL*4 FAIL,CONV
      INTEGER*4 DEGREE,CNT1,CNT2
C   Initialization of constants
      CALL MCON (ETA,INFIN,SMALNO,BASE)
      ARE = ETA
      MRE = 2.D0*DSQRT(2.D0)*ETA
      XX = .70710678D0
      YY = -XX
      COSR = -.069756474D0
      SINR = .99756405D0
      FAIL = .FALSE.
      NN=DEGREE+1
C   Algorithm fails if the leading coefficient is zero
      IF (OPR(1) .NE. 0.D0 .OR. OPI(1) .NE. 0.D0)  GO TO 10
          FAIL = .TRUE.
          RETURN
C   Remove the zeros at the origin, if any
10    IF (OPR(NN) .NE. 0.D0 .OR. OPI(NN) .NE. 0.D0)  GO TO 20
          IDNN2 = DEGREE-NN+2
          ZEROR(IDNN2) = 0.D0
          ZEROI(IDNN2) = 0.D0
          NN = NN-1
          GO TO 10
C   Make a copy of the coefficients
20    DO 30 I=1,NN
          PR(I) = OPR(I)
          PI(I) = OPI(I)
          SHR(I) = CMOD(PR(I),PI(I))
30        CONTINUE
C   Scale the polynomial
      BND = SKALE(NN,SHR,ETA,INFIN,SMALNO,BASE)
      IF (BND .EQ. 1.D0)  GO TO 40
      DO 35 I=1,NN
        PR(I) = BND*PR(I)
        PI(I) = BND*PI(I)
35      CONTINUE
C   Start the algorithm for one zero
40    IF (NN .GT. 2)  GO TO 50
      IF (NN .EQ. 1)  RETURN
C   Calculate the final zero and return
        CALL CDIVID (-PR(2),-PI(2),PR(1),PI(1),ZEROR(DEGREE),
     *               ZEROI(DEGREE))
        RETURN
C   Calculate BND, a lower bound on the modulus of the zeros
50    DO 60 I=1,NN
        SHR(I) = CMOD(PR(I),PI(I))
60      CONTINUE
      BND = CAUCHY(NN,SHR,SHI)
C   Outer loop to control two major passes with different sequences
C   of shifts
      DO 100 CNT1=1,2
C   First stage calculation, no shift
        CALL NOSHFT (5)
C   Inner loop to select a shift
        DO 90 CNT2=1,9
C   Shift is chosen with modulus BND and amplitude rotated by 94 degrees
C   from the previous shift
          XXX = COSR*XX-SINR*YY
          YY = SINR*XX+COSR*YY
          XX = XXX
          SR = BND*XX
          SI = BND*YY
C   Second stage calculation, fixed shift
          CALL FXSHFT (10*CNT2,ZR,ZI,CONV)
          IF (.NOT. CONV)  GO TO 80
C   The second stage jumps directly to the third-stage iteration; if
C   successful, the zero is stored and the polynomial deflated.
            IDNN2 = DEGREE-NN+2
            ZEROR(IDNN2) = ZR
            ZEROI(IDNN2) = ZI
            NN = NN-1
            DO 70 I=1,NN
              PR(I) = QPR(I)
              PI(I) = QPI(I)
70            CONTINUE
            GO TO 40
80        CONTINUE
C   If the iteration is unsuccessful, another shift is chosen
90      CONTINUE
C   If nine shifts fail, the outer loop is repeated with another
C   sequence of shifts
100   CONTINUE
C   The zerofinder has failed on two major passes; return empty-handed
      FAIL = .TRUE.
      RETURN
      END


      SUBROUTINE NOSHFT (L1)
#include <f77/lhdrsz.h>
C   Computes the derivative polynomial as the initial H
C   polynomial and computes L1 no-shift H polynomials
C
      COMMON /GLOBAL/PR,PI,HR,HI,QPR,QPI,QHR,QHI,SHR,SHI,
     *               SR,SI,TR,TI,PVR,PVI,ARE,MRE,ETA,INFIN,NN
      REAL*8 SR,SI,TR,TI,PVR,PVI,ARE,MRE,ETA,INFIN
      REAL*8 PR(SZLNHD),PI(SZLNHD),HR(1000),HI(1000),QPR(1000),QPI(1000)
      REAL*8 QHR(1000),QHI(1000),SHR(1000),SHI(1000)
      REAL*8 XNI,T1,T2,CMOD
      N = NN-1
      NM1 = N-1
      DO 10 I=1,N
        XNI = NN-I
        HR(I) = XNI*PR(I)/FLOAT(N)
        HI(I) = XNI*PI(I)/FLOAT(N)
10      CONTINUE
      DO 50 JJ=1,L1
        IF (CMOD(HR(N),HI(N)) .LE. ETA*10.D0*CMOD(PR(N),PI(N)))
     *                GO TO 30
        CALL CDIVID (-PR(NN),-PI(NN),HR(N),HI(N),TR,TI)
        DO 20 I=1,NM1
          J = NN-I
          T1 = HR(J-1)
          T2 = HI(J-1)
          HR(J) = TR*T1-TI*T2+PR(J)
          HI(J) = TR*T2+TI*T1+PI(J)
20        CONTINUE
        HR(1) = PR(1)
        HI(1) = PI(1)
        GO TO 50
C   If the constant term is essentially zero, shift H coefficients
30      DO 40 I=1,NM1
          J = NN-I
          HR(J) = HR(J-1)
          HI(J) = HI(J-1)
40        CONTINUE
        HR(1) = 0.D0
        HI(1) = 0.D0
50      CONTINUE
      RETURN
      END


      SUBROUTINE FXSHFT (L2,ZR,ZI,CONV)
#include <f77/lhdrsz.h>
C   Computes L2 fixed-shift H polynomials and tests for convergence.
C   Initiates a variable-shift iteration and returns with the 
C   approximate zero if successful.
C
C   L2    - Limit of fixed-shift steps
C   ZR,ZI - Approximate zero if CONV is .TRUE.
C   CONV  - Logical indicating convergence of stage three iteration
C
      COMMON /GLOBAL/PR,PI,HR,HI,QPR,QPI,QHR,QHI,SHR,SHI,SR,SI,
     *               TR,TI,PVR,PVI,ARE,MRE,ETA,INFIN,NN
      REAL*8 SR,SI,TR,TI,PVR,PVI,ARE,MRE,ETA,INFIN
      REAL*8 PR(SZLNHD),PI(SZLNHD),HR(1000),HI(1000),QPR(1000),QPI(1000)
      REAL*8 QHR(1000),QHI(1000),SHR(1000),SHI(1000)
      REAL*8 ZR,ZI,OTR,OTI,SVSR,SVSI,CMOD
      LOGICAL*4 CONV,TEST,PASD,BOOL
      N = NN-1
C   Evaluate P at S
      CALL POLYEV (NN,SR,SI,PR,PI,QPR,QPI,PVR,PVI)
      TEST = .TRUE.
      PASD = .FALSE.
C   Calculate first T = -P(S)/H(S)
      CALL CALCT (BOOL)
C   Main loop for one second-stage step
      DO 50 J=1,L2
        OTR = TR
        OTI = TI
C   Compute next H polynomial and new T
        CALL NEXTH (BOOL)
        CALL CALCT (BOOL)
        ZR = SR+TR
        ZI = SI+TI
C   Test for convergence, unless stage three has failed once or this
C   is the last H polynomial
        IF (BOOL .OR. .NOT. TEST .OR. J .EQ. L2)  GO TO 50
        IF (CMOD(TR-OTR,TI-OTI) .GE. .5D0*CMOD(ZR,ZI))  GO TO 40
          IF (.NOT. PASD)  GO TO 30
C   The weak convergence test has been passed twice; start the third-
C   stage iteration, after saving the current H polynomial and shift
            DO 10 I=1,N
              SHR(I) = HR(I)
              SHI(I) = HI(I)
10            CONTINUE
            SVSR = SR
            SVSI = SI
            CALL VRSHFT (10,ZR,ZI,CONV)
            IF (CONV)  RETURN
C   The iteration failed to converge.  Turn off testing and restore H,
C   S,PV, and T
            TEST = .FALSE.
            DO 20 I=1,N
              HR(I) = SHR(I)
              HI(I) = SHI(I)
20            CONTINUE
            SR = SVSR
            SI = SVSI
            CALL POLYEV (NN,SR,SI,PR,PI,QPR,QPI,PVR,PVI)
            CALL CALCT (BOOL)
            GO TO 50
30        PASD = .TRUE.
          GO TO 50
40      PASD = .FALSE.
50    CONTINUE
C   Attempt an iteration with final H polynomial from second stage
      CALL VRSHFT (10,ZR,ZI,CONV)
      RETURN
      END


      SUBROUTINE VRSHFT (L3,ZR,ZI,CONV)
#include <f77/lhdrsz.h>
C   Carries out the third-stage iteration
C
C   L3    - Limit of steps in stage three
C   ZR,ZI - On entry contains the initial iterate; if the iteration
C           converges, it contains the final iterate on exit
C   CONV  - .TRUE. if iteration converges
C
      COMMON /GLOBAL/PR,PI,HR,HI,QPR,QPI,QHR,QHI,SHR,SHI,SR,SI,
     *               TR,TI,PVR,PVI,ARE,MRE,ETA,INFIN,NN
      REAL*8 SR,SI,TR,TI,PVR,PVI,ARE,MRE,ETA,INFIN
      REAL*8 PR(SZLNHD),PI(SZLNHD),HR(1000),HI(1000),QPR(1000),QPI(1000)
      REAL*8 QHR(1000),QHI(1000),SHR(1000),SHI(1000)
      REAL*8 ZR,ZI,MP,MS,OMP,RELSTP,R1,R2,CMOD,DSQRT,ERREV,TP
      LOGICAL*4 CONV,B,BOOL
      CONV = .FALSE.
      B = .FALSE.
      SR = ZR
      SI = ZI
C   Main loop for stage three
      DO 60 I=1,L3
C   Evaluate P at S and test for convergence
        CALL POLYEV (NN,SR,SI,PR,PI,QPR,QPI,PVR,PVI)
        MP = CMOD(PVR,PVI)
        MS = CMOD(SR,SI)
        IF (MP .GT. 20.D0*ERREV(NN,QPR,QPI,MS,MP,ARE,MRE))  GO TO 10
C   Polynomial value is smaller in value than a bound on the error
C   in evaluating P; terminate the iteration
          CONV = .TRUE.
          ZR = SR
          ZI = SI
          RETURN
10      IF (I .EQ. 1)  GO TO 40
          IF (B .OR. MP .LT. OMP .OR. RELSTP .GE. .05D0)  GO TO 30
C   Iteration has stalled.  Probably a cluster of zeros.  Do five
C   fixed-shift steps into the cluster to force one zero to dominate.
            TP = RELSTP
            B = .TRUE.
            IF (RELSTP .LT. ETA)  TP = ETA
            R1 = DSQRT(TP)
            R2 = SR*(1.D0+R1)-SI*R1
            SI = SR*R1+SI*(1.D0+R1)
            SR = R2
            CALL POLYEV (NN,SR,SI,PR,PI,QPR,QPI,PVR,PVI)
            DO 20 J=1,5
              CALL CALCT (BOOL)
              CALL NEXTH (BOOL)
20            CONTINUE
            OMP = INFIN
            GO TO 50
C   Exit if the polynomial value increases significantly
30      IF (MP*.1D0 .GT. OMP)  RETURN
40    OMP =MP
C   Calculate next iterate
50    CALL CALCT (BOOL)
      CALL NEXTH (BOOL)
      CALL CALCT (BOOL)
      IF (BOOL)  GO TO 60
      RELSTP = CMOD(TR,TI)/CMOD(SR,SI)
      SR = SR+TR
      SI = SI+TI
60    CONTINUE
      RETURN
      END


      SUBROUTINE CALCT (BOOL)
#include <f77/lhdrsz.h>
C   Computes T = -P(S)/H(S)
C
C   BOOL - Logical, set .TRUE. if H(S) is essentially zero
      COMMON /GLOBAL/PR,PI,HR,HI,QPR,QPI,QHR,QHI,SHR,SHI,SR,SI,
     *               TR,TI,PVR,PVI,ARE,MRE,ETA,INFIN,NN
      REAL*8 SR,SI,TR,TI,PVR,PVI,ARE,MRE,ETA,INFIN
      REAL*8 PR(SZLNHD),PI(SZLNHD),HR(1000),HI(1000),QPR(1000),QPI(1000)
      REAL*8 QHR(1000),QHI(1000),SHR(1000),SHI(1000)
      REAL*8 HVR,HVI,CMOD
      LOGICAL*4 BOOL
      N = NN-1
C   Evaluate H(S)
      CALL POLYEV (N,SR,SI,HR,HI,QHR,QHI,HVR,HVI)
      BOOL = CMOD(HVR,HVI) .LE. ARE*10.D0*CMOD(HR(N),HI(N))
      IF (BOOL)  GO TO 10
        CALL CDIVID (-PVR,-PVI,HVR,HVI,TR,TI)
        RETURN
10    TR = 0.D0
      TI = 0.D0
      RETURN
      END


      SUBROUTINE NEXTH (BOOL)
#include <f77/lhdrsz.h>
C   Calculates the next shifted H polynomial
C
C   BOOL - Logical, if .TRUE. H(S) is essentially zero
      COMMON /GLOBAL/PR,PI,HR,HI,QPR,QPI,QHR,QHI,SHR,SHI,SR,SI,
     *               TR,TI,PVR,PVI,ARE,MRE,ETA,INFIN,NN
      REAL*8 SR,SI,TR,TI,PVR,PVI,ARE,MRE,ETA,INFIN
      REAL*8 PR(SZLNHD),PI(SZLNHD),HR(1000),HI(1000),QPR(1000),QPI(1000)
      REAL*8  QHR(1000),QHI(1000),SHR(1000),SHI(1000),T1,T2
      LOGICAL*4 BOOL
      N = NN-1
      IF (BOOL)  GO TO 20
        DO 10 J=2,N
          T1 = QHR(J-1)
          T2 = QHI(J-1)
          HR(J) = TR*T1-TI*T2+QPR(J)
          HI(J) = TR*T2+TI*T1+QPI(J)
10        CONTINUE
        HR(1) = QPR(1)
        HI(1) = QPI(1)
        RETURN
C   If H(S) is zero, replace H with QH
20    DO 30 J=2,N
        HR(J) = QHR(J-1)
        HI(J) = QHI(J-1)
30      CONTINUE
      HR(1) = 0.D0
      HI(1) = 0.D0
      RETURN
      END


      SUBROUTINE POLYEV (NN,SR,SI,PR,PI,QR,QI,PVR,PVI)
C   Evaluates a polynomial P at S by the Horner recurrence,
C   placing the partial sums in Q and the computed value in PV
      REAL*8 PR(NN),PI(NN),QR(NN),QI(NN),SR,SI,PVR,PVI,T
      QR(1) = PR(1)
      QI(1) = PI(1)
      PVR = QR(1)
      PVI = QI(1)
      DO 10 I=2,NN
        T = PVR*SR-PVI*SI+PR(I)
        PVI = PVR*SI+PVI*SR+PI(I)
        PVR = T
        QR(I) = PVR
        QI(I) = PVI
10      CONTINUE
      RETURN
      END


      REAL*8 FUNCTION ERREV(NN,QR,QI,MS,MP,ARE,MRE)
C   Bounds the error in evaluating the polynomial by the Horner recurrence
C
C   QR,QI   - The partial sums
C   MS      - Modulus of the point
C   MP      - Modulus of the polynomial value
C   ARE,MRE - Error bounds on complex addition and multiplication
      REAL*8 QR(NN),QI(NN),MS,MP,ARE,MRE,E,CMOD
      E = CMOD(QR(1),QI(1))*MRE/(ARE+MRE)
      DO 10 I=1,NN
        E = E*MS+CMOD(QR(I),QI(I))
10      CONTINUE
      ERREV = E*(ARE+MRE)-MP*MRE
      RETURN
      END


      REAL*8 FUNCTION CAUCHY(NN,PT,Q)
C   Cauchy computes a lower bound on the moduli of the zeros of a
C   polynomial - PT is the modulus of the coefficients
      REAL*8 Q(1000),PT(1000),X,XM,F,DX,DF,DABS,DEXP,DLOG
      PT(NN) = -PT(NN)
C   Compute upper estimate of bound
      N = NN-1
      X = DEXP((DLOG(-PT(NN))-DLOG(PT(1)))/FLOAT(N))
      IF (PT(N) .EQ. 0.D0)  GO TO 20
C   If Newton step at the origin is better, use it
        XM = -PT(NN)/PT(N)
        IF (XM .LT. X)  X = XM
C   Chop the interval (0,X) until F<=0
20    XM = X*.1D0
      F = PT(1)
      DO 30 I=2,NN
        F = F*XM+PT(I)
30      CONTINUE
      IF (F .LE. 0.D0)  GO TO 40
        X = XM
        GO TO 20
40    DX = X
C   Do Newton iteration until X converges to two decimal places
50    IF (DABS(DX/X) .LE. .005D0)  GO TO 70
        Q(1) = PT(1)
        DO 60 I=2,NN
          Q(I) = Q(I-1)*X+PT(I)
60        CONTINUE
        F = Q(NN)
        DF = Q(1)
        DO 65 I=2,N
          DF = DF*X+Q(I)
65        CONTINUE
        DX = F/DF
        X = X-DX
        GO TO 50
70    CAUCHY = X
      RETURN
      END


      REAL*8 FUNCTION SKALE(NN,PT,ETA,INFIN,SMALNO,BASE)
C   Returns a scale factor to multiply the coefficients of the
C   polynomial.  The scaling is done to avoid overflow and to
C   avoid undetected underflow interfering with the convergence
C   criterion.  The factor is a power of the base.
C
C   PT - Modulus of the coefficients of P
C   ETA,INFIN,SMALNO,BASE - Constants describing the floating-
C                           point arithmetic
      REAL*8 PT(NN),ETA,INFIN,SMALNO,BASE,HI,LO,MAX,MIN,X,SC
      REAL*8 DSQRT,DLOG
C   Find largest and smallest moduli of coefficients
      HI = DSQRT(INFIN)
      LO = SMALNO/ETA
      MAX = 0.D0
      MIN = INFIN
      DO 10 I=1,NN
        X = PT(I)
        IF (X .GT. MAX)  MAX = X
        IF (X .NE. 0.D0 .AND. X .LT. MIN)  MIN = X
10      CONTINUE
C   Scale only if there are very large or very small components.
      SKALE = 1.D0
      IF (MIN .GE. LO .AND. MAX .LE. HI)  RETURN
      X = LO/MIN
      IF (X .GT. 1.D0)  GO TO 20
        SC = 1.D0/(DSQRT(MAX)*DSQRT(MIN))
        GO TO 30
20    SC = X
      IF (INFIN/SC .GT. MAX)  SC = 1.D0
30    L = DLOG(SC)/DLOG(BASE)+.5D0
      SKALE = BASE**L
      RETURN
      END


      SUBROUTINE CDIVID (AR,AI,BR,BI,CR,CI)
C   Complex division C=A/B, avoiding overflow.
      REAL*8 AR,AI,BR,BI,CR,CI,R,D,T,INFIN,DABS
      IF (BR .NE. 0.D0 .OR. BI .NE. 0.D0)  GO TO 10
C   Division by zero, C=infinity
        CALL MCON (T,INFIN,T,T)
        CR = INFIN
        CI = INFIN
        RETURN
10    IF (DABS(BR) .GE. DABS(BI))  GO TO 20
        R = BR/BI
        D = BI+R*BR
        CR = (AR*R+AI)/D
        CI = (AI*R-AR)/D
        RETURN
20    R = BI/BR
      D = BR+R*BI
      CR = (AR+AI*R)/D
      CI = (AI-AR*R)/D
      RETURN
      END


      REAL*8 FUNCTION CMOD(R,I)
C   Modulus of a complex number avoiding overflow
      REAL*8 R,I,AR,AI,DABS,DSQRT
      AR = DABS(R)
      AI = DABS(I)
      IF (AR .GE. AI)  GO TO 10
        CMOD = AI*DSQRT(1.D0+(AR/AI)**2)
        RETURN
10    IF (AR .LE. AI)  GO TO 20
        CMOD = AR*DSQRT(1.D0+(AI/AR)**2)
        RETURN
20    CMOD = AR*DSQRT(2.D0)
      RETURN
      END


      SUBROUTINE MCON (ETA,INFINY,SMALNO,BASE)
C   MCON provides machine constants used in various parts of the program.
C   The user may either set them directly or use the statements below to
C   compute them.  The meaning of the four constants are:
C
C   ETA    - Maximum relative representation error which can be described
C            as the smallest positive floating-point number such that
C            1.D0+ETA is greater than 1.D0
C   INFINY - The largest floating-point number
C   SMALNO - The smallest positive floating-point number
C   BASE   - The base of the floating-point number system used
C
C   Let T be the number of base digits in each floating-point number
C   (double precision).  Then ETA is either .5*B**(1-T) or B**(1-T)
C   depending on whether rounding or truncation is used.
C   Let M be the largest exponent and N the smallest exponent in the
C   number system.  Then INFINY is (1-BASE**(-T))*BASE**M and SMALNO
C   is BASE**N.
C   The values for BASE,T,M,N below correspond to VAX 11/780
      REAL*8 ETA,INFINY,SMALNO,BASE
      INTEGER*4 M,N,T
      BASE = 2.D0
      T = 56
      M = 127
      N = -127
      ETA = BASE**(1-T)
      INFINY = BASE*(1.D0-BASE**(-T))*BASE**(M-1)
      SMALNO = (BASE**(N+3))/BASE**3
      RETURN
      END

      SUBROUTINE EXPARAMS (IP,T,H,Z,AMP,DAMP,FREQ,PHASE,N)
#include <f77/lhdrsz.h>
C
C   Routine that produces the real amplitude, damping, frequency, and
C   phase from the complex amplitude and exponential parameter arrays.
C
C   Input Parameters:
C
C     IP    - Order of exponential model (integer)
C     T     - Real variable representing the sample interval (sec.)
C     H     - Array of complex amplitude parameters, H(1) to H(IP)
C     Z     - Array of complex exponential parameters, Z(1) to Z(IP)
C
C   Output Parameters:
C
C     AMP   - Array of real amplitudes (arbitrary units)
C     DAMP  - Array of real damping factors (1/sec)
C     FREQ  - Array of real frequencies (Hz)
C     PHASE - Array of real phase (radians)
C
C   Notes:
C
C     External arrays H,Z,AMP,DAMP,FREQ,PHASE must be dimensioned
C     .GE. IP in the calling program.    If the original data was
C     data was real, the phase reported represents that of a
C     cosine waveform.
C

      integer N

      COMPLEX H(N),Z(N)
      REAL AMP(SZLNHD),DAMP(SZLNHD),FREQ(SZLNHD),PHASE(SZLNHD)

      TWOPI=8.*ATAN(1.)
      DO 10 K=1,IP
        FREQ(K)=ATAN2(AIMAG(Z(K)),REAL(Z(K)))/(TWOPI*T)
        DAMP(K)=LOG(CABS(Z(K)))/T
        AMP(K)=CABS(H(K))
10      PHASE(K)=ATAN2(AIMAG(H(K)),REAL(H(K)))
      RETURN
      END

      SUBROUTINE ESD (METHOD,IP,T,H,Z,N,CFOUR,SD)
C
C   Routine to compute the one-sided or two-sided energy spectral
C   density of the exponential model at a given frequency F .
C
C   Input Parameters:
C
C     METHOD - Indicates desired ESD:  1 for one-sided or 2 for
C              two-sided (integer)
C     IP     - Order of the exponential model (integer)
C     T      - Real variable representing the sample interval (sec.)
C     F      - Real variable representing the fraction of the
C              sampling frequency (ranges from -.5 to .5)
C     H      - Array of complex amplitude parameters, H(1) to H(IP)
C     Z      - Array of complex exponential parameters, Z(1) to Z(IP)
C     N      - Number of points in the frequency domain
C
C   Output Parameters:
C
C     SD     - Real energy spectral density value at frequency F
C     CFOUR  - Complex spectrum, including both phase and amplitude
C
C
C   Notes:
C
C     External arrays H and Z must be dimensioned .GE. IP in the
C     calling program.
C
      COMPLEX H(N),Z(N),ZK,ZKC,ZI,SUM,CFOUR(N)
      REAL F
      TWOPI=8.*ATAN(1.)
      N1=(N/2)+1
      DO  50 I=1,N
      IF(I.LE.N1) F=FLOAT(I-1)/FLOAT(N)
      IF(I.GT.N1) F=FLOAT(I-N-1)/FLOAT(N)
      ZI=CEXP(CMPLX(0.,-TWOPI*F))
c     zi=cexp(cmplx(0.,twopi*f))
      SUM=(0.,0.)
      IF (METHOD .EQ. 2)  GO TO 20
      DO 10 K=1,IP
10      SUM=SUM+H(K)/(1.-Z(K)*ZI)
      GO TO 40
20    DO 30 K=1,IP
        ZK=Z(K)
        ZKC=1./CONJG(ZK)
30      SUM=SUM+(H(K)*(ZK-ZKC)*ZI)/(1.-(ZK+ZKC)*ZI+ZK*ZKC*ZI*ZI)
40    SD=T*(REAL(SUM)**2+AIMAG(SUM)**2)
      CFOUR(I)=SUM
50    CONTINUE
      RETURN
      END

      SUBROUTINE CHOLESKY (M,EPS,A,B,ISTAT)

C
C This program solves a Hermitian symmetric set of complex linear
C simultaneous equations using the Cholesky decomposition method.
C The solution replaces the original contents of array B. Contents
C of array A are destroyed after this routine is called.
C
C                    AX = B
C
C   Input Parameters:
C        
C      M   - Order of the matrix (number of linear equations)
C      EPS - Epsilon (quantity for testing loss of significance;
C            depends on machine precision; suggest 1.E-15)
C      A   - Array of complex matrix elements stored columnwise
C            (i.e., A(1,1) is stored as A(1), A(1,2) as A(2),
C            A(2,2) as A(3), etc.  Only the top triangular part
C            of the A matrix is stored since the other half is
C            obtained by Hermitian symmetry)
C      B   - Array of complex elements of right-hand-side vector
C
C
C   Output Parameters:
C
C      B   - Complex solution X vector stored in place of B vector
C      ISTAT - Integer status indicator at time of exit
C              0 for normal exit
C              -1 if matrix is singular
C              +K if there is loss of numerical significance or if
C                 a nonpositive-definite matrix detected at pivot K
C
C   Notes:
C
C     External array A must be dimensioned .GE. M(M+1)/2 and array B
C     must be dimensioned .GE. M in the calling program.
C
      COMPLEX A(1),B(1),SUM
C
C   Factor into triangular and diagonal form   !  Eq. (3.76)
C
      ISTAT=0
      KPIV=0
      DO 100 K=1,M
        KPIV=KPIV+K
        IND=KPIV
        LEND=K-1
        TINY=ABS(EPS*REAL(A(KPIV)))
        DO 100 I=K,M
          SUM=(0.,0.)
          IF (LEND .EQ. 0)  GO TO 40
          LPIV=KPIV
          DO 30 L=1,LEND
            LPIV=LPIV+L-K-1
30          SUM=SUM+REAL(A(LPIV))*A(IND-L)*CONJG(A(KPIV-L))
40        SUM=A(IND)-SUM
          IF (I .NE. K)  GO TO 80
C
C   Test for negative pivot element and loss of significance
C
          IF (REAL(SUM) .GT. TINY)  GO TO 90
          IF (REAL(SUM) .GT. 0.)  GO TO 70
          ISTAT=-1
          RETURN
70        IF (ISTAT .GT. 0)  GO TO 90
          ISTAT=K
90        A(KPIV)=CMPLX(REAL(SUM),0.)
          DPIV=1./REAL(SUM)
          GO TO 100
80        A(IND)=SUM*DPIV
100       IND=IND+I
C
C   Back solution for intermediate column vector solution  ! Eq. (3.74)
C
      KPIV=1
      DO 200 K=2,M
        KPIV=KPIV+K
        SUM=B(K)
        DO 210 J=1,K-1
210       SUM=SUM-B(K-J)*CONJG(A(KPIV-J))
200     B(K)=SUM
C
C   Back solution for final column vector solution    !  Eq. (3.75)
C
      KPIV=(M*(M+1))/2
      B(M)=B(M)/REAL(A(KPIV))
      DO 300 K=M,2,-1
        KPIV=KPIV-K
        IND=KPIV
        SUM=B(K-1)/REAL(A(KPIV))
        DO 310 J=K,M
          IND=IND+(J-1)
310       SUM=SUM-B(J)*A(IND)
300     B(K-1)=SUM
      RETURN
      END
C
      SUBROUTINE PREFFT (N, MODE, NEXP, W )
C
C   Input Parameters:
C
C     N     - Number of data samples to be processed (integer-must be a
C             power of two)
C     MODE  - Set to 0 for discrete-time Fourier series (Eq. 2.C.1) or
C             1 for inverse (Eq. 2.C.2)
C
C   Output Parameters:
C
C     NEXP  - Indicates power-of-2 exponent such that N=2**NEXP .
C             Will be set to -1 to indicate error condition if N
C             is not a power of 2 (this integer used by sub. FFT)
C     W     - Complex exponential array
C
C   Notes:
C
C     External array W must be dimensioned .GE. N by calling program.
C
#include <f77/lhdrsz.h>

      COMPLEX W(2*SZLNHD),C1,C2
      NEXP=1
5     NT=2**NEXP
      IF (NT .GE. N)  GO TO 10
      NEXP=NEXP+1
      GO TO 5
10    IF (NT .EQ. N)  GO TO 15
      NEXP=-1                 ! Error:  N is not a power of 2
      RETURN
15    S=8.*ATAN(1.)/FLOAT(NT)
      C1=CMPLX(COS(S),-SIN(S))
      IF (MODE .NE. 0)  C1=CONJG(C1)
      C2=(1.,0.)
      DO 20 K=1,NT
        W(K)=C2
20      C2=C2*C1
      RETURN
      END


      SUBROUTINE FFT ( N, MODE, T, NEXP, W, X )
C
C   Input Parameters:
C
C     N,MODE,NEXP,W - See parameter list for subroutine PREFFT
C     T             - Sample interval in seconds
C     X             - Array of N complex data samples, X(1) to X(N)
C
C   Output Parameters:
C
C     X - N complex transform values replace original data samples
C         indexed from k=1 to k=N, representing the frequencies
C         (k-1)/NT hertz
C
C   Notes:
C
C     External array X must be dimensioned .GE. N by calling program.
C
#include <f77/lhdrsz.h>

      integer N
      COMPLEX X(N),W(2*SZLNHD),C1,C2
      MM=1
      LL=N
      DO 70 K=1,NEXP
        NN=LL/2
        JJ=MM+1
        DO 40 I=1,N,LL
          KK=I+NN
          C1=X(I)+X(KK)
          X(KK)=X(I)-X(KK)
40        X(I)=C1
        IF (NN .EQ. 1) GO TO 70
        DO 60 J=2,NN
          C2=W(JJ)
          DO 50 I=J,N,LL
            KK=I+NN
            C1=X(I)+X(KK)
            X(KK)=(X(I)-X(KK))*C2
50          X(I)=C1
60        JJ=JJ+MM
        LL=NN
        MM=MM*2
70      CONTINUE
      NV2=N/2
      NM1=N-1
      J=1
      DO 90 I=1,NM1
        IF (I .GE. J)  GO TO 80
        C1=X(J)
        X(J)=X(I)
        X(I)=C1
80      K=NV2
85      IF (K .GE. J)  GO TO 90
        J=J-K
        K=K/2
        GO TO 85
90      J=J+K
      IF (MODE .EQ. 0)  S=T
      IF (MODE .NE. 0)  S=1./(T*FLOAT(N))
      DO 100 I=1,N
100     X(I)=X(I)*S
      RETURN
      END
C
      SUBROUTINE BURG ( N, IP, X, P, A, ISTAT )

#include <f77/lhdrsz.h>

C
C   Program to estimate the complex autoregressive parameters by
C   the Burg algorithm.
C
C   Input Parameters:
C
C     N  - Number of data samples
C     IP - Order of autoregressive process
C     X  - Array of complex data samples X(1) through X(N)
C
C   Output Parameters:
C
C     P  - Real variable representing driving noise variance
C     A  - Array of complex autoregressive parameters A(1) to A(IP)
C     ISTAT - Integer status indicator at time of exit
C             0 for normal exit
C             1 for numerical ill-conditioning (P < 0)
C
C   Notes:
C
C     External array X must be dimensioned .GE. N and array A
C     must be dimensioned .GE. IP in the calling program.
C     Internal arrays EF and EB must be dimensioned .GE. N .
C
      integer N, IP, ISTAT

      COMPLEX X(N), A(SZLNHD), EF(2*SZLNHD), EB(2*SZLNHD), NUM
      complex SAVE1, SAVE2

      double precision f11, f22, p, den, temp
C
C   Initialization
C
      ISTAT=0
      P=0.
      DO 10 J=1,N
10      P=P+REAL(X(J))**2+AIMAG(X(J))**2   ! Eq. (8.12)
      DEN=P*2.
      P=P/N
      PIN=P
      IF (IP .EQ. 0)  RETURN
      DO 20 J=1,N
        EF(J)=X(J)
20      EB(J)=X(J)                         ! Eq. (8.11)
      TEMP=1.
      K=0
C
C   Main recursion
C
100   K=K+1
      NUM=(0.,0.)
      DO 30 J=K+1,N
30      NUM=NUM+EF(J)*CONJG(EB(J-1))
      DEN=TEMP*DEN-REAL(EF(K))**2-AIMAG(EF(K))**2
     *    -REAL(EB(N))**2-AIMAG(EB(N))**2  ! Eq. (8.10)
      SAVE1=-2.*NUM/DEN                    ! Eq. (8.14)
      f11=real(save1)**2
      f22=aimag(save1)**2
c
c -- the Absoft compiler complained - try casting real - joe wade 3/18/98
c
c     f33=amin1(1.,(f11+f22))
      f33=amin1(1.,real(f11+f22))
c     TEMP=1.-REAL(SAVE1)**2-AIMAG(SAVE1)**2
      temp=1.-f33
c     if(temp.eq.0) temp=1.0e-10
      P=P*TEMP                             ! Eq. (8.4)
c     write(*,92) k,num,den,save1,temp,p
c     write(*,97) k,p
 97   format(2x,i5,2x,e12.6)
c     write(*,93) f11,f22,f11+f22
 93   format(3(1x,d24.18))
 92   format(1x,i2,7(1x,e10.4))
      IF (TEMP .GT. 0.)  GO TO 40
c     if(temp.eq.0.) go to 40
      if(temp.eq.0.) istat=0
      if(temp.eq.0.) go to 47
      ISTAT=1
47    RETURN
41    istat=0
      return
40    A(K)=SAVE1
      IF (K .EQ. 1)  GO TO 60
      KHALF=K/2
      DO 50 J=1,KHALF
        KJ=K-J
        SAVE2=A(J)
        A(J)=SAVE2+SAVE1*CONJG(A(KJ))      ! Eq. (8.2)
        IF (J .EQ. KJ)  GO TO 50
        A(KJ)=A(KJ)+SAVE1*CONJG(SAVE2)     ! Eq. (8.2)
50      CONTINUE
      if(temp.eq.0) return
60    IF (K .EQ. IP)  RETURN
      DO 70 J=N,K+1,-1
        SAVE2=EF(J)
        EF(J)=SAVE2+SAVE1*EB(J-1)          ! Eq. (8.7)
70      EB(J)=EB(J-1)+CONJG(SAVE1)*SAVE2
      if(p.lt.4.0e-11) go to 117
c     ratio=p/PIN
c     if(ratio.lt.1.0e-04) go to 117
      GO TO 100
 117  continue
      END
C
C
