C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine d3hsmg(a,iorda,iord,iord2,b)

c*************************************************************************
c     This subroutine fills in the symmetric matrix given by a column by
c     row multiplication of a two dimensional power series using a high
c     speed matrix generator.
c
c     Thompson, G.T., 1992, "The Grand Unified Theory of Least Squares",
c     Computers and Geosciences, Vol. 18, No. 7, pp. 815-822.
c
c**** Subroutine inputs
c
c     a = The vector of powers of one independent variable up to twice the
c         order of the matrix being filled.
c     iorda = The length of the vector, a.
c     iord = The order of the polynomial being tested.
c     iord2 = The number of coeficients in the polynomial being tested.
c
c**** Subroutine outputs
c
c     b = The symmetric matrix of normal equations.
c
c**** Subroutines required.
c
c     dmmove
c
c*************************************************************************

      integer iorda,iord,iord2,i,j,nmove,icol,iloc,ilocb,isum,iscan,im1
      integer iord1
      real*8 a(iorda),b(iord2 * iord2)

      iord1 = iord + 1
      icol = 1

      do i = 1,iord1

        im1 = i - 1

        do j = 1,i

          isum = 1

          do nmove = 1,iord1

            iscan = im1 + nmove
            iloc = iscan * (iscan - 1) / 2 + j
            ilocb = (icol - 1) * iord2 + isum
            call dmmove(nmove,a(iloc),b(ilocb))
            isum = isum + nmove

          enddo

          icol = icol + 1

        enddo

      enddo

      return
      end
      subroutine peval3(x,y,coef,iord,p)

c***********************************************************************
c     Evaluates the estimated value of a three dimensional polynomial
c     with two independent variables.
c
c**** Subroutine inputs.
c
c     x = The value of the first independent variable.
c     y = The value of the second independent variable.
c     coef = The array of polynomial coefficients.
c     iord = The order of the polynomial.
c
c**** Subroutine outputs.
c
c     p = The value of the polynomial at the point x,y.
c
c***********************************************************************

      integer i,iord,iord2,jstart,nj,nstep,j,ij
      real*8 x,y,p,prow
      real*8 coef((iord + 1) * (iord + 2) / 2)

      iord2 = (iord + 1) * (iord + 2) / 2
      jstart = iord2
      nj = 0
      p = coef(iord2)

      do i = iord,1,-1

        jstart = jstart - 1
        nj = nj + 1
        nstep = iord
        prow = coef(jstart)
        ij = jstart

        do j = nj,1,-1

          ij = ij - nstep
          prow = prow * x + coef(ij)
          nstep = nstep - 1

        enddo

        p = p * y + prow

      enddo

      return
      end
      subroutine rob3sb(x,y,v,nn,irtype,limord,coef,reg,ssr,ssd,sst,ndf1
     :,ndf2,ndf3,f1,minord,if1max,ifumax,ipoly,ilim,delta,ierr,itrial,sp
     :ace,isord)

c***********************************************************************
c     This subroutine computes the coefficients of either the least
c     square or the robust polynomials of orders 1 through 14 that fit a
c     set of three dimensional data points.
c
c     Further details can be found in
c
c     Davies, "Statistics and Data analysis in Geology", pages 322-352.
c
c     Beltrao, J.F., Silva, J.B.C., and Costa J.C., 1991, "Robust
c     Polynomial Fitting Method for Regional Gravity Estimation,"
c     Geophysics, vol.56,80-89.
c
c     Thompson, G.T., 1992, "The Grand Unified Theory of Least Squares,"
c     Computers and Geosciences, vol.18,n7,815-822
c
c**** Subroutine inputs
c
c     x(i) = The first independent variable .
c     y(i) = The second independent variable .
c     v(i) = The dependent variable.
c     nn = The number of data points.
c     irtype = Type of fitting algorithm flag
c              irtype = 0   Fitting is least squares
c              irtype = 1   Fitting is weighted robust.
c     limord = The maximum order of polynomial tested for fit.
c              0 <= limord <= 14
c              If limord = 0 the maximum order of fit is determined
c              internally, dependent on the number of data points and
c              the magnitude of the numbers.
c     ilim = The maximum number of iterations allowed to find the robust
c            fit
c     delta = The minimum  relative difference of the median between
c             iterations
c     isord = Order of fit for for constant order computation.
c             if isord = 0 the optimum order of fit is determined by a
c             search over all possible orders.
c
c**** Subroutine outputs
c
c     coef(680) = The polynomial coefficients.
c
c                 The number of coefficients of the polynomial of order I
c                 is given by:-
c
c                 ncof = (I + 1) * (I + 2) / 2
c
c                 The first coefficient of the polynomial of order, I is
c                 at index iloc in the coef array where:-
c
c                 iloc = I * (I + 1) * (I + 2) / 6
c
c     reg(14) = The regression coefficient of each order
c     ssr(14) = The variance of the regression for each order
c     ssd(14) = The variance of the residuals for each order
c     sst = The variance of the independent variable
c     ndf1(14) = Number of degrees of freedom of each regression
c     ndf2(14) = Number of degrees of freedom of each set of residuals
c     ndf3 = The total number of degrees of freedom
c     f1(14) = The f-test value of each order
c     minord = Minimum significant order of fit.
c     if1max = Order of fit with the maximum f-test value up to 10,000.
c     ifumax = Order of fit with the maximum f-test value.
c     ipoly = Recommended order of fit by the subroutine.
c     ierr = Error code on exit from reading data
c            ierr = 0  Normal completion.
c            ierr = 1  Insufficient data points.
c            ierr = 2  No significant polynomial relationship between
c                      the variables.
c            ierr = 3  The selected constant order of fit, isord is greater
c                      than the maximum allowed for the input data set.
c     itrial(14) = The number of iterations taken at each order of fit.
c                  itrial(i) = 1 means the solution is equivalent to
c                  least square.
c     space = Working space of 435 + 2 * nn elements.
c
c**** Partition of working space
c
c     space(1 to 435) = The array of powers of the polynomial of twice the
c                       maximum order
c     space(436 to 436 + nn) = The array of weights
c     space(436 to 436 + nn) = The array absolute residuals (uses the same
c                              memory locations as the weights)
c     space(436 to 436 + nn * 2) = The array of residuals
c
c**** Subroutines required
c
c     d3hsmg
c     dhsort
c     dcmpwt
c     dcrout
c     dmdian
c     dmxadd
c     dmmove
c     dordr2
c     dscalm
c     dsdev
c     dzero
c     p3ind
c     peval3
c     pncoef
c     pordmx
c     tsig01
c
c***********************************************************************

      integer nn,ndim,iord,iord4,iord2,iordw,iordw2,iordw4,if1max,minord
      real*8 x(nn),y(nn),v(nn),space(435 + 2 * nn)
      real*8 awa(14520),coef(680),apwr(435),b(120),cold(120)
      real*8 ssr(14),ssd(14),reg(14),f1(14)
      real*8 syc(14),syyc(14),ssri(14),amsr(14),amsd(14)
      real*8 amsri(14),r2(14),fi(14)
      real*8 maxin,pv,s,sold,delta,sd,sdest,dephuge,depmrsp
      real*8 sy,syy,sst,f1max,infity,reglim,varlim,r2max,fumax
      real*8 wmean
      integer ndf1(14),ndf2(14),iflag(14),ndf1i(14),itrial(14)
      integer maxord,ndf3,itest,iftest,iopt,irtype,ir2max,ifumax
      integer i,nmove,ilim,iordf,iordf2
      integer iordh,iord2h,iord4h
      integer iw,ires,iabsr,iloc,isord,ifirst

      if(nn.ge.3) then

        ierr = 0

        else

        ierr = 1
        return

      endif

c***********************************************************************
c     Initialise variables
c***********************************************************************

      varlim = 1.0d-2
      sdest = 0.6745d0
      maxin = 0.0d0
      r2max = 0.0d0
      sy = 0.0d0
      syy = 0.0d0
      f1max = 0.0d0
      fumax = 0.0d0
      reglim = dsqrt(0.5d0)
      infity = dephuge(maxin)
      call dzero(14,syc)
      call dzero(14,syyc)
      iftest = 0
      if1max = 0
      ifumax = 0
      ir2max = 0
      minord = 0
      ndim = 3
      iopt = -1
      iw = 435
      ires = 435 + nn
      iabsr = iw
      ndf3 = nn - 1
      ifirst = 1
      isord = iabs(isord)

c***********************************************************************
c     Calculate the sum and sum of squares of the dependent variable.
c***********************************************************************

      do i = 1,nn

        maxin = dmax1(dabs(maxin),dabs(x(i)),dabs(y(i)),dabs(v(i)))
        sy = sy + v(i)
        syy = syy + v(i) * * 2

      enddo

      sst = syy - sy * sy / dble(nn)

c***********************************************************************
c     Determine the maximum order of fit that can be made.
c***********************************************************************

      call pordmx(iopt,ndim,nn,maxin,iord,iord2,iord4,ierr)
      if(isord.gt.iord) ierr = 3
      if(ierr.ne.0) return

      if(isord.ne.0) then
         
         iord = isord
         ifirst = isord
         call pncoef(ndim,iord,iord2,iord4)
         
         else if(limord.ne.0.and.limord.lt.iord) then
         
         iord = limord
         
      endif

c***********************************************************************
c     Calculate coefficients and statistics for each order of fit up to
c     the maximum.
c***********************************************************************

      do iordw = ifirst,iord

        itrial(iordw) = 1
        iflag(iordw) = 0
        iloc = iordw * (iordw + 1) * (iordw + 2) / 6

c***********************************************************************
c     Initialise the weight matrix with unit values
c***********************************************************************

        if(irtype.eq.1.or.iordw.eq.1) then

          do i = 1,nn

            space(iw + i) = 1.0d0

          enddo

          wmean = 1.0d0

        endif

c***********************************************************************
c     Determine the parameters of the matrix of normal equations
c***********************************************************************

        call pncoef(ndim,iordw,iordw2,iordw4)

        if(irtype.eq.0.or.ilim.eq.1) then

          iordf = iord
          iordf2 = iord2

          else

          iordf = iordw
          iordf2 = iordw2

        endif

        iordh = 2 * iordf
        call pncoef(ndim,iordh,iord2h,iord4h)

c***********************************************************************
c     Enter the iteraive loop for coefficient calculation. (only 1.0d0
c     pass through for the least squares option).
c***********************************************************************

        do while(itrial(iordw).le.ilim)

          if(irtype.eq.1.or.iordw.eq.1) then

            call dzero(iordf2,b)
            call dzero(iord2h,apwr)

            do i = 1,nn

c***********************************************************************
c     Calculate the powers of the independent variables up to twice the
c     current order of fit.
c***********************************************************************

              call p3ind(x(i),y(i),iordh,space)

c***********************************************************************
c     Scale the powers by the weight of the current data point and then
c     accumulate the sum.
c***********************************************************************

              call dscalm(space,iord2h,space(iw + i))
              call dmxadd(apwr,space,apwr,iord2h,1)

c***********************************************************************
c     Accumulate the sum of right hand sides of the normal equations.
c***********************************************************************

              if(v(i).eq.0.0d0) then

                call dzero(iordf2,space)

                else

                call dscalm(space,iordf2,v(i))

              endif

              call dmxadd(space,b,b,iordf2,1)

            enddo

          endif

c***********************************************************************
c     Form the matrix of normal equations using a high speed matrix
c     generator.
c***********************************************************************

          call d3hsmg(apwr,iord2h,iordw,iordw2,awa)
          call dmmove(iordw2,b,awa(iordw4 + 1))

c***********************************************************************
c     Solve the least squares equations.
c***********************************************************************

          call dcrout(iordw2,iordw2 + 1,awa,awa,1,cold)

c***********************************************************************
c     Compute residuals for weight calculation of robust fit, for least
c     squares fit (irtype = 0) compute statistics of fit.
c***********************************************************************

          if(irtype.eq.0.or.ilim.eq.1) then

            call dmmove(iordw2,cold,coef(iloc))
            goto 1

          endif

          do i = 1,nn

            call peval3(x(i),y(i),cold,iordw,pv)
            space(ires + i) = v(i) - pv
            space(iabsr + i) = dabs(space(ires + i))

          enddo

c***********************************************************************
c     Determine the median value of absr(i)
c***********************************************************************

          call dmdian(nn,space(iabsr + 1),s)

c***********************************************************************
c     Check exit criteria.
c***********************************************************************

          if(s.eq.0.0d0) then

            call dmmove(iordw2,cold,coef(iloc))
            sd = 0.0d0
            goto 1

          endif

          if(itrial(iordw).gt.1) then

            if(s.gt.sold) then

              itrial(iordw) = itrial(iordw) - 1
              sd = sold / sdest
              goto 1

              else if((dabs((s - sold) / sold).le.delta).or.itrial(iordw
     :).eq.ilim) then

              call dmmove(iordw2,cold,coef(iloc))
              sd = s / sdest
              goto 1

            endif

          endif

c***********************************************************************
c     Update the weights
c***********************************************************************

          sd = s / sdest
          call dcmpwt(nn,space(ires + 1),sd,space(iw + 1))
          call dxmean(space(iw + 1),nn,wmean)

          if(dabs(wmean).lt.0.25d0) then

              itrial(iordw) = itrial(iordw) - 1
              sd = sold / sdest
              goto 1

          endif

c***********************************************************************
c     Set the values of x for the next iteration.
c***********************************************************************

          sold = s
          call dmmove(iordw2,cold,coef(iloc))
          itrial(iordw) = itrial(iordw) + 1

        enddo

c***********************************************************************
c     Calculate error measures for statistical analysis
c***********************************************************************

    1   ndf1(iordw) = iordw2 - 1
        ndf2(iordw) = nn - iordw2

c***********************************************************************
c     Compute the sum of squares of the regression and the deviations
c***********************************************************************

        if(irtype.eq.0.or.itrial(iordw).eq.1) then

c***********************************************************************
c     Calculate the estimated value and residual for each observation
c***********************************************************************

          do i = 1,nn

            call peval3(x(i),y(i),coef(iloc),iordw,pv)
            space(ires + i) = v(i) - pv
            syc(iordw) = syc(iordw) + space(ires + i)
            syyc(iordw) = syyc(iordw) + space(ires + i) * * 2

          enddo

          ssd(iordw) = syyc(iordw) - syc(iordw) * syc(iordw) / dble(nn)

          else

          ssd(iordw) = sd * sd * dble(ndf2(iordw))

        endif

        ssr(iordw) = sst - ssd(iordw)
        amsr(iordw) = ssr(iordw) / dble(ndf1(iordw))

c***********************************************************************
c     Test if the order of the regression is the maximum possible with
c     the input number of data points.
c***********************************************************************

        if(nn.eq.iordw2) then

          amsd(iordw) = 0.0d0
          r2(iordw) = 1.0d0
          reg(iordw) = 1.0d0
          iflag(iordw) = 1
          itest = 0
          f1(iordw) = 0.0d0
          goto 2

        endif

c***********************************************************************
c     If ssr(iordw) < 0.0 then the fit is not significant
c     the input number of data points.
c***********************************************************************

        if(ssr(iordw).lt.0.0d0) then

          amsd(iordw) = 0.0d0
          r2(iordw) = 0.0d0
          reg(iordw) = 0.0d0
          iflag(iordw) = 2
          itest = 0
          f1(iordw) = 0.0d0
          goto 2

        endif

c***********************************************************************
c     Compute corrected sum of squares of the deviations and the
c     goodness of fit
c***********************************************************************

        amsd(iordw) = ssd(iordw) / dble(ndf2(iordw))
        r2(iordw) = ssr(iordw) / sst

c***********************************************************************
c     Verify the stability of the solution within the limits of
c     truncation error of the solution.
c***********************************************************************

        if(r2(iordw) - 1.0d0.gt.dble(nn) * depmrsp(sst)) then

          if(iordw.eq.1) minord = 0
          iflag(iordw) = 2
          goto 3

        endif

c***********************************************************************
c     Compute the regression coefficient.
c***********************************************************************

        if(r2(iordw).gt.1.0d0) then

          r2(iordw) = 1.0d0
          amsd(iordw) = 0.0d0

        endif

        if(r2(iordw).gt.0.0d0) reg(iordw) = dsqrt(r2(iordw))

c***********************************************************************
c     Test the regression for statistical significance
c***********************************************************************

        if(amsd(iordw).eq.0.0d0) then

          itest = 1
          f1(iordw) = infity
          fumax = infity
          ifumax = iordw

          if(f1max.lt.1.0d4) then

            f1max = f1(iordw)
            if1max = iordw

          endif

          r2max = r2(iordw)
          ir2max = iordw

          else

          f1(iordw) = amsr(iordw) / amsd(iordw)
          call tsig01(ndf1(iordw),ndf2(iordw),f1(iordw),itest)

        endif

    2   sd = dsqrt(amsd(iordw))

        if(itest.eq.0) then

          if(iftest.eq.0) minord = 0

          if(iflag(iordw).eq.1.and.minord.eq.0) then

            ipoly = 0
            return

          endif

          else

c***********************************************************************
c     Test to see if at least 1.0d0 order is significant (iftest = 0 if a
c     significant order has not been found). If this is the first
c     significant order set minord = iordw and flag iftest = 1.
c***********************************************************************

          if(iftest.eq.0) minord = iordw
          iftest = 1

c***********************************************************************
c     Reset the maximum F-test value that has been found up to 10000
c***********************************************************************

          if(f1max.lt.f1(iordw).and.f1max.lt.1.0d4) then

            f1max = f1(iordw)
            if1max = iordw

          endif

c***********************************************************************
c     Reset the maximum F-test value that has been found.
c***********************************************************************

          if(fumax.lt.f1(iordw)) then

            fumax = f1(iordw)
            ifumax = iordw

          endif

c***********************************************************************
c     Reset the maximum R2 value that has been found.
c***********************************************************************

          if(r2max.lt.r2(iordw)) then

            r2max = r2(iordw)
            ir2max = iordw

          endif

        endif

        if((iflag(iordw).eq.0).and.(itest.eq.0.and.iftest.ne.0).or.f1(io
     :rdw).ge.infity.or.r2(iordw).eq.1.0d0.or.(r2(iordw).lt.r2max.and.r2
     :max.gt.0.0d0.and.irtype.eq.0)) goto 3

      enddo

c***********************************************************************
c     On exit from the loop iordw is 1 greater than the maximum so reset
c     iordw and search from if1max to iordw for the order above which
c     the polynomial is varying according to noise, (r2 varies slowly)
c***********************************************************************

    3 if(iordw.eq.0.or.minord.eq.0) then

        ierr = 2
        return

      endif

      if(f1(iordw).lt.infity) iordw = iordw - 1

      if(minord.eq.0) then

        ipoly = 0
        return

      endif

c***********************************************************************
c     Determine the best order of fit to occur when there is negligible
c     improvement in the goodness of fit above the maximum F-test.
c***********************************************************************

      nmove = ir2max - if1max + 1

      if(nmove.le.1) then

        ipoly = if1max

        else

        call dmmove(nmove,r2(if1max),space)
        call dordr2(space,nmove,varlim,ipoly)
        ipoly = ipoly + if1max - 1

      endif

c***********************************************************************
c     Compare the significant orders of fit from the minimum to the best
c***********************************************************************

      maxord = minord
      if(ipoly.le.1) return

      do  i = minord + 1,ipoly

        if(iflag(maxord).gt.0.or.iflag(i).gt.0) then

c***********************************************************************
c     Compute the ANOVA table for the comparison between orders and test
c     for significance.
c***********************************************************************

          ssri(maxord) = ssr(i) - ssr(maxord)
          ndf1i(maxord) = ndf1(i) - ndf1(maxord)
          amsri(maxord) = ssri(maxord) / dble(ndf1i(maxord))

          if(amsd(i).eq.0.0d0) then

            itest = 1
            fi(maxord) = infity

            else

            fi(maxord) = amsri(maxord) / amsd(i)
            call tsig01(ndf1i(maxord),ndf2(i),fi(maxord),itest)

          endif

          if(itest.ne.0) maxord = i

        endif

      enddo

      itest = 0
      if(iflag(ipoly).ne.0.or.iflag(maxord).ne.0.or.(ipoly - maxord).lt.
     :2) return

c***********************************************************************
c     Compare the the best order of fit and the order at which no
c     significant improvement occurs.
c***********************************************************************

      ssri(maxord) = ssr(ipoly) - ssr(maxord)
      ndf1i(maxord) = ndf1(ipoly) - ndf1(maxord)
      amsri(maxord) = ssri(maxord) / dble(ndf1i(maxord))
      fi(maxord) = amsri(maxord) / amsd(ipoly)
      call tsig01(ndf1i(maxord),ndf2(ipoly),fi(maxord),itest)
      if(itest.eq.0) ipoly = min0(ipoly,maxord)
      return
      end
