/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/* ---------------------------------------------------------------------------

	Package	:	sisio
	File	:	lbopen.c
	Author	:	Dean Kopesky
	Company	:	Minnesota Supercomputer Center, Inc.


	LBOPEN opens a file with a given name and access mode, and
	assigns a unit number to it.  Unit number 0 is preconnected
	to standard input; unit number 1 is preconnected to standard
	output.  

	If the file name is "/dev/null", LBOPEN sets the doio field to
	FALSE so that RTAPE and WRTAPE won't do anything, and sets the
	isdisk field to TRUE (even though /dev/null is a character
	special device).

	FORTRAN:

	CALL LBOPEN( unit, name, mode )

	unit	output	integer	Logical unit number assigned.
	name	input	charact	File name.
	mode	input	charact	Access mode.

	C:

	lbopen( unit, name, mode );

	unit	output	int *	Logical unit number assigned.
	name	input	char *	File name.
	mode	input	char *	Access mode.


	$Header: /m/s1/dmk/xlate/sisio/RCS/lbopen.c,v 4.3 91/09/11 08:47:28 dmk Exp $

	$Log:	lbopen.c,v $
	Revision 4.3  91/09/11  08:47:28  dmk
	MODSET: 4.3
	Changed entry point names to macros.  Tests for STRLEN_IS_ARG, not sun.
	
	Revision 4.1  90/07/13  09:17:06  dmk
	MODSET: 4.1
	Added code to store the filename in the FIT.  Changed SUN to sun.
	
	Revision 3.1  89/09/27  09:54:15  dmk
	MODSET: 3.1
	Added C entry point.
	
	Revision 2.3  88/08/16  12:45:05  dmk
	MODSET: 2.3
	Added the namlen and modlen parameters to fix a bug on the SUN version.
	Also, changed the flags set by the "a", "a+" and "w" modes.
	
	Revision 2.2  88/07/20  08:52:38  dmk
	MODSET: 2.2
	If file name is "/dev/null", will no longer test to see if it's a disk
	file or pipe; just sets .isdisk to TRUE and .doio to FALSE.
	
	Revision 2.1  88/07/12  15:20:32  dmk
	MODSET: 2.1
	LBOPEN picks unit numbers for you now.  Can no longer open stdin and
	stdout by using a null file name.  No longer _fitinit's in case of an
	error.  Added lower case entry point for SUN.  Sets cwlen to WORDSIZE
	instead of 8.
	
	Revision 1.1  88/06/27  11:53:16  dmk
	Initial revision
	
--------------------------------------------------------------------------- */


#ifdef	CRAY
#include	<fortran.h>
#endif

#include	<fcntl.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	"sisio.h"

#define	MODULE	"LBOPEN"

#define	STRLEN	128
#define PERM	0644


#ifdef	CRAY
void	F_LBOPEN( unit, name, mode )

	int *	unit;
	_fcd	name;
	_fcd	mode;
#endif
#ifdef	STRLEN_IS_ARG
void	F_LBOPEN( unit, name, mode, namlen, modlen )

	int *	unit;
	char *	name;
	char *	mode;
	int	namlen;
	int	modlen;
#endif

{
	char	cname[ STRLEN ];
	char	cmode[ STRLEN ];

	/*
		Translate filename and access mode to C character pointers.
	*/

#ifdef	CRAY
	if ( _strf2c( name, cname, STRLEN ) <= 0 )
#endif
#ifdef	STRLEN_IS_ARG
	if ( _strf2c( name, namlen, cname, STRLEN ) <= 0 )
#endif
	{
		_error( MODULE, *unit, E_BADNAME );
		return;
	}

#ifdef	CRAY
	if ( _strf2c( mode, cmode, STRLEN ) <= 0 )
#endif
#ifdef	STRLEN_IS_ARG
	if ( _strf2c( mode, modlen, cmode, STRLEN ) <= 0 )
#endif
	{
		_error( MODULE, *unit, E_BADMODE );
		return;
	}

	/*
		Call the C version.
	*/

	C_LBOPEN( unit, cname, cmode );
}


void	C_LBOPEN( unit, name, mode )

	int *	unit;
	char *	name;
	char *	mode;

{
	int	md, fd, isdisk;

	/*
		Initialize sisio iff necessary.
	*/

	INITIALIZE;

	/*
		Select an unopen, error-free unit number.
	*/

	*unit = 2;

	while ( ( _sisfit[ *unit ].isopen || _sisfit[ *unit ].error )
			&& *unit < MAXUNIT )
		*unit += 1;

	if ( _sisfit[ *unit ].isopen || _sisfit[ *unit ].error )
	{
		_error( MODULE, -1, E_NOUNITS );
		return;
	}

	/*
		Store the file name in the FIT.
	*/

	_sisfit[ *unit ].filename = malloc( (unsigned) strlen( name ) + 1 );
	(void) strcpy( _sisfit[ *unit ].filename, name );

	/*
		Translate the access mode.
	*/

	_strlower( mode );

	if( ! strcmp( mode, "r" ) )
		md = O_RDONLY;
	else if( ! strcmp( mode, "w" ) )
		md = O_WRONLY | O_CREAT | O_TRUNC;
	else if( ! strcmp( mode, "a" ) )
		md = O_WRONLY | O_CREAT;
	else if( ! strcmp( mode, "r+" ) )
		md = O_RDWR;
	else if( ! strcmp( mode, "w+" ) )
		md = O_RDWR | O_CREAT | O_TRUNC;
	else if( ! strcmp( mode, "a+" ) )
		md = O_RDWR | O_CREAT;
	else
	{
		_error( MODULE, *unit, E_BADMODE );
		return;
	}

	/*
		Open the file.
	*/

	fd = open( name, md, PERM );
	if( fd < 0 )
	{
		_error( MODULE, *unit, E_OPEN );
		return;
	}
	_sisfit[ *unit ].filedes = fd;

	/*
		See if the file is a pipe, unless it's /dev/null.
	*/

	if ( ! strcmp( name, "/dev/null" ) )
	{
		_sisfit[ *unit ].doio = FALSE;
		_sisfit[ *unit ].isdisk = TRUE;
	}
	else
	{
		isdisk = _isdisk( _sisfit[ *unit ].filedes );
		if ( isdisk == -1 )
		{
			_error( MODULE, *unit, E_NOTFILE );
			( void ) close( _sisfit[ *unit ].filedes );
			_sisfit[ *unit ].filedes = 0;
			return;
		}
		_sisfit[ *unit ].isdisk = isdisk;
	}

	/*
		Set the control word length.
	*/

	if ( _sisfit[ *unit ].trwhen == W_ALWAYS || ( _sisfit[ *unit ].trwhen
			== W_IFDISK && _sisfit[ *unit ].isdisk ) )
		_sisfit[ *unit ].cwlen = 4;
	else
		_sisfit[ *unit ].cwlen = WORDSIZE;

	/*
		Allocate the buffer.
	*/

	if ( _sisfit[ *unit ].largebuf )
	{
		_sisfit[ *unit ].buffer = malloc( LGBUFSIZE );
		_sisfit[ *unit ].bufsize = LGBUFSIZE;
	}
	else
	{
		_sisfit[ *unit ].buffer = malloc( SMBUFSIZE );
		_sisfit[ *unit ].bufsize = SMBUFSIZE;
	}

	if ( _sisfit[ *unit ].buffer == NULL )
	{
		_error( MODULE, *unit, E_NOBUFFER );
		( void ) close( _sisfit[ *unit ].filedes );
		_sisfit[ *unit ].filedes = 0;
		_sisfit[ *unit ].doio = TRUE;
		return;
	}

	_sisfit[ *unit ].isopen = TRUE;
}
