C*****  AMX002  COMPLEX FFT (MIXED-RADIX)       REV 1.0     JAN 88   C
C
C  PURPOSE:
C       THIS IS A NESTED VERSION OF THE MIXED-RADIX COMPLEX FFT WHICH
C       COMPUTES A FORWARD OR INVERSE COMPLEX IN-PLACE FFT WHERE
C       THE LENGTH OF THE ARRAY IS FACTORABLE BY 2, 3, 5, AND/OR 7.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       ORIGINAL                DEC 86          D.R. BENUA, QTC
C       MATH ADVANTAGE VERSION  JAN 88          R.D. COLEMAN, QTC
C          CHANGED SUBROUTINE NAMES AND REMOVED SCALING.
C       INSERT NESTING          JAN 88          P.G. CROSE, QTC
C          REVISION OF AMX002
C
C  CALLING FORMAT:
C       CALL AMX002 (A, B, INCE, INCV, N, NV, IFLG, NFAC, M, CNST,
C      &             STABLE,CTABLE, ISRC, IDST, NPERM,
C      &             AK1, AK2, AK3, AK4, AK5, AK6, AK7,
C      &             BK1, BK2, BK3, BK4, BK5, BK6, BK7 )
C      &                      AA, BB, AJ, BJ )
C
C  PARAMETERS:
C
C       A       REAL INPUT/OUTPUT VECTOR OF LENGTH N
C               REAL ARRAY ELEMENTS.
C
C       B       REAL INPUT/OUTPUT VECTOR OF LENGTH N
C               IMAGINARY ARRAY ELEMENTS.
C
C       INCE    INTEGER INPUT SCALAR
C               STRIDE BETWEEN ELEMENTS OF SUBVECTORS A AND B.
C
C       INCV    INTEGER INPUT SCALAR
C               STRIDE BETWEEN SUBVECTORS OF A AND B.
C
C       N       INTEGER INPUT SCALAR
C               LENGTH OF SUBVECTORS OF A, B, STABLE, AND
C               CTABLE.
C
C       NV      INTEGER INPUT SCALAR
C               NUMBER OF SUBVECTORS IN A, B, STABLE, AND
C               CTABLE.
C
C       IFLG    INTEGER INPUT SCALAR
C               TRANSFORM DIRECTION FLAG, POSITIVE
C               FOR FORWARD, NEGATIVE FOR INVERSE.
C
C       NFAC    INTEGER INPUT ARRAY OF LENGTH M
C               PRIME FACTORS OF N.
C
C       M       INTEGER INPUT SCALAR
C               NUMBER OF PRIME FACTORS IN ARRAY NFAC.
C
C       CNST    REAL INPUT ARRAY OF LENGTH 11
C               CONSTANTS, DEPENDENT ON IFLG.
C
C       STABLE  REAL INPUT ARRAY OF LENGTH N
C               SINE TWIDDLE FACTORS.
C
C       CTABLE  REAL INPUT ARRAY OF LENGTH N
C               COSINE TWIDDLE FACTORS.
C
C       ISRC    INTEGER INPUT ARRAY OF LENGTH NPERM
C               PERMUTATION INDICIES.
C
C       IDST    INTEGER INPUT ARRAY OF LENGTH NPERM
C               PERMUTATION INDICIES.
C
C       NPERM   INTEGER INPUT SCALAR
C               NUMBER OF ELEMENTS IN ARRAYS ISRC
C               AND IDST, ALWAYS LESS THAN N.
C
C  DESCRIPTION:
C       THIS ROUTINE COMPUTES THE FORWARD OR INVERSE FAST FOURIER
C       TRANSFORM OF NV COMPLEX SUBVECTORS. THE LENGTH OF THE SUBVECTOR,
C       N, MUST BE FACTORABLE BY THE PRIME FACTORS 2, 3, 5, AND/OR
C       7.  THE TRANFORM IS PERFORMED IN PLACE USING PRE-COMPUTED
C       SINE, COSINE, FACTOR, CONSTANT, AND PERMUTATION TABLES.
C       THE TABLES ARE UNIQUE FOR EACH VALUE OF N, INC, AND IFLG.
C
C       THE ALGORITHM USED HERE IS ADAPTED FROM THE MIXED RADIX
C       FFT PROGRAM BY R. C. SINGLETON PUBLISHED IN "PROGRAMS FOR
C       DIGITAL SIGNAL PROCESSING", IEEE PRESS, 1979, SEC. 1.4.
C       THE PRIMARY DIFFERENCES ARE THAT THE PRIME FACTORS ARE
C       RESTRICTED TO 2, 3, 5, AND 7, SINE AND COSINE TABLES ARE
C       USED, A PERMUTATION TABLE IS USED, AND THE ORDER OF THE
C       RESULT ARRAY IS FLIPED ABOUT THE N/2 POINT. (EG. FOR A
C       FORWARD TRANFORM, THE POSITIVE FREQUENCY COEFFICIENTS
C       APPEAR IN THE FIRST HALF OF THE ARRAY RATHER THAN IN THE
C       LAST HALF AS THEY DO IN THE SINGLETON ALGORITHM)  THE
C       RESULTS PRODUCED BY THIS ROUTINE FOLLOW THE ORDER DEFINED
C       IN THE FORMAL DEFINITION OF THE DISCRETE FOURIER TRANSFORM
C       GIVEN BY MOST AUTHORS.
C
C       THE DESIGN STRATEGY FOR THIS ROUTINE ASSUMES THAT IT WILL
C       BE CALLED MANY TIMES WITH THE SAME VALUES OF N, INC, AND
C       IFLG, SO THAT THE OVERHEAD ASSOCIATED WITH CREATING THE
C       TABLES IS INSIGNIFICANT.  SEE THE ROUTINE AMX001.FOR FOR
C       THE TABLE INITIALIZATION ALGORITHMS.  WHEN SETTING UP FOR
C       AN INVERSE TRANSFORM AFTER DOING A NUMBER OF FORWARD
C       TRANSFORMS WITH THE SAME N AND INC, IT IS ONLY NECESSARY
C       TO CHANGE THE SIGNS OF THE VALUES IN STABLE AND SOME OF
C       THE CONSTANTS IN CNST.
C
C       THE CODE IN THESE SUBROUTINES WAS DESIGNED TO PROVIDE A CLEAR
C       MODEL FOR A TRANSLATION INTO MICROCODE OR ASSEMBLY LANGUAGE.
C       THE AUTHOR DOES NOT CLAIM THAT THIS IS THE BEST WAY TO
C       IMPLEMENT THE ALGORITHM FOR A GENERAL FORTRAN MACHINE.
C       THE DESIGN ALSO ASSUMES THAT A LARGE CACHE OR FAST MEMORY
C       IS AVAILABLE TO STORE FULL LENGTH SINE AND COSINE TABLES AND
C       PERMUTATION VECTORS. OBVIOUSLY, MEMORY REQUIREMENTS FOR THE
C       TWIDDLE FACTORS TABLES COULD BE REDUCED, BUT ONLY AT THE
C       EXPENSE OF INCREASED COMPUTATION TO DERIVE THEM AT RUN TIME.
C       REGARDING THE PERMUTATION VECTORS, NOTE THAT IN THE SORTING
C       LOOP AT THE END OF THIS ROUTINE THERE IS A CONDITIONAL TEST.
C       THE RESULT OF THIS TEST DEPENDS ONLY ON THE PRECOMPUTED
C       VALUES IN THE ARRAYS ISRC AND IDST.  IF IT PROVES MORE
C       EFFICIENT ON THE TARGET MACHINE, THE LOGICAL VALUE OF THE
C       TEST COULD BE ENCODED IN A THIRD PERMUTATION VECTOR, OR IN
C       THE HIGH BITS OF THE PRESENT PERMUTATION VECTORS.
C
C       A SET OF 18 WORKING VECTOR ARRAYS ARE INCLUDED IN THE ARGUMENT
C       LIST TO ALLOW FOR OPTIMIZING THE VECTOR EQUATIONS. THIS MANY
C       WORKING VECTORS ARE REQUIRED FOR THE RADIX 7 IMPLEMENTATION.
C
C  SUBPROGRAMS CALLED:
C       NONE
C
C  ERROR CONDITIONS:
C       NONE
C
C-----------------------------------------------------------------------
C
        SUBROUTINE AMX002 (A, B, INCE, INCV, N, NV, IFLG, NFAC, M, CNST,
     &                      STABLE,CTABLE, ISRC, IDST, NPERM,
     &                      AK1, AK2, AK3, AK4, AK5, AK6, AK7,
     &                      BK1, BK2, BK3, BK4, BK5, BK6, BK7,
     &                      AA, BB, AJ, BJ )
 
 
        REAL A(1), B(1), CNST(1), STABLE(1), CTABLE(1)
        REAL AK1(1), AK2(1), AK3(1), AK4(1), AK5(1), AK6(1), AK7(1)
        REAL BK1(1), BK2(1), BK3(1), BK4(1), BK5(1), BK6(1), BK7(1)
        REAL AA(1), BB(1), AJ(1), BJ(1)
        INTEGER INCE, INCV, N, NV, IFLG, NFAC(1), M, ISRC(1), IDST(1)
        INTEGER NPERM
C
C        LOCAL VARIABLES
C
        INTEGER NT, KSPAN, KSPNN, NN, I, KK, K1, K2, K3, K4, K5, K6
        INTEGER INCD,THETA1,THETA2, K
        INTEGER IL, LKK, LK1, LK2, LK3, LK4, LK5, LK6
        REAL C1, C2, S1, S2
C
C-----------------------------------------------------------------------
C
        IF (N .LT. 2 .OR. M .LE. 0) RETURN
C
C---------------------------------
C  Initialize indexing variables
C---------------------------------
C
        NT = INCE*N
        KSPAN = NT
        NN = NT - INCE
        I = 0
        INCD = 1
C
C-------------------------------------
C  Top of the main loop, one pass of
C  this loop for each factor in NFAC
C-------------------------------------
C
20      KK = 1
        IF (I .GT. 0) INCD = INCD * NFAC(I)
        I = I + 1
        K = NFAC(I)
        KSPNN = KSPAN
        KSPAN = KSPAN/K
C
C------------------------------------------------------------------
C  Radix 2 pass, this code includes twiddle factor multiplications.
C  Note that there are two loops here, the first one is used when
C  there are no twiddle factors.
C------------------------------------------------------------------
C
        IF (K .NE. 2) GOTO 80
        K1 = KSPAN + 2
 
  30    K2 = KK + KSPAN
        LK2 = K2
 
        DO 31 IL = 1, NV
            AK1(IL) = A(LK2)
            BK1(IL) = B(LK2)
            LK2 = LK2 + INCV
  31    CONTINUE
 
        LKK = KK
        LK2 = K2
 
CDIR$ IVDEP
        DO 32 IL = 1, NV
            A(LK2) = A(LKK) - AK1(IL)
            B(LK2) = B(LKK) - BK1(IL)
            A(LKK) = A(LKK) + AK1(IL)
            B(LKK) = B(LKK) + BK1(IL)
            LK2 = LK2 + INCV
            LKK = LKK + INCV
  32    CONTINUE
 
        KK = K2 + KSPAN
        IF (KK .LE. NN) GO TO 30
C
        KK = KK - NN
        IF (KK .LE. INCE) GO TO 30
C
        IF (KK .GT. KSPAN) GO TO 190
 
  40    THETA1 = INCD
        GO TO 60
 
  50    THETA1 = THETA1 + INCD
        IF (THETA1 .GT. N) THETA1 = THETA1 - N
C
  60    C1 = CTABLE(THETA1)
        S1 = STABLE(THETA1)
 
  70    K2 = KK + KSPAN
        LKK = KK
        LK2 = K2
 
        DO 71 IL = 1, NV
            AK1(IL) = A(LKK) - A(LK2)
            BK1(IL) = B(LKK) - B(LK2)
            LK2 = LK2 + INCV
            LKK = LKK + INCV
  71    CONTINUE
 
        LKK = KK
        LK2 = K2
 
CDIR$ IVDEP
        DO 72 IL = 1, NV
            A(LKK) = A(LKK) + A(LK2)
            B(LKK) = B(LKK) + B(LK2)
            LK2 = LK2 + INCV
            LKK = LKK + INCV
  72    CONTINUE
 
        LK2 = K2
 
        DO 73 IL = 1, NV
            A(LK2) = C1*AK1(IL) - S1*BK1(IL)
            B(LK2) = S1*AK1(IL) + C1*BK1(IL)
            LK2 = LK2 + INCV
  73    CONTINUE
 
        KK = K2 + KSPAN
        IF (KK .LT. NT) GO TO 70
C
        K2 = KK - NT
        THETA1 = N/2 - THETA1
        IF (THETA1 .LE. 0) THETA1 = THETA1 + N
        KK = K1 - K2
        IF (KK .GT. K2) GO TO 60
        KK = KK + INCE
        IF (KK .LE. K1/2) GO TO 50
C
        K1 = K1 + INCE + INCE
        KK = (K1-KSPAN)/2 + INCE
        IF (KK .LE. INCE+INCE) GO TO 40
        GO TO 20
C
C----------------------------------------------------------------
C  Radix 3 Pass.  Note that this pass does not include twiddle
C  factor multiplication.  See code at label 130.
C----------------------------------------------------------------
C
80      IF (K .NE. 3) GOTO 100
C
90      K1 = KK + KSPAN
        K2 = K1 + KSPAN
 
        LK1 = K1
        LK2 = K2
        LKK = KK
 
        DO 91 IL = 1, NV
            AK1(IL) = A(LKK)
            BK1(IL) = B(LKK)
            AJ(IL) = A(LK1) + A(LK2)
            BJ(IL) = B(LK1) + B(LK2)
            LKK = LKK + INCV
            LK1 = LK1 + INCV
            LK2 = LK2 + INCV
  91    CONTINUE
 
        LKK = KK
 
        DO 93 IL = 1, NV
            A(LKK) = AK1(IL) + AJ(IL)
            B(LKK) = BK1(IL) + BJ(IL)
            LKK = LKK + INCV
  93    CONTINUE
 
 
        DO 94 IL = 1, NV
            AK1(IL) = -0.5 * AJ(IL) + AK1(IL)
            BK1(IL) = -0.5 * BJ(IL) + BK1(IL)
  94    CONTINUE
 
        LK1 = K1
        LK2 = K2
 
        DO 95 IL = 1, NV
            AJ(IL) = (A(LK1) - A(LK2)) * CNST(1)
            BJ(IL) = (B(LK1) - B(LK2)) * CNST(1)
            LK1 = LK1 + INCV
            LK2 = LK2 + INCV
  95    CONTINUE
 
        LK1 = K1
 
        DO 96 IL = 1, NV
            A(LK1) = AK1(IL) - BJ(IL)
            B(LK1) = BK1(IL) + AJ(IL)
            LK1 = LK1 + INCV
  96    CONTINUE
 
        LK2 = K2
 
        DO 97 IL = 1, NV
            A(LK2) = AK1(IL) + BJ(IL)
            B(LK2) = BK1(IL) - AJ(IL)
            LK2 = LK2 + INCV
  97    CONTINUE
 
        KK = K2 + KSPAN
        IF (KK.LT.NN) GO TO 90
C
        KK = KK - NN
        IF (KK.LE.KSPAN) GO TO 90
C
        GO TO 130
C
C-------------------------------------------------------------------
C  Radix 5 Pass. See code at 130 for twiddle factor multiplication
C-------------------------------------------------------------------
C
 100    IF (K .NE. 5) GOTO 120
C
 110    K1 = KK + KSPAN
        K2 = K1 + KSPAN
        K3 = K2 + KSPAN
        K4 = K3 + KSPAN
 
        LK1 = K1
        LK2 = K2
        LK3 = K3
        LK4 = K4
        LKK = KK
 
        DO 111 IL = 1, NV
            AK2(IL) = A(LK1) + A(LK4)
            AK3(IL) = A(LK1) - A(LK4)
            BK2(IL) = B(LK1) + B(LK4)
            BK3(IL) = B(LK1) - B(LK4)
            AK4(IL) = A(LK2) + A(LK3)
            AK5(IL) = A(LK2) - A(LK3)
            BK4(IL) = B(LK2) + B(LK3)
            BK5(IL) = B(LK2) - B(LK3)
            AA(IL)  = A(LKK)
            BB(IL)  = B(LKK)
            LK1 = LK1 + INCV
            LK2 = LK2 + INCV
            LK3 = LK3 + INCV
            LK4 = LK4 + INCV
            LKK = LKK + INCV
 111    CONTINUE
 
        LKK = KK
 
        DO 112 IL = 1, NV
            A(LKK) = AA(IL) + AK2(IL) + AK4(IL)
            B(LKK) = BB(IL) + BK2(IL) + BK4(IL)
            AK1(IL) = AK2(IL) * CNST(4) + AK4(IL) * CNST(2) + AA(IL)
            BK1(IL) = BK2(IL) * CNST(4) + BK4(IL) * CNST(2) + BB(IL)
            AJ(IL)  = AK3(IL) * CNST(5) + AK5(IL) * CNST(3)
            BJ(IL)  = BK3(IL) * CNST(5) + BK5(IL) * CNST(3)
            LKK = LKK + INCV
 112    CONTINUE
 
        LK1 = K1
 
        DO 113 IL = 1, NV
            A(LK1) = AK1(IL) - BJ(IL)
            B(LK1) = BK1(IL) + AJ(IL)
            LK1 = LK1 + INCV
 113    CONTINUE
 
        LK4 = K4
 
        DO 114 IL = 1,NV
            A(LK4) = AK1(IL) + BJ(IL)
            B(LK4) = BK1(IL) - AJ(IL)
            LK4 = LK4 + INCV
 114    CONTINUE
 
        DO 115 IL = 1,NV
            AK1(IL) = AK2(IL) * CNST(2) + AK4(IL) * CNST(4) + AA(IL)
            BK1(IL) = BK2(IL) * CNST(2) + BK4(IL) * CNST(4) + BB(IL)
            AJ(IL)  = AK3(IL) * CNST(3) - AK5(IL) * CNST(5)
            BJ(IL)  = BK3(IL) * CNST(3) - BK5(IL) * CNST(5)
 115    CONTINUE
 
        LK2 = K2
 
        DO 116 IL = 1,NV
            A(LK2) = AK1(IL) - BJ(IL)
            B(LK2) = BK1(IL) + AJ(IL)
            LK2 = LK2 + INCV
 116    CONTINUE
 
        LK3 = K3
 
        DO 117 IL = 1, NV
            A(LK3) = AK1(IL) + BJ(IL)
            B(LK3) = BK1(IL) - AJ(IL)
            LK3 = LK3 + INCV
 117    CONTINUE
 
        KK = K4 + KSPAN
        IF (KK.LT.NN) GO TO 110
C
        KK = KK - NN
        IF (KK.LE.KSPAN) GO TO 110
C
        GO TO 130
C
C--------------------------------------------
C  Radix 7 Pass.  See code at label 130 for
C  twiddle factor multiplication.
C--------------------------------------------
C
120     K1 = KK + KSPAN
        K2 = K1 + KSPAN
        K3 = K2 + KSPAN
        K4 = K3 + KSPAN
        K5 = K4 + KSPAN
        K6 = K5 + KSPAN
 
        LK1 = K1
        LK2 = K2
        LK3 = K3
        LK4 = K4
        LK5 = K5
        LK6 = K6
        LKK = KK
 
       DO 121 IL = 1, NV
            AK2(IL) = A(LK1) + A(LK6)
            BK2(IL) = B(LK1) + B(LK6)
            AK3(IL) = A(LK1) - A(LK6)
            BK3(IL) = B(LK1) - B(LK6)
            AK4(IL) = A(LK2) + A(LK5)
            BK4(IL) = B(LK2) + B(LK5)
            AK5(IL) = A(LK2) - A(LK5)
            BK5(IL) = B(LK2) - B(LK5)
            AK6(IL) = A(LK3) + A(LK4)
            BK6(IL) = B(LK3) + B(LK4)
            AK7(IL) = A(LK3) - A(LK4)
            BK7(IL) = B(LK3) - B(LK4)
C
            AA(IL) = A(LKK)
            BB(IL) = B(LKK)
            LK1 = LK1 + INCV
            LK2 = LK2 + INCV
            LK3 = LK3 + INCV
            LK4 = LK4 + INCV
            LK5 = LK5 + INCV
            LK6 = LK6 + INCV
            LKK = LKK + INCV
 121    CONTINUE
 
        LKK = KK
 
        DO 122 IL = 1, NV
            A(LKK) = AA(IL) + AK2(IL) + AK4(IL) + AK6(IL)
            B(LKK) = BB(IL) + BK2(IL) + BK4(IL) + BK6(IL)
            LKK = LKK + INCV
 122    CONTINUE
 
 
        DO 123 IL = 1, NV
            AK1(IL) = AA(IL) + AK2(IL)*CNST(6) + AK4(IL)*CNST(8)
     &                 + AK6(IL)*CNST(10)
            BK1(IL) = BB(IL) + BK2(IL)*CNST(6) + BK4(IL)*CNST(8)
     &                 + BK6(IL)*CNST(10)
            AJ(IL) = AK3(IL)*CNST(7) + AK5(IL)*CNST(9)
     &                 + AK7(IL)*CNST(11)
            BJ(IL) = BK3(IL)*CNST(7) + BK5(IL)*CNST(9)
     &                 + BK7(IL)*CNST(11)
 123    CONTINUE
 
        LK1 = K1
 
        DO 124 IL = 1, NV
            A(LK1) = AK1(IL) - BJ(IL)
            B(LK1) = BK1(IL) + AJ(IL)
            LK1 = LK1 + INCV
 124    CONTINUE
 
        LK6 = K6
 
        DO 125 IL = 1, NV
            A(LK6) = AK1(IL) + BJ(IL)
            B(LK6) = BK1(IL) - AJ(IL)
            LK6 = LK6 + INCV
 125    CONTINUE
 
 
C
 
        DO 126 IL = 1, NV
            AK1(IL) = AA(IL) + AK2(IL)*CNST(8) + AK4(IL)*CNST(10)
     &                 + AK6(IL)*CNST(6)
            BK1(IL) = BB(IL) + BK2(IL)*CNST(8) + BK4(IL)*CNST(10)
     &                 + BK6(IL)*CNST(6)
            AJ(IL) = AK3(IL)*CNST(9) - AK5(IL)*CNST(11)
     &                 - AK7(IL)*CNST(7)
            BJ(IL) = BK3(IL)*CNST(9) - BK5(IL)*CNST(11)
     &                - BK7(IL)*CNST(7)
 
 126    CONTINUE
 
        LK2 = K2
 
        DO 127 IL = 1, NV
            A(LK2) = AK1(IL) - BJ(IL)
            B(LK2) = BK1(IL) + AJ(IL)
            LK2 = LK2 + INCV
 127    CONTINUE
 
        LK5 = K5
 
        DO 128 IL = 1, NV
            A(LK5) = AK1(IL) + BJ(IL)
            B(LK5) = BK1(IL) - AJ(IL)
            LK5 = LK5 + INCV
 128    CONTINUE
C
 
        DO 129 IL = 1, NV
            AK1(IL) = AA(IL) + AK2(IL)*CNST(10) + AK4(IL)*CNST(6)
     &                 + AK6(IL)*CNST(8)
            BK1(IL) = BB(IL) + BK2(IL)*CNST(10) + BK4(IL)*CNST(6)
     &                 + BK6(IL)*CNST(8)
            AJ(IL) = AK3(IL)*CNST(11) - AK5(IL)*CNST(7)
     &                 + AK7(IL)*CNST(9)
            BJ(IL) = BK3(IL)*CNST(11) - BK5(IL)*CNST(7)
     &                 + BK7(IL)*CNST(9)
 129    CONTINUE
 
        LK3 = K3
 
        DO 1129 IL = 1, NV
            A(LK3) = AK1(IL) - BJ(IL)
            B(LK3) = BK1(IL) + AJ(IL)
            LK3 = LK3 + INCV
 1129  CONTINUE
 
        LK4 = K4
 
        DO 2129 IL = 1, NV
            A(LK4) = AK1(IL) + BJ(IL)
            B(LK4) = BK1(IL) - AJ(IL)
            LK4 = LK4 + INCV
 2129   CONTINUE
C
        KK = KK + KSPNN
        IF (KK.LE.NN) GO TO 120
C
        KK = KK - NN
        IF (KK.LE.KSPAN) GO TO 120
C
C----------------------------------
C  Twiddle Factor Multiplications
C----------------------------------
C
130     CONTINUE
        IF (I.EQ.M) GO TO 190
        KK = INCE + 1
C
140     THETA2 = INCD
        GO TO 160
C
150     THETA2 = THETA1 + INCD
        IF (THETA2 .GT. N) THETA2 = THETA2 - N
C
160     KK = KK + KSPAN
        THETA1 = THETA2
C
170     C2 = CTABLE(THETA2)
        S2 = STABLE(THETA2)
C
180     CONTINUE
        LKK = KK
 
        DO 181 IL = 1, NV
            AK1(IL) = A(LKK)
            LKK = LKK + INCV
 181    CONTINUE
 
        LKK = KK
 
        DO 182 IL = 1, NV
            A(LKK) = C2*AK1(IL) - S2*B(LKK)
            B(LKK) = S2*AK1(IL) + C2*B(LKK)
            LKK = LKK + INCV
 182    CONTINUE
 
        KK = KK + KSPNN
        IF (KK.LE.NT) GO TO 180
C
        THETA2 = THETA2 + THETA1
        IF (THETA2 .GT. N) THETA2 = THETA2 - N
        KK = KK - NT + KSPAN
        IF (KK.LE.KSPNN) GO TO 170
C
        KK = KK - KSPNN + INCE
        IF (KK .LE. KSPAN) GO TO 150
C
        KK = KK - KSPAN
        IF (KK.GT.0) GO TO 20
C
        KK = KK + INCE + INCE
C       GOTO 140
C
C--------------------------------------------
C  Sort the arrays A and B according to the
C  index values stored in ISRC and IDST.
C--------------------------------------------
C
190     IF (NPERM .EQ. 0) RETURN
C
 
        LKK = IDST(1)
 
        DO 191 IL = 1, NV
            AK1(IL) = A(LKK)
            BK1(IL) = B(LKK)
            LKK = LKK + INCV
 191    CONTINUE
 
        LKK = IDST(1)
        LK1 = ISRC(1)
 
CDIR$ IVDEP
        DO 192 IL = 1, NV
            A(LKK) = A(LK1)
            B(LKK) = B(LK1)
            LK1 = LK1 + INCV
            LKK = LKK + INCV
 192    CONTINUE
C
        DO 200 I=2,NPERM
 
            LKK = IDST(I)
 
            DO 193 IL = 1, NV
                AK2(IL) = A(LKK)
                BK2(IL) = B(LKK)
                LKK = LKK + INCV
 193        CONTINUE
 
                IF (IDST(I-1) .EQ. ISRC(I)) THEN
 
                    LKK = IDST(I)
 
                    DO 194 IL = 1, NV
                        A(LKK) = AK1(IL)
                        B(LKK) = BK1(IL)
                        LKK = LKK + INCV
 194                CONTINUE
 
                ELSE
 
                    LK1 = ISRC(I)
                    LKK = IDST(I)
 
CDIR$ IVDEP
                    DO 195 IL = 1, NV
                        A(LKK) = A(LK1)
                        B(LKK) = B(LK1)
                        LK1 = LK1 + INCV
                        LKK = LKK + INCV
 195                CONTINUE
 
                ENDIF
 
 
                DO 196 IL = 1, NV
                    AK1(IL) = AK2(IL)
                    BK1(IL) = BK2(IL)
 196            CONTINUE
 
200     CONTINUE
C
        RETURN
        END
C
