C****** BWCOEF  BUTTERWORTH BANDPASS FILTER DESIGN         MTADV EXT REL     1.0
C
C      ** COPYRIGHT 1988  QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C        CALL BWCOEF (FL,FH,DT,B,YNORM,NORDER,IFT)
C
C        where,
C
C        FL       Real input scaler, low frequency corner in Hz.
C
C        FH       Real input scaler, high frequency corner in Hz.
C
C        DT       Real input scaler, sampling interval.
C
C        B        Real output matrix of dimension 2 by NORDER.
C                 Contains the filter coefficents.
C
C        YNORM    Real output scaler.  The normalization constant for the
C                 filter.
C
C        NORDER   Integer input scaler order of the Butterworth filter
C
C        IFT      Integer output scaler.  IFT is one for a low pass filter and
C                 zero otherwise.
C
C  DESCRIPTION
C
C        This routine will construct a digital band pass filter based on
C        a transformed, low pass Butterworth filter.  The algorithm implemented
C        is derived by transforming in the analog domain (also known as
C        the Laplace transform domain) from a low pass Butterworth filter to
C        a band pass filter.  The bilinear transform is then applied to the
C        resulting transfer function.
C
C        The system function constucted in this routine has a form best
C        implemented as a cascade of NORDER second order structures;
C
C                  H(z)   =  H(1) * H(2) * ..... * H(i) *..... * H(NORDER)
C
C        where the individual structures are,
C
C                               K * (1 - z**2)
C                  H(i)   = ----------------------- .
C                             1 + D1 * z + D2 * z**2
C
C        The constant K assures that the overall filter has a normalized
C        frequency responce.   The K's for each section are combined to yield
C        the overall filter normalization constant called YNORM.  The user can
C        apply a filter based on this system function by a call to BWFILT or
C        BWFLTM.
C
C        The frequency corners for the band pass filter are given by FH and FL
C        for the high and low frequency corners respectively.  If FL is 0, the
C        filter is a traditional, Butterworth low pass filter with critical
C        frequency at FH.  If FH exceeds the Nyquist freqency, the resulting
C        filter is a high pass filter with corner at FL.  The units for FH and
C        FL are Hz.
C
C        If NORDER is zero, this routine exits immediately.  Note that this
C        algorithm leads to potentially erroneous filter responces for high
C        filter orders due to "numerical noise propagation".
C
C  REFERENCE
C
C        Oppenheim A.V. and Schafer R.W., 1975, Digital Signal Processing,
C        (Englewood Cliffs, New Jersey, Prentice-Hall Co). Chapters 4 and 5.
C
C        Kanasewich E.R., Time Sequence Analysis in Geophysics.
C
C        This code is a direct modification of parts of the program FILT
C        written by R.B. Herrmann and T. Mokhtar of Saint Louis University
C        in 1986.
C
C  EXAMPLE
C
C        CALL BWCOEF (0.0, 11.6455, 0.01, B, YNORM, 6, IFT)
C
C        Output Operands:
C
C           B       =    -2.00000000E+00      1.00000000E+00
C                        -2.00000000E+00      1.00000000E+00
C                        -2.00000000E+00      1.00000000E+00
C                        -9.04394269E-01      2.15528116E-01
C                        -1.01060855E+00      3.58282685E-01
C                        -1.26867843E+00      7.05134928E-01
C
C           YNORM   =     7.37701077E-04
C
C           IFT     =     1
C
C  SUBPROGRAMS
C
C        INTRINSICS:  COS, SIN, ABS, SQRT
C
C  HISTORY
C        (1) JUNE 88           T.G. Mattson           Original
C
C-----------------------------------------------------------------------
C
      SUBROUTINE BWCOEF (FL, FH, DT, B, YNORM, NORDER, IFT)
C
      INTEGER NORDER, IFT, NPOLE, I, NPOLE2
      REAL    B(2,*), YNORM,FL,FH,DT
      REAL    WDL, WDH, X, WA, S1, SR, SI, XN, XD, ANG,PI,FNYQ
      REAL    AZR,AZI,CZR,CZI,ZCR,ZCI,RTMP,Z1R,Z1I,Z2R,Z2I
      REAL    ZT1R, ZT1I, ZT2R, ZT2I, Z1MAG, Z2MAG,S1MAG,SMAG
      PARAMETER (PI = 3.1415926535897932384626433)
C
C-----------------------------------------------------------------------
C
      IF (NORDER .LE. 0) RETURN
C
C  Check the validity of the corner frequencies and clean them up if
C  nessesary.   Default to a low pass filter if FL is zero and a high
C  pass filter if FH is greater than the Nyquist frequency (FNYQ).
C  Otherwise, a pass band filter is generated.
C
      IFT     = 0
      FNYQ    = 0.5 / DT
      IF (FL .LE. 0.0)  THEN
          FL  = 0.0
          IFT = 1
      ENDIF
      IF (FH .GT. FNYQ) FH  = FNYQ
C
      YNORM   = 1.0
      NPOLE   = NORDER
      WDL     = PI * FL * DT
      WDH     = PI * FH * DT
      XN      = COS (WDL + WDH)
      XD      = COS (WDH - WDL)
      X       = XN / XD
      WA      = SIN (WDH - WDL) / XD
      S1      = ABS (WA)
      NPOLE2  = (NPOLE + 1 ) / 2
C
C  Find the filter coefficents.  In the following code, SR and SI are related
C  to the real and imaginary parts of the S plane poles.  There are NPOLE2 pairs
C  of these poles where the members of each pair are complex conjugates of each
C  other.  These poles fall on the perimeter of a circle of radius S1.
C
      DO 1000 I=1,NPOLE2,1
         ANG =  0.5  * PI * (1.0 + (2.0 * I - 1.0) / NPOLE )
         SR  = -S1   * COS(ANG)
         SI  = -S1   * SIN(ANG)
         AZR =  1.0  + SR
         AZI =  SI
         CZR =  1.0  - SR
         CZI = -SI
         ZCR =  (AZR * CZR - AZI * CZI)
         ZCI = -(AZI * CZR + AZR * CZI)
         ZCR =  X    * X   - ZCR
C
C  Find the square root of zc = real(zc) + i*imag(zc) = ZCR + i * ZCI
C
         ANG   = ATAN2 (ZCI, ZCR)/ 2.0
         RTMP  = SQRT (ZCR * ZCR + ZCI * ZCI)
         RTMP  = SQRT (RTMP)
         ZCR   = RTMP * COS (ANG)
         ZCI   = RTMP * SIN (ANG)
C
         ZT1R  =  X + ZCR
         ZT1I  =  ZCI
         ZT2R  =  X - ZCR
         ZT2I  = -ZCI
C
C  Find Z1 = Z1R + i * Z1I = (ZT1R + i*ZT1I)/(AZR + i * AZI) and likewise for
C  Z2 = ZT2R + i * ZT2I over AZ.
C
         RTMP  =  AZR  * AZR  + AZI  * AZI
         Z1R   = (ZT1R * AZR  + AZI  * ZT1I)/RTMP
         Z1I   = (ZT1I * AZR  - ZT1R * AZI) /RTMP
         Z2R   = (ZT2R * AZR  + AZI  * ZT2I)/RTMP
         Z2I   = (ZT2I * AZR  - ZT2R * AZI) /RTMP
C
C  Find the filter coeficents for the paired poles located off of the real axis
C
         IF (I .NE. (NPOLE+1-I) ) THEN
            Z1MAG          =  Z1R * Z1R + Z1I * Z1I
            Z2MAG          =  Z2R * Z2R + Z2I * Z2I
C            SMAG          =  SR  * SR  + SI  * SI
            SMAG           =  S1*S1
            S1MAG          =  1.0 + 2.0 * SR  + SMAG
            B(1,I)         = -2.0 * Z1R
            B(2,I)         =  Z1MAG
            B(1,NPOLE+1-I) = -2.0 * Z2R
            B(2,NPOLE+1-I) =  Z2MAG
            YNORM          =  YNORM * SMAG / S1MAG
         ELSE
C
C  This section of the routine is aplicable for odd values of NORDER only
C  and corresponds to the final S plane pole which found on the negative
C  real axis.
C
            RTMP           = 1.0/(1.0 + S1)
            B(1,I)         = -2.0  * X   * RTMP
            B(2,I)         = (1.0  - S1) * RTMP
            YNORM          = YNORM * S1  * RTMP
         ENDIF
 1000 CONTINUE
C
      RETURN
      END
