C****** BWFLTM  MULTIPLE TRACE PASSBAND FILTER APPLICATION     MTADV EXT REL 1.0
C
C      ** COPYRIGHT 1988  QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C        CALL BWFLTM (X,Y,XD,YD,A,WRK,YNORM,NSECT,NSAMP,NDIM,NTRC,INIT,IFT)
C
C        where,
C
C        X        Real input array of dimension NDIM by NTRC.  The set of
C                 NTRC input sequences to which the filter will be applied.
C
C        Y        Real output array of dimension NDIM by NTRC.  The set of
C                 NTRC filtered sequences.
C
C        XD       Real work array of dimension NTRC by 3.  Contains the
C                 delayed input samples.
C
C        YD       Real work array of dimension NTRC by 3*NSECT.  Contains the
C                 delayed output samples for each section.
C
C        A        Real input array of dimension 2 by NSECT.  The filter
C                 coefficents usually computed by BWCOEF.
C
C        WRK      Real work array of dimension NTRC.
C
C        YNORM    Real input scaler filter normalization constant.
C
C        NSECT    Integer input scaler.  The number of sections in the Cascade
C                 realization of the filter.  This is equal to the filter order
C                 for a Butterworth filter.
C
C        NSAMP    Integer input scaler. Length of the sequences in X and Y.
C                 NSAMP must be less than or equal to NDIM.
C
C        NDIM     Integer input scaler. The dimension of the arrays X and Y
C                 in the NSAMP (row) dimension.  This number should be odd
C                 to work best with the CRAY 2 memory interleave.
C
C        NTRC     Integer input scaler.  The number of traces to be filtered.
C
C        INIT     Integer input scaler. If greater than zero, the filter
C                 is initialized in this subroutine.
C
C        IFT      Integer input scaler.  If IFT is set to one, the value of
C                 the first element of each input trace is added to each
C                 element of the corresponding output trace.
C
C  DESCRIPTION
C
C        This routine applies a BUTTERWORTH filter to a set of traces.  The
C        code is designed so that vectorization occurs accross a row of traces.
C        The filter is realized as a cascade of NSECT second order structures;
C
C                  H(z)   =  H(1) * H(2) * ..... * H(i) *..... * H(NSECT)
C
C        where the individual structures are,
C
C                               K * (1 - z**2)
C                  H(i)   = ----------------------- .
C                             1 + D1 * z + D2 * z**2
C
C        The constant K assures that the overall filter has a normalized
C        frequency responce.   The K's for each section are combined to yield
C        the overall filter normalization constant called YNORM.  The filter
C        applied by this routine is synthesized in the subroutine, BWCOEF.
C
C        The filter must be initialized by setting the values in the delay
C        arrays, XD and YD.   The delay array, XD, contains the three delayed
C        input values for each trace.   YD contains the three delayed "output"
C        values for each section with the section and delay indices packed
C        into a single index.   The initialization is done in this  subroutine
C        if the flag INIT is set to a non-negative value.  This initialization
C        is:
C
C              XD(j,n) = X(1,j)  for  j = 1,NTRC, and n = 1,3
C              YD(j,i) = 0       for  j = 1,NTRC, and i = 1,3*NSECT
C
C        If other initializations are desired, the delay arrays can be set in
C        the calling program unit and INIT set to a negative value.
C
C        If illeagal values for the scalers NSAMP or NSECT are  encountered,
C        this subroutine returns immediately.
C
C  REFERENCE
C
C        Oppenheim A.V. and Schafer R.W., 1975, Digital Signal Processing,
C        (Englewood Cliffs, New Jersey, Prentice-Hall Co). Chapters 4 and 5.
C
C        Kanasewich E.R., Time Sequence Analysis in Geophysics.
C
C  EXAMPLE
C
C        CALL BWFLTM (X, Y, XD, YD, A, WRK, YNORM, 6, 10, 10,
C    &                                   5, 1, IFT)
C
C        Input Operands:
C
C             X = 31.52851   53.27586   74.99746   96.68282   118.3214
C                 62.87141   106.2230   149.3692   192.2267   234.7126
C                 94.08928   158.7596   222.7395   285.7507   347.5192
C                 125.1200   210.6829   294.6173   376.2744   455.0227
C                 155.9021   261.7931   364.5236   462.8525   555.5926
C                 186.3744   311.8944   431.9944   544.5859   647.7120
C                 216.4770   360.7959   496.5846   620.6317   730.0038
C                 246.1506   408.3123   557.8725   690.2136   801.2514
C                 275.3372   454.2651   615.4609   752.6306   860.4188
C                 303.9803   498.4835   668.9821   807.2639   906.6690
C
C             A = -2.00000000E+00       1.00000000E+00
C                 -2.00000000E+00       1.00000000E+00
C                 -2.00000000E+00       1.00000000E+00
C                 -9.04394269E-01       2.15528116E-01
C                 -1.01060855E+00       3.58282685E-01
C                 -1.26867843E+00       7.05134928E-01
C
C             YNORM = 7.37701077E-04
C
C             IFT   = 1
C
C        Output Operands:
C
C             Y = 31.52851   53.27586   74.99746   96.68282   118.3214
C                 31.55163   53.31492   75.05232   96.75330   118.4073
C                 31.78700   53.71238   75.61031   97.46959   119.2790
C                 32.93737   55.65396   78.33386   100.9622   123.5240
C                 36.56709   61.77575   86.91183   111.9460   136.8492
C                 44.96267   75.92094   106.7013   137.2324   167.4442
C                 60.28524   101.6986   142.6834   183.0690   222.6897
C                 83.45496   140.5930   196.7944   251.6898   304.9242
C                 113.5175   190.8945   266.4279   339.3991   409.1257
C                 147.9126   248.1628   345.1063   437.4725   524.0734
C
C  SUBPROGRAMS
C
C        Math Advantage: VMOV, VADD, VSMA, VSMUL
C
C  HISTORY
C        (1) JUNE 88           T.G. Mattson           Original
C
C-----------------------------------------------------------------------
        SUBROUTINE BWFLTM (X, Y, XD, YD, A, WRK, YNORM, NSECT, NSAMP,
     &                                     NDIM, NTRC, INIT, IFT)
C
        INTEGER NSECT, NSAMP, NDIM, NTRC, INIT, IFT
        REAL    X(NDIM,*), Y(NDIM,*), XD(NTRC,*), YD(NTRC,*), A(2,*)
        REAL    YNORM, WRK(*)
        INTEGER JJ0, JJ1, JJ2, JM0, JM1, JM2, NS0, NS1, NS2
        INTEGER ISMP, ITRC, J
        REAL    XTMP
C
C-----------------------------------------------------------------------
C
        IF ( (NSECT.LE.0) .OR. (NSAMP.GT.NDIM) .OR. (NSAMP.LE.0))THEN
           RETURN
        ENDIF
C
C  Initialize the DELAY array (if appropriate).  The three "Y" delays
C  per section are packed into one dimension.  NS0, NS1, and NS2
C  refer to the indices for the last sections's three Y delays.
C
        IF ( INIT .GT. 0) THEN
           DO 150 ITRC=1,NTRC
              XTMP       = X(1,ITRC)
              XD(ITRC,1) = XTMP
              XD(ITRC,2) = XTMP
              XD(ITRC,3) = XTMP
              DO 100 J=1,3*NSECT
                 YD(ITRC,J) = 0.0
 100          CONTINUE
 150       CONTINUE
        ENDIF
        NS0 = 3 * (NSECT - 1) + 1
        NS1 = NS0 + 1
        NS2 = NS0 + 2
C
        DO 300 ISMP = 1,NSAMP
C
C  Compute the first filter section.
C
           CALL VMOV (X(ISMP,1), NDIM, XD, 1, NTRC)
           CALL VSMUL (YD(1,2), 1, A(1,1), WRK, 1, NTRC)
           CALL VSMA  (YD(1,3), 1, A(2,1), WRK, 1, WRK, 1, NTRC)
           DO 205 ITRC=1,NTRC
              YD(ITRC,1) = XD(ITRC,1) -XD(ITRC,3) - WRK(ITRC)
 205       CONTINUE
C
C  All remaining sections are now computed.  Each section takes its input from
C  the preceeding section.  JJ0, JJ1 and JJ2 index the Y delays for the current
C  section while JM0, JM1 and JM2 index the Y delays from the previous section.
C
           JJ0 = 4
           JM0 = 1
           JM1 = 2
           JM2 = 3
           DO 230 J = 2,NSECT
              JJ1 = JJ0 + 1
              JJ2 = JJ0 + 2
              CALL VSMUL (YD(1,JJ1), 1, A(1,J), WRK, 1, NTRC)
              CALL VSMA  (YD(1,JJ2), 1, A(2,J), WRK, 1, WRK, 1, NTRC)
              DO 220 ITRC = 1,NTRC
                 YD(ITRC,JJ0) = YD(ITRC,JM0)-YD(ITRC,JM2)-WRK(ITRC)
                 YD(ITRC,JM2) = YD(ITRC,JM1)
                 YD(ITRC,JM1) = YD(ITRC,JM0)
 220          CONTINUE
              JM0 = JJ0
              JM1 = JJ1
              JM2 = JJ2
              JJ0 = JJ0 + 3
 230       CONTINUE
C
C  Finish updating the delay arrays and find the normalized output trace.
C
           DO 250 ITRC=1,NTRC
              XD(ITRC,3)   = XD(ITRC,2)
              XD(ITRC,2)   = XD(ITRC,1)
              YD(ITRC,NS2) = YD(ITRC,NS1)
              YD(ITRC,NS1) = YD(ITRC,NS0)
              Y(ISMP,ITRC) = YNORM * YD(ITRC,NS0)
 250       CONTINUE
 300    CONTINUE
C
C  Add in the first element of the input trace when appropriate
C
        IF (IFT .EQ. 1)THEN
           CALL VMOV (X(1,1), NDIM, WRK,1, NTRC)
           DO 330 ISMP=1,NSAMP
              CALL VADD (Y(ISMP,1),NDIM,WRK,1,Y(ISMP,1),NDIM,NTRC)
 330       CONTINUE
        ENDIF
C
        RETURN
        END
