C*****  DNRM2    Euclidean Length of Double Prec Real      MATH ADV REL 3.0
C
C    ** COPYRIGHT 1984-1985 QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       DW = DNRM2 (N,DX,INCX)
C
C       where,
C
C       N       Integer input element count.
C
C       DX      Double precision real input vector.
C
C       INCX    Integer input stride for vector DX.
C               May not be negative.
C
C
C  DESCRIPTION
C
C       This function returns the Euclidean length, also know as the
C       Euclidean norm, of the elements of double precision vector DX.
C
C       If N <= 0 return with result = 0.0
C
C       Four phase method using two calculated constants:
C         CUTLO = DSQRT(U/EPS)
C         CUTHI = DSQRT(V)
C       where
C         EPS = smallest number such that   EPS + 1.0 > 1.0
C         U   = smallest positive number   (underflow limit)
C         V   = largest number             (overflow  limit)
C
C       Brief outline of algorithm..
C
C       Phase 1 scans zero components.
C       Move to phase 2 when a component is nonzero and <= CUTLO
C       Move to phase 3 when a component is > CUTLO
C       Move to phase 4 when a component is >= CUTHI/N
C
C
C  REFERENCE
C
C       J.J. Dongarra and C.B. Moler, J.R. Bunch, G.W. Stewart.  1979.
C       LINPACK Users' Guide.  Appendix A: Basic Linear Algebra Subprograms.
C       SIAM
C
C  EXAMPLE
C
C       DW = DNRM2 (5,DX,1)
C
C       Input Operands:
C
C       DX =  1.000
C             2.000
C            -3.000
C             4.000
C             5.000
C
C       Output Result:
C
C       DW = 7.416198
C
C  HISTORY
C         1) Jan 88     L. Shanbeck     Original.
C
C
      DOUBLE PRECISION FUNCTION DNRM2(N,DX,INCX)
      DOUBLE PRECISION DX(1),CUTLO,CUTHI,HITEST,SUM,XMAX,ZERO,ONE
      DOUBLE PRECISION EPS,TINY,HUGE,S
      INTEGER N,INCX,I,J
C
C   SMALL is used to prevent values below SMALL from being returned.
C   See the end of the program for usage.  Only for Microsoft Fortran.
C     REAL SMALL
C     DATA SMALL / 8.43E-37 /
C
      DATA ZERO, ONE /0.0D0, 1.0D0/
      DATA EPS, TINY, HUGE /0.0D0, 0.0D0, 0.0D0/
C
      DNRM2 = ZERO
      IF(N.LE.0 .OR. INCX.LT.0) GO TO 999
C
C  SET MACHINE DEPENDENT VALUES, IF NECESSARY
      IF (HUGE .EQ. 0.0D0) THEN
C
         EPS = 1.0D0
   1     EPS = EPS/2.0D0
         S = 1.0D0 + EPS
         IF (S .GT. 1.0D0) GO TO 1
         EPS = 2.0D0*EPS
C
         S = 1.0D0
   2     TINY = S
         S = S/16.0D0
         IF (S*1.0D0 .NE. 0.0D0) GO TO 2
         TINY = (TINY/EPS)*100.0D0
         HUGE = 1.0D0/TINY
C
         CUTLO = DSQRT(TINY/EPS)
         CUTHI = DSQRT(HUGE)
      ENDIF
C
      SUM = ZERO
      NN = N * INCX
C
C  FOR SINGLE OR DOUBLE PRECISION SET HITEST = CUTHI/N
      HITEST = CUTHI/DBLE( N )
      XMAX = ZERO
C                        BEGIN MAIN LOOP
      I = 1
   20 CONTINUE
C
C                        PHASE 1.  SUM IS ZERO
C
      IF( DX(I) .EQ. ZERO) THEN
         CONTINUE
C
C                        PHASE 2.  SUM IS SMALL
C                        SCALE TO AVOID DESTRUCTIVE UNDERFLOW.
C
      ELSE IF( DABS(DX(I)) .LE. CUTLO) THEN
         IF( DABS(DX(I)) .GT. XMAX) THEN
            SUM = ONE + SUM * (XMAX / DX(I))**2
            XMAX = DABS(DX(I))
         ELSE
            SUM = SUM + (DX(I)/XMAX)**2
         ENDIF
C
C                        PHASE 3.  SUM IS MID-RANGE.  NO SCALING.
C
      ELSE
         SUM = (SUM * XMAX) * XMAX
         DO 95 J =I,NN,INCX
            IF(DABS(DX(J)) .GE. HITEST) GO TO 100
            SUM = SUM + DX(J)**2
   95    CONTINUE
         DNRM2 = DSQRT( SUM )
         GO TO 300
C
C                        PHASE 4.
C                        SUM IS LARGE.  SCALE TO AVOID OVERFLOW.
C
  100    SUM = (SUM / DX(J)) / DX(J)
         XMAX = DABS(DX(J))
         DO 120 I=J,NN,INCX
            IF( DABS(DX(I)) .GT. XMAX) THEN
               SUM = ONE + SUM * (XMAX / DX(I))**2
               XMAX = DABS(DX(I))
            ELSE
               SUM = SUM + (DX(I)/XMAX)**2
            ENDIF
  120    CONTINUE
      ENDIF
C
C                        END OF MAIN LOOP.
  200 CONTINUE
      I = I + INCX
      IF (I .LE. NN) GO TO 20
C
C              COMPUTE SQUARE ROOT AND ADJUST FOR SCALING.
C
      DNRM2 = XMAX * DSQRT(SUM)
  300 CONTINUE
C
C *** 6 Jan 84, J. Fried :
C     This next line of FORTRAN was added to prevent values below
C     MS-FORTRAN's specified minimum from being returned.  Such
C     numbers have appeared during testing of the SQRDC routine
C     (test #9 in SQ).  The value of SMALL was taken from the
C     MS-FORTRAN manual.  Remember, this only applies to PC/MS-FORTRAN.
C
C     IF ( DNRM2 .LE. SMALL ) DNRM2 = SMALL
999   RETURN
      END
