C***** QTC078  BUILD SPARSE UNSYMM. STRUCTURE  REV 1.0         JAN 88
C
C  PURPOSE:
C       BUILDS THE DATA STRUCTURE FOR A REORDERED SPARSE UNSYMMETRIC
C       MATRIX.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       ORIGINAL                JAN 88          R.D. COLEMAN, QTC
C
C  CALLING FORMAT:
C       CALL QTC078 (N, IADJ, JADJ, IPERMR, IPERMF, ICP, IDP, IRN,
C                    MAXNS, NS, IWRK1, IWRK2, IWRK3, IERR)
C
C  PARAMETERS:
C       N       INTEGER INPUT SCALAR
C               NUMBER OF EQUATIONS.
C
C       IADJ    INTEGER INPUT VECTOR OF LENGTH N+1
C               POINTERS INTO THE ADJACENCY VECTOR.  IADJ(N+1) = 2*NE+1,
C               WHERE NE IS THE NUMBER OF EDGES.
C
C       JADJ    INTEGER INPUT VECTOR OF LENGTH 2*NE
C               THE ADJACENCY STRUCTURE OF THE GRAPH OF THE MATRIX.
C
C       IPERMR  INTEGER INPUT VECTOR OF LENGTH N
C               CONTAINS THE REVERSE PERMUTATION VECTOR.
C
C       IPERMF  INTEGER INPUT VECTOR OF LENGTH N
C               CONTAINS THE FORWARD PERMUTATION VECTOR.
C
C       ICP     INTEGER OUTPUT VECTOR OF LENGTH N+1
C               COLUMN POINTERS INTO THE ARRAY IRN.
C
C       IDP     INTEGER OUTPUT VECTOR OF LENGTH N
C               DIAGONAL ELEMENT POINTERS INTO THE ARRAY IRN.
C
C       IRN     INTEGER OUTPUT VECTOR OF LENGTH MAXNS
C               ROW NUMBERS ARRAY.
C
C       MAXNS   INTEGER INPUT SCALAR
C               NUMBERS OF WORDS AVAILABLE IN ARRAY IRN.
C
C       NS      INTEGER OUTPUT SCALAR
C               NUMBERS OF WORDS USED IN ARRAY IRN; I.E., THE NUMBER
C               OF SPARSE ELEMENTS (NON-ZERO AND FILLIN) IN THE FULL
C               MATRIX.
C
C       IWRK1   INTEGER SCRATCH VECTOR OF LENGTH N+1
C
C       IWRK2   INTEGER SCRATCH VECTOR OF LENGTH N
C
C       IWRK3   INTEGER SCRATCH VECTOR OF LENGTH N
C
C       IERR    INTEGER OUTPUT SCALAR
C               COMPLETION CODE: IERR = 0 - NORMAL COMPLETION
C                                IERR = 1 - INSUFFICIENT SPACE IN IRN
C
C  DESCRIPTION:
C       GIVEN A SPARSE UNSYMMETRIC MATRIX OF ORDER N WHOSE GRAPH IS
C       SPECIFIED BY THE ADJACENCY STRUCTURE (IADJ, JADJ) AND A
C       REORDERING OF THE MATRIX SPECIFIED BY THE PERMUTATION VECTORS
C       (IPERMR, IPERMF), THIS ROUTINE BUILDS THE MATH ADVANTAGE DATA
C       STRUCTURE (ICP, IRN) FOR THE REODERED MATRIX.  THE ROUTINE
C       QTC079 IS FIRST CALLED TO SYMBOLICALLY FACTOR THE MATRIX AND
C       BUILD THE COMPRESSED ROW NUMBER VECTOR.  THE COMPRESSED ROW
C       NUMBER VECTOR IS THEN EXPANDED IN-PLACE.
C
C  REFERENCE:
C       Alan George and Joseph W-H. Liu.  1981.  Computer Solution of
C       Large Sparse Positive Definite Systems, Prentice-Hall, Englewood
C       Cliffs, N.J. (1981)
C
C  SUBPROGRAMS CALLED:
C       QTC079
C
C  ERROR CONDITIONS:
C       NONE
C
C-----------------------------------------------------------------------
C
      SUBROUTINE QTC078 (N, IADJ, JADJ, IPERMR, IPERMF, ICP, IDP, IRN,
     &                   MAXNS, NS, IWRK1, IWRK2, IWRK3, IERR)
C
      INTEGER N, IADJ(1), JADJ(1), IPERMR(1), IPERMF(1), ICP(1), IRN(1),
     &        IDP(1), MAXNS, NS, IWRK1(1), IWRK2(1), IWRK3(1), IERR
      INTEGER I, J, K, K1, K2, KSTRT, KSTOP, L, LSTOP, LSTRT, NK
C
C-----------------------------------------------------------------------
C
C     ------------------------------
C     PERFORM SYMBOLIC FACTORIZATION
C     ------------------------------
C
      CALL QTC079 (N, IADJ, JADJ, IPERMR, IPERMF, ICP, IRN, MAXNS,
     &             IWRK1, IWRK2, IWRK3, IDP, IERR)
C
      IF (IERR .NE. 0) GO TO 810
C
      NS = 2 * (IWRK1(N) - 1) + N
      IF (NS .GT. MAXNS) GO TO 810
C
C     -----------------------------------------------------------------
C     IWRK1(J) = NUMBER OF NON-ZEROS BELOW DIAGONAL IN J-TH COLUMN
C     IWRK2(J) = NUMBER OF NON-ZEROS ABOVE DIAGONAL IN J-TH COLUMN
C     IWRK3(J) = POINTER INTO COMPRESSED ROW NUMBER VECTOR FOR J-TH COL
C     -----------------------------------------------------------------
C
      DO 110 J = 1, N
         IWRK2(J) = 0
         IWRK3(J) = ICP(J)
  110 CONTINUE
C
      DO 130 J = 1, N
         NK = IWRK1(J+1) - IWRK1(J)
         IWRK1(J) = NK
         KSTRT = IWRK3(J)
         KSTOP = KSTRT + NK - 1
         DO 120 K = KSTRT, KSTOP
            I = IRN(K)
            IWRK2(I) = IWRK2(I) + 1
  120    CONTINUE
  130 CONTINUE
C
C     ----------------------------------------------------
C     CALCULATE NEW COLUMNS POINTERS AND DIAGONAL POINTERS
C     ----------------------------------------------------
C
      ICP(1) = 1
      DO 140 J = 1, N
         IDP(J  ) = ICP(J) + IWRK2(J)
         ICP(J+1) = IDP(J) + IWRK1(J) + 1
         IWRK2(J) = ICP(J)
  140 CONTINUE
C
C     ---------------------------------------------------------------
C     EXPAND COMPRESSED LOWER TRIANGLE ROW NUMBERS INTO NEW LOCATIONS
C     ---------------------------------------------------------------
C
      DO 220 J = N-1, 1, -1
         NK = IWRK1(J)
         IF (NK .GT. 0) THEN
            K1 = IWRK3(J) + NK
            K2 = ICP(J+1)
            DO 210 K = 1, NK
               K1 = K1 - 1
               K2 = K2 - 1
               IRN(K2) = IRN(K1)
  210       CONTINUE
         ENDIF
  220 CONTINUE
C
C     --------------------------------
C     NOW ADD THE DIAGONAL ROW NUMBERS
C     --------------------------------
C
      DO 230 J = 1, N
         IRN(IDP(J)) = J
  230 CONTINUE
C
C     ----------------------------------------------
C     FINALLY, INSERT THE UPPER TRIANGLE ROW NUMBERS
C     ----------------------------------------------
C
      DO 250 J = 1, N-1
         LSTRT = IDP(J)   + 1
         LSTOP = ICP(J+1) - 1
         IF (LSTRT .LE. LSTOP) THEN
            DO 240 L = LSTRT, LSTOP
               I = IRN(L)
               K = IWRK2(I)
               IRN(K) = J
               IWRK2(I) = IWRK2(I) + 1
  240       CONTINUE
         ENDIF
  250 CONTINUE
C
      RETURN
C
C     ----------------------------------------------------
C     ERROR - INSUFFICIENT STORAGE FOR NONZERO SUBSCRIPTS.
C     ----------------------------------------------------
C
  810 CONTINUE
      IERR = 1
      RETURN
C
      END
