C*****  RMTUSV  Real Matrix Tridiagonal Unsym Solve       MATH ADV REL 3.0
C
C    ** COPYRIGHT 1984-1985 QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       CALL RMTUSV (AD, AU, AL, N, BX, M)
C
C       where,
C
C       AD      Real input vector of length N.
C               AD contains the reciprocals of the main diagonal
C               of the factored matrix.
C
C       AU      Real input vector of length N-1.
C               AU contains the reciprocals of the upper diagonal
C               of the factored matrix.
C
C       AL      Real input vector of length N-1.
C               AL contains the reciprocals of the lower diagonal
C               of the factored matrix.
C
C       N       Integer input number of rows in full matrix.
C
C       BX      Real input/output matrix of length M*N.
C               On input, BX contains M right-hand sides.
C               On output, BX contains M solution vectors.
C
C       M       Integer input number of right-hand sides.
C
C
C  DESCRIPTION
C
C       This routine solves a real linear system Ax = b,
C       where A has been factored into LU by RMTUFC, and is
C       represented in tridiagonal format by AD, AU and AL.
C
C
C  REFERENCE
C
C       G. W. Stewart.  1973.  Introduction to matrix computa-
C       tions.  New York: Academic Press.
C
C
C  EXAMPLE
C
C       CALL RMTUSV (AD, AU, AL, 5, BX, 2)
C
C       Input Operands:
C
C       AD = -0.0625   AU = -0.5000   AL =  4.000
C             0.0769         0.3077        -2.000
C            -0.1354         0.5417         5.000
C             0.0700         0.2799         0.000
C             0.2000
C
C       BX =  64.000   -80.000
C           -134.000    92.000
C            -44.000   -32.000
C            205.000   -67.000
C             25.000   -30.000
C
C       Output Operands:
C
C       BX =  -9.000     7.000
C            -10.000     4.000
C              3.000     5.000
C             10.000    -4.000
C              5.000    -6.000
C
C  HISTORY
C         1) Jan 88     L. Shanbeck     Original.
C                       R. Coleman
C
      SUBROUTINE RMTUSV (AD, AU, AL, N, BX, M)
C
      REAL    AD(1), AU(1), AL(1), BX(1), SD, SL
      INTEGER N, M, I, J, JB
C
C     DO VALIDITY CHECKS AND INITIALIZATION
C
      IF (N.LE.0  .OR. M.LE.0) GOTO 800
      IF (M .GT. 1) GO TO 200
C
C  PATH FOR M = 1
C
C  FORWARD ELIMINATION
C
      BX(1) = AD(1) * BX(1)
      IF (N .EQ. 1) GO TO 800
C
      DO 110 I = 2, N
         BX(I) = AD(I) * (BX(I) - AL(I-1) * BX(I-1))
  110 CONTINUE
C
C  BACKWARD SUBSTITUTION
C
      DO 120 I = N-1, 1, -1
         BX(I) = BX(I) - AU(I) * BX(I+1)
  120 CONTINUE
      GO TO 800
C
C  PATH FOR M > 1
C
C  FORWARD ELIMINATION
C
  200 CONTINUE
      JB = 1
      SD = AD(1)
      DO 210 J = 1, M
         BX(JB) = SD * BX(JB)
         JB = JB + N
  210 CONTINUE
      IF (N .EQ. 1) GO TO 800
C
      DO 230 I = 2, N
         JB = I
         SD = AD(I)
         SL = AL(I-1)
         DO 220 J = 1, M
            BX(JB) = SD * (BX(JB) - SL * BX(JB-1))
            JB = JB + N
  220    CONTINUE
  230 CONTINUE
C
C  BACKWARD SUBSTITUTION
C
      DO 250 I = N-1, 1, -1
         JB = I
         SU = AU(I)
         DO 240 J = 1, M
            BX(JB) = BX(JB) - SU * BX(JB+1)
            JB = JB + N
  240    CONTINUE
  250 CONTINUE
C
  800 CONTINUE
      RETURN
      END
