C*****  SNRM2    Euclidean Length of Single Prec Real      MATH ADV REL 3.0
C
C    ** COPYRIGHT 1984-1985 QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       SW = SNRM2 (N,SX,INCX)
C
C       where,
C
C       N       Integer input element count.
C
C       SX      Real input vector.
C
C       INCX    Integer input stride for vector SX.
C               May not be negative.
C
C
C  DESCRIPTION
C
C       This function returns the Euclidean length, also know as the
C       Euclidean norm, of the elements of vector SX.
C
C       If N <= 0 return with result = 0.0
C
C       Four phase method using two calculated constants:
C         CUTLO = SQRT(U/EPS)
C         CUTHI = SQRT(V)
C       where
C         EPS = smallest number such that   EPS + 1.0 > 1.0
C         U   = smallest positive number   (underflow limit)
C         V   = largest number             (overflow  limit)
C
C       Brief outline of algorithm..
C
C       Phase 1 scans zero components.
C       Move to phase 2 when a component is nonzero and <= CUTLO
C       Move to phase 3 when a component is > CUTLO
C       Move to phase 4 when a component is >= CUTHI/N
C
C
C  REFERENCE
C
C       J.J. Dongarra and C.B. Moler, J.R. Bunch, G.W. Stewart.  1979.
C       LINPACK Users' Guide.  Appendix A: Basic Linear Algebra Subprograms.
C       SIAM
C
C  EXAMPLE
C
C       SW = SNRM2 (5,SX,1)
C
C       Input Operands:
C
C       SX =  1.000
C             2.000
C            -3.000
C             4.000
C             5.000
C
C       Output Result:
C
C       SW = 7.416198
C
C  HISTORY
C         1) Jan 88     L. Shanbeck     Original.
C
C
      REAL FUNCTION SNRM2(N,SX,INCX)
      REAL SX(1),CUTLO,CUTHI,HITEST,SUM,XMAX,ZERO,ONE
      REAL EPS,TINY,HUGE,S
      INTEGER N,INCX,I,J
C
C   SMALL is used to prevent values below SMALL from being returned.
C   See the end of the program for usage.  Only for Microsoft Fortran.
C     REAL SMALL
C     DATA SMALL / 8.43E-37 /
C
      DATA ZERO, ONE /0.0, 1.0/
      DATA EPS, TINY, HUGE /0.0, 0.0, 0.0/
C
      SNRM2  = ZERO
      IF(N.LE.0 .OR. INCX.LT.0) GO TO 999
C
C  SET MACHINE DEPENDENT VALUES, IF NECESSARY
      IF (HUGE .EQ. 0.0) THEN
C
         EPS = 1.0E0
   1     EPS = EPS/2.0E0
         S = 1.0E0 + EPS
         IF (S .GT. 1.0E0) GO TO 1
         EPS = 2.0E0*EPS
C
         S = 1.0E0
   2     TINY = S
         S = S/16.0E0
         IF (S*1.0 .NE. 0.0E0) GO TO 2
         TINY = (TINY/EPS)*100.0E0
         HUGE = 1.0E0/TINY
C
         CUTLO = SQRT(TINY/EPS)
         CUTHI = SQRT(HUGE)
      ENDIF
C
      SUM = ZERO
      NN = N * INCX
C
C  FOR SINGLE OR DOUBLE PRECISION SET HITEST = CUTHI/N
      HITEST = CUTHI/FLOAT( N )
      XMAX = ZERO
C                        BEGIN MAIN LOOP
      I = 1
   20 CONTINUE
C
C                        PHASE 1.  SUM IS ZERO
C
      IF( SX(I) .EQ. ZERO) THEN
         CONTINUE
C
C                        PHASE 2.  SUM IS SMALL
C                        SCALE TO AVOID DESTRUCTIVE UNDERFLOW.
C
      ELSE IF( ABS(SX(I)) .LE. CUTLO) THEN
         IF( ABS(SX(I)) .GT. XMAX) THEN
            SUM = ONE + SUM * (XMAX / SX(I))**2
            XMAX = ABS(SX(I))
         ELSE
            SUM = SUM + (SX(I)/XMAX)**2
         ENDIF
C
C                        PHASE 3.  SUM IS MID-RANGE.  NO SCALING.
C
      ELSE
         SUM = (SUM * XMAX) * XMAX
         DO 95 J =I,NN,INCX
            IF(ABS(SX(J)) .GE. HITEST) GO TO 100
            SUM = SUM + SX(J)**2
   95    CONTINUE
         SNRM2 = SQRT( SUM )
         GO TO 300
C
C                        PHASE 4.
C                        SUM IS LARGE.  SCALE TO AVOID OVERFLOW.
C
  100    SUM = (SUM / SX(J)) / SX(J)
         XMAX = ABS(SX(J))
         DO 120 I=J,NN,INCX
            IF( ABS(SX(I)) .GT. XMAX) THEN
               SUM = ONE + SUM * (XMAX / SX(I))**2
               XMAX = ABS(SX(I))
            ELSE
               SUM = SUM + (SX(I)/XMAX)**2
            ENDIF
  120    CONTINUE
      ENDIF
C
C                        END OF MAIN LOOP.
  200 CONTINUE
      I = I + INCX
      IF (I .LE. NN) GO TO 20
C
C              COMPUTE SQUARE ROOT AND ADJUST FOR SCALING.
C
      SNRM2 = XMAX * SQRT(SUM)
  300 CONTINUE
C
C *** 6 Jan 84, J. Fried :
C     This next line of FORTRAN was added to prevent values below
C     MS-FORTRAN's specified minimum from being returned.  Such
C     numbers have appeared during testing of the SQRDC routine
C     (test #9 in SQ).  The value of SMALL was taken from the
C     MS-FORTRAN manual.  Remember, this only applies to PC/MS-FORTRAN.
C
C     IF ( SNRM2 .LE. SMALL ) SNRM2 = SMALL
999   RETURN
      END
