c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
C***********************************************************************
C Copyright 2001, Allied Geophysics, Inc. All Rights Reserved          *
C***********************************************************************
C Portions of this code and/or subroutines  used by this code are      *
C protected by the following copyright(s):                             *
C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c -----------------  Main Routine -----------------------
c
c mapfill - fill holes in a map that are flagged with emask
c
c  Program Description:
c
c   mapfill does simple linear interpolation in 2 directions to fill
c     gaps in a map. The result is the average solution of the 2
c     interpolations.
c
c  Program History:
c
c    Jan 21, 2002 - add implicit none and indicated declarations --- PGAG
c    Dec 27, 2001 - Add logic preventing interp into large data gaps
c    Oct 30, 2001 - Adopted emask as name for embedded mask
c    Oct 27, 2001 - Original version
c
c get machine dependent parameters 

      implicit none

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c declare standard USP variables 
      integer     ilhd(3*SZLNHD)
     
      integer     luin , luout
      integer     obytes, lbytes, lbyout, nbytes
      integer     n1,n2,n3,SmpInt,Dz1000,Dx1000,Dy1000
      integer     argis, JERR
      real        UnitSc,d1,d2,d3

      character   ntap*255, otap*255, name*7

      logical     lmask, verbose

c Program Specific - dynamic memory variables
      integer TrcSize,RecSize,HdrSize,Ar1Size,Ar2Size
      integer TotalMem, npt_mask
      integer ier1,ier2,ier3,ier4,ier5,ier6,ier7,ier8
      integer ier9,iera,ierb,ierc,ierd,iere,ierf,ierg,abort

      integer Trace(2), Headers(2)
      integer rmask_i1(2),rmask_i2(2)
      real    Pts_in1(2),Pts_out1(2),Coord_in1(2),Coord_out1(2)
      real    Pts_in2(2),Pts_out2(2),Coord_in2(2),Coord_out2(2)
      real    RecIn(2),RecInterp1(2),RecInterp2(2),RecOut(2)

      pointer (ptr_Trace,Trace)
      pointer (ptr_Headers,Headers)
      pointer (ptr_rmask_i1,rmask_i1)
      pointer (ptr_rmask_i2,rmask_i2)
      pointer (ptr_Pts_in1,Pts_in1)
      pointer (ptr_Pts_out1,Pts_out1)
      pointer (ptr_Coord_in1,Coord_in1)
      pointer (ptr_Coord_out1,Coord_out1)
      pointer (ptr_Pts_in2,Pts_in2)
      pointer (ptr_Pts_out2,Pts_out2)
      pointer (ptr_Coord_in2,Coord_in2)
      pointer (ptr_Coord_out2,Coord_out2)
      pointer (ptr_RecIn,RecIn)
      pointer (ptr_RecInterp1,RecInterp1)
      pointer (ptr_RecInterp2,RecInterp2)
      pointer (ptr_RecOut,RecOut)

c Program Specific - static memory variables
      real    emask, d1u,d2u, radius

      integer hdr_index, tr_index
      integer irec, itrc

c Initialize variables
      data abort/0/
      data name/"MAPFILL"/

c give command line help if requested
      if ( argis('-?')    .gt. 0 .or. 
     :     argis('-h')    .gt. 0 .or.
     :     argis('-help') .gt. 0 ) then
        call help(name)
        stop
      endif

c open printout file
#include <f77/open.h>

c get command line input parameters
      call cmdln (ntap,otap, d1u,d2u, radius, emask, name, verbose)

c Open the input and output files
      call getln(luin,ntap,'r',0)
      call getln(luout,otap,'w',1)

c read input line header and save certain parameters
      call rtape(luin,ilhd,lbytes)
      if(lbytes.eq.0)then
        write(LER,*)name,': no line header on input file',ntap
        write(LER,*)'FATAL'
        stop
      endif

c print HLH to printout file
      call hlhprt (ilhd, lbytes, name, 4, LERR)

c save out hlh and line header
      call savhlh (ilhd, lbytes, lbyout)
      call wrtape (luout, ilhd, lbyout)

c get axis deltas
      call saver(ilhd, 'UnitSc', UnitSc, LINHED)
      call saver(ilhd, 'SmpInt', SmpInt, LINHED)
      call saver(ilhd, 'Dz1000', Dz1000, LINHED)
      call saver(ilhd, 'Dx1000', Dx1000, LINHED)
      call saver(ilhd, 'Dy1000', Dy1000, LINHED)

      d1 = real(Dx1000)/1000.
      d2 = real(Dy1000)/1000.
      d3 = real(Dz1000)/1000.
      if (d1.eq.0) d1=SmpInt*UnitSc
      if (d1u.ne.0.0) d1 = d1u
      if (d2u.ne.0.0) d2 = d2u

      if(d1.eq.0) d1=1.0
      if(d2.eq.0) d2=1.0

c get sizes for dynamic allocation
      call saver(ilhd, 'NumSmp', n1, LINHED)
      call saver(ilhd, 'NumTrc', n2, LINHED)
      call saver(ilhd, 'NumRec', n3, LINHED)

c     for traces
      TrcSize = SZTRHD + n1*SZSMPD

c     for arrays
      RecSize = n1*n2*SZSMPD
      HdrSize = n2*SZTRHD
      Ar1Size = n1*SZSMPD
      Ar2Size = n2*SZSMPD

      TotalMem = TrcSize + 4*Ar1Size + 4*Ar2Size + 4*RecSize + HdrSize

c     number bytes on input/output
      obytes = TrcSize

c now get info for radius-based interpolation mask
      if (radius .lt. n1*d1/2.0 .and.
     :    radius .lt. n2*d2/2.0) then
        lmask = .true.
        call count_mask_pts(d1,d2,radius,npt_mask)
        TotalMem = TotalMem + 2*npt_mask*SZSMPD
      else
        lmask = .false.
      endif

c dynamic memory allocation:  
      call galloc(ptr_Trace,      TrcSize, ier1, abort)
      call galloc(ptr_Headers,    HdrSize, ier2, abort)
      call galloc(ptr_Pts_in1,    Ar1Size, ier3, abort)
      call galloc(ptr_Pts_out1,   Ar1Size, ier4, abort)
      call galloc(ptr_Coord_in1,  Ar1Size, ier5, abort)
      call galloc(ptr_Coord_out1, Ar1Size, ier6, abort)
      call galloc(ptr_Pts_in2,    Ar2Size, ier7, abort)
      call galloc(ptr_Pts_out2,   Ar2Size, ier8, abort)
      call galloc(ptr_Coord_in2,  Ar2Size, ier9, abort)
      call galloc(ptr_Coord_out2, Ar2Size, iera, abort)
      call galloc(ptr_RecIn,      RecSize, ierb, abort)
      call galloc(ptr_RecInterp1, RecSize, ierc, abort)
      call galloc(ptr_RecInterp2, RecSize, ierd, abort)
      call galloc(ptr_RecOut,     RecSize, iere, abort)
      if (lmask) then
        call galloc(ptr_rmask_i1, npt_mask*SZSMPD, ierf, abort)
        call galloc(ptr_rmask_i2, npt_mask*SZSMPD, ierg, abort)
      else
        ierf = 0
        ierg = 0
      endif
    
      if (ier1.ne.0 .or. ier2.ne.0 .or. ier3.ne.0 .or. ier4.ne.0 .or.
     :    ier5.ne.0 .or. ier6.ne.0 .or. ier7.ne.0 .or. ier8.ne.0 .or.
     :    ier9.ne.0 .or. iera.ne.0 .or. ierb.ne.0 .or. ierc.ne.0 .or.
     :    ierd.ne.0 .or. iere.ne.0 .or. ierf.ne.0 .or. ierg.ne.0) then
        write(LERR,*)' '
        write(LERR,*)
     :    'Unable to allocate workspace: ',TotalMem,' bytes'
        write(LERR,*)'FATAL'
        write(LERR,*)' '
        write(LER,*)' '
        write(LER,*)
     :    'Unable to allocate workspace: ',TotalMem,' bytes'
        write(LER,*)'FATAL'
        write(LER,*)' '
        stop
      else
        write(LERR,*)' '
        write(LERR,*)'Allocated ',TotalMem,' bytes of workspace'
        write(LERR,*)' '
      endif

c initialize memory
      call vclr (Trace, 1, ITRWRD+n1)
      call vclr (Headers, 1, ITRWRD*n2)
      call vclr (Pts_in1, 1, n1)
      call vclr (Pts_out1, 1, n1)
      call vclr (Coord_in1, 1, n1)
      call vclr (Coord_out1, 1, n1)
      call vclr (Pts_in2, 1, n2)
      call vclr (Pts_out2, 1, n2)
      call vclr (Coord_in2, 1, n2)
      call vclr (Coord_out2, 1, n2)
      call vclr (RecIn, 1, n1*n2)
      call vfill (emask, RecInterp1, 1, n1*n2)
      call vfill (emask, RecInterp2, 1, n1*n2)
      call vfill (emask, RecOut, 1, n1*n2)

c this sets up indices relative to a current grid location that
c describe a circle with 'radius' around that grid loc.
      if (lmask) then
        call set_mask_pts(d1,d2,radius,rmask_i1,rmask_i2,npt_mask)
      endif

c verbose output of all pertinent information before processing begins
      call verbal (ntap,otap, emask, verbose, n1,n2,n3, d1,d2,radius)

c BEGIN PROCESSING

c Loop over records processing each independantly
      do irec = 1,n3

c Loop over input traces loading a record and saving headers
        tr_index = 1 - n1
        hdr_index = 1 - ITRWRD
        do itrc = 1,n2

          nbytes = 0
          call rtape(luin, Trace, nbytes)
          if(nbytes.eq.0)then
            write(LERR,*)name,
     :        ': Read error on input at record = ',irec,' trace= ',itrc
            stop
          endif
c         split data and headers
          tr_index = tr_index + n1
          hdr_index = hdr_index + ITRWRD
          call vmov(Trace, 1, Headers(hdr_index), 1, ITRWRD)
          call vmov(Trace(ITHWP1), 1, RecIn(tr_index), 1, n1)
        enddo

c process the record
        call fillmap (RecIn,n1,n2, RecOut, emask,
     :                rmask_i1,rmask_i2,npt_mask,lmask,
     :                RecInterp1,RecInterp2,
     :                Pts_in1,Pts_out1,Coord_in1,Coord_out1,
     :                Pts_in2,Pts_out2,Coord_in2,Coord_out2)

c write the output record
        tr_index = 1 - n1
        hdr_index = 1 - ITRWRD
        do itrc = 1,n2

c         First get the ptrs into the processed data and header arrays
          tr_index = tr_index + n1
          hdr_index = hdr_index + ITRWRD

c         Now glue headers and data together into an output vector
          call vmov (Headers(hdr_index), 1, Trace(1), 1, ITRWRD)
          call vmov (RecOut(tr_index), 1, Trace(ITHWP1), 1, n1)

c         Now write the trace
          call wrtape (luout, Trace, obytes)

        enddo

c end of loop over axis 3 (records)
      enddo


c Close data files 
      call lbclos (luin)
      call lbclos (luout)
      write(LERR,*)name,': Normal Termination'
      write(LER,*)name,': Normal Termination'
      stop

      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c provide terse online help [detailed help goes in man page]
      subroutine help(name)

      character name*(*)

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for ',name
      write(LER,*)'    fill holes in a map that are flagged with emask'
      write(LER,*)' '
      write(LER,*)'Input......................................... (def)'
      write(LER,*)' '
      write(LER,*)'-N[]  -- input data set                     (stdin:)'
      write(LER,*)'-O[]  -- output data set                   (stdout:)'
      write(LER,*)' '
      write(LER,*)'-radius[]-- radius for interpolation into    (1e+37)'
      write(LER,*)'            large data gaps'
      write(LER,*)'-d1[] -- delta for axis 1              (1e-3*Dz1000)'
      write(LER,*)'-d2[] -- delta for axis 2              (1e-3*Dx1000)'
      write(LER,*)' '
      write(LER,*)'-emask[] -- value for samples that are      (-1e+37)'
      write(LER,*)'            masked'
      write(LER,*)' '
      write(LER,*)'-V    -- Verbose printout                  (.false.)'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'  mapfill -N[] -O[] -d1[] -d2[] -radius[]'
      write(LER,*)'          -emask[] -V'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c pick up command line arguments 
      subroutine cmdln (ntap,otap, d1u,d2u,radius, emask, name, verbose)

      implicit none

#include <f77/iounit.h>

      integer   argis
      real      d1u,d2u,radius,emask
      character ntap*(*), otap*(*), name*(*)
      logical   verbose

      call argstr('-N', ntap, ' ', ' ') 
      call argstr('-O', otap, ' ', ' ') 
      call argr4('-d1', d1u, 0.0, 0.0)
      call argr4('-d2', d2u, 0.0, 0.0)
      call argr4('-radius', radius, 1.0e+37, 1.0e+37)
      call argr4('-emask', emask, -1.0e+37, -1.0e+37)

      verbose= .false.
      verbose= (argis('-V').gt.0)

c alternate parameter retrieval

      call argstr('in=', ntap, ntap, ntap) 
      call argstr('out=', otap, otap, otap) 
      call argr4('d1=', d1u, d1u, d1u)
      call argr4('d2=', d2u, d2u, d2u)
      call argr4('radius=', radius, radius, radius)
      call argr4('emask=', emask, emask, emask)
      if (.not.verbose) verbose = (argis('verbose=').gt.0)

c check for extraneous arguments and abort if found to
c catch all manner of user typo's
      call xtrarg ( name, ler, .FALSE., .FALSE. )
      call xtrarg ( name, lerr, .FALSE., .TRUE. )

      return
      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c verbal printout of pertinent program particulars
      subroutine verbal (ntap,otap, emask, verbose, n1,n2,n3,
     :                   d1,d2,radius)

      implicit none

#include <f77/iounit.h>

      character  ntap*(*), otap*(*)
      real       emask
      integer    n1,n2,n3
      real       d1,d2,radius
      logical    verbose
      integer    nblen

      write(LERR,*)' '
      write(LERR,*)' File names for input and output'
      write(LERR,*)' '
      if (ntap(1:1) .ne. ' ') then
        write(LERR,*)'   input data set name   = ',ntap(1:nblen(ntap))
      else
        write(LERR,*)'   input data is on         stdin:'
      endif
      if (otap(1:1) .ne. ' ') then
        write(LERR,*)'   output data set name  = ',otap(1:nblen(otap))
      else
        write(LERR,*)'   output data is on        stdout:'
      endif
      write(LERR,*)' '
      write(LERR,*)'      Number of samples   = ',n1
      write(LERR,*)'      Number of traces    = ',n2
      write(LERR,*)'      Number of records   = ',n3
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Other command line parameters'
      write(LERR,*)' '
      write(LERR,*)'   Axis 1 delta           = ',d1
      write(LERR,*)'   Axis 2 delta           = ',d2
      write(LERR,*)'   Radius for fill mask   = ',radius
      write(LERR,*)'   Embedded mask value    = ',emask
      if (verbose) then
        write(LERR,*)' '
        write(LERR,*)'   Verbose printout requested'
      endif
      write(LERR,*)' '

      return
      end
