c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
C***********************************************************************
C Copyright 2001, Allied Geophysics, Inc. All Rights Reserved          *
C***********************************************************************
C Portions of this code and/or subroutines  used by this code are      *
C protected by the following copyright(s):                             *
C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c -----------------  Main Routine -----------------------
c
c  mapreplace - Replace selected values in a 3D volume with some
c    other function.
c
c  Program Description:
c
c     Replace selected values in a 3D volume with some other function.
c     The values are selected by one or two maps. An upper map
c     determines the first sample of the volume to replace, a lower map
c     determined the last. Maps and input data must be conformable
c     Functions are initially limitted to a linear function in Z hung
c     from a constant datum. Future functions might include
c       V = V0 + z*k           (Already here)
c       V = V0 + (z-datum)*k   where datum can be variable
c       V = V from secondary input volume
c     Others...
c
c  Program Changes:
c    Jan 21, 2002 - added implicit none and associated declarations -- PGAG
c    Dec 28, 2001 - minor cleanup of code and man page
c    Nov 23, 2001 - added access to license info through command line
c    Oct 30, 2001 - adopted emask as name for embedded mask
c    Oct 24, 2001 - original version
c

c get machine dependent parameters 

      implicit none

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c dimension standard USP variables 

      integer     itr_vol(3*SZLNHD)
      integer     itr_map1(3*SZLNHD)
      integer     itr_map2(3*SZLNHD)
     
      integer     lbytes_vol, lbytes_map1, lbytes_map2
      integer     nbytes, lbyout, obytes
      integer     luin,lumap1,lumap2, luout
      integer     argis, JERR

      character   map1file*255, map2file*255
      character   ntap*(255), otap*255, name*10

      logical     verbos

c Program Specific _ dynamic memory variables

      integer MapSize, TrcSize, InitialMem
      integer errcd1, errcd2, errcd3, abort

      integer Map1(2), Map2(2)
      real    Trc_in(2)

      pointer (ptr_Map1, Map1)
      pointer (ptr_Map2, Map2)
      pointer (ptr_Trc_in, Trc_in)

c Program Specific static memory variables

      integer tr_index, i2,i3, im,it, id1, iz1,iz2, iz_here
      integer n1_vol,n2_vol,n3_vol, n1_map1,n2_map1, n1_map2,n2_map2
      integer nblen, map_index
      real    d1, vbias,vslope,z0,dz, z1,z2, emask
      logical lfatal, lmap1,lmap2

c Initialize variables

      data abort/0/
      data name/"MAPREPLACE"/
      data lfatal/.false./

c
c give command line help if requested
c
      if ( argis('-?')    .gt. 0 .or.
     :     argis('-h')    .gt. 0 .or.
     :     argis('-help') .gt. 0 ) then
        call help(name)
        stop
      endif

c
c open printout file
c
#include <f77/open.h>

c
c get command line input parameters
c
      call cmdln ( ntap, otap, map1file, map2file,
     :             vbias,vslope,z0,dz, z1,z2,
     :             emask, name, verbos )

      lmap1 = .true.
      if (map1file(1:1).eq.' ') lmap1 = .false.
      lmap2 = .true.
      if (map2file(1:1).eq.' ') lmap2 = .false.

c
c open input and output files
c
      call getln(luin, ntap,'r', 0)
      if (lmap1) call getln(lumap1, map1file,'r', 0)
      if (lmap2) call getln(lumap2, map2file,'r', 0)
      call getln(luout, otap,'w', 1)

c
c read the Line header info from inputs and check for inconsistencies
c
      call rtape(luin,itr_vol,lbytes_vol)
      if(lbytes_vol.eq.0)then
        write(LER,*)name,
     :    ': no line header on input file ',ntap(1:nblen(ntap))
        write(LER,*)'FATAL'
        lfatal = .true.
      endif

      call saver(itr_vol, 'NumSmp', n1_vol, LINHED)
      call saver(itr_vol, 'NumTrc', n2_vol, LINHED)
      call saver(itr_vol, 'NumRec', n3_vol, LINHED)
      call saver(itr_vol, 'Dz1000', id1, LINHED)
      d1 = float(id1)*0.001
      if (dz.gt.0.0) d1 = dz

      if(lmap1) then
        call rtape(lumap1,itr_map1,lbytes_map1)
        if(lbytes_map1.eq.0)then
          write(LER,*)name,
     :     ': no line header on input file ',map1file(1:nblen(map1file))
          write(LER,*)'FATAL'
          lfatal = .true.
        endif
        call saver(itr_map1, 'NumSmp', n1_map1, LINHED)
        call saver(itr_map1, 'NumTrc', n2_map1, LINHED)
      else
        n1_map1 = n2_vol
        n2_map1 = n3_vol
      endif

      if(lmap2) then
        call rtape(lumap2,itr_map2,lbytes_map2)
        if(lbytes_map2.eq.0)then
          write(LER,*)name,
     :     ': no line header on input file ',map2file(1:nblen(map2file))
          write(LER,*)'FATAL'
          lfatal = .true.
        endif
        call saver(itr_map2, 'NumSmp', n1_map2, LINHED)
        call saver(itr_map2, 'NumTrc', n2_map2, LINHED)
      else
        n1_map2 = n2_vol
        n2_map2 = n3_vol
      endif

      if (lfatal) stop

c     inconsistency check
      if (n2_vol.ne.n1_map1 .or. n3_vol.ne.n2_map1) then
        write(LER,*)name,': File ',map1file(1:nblen(map1file)),
     :    ' is not conformable with input volume ',ntap(1:nblen(ntap))
        write(LER,*)' '
        write(LER,*)'   ',ntap(1:nblen(ntap)),' parameters:'
        write(LER,*)'           NumSmp= ',n1_vol
        write(LER,*)'           NumTrc= ',n2_vol
        write(LER,*)'           NumRec= ',n3_vol
        write(LER,*)' '
        write(LER,*)'   ',map1file(1:nblen(map1file)),' parameters:'
        write(LER,*)'           NumSmp= ',n1_map1
        write(LER,*)'           NumTrc= ',n2_map1
        write(LER,*)'FATAL'
        lfatal = .true.
      endif

      if (n2_vol.ne.n1_map2 .or. n3_vol.ne.n2_map2) then
        write(LER,*)name,': File ',map2file(1:nblen(map2file)),
     :    ' is not conformable with input volume ',ntap(1:nblen(ntap))
        write(LER,*)' '
        write(LER,*)'   ',ntap(1:nblen(ntap)),' parameters:'
        write(LER,*)'           NumSmp= ',n1_vol
        write(LER,*)'           NumTrc= ',n2_vol
        write(LER,*)'           NumRec= ',n3_vol
        write(LER,*)' '
        write(LER,*)'   ',map2file(1:nblen(map2file)),' parameters:'
        write(LER,*)'           NumSmp= ',n1_map2
        write(LER,*)'           NumTrc= ',n2_map2
        write(LER,*)'FATAL'
        lfatal = .true.
      endif

      if(lmap1.and.lmap2) then
        if (n1_map1.ne.n1_map2 .or. n2_map1.ne.n2_map2) then
          write(LER,*)name,': File ',map2file(1:nblen(map2file)),
     :      ' is not conformable with file ',map2file(1:nblen(map2file))
          write(LER,*)' '
          write(LER,*)'   ',map1file(1:nblen(map1file)),' parameters:'
          write(LER,*)'           NumSmp= ',n1_map1
          write(LER,*)'           NumTrc= ',n2_map1
          write(LER,*)' '
          write(LER,*)'   ',map2file(1:nblen(map2file)),' parameters:'
          write(LER,*)'           NumSmp= ',n1_map2
          write(LER,*)'           NumTrc= ',n2_map2
          write(LER,*)'FATAL'
          lfatal = .true.
        endif
      endif

      if (lfatal) stop

c
c print HLH to printout file (dump headers from all input files)
c
      call hlhprt (itr_vol, lbytes_vol, name, 4, LERR)
      if (lmap1) call hlhprt (itr_map1, lbytes_map1, name, 4, LERR)
      if (lmap2) call hlhprt (itr_map2, lbytes_map2, name, 4, LERR)

c
c number output bytes in a trace
c
      obytes = SZTRHD + SZSMPD * n1_vol

c
c save out hlh and line header
c
      call savhlh (itr_vol, lbytes_vol, lbyout)
      call wrtape (luout, itr_vol, lbyout)

c
c verbose output of all pertinent information before processing begins
c
      call verbal
     :       ( ntap,otap,map1file,map2file,
     :         vbias,vslope,z0,d1, z1,z2,
     :         n1_vol,n2_vol,n3_vol, emask, verbos )

c
c dynamic memory allocation:
c
      MapSize = n1_map1*n2_map1*SZSMPD
      TrcSize = SZTRHD + SZSMPD*(max(n1_vol,n1_map1))
      InitialMem = 2*MapSize+TrcSize

      call galloc (ptr_Map1,MapSize,errcd1,abort)
      call galloc (ptr_Map2,MapSize,errcd2,abort)
      call galloc (ptr_Trc_in,TrcSize,errcd3,abort)
    
      if (errcd1.ne.0 .or. errcd2.ne.0 .or. errcd3.ne.0 ) then

        write(LERR,*)' '
        write(LERR,*)
     :    name,': Unable to allocate workspace:',InitialMem,' bytes'
        write(LERR,*)name,': ABNORMAL Termination'
        write(LERR,*)' '

        write(LER,*)' '
        write(LER,*)
     :    name,': Unable to allocate workspace:',InitialMem,' bytes'
        write(LER,*)name,': ABNORMAL Termination'
        write(LER,*)' '

        call lbclos (luin)
        if(map1file.ne.' ') call lbclos (lumap1)
        if(map2file.ne.' ') call lbclos (lumap2)
        call lbclos (luout)
        stop

      else

        write(LERR,*)' '
        write(LERR,*)name,': Allocating workspace:',InitialMem,' bytes'
        write(LERR,*)' '

      endif

c
c set up default values for map surfaces
c
      if (.not.lmap1 .and. z1.lt.z0) z1 = z0
      if (.not.lmap2 .and. z2.lt.z0) z2 = z0 + d1*(n1_vol-1)

c     These are only used in absence of input maps
      iz1 = int((z1-z0)/d1) + 1
      iz2 = int((z2-z0)/d1) + 1
      iz1 = min(n1_vol,max(1,iz1))
      iz2 = min(n1_vol,max(1,iz2))

c
c adjust vslope to be index based
c
      vslope = vslope*d1

c
c  Load Maps into Memory
c  Maps are stored in integer arrays of input volume sample number
c
      tr_index = 1 - n1_map1
      DO i3 = 1, n2_map1

        tr_index = tr_index + n1_map1

        if (lmap1) then
          nbytes = 0
          call rtape(lumap1, Trc_in, nbytes)
          if(nbytes .eq. 0) then
            write(LERR,*)name,': Premature EOF on Map1 at trace ',i3
            lfatal = .true.
          else
            im=tr_index-1
            do it = ITHWP1,ITHWP1+n1_map1-1
              im = im+1
              if (Trc_in(it) .eq. emask) then
                Map1(im) = -1
              else
                iz_here = int((Trc_in(it)-z0)/d1) + 2
                if ((iz_here-1)*d1+z0 .eq. Trc_in(it)) iz_here=iz_here-1
                Map1(im) = min(n1_vol,max(1,iz_here))
              endif
            enddo
          endif
        else
          do im = tr_index,tr_index-1+n1_map1
            Map1(im) = iz1
          enddo
        endif

        if (lmap2) then
          nbytes = 0
          call rtape(lumap2, Trc_in, nbytes)
          if(nbytes .eq. 0) then
            write(LERR,*)name,': Premature EOF on Map2 at trace ',i3
            lfatal = .true.
          else
            im=tr_index-1
            do it = ITHWP1,ITHWP1+n1_map1-1
              im = im+1 
              if (Trc_in(it) .eq. emask) then
                Map2(im) = -1
              else
                iz_here = int((Trc_in(it)-z0)/d1) + 1
                if ((iz_here-1)*d1+z0 .eq. Trc_in(it)) iz_here=iz_here-1
                Map2(im) = min(n1_vol,max(1,iz_here))
              endif
            enddo
          endif
        else
          do im = tr_index,tr_index-1+n1_map1
            Map2(im) = iz2
          enddo
        endif

        if (lfatal) then
          call lbclos (luin)
          if(lmap1) call lbclos (lumap1)
          if(lmap2) call lbclos (lumap2)
          call lbclos (luout)
          stop
        endif

      ENDDO

c
c Maps are here.
c Loop over the input volume replacing values as we go
c

      if(verbos) then
        write(LERR,*)' '
        write(LERR,*)name,': Starting to process ',n3_vol,
     :               ' records with ',n2_vol,' traces.'
        write(LERR,*)' '
      endif

      map_index = 0
      do i3 = 1,n3_vol
        do i2 = 1,n2_vol

c         read a trace
          nbytes = 0
          call rtape(luin,Trc_in,nbytes)
          if(nbytes.eq.0) then
            write(LERR,*)name,': Premature EOF on input at record ',i3,
     :                   ' trace ',i2
            call lbclos (luin)
            if(lmap1) call lbclos (lumap1)
            if(lmap2) call lbclos (lumap2)
            call lbclos (luout)
            stop
          endif

c         replace samples
          map_index = map_index + 1
          call replace_vz
     :           ( Trc_in(ITHWP1),n1_vol,
     :             Map1(map_index),Map2(map_index),
     :             vbias,vslope )

c         write the result
          call wrtape (luout,Trc_in,obytes)

        enddo

        if(verbos) then
          write(LERR,*)name,':    Done with record ',i3,' of ',n3_vol
        endif

      enddo

c close data files 

      call lbclos (luin)
      if(lmap1) call lbclos (lumap1)
      if(lmap2) call lbclos (lumap2)
      call lbclos (luout)
      write(LERR,*)name,': Normal Termination'
      write(LER,*)name,': Normal Termination'
      write(LER,*)' '

      stop
      end

c -----------------  Subroutine -----------------------
c provide terse online help [detailed help goes in man page]

      subroutine help(name)

      character  name*(*)
      integer nblen, len1

#include <f77/iounit.h>

      len1 = nblen(name)

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for ',name(1:len1)
      write(LER,*)'   Replace selected values in a 3D volume with some'
      write(LER,*)'             other function'
      write(LER,*)' '
      write(LER,*)'Input..................................... (default)'
      write(LER,*)' '
      write(LER,*)'-N[]     -- Input volume                    (stdin:)'
      write(LER,*)'-O[]     -- Output volume                  (stdout:)'
      write(LER,*)'-map1[]  -- Upper map surface            (first smp)'
      write(LER,*)'-map2[]  -- Lower map surface             (last smp)'
      write(LER,*)' '
      write(LER,*)'-v0[]    -- Bias  for v0 + z*k function     (3000.0)'
      write(LER,*)'-k[]     -- Slope for v0 + z*k function        (0.0)'
      write(LER,*)'-z0[]    -- Z axis origin for v0 + z*k         (0.0)'
      write(LER,*)'-dz[]    -- Delta Z override           (1e-3*Dz1000)'
      write(LER,*)' '
      write(LER,*)'-z1[]    -- Constant depth for map1           (none)'
      write(LER,*)'-z2[]    -- Constant depth for map2           (none)'
      write(LER,*)' '
      write(LER,*)'-emask[] -- Value to indicate undefined     (-1e+37)'
      write(LER,*)'              regions in maps.'
      write(LER,*)' '
      write(LER,*)'-License -- Print license info and quit    (.false.)'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'    mapreplace -N[] -map1[] -map2[] -O[]'
      write(LER,*)'               -v0[] -k[] -z0[] -dz[] -z1[] -z2[]'
      write(LER,*)'               -emask[] -V'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end

c -----------------  Subroutine -----------------------
c pick up command line arguments 

      subroutine cmdln ( ntap, otap, map1file, map2file,
     :                   vbias,vslope,z0,dz, z1,z2,
     :                   emask, name, verbos )

      implicit none

#include <f77/iounit.h>

      integer    argis
      real       vbias,vslope,z0,dz, z1,z2, emask
      character  map1file*(*),map2file*(*)
      character  ntap*(*), otap*(*), name*(*)
      logical    verbos

      call argstr ('-N', ntap, ' ', ' ') 
      call argstr ('-O', otap, ' ', ' ') 
      call argstr ('-map1', map1file, ' ', ' ') 
      call argstr ('-map2', map2file, ' ', ' ') 

      call argr4  ('-v0', vbias, 3000.0, 3000.0)
      call argr4  ('-k', vslope,    0.0,    0.0)
      call argr4  ('-z0',    z0,    0.0,    0.0)
      call argr4  ('-dz',    dz,    0.0,    0.0)

      call argr4  ('-z1', z1, z0-1.0, z0-1.0)
      call argr4  ('-z2', z2, z0-1.0, z0-1.0)

      call argr4  ('-emask',emask,-1.0e-37,-1.0e-37)

      verbos = (argis('-V') .gt. 0)

c
c now look for values passed in with alternative style
c
      call argstr ('in=', ntap, ntap, ntap) 
      call argstr ('out=', otap, otap, otap) 
      call argstr ('map1=', map1file, map1file, map1file) 
      call argstr ('map2=', map2file, map2file, map2file) 
      call argr4  ('v0=', vbias, vbias, vbias)
      call argr4  ('k=', vslope, vslope, vslope)
      call argr4  ('z0=', z0, z0, z0)
      call argr4  ('dz=', dz, dz, dz)
      call argr4  ('z1=', z1, z1, z1)
      call argr4  ('z2=', z2, z2, z2)
      call argr4  ('emask=', emask, emask, emask)
      if (.not.verbos) verbos = (argis('verbose=') .gt. 0)

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

      call xtrarg (name, LER,  .FALSE., .FALSE.)
      call xtrarg (name, LERR, .FALSE., .TRUE.)

      return
      end

c -----------------  Subroutine -----------------------

c verbal printout of pertinent program particulars

      subroutine verbal
     :       ( ntap,otap,map1file,map2file,
     :         vbias,vslope,z0,d1, z1,z2,
     :         n1,n2,n3, emask, verbos )

      implicit none

#include <f77/iounit.h>

      character ntap*(*), otap*(*), map1file*(*), map2file*(*)
      integer   n1,n2,n3
      real      vbias,vslope,z0,d1, z1,z2, emask
      logical   verbos
      integer   len1,len2,len3,len4
      integer   nblen

      len1 = nblen(ntap)
      len2 = nblen(otap)
      len3 = nblen(map1file)
      len4 = nblen(map2file)

      write(LERR,*)' '
      write(LERR,*)' Input Line Header Parameters'
      write(LERR,*)' '
      write(LERR,*) ' Input data volume       = ', ntap(1:len1)
      write(LERR,*) '   Number of samples     = ', n1
      write(LERR,*) '   Number of traces      = ', n2
      write(LERR,*) '   Number of records     = ', n3
      write(LERR,*) '   Sample axis origin    = ', z0
      write(LERR,*) '   Sample increment      = ', d1
      write(LERR,*)' '
      if (len3.gt.0) then
        write(LERR,*) ' Upper map               = ', map1file(1:len3)
        write(LERR,*) '   Number of samples     = ', n2
        write(LERR,*) '   Number of traces      = ', n3
        write(LERR,*)' '
      elseif(z1.ge.z0) then
        write(LERR,*) ' Upper map has constant Z = ',z1
        write(LERR,*)' '
      else
        write(LERR,*) ' Upper map defaulted to Z0 = ',z0
        write(LERR,*)' '
      endif
      if (len4.gt.0) then
        write(LERR,*) ' Lower map               = ', map2file(1:len4)
        write(LERR,*) '   Number of samples     = ', n2
        write(LERR,*) '   Number of traces      = ', n3
        write(LERR,*)' '
      elseif(z2.ge.z0) then
        write(LERR,*) ' Lower map has constant Z = ',z2
        write(LERR,*)' '
      else
        write(LERR,*) ' Lower map defaulted to Z = ',z0+d1*(n1-1)
        write(LERR,*)' '
      endif
      write(LERR,*)' '
      write(LERR,*) ' Output volume           = ', otap(1:len2)
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*) ' Bias for V(z) fxn      v0 = ', vbias
      write(LERR,*) ' Slope for V(z) fxn      k = ', vslope
      write(LERR,*)' '
      if ( verbos ) then
        write(LERR,*) ' verbose printout requested'
        write(LERR,*)' '
      endif
      write(LERR,*)'================================================== '
      write(LERR,*)' '

      return
      end
