/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  This file contains the functions to get the
  seismic data from the Cray.

  Written by Douglas Horn.
*/

#include <signal.h>
#include <fcntl.h>
#include <stdio.h>

extern int verbose;

static FILE *out_pipe,*in_pipe;
static int sk_client[2],sk_server[2];
static float scalar;
static int offset,seq_number=0;


/*
  This function performs an "rsh" to start
  the server program.
*/
amoco_start_server(hostname,login,command,machine)
char *hostname,*login,*command,*machine;
{
 
  signal(SIGPIPE,SIG_IGN);

  if ((pipe(sk_server)<0)||(pipe(sk_client)<0)) {
    fprintf(stderr,"Error openning connection.\n");
    exit(-1);
  }

  if (fork()==0) {
    if (sk_server[0]!=0) {
      close(0);
      if (dup(sk_server[0])!=0) {
	fprintf(stderr,"Error openning connection.\n");
	exit(-1);
      }
      close(sk_server[0]);
    }
    if (sk_client[1]!=1) {
      close(1);
      if (dup(sk_client[1])!=1) {
	fprintf(stderr,"Error openning connection.\n");
	exit(-1);
      }
      close(sk_client[1]);
    }
    close(sk_server[1]);
    close(sk_client[0]);

    if (hostname[0]=='=') {
      if (verbose)
	fprintf(stderr,"Starting local data server %s.\n",command);
      execlp(command,command,machine,(char *)0);
    } else {
      if (verbose)
	fprintf(stderr,"Starting remote data server.\n");
      execl("/usr/ucb/rsh",hostname,"-l",login,command,machine,(char *)0);
    }
  }

  close(sk_server[0]);
  close(sk_client[1]);

  out_pipe=fdopen(sk_server[1],"w");
  if (out_pipe==NULL) {
    fprintf(stderr,"Error openning write connection.\n");
      exit(-1);
  }
  in_pipe=fdopen(sk_client[0],"r");
  if (in_pipe==NULL) {
    fprintf(stderr,"Error openning write connection.\n");
      exit(-1);
  } 
}



/*
  This function send a command to the server
  to check that it still running.
*/
amoco_check_server()
{
  int i;
  char buf[10];

  seq_number++;
  fprintf(out_pipe,"%d Check\n",seq_number);
  fflush(out_pipe);
  strcpy(buf,"");
  i=read(sk_client[0],buf,3);
  if (strcmp(buf,"OK")!=0)
    return(-1);
}



/*
  This function sends a command to the server
  to set the color range of the raster data.
*/
amoco_color_range(start,end)
int start,end;
{
  seq_number++;
  fprintf(out_pipe,"%d ColorR %d %d\n",seq_number,start,end);
  fflush(out_pipe);
}



/*
  This function sends a command to the server
  for reading the line header of the data.
*/
amoco_read_line_header(header,file)
char *header;
char *file;
{
  char buf[10];
  int nbytes,error;

  error=amoco_check_server();
  if (error==-1) return(-1);

  seq_number++;
  fprintf(out_pipe,"%d LineH %s\n",seq_number,file);
  fflush(out_pipe);
  strcpy(buf,"");
  read(sk_client[0],buf,10);
  sscanf(buf,"%d",&nbytes);
  if (nbytes==-1) return(-1);

  error=amoco_reader(nbytes,header);

  return(error);

}
  


/*
  This function send a command to the server
  for reading the data. 
*/
amoco_read_data(raster,traceH,file,srec,erec,irec,strace,etrace,itrace,ssamp,esamp,isamp,type,s_offset,s_scalar,sort_flag,trace_flag)
unsigned char *raster;
unsigned char *traceH;
char *file;
int srec,erec,irec,strace,etrace,itrace,ssamp,esamp,isamp,type,s_offset;
float s_scalar;
int sort_flag,trace_flag;
{
  char buf[80];
  int nbytes,error,nbytes_trace;
  unsigned char *data;

  seq_number++;
  if (sort_flag)
    fprintf(out_pipe,"%d Data %s %d %d %d %d %d %d %d %d %d %d %d %e %d\n",
	    seq_number,file,srec,erec,irec,strace,etrace,itrace,ssamp,esamp,
	    isamp,type,s_offset,s_scalar,trace_flag);
  else
    fprintf(out_pipe,"%d DataNS %s %d %d %d %d %d %d %d %d %d %d %d %e %d\n",
	    seq_number,file,srec,erec,irec,strace,etrace,itrace,ssamp,esamp,
	    isamp,type,s_offset,s_scalar,trace_flag);
  fflush(out_pipe);
  error=read(sk_client[0],buf,80);
  sscanf(buf,"%d %d %e %d",&nbytes,&offset,&scalar,&nbytes_trace);
  if (verbose) {
    fprintf(stderr,"%d Data? %s %d %d %d %d %d %d %d %d %d %d %d %e\n",
	    seq_number,file,srec,erec,irec,strace,etrace,itrace,ssamp,
	    esamp,isamp,type,s_offset,s_scalar);
    fprintf(stderr,"%d bytes of data being transfered\n",nbytes);
  }
  if (nbytes==-1) return(-1);

  data = raster;
  error=amoco_reader(nbytes,data);

  if (verbose)
    fprintf(stderr,"%d bytes of data being transfered\n",nbytes_trace);
  if (nbytes_trace==-1) return(-1);

  data = traceH;
  error=amoco_reader(nbytes_trace,data);
  return(error);
}

amoco_offset()
{
  return(offset);
}

float amoco_scalar()
{
  return(scalar);
}
	  

/*
  This function find the length
  needed to store the file listing.
*/
amoco_ls_size(file)
char *file;
{
  char buf[80];
  int nbytes,error;

  error=amoco_check_server();
  if (error==-1) return(-1);

  seq_number++;
  fprintf(out_pipe,"%d Files %s\n",seq_number,file);
  fflush(out_pipe);
  read(sk_client[0],buf,10);
  sscanf(buf,"%d",&nbytes);
  if (verbose)
    fprintf(stderr,"%d bytes of data being transfered\n",nbytes);
  if (nbytes==-1) return(-1);
  return(nbytes);
}

/*
  This function retrieve the file
  listing.
*/
amoco_ls_files(string,nbytes)
char *string;
int nbytes;
{
  int error;
  char *data;

  data = string;
  error=amoco_reader(nbytes,data);

  return(error);
}



/* 
  This function sends a command to the server
  to stop the server program. 
*/
amoco_end_server()
{
  seq_number++;
  fprintf(out_pipe,"%d End\n",seq_number);
  fflush(out_pipe);
}



/*
  This function reads from the pipe
  and prints a hash mark ever 10,000
  bytes.
*/
amoco_reader(nbytes,data)
int nbytes;
unsigned char *data;
{
  int n,l,count,hash_count;

  if (verbose)
    fprintf(stderr,"\n");
  hash_count=0;
  count=0;
  for(n=nbytes; n > 0; n -= l) {
    l = read(sk_client[0], data, n);
    data += l;
    if (l==0)                     /* No data received. */
      count++;
    if ((count>1000)||(l<0))      /* After 1000 tries assume error. */
      return(-1);
    hash_count+=l;
    if (hash_count>10000) {
      hash_count=0;
      if (verbose)
	fprintf(stderr,"#");
    }
  }
  if (verbose)
    fprintf(stderr,"\n");
  return(0);
}
