C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C NAME: CMMVZ3D   3D, V(Z) MIGRATION - CDP             REV 2.0  MAR 95 *
C***********************************************************************
C
C  PURPOSE:
C       Performs a 3D, V(z), post-stack migration.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  FILES:
C
C  DESCRIPTION:
C
C  HISTORY:
C       Rev. 2.0                MAR 95            R.D. Coleman, CETech
C               Complete rewrite from scratch.
C       Rev. 2.2                MAR 95               Mary Ann Thornton   
C               change "stop" to call exitfu(0) so the script can 
c               capture and check the status upon exit of the program
c       Rev. 2.3                OCT 96            Dan Whitmore
c               removed the commenting of the command line argument
c               "stretch" in c3getpar.F
C---------------------------------------------------------------------
C
      program cmmvz3d

      implicit none

#include     <save_defs.h>
#include     <f77/iounit.h>
#include     <f77/lhdrsz.h>
#include     <f77/sisdef.h>

c
c parameters
c
      integer   ntab            ! number of entries in cxftab
      integer   jerr
      real      tabmax          ! maximum argument for cxftab
      real      zero            ! zero

      parameter( ntab   = 100001   )
      parameter( tabmax =     10.0 )  ! note: 10.0 > pi**2
      parameter( zero   =      0.0 )
c
c variables
c
      real      akx(1)          ! x wave number (kx) values
      real      akxbias         ! kx origin minus delta kx
      real      aky(1)          ! y wave number (ky) values
      real      akybias         ! ky origin minus delta ky
      character axis3*2         ! third axis name
      real*8    cxftab(ntab+5)  ! complex exponential function table
      integer   debug           ! debug output flag
      real      di(1)           ! imaginary components of depth section
      real      dip             ! maximum dip (degrees)
      real      dkx             ! delta kx
      real      dky             ! delta ky
      real      dr(1)           ! real components of depth section
      real      dtgrid          ! delta t
      real      dw              ! delta omega
      real      dzgrid          ! delta z
      real      fmax            ! maximum frequency (Hz)
      character fn_data*100     ! name of input data file
      character fn_ddout*100    ! name of output data dictionary file
      character fn_image*100    ! name of output image file
      character fn_vel*100      ! name of input velocity file
      character fn_zeta*100     ! name of output zeta file
      character name*7          ! name of code
      integer   header(1)       ! trace header array
      integer   ierr            ! error flag
      integer   izsnz(1)        ! number of z steps per z segment
      integer   jkx             ! loop index for kx
      integer   jky             ! loop index for ky
      integer   jw              ! loop index for w
      integer   jzseg           ! loop index for z segments
      real      kx0             ! kx origin
      integer   kxbias          ! index offset to zero x wave number
      integer   kxlim           ! kx index limit about zero wave number
      integer   kxlim1          ! lower limit for kx index
      integer   kxlim2          ! upper limit for kx index
      real      ky0             ! ky origin
      integer   kybias          ! index offset to zero y wave number
      integer   kylim           ! ky index limit about zero wave number
      integer   kylim1          ! lower limit for ky index
      integer   kylim2          ! upper limit for ky index
      integer   lbyout          ! number of bytes for usp i/o lineheader
      integer   ldi             ! length of array di in words
      integer   ldr             ! length of array dr in words
      integer   lhdr            ! length of array header in words
      integer*2 linehdr(SZLNHD) ! line header
      integer   lpsi            ! length of array psi in words
      integer   luinp           ! logical unit of input dataset
      integer   luout           ! logical unit of output dataset
      integer   lwork           ! length of array work in words
      integer   lz              ! leading dimension of array yblk
      integer   nbytes          ! number of bytes for usp i/o
      integer   nfopened        ! number of files opened
      integer   nkx             ! number of x wave numbers
      integer   nky             ! number of y wave numbers
      integer   nrec            ! number of input records
      integer   nsmp            ! number of input samples per trace
      integer   nt              ! original number of time samples/trace
      integer   ntrc            ! number of input traces per record
      integer   nvel            ! number of velocities in the vel. file
      integer   nw              ! number of omegas
      integer   nx              ! original number of x's
      integer   ny              ! original number of y's
      integer   nzbar           ! number of z's (irregular grid)
      integer   nzseg           ! number of z segments
      real      p               ! sin( angle ) / vref
      real      phase(1)        ! phase shift vector
      character ppname*7        ! program name
      real      psi(1)          ! wave field
      real      strch           ! stretch factor
      logical   tablelu         ! table loop-up flag
      character title*66        ! title for printout banner
      real      tmax            ! maximum time for migration
      real      twopi           ! 2.0 * pi
      real      vref            ! reference velocity
      logical   verbos          ! verbose output flag
      character version*3       ! version number
      real      w               ! omega (angular frequency)          
      real      w0              ! omega origin
      real      work(1)         ! scratch space for migration
      real      zmax            ! maximum depth for migration
      real      zsdz(1)         ! delta z per z segment
      real      zsslo(1)        ! slowness per z segment
c
c pointers
c
      pointer ( pakx   , akx    ) 
      pointer ( paky   , aky    )
      pointer ( pdi    , di     ) 
      pointer ( pdr    , dr     ) 
      pointer ( pheader, header ) 
      pointer ( pizsnz , izsnz  )
      pointer ( pphase , phase  ) 
      pointer ( ppsi   , psi    )
      pointer ( pwork  , work   )
      pointer ( pzsdz  , zsdz   )
      pointer ( pzsslo , zsslo  )
c
c functions
c
      integer   argis           ! is argument present function
c
c data initialization
c
      data name /'CMMVZ3D'/
      data nfopened / 0 /
      data ierr    / 0 /
      data ppname  / 'CMMVZ3D' /
      data title /                                                     '
     &                     3D V(Z) MIGRATION - CDP
     &' /
      data version / '2.3' /

c-----------------------------------------------------------------------

  900 format(/
     :'***************************************************************'/
     :'cmmvz3d performs a 3D post-stack, v(z) migration.'/
     :'execute cmmvz3d by typing cmmvz3d and the list of program'/
     :'parameters.'/
     :'  vel=<vel_model>       - Velocity model file name        ',
     :                           '(no default)'/
     :'  out=<ddout>           - Output data dictionary          ',
     :                           '(no default)'/
     :'  data_path=<path>      - Output data/zeta file path      ',
     :                           '(default=".")'/
     :'  data_suffix=<dsuffix> - Output data file suffix         ',
     :                           '(default=".usp")'/
     :'  zeta_suffix=<zsuffix> - Output zeta file suffix         ',
     :                           '(default=".zet")')
  901 format(
     :'  dz=<dzgrid>           - delta z                         ',
     :                           '(default - see note)'/
     :'  dtms=<dtgrid>         - delta t                         ',
     :                           '(default - see note)'/
     :'  zmax=<zmax>           - maximum depth of migration (ms) ',
     :                           '(default=model depth)'/
     :'  tmax=<tmax>           - maximum time  of migration (ms) ',
     :                           '(default=model time)'/
     :'  vref=<vref>           - reference velocity              ',
     :                           '(no default)'/
     :'  maxdip=<dipmax>       - maximum dip angle (degrees)     ',
     :                           '(default=90)'/
     :'  stretch=<strch>       - stretch factor                  ',
     :                           '(default=1)'/
     :'  flags=<flag_str>      - string of flags'/
     :'     V - Verbose printout'/
     :'     C - Compute phase-shift operators (default)'/
     :'     T - Table look-up phase-shift operators'/)
  902 format(
     :'  note: dz and dt are used only to force the boundary between'/
     :'  velocities to an integer multiple of either dz or dt.  Thus,'/
     :'  dz and dt cannot both be greater than zero.  If both are less'/
     :'  than or equal to zero, then dz is set to vref/200.'//
     :'usage:  cmmvz3d <<ddin> vel=<vel_model> out=<ddout>'/
     :'                [data_path=<path>] [data_suffix=<dsuffix>]'/
     :'                [zeta_suffix=<zsuffix>]'/
     :'                [dz=<dzgrid> | dtms=<dtgrid>]'/
     :'                [zmax=<zmax>] [tmax=<tmax>] vref=<vref>'/
     :'                [maxdip=<dipmax>] [stretch=<strch>]'/
     :'                [flags=[V][C|T]]'//
     :'***************************************************************')
  910 format( /' ', 'Input file name       = ', A100/
     1         ' ', 'Velocity file name    = ', A100/
     2         ' ', 'Output DD file name   = ', A100/
     3         ' ', 'Output data file name = ', A100/
     4         ' ', 'Output zeta file name = ', A100)
  911 format( /' ', 'Parameters:'/
     1         ' ', '   NW,  W0,  DW            =', i6, 2e15.5/
     2         ' ', '   NKX, KX0, DKX           =', i6, 2e15.5/
     3         ' ', '   NKY, KY0, DKY           =', i6, 2e15.5)
  912 format(  ' ', '   Original NT,NX,NY       =', 3i6)
  913 format(  ' ', '   Reference velocity      =', f13.6/
     2         ' ', '   Maximum dip             =', f13.6/
     3         ' ', '   Stretch factor          =', f13.6)
  921 format( /' ', 'Velocity model parameters:'/
     1         ' ', '   Number of z segments    =', i6/
     2         ' ', '   Number of z-s           =', i6/
     3         ' ', '   Maximum frequency (Hz)  =', f13.6/
     4         ' ', '   Maximum depth           =', f13.6/
     5         ' ', '   Maximum time            =', f13.6)
  922 format( /' ', '   i  izsnz(i)   zsdz(i)    zsslo(i)' )
  923 format(  ' ', i4, i10, f10.3, f12.8 )
  931 format( /' ', a6, ' file line header values:'/
     1         ' ', '   Number of records       =', i6/
     2         ' ', '   Number of traces/record =', i6/
     3         ' ', '   Number of samples/trace =', i6)

c-----------------------------------------------------------------------
c
c check for help
c
      if( argis( '-h' ) .gt. 0 .or. argis( '-?' ) .gt. 0 ) then
         write( LER, 900 )
         write( LER, 901 )
         write( LER, 902 )
         call exitfu(0)
      endif
c
c open printout file and print banner
c
      ierr = 0
      call openpr( LUPPRT, LUPRT, ppname, ierr)
      if( ierr .ne. 0 ) then
         write( LER, * ) '***** ERROR: reported by openpr *****'
         ierr = 1001
         go to 800
      endif

cinclude <f77/open.h>

      call gamoco( title, 1, LUPRT )
c
c read program parameters from command line
c
      call c3getpar( fn_data, fn_vel, fn_ddout, fn_image, fn_zeta,
     &               nw, w0, dw, nkx, kx0, dkx, nky, ky0, dky,
     &               nt, nx, ny, dzgrid, zmax, dtgrid, tmax, vref, dip,
     &               strch, tablelu, axis3, verbos, debug )

      write( LUPRT, 910 ) fn_data, fn_vel, fn_ddout, fn_image, fn_zeta
      write( LUPRT, 911 ) nw, w0, dw, nkx, kx0, dkx, nky, ky0, dky
      write( LUPRT, 912 ) nt, nx, ny
      write( LUPRT, 913 ) vref, dip, strch
c
c apply stretch factor to dzgrid and zmax and print command line arguments
c
      zmax   = zmax   * strch
      dzgrid = dzgrid * strch
c
c get the velocity model
c
      twopi = 2.0 * acos( -1.0 )
      fmax  = ( w0 + dw * ( nw - 1 ) ) / twopi

      open( unit=LUCARD, file=fn_vel, status='old', iostat=ierr )
      if( ierr .ne. 0 ) then
         write( LER, * ) '*** ERROR: cannot open velocity file ***'
         call exitfu(1)
      endif

      call c3rdnvel( verbos, debug, nvel, ierr )
      if( ierr .ne. 0 ) call exitfu(1)

      call galloc( pzsdz , nvel*ISZBYT, ierr, 'ABORT' )
      call galloc( pzsslo, nvel*ISZBYT, ierr, 'ABORT' )
      call galloc( pizsnz, nvel*ISZBYT, ierr, 'ABORT' )

      call c3rdvmod( verbos, debug, nvel, fmax, dtgrid, tmax,
     &               dzgrid, zmax, strch,
     &               nzbar, nzseg, izsnz, zsdz, zsslo, ierr )
      if( ierr .ne. 0 ) call exitfu(1)

      close( LUCARD )

      write( LUPRT, 921 ) nzseg, nzbar, fmax, zmax, tmax

      if( debug .gt. 0 ) then
         write( LUPRT, 922 )
         do jzseg = 1, nzseg
            write( LUPRT, 923 ) jzseg, izsnz(jzseg), zsdz(jzseg),
     &                          zsslo(jzseg)
         end do
      endif
c
c write output data dictionary
c
      call c3putpar( LUDISK, fn_ddout, fn_image, fn_zeta,
     &               nzbar, nkx, kx0, dkx, nky, ky0, dky,
     &               nt, nx, ny, dzgrid, zmax, dtgrid, tmax,
     &               axis3, verbos, debug )
c
c calculate z & t grids and write them to the zeta file
c 
      call c3ggrids( verbos, debug, LUDISK, fn_zeta, zmax, fmax,
     &               nzbar, nzseg, izsnz, zsdz, zsslo )
c
c open input and output files
c
      call lbopen( luinp, fn_data, 'r' )
      call lbopen( luout, fn_image, 'w' )
      nfopened = 2
c
c read line header and extract dimensions
c
      nbytes = 0
      call rtape( luinp, linehdr, nbytes )
      if( nbytes .eq. 0) then
         write( LER, * ) '*** ERROR: cannot read line header ***'
         ierr = 1003
         go to 800
      endif

      call saver( linehdr, 'NumSmp', nsmp, LINHED )
      call saver( linehdr, 'NumTrc', ntrc, LINHED )
      call saver( linehdr, 'NumRec', nrec, LINHED )

      call hlhprt( linehdr, nbytes, ppname, len( ppname ), LUPRT )

      write( LUPRT, 931 ) 'Input ', nrec, ntrc, nsmp
c
c update line header
c
      nsmp = 2 * nzbar
      call savew( linehdr, 'NumSmp', nsmp, LINHED )
      call savew( linehdr, 'NumTrc', ntrc, LINHED )
      call savew( linehdr, 'NumRec', nrec, LINHED )
      call savhlh( linehdr, nbytes, lbyout )
c
c write output line header
c
      call wrtape(  luout, linehdr, lbyout )
      write( LUPRT, 931 ) 'Output', nrec, ntrc, nsmp
c
c allocate remaining arrays
c
cndw sept 7, 1995 changed memory allocation of work 
      lwork = max0( nkx, 2*nw+ITRWRD, 2*nzbar+ITRWRD )
cmat
      lwork = lwork * 3
      lhdr  = ITRWRD * nkx
      lpsi  = 2 * nkx * nw
      ldr   = nkx * nzbar
      ldi   = nkx * nzbar
      call galloc( pakx   ,   nkx*ISZBYT, ierr, 'ABORT' )
      call galloc( paky   ,   nky*ISZBYT, ierr, 'ABORT' )
      call galloc( pheader,  lhdr*ISZBYT, ierr, 'ABORT' )
      call galloc( pphase , 2*nkx*ISZBYT, ierr, 'ABORT' )
      call galloc( pwork  , lwork*ISZBYT, ierr, 'ABORT' )
      call galloc( ppsi   ,  lpsi*ISZBYT, ierr, 'ABORT' )
      call galloc( pdr    ,   ldr*ISZBYT, ierr, 'ABORT' )
      call galloc( pdi    ,   ldi*ISZBYT, ierr, 'ABORT' )
c
c generate tables
c
c     call gcftlp( dummy, dummy, 0, ntab, 0.0, tabmax, 2, cxftab, ierr )
c     if( ierr .ne. 0 ) then
c        write( LER, * ) '*** ERROR: gcftlp reports ierr = ',ierr,' ***'
c        ierr = 1006
c        go to 800
c     endif

      akxbias = - ( ( nkx + 1 ) / 2 ) * dkx
      do jkx = 1, nkx
         akx(jkx) = jkx * dkx + akxbias
      end do

      akybias = - ( ( nky + 1 ) / 2 ) * dky
      do jky = 1, nky
         aky(jky) = jky * dky + akybias
      end do
c
c=======================================================================
c                         main migration loop
c=======================================================================
c
      kxbias = ( nkx + 1 ) / 2
      kybias = ( nky + 1 ) / 2
      p      = sin( dip * twopi / 360.0 ) / ( 0.5 * vref )

      do jky = 1, nky
         write( LER, * ) '  Migrating record ', jky
         if( debug .gt. 0 ) write( LUPRT, * ) '  Migrating record ', jky

         call c3rdrec( verbos, debug, luinp, jky, nkx, nw, work,
     &                 header, psi, ierr)
         if( ierr .ne. 0 ) go to 800

         call vclr( dr, 1, nkx*nzbar )
         call vclr( di, 1, nkx*nzbar )

         do jw = 0, nw-1
            w = w0 + dw * jw
            kxlim = nint( p * w / dkx )
            kylim  = nint( p * w / dky )
            kylim1 = kybias - kylim
            kylim2 = kybias + kylim
            if( kylim1 .lt.   1 ) kylim1 = 1
            if( kylim2 .gt. nky ) kylim2 = nky
            if( debug .gt. 0 ) write(LUPRT,*) '  KYLIM = ',kylim1,kylim2

            if( jky .ge. kylim1 .and. jky .le. kylim2 ) then
               kxlim  = nint( p * w / dkx )
               kxlim1 = kxbias - kxlim
               kxlim2 = kxbias + kxlim
               if( kxlim1 .lt.   1 ) kxlim1 = 1
               if( kxlim2 .gt. nkx ) kxlim2 = nkx

               call c3cont( verbos, debug, nkx, akx, aky(jky), nzseg,
     &                      izsnz, zsdz, zsslo, w, phase, work, kxlim1,
     &                      kxlim2, psi(1+2*nkx*jw), dr, di, ierr )
               if( ierr .ne. 0 ) go to 800
            endif
         enddo

         call c3wrrec( verbos, debug, luout, jky, nkx, nzbar, work,
     &                 header, dr, di, ierr )
         if( ierr .ne. 0 ) go to 800

      enddo
c
c=======================================================================
c                     end of main migration loop
c=======================================================================
c
c
c exit program
c
  800 continue
      if( nfopened .ge. 1 ) call lbclos( luinp )
      if( nfopened .ge. 2 ) call lbclos( luout )

      if( ierr .eq. 0 ) then
         write( LER, * ) '*** NORMAL COMPLETION ***'
         write( LUPRT, * ) '*** NORMAL COMPLETION ***'
         call exitfu(0)
      else
         call exitfu(1)
      endif

      end
