C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C NAME: CUBEVZ    V OF Z MIGRATION CUBE                REV 3.1  OCT 92 *
C***********************************************************************
C
C  PURPOSE:
C       CUBEVZ performs a plane wave, V of Z migration for a number of
C       angles and a number of velocity models.  For each velocity
C       model, the results are summed over the angles.
C
C  USAGE:
C       cubevz [-Nintape] [-Oouttape] [-Ccard] [-VELvtape] [-V] [-D] \
C              [-h]
C
C       -Nintape
C               Specifies 'intape' as the input SIS data set.  If
C               omitted, standard input is used.
C
C       -Oouttape
C               Specifies 'outtape' as the output SIS data set.  If
C               omitted, standard output is used.
C
C       -Ccard
C               Specifies 'card' as the input card file.  If omitted,
C               cubevz.crd is used.
C
C       -VELvtape
C               Specifies 'vtape' as the input velocity SIS data set.
C
C       -V
C               Causes the program to generate verbose print output.
C
C       -D
C               Causes the program to generate debugging print output
C               which includes the verbose print output.
C
C       -P
C               Causes the program to only calculate the job
C               parameters; i.e., the migration is not perfrormed.
C               Both the debug and verbose print output are generated.
C
C       -h
C               Causes the program to write help information to the
C               standard output and then halts.
C
C  FILES:
C       intape
C               Input standard SIS data set.
C
C       outtape
C               Output standard SIS data set.
C
C       card
C               Input card file.
C
C       vcrd
C               Input velocity SIS data set.
C
C       pfile
C               Output print file.  This file contains informational
C               output and error messages.  The print file name is
C               CU.xxxxx.yyyyy, where xxxxx and yyyyy are 5 digit
C               numbers that are generated from the process ID.
C
C  EXAMPLE:
C       The following command line will execute cubevz with an input SIS
C       data set named cubevz.inp, an output SIS data set named
C       cubevz.out, an input card file named card.dat, a input velocity
C       SIS data set called vel.inp and with the verbose output option
C       set:
C
C    cubevz -Ncubevz.inp -Ocubevz.out -Ccard.dat -VELvel.inp -V
C
C       The following is an example of a card file:
C
C          1111111111222222222233333333334444444444555555555566666666667
C 1234567890123456789012345678901234567890123456789012345678901234567890
C
C TIME       ENDTIMEMS PADTIMEMS BEGTIMEMS MIGTIMEMS
C TIME            4000       200         0      3000
C FREQUENCY  MINFREQHZ  F2FREQHZ  F3FREQHZ MAXFREQHZ
C FREQUENCY         20        25        55        60
C WIDTH       BEGTRACE  ENDTRACE  PADTRACE    DELTAX
C WIDTH              1       400        50        80
C RECORDS      BEG REC   END REC   INC REC
C RECORDS            1        51         2
C ANGLES      # ANGLES     START INCREMENT BEAMWIDTH   REF VEL
C ANGLES             5       -60         4        80      6000
C ANGLES            41       -40         2
C ANGLES             5        44         4
C T SCALE        ALPHA     XMULT
C T SCALE         0.80      0.02
C
C       The following facts should be noted about the card files:
C          (1) Card sets may appear in any order.  (A card set is
C              a header card and all of its associated data cards.)
C          (2) If a card has a asterisk (*) or pound sign (#) in
C              column 1, then it is a comment card and is ignored.
C              Comment cards may appear anywhere.
C          (3) The 'T SCALE' card set is optional.  All other card
C              sets must be included.
C          (4) All card set types, except ANGLES, can can have only
C              one data card.
C          (5) The BEAMWIDTH and 'REF VEL' specification must be on
C              first ANGLE data card.
C
C  HISTORY:
C       JUL 90          REL 1.0         R.D. Coleman, QTC
C       DEC 90          REL 2.0         R.D. Coleman, QTC
C               Input velocity models from a velocity SIS dataset.
C       DEC 90          REL 2.1         R.D. Coleman, QTC
C               Put data in natural order instead of FFT order and
C               replaced call to XVZPK3 with call to XVZPK4.
C       JAN 91          REL 2.2         R.D. Coleman, QTC
C               Corrected K limits calculation.
C       MAR 92          REL 2.3         R.D. Coleman, CETech
C               Replace MOD I/O with FORTRAN I/O.
C       JUN 92          REL 3.0         R.D. Coleman, CETech
C               Upgraded to same revision level as PWMVZN including
C               insuring that the program is portable.
C       OCT 92          REL 3.1         R.D. Coleman, CETech
C               Added output for t=0 and fixed work space allocation
C               bug.
C       MAY 93          REL 3.2         Mary Ann Thornton   
C               Change writes to lutrm to writes to ler and add the include
C               file for the HP logical unit LER
C               Recompile to take advantage of new gtfltr.f routine in library
C               Change size of line header to SZLNHD        
C-----------------------------------------------------------------------
C
C  DATA DICTIONARY
C       The Table below shows usage of the major variables. Symbolic
C       constants, such as LUPRT, are documented in the file cubevz.h
C
C                          M  I  P  R  D  M
C                          A  N  A  C  U  I
C                          I  I  R  R  M  G
C  Type    Variable        N  T  M  D  P  R  Description
C
C  REAL    AK      ( )     D     O  O     I  Wave number vector
C  REAL    ANGLE   ( )     D     O  O  I  I  Angle vector
C  REAL    BWIDTH          D     O  O  I  I  Beam width
C  REAL    CXFTAB          D     O        I  Complex function table
C  LOGICAL DEBUG           D  O              Debug flag
C  REAL    DK              D     O  O  I  I  Delta K
C  REAL    DT              D     O  O  I     Delta T
C  REAL    DZVM            D     O           Delta Z in Velocity Model
C  REAL    FILTR   ( )     D     O  O  I     Frequency filter
C  REAL    FMAX            D     O  O  I     Maximum frequency (Hz)
C  INTEGER IDT             D     O  I  I     Sampling interval used
C  INTEGER IERR            D  O  O  O  O  O  Error flag
C  INTEGER INCREC          D     O  O  I  I  Record increment
C  INTEGER IREC1           D     O  O  I  I  First input record to keep
C  INTEGER IREC2           D     O  O  I     Last input record to keep
C  INTEGER IT1             D     O  O  I     Index of first time sample used
C  INTEGER IT2             D     O  O  I     Index of last time sample used
C  INTEGER ITRC1           D     O  O  I     First trace to keep
C  INTEGER ITRC2           D     O  O  I     Last trace to keep
C  INTEGER IW1             D     O  O  I     First frequency to keep
C  INTEGER IW2             D     O  O  I     Last frequency to keep
C  INTEGER IZSNZ   ( )     D              I  Number of Z's per Z segment
C  INTEGER LUINP           D  O  I           LUN of input SIS dataset
C  INTEGER LUOUT           D  O  I           LUN of output SIS dataset
C  INTEGER LUVEL           D  I  I           LUN of velocity dataset
C  INTEGER MK              D              I  Leading dimension of IMAGE
C  INTEGER NK              D     O  O  I  I  Number of K's (wavenumbers)
C  INTEGER NP              D     O  O     I  Number of angles
C  INTEGER NREC            D     O  I  I     Number of input records
C  INTEGER NSMP            D     O  I  I     Number of samples per trace
C  INTEGER NT              D     O  O  I     Number times after padding
C  INTEGER NTOFF           D     O  O  I     Padding at start of trace
C  INTEGER NTOUT           D     O  O  I     Number of t samples in output
C  INTEGER NTPAD           D     O  O  I     Padding at end of trace
C  INTEGER NTRC            D     O  I  I     Number of traces per record
C  INTEGER NVMOD           D     O     I     Number of velocity models
C  INTEGER NVSMP           D     O     I     Number of velocity samples
C  INTEGER NW              D     O  O  I  I  Number of frequencies kept
C  INTEGER NX              D     O  O  I     Number of X's
C  INTEGER NZSEG           D              I  Number of Z segments
C  INTEGER NZT             D              I  Number of Z's for migration grid
C  REAL    OMEGA   ( )     D     O  O  I  I  Angular frequency vector
C  LOGICAL PARM            D  O              Flag to run CUPARM only
C  REAL    SCALE   ( )     D     O  O        Time scaling vector
C  REAL    TMIG            D     O  O  I     Time to migrate
C  REAL    VELREF          D     O  O  I  I  Reference velocity
C  LOGICAL VERBOS          D  O  I           Verbose flag
C  REAL    ZSDZ    ( )     D              I  Delta Z per Z segment
C  REAL    ZSSLOW  ( )     D              I  Slowness per Z segment
C
C  KEY:
C  D = DECLARED; I = INPUT; O = OUTPUT; W = WORKSPACE
C
C  * Indicates that the CALL and SUBROUTINE use different names
C  + Indicates that the space is declared under a different name
C
C-----------------------------------------------------------------------
C
      PROGRAM CUBEVZ
C
#ifdef CRAY
      IMPLICIT NONE
#endif
C
#include <localsys.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
C
#include <cubevz.h>
C
      LOGICAL DEBUG, VERBOS, PARM
      INTEGER LUINP, LUOUT, LUVEL, NREC, NTRC, NSMP, IDT, IREC1, 
     &        IREC2, INCREC, ITRC1, ITRC2, IT1, IT2, NP, NT, NTOFF,
     &        NTOUT, NTPAD, NW, IW1, IW2, NX, MK, NK, NVMOD, NVSMP,
     &        NZT, NZSEG, IZSNZ(MAXSEG), IERR
      REAL    DK, DT, DZVM, FMAX, BWIDTH, VELREF, ANGLE(MAXNP),
     &        OMEGA(MAXNW), SCALE(MAXNT), TMIG, FILTR(MAXNW),
     &        AK(MAXNK), ZSDZ(MAXSEG), ZSSLOW(MAXSEG)
C
C===  SYSTEM DEPENDENT DECLARATIONS
C
#ifdef CRAY
      REAL    CXFTAB(NTAB+5)
#else
      REAL*8  CXFTAB(NTAB+5)
#endif
C
C  LOCAL VARIABLES:
C
      INTEGER  JP
C                               Angle index
      INTEGER  JREC
C                               Input record index
      INTEGER  JVMOD
C                               Velocity model index
      INTEGER  LRECL
C                               Record Length
      INTEGER  LWRK0
C                               Length of WRK0 in words
      INTEGER  LWRK1
C                               Length of WRK1 in words
      INTEGER  LWRK2
C                               Length of WRK2 in words
      INTEGER  MT
C                               Leading dimension of PSI = NT+3
      INTEGER  NBTRC
C                               Number of bytes per input trace
      INTEGER  NEXTRC
C                               Next input record to process
      INTEGER  NZSMAX
C                               Maximum value of NZSEG
      INTEGER  NZTMAX
C                               Maximum value of NZT
      POINTER (PWRK0, WRK0 )
C                               Pointer to WRK0
      POINTER (PWRK1, WRK1 )
C                               Pointer to WRK1
      POINTER (PWRK2, WRK2 )
C                               Pointer to WRK2
      REAL     T1GRID(MAXNZT)
C                               Internal irregular T grid
      REAL     T2GRID(MAXNT)
C                               Output regular T grid
      REAL     WRK0  (1)
C                               Work pointee array #0
      REAL     WRK1  (1)
C                               Work pointee array #1
      REAL     WRK2  (1)
C                               Work pointee array #2
      REAL     ZSDT(MAXSEG)
C                               Delta t for each step for each Z segment
C
C  SUBPROGRAMS CALLED:
C
      EXTERNAL CUINIT
C                               Initialize program, open files, etc.
      EXTERNAL CUPARM
C                               Get job parameters, write line header
      EXTERNAL CUMIGR
C                               Downward continue and image
      EXTERNAL FFT2DF
C                               2D forward FFT
      EXTERNAL FFTIXR
C                               Inverse complex FFT and extract reals
      EXTERNAL GALLOC
C                               Allocate space
      EXTERNAL GGRID
C                               Generate irregular grid
      EXTERNAL MEXIT
C                               MBS exit
      EXTERNAL RDRECNH
C                               Read record (no headers)
      EXTERNAL RDVMOD
C                               Read a velocity model
      EXTERNAL RESAMP
C                               Resample
      EXTERNAL RMSCAL
C                               Real matrix scale
      EXTERNAL SKIPT
C                               Skip input SIS dataset traces
      EXTERNAL VMUL
C                               Vector multiply
      EXTERNAL VRAMP
C                               Vector ramp
      EXTERNAL VSMUL
C                               Vector scalar multiply
      EXTERNAL WRM2FILE
C                               Write matrix to file
      EXTERNAL WRRECNH
C                               Write record (no headers)
C
C  GLOBAL VARIABLE INITIALIZATIONS:
C
      DATA LUINP, LUOUT, LUVEL / 3 * -1 /
C
C-----------------------------------------------------------------------
C
  991 FORMAT (/' ', '***** ERROR: ERROR ATTEMPTING TO REAL VELOCITY ',
     &              'MODEL', I3)
C
C-----------------------------------------------------------------------
C
C===  A. INITIALIZE, OPEN FILES, ETC.
C===  B. READ LINE HEADER & JOB PARAMETERS, WRITE OUTPUT LINE HEADER
C===  C. INITIALIZE AND OPEN SCRATCH SCRATCH FILE
C===  D. ALLOCATE SPACE FOR WORK ARRAYS
C
      CALL CUINIT( VERBOS, DEBUG, PARM, LUINP, LUOUT, LUVEL, IERR )
      IF( DEBUG ) PRINT *, ' '
      IF( DEBUG ) PRINT *, '   finished CUINIT'
      IF( IERR .NE. 0 ) GO TO 800
C
      CALL CUPARM( LUINP, LUOUT, LUVEL, VERBOS, NREC, NTRC, NSMP,
     &    IDT, IREC1, IREC2, INCREC, ITRC1, ITRC2, IT1, IT2, NP,
     &    NT, NTOFF, NTOUT, NTPAD, NW, IW1, IW2, NX, NK, NVMOD,
     &    NVSMP, DK, DT, DZVM, FMAX, BWIDTH, VELREF, ANGLE, OMEGA,
     &    SCALE, TMIG, FILTR, AK, CXFTAB, IERR )
C
      IF( DEBUG ) PRINT *, '   finished CUPARM'
C
      IF( IERR .NE. 0 ) GO TO 800
      IF( PARM ) GO TO 800
C
      MK    = NK + 1
      MT    = NT + 3
      LRECL = 2 * NK * ISZBYT
C
      OPEN( UNIT=LUTMP, ACCESS='DIRECT', FORM='UNFORMATTED',
     &      RECL=LRECL, STATUS='SCRATCH' )
C
C===  Compute memory requirements and allocate space
C
      LWRK0 = MAX0( NVSMP, NSMP, NTOUT ) + ITRWRD
      CALL GALLOC( PWRK0, ISZBYT * LWRK0, IERR, 'ABORT' )
      NZTMAX = 0
      NZSMAX = 0
      DO 100 JVMOD = 1, NVMOD
         CALL RDVMOD (LUVEL, NVSMP, DZVM, FMAX, TMIG, WRK0,
     &                NZSEG, NZT, IZSNZ, ZSDZ, ZSSLOW, IERR)
         IF (IERR .NE. 0) THEN
            IERR = 1600 + IABS( IERR )
            GO TO 800
         ENDIF
         IF( NZT   .GT. NZTMAX ) NZTMAX = NZT
         IF( NZSEG .GT. NZSMAX ) NZSMAX = NZSEG
  100 CONTINUE
      CALL SISSEEK( LUVEL, 1 )
      CALL GFREE( PWRK0 )
          
      LWRK0 = MAX0( LWRK0, 7*NK, 7*NZSMAX )
      LWRK1 = MAX0( MT*NK, 2*MK*(NZTMAX+1) )
#ifdef CRAY
      LWRK2 = MAX0( 2*NT*NX+2*NT, 4*NW*NK+2*NK, 4*NK*(NZTMAX+1)+2*NK,
     &              NX*NTOUT, 2*NK*NP )
#else
      LWRK2 = MAX0( 9*NT/2+18*NX+41, 6*NK+18*NW+41,
     &              6*NK+18*(NZTMAX+1)+41, NX*NTOUT, 2*NK*NP )
#endif
C
      CALL GALLOC( PWRK0, ISZBYT * LWRK0, IERR, 'ABORT' )
      CALL GALLOC( PWRK1, ISZBYT * LWRK1, IERR, 'ABORT' )
      CALL GALLOC( PWRK2, ISZBYT * LWRK2, IERR, 'ABORT' )
      IF( DEBUG ) PRINT *, '   finished GALLOCs'
      IF( DEBUG ) PRINT *, '   LWRK0, LWRK1, LWRK2 = '
      IF( DEBUG ) PRINT *, '   ', LWRK0, LWRK1, LWRK2
C
C===  LOOP OVER ANGLES (INPUT RECORDS) SKIPPING UNWANTED ANGLES
C===     A. READ INPUT DATA
C===     B. FFT AND TRANSPOSE DATA
C===     C. WRITE DATA TO A SCRATCH SCRATCH FILE
C
      NBTRC  = NSMP * SZSAMP + SZDTHD
      NEXTRC = IREC1
      JP    = 0
      DO 110 JREC = 1, IREC2
         IF (JREC .EQ. NEXTRC) THEN
            IF (DEBUG) PRINT *, ' PROCESSING RECORD ', JREC
C
            NEXTRC = NEXTRC + INCREC
            JP     = JP + 1
C
C======     READ SEISMIC RECORD
C======     NOTE: WRK1 will now contain PSI(t,x,z=0)
C======           WRK0 is scratch
C
            CALL RDRECNH( LUINP, LUPRT, JREC, NTRC, ITRC1, ITRC2,
     &                    NSMP, IT1, IT2, NTOFF, NT, 1, NX, MT,
     &                    WRK0, WRK1, IERR )
            IF( DEBUG ) PRINT *, '   finished RDRECNH'
            IF (IERR .NE. 0) THEN
               IERR = 1500 + IABS( IERR )
               GO TO 800
            ENDIF
C
C======     PERFORM 2D FFT
C======     NOTE: WRK2 will now contain PSI(kx,w,z=0)
C
            CALL FFT2DF( MT, NT, NX, NK, NW, IW1, FILTR, WRK1, WRK2 )
            IF( DEBUG ) PRINT *, '   finished FFT2DF'
C
C======     WRITE TO SCRATCH FILE
C
            CALL WRM2FILE( WRK2, 2*NK, 2*NK, NW, LUTMP, JP, NP )
            IF( DEBUG ) PRINT *, '   finished WRM2FILE'
C
         ELSE
            IF (DEBUG) PRINT *, ' SKIPPING   RECORD ', JREC
            CALL SKIPT (LUINP, NTRC, NBTRC)
         ENDIF
  110 CONTINUE
C
C===  LOOP OVER VELOCITY MODELS (OUTPUT RECORDS)
C===     A. CALCULATE VELOCITIES AND VELOCITY DEPENDENT Z PARAMETERS
C===     B. LOOP OVER OMEGA (IN CUMIGR)
C===        1. READ DATA FROM SCRATCH FILE
C===        2. CALCULATE K LIMITS
C===        3. DOWNWARD CONTINUE AND IMAGE
C===     C. RESAMPLE AND TRANSPOSE
C===     D. WRITE TO OUTPUT DATASET
C
      CALL VRAMP( 0.0, DT, T2GRID, 1, NTOUT )
      DO 210 JVMOD = 1, NVMOD
         IF (DEBUG) PRINT *, ' PROCESSING VEL. MODEL ', JVMOD
C
C======  READ VELOCITIES
C======  NOTE: WRK0 is scratch
C
         CALL RDVMOD (LUVEL, NVSMP, DZVM, FMAX, TMIG, WRK0,
     &                NZSEG, NZT, IZSNZ, ZSDZ, ZSSLOW, IERR)
         IF (IERR .NE. 0) THEN
            IERR = 1600 + IABS( IERR )
            GO TO 800
         ENDIF
         IF( DEBUG ) PRINT *, '   finished RDVMOD'
C
         CALL VMUL( ZSDZ, 1, ZSSLOW, 1, ZSDT, 1, NZSEG )
         CALL VSMUL( ZSDT, 1, 2.0, ZSDT, 1, NZSEG )
         T1GRID(1) = 0.0
         CALL GGRID( NZSEG, IZSNZ, ZSDT, T1GRID(2) )
         IF( DEBUG ) PRINT *, '   finished GGRID'
C
C
         IF (IERR .NE. 0) THEN
            WRITE (LUPRT, 991) JVMOD
            GO TO 800
         ENDIF
C
C======  MIGRATE
C======  NOTE: WRK1 will now contain the complex IMAGE(kx,t1) where t1
C======             is t with an irregular grid
C======        WRK0 is scratch
C======        WRK2 is scratch
C
         CALL CUMIGR( MK, NK, NW, NZSEG, NZT, NP, IREC1, INCREC, DK,
     &                ANGLE, BWIDTH, VELREF, OMEGA, AK, IZSNZ, ZSDZ,
     &                ZSSLOW, CXFTAB, WRK0, WRK2, WRK1, IERR )
         IF( DEBUG ) PRINT *, '   finished CUMIGR'
         IF( IERR .NE. 0 ) GO TO 800
C
C======  INVERSE FFT
C======  NOTE: WRK1 will now contain the real IMAGE(x,t1)
C======        WRK2 is scratch
C
         CALL FFTIXR( MK, NK, NX, NZT+1, WRK2, WRK1 )
         IF( DEBUG ) PRINT *, '   finished FFTIXR'
C
C======  RESAMPLE
C======  NOTE: WRK2 will now contain the IMAGE(t,x)
C======        WRK0 is scratch
C
         CALL RESAMP (NX, 1, NTOUT, NZT+1, NX, NTOUT, 1, T1GRID, T2GRID,
     &                WRK0, WRK1, WRK2 )
         IF( DEBUG ) PRINT *, '   finished RESAMP'
C
C======  TIME SCALING
C======  NOTE: WRK2 will now contain the time scaled IMAGE(t,x)
C
         CALL RMSCAL( WRK2, 1, NTOUT, SCALE, 1, 0, WRK2, 1, NTOUT,
     &                NTOUT, NX )
         IF( DEBUG ) PRINT *, '   finished RMSCAL'
C
C======  WRITE RECORD
C======  NOTE: WRK0 is scratch
C
         CALL WRRECNH( LUOUT, LUPRT, JVMOD, NTOUT, 1, NX, NTOUT, WRK0,
     &                 WRK2, IERR )
         IF( DEBUG ) PRINT *, '   finished WRRECNH'
         IF( IERR .NE. 0 ) THEN
            IERR = 1700 + IABS( IERR )
            GO TO 800
         ENDIF
C
  210 CONTINUE
C
C===  CLOSE FILES, CLEAN-UP, & EXIT
C
      IERR = 0
C
  800 CONTINUE
      CALL MEXIT( IERR, LUPRT, LUINP, LUOUT )
C
      END
C***********************************************************************
C NAME: CUINIT                                                         *
C***********************************************************************
C
C  PURPOSE:
C      CUINIT INITIALIZES THE PROGRAM INCLUDING OPENING ALL REQUIRED
C      FILES.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE CUINIT( VERBOS, DEBUG, PARM, LUINP, LUOUT, LUVEL, IERR)
C
#ifdef CRAY
      IMPLICIT NONE
#endif
#include <f77/hp.h>
#include <cubevz.h>
C
C  FORMAL PARAMETERS:
C
      LOGICAL VERBOS, DEBUG, PARM
      INTEGER LUINP, LUOUT, LUVEL, IERR
C
C  LOCAL VARIABLES:
C
      CHARACTER*128 CARDS
C                                 CARD FILE NAME
      CHARACTER*128 NTAPE
C                                 INPUT  SIS DATASET NAME
      CHARACTER*128 OTAPE
C                                 OUTPUT SIS DATASET NAME
      CHARACTER*128 VTAPE
C                                 VELOCITY SIS DATASET NAME
C
C  SUBPROGRAMS CALLED:
C
      INTEGER ARGIS, ICOPEN
C
      EXTERNAL ARGIS
C                                 CHECK FOR PRESENCE OF COMMAND LINE SWITCH
      EXTERNAL ARGSTR
C                                 GET COMMAND LINE STRING ARGUMENT
      EXTERNAL GAMOCO
C                                 PRINT AMOCO LOGO
      EXTERNAL ICOPEN
C                                 OPEN INLINE CARD STREAM
      EXTERNAL LBOPEN
C                                 OPEN SIS DATASET
      EXTERNAL OPENPR
C                                 OPEN PRINT FILE
C
C-----------------------------------------------------------------------
C
  900 FORMAT( /' ', 'CUBEVZ COMMAND LINE ARGUMENTS',
     &        /' ',
     &        /' ', 'INPUTS:',
     &        /' ', '  -N[ntape]    - INPUT  DATASET NAME',
     &        /' ', '  -O[otape]    - OUTPUT DATASET NAME',
     &        /' ', '  -VEL[vcards] - VELOCITY DATASET NAME',
     &        /' ', '  -C[cards]    - CARD FILE NAME',
     &        /' ', '  -V           - VERBOSE PRINTOUT',
     &        /' ',
     &        /' ', 'USAGE:',
     &        /' ', '  cubevz -N[] -O[] -VEL[] -C[] -V'/ )
  901 FORMAT(  ' ' )
  902 FORMAT(  ' ', 'INPUT DATASET    = ', A )
  903 FORMAT(  ' ', 'OUTPUT DATASET   = ', A )
  904 FORMAT(  ' ', 'VELOCITY DATASET = ', A )
  905 FORMAT(  ' ', 'CARD FILE        = ', A )
  991 FORMAT( /' ', '***** ERROR: UNABLE TO OPEN PRINT FILE - ',
     &              'OPENPR ERROR CODE = ', I10/ )
  992 FORMAT( /' ', '***** ERROR: UNABLE TO OPEN CARD FILE - ',
     &              'CFT77 MESSAGE NUMBER = ', I10/ )
  993 FORMAT( /' ', '***** ERROR: UNABLE TO OPEN CARD FILE'/ )
  994 FORMAT( /' ', '***** ERROR: VELOCITY FILE MUST BE SPECIFIED'/ )
C
C-----------------------------------------------------------------------
C
C  IF HELP, WRITE INSTRUCTIONS TO SCREEN
C
      IF( ARGIS( '-h' ) .GT. 0 ) THEN
         WRITE( LER, 900 )
         IERR = -1
         RETURN
      ENDIF
C
C  CREATE UNIQUE PRINT FILE NAME FROM THE PROCESS ID & OPEN PRINT FILE
C
      CALL OPENPR( LULST, LUPRT, PPNAME, IERR )
      IF( IERR .NE. 0 ) THEN
         WRITE( LER, 991 ) IERR
         IERR = 2100 + IABS( IERR )
         RETURN
      ENDIF
C
#include <mbsdate.h>
C
      CALL GAMOCO( TITLE, 1, LUPRT )
C
      PARM   = ARGIS( '-P' ) .GT. 0
      DEBUG  = ARGIS( '-D' ) .GT. 0 .OR. PARM
      VERBOS = ARGIS( '-V' ) .GT. 0 .OR. DEBUG
C
      WRITE( LUPRT, 901 )
C
C  GET REMAINING COMMAND LINE ARGUMENTS
C
      CALL ARGSTR( '-N',   NTAPE, ' ', ' ' )
      CALL ARGSTR( '-O',   OTAPE, ' ', ' ' )
      CALL ARGSTR( '-C',   CARDS, ' ', ' ' )
      CALL ARGSTR( '-VEL', VTAPE, ' ', ' ' )
C
C  DEFAULT FOR PIPES
C
      LUINP = 0
      LUOUT = 1
C
C  OPEN SIS DATASETS
C
      IF( NTAPE .NE. ' ' ) THEN
         CALL LBOPEN( LUINP, NTAPE, 'r' )
         WRITE( LUPRT, 902 ) NTAPE
      ENDIF
C
      IF( OTAPE .NE. ' ' ) THEN
         CALL LBOPEN( LUOUT, OTAPE, 'w' )
         WRITE( LUPRT, 903 ) OTAPE
      ENDIF
C
      IF( VTAPE .NE. ' ' ) THEN
         CALL LBOPEN( LUVEL, VTAPE, 'r' )
         WRITE( LUPRT, 904 ) VTAPE
      ELSE
         WRITE( LUPRT, 994 )
         IERR = 2002
         RETURN
      ENDIF
C
C  OPEN CARD FILE
C
      IF( CARDS .NE. ' ' ) THEN
         OPEN( UNIT=LUCRD, FILE=CARDS, STATUS='OLD', IOSTAT=IERR )
         IF( IERR .EQ. 0 ) THEN
            WRITE( LUPRT, 905 ) CARDS
         ELSE
            WRITE( LUPRT, 992 ) IERR
            IERR = 2001
            RETURN
         ENDIF
      ELSE
         IERR = ICOPEN( '-cubevz.crd', LUCRD )
         IF( IERR .EQ. 0 ) THEN
            WRITE( LUPRT, 993 )
            IERR = 2001
            RETURN
         ENDIF
      ENDIF
C
      IERR = 0
      RETURN
      END
C***********************************************************************
C NAME: CUPARM                                                         *
C***********************************************************************
C
C  PURPOSE:
C      CUPARM READS THE INPUT LINE HEADER, VELOCITY LINE HEADER, JOB
C      PARAMETERS, AND WRITES THE OUTPUT LINE HEADER.  IN ADDITION,
C      CUPARM CALCULATES SOME JOB PARAMETERS.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE CUPARM( LUINP, LUOUT, LUVEL, VERBOS, NREC, NTRC, NSMP,
     &    IDT, IREC1, IREC2, INCREC, ITRC1, ITRC2, IT1, IT2, NP,
     &    NT, NTOFF, NTOUT, NTPAD, NW, IW1, IW2, NX, NK, NVMOD,
     &    NVSMP, DK, DT, DZVM, FMAX, BWIDTH, VELREF, ANGLE, OMEGA,
     &    SCALE, TMIG, FILTR, AK, CXFTAB, IERR )
C
C
#ifdef CRAY
      IMPLICIT NONE
#endif
#include <cubevz.h>
#include <localsys.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
C
C
C  SUBROUTINE ARGUMENTS
C
      LOGICAL VERBOS
      INTEGER LUINP, LUOUT, LUVEL, NREC, NTRC, NSMP, IDT, IREC1, 
     &        IREC2, INCREC, ITRC1, ITRC2, IT1, IT2, NP, NT, NTOFF,
     &        NTOUT, NTPAD, NW, IW1, IW2, NX, NK, NVMOD, NVSMP, IERR,
     &        IDZVM
      REAL    DK, DT, DZVM, FMAX, BWIDTH, VELREF, ANGLE(*), OMEGA(*),
     &        SCALE(*), TMIG, FILTR(*), AK(*), CXFTAB(*)
C
C  LOCAL VARIABLES
C
      REAL         DUMMY
C                                 DUMMY PLACE HOLDER
      INTEGER      IFORM
C                                 FORMAT OF SIS DATASET
      INTEGER      LHEAD (SZLNHD)
C                                 LINE HEADER
      INTEGER      NBYTES
C                                 BYTE COUNT FOR RTAPE AND WRTAPE
C
C  SUBPROGRAMS CALLED:
C
      EXTERNAL CURCRD
C                                 Read card files
      EXTERNAL CUDUMP
C                                 Write global variables
      EXTERNAL GCFTLP
C                                 Generate complex exponential function table
      EXTERNAL HLHPRT
C                                 Update line header
      EXTERNAL RTAPE
C                                 Read SIS dataset
      EXTERNAL WRTAPE
C                                 Write SIS dataset
C
C-----------------------------------------------------------------------
C
  901 FORMAT (/' ', 'INPUT LINE HEADER PARAMETERS:')
  902 FORMAT (/' ', 'OUTPUT LINE HEADER PARAMETERS:')
  903 FORMAT (/' ', '   NUMBER OF SEISMIC RECORDS IN THIS JOB =', I5,
     2        /' ', '   NUMBER OF TRACES PER SEISMIC RECORD   =', I5,
     3        /' ', '   NUMBER OF DATA SAMPLES PER TRACE      =', I5,
     4        /' ', '   SAMPLE INTERVAL (MILLISECONDS)        =', I5,
     5        /' ', '   FORMAT                                =', I5)
  904 FORMAT (/' ', 'VELOCITY LINE HEADER PARAMETERS:',
     2        /' ', '   NUMBER OF VELOCITY MODELS             =', I5,
     3        /' ', '   NUMBER OF VELOCITY SAMPLES PER MODEL  =', I5,
     4        /' ', '   DEPTH SAMPLE INTERVAL                 =', F8.2)
  991 FORMAT (/' ', '***** ERROR - INPUT LINE HEADER READ ERROR *****')
  992 FORMAT (/' ', '***** ERROR - INPUT DATASET MUST HAVE FORMAT 3 ',
     &              '*****')
  993 FORMAT (/' ', '***** ERROR - NUMBER OF RECORDS EXCEEDS MAXIMUM ',
     &              'ALLOWED *****',
     2        /' ', '              NUMBER OF RECORDS       =', I8,
     3        /' ', '              MAXIMUM RECORDS ALLOWED =', I8)
  994 FORMAT (/' ', '***** ERROR - NUMBER OF SAMPLES EXCEEDS MAXIMUM ',
     &              'ALLOWED *****',
     2        /' ', '              NUMBER OF SAMPLES       =', I8,
     3        /' ', '              MAXIMUM SAMPLES ALLOWED =', I8)
  995 FORMAT (/' ', '***** ERROR - VELOCITY LINE HEADER READ ERROR ',
     &              '*****')
  996 FORMAT (/' ', '***** ERROR - ERROR REPORTED BY ROUTINE GCFTLP ',
     &              '*****',
     2        /' ', '              GCFTLP ERROR CODE =', I8)
C
C=======================================================================
C                  PROCESS VELOCITY LINE HEADER
C=======================================================================
C
      NBYTES = 0
      CALL RTAPE (LUVEL, LHEAD, NBYTES)
      IF (NBYTES .EQ. 0) THEN
         WRITE (LUPRT, 995)
         IERR = 3002
         RETURN
      ENDIF
C
C  GET PARAMETERS FROM LINE HEADER
C
CMAT  NVMOD = LHEAD(11)
CMAT  NVSMP = LHEAD(14)
CMAT  DZVM  = 0.001 * FLOAT( LHEAD(54) )
      CALL SAVER(LHEAD, 'NumTrc', NVMOD, LINHED)
      CALL SAVER(LHEAD, 'NumSmp', NVSMP, LINHED)
      CALL SAVER(LHEAD, 'Dz1000', IDZVM, LINHED)
      DZVM = 0.001 * FLOAT( IDZVM )
C
      IF (VERBOS) THEN
         WRITE (LUPRT, 904) NVMOD, NVSMP, DZVM
      ENDIF
C
C=======================================================================
C                  PROCESS INPUT LINE HEADER
C=======================================================================
C
      NBYTES = 0
      CALL RTAPE (LUINP, LHEAD, NBYTES)
      IF (NBYTES .EQ. 0) THEN
         WRITE (LUPRT, 991)
         IERR = 3002
         RETURN
      ENDIF
C
C  GET PARAMETERS FROM LINE HEADER
C
CMAT  NTRC  = LHEAD(11)
CMAT  NREC  = LHEAD(12)
CMAT  IDT   = LHEAD(13)
CMAT  NSMP  = LHEAD(14)
CMAT  IFORM = LHEAD(15)
      CALL SAVER(LHEAD, 'NumTrc', NTRC , LINHED)
      CALL SAVER(LHEAD, 'NumRec', NREC , LINHED)
      CALL SAVER(LHEAD, 'SmpInt', IDT  , LINHED)
      CALL SAVER(LHEAD, 'NumSmp', NSMP , LINHED)
      CALL SAVER(LHEAD, 'Format', IFORM, LINHED)
C
      IF (VERBOS) THEN
         WRITE (LUPRT, 901)
         WRITE (LUPRT, 903) NREC, NTRC, NSMP, IDT, IFORM
      ENDIF
C
      IF (IFORM .NE. 3) THEN
        WRITE (LUPRT, 992)
        IERR = 3011
        RETURN
      ENDIF
C
      IF (NREC .GT. MAXNP) THEN
         WRITE (LUPRT, 993) NREC, MAXNP
         IERR = 3003
         RETURN
      ENDIF
C
      IF (NSMP .GT. MAXNT) THEN
         WRITE (LUPRT, 994) NSMP, MAXNT
         IERR = 3003
         RETURN
      ENDIF
C
C=======================================================================
C                  READ PARAMETERS FROM CARD FILE
C=======================================================================
C
      CALL CURCRD( NREC, NTRC, NSMP, IDT, IREC1, IREC2, INCREC,
     &             ITRC1, ITRC2, IT1, IT2, NP, NT, NTOFF, NTOUT,
     &             NTPAD, NW, IW1, IW2, NX, NK, DK, DT, FMAX,
     &             BWIDTH, VELREF, ANGLE, OMEGA, SCALE, TMIG,
     &             FILTR, AK, IERR )
      IF( IERR .NE. 0 ) RETURN
C
C=======================================================================
C                  UPDATE AND OUTPUT LINE HEADER
C=======================================================================
C
      CALL HLHPRT (LHEAD, NBYTES, PPNAME, LEN(PPNAME), LUPRT)
C
CMAT  LHEAD(11) = NX
CMAT  LHEAD(12) = NVMOD
CMAT  LHEAD(13) = IDT
CMAT  LHEAD(14) = NTOUT
CMAT  LHEAD(15) = IFORM
      CALL SAVEW(LHEAD, 'NumTrc', NX   , LINHED)
      CALL SAVEW(LHEAD, 'NumRec', NVMOD, LINHED)
      CALL SAVEW(LHEAD, 'SmpInt', IDT  , LINHED)
      CALL SAVEW(LHEAD, 'NumSmp', NTOUT, LINHED)
      CALL SAVEW(LHEAD, 'Format', IFORM, LINHED)

      CALL WRTAPE (LUOUT, LHEAD, NBYTES)
C
      IF (VERBOS) THEN
         WRITE (LUPRT, 902)
CMAT     WRITE (LUPRT, 903) LHEAD(12), LHEAD(11), LHEAD(14), LHEAD(13),
CMAT &                      LHEAD(15)
         WRITE (LUPRT, 903) NVMOD, NX, NTOUT, IDT, IFORM
      ENDIF
C
C=======================================================================
C                  BUILD TABLES
C=======================================================================
C
      CALL GCFTLP (DUMMY, DUMMY, 0, NTAB, 0.0, TABMAX, 2, CXFTAB, IERR)
C
      IF (IERR .NE. 0) THEN
         WRITE (LUPRT, 996) IERR
         IERR = 3300 + IABS( IERR )
         RETURN
      ENDIF
C
C=======================================================================
C                  EXIT SUBROUTINE
C=======================================================================
C
      IF (VERBOS) CALL CUDUMP( NREC, NTRC, NSMP, IDT, IREC1, IREC2,
     &    INCREC, ITRC1, ITRC2, IT1, IT2, NT, NTOFF, NTOUT, NTPAD, NW,
     &    IW1, IW2, NX, NK, NVMOD, NVSMP, DK, DT, DZVM, FMAX, BWIDTH,
     &    VELREF, ANGLE, OMEGA, TMIG, FILTR )
C
      IERR = 0
      RETURN
      END
C***********************************************************************
C NAME: CURCRD                                                         *
C***********************************************************************
C
C  PURPOSE:
C       CURCRD READS AND PROCESSES THE CONTENTS OF THE CARD FILE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE CURCRD( NREC, NTRC, NSMP, IDT, IREC1, IREC2, INCREC,
     &                   ITRC1, ITRC2, IT1, IT2, NP, NT, NTOFF, NTOUT,
     &                   NTPAD, NW, IW1, IW2, NX, NK, DK, DT, FMAX,
     &                   BWIDTH, VELREF, ANGLE, OMEGA, SCALE, TMIG,
     &                   FILTR, AK, IERR )
C
#ifdef CRAY
      IMPLICIT NONE
#endif
C
#include <cubevz.h>
C
C  SUBROUTINE ARGUMENTS
C
      INTEGER NREC, NTRC, NSMP, IDT, IREC1, IREC2, INCREC, ITRC1,
     &        ITRC2, IT1, IT2, NP, NT, NTOFF, NTOUT, NTPAD, NW,
     &        IW1, IW2, NX, NK, IERR
      REAL    DK, DT, FMAX, BWIDTH, VELREF, ANGLE(*), OMEGA(*),
     &        SCALE(*), TMIG, FILTR(*), AK(*)
C
C  PARAMETERS:
C
      INTEGER NTYPES
C
      PARAMETER (NTYPES = 6)
C                                   Number of card types
C
C  LOCAL VARIABLES:
C
      REAL         A
C                                   Angle
      REAL         AINC
C                                   Angle increment
      REAL         ANG   (MAXNP)
C                                   Angle spec start angle
      REAL         ANGINC(MAXNP)
C                                   Angle spec angle increment
      CHARACTER*80 CARD
C                                   Card image buffer
      LOGICAL      CRDCHK(NTYPES)
C                                   Card type checkoff
      CHARACTER*10 CTYPE (NTYPES)
C                                   Card type names
      REAL         DF
C                                   Delta frequency (hertz)
      REAL         DW
C                                   Delta W
      REAL         DX
C                                   Delta X
      REAL         F1
C                                   Freq. to start ramp up   (hertz)
      REAL         F2
C                                   Freq. to end   ramp up   (hertz)
      REAL         F3
C                                   Freq. to start ramp down (hertz)
      REAL         F4
C                                   Freq. to end   ramp down (hertz)
      REAL         FCUT
C                                   Cutoff frequency (FNYQ - DF)
      REAL         FNYQ
C                                   Nyquist frequency
      INTEGER      ITBEG
C                                   Time to begin using data (MS)
      INTEGER      ITEND
C                                   Time to end   using data (MS)
      INTEGER      ITMAX
C                                   Maximum time data is available (MS)
      INTEGER      ITMIG
C                                   Time to migrate
      INTEGER      ITPAD
C                                   Time to pad at end of trace (MS)
      INTEGER      ITYPE
C                                   Card type index
      INTEGER      JA
C                                   Angle index
      INTEGER      JAS
C                                   Angle spec. index
      INTEGER      JK
C                                   Wavenumber (K) index
      INTEGER      JT
C                                   Time index
      INTEGER      JW
C                                   Frequency index
      INTEGER      K
C                                   Wavenumber index
      INTEGER      KBIAS
C                                   Offset to zero wavenumber
      INTEGER      NA
C                                   Number of angles
      INTEGER      NANG  (MAXNP)
C                                   Angle spec angle count
      INTEGER      NAS
C                                   Number of angle specification cards
      INTEGER      NT0
C                                   Number of times before using NRFFT5
      INTEGER      NXPAD
C                                   Number X's to pad width
      LOGICAL      TSCALE
C                                   Time scaling flag
      REAL         TSCEXP
C                                   Time scaling exponent
      REAL         TSCMLT
C                                   Time scaling multiplier
      REAL         W1
C                                   Freq. to start ramp up   (rad/sec)
      REAL         W2
C                                   Freq. to end   ramp up   (rad/sec)
      REAL         W3
C                                   Freq. to start ramp down (rad/sec)
      REAL         W4
C                                   Freq. to end   ramp down (rad/sec)
C
C  SUBPROGRAMS CALLED:
C
      INTEGER NCFFT5, NRFFT5
C
      EXTERNAL NCFFT5
C                                   Find smallest N to use with CFFTMLT
      EXTERNAL NRFFT5
C                                   Find smallest N to use with RFFTMLT
C
C  STATEMENT FUNCTIONS:
C
      INTEGER IROUND, I, J
      IROUND(I,J) = J * IFIX( FLOAT( I ) / FLOAT( J ) + 0.5 )
C
C  DATA INITIALIZATION:
C
      DATA CRDCHK / NTYPES * .FALSE. /
      DATA CTYPE  / 'TIME      ','FREQUENCY ','WIDTH     ','RECORDS   ',
     &              'ANGLES    ','T SCALE   '/
C
C-----------------------------------------------------------------------
C
 9000 FORMAT (A80)
 9001 FORMAT (10X, 4I10  )
 9002 FORMAT (10X, 4F10.0)
 9003 FORMAT (10X, 3I10  ,  F10.0)
 9004 FORMAT (10X, 3I10  )
 9005 FORMAT (10X,  I10  , 4F10.0)
 9006 FORMAT (10X, 2F10.0)
 9101 FORMAT (/' ', 'TIME       ENDTIMEMS PADTIMEMS BEGTIMEMS',
     &              ' MIGTIMEMS')
 9102 FORMAT (/' ', 'FREQUENCY  MINFREQHZ  F2FREQHZ  F3FREQHZ',
     &              ' MAXFREQHZ')
 9103 FORMAT (/' ', 'WIDTH       BEGTRACE  ENDTRACE  PADTRACE',
     &              '    DELTAX')
 9104 FORMAT (/' ', 'RECORDS      BEG REC   END REC   REC INC')
 9105 FORMAT (/' ', 'ANGLES      # ANGLES     START INCREMENT',
     &              ' BEAMWIDTH   REF VEL')
 9106 FORMAT (/' ', 'T SCALE        ALPHA     XMULT')
 9201 FORMAT ( ' ', 'TIME      ', 4I10  )
 9202 FORMAT ( ' ', 'FREQUENCY ', 4F10.1)
 9203 FORMAT ( ' ', 'WIDTH     ', 3I10  , F10.1)
 9204 FORMAT ( ' ', 'RECORDS   ', 3I10  )
 9205 FORMAT ( ' ', 'ANGLES    ',  I10  , 4F10.1)
 9206 FORMAT ( ' ', 'T SCALE   ', 3F10.4)
 9900 FORMAT ('               CARD TYPE = ', A10/)
 9901 FORMAT (/
     1 ' ***** ERROR - UNRECOGNIZED CARD TYPE *****'/
     2 '               CARD TYPE = ', A10/)
 9903 FORMAT (/
     1 ' ***** ERROR - DIMENSIONS EXCEED ARRAY SIZES *****')
 9904 FORMAT (/
     1 ' ***** ERROR - UNEXPECTED END OF FILE ATTEMPTING TO READ *****'/
     2 '               CARD TYPE = ', A10/)
 9905 FORMAT (/
     1 ' ***** ERROR - INVALID PARAMETER VALUE *****')
 9906 FORMAT (/
     1 ' ***** ERROR - DATA CONVERSION ERROR  *****'/
     2 '               CARD TYPE = ', A10/)
 9907 FORMAT (/
     1 ' ***** ERROR - MISSING CARD TYPE *****'/
     2 '               CARD TYPE = ', A10/)
 9908 FORMAT (/
     1 ' ***** ERROR - UNEXPECTED DATA CARD TYPE *****'/
     2 '               EXPECTED CARD TYPE = ', A10/
     3 '               ACTUAL   CARD TYPE = ', A10/)
 9909 FORMAT (/
     1 ' ***** ERROR - DUPLICATE CARD TYPE *****'/
     2 '               CARD TYPE = ', A10/)
 9910 FORMAT (/
     1 ' ***** ERROR - INSUFFICIENT NUMBER OF ANGLES SPECIFIED *****')
C
C-----------------------------------------------------------------------
C
      TSCALE = .FALSE.
C
C=======================================================================
C                 READ SCALAR PARAMETERS FROM CARD FILE
C=======================================================================
C
 1000 CONTINUE
C
C        READ HEADER CARD
C
         READ (LUCRD, 9000, END=3000) CARD
         IF (CARD(1:1) .EQ. '*' .OR. CARD(1:1) .EQ. '#') GO TO 1000
C
 1010    CONTINUE
         DO 1020 ITYPE = 1, NTYPES
            IF (CARD(1:10) .EQ. CTYPE(ITYPE)) GO TO 1030
 1020    CONTINUE
         GO TO 8010
C
 1030    CONTINUE
         IF (CRDCHK(ITYPE)) GO TO 8090
         CRDCHK(ITYPE) = .TRUE.
C
C        READ DATA CARD
C
 1100    CONTINUE
         READ (LUCRD, 9000, END=8040) CARD
         IF (CARD(1:1) .EQ. '*' .OR. CARD(1:1) .EQ. '#') GO TO 1100
         IF (CARD(1:10) .NE. CTYPE(ITYPE)) GO TO 8080
C
         GO TO (2100, 2200, 2300, 2400, 2500, 2600), ITYPE
C
C        READ CARD TYPE 1 = TIME CARD
C
 2100    CONTINUE
         READ (CARD, 9001, ERR=8060) ITEND, ITPAD, ITBEG, ITMIG
         GO TO 1000
C
C        READ CARD TYPE 2 = FREQUENCY CARD
C
 2200    CONTINUE
         READ (CARD, 9002, ERR=8060) F1, F2, F3, F4
         GO TO 1000
C
C        READ CARD TYPE 3 = WIDTH CARD
C
 2300    CONTINUE
         READ (CARD, 9003, ERR=8060) ITRC1, ITRC2, NXPAD, DX
         GO TO 1000
C
C        READ CARD TYPE 4 = RECORDS CARD
C
 2400    CONTINUE
         READ (CARD, 9004, ERR=8060) IREC1, IREC2, INCREC
         GO TO 1000
C
C        READ CARD TYPE 5 = ANGLE CARDS
C
 2500    CONTINUE
         NAS = 1
         READ (CARD, 9005, ERR=8060) NANG(NAS), ANG(NAS), ANGINC(NAS),
     &                               BWIDTH, VELREF
C
 2510    CONTINUE
            READ (LUCRD, 9000, END=3000) CARD
            IF (CARD(1:1) .EQ. '*' .OR. CARD(1:1) .EQ. '#') GO TO 2510
            IF (CARD(1:10) .NE. CTYPE(ITYPE)) GO TO 1010
            NAS = NAS + 1
            IF (NAS .GT. MAXNP) GO TO 8030
            READ (CARD, 9005, ERR=8060) NANG(NAS), ANG(NAS), ANGINC(NAS)
            GO TO 2510
C
C        READ CARD TYPE 6 = T SCALE CARD
C
 2600    CONTINUE
         READ (CARD, 9006, ERR=8060) TSCEXP, TSCMLT
         TSCALE = .TRUE.
         GO TO 1000
C
C     END OF CARD FILE - INSURE THAT ALL CARD TYPES HAVE BEEN READ
C     EXCEPT FOR ITYPE = 6 (T SCALE) WHICH IS OPTIONAL
C
 3000 CONTINUE
      CRDCHK(6) = .TRUE.
      DO 3010 ITYPE = 1, NTYPES
         IF (.NOT. CRDCHK(ITYPE)) GO TO 8070
 3010 CONTINUE
C
      ITYPE = 0
C
C=======================================================================
C                        PROCESS TIME PARAMETERS
C=======================================================================
C
      ITMAX = IDT * (NSMP - 1)
      IF (ITEND .LE. 0 .OR. ITEND .GT. ITMAX) ITEND = ITMAX
      IF (ITMIG .LE. 0 .OR. ITMIG .GT. ITEND) ITMIG = ITEND
      IF (ITPAD .LE. 0) ITPAD = 0
C
      ITBEG = IROUND( ITBEG, IDT )
      ITEND = IROUND( ITEND, IDT )
      ITPAD = IROUND( ITPAD, IDT )
      ITMIG = IROUND( ITMIG, IDT )
      TMIG  = 0.001 * FLOAT( ITMIG )
C
C     CALCULATE:
C        IT1   = INDEX OF FIRST TIME SAMPLE TO USE
C        IT2   = INDEX OF LAST  TIME SAMPLE TO USE
C        NTOFF = NUMBER OF ZERO SAMPLES BEFORE FIRST SAMPLE
C        NTPAD = NUMBER OF ZERO SAMPLES AFTER LAST SAMPLE (PAD)
C
C        NOTE: FIRST SAMPLE IN TRACE CORRESPONDS TO TIME = 0
C
      IT2   = ITEND / IDT + 1
      NTOUT = ITMIG / IDT + 1
      NTPAD = ITPAD / IDT
C
      IF (ITBEG .GE. 0) THEN
         NTOFF = 0
         IT1   = ITBEG / IDT + 1
      ELSE
         NTOFF = - ITBEG / IDT
         IT1   = 1
      ENDIF
C
C     CALCULATE NT = NUMBER OF TIME SAMPLES FOR FFT, PAD AS NEEDED
C
      NT0   = IT2 - IT1 + 1 + NTOFF + NTPAD
      NT    = NRFFT5( NT0 )
      DT    = 0.001 * FLOAT( IDT )
      NTPAD = NTPAD + NT - NT0
C
C     RECALCULATE ITPAD AND ITEND
C
      ITPAD = IDT * NTPAD
      ITEND = IDT * (IT2 - 1)
C
      WRITE (LUPRT, 9101)
      WRITE (LUPRT, 9201) ITEND, ITPAD, ITBEG, ITMIG
C
      IF (ITEND .LE. ITBEG) GO TO 8050
      IF (NT  .GT. MAXNT) GO TO 8030
C
C=======================================================================
C                     PROCESS FREQUENCY PARAMETERS
C=======================================================================
C
      IF (F2 .LE. 0) F2 = F1
      IF (F3 .LE. 0) F3 = F4
C
      FNYQ = 0.5 / DT
      DF   = 1.0 / (NT * DT)
      FCUT = FNYQ - DF
C
      IF (F1 .GT. FCUT) F1 = FCUT
      IF (F2 .GT. FCUT) F2 = FCUT
      IF (F3 .GT. FCUT) F3 = FCUT
      IF (F4 .GT. FCUT) F4 = FCUT
C
      WRITE (LUPRT, 9102)
      WRITE (LUPRT, 9202) F1, F2, F3, F4
C
      IF (F1.LT.0.0 .OR. F2.LT.F1 .OR. F3.LT.F2 .OR. F4.LT.F3)GO TO 8050
C
C     CALCULATE REMAINING FREQUENCY PARAMETERS
C
      W1 = 2.0 * PI * F1
      W2 = 2.0 * PI * F2
      W3 = 2.0 * PI * F3
      W4 = 2.0 * PI * F4
C
      DW  = 2.0 * PI * DF
      IW1 = IFIX( W1 / DW ) + 1
      IF (DW*(IW1-1) .LT. W1) IW1 = IW1 + 1
      IW2 = IFIX( W4 / DW ) + 1
      IF (W2 .GT. W1 .AND. DW*(IW1-1) .EQ. W1) IW1 = IW1 + 1
      IF (W4 .GT. W3 .AND. DW*(IW2-1) .EQ. W4) IW2 = IW2 - 1
      NW  = IW2 - IW1 + 1
C
      IF (NW .GT. MAXNW) GO TO 8030
C
      DO 3210 JW = 1, NW
         OMEGA(JW) = DW * (IW1 + JW - 2)
         IF      (OMEGA(JW) .LT. W2) THEN
            FILTR(JW) = (OMEGA(JW) - W1) / (W2 - W1)
         ELSE IF (OMEGA(JW) .GT. W3) THEN
            FILTR(JW) = (W4 - OMEGA(JW)) / (W4 - W3)
         ELSE
            FILTR(JW) = 1.0
         ENDIF
 3210 CONTINUE
C
      FMAX = OMEGA(NW) / (2.0 * PI)
C
C=======================================================================
C                       PROCESS WIDTH PARAMETERS
C=======================================================================
C
      IF (ITRC1 .LE. 0) ITRC1  = 1
      IF (ITRC2 .LE. 0) ITRC2  = NTRC
C
      NX    = ITRC2 - ITRC1 + 1
      NK    = NCFFT5( NX + NXPAD )
      DK    = 2.0 * PI / (NK * DX)
      NXPAD = NK - NX
C
      WRITE (LUPRT, 9103)
      WRITE (LUPRT, 9203) ITRC1, ITRC2, NXPAD, DX
C
      IF (ITRC1 .GT. ITRC2) GO TO 8050
      IF (NX .GT. MAXNX .OR. NK .GT. MAXNK) GO TO 8030
C
C     CALCULATE REMAINING WIDTH RELATED PARAMETERS
C
      KBIAS = (NK + 1) / 2
      DO 3310 JK = 1, NK
         K = JK - KBIAS
         AK(JK) = K * DK
 3310 CONTINUE
C
C=======================================================================
C                       PROCESS RECORD PARAMETERS
C=======================================================================
C
      IF (IREC1  .LE. 0) IREC1  = 1
      IF (IREC2  .LE. 0 .OR. IREC2 .GT. NREC) IREC2  = NREC
      IF (INCREC .LE. 0) INCREC = 1
C
      NP    = (IREC2 - IREC1) / INCREC + 1
      IREC2 = IREC1 + (NP - 1) * INCREC
C
      WRITE (LUPRT, 9104)
      WRITE (LUPRT, 9204) IREC1, IREC2, INCREC
C
      IF (IREC1 .GT. IREC2) GO TO 8050
C
C=======================================================================
C                       PROCESS ANGLE PARAMETERS
C=======================================================================
C
      NA = 0
      WRITE (LUPRT, 9105)
      DO 3520 JAS = 1, NAS
         IF (JAS .EQ. 1) THEN
            WRITE (LUPRT, 9205) NANG(JAS), ANG(JAS), ANGINC(JAS),
     &                          BWIDTH, VELREF
         ELSE
            WRITE (LUPRT, 9205) NANG(JAS), ANG(JAS), ANGINC(JAS)
         ENDIF
C
         IF (NANG(JAS) .LE. 0) GO TO 8050
C
         A    = ANG(JAS)
         AINC = ANGINC(JAS)
         DO 3510 JA = 1, NANG(JAS)
            NA = NA + 1
            IF (NA .GT. MAXNP) GO TO 8030
            ANGLE(NA) = A
            A         = A + AINC
 3510    CONTINUE
 3520 CONTINUE
C
      IF (NA .LT. IREC2) GO TO 8100
C
C=======================================================================
C                       PROCESS T SCALE PARAMETERS
C=======================================================================
C
      IF (TSCALE) THEN
         WRITE (LUPRT, 9106)
         WRITE (LUPRT, 9206) TSCEXP, TSCMLT
         DO 3610 JT = 1, NTOUT
            SCALE(JT) = TSCMLT * FLOAT( JT ) ** TSCEXP
 3610    CONTINUE
      ELSE
         DO 3620 JT = 1, NTOUT
            SCALE(JT) = 1.0
 3620    CONTINUE
      ENDIF
C
C=======================================================================
C                              NORMAL EXIT
C=======================================================================
C
      RETURN
C
C=======================================================================
C                              ERROR EXITS
C=======================================================================
C
 8010 CONTINUE
      WRITE (LUPRT, 9901) CARD(1:10)
      IERR = 4001
      RETURN
C
 8030 CONTINUE
      WRITE (LUPRT, 9903)
      IF (ITYPE .GT. 0) WRITE (LUPRT, 9900) CTYPE(ITYPE)
      IERR = 4003
      RETURN
C
 8040 CONTINUE
      WRITE (LUPRT, 9904) CTYPE(ITYPE)
      IERR = 4004
      RETURN
C
 8050 CONTINUE
      WRITE (LUPRT, 9905)
      IERR = 4005
      RETURN
C
 8060 CONTINUE
      WRITE (LUPRT, 9906) CTYPE(ITYPE)
      IERR = 4006
      RETURN
C
 8070 CONTINUE
      WRITE (LUPRT, 9907) CTYPE(ITYPE)
      IERR = 4007
      RETURN
C
 8080 CONTINUE
      WRITE (LUPRT, 9908) CTYPE(ITYPE), CARD(1:10)
      IERR = 4008
      RETURN
C
 8090 CONTINUE
      WRITE (LUPRT, 9909) CARD(1:10)
      IERR = 4002
      RETURN
C
 8100 CONTINUE
      WRITE (LUPRT, 9910)
      IERR = 4010
      RETURN
C
      END
C***********************************************************************
C NAME: CUDUMP                                                         *
C***********************************************************************
C
C  PURPOSE:
C       CUDUMP WRITES SELECTED GLOBAL VARIABLES TO LUPRT.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE CUDUMP( NREC, NTRC, NSMP, IDT, IREC1, IREC2, INCREC,
     &                   ITRC1, ITRC2, IT1, IT2, NT, NTOFF, NTOUT,
     &                   NTPAD, NW, IW1, IW2, NX, NK, NVMOD, NVSMP,
     &                   DK, DT, DZVM, FMAX, BWIDTH, VELREF, ANGLE,
     &                   OMEGA, TMIG, FILTR )
C
#ifdef CRAY
      IMPLICIT NONE
#endif
#include <cubevz.h>
C
C  SUBROUTINE ARGUMENTS
C
      INTEGER NREC, NTRC, NSMP, IDT, IREC1, IREC2, INCREC, ITRC1,
     &        ITRC2, IT1, IT2, NT, NTOFF, NTOUT, NTPAD, NW, IW1, IW2,
     &        NX, NK, NVMOD, NVSMP
      REAL    DK, DT, DZVM, FMAX, BWIDTH, VELREF, ANGLE(*), OMEGA(*),
     &        TMIG, FILTR(*)
C
C  LOCAL VARIABLES:
C
      INTEGER I
C                                 Loop index
C
C-----------------------------------------------------------------------
C
  900 FORMAT (/' ', 'COMMON BLOCK PARAMETERS:')
  901 FORMAT (/' ', 'NUMBER OF INPUT RECORDS     =', I5,
     2        /' ', 'NUMBER OF TRACES PER RECORD =', I5,
     3        /' ', 'NUMBER OF SAMPLES PER TRACE =', I5,
     4        /' ', 'FIRST RECORD                =', I5,
     5        /' ', 'LAST  RECORD                =', I5,
     6        /' ', 'RECORD INCREMENT            =', I5,
     7        /' ', 'START TRACE IN EACH RECORD  =', I5,
     8        /' ', 'END   TRACE IN EACH RECORD  =', I5,
     9        /' ', 'SAMPLE RATE (MS)            =', I5,
     &        /' ', 'NUMBER OF OUTPUT RECORDS    =', I5,
     1        /' ', 'NUMBER OF TRACES/RECORD OUT =', I5,
     2        /' ', 'NUMBER OF SAMPLES/TRACE OUT =', I5)
  902 FORMAT (/' ', 'NUMBER OF W-S,              =', I5,
     2        /' ', 'NUMBER OF X-S,              =', I5,
     3        /' ', 'NUMBER OF T-S, DELTA T      =', I5, E14.5,
     4        /' ', 'NUMBER OF K-S, DELTA K      =', I5, E14.5)
  903 FORMAT (/' ', 'INDEX OF FIRST SAMPLE       =', I5,
     2        /' ', 'INDEX OF LAST  SAMPLE       =', I5,
     3        /' ', 'LEADING  ZERO PAD           =', I5,
     4        /' ', 'TRAILING ZERO PAD           =', I5)
  904 FORMAT (/' ', 'INDEX OF FIRST OMEGA        =', I5,
     2        /' ', 'INDEX OF LAST  OMEGA        =', I5)
  905 FORMAT (/' ', 'REFERENCE VELOCITY          =', F9.3,
     2        /' ', 'BEAMWIDTH                   =', F9.3)
  906 FORMAT (/' ', 'NUMBER OF VELOCITY MODELS   =', I5,
     2        /' ', 'NUMBER OF VELOCITY SAMPLES  =', I5,
     3        /' ', 'VELOCITY SAMPLE INTERVAL    =', F9.3,
     4        /' ', 'MIGRATION TIME (SECONDS)    =', F9.3,
     5        /' ', 'MAXIMUM FREQUENCY (HERTZ)   =', F9.3)
  913 FORMAT (/' ', '    I     OMEGA    FILTER'/)
  914 FORMAT ( ' ', I5, 2F10.3)
  915 FORMAT (/' ', '    I     ANGLE'/)
  916 FORMAT ( ' ', I5, F10.3)
C
C-----------------------------------------------------------------------
C
      WRITE (LUPRT, 900)
      WRITE (LUPRT, 901) NREC, NTRC, NSMP, IREC1, IREC2, INCREC,
     &                   ITRC1, ITRC2, IDT, NVMOD, NX, NTOUT
      WRITE (LUPRT, 902) NW, NX, NT, DT, NK, DK
      WRITE (LUPRT, 903) IT1, IT2, NTOFF, NTPAD
      WRITE (LUPRT, 904) IW1, IW2
      WRITE (LUPRT, 905) VELREF, BWIDTH
      WRITE (LUPRT, 906) NVMOD, NVSMP, DZVM, TMIG, FMAX
C
      WRITE (LUPRT, 913)
      DO 220 I = 1, NW
         WRITE (LUPRT, 914) I, OMEGA(I), FILTR(I)
  220 CONTINUE
C
      WRITE (LUPRT, 915)
      DO 230 I = 1, NREC
         WRITE (LUPRT, 916) I, ANGLE(I)
  230 CONTINUE
C
      RETURN
      END
C***********************************************************************
C NAME: CUMIGR                                                         *
C***********************************************************************
C
C  PURPOSE:
C       CUMIGR DOWNWARD CONTINUES THE WAVE FIELD AND IMAGES.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE CUMIGR( MK, NK, NW, NZSEG, NZT, NP, IREC1, INCREC, DK,
     &                   ANGLE, BWIDTH, VELREF, OMEGA, AK, IZSNZ, ZSDZ,
     &                   ZSSLOW, CXFTAB, WORK, PSI, IMAGE, IERR )
C
#ifdef CRAY
      IMPLICIT NONE
#endif
#include <cubevz.h>
C
C  FORMAL PARAMETERS:
C
      INTEGER MK, NK, NW, NZSEG, NZT, NP, IREC1, INCREC, IERR, IZSNZ(*)
      REAL    DK, ANGLE(*), BWIDTH, VELREF, OMEGA(*), AK(*), ZSDZ(*),
     &        ZSSLOW(*), CXFTAB(*), WORK(*), PSI(*), IMAGE(*)
C
C  LOCAL VARIABLES:
C
      REAL    A0
C                                 Angle (degrees)
      REAL    AM
C                                 Angle minus beam width (degrees)
      REAL    AP
C                                 Angle plus  beam width (degrees)
      INTEGER I1
C                                 Begining value for I
      INTEGER I2
C                                 Ending value for I
      INTEGER I
C                                 Index for reading from scratch file
      INTEGER JA
C                                 Index for ANGLE
      INTEGER JP
C                                 Index for P, PM, PP, & KLIM
      INTEGER JW
C                                 Index for OMEGA
      INTEGER LOC
C                                 Location pointer within scratch file
      INTEGER KLIM(2,MAXNP)
C                                 Lower and upper K limits
      REAL    P(MAXNP)
C                                 Angle ( sin(A) / VELREF )
      REAL    PM(MAXNP)
C                                 Angle minus beam width
      REAL    PP(MAXNP)
C                                 Angle plus  beam width
      REAL    WOVRDK
C                                 W / DK
C
C  SUBPROGRAMS CALLED:
C
      EXTERNAL VZDCIP0
C                                 V of Z downward continue and image
      EXTERNAL VCLR
C                                 Vector clear
C
C-----------------------------------------------------------------------
C
  992 FORMAT (/' ', '***** ERROR: FATAL ERROR REPORTED BY VZDCIP0 - ',
     &              'IERR = ', I8, ' *****'/)
C
C-----------------------------------------------------------------------
C
      CALL VCLR( IMAGE, 1, 2*MK*(NZT+1) )
C
      JA = IREC1
      DO 110 JP = 1, NP
         A0 = ANGLE(JA)
         AM = ANGLE(JA) - (BWIDTH + 10.0)
         AP = ANGLE(JA) + (BWIDTH + 10.0)
         IF (A0 .LT. -90.0) A0 = -90.0
         IF (A0 .GT.  90.0) A0 =  90.0
         IF (AM .LT. -90.0) AM = -90.0
         IF (AM .GT.  90.0) AM =  90.0
         IF (AP .LT. -90.0) AP = -90.0
         IF (AP .GT.  90.0) AP =  90.0
         P (JP) = SIN( A0 * DEG2RAD ) / VELREF
         PM(JP) = SIN( AM * DEG2RAD ) / VELREF
         PP(JP) = SIN( AP * DEG2RAD ) / VELREF
         JA     = JA + INCREC
  110 CONTINUE
C
      LOC = 1
      DO 220 JW = 1, NW
C
         I2 = 0
         DO 205  JP = 1, NP
            I1 = I2 + 1
            I2 = I2 + 2 * NK
            READ( LUTMP, REC = LOC ) ( PSI(I), I = I1, I2 )
            LOC = LOC + 1
  205    CONTINUE
C
         WOVRDK = OMEGA(JW) / DK
         DO 210 JP = 1, NP
            KLIM(1,JP) = NINT( 2.0 * PM(JP) * WOVRDK )
            KLIM(2,JP) = NINT( 2.0 * PP(JP) * WOVRDK )
  210    CONTINUE
C
         CALL VZDCIP0( MK, NK, NP, NZSEG, NZT, OMEGA(JW), AK, P, KLIM,
     &                 IZSNZ, ZSDZ, ZSSLOW, ZSSLOW, CXFTAB, WORK, PSI,
     &                 IMAGE, IERR )
C
         IF (IERR .NE. 0) THEN
            WRITE (LUPRT, 992) IERR
            IERR = 5600 + IABS( IERR )
            RETURN
         ENDIF
C
  220 CONTINUE
C
      RETURN
      END
