c***********************************************************************
c                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c***********************************************************************
c NAME: dspace  Determine disk space needed for the 3 steps of 3dvzmig:
c               (fft3d, cmmvz3d, zkk2zxy)
c
c       11-22-95             Version 1.0              Mary Ann Thornton
c***********************************************************************
      implicit none
c
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c     parameters:
      integer   MAXNW           ! maximum number of omegas
      integer   MAXNK           ! maximum number of k_s
      real      PI              ! PI
      parameter (MAXNW=2048, MAXNK=2048, PI=3.141592653589793)
c
c variables:
c
      real tmax,zmax,strch,twopi,fmax,dtgrid,z,t,dz
      real z_last,dzgrid,t_last
      real zsdz(2048),zsslo(2048)
      integer izsnz(2048),jerr
      logical timel,depth
      integer ispace,nvel,nzbar,nzseg,i,jzseg,jz,nzout
      integer ntr,nwrd,ihdwrd,igreen
      integer   dheadr(SZLNHD)  ! length of input file line header (words)
      real      dkx             ! delta kx
      real      dky             ! delta ky
      real      dx              ! delta x
      real      dy              ! delta y
      integer   idx1000         ! delta x from line header
      integer   idy1000         ! delta y from line header
      real      dt              ! delta t
      real      dw              ! delta w
      real      f1              ! low  cutoff frequency (Hz) for filter
      real      f2              ! low  corner frequency (Hz) for filter
      real      f3              ! high corner frequency (Hz) for filter
      real      f4              ! high cutoff frequency (Hz) for filter
      integer   idt             ! Sampling interval (milliseconds)
      integer   idt0            ! Sampling interval from line header
      integer   ierr            ! error flag
      integer   irec1           ! first input record to use
      integer   irec2           ! last  input record to use
      integer   ismp1           ! first input sample of trace to use
      integer   ismp2           ! last  input sample of trace to use
      integer   itrc1           ! beginning trace
      integer   itrc2           ! ending trace
      integer   iw1             ! index of first omega to keep
      real      kx0             ! first kx
      real      ky0             ! first ky
      integer   luin            ! logical unit of input dataset
      integer   luout           ! logical unit of output dataset
      integer   mbytes          ! MBytes for scratch (default=32, 64-cray)
      integer   nbytes          ! length of input file line header (bytes)
      integer   nkx             ! number of kx_s
      integer   nky             ! number of ky_s
      integer   nrec            ! number of records in input dataset
      integer   nrec2           ! number of records in output dataset
      integer   nrpad           ! number pad record
      integer   nsmp            ! number of samples per trace in input
      integer   nsmp2           ! number of samples per trace in output
      integer   nspad           ! number pad sample
      integer   nt              ! number of times ( samples-editing+pad )
      integer   ntout           ! number of output time samples out
      integer   ntpad           ! number pad trace
      integer   ntrc            ! number of traces per record in input
      integer   ntrc2           ! number of traces per record in output
      character ntap*128        ! name of input file
      integer   nx              ! number of traces to use
      integer   ny              ! number of records to use
      integer   nw              ! number frequencies (omega) kept
      character ppname*5        ! program name
      character title*66        ! title for printout banner
      logical   twod            ! if true, perform fft only on first 2 dim
      logical   verbos          ! verbose output flag
      character version*4       ! version number
      real      w0              ! first omega
      character outfil*54       ! output data dictionary
      character fn_vel*100      ! name of input velocity file
      character datpth*64       ! output data file path
      character datsfx*10       ! output data file suffix
      character output_type*4   ! output type: TIME or DEPT 
      character name*6
      data name/'DSPACE'/, version /' 1.0'/
c
c dynamically allocated arrays in step fft3d:
c
      real      filtr(MAXNW)    ! frequency filter
      real      omega(MAXNW)    ! angular frequency vector
c
c functions:
c
      integer   argis           ! is argument present function
      integer   ncfft5          ! computes integer for use in cfftlmtn
      integer   nrfft5          ! computes integer for use in rfftlmtn
c
c data initialization:
c
      data ierr   / 0  /
      data luin   / -1 /
      data luout  / -1 /
      data ppname / 'DSPACE' /
      data title   /                                                   '
     &                   DSPACE -- Determine Disk Space
     &' /
      data verbos  /.false./
c
c
c===  check for help
c
      if( argis( '-h' ) .gt. 0 .or. argis( '-?' ) .gt. 0 ) then
         call help
         stop
      endif
#include <f77/mbsopen.h>
c-----------------------------------------------------------------------
c     determine space for fft3d step
c-----------------------------------------------------------------------
c
c===  read program parameters from command line
c
      call gcmdln(LER, ntap, f1, f2, f3, f4, irec1, irec2,
     &            nrpad, itrc1, itrc2, ntpad, ismp1, ismp2, nspad,
     &            idt, dx, dy, mbytes, twod, verbos, dzgrid, dtgrid,
     &            outfil, fn_vel, datpth, datsfx, tmax, zmax, strch,
     &            output_type )
c
      call lbopen( luin, ntap, 'r' )
c
c===  read data line header
c
      nbytes = 0
      call rtape( luin, dheadr, nbytes )
      if( nbytes .eq. 0 ) then
         write( LER,* ) ' ***** ERROR reading line header ***** '
         call lbclos(luin)
         stop 100
      endif
c
c===  get parameters from data line header
c
      call saver( dheadr, 'SmpInt', idt0, LINHED )
      call saver( dheadr, 'NumSmp', nsmp, LINHED )
      call saver( dheadr, 'NumTrc', ntrc, LINHED )
      call saver( dheadr, 'NumRec', nrec, LINHED )
c
      if ( dx .le. 0.0 ) then
         call saver( dheadr, 'Dx1000', idx1000, LINHED )
         dx = float(idx1000)/1000.0
         if ( dx .le. 0.0 ) then
             write( LER, * ) ' ***** ERROR: dx not set in line'
     &           ,' header. You must enter a value for dx*****'
             ierr = 1
         endif
      endif
c
      if ( dy .le. 0.0 ) then
         call saver( dheadr, 'Dy1000', idy1000, LINHED )
         dy = float(idy1000)/1000.0
         if ( dy .le. 0.0 ) then
             write( LER, * ) ' ***** ERROR: dy not set in line'
     &           ,' header. You must enter a value for dy*****'
             ierr = 1
         endif
      endif
c
c===  set some parameters using these values
c
      if( idt   .lt. 0 ) idt   = idt0
      if( ismp2 .lt. 0 ) ismp2 = nsmp
      if( itrc2 .lt. 0 ) itrc2 = ntrc
      if( irec2 .lt. 0 ) irec2 = nrec
      if(output_type.eq.'TIME' .and. dtgrid.le.0.0)dtgrid=idt
c
c===  check parameters
c
      if( ( irec1 .le. 0 ) .or. ( irec1 .gt. irec2 ) ) then
          write( LER, * ) ' ***** ERROR: first record must be less  ',
     &        'or equal to last record on input dataset *****'
         irec1 = 1
         ierr = 1
      endif
c
      if( irec2 .gt. nrec ) then
          write( LER, * )
     &       ' ***** ERROR: last record must not be greater than ',
     &       '             ',nrec
          irec2 = nrec
          ierr = 1
      endif
c
      if( ( ismp1 .le. 0 ) .or. ( ismp1 .gt. ismp2 ) ) then
         write( LER, * ) ' ***** ERROR: first sample to use must ',
     &         'be less than last sample on input trace ****'
         ismp1 = 1
         ierr = 1
      endif
c
      if( ismp2 .gt. nsmp ) then
         write( LER, * )
     &        ' ***** ERROR: last sample must be less than or ',
     &        '              equal to last sample on input ***** '
         ismp2 = nsmp
         ierr = 1
      endif
c
c===  calculate dt and number of time samples for fft, pad as needed
c
      dt = 0.001 * float( idt )
cdan
      ntout = ismp2 - ismp1 + 1
      nt = nrfft5( ismp2 - ismp1 + 1 + nspad )
c
      if(dt.le.0.0) then
         write( LER, * ) ' ***** ERROR: dt must be positive *****'
         call lbclos(luin)
         stop 100
      endif
c
c===  calculate nx, ny, nkx, nky, dkx, dky, kx0, ky0
c
      nx  = itrc2 - itrc1 + 1
      ny  = irec2 - irec1 + 1
c
c===  check if 2D because only one record to use
c
      if( ny .eq. 1 ) twod = .true.
c
      nkx = ncfft5( nx + ntpad )
      dkx = ( 2.0*PI ) / ( nkx*dx )
      kx0 = ( 1 - nkx ) / 2 * dkx
c
      if( ( nkx .le. 0 ) .or. ( nkx .gt. MAXNK ) ) then
         write( LER, * ) ' ***** ERROR: number of traces plus',
     &          ' the pad must be less than or equal to',MAXNK,
     &          ' after the fft ***** '
         call lbclos (luin)
         stop 100
      endif
c
      if( .not. twod ) then
cdan**** nky = ncfft5( nrec + nrpad )
         nky = ncfft5( ny + nrpad )
         dky = ( 2.0*PI ) / ( nky*dy )
         ky0 = ( 1 - nky ) / 2 * dky
c
         if( ( nky .le. 0 ) .or. ( nky .gt. MAXNK ) ) then
            write( LER, * ) ' ***** ERROR: number of records plus',
     &             ' the pad must be less than',MAXNK,' after fft *****'
            call lbclos(luin)
            stop 100 
         endif
      endif
c
c===  check and adjust frequency filter parameters
c
      if( f1 .le. 0.0 ) f1 = 0.0
      if( f4 .le. 0.0 ) f4 = float( nt/2 - 1 ) / ( float( nt ) * dt )
      if( f2 .le. 0.0 ) f2 = f1
      if( f3 .le. 0.0 ) f3 = f4
c
c===  determine no. omegas (angular frequencies)
c
      call gtfltr( f1, f2, f3, f4, nt, dt, MAXNW,
     &             nw, iw1, omega, filtr, ierr )

      w0 = omega(1)
      dw = omega(2) - omega(1)
c
      if (ierr .ne. 0) then
         write( LER, * ) ' ***** ERROR in gtfltr *****'
         write( LER, * ) ' check frequencies, Maximum number'
         write( LER, * ) ' allowed is ',MAXNW
         call lbclos(luin)
         stop 100
      endif
c
c===  determine space needed for this step
c
      nsmp2 = 2 * nw
      ntrc2 = nkx
      if( .not. twod ) then
         nrec2 = nky
      else
         nrec2 = ny
      endif
c
      write(LERR,*)' APPROXIMATE SIZES: '
      write(LER ,*)' APPROXIMATE SIZES: '
      write(LERR,*)' FFT3D: '
      write(LER ,*)' FFT3D: '
      write(LERR,*)' Output no. samples/trace: ',nsmp2
      write(LER ,*)' Output no. samples/trace: ',nsmp2
      write(LERR,*)' Output no. traces/record: ',ntrc2
      write(LER ,*)' Output no. traces/record: ',ntrc2
      write(LERR,*)' Output no. records      : ',nrec2
      write(LER ,*)' Output no. records      : ',nrec2
      ntr = ntrc2*nrec2
      nwrd= ntr*nsmp2
      ihdwrd=ntr*ITRWRD
      igreen=ntr*4+100
      nwrd = nwrd+ihdwrd
      ispace = nwrd*ISZBYT + nbytes + igreen
      write(LERR,*)' Output dataset size in bytes ',ispace
      write(LER ,*)' Output dataset size in bytes ',ispace

c-----------------------------------------------------------------------
c     determine space for cmmvz3d step
c-----------------------------------------------------------------------
c read program parameters from command line
c
      zmax   = zmax   * strch
      dzgrid = dzgrid * strch
c
c get the velocity model
c
      twopi = 2.0 * acos( -1.0 )
      fmax  = ( w0 + dw * ( nw - 1 ) ) / twopi

      open( unit=LUCARD, file=fn_vel, status='old', iostat=ierr )
      if( ierr .ne. 0 ) then
         write( LER, * ) '*** ERROR: cannot open velocity file ***'
         ierr = 1
         stop 100
      endif

      call rdvmod( verbos, nvel, fmax, dtgrid, tmax,
     &             dzgrid, zmax, strch,
     &             nzbar, nzseg, izsnz, zsdz, zsslo, ierr )

      close(LUCARD)

      nsmp2 = 2 * nzbar
      write(LERR,*)' CMMVZ3D: '
      write(LER ,*)' CMMVZ3D: '
      write(LERR,*)' Output no. samples/trace: ',nsmp2
      write(LER ,*)' Output no. samples/trace: ',nsmp2
      write(LERR,*)' Output no. traces/record: ',ntrc2
      write(LER ,*)' Output no. traces/record: ',ntrc2
      write(LERR,*)' Output no. records      : ',nrec2
      write(LER ,*)' Output no. records      : ',nrec2
      ntr = ntrc2*nrec2
      nwrd= ntr*nsmp2
      ihdwrd=ntr*ITRWRD
      igreen=ntr*4+100
      nwrd = nwrd+ihdwrd
      ispace = nwrd*ISZBYT + nbytes + igreen
      write(LERR,*)'Output dataset size in bytes ',ispace
      write(LER ,*)'Output dataset size in bytes ',ispace
c-----------------------------------------------------------------------
c     determine space for zkk2zxy step
c-----------------------------------------------------------------------
      i = 0
      z = 0.0
      t = 0.0
      do jzseg = 1, nzseg
         dz = zsdz(jzseg)
         dt = 1000.0 * dz * zsslo(jzseg)
         do jz = 1, izsnz(jzseg)
            i = i + 1
            z = z + dz
            t = t + dt
         enddo
      enddo

      z_last = z
      t_last = t

      depth = .false.
      timel = .false.
      if( output_type .eq. 'DEPT') depth = .true.
      if( output_type .eq. 'TIME') timel = .true.
      if(.not.depth .and. .not.timel)then
         write(LER,*)' You must enter the output type: DEPTH or TIME'
         call lbclos(luin)
         stop 100
      endif

      if( depth ) then
         if( zmax .gt. 0.0 ) then
            zmax = min (zmax, z_last )
         else
            zmax = z_last
         endif
         nzout = nint( zmax / dzgrid )
         ntout = nzout
      endif
c
      if( timel ) then
         if( tmax .gt. 0.0 ) then
            tmax = min (tmax, t_last )
         else
            tmax = t_last
         endif
         ntout = nint( tmax / dtgrid )
         nzout = ntout
      endif

      nsmp2 = nzout+1
      ntrc2 = itrc2-itrc1+1
      nrec2 = irec2-irec1+1

      write(LERR,*)' ZKK2ZXY: '
      write(LER ,*)' ZKK2ZXY: '
      write(LERR,*)' Output no. samples/trace: ',nsmp2
      write(LER ,*)' Output no. samples/trace: ',nsmp2
      write(LERR,*)' Output no. traces/record: ',ntrc2
      write(LER ,*)' Output no. traces/record: ',ntrc2
      write(LERR,*)' Output no. records      : ',nrec2
      write(LER ,*)' Output no. records      : ',nrec2
      ntr = ntrc2*nrec2
      nwrd= ntr*nsmp2
      ihdwrd=ntr*ITRWRD
      igreen=ntr*4+100
      nwrd = nwrd+ihdwrd
      ispace = nwrd*ISZBYT + nbytes + igreen
      write(LERR,*)'Output dataset size in bytes ',ispace
      write(LER ,*)'Output dataset size in bytes ',ispace
      stop
      end
C * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'dspace will determine the amount of disk space necessary'
        write(LER,*)
     :' for each of the three steps of 3dvzmig ',
     :' (fft3d, cmmvz3d, & zkk2zxy)'
        write(LER,*)
     :'The easiest way to use this program is to build an '
        write(LER,*)
     :'executable script by typing catpat 3dvzmig > 3dvzmig.job .'
        write(LER,*)
     :'Then edit the 3dvzmig.job and set DSPACE=YES '
        write(LER,*)
     :'and set FFT3D=NO and CMMVZ3D=NO before you execute the script.'
        write(LER,*)
     :'***************************************************************'
      return
      end
 
