C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C NAME: CFFTMLTU  MULTIPLE CFFT, USP OUT-OF-CORE       REV 1.0  MAR 95 *
C***********************************************************************
C
C  PURPOSE:
C       Performs multiple in-place, mixed-radix, complex FFT's on the
C       slowest varying dimension (i.e., across records) of a 3D USP
C       data set.  The results are in natural order.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       Revision 1.0    MAR 95          Jerry Cooperstein, CETech
C
C  CALLING FORMAT:
C       CALL CFFTMLTU( LTAPE, IDIR, NSMP, NTRC, NREC, NWORK, WORK, IERR )
C
C  PARAMETERS:
C       LTAPE   INTEGER INPUT SCALAR
C               Logical unit number of USP data set.
C
C       IDIR    INTEGER INPUT SCALAR
C               Direction flag:
C                  IDIR = -1,  perform forward FFT
C                  IDIR =  1,  perform inverse FFT
C
C       NSMP    INTEGER INPUT SCALAR
C               Number of samples per trace.  Note: The input data is
C               complex; thus, each trace contains NSMP/2 complex
C               elements with the real and imaginary components in
C               alternate samples.
C
C       NTRC    INTEGER INPUT SCALAR
C               Number of traces per record.
C
C       NREC    INTEGER INPUT SCALAR
C               Number of records per line.  Note: NREC must equal
C               (2**p) * (3**q) * (5**r) where p, q, and r are
C               non-negative integers.
C
C       NWORK   INTEGER INPUT SCALAR
C               Size in words of scratch array WORK.
C
C       WORK    REAL SCRATCH ARRAY OF LENGTH NWORK
C               Scratch array.
C
C       IERR    INTEGER OUTPUT SCALAR
C               Completion code:
C                  IERR = 0,  normal completion
C                  IERR = 1,  invalid direction flag
C                  IERR = 2,  invalid dimension
C                  IERR = 3,  input/output error
C                  IERR = n,  n > 3, insufficient scratch space.  n =
C                             minimum number of words required.
C
C  DESCRIPTION:
C       Performs multiple in-place, mixed-radix, complex FFT's on the
C       slowest varying dimension (i.e., across records) of a 3D USP
C       data set.  The results are in natural order.
C
C       To be valid, the dimensions must be positive plus NSMP must be
C       even and NREC must be (2**p)*(3**q)*(5**r) where p, q, and r are
C       non-negative integers.
C
C       In general, the larger that NWORK is, the faster the subroutine
C       will execute.  The minimun value for NWORK is
C          NREC*( 3*NSMP + ITRWRD + 2 ) for Cray systems,
C          NREC*( NSMP + ITRWRD + 6 ) + 9*NSMP + 41 for non-Cray systems
C
C  SUBPROGRAMS CALLED:
C       CFFTMLTN, SISSEEK, RDTAPE, WRTAPE
C
C  ERROR CONDITIONS:
C       If an error is detected (see above), the completion code is set
C       to the appropriate value and the subroutine is aborted.
C
c***********************************************************************
      subroutine cfftmltu( ltape, idir, nsmp, ntrc, nrec, nwork, work
     &                     , ierr )
c***********************************************************************
      implicit none
c
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c
c===  input parameters:
c
      integer   ltape   ! Logical unit number of USP data set.
      integer   idir    ! direction flag -1(+1), forward(inverse) FFT
      integer   nsmp    ! number of samples per trace
      integer   ntrc    ! number of traces per record
      integer   nrec    ! number of records per line
      integer   nwork   ! size (words) of array work
c
c===  scratch parameters:
c
      real      work(*) ! scratch array; 
c
c===  output parameters:
c
      integer   ierr    ! completion code
c
c===  local variables:
c
      integer   idata   ! starting index for data array in array work
      integer   ifftwrk ! starting index for fftwrk array in array work
      integer   itrig   ! starting index for fft table in array work
      integer   ld      ! leading dimension of data array
      integer   ldata   ! length of data slice array
      integer   lfftwrk ! length of fft scratch space array
      integer   lplane  ! words required per each (smp,rec) plane
      integer   ltrig   ! length of fft table
      integer   nchk    ! check to see if nrec is a valid fft length
      integer   nplanes ! number of planes per slice
      integer   nwmin   ! minimum size for nwork
c
c===  functions
c
      integer   ncfft5          ! computes integer for use in cfftmlt
c***********************************************************************
      ierr = 0
c
c===  validate values of idir, nsmp, ntrc, & nrec
c
      if( ( idir .ne. 1 ) .and. ( idir .ne. -1 ) ) then
         write( LER, * )' ***** idir must be +1 or -1, not idir= '
     &           , idir, ' *****'
         ierr = 1
         return
      endif
c
      if( ( nsmp .le. 0 ) .or. ( mod( nsmp, 2 ) .ne. 0 ) ) then
         write( LER, * )' ***** nsmp must be > 0 and even, not nsmp= '
     &          , nsmp, ' *****'
         ierr = 2
         return
      endif
c
      if( ntrc .le. 0  ) then
         write( LER, * )' ***** ntrc must be > 0,  not ntrc= '
     &          , ntrc, ' *****'
         ierr = 2
         return
      endif
c
      if( nrec .le. 0  ) then
         write( LER, * )' ***** nrec must be > 0,  not nrec= '
     &          , nrec, ' *****'
         ierr = 2
         return
      endif
c
      nchk = ncfft5( nrec )
      if( nchk .ne. nrec ) then
         write( LER, * )' ***** nrec = ', nrec, ' is not a valid fft'
     &        , ' length, next valid length is =', nchk, ' *****'
         ierr = 2
         return
      endif
c
c=== NOTE MORE COMPLEX CHECK IS NEEDED!!!!!!!
c
c***********************************************************************
c
c===  verify nwork >= nwmin_nwork
c
#ifdef CRAY
      nwmin = nrec * ( 3*nsmp + ITRWRD + 2 )
#else
      nwmin = nrec * ( nsmp   + ITRWRD + 6 ) + 9*nsmp + 41
#endif
c
      if( nwork .lt. nwmin ) then
         write( LER, * )' ***** nwork must be > nwmin, nwmin = '
     &          , nwmin, ' nwork = ', nwork
         ierr = nwmin
         return
      endif
c
c***********************************************************************
c
c===  calculate lengths of work arrays and allocate space
c
      ltrig   = 2 * nrec
c
#ifdef CRAY
      lfftwrk = 2 * nsmp * nrec
#else
      lfftwrk = 4 * nrec + 9 * nsmp + 41
#endif

c      ldatamin = nwmin - ltrig - lfftwrk = nrec*(nsmp+ITRWRD)
c
      ldata   = nwork - ltrig - lfftwrk
      itrig   = 1
      ifftwrk = itrig   + ltrig
      idata   = ifftwrk + lfftwrk
c
      ld      = ITRWRD + nsmp
      lplane  = nrec * ld
      nplanes = ldata / lplane
c
c***********************************************************************
c
c===  calculate transform
c
      call cfftmltu01( ltape, idir, nsmp, ntrc, nrec, ld, nplanes,
     &             work(itrig), work(ifftwrk), work(idata), ierr )
c
      if( ierr .ne. 0 ) then
         write( LER, * ) ' **** error in cfftmltu01, ierr = '
     &           , ierr, ' *****'
         return
      endif
c
c***********************************************************************
      return
      end
c***********************************************************************
c***********************************************************************
      subroutine cfftmltu01( ltape, idir, nsmp, ntrc, nrec, ld, nplanes,
     &                       trig, fftwrk, data, ierr )
c***********************************************************************
c
      implicit none
c
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c
c===  input parameters:
c
      integer   ltape   ! logical unit number of USP data set.
      integer   idir    ! direction flag -1(+1), forward(inverse) FFT
      integer   nrec    ! number of records per line
      integer   nsmp    ! number of samples per trace
      integer   ntrc    ! number of traces per record
      integer   ld      ! leading dimension of data array
      integer   nplanes ! number of planes per slice
c
c===  scratch parameters:
c
      real      trig(*)    ! table for FFTs
      real      fftwrk(*)  ! workspace for FFT
      real      data(ld,nplanes,nrec) ! data slice
c
c===  output parameters:
c
      integer   ierr    ! completion code
c
c===  local variables:
c
      integer   ifac(19)! factors of nrec from cftfax
      integer   itrc1   ! first trace in slice
      integer   itrc2   ! last trace in slice
      integer   jplane  ! plane index
      integer   jrec    ! record index
      integer   jslice  ! slice index
      integer   nbytes  ! number of bytes in record
      integer   npslice ! number of planes in slice
      integer   nslices ! number of slices
      real      scale   ! scale factor for transform
c
c     real second, time0, timefft, timerd, timewr ! timing variables
      real timefft, timerd, timewr ! timing variables
c
c***********************************************************************
      ierr = 0
c
      timefft  = 0.0
      timerd   = 0.0
      timewr   = 0.0
c
c===  generate fft table
c
      call cftfax( nrec, ifac, trig )
c
c===  get number of slices
c
      nslices = ( ntrc - 1 ) / nplanes + 1
c
c===  begin loop over slices: note last slice may be partial
c
      do jslice = 1, nslices
c
         itrc1 = 1 + nplanes*(jslice-1)
         itrc2 = itrc1 + nplanes - 1
         if( itrc2 .gt. ntrc ) itrc2 = ntrc
         npslice = itrc2 - itrc1 + 1
c
c======  load a slice of data set into data array
c
cmat     time0 = second()
c
c======  begin loop over records
         do jrec = 1, nrec
c
            call rwd( ltape )
            call sisseek( ltape, itrc1 + ntrc*(jrec-1) )
c
c=========  begin loop over planes within current slice
            do jplane = 1, npslice
c
               call rtape( ltape, data(1,jplane,jrec), nbytes )
c
               if( nbytes .eq. 0 ) then
                  write( LER , * )' ***** ERROR reading tape, '
     &              , ' jrec,jslice,jplane = ', jrec, jslice, jplane
                  ierr = 3
                  return
               endif
c
            enddo
c=========  end of loop over planes within current slice
c
         enddo
c======  end of loop over records
c
cmat     timerd = timerd + second() - time0
c
c======  perform fft on slice a plane at a time
c
cmat     time0 = second()
c
c======  begin loop over planes within current slice
         do jplane = 1, npslice
c
            call cfftmltn( data(1+ITRWRD,jplane,1),
     &                     data(2+ITRWRD,jplane,1),
     &                     fftwrk, trig, ifac,
     &                     nplanes*(ITRWRD+nsmp), 2,
     &                     nrec, nsmp/2, idir )
c
         enddo
c======  end of loop over planes within current slice
c
cmat     timefft = timefft + second()-time0
c
c======  store a slice of data set back to where it came from on disk
c
         scale = 1.0 / float( nrec )
cmat     time0 = second()
c
c======  begin loop over records
         do jrec = 1, nrec
c
            call rwd( ltape )
            call sisseek( ltape, itrc1 + ntrc*(jrec-1) )
c
c=========  begin loop over planes within current slice
            do jplane = 1, npslice
c
c============  scale data on the fly if idir = -1
c
               if (idir .eq. -1 )
     &         call vsmul( data(1+ITRWRD,jplane,jrec), 1, scale,
     &                     data(1+ITRWRD,jplane,jrec), 1, nsmp )
c
               call wrtape( ltape, data(1,jplane,jrec), nbytes )
c
               if( nbytes .eq. 0 ) then
                  write( LER , * )' ***** ERROR writing tape, '
     &              , ' jrec = ', jrec
                  ierr = 3
                  return
               endif
c
            enddo
c=========  end of loop over planes within current slice
c
         enddo
c======  end of loop over records
c
cmat     timewr = timewr + second() - time0
c
      enddo
c===  end of loop over slices
c
c===  print out timing data
c
cmat  write( *, '('' timefft = '',f10.3,'' timeread = '',f10.3
cmat &           , '' timewrite = '',f10.3)' ) timefft, timerd, timewr

c***********************************************************************
      return
      end
c***********************************************************************
