c***********************************************************************
c                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c***********************************************************************
c NAME: FFT3D    3D FFT                         REV 1.0  MAR 95 ********
c***********************************************************************
c
c  HISTORY:       MAR 95          REV 1.0         J. Cooperstein, CETech
c
c  Revised: Mary Ann Thornton  Rev 1.4         December 12, 1995
c     changed "stop" to "call exitfu(-)" so the status upon exit could be
c     captured and checked in the script 3dvzmig.pat
c  Revised: Mary Ann Thornton  Rev 1.5         April 3, 1996
c     (1)Changed maxnk from 1024 to 2048
c     (2)rdrec.F (from ~mbs/src/lib/libmbs) was moved into this   
c       directory, and the 30000 in the StaCor trace header word set to
c       zero. Trace headers that were previously attached to incoming
c       zero traces, were being attached to outgoing non-zero traces and
c       causing these traces to get zeroed out when the data was read in
c       in the next step (cmmvz3d). This version of rdrec 
c       (called rdrecn.F) will set the StaCor to 0 on dead input traces.
c***********************************************************************
      program fft3d
c***********************************************************************
c     fft3d - inputs a usp file in (t,x,y), performs a 3D fft and applies a
c     trapezoidal filter over frequency, outputs a usp file in
c     (w,kx,ky) and a DDS style data dictionary.  Also performs any
c     required data editing including padding.
c***********************************************************************
      implicit none
c
#include     <save_defs.h>
#include     <f77/iounit.h>
#include     <f77/lhdrsz.h>
#include     <f77/sisdef.h>

c
c parameters:
c
      integer   MAXNW           ! maximum number of omegas
      integer   MAXNK           ! maximum number of k_s
      integer   jerr            ! 
      real      PI              ! PI
      parameter (MAXNW = 4096, MAXNK = 8192 , PI = 3.141592653589793 )
c
c variables:
c
cmat  real time0, time2d, time3d, second ! timing variables
c
      integer   debug           ! debug output flag
      integer   dheadr(SZLNHD)  ! length of input file line header (words)
      real      dkx             ! delta kx
      real      dky             ! delta ky
      real      dx              ! delta x
      real      dy              ! delta y
      real      dt              ! delta t
      real      dw              ! delta w
      real      f1              ! low  cutoff frequency (Hz) for filter
      real      f2              ! low  corner frequency (Hz) for filter
      real      f3              ! high corner frequency (Hz) for filter
      real      f4              ! high cutoff frequency (Hz) for filter
      integer   ic              ! character index in title construction
      integer   idir            ! direction flag -1(+1), forward(inverse) FFT
      integer   idt             ! Sampling interval (milliseconds)
      integer   idt0            ! Sampling interval from line header
      integer   idx1000         ! dx * 1000 in line header
      integer   idy1000         ! dy * 1000 in line header
      integer   ierr            ! error flag
      logical   ifft            ! perform inverse fft_s (NOT IMPLEMENTED)
      integer   irec1           ! first input record to use
      integer   irec2           ! last  input record to use
      integer   ismp1           ! first input sample of trace to use
      integer   ismp2           ! last  input sample of trace to use
      integer   itrc1           ! beginning trace
      integer   itrc2           ! ending trace
      integer   iw1             ! index of first omega to keep
      integer   jc              ! character index in title construction
      integer   jk              ! k index for transposition
      integer   jrec            ! index of current input record
      real      kx0             ! first kx
      real      ky0             ! first ky
      integer   luinp           ! logical unit of input dataset
      integer   luout           ! logical unit of output dataset
      integer   mbytes          ! MBytes for scratch (default=32, 64-cray)
      integer   mt              ! row dimension of input DATA (MT = NT+3)
      integer   nbytes          ! length of input file line header (bytes)
      integer   nkx             ! number of kx_s
      integer   nky             ! number of ky_s
      integer   nrec            ! number of records in input dataset
      integer   nrec2           ! number of records in output dataset
      integer   nrpad           ! number pad record
      integer   nsmp            ! number of samples per trace in input
      integer   nsmp2           ! number of samples per trace in output
      integer   nspad           ! number pad sample
      integer   nt              ! number of times ( samples-editing+pad )
      integer   ntoff           ! number of zero samples at start of trace
      integer   ntout           ! number of output time samples out
      integer   ntpad           ! number pad trace
      integer   ntrc            ! number of traces per record in input
      integer   ntrc2           ! number of traces per record in output
      character ntap*128        ! name of input file
      integer   nx              ! number of traces to use
      integer   ny              ! number of records to use
      integer   nw              ! number frequencies (omega) kept
      character otap*128        ! name of output file
      character ppname*5        ! program name
      character title*66        ! title for printout banner
      logical   twod            ! if true, perform fft only on first 2 dim
      logical   verbos          ! verbose output flag
      character version*4       ! version number
      real      w0              ! first omega
      character name*5          ! name of code
      character outfil*54       ! output data dictionary
      character datpth*64       ! output data file path
      character datsfx*10       ! output data file suffix
c
c dynamically allocated arrays:
c
      real      filtr(1)        ! frequency filter
      integer*2 headr(1)        ! trace headers, current record
      real      omega(1)        ! angular frequency vector
      real      rec(1)          ! scratch space for nt*nx input record
      real      trace(1)        ! scratch space, trace (including header)
      real      work(1)         ! scratch space for output rec, cfftlmtu
c
c variables for length (in words) of dynamically allocated arrays:
c
      integer   lfiltr, lheadr, lomega, lrec, ltrace, lwork2, lwork
cdan
c
c Absoft doesn't support integer*8 - we wouldn't have that much space anyway
c
#ifdef _NO_INTEGER_8
      integer lwork_8
#else
      integer*8 lwork_8
#endif
c
c functions:
c
      integer   argis           ! is argument present function
      integer   ncfft5          ! computes integer for use in cfftlmtn
      integer   nrfft5          ! computes integer for use in rfftlmtn
c
c pointers:
c
      pointer ( pfiltr,  filtr )
      pointer ( pheadr,  headr )
      pointer ( pomega,  omega )
      pointer ( prec,    rec  )
      pointer ( ptrace,  trace )
      pointer ( pwork,   work  )
c
c data initialization:
c
      data name /'FFT3D'/
      data ierr   / 0  /
      data luinp  / -1 /
      data luout  / -1 /
      data ntoff  / 0  /
      data ppname / 'FFT3D' /
      data title   /                                                   '
     &                        FFT3D -- 3D FFT
     &' /
      data verbos  /.false./
      data version /'1.5 ' /
c
c***********************************************************************
c
c format statements:
c
  900 format(
     &'***************************************************************'/
     &'  fft3d - inputs a usp file in (t,x,y), performs a 3D fft '/
     &'  and applies a trapezoidal filter over frequency, outputs'/
     &'  a usp file in (w,kx,ky) and a DDS style data dictionary.'/
     &'  Also performs any required data editing including padding.'//
     &' Execute fft3d by typing fft3d and the list of program'/
     &' parameters.  Note that each parameter is proceeded by -a where'/
     &' "a" is a character(s) corresponding to some parameter.'/
     &' Users enter the following parameters, or use default values:'/)
  901 format(
     &' -N<ntap>   - Full path of input file           (no default)'/
     &' -f<f1>     - Lower cut-off frequency of filter (default=0.0)'/
     &' -fc<f2>    - Lower corner frequency of filter  (default=f1)'/
     &' -FC<f3>    - Upper corner frequency of filter  (default=f4)'/
     &' -F<f4>     - Upper cut-off frequency of filter (default=nyquist'
     &                                                          ,'-1)'/
     &' -BR<irec1> - Begining record                 (default=1)'/
     &' -ER<irec2> - Ending record                   (default=NumRec)'/
     &' -PR<nrpad> - Pad record                      (default=0)'/
     &' -BT<itrc1> - Begining trace                  (default=1)'/
     &' -ET<itrc2> - Ending trace                    (default=NumTrc)'/
     &' -PT<ntpad> - Pad trace                       (default=0)'/
     &' -BS<ismp1> - Begining sample                 (default=1)'/
     &' -ES<ismp2> - Ending sample                   (default=NumSmp)'/
     &' -PS<nspad> - Pad sample                      (default=0)'/
     &' -DTMS<dt>  - sample interval in ms           (default=SmpInt)'/
     &' -DX<dx>    - delta x                         (default=Dx1000)'/
     &' -DY<dy>    - delta y                         (default=Dy1000)')
#ifdef CRAY
  902 format(
     &' -M<mbytes> - number of MBytes for scratch      (default=64)')
#else
  902 format(
     &' -M<mbytes> - number of MBytes for scratch      (default=32)')
#endif
  903 format(
     &' -2D        - only perform fft on first 2 dimensions'/
     &' -IFFT      - perform inverse fft_s (NOT IMPLEMENTED)'/
     &' -V         - specifies verbose printout')
  904 format(
     &' out=<outfile>        - Output data dictionary  (no default)'/
     &' data_path=<path>     - Output data file path   (default=".")'/
     &' data_suffix=<suffix> - Output data file suffix (default=".usp")'
     &,' '//
     &' usage: fft3d -N<ntap> -f<f1> -fc<f2> -FC<f3> -F<f4> '/
     &'             -BR<irec1> -ER<irec2> -PR<nrpad>'/
     &'             -BT<itrc1> -ET<itrc2> -PT<ntpad>'/
     &'             -BS<ismp1> -ES<ismp2> -PS<nspad> -DTMS<dt>'/
     &'             -DX<dx> -DY<dy> -M<mbytes> -2D -IFFT -V'/
     &'       out=<outfile> data_path=<path> data_suffix=<suffix>'/
     &'***************************************************************')
  910 format( /' ', ' INPUT DATASET    = '       / a128,
     &        /' ', ' OUTPUT DATASET   = '       / a128 )
  911 format(/10x,' *********** PERFORMING 2D TRANSFORM ************'/ )
  912 format(/10x,' *********** PERFORMING 3D TRANSFORM ************'/ )
  913 format( /' ', 'Input file line header values:'/
     1         ' ', '   Number of records       =', i6/
     2         ' ', '   Number of traces/record =', i6/
     3         ' ', '   Number of samples/trace =', i6)
  914 format( /' ', 'Output file line header values:'/
     1         ' ', '   Number of records       =', i6/
     2         ' ', '   Number of traces/record =', i6/
     3         ' ', '   Number of samples/trace =', i6)
  920 format( /' ', 'Command line arguments (adjusted):'/
     &         '    irec1    =', i6/
     &         '    irec2    =', i6/
     &         '    nrpad    =', i6/
     &         '    itrc1    =', i6/
     &         '    itrc2    =', i6/
     &         '    ntpad    =', i6/
     &         '    ismp1    =', i6/
     &         '    ismp2    =', i6/
     &         '    nspad    =', i6/
     &         '    idt      =', i6/
     &         '    dx       =', f13.6/
     &         '    dy       =', f13.6/
     &         '    mbytes   =', i6/
     &         '    twod     =', l6/
     &         '    ifft     =', l6)
  921  format( '    f1       =', f8.1/
     &         '    f2       =', f8.1/
     &         '    f3       =', f8.1/
     &         '    f4       =', f8.1/
     &         '    nw       =', i6/
     &         '    iw1      =', i6/
     &         '    w1       =', f13.6/
     &         '    dw       =', f13.6 )
  922 format(  '    nkx      =', i6/
     &         '    nky      =', i6/
     &         '    nt       =', i6/
     &         '    mt       =', i6/
     &         '    nx       =', i6)
  923 format(/' Dynamically allocated arrays: '/
     &        '    lheadr   =', i10/
     &        '    ltrace   =', i10/
     &        '    lrec     =', i10/
     &        '    lwork    =', i10/
     &        '    lwork2   =', i10)
9998  format( /' ', '***** NORMAL COMPLETION *****'/ )
9999  format( /' ', '***** ABNORMAL COMPLETION CODE = ', i4, ' *****'/ )
c
c***********************************************************************
c
c===  check for help
c
      if( argis( '-h' ) .gt. 0 .or. argis( '-?' ) .gt. 0 ) then
         write( LER, 900 )
         write( LER, 901 )
         write( LER, 902 )
         write( LER, 903 )
         write( LER, 904 )
         call exitfu(0)
      endif
c
c===  open printout file
c
      call openpr( LUPPRT, LUPRT, ppname, ierr)
c
      if( ierr .ne. 0 ) then
         write( LER,* ) '***** ERROR: ierr = ',ierr
     &       ,' reported by openpr *****'
         go to 800
      endif
c
c===  create print banner
c
cinclude <f77/open.h>
c
      call gamoco( title, 1, LUPRT )
c
c===  read program parameters from command line
c
      call f3gcmdln( LER, ntap, f1, f2, f3, f4, irec1, irec2
     &            , nrpad, itrc1, itrc2, ntpad, ismp1, ismp2, nspad
     &            , idt, dx, dy, mbytes, twod, ifft, verbos, debug
     &            , outfil, datpth, datsfx )
c
c
c=== construct full output file name (packing to left)
c=== make sure a / (backslash) is at the end of the data path
c
      do jc = len(datpth), 1, -1
          if( datpth(jc:jc) .ne. ' ' ) then
             if(datpth(jc:jc) .ne. '/' ) datpth(jc+1:jc+1) = '/'
             go to 1000
          endif
      enddo
1000  continue

      otap = datpth // outfil // datsfx
      ic = 0
      do jc = 1, 128
         if(otap(jc:jc).ne.' ') then
            ic = ic+1
            otap(ic:ic) = otap(jc:jc)
         endif
      enddo
      if( ic.lt.128) then
         do jc = ic+1,128
            otap(jc:jc) = ' '
         enddo
      endif
c
      write( LUPRT,  910 ) ntap, otap
c
      if( ntap .eq. ' ' .or. otap .eq. ' ' ) then
         write( LER, * ) ' ***** ERROR: input and output files must be'
     &                 , ' specified *****'
         ierr = 1
         go to 800
      endif
c
c===  open data files:
c
      call lbopen( luinp, ntap, 'r' )
      call lbopen( luout, otap, 'w+' )
c
c===  read data line header
c
      nbytes = 0
      call rtape( luinp, dheadr, nbytes )
      if( nbytes .eq. 0 ) then
         write( LER,* ) ' ***** ERROR - DATA LINE HEADER READ ERROR ***'
         ierr = 1
         go to 800
      endif
c
c===  get parameters from data line header
c
      call saver( dheadr, 'SmpInt', idt0, LINHED )
      call saver( dheadr, 'NumSmp', nsmp, LINHED )
      call saver( dheadr, 'NumTrc', ntrc, LINHED )
      call saver( dheadr, 'NumRec', nrec, LINHED )
c
c===  set some parameters using these values
c
      if( idt   .lt. 0 ) idt   = idt0
      if( ismp2 .lt. 0 ) ismp2 = nsmp
      if( itrc2 .lt. 0 ) itrc2 = ntrc
      if( irec2 .lt. 0 ) irec2 = nrec
c
c===  some consistency checks
c
      if( ( irec1 .le. 0 ) .or. ( irec1 .gt. irec2 ) ) then
          write( LER, * ) ' ***** ERROR: irec1 must be positive and '
     &        ,'no greater than irec2 *****'
         ierr = 1
      endif
c
      if( irec2 .gt. nrec ) then
          write( LER, * )
     &       ' ***** ERROR: irec2 must be no greater than nrec *****'
          ierr = 1
      endif
c
      if( ( ismp1 .le. 0 ) .or. ( ismp1 .gt. ismp2 ) ) then
         write( LER, * ) ' ***** ERROR: ismp1 must be positive '
     &         ,'and no more than ismp2 *****'
         ierr = 1
      endif
c
      if( ismp2 .gt. nsmp ) then
         write( LER, * )
     &        ' ***** ERROR: ismp2 must be no greater than nsmp *****'
         ierr = 1
      endif
c
c===  set dx and dy from the line header if not set by f3gcmdln
c
      if ( dx .le. 0.0 ) then
         call saver( dheadr, 'Dx1000', idx1000, LINHED )
         dx = float(idx1000)/1000.0
         if ( dx .le. 0.0 ) then
             write( LER, * ) ' ***** ERROR: dx not set in line'
     &           ,' header, must be set from command line *****'
     &           ,' idx1000 = ', idx1000
             ierr = 1
         endif
      endif
c
      if ( dy .le. 0.0 ) then
         call saver( dheadr, 'Dy1000', idy1000, LINHED )
         dy = float(idy1000)/1000.0
         if ( dy .le. 0.0 ) then
             write( LER, * ) ' ***** ERROR: dy not set in line'
     &           ,' header, must be set from command line *****'
     &           ,' idy1000 = ', idy1000
             ierr = 1
         endif
      endif
c
      if( ierr .ne. 0 ) go to 800
c
c===  calculate dt and number of time samples for fft, pad as needed
c
      dt = 0.001 * float( idt )
cdan
      ntout = ismp2 - ismp1 + 1
      nt = nrfft5( ismp2 - ismp1 + 1 + nspad + ntoff )
      mt = nt + 3
c
      if(dt.le.0.0) then
         write( LER, * ) ' ***** ERROR: dt must be positive *****'
         ierr = 1
         go to 800
      endif
c
c===  calculate nx, ny, nkx, nky, dkx, dky, kx0, ky0
c
      nx  = itrc2 - itrc1 + 1
      ny  = irec2 - irec1 + 1
c
c===  check if 2D because only one record to use
c
      if( ny .eq. 1 ) twod = .true.
c
      nkx = ncfft5( nx + ntpad )
      dkx = ( 2.0*PI ) / ( nkx*dx )
      kx0 = ( 1 - nkx ) / 2 * dkx
c
      if( ( nkx .le. 0 ) .or. ( nkx .gt. MAXNK ) ) then
         write( LER, * ) ' ***** ERROR: nkx must be positive '
     &          ,'and no more than MAXNK *****'
         ierr = 1
         go to 800
      endif
c
      if( .not. twod ) then
cdan**** nky = ncfft5( nrec + nrpad )
         nky = ncfft5( ny + nrpad )
         dky = ( 2.0*PI ) / ( nky*dy )
         ky0 = ( 1 - nky ) / 2 * dky
c
         if( ( nky .le. 0 ) .or. ( nky .gt. MAXNK ) ) then
            write( LER, * ) ' ***** ERROR: nky must be positive '
     &             ,'and no more than MAXNK *****'
            ierr = 1
            go to 800
         endif
      endif
c
c===  check and adjust frequency filter parameters
c
      if( f1 .le. 0.0 ) f1 = 0.0
      if( f4 .le. 0.0 ) f4 = float( nt/2 - 1 ) / ( float( nt ) * dt )
      if( f2 .le. 0.0 ) f2 = f1
      if( f3 .le. 0.0 ) f3 = f4
c
c===  allocate space for and compute filter and angular frequency
c
      lfiltr = MAXNW
      lomega = MAXNW
c
      call galloc( pfiltr,  ISZBYT*lfiltr,    ierr, 'ABORT' )
      call galloc( pomega,  ISZBYT*lomega,    ierr, 'ABORT' )
c
      call gtfltr( f1, f2, f3, f4, nt, dt, MAXNW,
     &             nw, iw1, omega, filtr, ierr )

      w0 = omega(1)
      dw = omega(2) - omega(1)
c
      if (ierr .ne. 0) then
         write( LER, * ) ' ***** ERROR in gtfltr, ierr = ',ierr,' *****'
         go to 800
      endif
c
c===  update and output line header
c
      nsmp2 = 2 * nw
      ntrc2 = nkx
      if( .not. twod ) then
         nrec2 = nky
      else
         nrec2 = ny
      endif
c
      call hlhprt( dheadr, nbytes, PPNAME, 5, LUPRT )
c
      call savew( dheadr, 'SmpInt', idt    , LINHED )
      call savew( dheadr, 'NumSmp', nsmp2  , LINHED )
      call savew( dheadr, 'NumTrc', ntrc2  , LINHED )
      call savew( dheadr, 'NumRec', nrec2  , LINHED )
      call wrtape( luout, dheadr, nbytes )
c
      if( twod ) then
         write( LUPRT , 911 )
      else
         write( LUPRT , 912 )
      endif
      write( LUPRT, 913 ) nrec , ntrc , nsmp
      write( LUPRT, 914 ) nrec2, ntrc2, nsmp2
c
c===  output dds data file
c
cdan changed nsmp,ntrc,nrec to ntout,nx,ny
      call f3wdds( outfil, otap, nw, nkx, nky, dw, dkx, dky, w0
     &                     , kx0, ky0, dy, ny, ntout, nx, ny
     &                     , twod, verbos )
c
c===  allocate space for dynamic arrays
c
      lheadr  = ITRWRD * nkx
      ltrace  = ITRWRD + max0( nsmp , nsmp2 )
      lrec    = max0( mt * nkx, nsmp2 * ntrc2 )
c
#ifdef CRAY
      lwork2 = max0( 2*nt*nkx+2*nt, 4*nw*nkx+2*nkx)
#else
      lwork2 = max0( 9*nt/2+18*nkx+41, 6*nkx+18*nw+41, 2*nw*nkx )
#endif
c
      if( twod ) then
         lwork_8 = lwork2
      else
#ifdef CRAY
         lwork_8  = 2*nrec2 + 2*nsmp2*nrec2
     &                    + (ITRWRD+nsmp2)*nrec2*ntrc2
#else
cdan 
         lwork_8  = nrec2*ntrc2
         lwork_8  = (ITRWRD+nsmp2)*lwork_8
     &            +  2*nrec2 + 4*nrec2+9*nsmp2+41 
#endif
c
c======  lwork would hold the entire seismic data set
c
cdan
         lwork_8 = min0( lwork_8, 1048576 * mbytes / ISZBYT )
         lwork = lwork_8 
         write(0,*) 'lwork,lwork_8 = ',lwork,lwork_8
c
         if( lwork .lt. lwork2 ) then
            write( LER, * )' ***** mbytes not big enough for 2D'
     &         , ' transform, use a bigger value ******'
            ierr = 1
            go to 800
         endif
c
      endif
c
      call galloc( pheadr,   ISZBYT * lheadr,   ierr, 'ABORT' )
      call galloc( ptrace,   ISZBYT * ltrace,   ierr, 'ABORT' )
      call galloc( prec,     ISZBYT * lrec,     ierr, 'ABORT' )
      call galloc( pwork,    ISZBYT * lwork,    ierr, 'ABORT' )
c
c===  verbose output
c
      if( verbos ) then
         write( LUPRT , 920 ) irec1, irec2, nrpad, itrc1, itrc2
     &                      , ntpad, ismp1, ismp2, nspad, idt
     &                      , dx, dy, mbytes, twod, ifft
         write( LUPRT , 921 ) f1, f2, f3, f4, nw, iw1, w0, dw
         write( LUPRT , 922 ) nkx, nky, nt, mt, nx
         write( LUPRT , 923 ) lheadr, ltrace, lrec, lwork, lwork2
      endif
c
c***********************************************************************
c
c===  begin the calculation
c
c***********************************************************************
c
cmat  time0 = second()
cdan  skip over the input records       
       if(irec1.gt.1) then
        nbytes = 4*nsmp +2*128
        call skipt(luinp,(irec1-1)*ntrc,nbytes)
       endif
cdan
c
c===  begin loop over records
      do jrec = irec1, irec2
cmat     time2d = second() - time0
	 write( LUPRT, * ) 'PROCESSING RECORD ', jrec
         write(0,*) 'PROCESSING RECORD ', jrec
cmat &         , ' time2d = ',time2d
c
c======  read record data
c======  rec will contain input data in (t,x)
c
         call rdrecn( luinp, LUPRT, jrec, ntrc, itrc1, itrc2
     &             , nsmp, ismp1, ismp2, ntoff, nt, 1, nkx, mt
     &             , trace, headr, rec, ierr )
c
         if( ierr .ne. 0 ) then
            write( LER, * ) ' ***** ERROR reading record ***** '
            go to 800
         endif
c
c======  forward 2d FFT (with transpose)
c======  work will contain the filter input data in (k,w)
c
         call fft2df( mt, nt, nx, nkx, nw, iw1, filtr, rec, work )
c
c======  transpose output record, putting complex elements on w axis
c======  outfile will be (w,k) with w the complex (real/imag) axis
c
         do jk = 1,nkx
         call vmov(work(1+(jk-1)*2), 2*nkx, rec(1+(jk-1)*2*nw), 2, nw )
         call vmov(work(2+(jk-1)*2), 2*nkx, rec(2+(jk-1)*2*nw), 2, nw )
         enddo

c
c======  write record to output file
c
         call wrrec( luout, LUPRT, jrec, nsmp2, 1, ntrc2, nsmp2,
     &               trace, headr, rec, ierr )
         if( ierr .ne. 0 ) then
            write( LER, * ) ' ***** ERROR writing record ',jrec,' *****'
            go to 800
         endif
c
      enddo
c
cmat  time2d = second() - time0
cmat  write( * , * ) ' TOTAL TIME 2D = ',time2d
c
c===  exit if 2D calculation
c
      if( twod ) go to 800
c
c===  pad with extra blank records
c
cdan  if( nrec2 .gt. irec2 ) then
      if( nky .gt. ny ) then
         call vclr( rec, 1, nsmp2*ntrc2 )
cdan     do jrec = irec2+1, nrec2
         do jrec = ny+1, nky
            call wrrec( luout, LUPRT, jrec, nsmp2, 1, ntrc2, nsmp2,
     &               trace, headr, rec, ierr )
            if( ierr .ne. 0 ) then
               write( LER, * ) ' ***** ERROR writing record '
     &                    , jrec,' *****'
               go to 800
            endif
         enddo
      endif
c
cmat  time2d = second() - time0
cmat  write( * , * ) ' TOTAL TIME 2D AFTER RECORD PADDING = ',time2d
c
c===  flush output buffer
c
       call rwd( luout )
c***********************************************************************
c
c===  perform complex fft across slowest dimension (records)
c
      idir = -1
c
cmat  time0 = second()
c
      call cfftmltu( luout, idir, nsmp2, ntrc2, nrec2, lwork, work
     &              , ierr )
c
cmat  time3d = second()-time0
cmat  write( LUPRT, * )' TIME FOR THE THIRD DIM. = ',time3d
cmat  write( LER, * )' TIME FOR THE THIRD DIM. = ',time3d
c
      if( ierr .ne. 0 ) then
        write( LER, * )' ***** ERROR in CFFTMLTU ****** '
        go to 800
      endif
c***********************************************************************
c
      ierr = 0
c
c===  close files, clean-up, & exit
c
800   continue
c
      if( luinp .gt. 1 ) call lbclos (luinp)
      if( luout .gt. 1 ) call lbclos (luout)
c
      if( ierr .le. 0 ) then
         if( LUPRT .gt. 0) write( LUPRT, 9998 )
         call exitfu(0)
      else
         if( LUPRT .gt. 0) write( LUPRT, 9999 ) ierr
         call exitfu(1)
      endif
c***********************************************************************
      end
c***********************************************************************
