c***********************************************************************
c                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c***********************************************************************
c NAME: FXPREP   F(X) PREP FOR FXSMIG            REV 1.0 SEP 95 ********
c***********************************************************************
c
c  HISTORY:
c
c    SEP 95                       REV 1.0         J. Cooperstein, CETech
c
c    OCT 95                       REV 1.1         N. D. Whitmore, Jr.   
c    Changed fxpgeom.f to write the correct no. of traces in the 
c    geometry file header.
c
c    NOV 95                       REV 1.2         M. A. Thornton      
c    Changed fxpgeom to read the newest geometry file format.  A variable
c    "ioversion" is available in each geometry file to determine how it
c    was written.  Also a change in fxpclean to mark the "dead" traces
c    with a 30000 in the StaCor location of the trace header.
c
c***********************************************************************
c***********************************************************************
c
      program fxprep
c
      implicit none
c
      integer   LPPNAM          ! number of characters in program name
      parameter ( LPPNAM = 6 )
c
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c
c variables:
c
      logical   doit            ! logical switch
      integer   dheadr(SZLNHD)  ! length of input file line header (words)
      character gitape*128      ! name of geometry file from PREPMIG
      character gotape*128      ! name of output geometry file for FXSMIG
      integer   idt             ! Sampling interval used
      integer   ierr            ! error flag
      integer   increc          ! increment between input records
      integer   irec1           ! first input record to use
      integer   irec2           ! last  input record to use
      integer   jrec            ! index of current input record
      integer   jrecprev        ! index of last input record
      integer   krec            ! index of current output record
      integer   luinp           ! logical unit of input dataset
      integer   luout           ! logical unit of output dataset
      integer   ndbytes         ! length of input file line header (bytes)
      integer   lbyout          ! length of input file line header (bytes)
      integer   maxsta          ! maximum number of stations
      integer   nrec            ! number of records in input dataset
      integer   nrec2           ! number of records in output dataset
      integer   nsmp            ! number of samples per trace in input
      character ntape*128       ! name of input file
      integer   ntrc            ! number of traces per record in input
      integer   ntrc2           ! number of traces per record in output
      integer   ntrcb2          ! ntrc2(out)/2
      character otape*128       ! name of output file
      character ppname*(LPPNAM) ! program name
      character title*66        ! title for printout banner
      character tunits*5        ! name of time units
      logical   verbos          ! verbose flag
      character version*4       ! version number
c
c dynamically allocated arrays:
c
      real      data(1)         ! input/output array
      real      dataout(1)      ! input/output array for final image
      integer*2 headr(1)        ! trace headers, input record
      integer*2 headrout(1)     ! trace headers, output record
      integer   isleft(1)       ! station number, first live trace
      integer   isright(1)      ! station number, last live trace
      integer   istalv(1)       ! station index (of live stations)
      logical   live(1)         ! station live or dead?
      integer   nlive(1)        ! number of live stations in shot
      integer   soptnm(1)       ! station number of shot
      real      trace(1)        ! scratch space, trace (including header)
c
c variables for length (in words) of dynamically allocated arrays:
c
      integer   ldata, ldataout, lheadr, lheadrout, listalv, llive
     &         , ltrace, ltotal
c
c functions:
c
      integer   argis           ! is argument present function
c
c pointers:
c
      pointer ( pdata,   data   )
      pointer ( pdataout,dataout)
      pointer ( pheadr,  headr  )
      pointer ( pheadrout,  headrout  )
      pointer ( pistalv, istalv )
      pointer ( plive,   live  )
      pointer ( pnlive,  nlive  )
      pointer ( psoptnm, soptnm )
      pointer ( ptrace,  trace  )
      pointer ( pisleft, isleft )
      pointer ( pisright, isright )
c
c data initialization:
c
      data ierr    /  0 /
      data ppname  /'FXPREP'/
      data title   /                                                   '
     &                        F(X) Shot Migration PREP
     &' /
      data verbos  /.false./
      data version /' 1.2'/
c***********************************************************************
c
c format statements:
c
9000  format(/
     &'***************************************************************'/
     & ' FXPREP performs a pre-stack shot migration.'//
     &'      execute fxprep by typing fxprep and the list of program'/
     &'      parameters.  Note that each parameter is proceeded by -a '/
     &'      where "a" is a character(s) corresponding to some '/
     &'      parameter.'/
     &'   Users enter the following parameters, or use default values:'/
     &'***************************************************************'/
     &   ' ', 'FXPREP COMMAND LINE ARGUMENTS:',
     &        /' ', '  -N[ntape]       - INPUT  DATASET NAME          ',
     &              ' default = stdin',
     &        /' ', '  -O[otape]       - OUTPUT DATASET NAME          ',
     &              ' default = stdout',
     &        /' ', '  -GIN[gitape]    - GEOMETRY FILE FROM PREPMIG   ',
     &              ' none',
     &        /' ', '  -GOUT[gotape]   - GEOMETRY FILE FOR FXSMIG     ',
     &              ' ./fxgeo.dat')
9001  format(
     &         ' ', '  -BR[irec1]   - BEGINNING RECORD                ',
     &              ' default = 1',
     &        /' ', '  -ER[irec2]   - ENDING RECORD                   ',
     &              ' default = NumRec from header',
     &        /' ', '  -IR[increc]  - RECORD INCREMENT                ',
     &              ' default = 1',
     &        /' ', '  -MAXSTA[maxsta] - max number of stations       ',
     &              ' default = 10000',
     &        /' ', '  -V              - VERBOSE PRINTOUT' )
9002  format(
     &'***************************************************************'/
     &' usage:  fxprep -N[ntape]  -O[otape] '
     &               ,' -GIN[gitape] -GOUT[gotape]'/
     &'                -BR[irec1] -ER[irec2] -IR[increc]'/
     &'                -MAXSTA[maxsta]     -V'//
     &'***************************************************************')
9100  format( /' INPUT DATASET      = '       / a128 /
     &         ' OUTPUT DATASET     = '       / a128 /
     &         ' GEOMETRY FILE(IN)  = '       / a128 /
     &         ' GEOMETRY FILE(OUT) = '       / a128 )
9110  format( /' ', 'INPUT LINE HEADER PARAMETERS:' )
9120  format( /' ', 'OUTPUT LINE HEADER PARAMETERS:' )
9130  format(
     &         ' ', '   NUMBER OF SEISMIC RECORDS IN THIS JOB =', i5
     &        /' ', '   NUMBER OF TRACES PER SEISMIC RECORD   =', i5
     &        /' ', '   NUMBER OF DATA SAMPLES PER TRACE      =', i5
     &        /' ', '   SAMPLE INTERVAL (',a5,'SECONDS)        =', i5 )
9150  format( /'    ntrc     =' ,i6
     &        /'    ntrc2    =' ,i6
     &        /'    nrec     =', i6
     &        /'    nrec2    =', i6
     &        /'    nsmp     =', i6 )
9200  format( /'    MEMORY ALLOCATION:',
     &        /'    ldata    =', i10,
     &        /'    ldataout =', i10,
     &        /'    lheadr   =', i10,
     &        /'    lheadrout=', i10,
     &        /'    listalv  =', i10,
     &        /'    llive    =', i10,
     &        /'    ltrace   =', i10,
     &        /'    lisleft  =', i10,
     &        /'    liright  =', i10,
     &        /'    lsoptnm  =', i10,
     &        /'    lnlive   =', i10,
     &        /'               ----------',
     &        /'    TOTAL    =', i10 )
9998  format( /' ', '***** NORMAL COMPLETION *****'/ )
9999  format( /' ', '***** ABNORMAL COMPLETION CODE = ', i4, ' *****'/ )
c***********************************************************************
c***********************************************************************
c
c===  check for help
c
      if( argis( '-h' ) .gt. 0 .or. argis( '-?' ) .gt. 0 ) then
         write( LER, 9000 )
         write( LER, 9001 )
         write( LER, 9002 )
         stop
      endif
c
c===  create unique print file name from process id, open print file
c
      call openpr( LUPPRT, LUPRT, ppname, ierr)
      if( ierr .ne. 0 ) then
         write( LER,* ) '***** ERROR: ierr = ',ierr
     &       ,' reported by openpr *****'
         go to 800
      endif
c
c===  create print banner
c
#include <mbsdate.h>
c
      call gamoco( title, 1, LUPRT )
c
c===  read program parameters from command line
c
      call fxpcmdln( LER, verbos, ntape, otape, gitape, gotape
     &                   , increc, irec1, irec2, maxsta )
c
      write( LUPRT,  9100 ) ntape, otape, gitape, gotape
c
c===  open files:     default for pipes: luinp = 0, luout = 1
c
      luinp = 0
      luout = 1
      if( ntape .ne. ' ' ) call lbopen( luinp, ntape, 'r' )
      if( otape .ne. ' ' ) call lbopen( luout, otape, 'w' )
c
c===  read data line header
c
      ndbytes = 0
      call rtape( luinp, dheadr, ndbytes )
      if( ndbytes .eq. 0 ) then
         write( LER,* ) ' ***** ERROR - DATA LINE HEADER READ ERROR ***'
         ierr = 1
         go to 800
      endif
c
c===  get parameters from data line header
c
      call saver( dheadr, 'SmpInt', idt, LINHED )
      call saver( dheadr, 'NumRec', nrec, LINHED )
      call saver( dheadr, 'NumSmp', nsmp, LINHED )
      call saver( dheadr, 'NumTrc', ntrc, LINHED )
c
c***********************************************************************
c
c===  allocate space for dynamic arrays
c
      ldata     = max0( nsmp * ntrc, maxsta )
      lheadr    = ITRWRD * ntrc
      listalv   = ntrc * nrec
      llive     = ntrc * nrec
      ltrace    = ITRWRD + nsmp
c
      call galloc( pisleft,  ISZBYT * nrec,     ierr, 'ABORT' )
      call galloc( pisright, ISZBYT * nrec,     ierr, 'ABORT' )
      call galloc( pdata,    ISZBYT * ldata,    ierr, 'ABORT' )
      call galloc( pheadr,   ISZBYT * lheadr,   ierr, 'ABORT' )
      call galloc( pistalv,  ISZBYT * listalv,  ierr, 'ABORT' )
      call galloc( plive,    ISZBYT * llive,    ierr, 'ABORT' )
      call galloc( pnlive,   ISZBYT * nrec,     ierr, 'ABORT' )
      call galloc( psoptnm,  ISZBYT * nrec,     ierr, 'ABORT' )
      call galloc( ptrace,   ISZBYT * ltrace,   ierr, 'ABORT' )
c
c***********************************************************************
c
c===  read global shot geometry: get:
c                  some global informaton;
c                  shot locations and number of live stations;
c                  depth and integer depth index for stations.
c
      if( irec2 .lt. 0 ) irec2 = nrec
      irec2 = min0( irec2, nrec )
      nrec2 = ( irec2 - irec1 ) / increc + 1
c
      call fxpgeom( LER, LUDISK, LUCARD, gitape, gotape, ntrc, nrec
     &        , maxsta, nlive, live, soptnm, isleft, isright, istalv
     &        , data, irec1, irec2, increc, ntrc2, ierr )
c
c
      if( ierr .ne .0 ) then
         write( LER, * ) ' ***** ERROR reading geometry file ***** '
         go to 800
      endif
c
c===  ntrc2 must be at least ntrc + 1 for shot position and should be
c     odd for use of fxsmig
c
      ntrc2 = max0( ntrc2, ntrc + 1 )
      if( mod( ntrc2 , 2 ) .eq. 0 ) ntrc2 = ntrc2 + 1
c
      ntrcb2 = ntrc2/2
      if( ntrc2 .lt. ntrc + 1 ) then
         ierr =1
         write( LER, * )' ***** ERROR: ntrc2 must be at least ntrc+1, '
     &        ,' ntrc, ntrc2 = ', ntrc, ntrc2
         go to 800
      endif
c
c===  allocate space for dynamic arrays for output data
c
      ldataout  = nsmp * ntrc2
      lheadrout = ITRWRD * ntrc2
      call galloc( pdataout, ISZBYT * ldataout, ierr, 'ABORT' )
      call galloc( pheadrout,ISZBYT * lheadrout,ierr, 'ABORT' )
c
c***********************************************************************
c
c===  begin verbose output
c
      if( verbos ) then
c
         if( idt .le. 32 ) then
            tunits = 'MILLI'
         else
            tunits = 'MICRO'
         endif
c
         write( LUPRT, 9110 )
         write( LUPRT, 9130 ) nrec, ntrc, nsmp, tunits, idt
         write( LUPRT, 9120 )
         write( LUPRT, 9130 ) nrec2, ntrc2, nsmp, tunits, idt
         ltotal = ldata + ldataout + lheadr + lheadrout
     &          + listalv + llive + ltrace + 4*nrec
         write( LUPRT, 9200 ) ldata, ldataout, lheadr, lheadrout
     &                      , listalv, llive, ltrace
     &                      , nrec, nrec, nrec, nrec, ltotal
c
         write( LUPRT, 9150) ntrc, ntrc2, nrec, nrec2, nsmp
      endif
c
c===  end verbose output
c
c***********************************************************************
c
c===  update and output line header
c
      call savew( dheadr, 'NumTrc', ntrc2, LINHED )
      call savew( dheadr, 'NumRec', nrec2, LINHED )
c
      call hlhprt( dheadr, ndbytes, PPNAME, LPPNAM, LUPRT )
c
c     store command line arguments in historical part of line header
c
      call savhlh( dheadr, ndbytes, lbyout )
      call wrtape(  luout, dheadr,  lbyout )
c
c***********************************************************************
c===  BEGIN THE WORK
c***********************************************************************
c
c===  beginning of loop over records
      jrecprev = 0
      krec = 0
      do jrec = irec1, irec2, increc
	 write( LUPRT, * ) 'PROCESSING RECORD ', jrec
c
c======  read input record
c
         if (verbos) write( LUPRT, * ) ' READING RECORD ', jrec
         doit =  jrec .ne. ( jrecprev + 1 )
         if( doit ) call sisseek( luinp, 1 + ( jrec - 1 ) * ntrc )
         call rdrec( luinp, LUPRT, jrec, ntrc, 1, ntrc, nsmp, 1, nsmp
     &             , 0, nsmp, 1, ntrc, nsmp, trace, headr, data, ierr )
         if( ierr .ne. 0 ) then
            write( LER, * ) ' ***** ERROR reading record ***** '
            go to 800
         endif
c
c======  clean up record for output:
c          - insert zero traces for dead stations
c          - fix up trace headers
c
         call fxpclean( LER, jrec, nsmp, ntrc, nrec, ntrc2/2, nlive
     &    , istalv, live, soptnm, headr, data, dataout, headrout, ierr )
         if( ierr. ne. 0 ) go to 800
c
c======  write output record
c
         krec = krec + 1
         if (verbos) write( LUPRT, * ) ' WRITING RECORD ', krec
         call wrrec( luout, LUPRT, krec, nsmp, 1, ntrc2, nsmp,
     &               trace, headrout, dataout, ierr )
         if( ierr .ne. 0 ) then
            write( LER, * ) ' ***** ERROR writing record ***** '
            go to 800
         endif
c
         jrecprev = jrec
      enddo
c===  end of loop over records
c***********************************************************************
      ierr = 0
c
c===  close files, clean-up, & exit
c
800   continue
c
      if( luinp .ge. 0 ) call lbclos (luinp)
      if( luout .ge. 0 ) call lbclos (luout)
c
      if( ierr .eq. 0 ) then
         write( LUPRT, 9998 )
         stop 0
      else
         write( LUPRT, 9999 ) ierr
         stop 1
      endif
c***********************************************************************
      end
c***********************************************************************
