C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C                                                                      *
C  ROUTINE:       RKRAYD                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C       Performs the time domain loop of the Runge-Kutta raytracer.    *
C       Calculates a Runge-Kutta update on the incident angle, THETA,  *
C       and updates the wavefront position, (XRAY,ZRAY), with an       *
C       Euler recursion.                                               *
C                                                                      *
C  ENTRY POINTS:                                                       *
C      RKRAYD  (NRMAX,NZMAX,NRAY,NTSKIP,JT,DXOVER,DZOVER,VDSX,VDSZ,    *
C               VDT,CSTAB,THETA,WCOS,WSIN,WORK,XRAY,ZRAY)              *
C      XRKRAYD  (NRMAX,NZMAX,NRAY,NTSKIP,JT,DXOVER,DZOVER,VDSX,VDSZ,   *
C                VDT,CSTAB,THETA,WCOS,WSIN,WORK,XRAY,ZRAY)             *
C      CRKRAYD  (NRMAX,NZMAX,NRAY,NTSKIP,JT,DXOVER,DZOVER,VDSX,VDSZ,   *
C                VDT,CSTAB,THETA,WCOS,WSIN,WORK,XRAY,ZRAY)             *
C  ARGUMENTS:                                                          *
C      NRMAX   INTEGER  ??IOU*      -                                  *
C      NZMAX   INTEGER  ??IOU*      -                                  *
C      NRAY    INTEGER  ??IOU*      -                                  *
C      NTSKIP  INTEGER  ??IOU*      -                                  *
C      JT      INTEGER  ??IOU*      -                                  *
C      DXOVER  REAL     ??IOU*      -                                  *
C      DZOVER  REAL     ??IOU*      -                                  *
C      VDSX    REAL     ??IOU*  (*) -                                  *
C      VDSZ    REAL     ??IOU*  (*) -                                  *
C      VDT     REAL     ??IOU*  (*) -                                  *
C      CSTAB   REAL     ??IOU*  (*) -                                  *
C      THETA   REAL     ??IOU*  (*) -                                  *
C      WCOS    REAL     ??IOU*  (*) -                                  *
C      WSIN    REAL     ??IOU*  (*) -                                  *
C      WORK    REAL     ??IOU*  (*) -                                  *
C      XRAY    REAL     ??IOU*  (*) -                                  *
C      ZRAY    REAL     ??IOU*  (*) -                                  *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   Dan Whitmore                       ORIGIN DATE: 92/07/22  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 92/07/22  *
C       FORTRAN CFT77                                                  *
C                                                                      *
C  HISTORY:                                                            *
C       Original        May 90                  R.D. Coleman, QTC      *
C       Rev 2.0         Feb 92                  R.D. Coleman, CETech   *
C               Portable FORTRAN version - added additional entry point*
C                                                                      *
C  CALLING FORMAT:                                                     *
C       CALL  RKRAYD (NRMAX, NZMAX, NRAY, NTSKIP, JT,                  *
C      &              DXOVER, DZOVER, VDSX, VDSZ, VDT, CSTAB,          *
C      &              THETA, WCOS, WSIN, WORK, XRAY, ZRAY)             *
C       CALL CRKRAYD (NRMAX, NZMAX, NRAY, NTSKIP, JT,                  *
C      &              DXOVER, DZOVER, VDSX, VDSZ, VDT, CSTAB,          *
C      &              THETA, WCOS, WSIN, WORK, XRAY, ZRAY)             *
C       CALL XRKRAYD (NRMAX, NZMAX, NRAY, NTSKIP, JT,                  *
C      &              DXOVER, DZOVER, VDSX, VDSZ, VDT, CSTAB,          *
C      &              THETA, WCOS, WSIN, WORK, XRAY, ZRAY)             *
C                                                                      *
C  PARAMETERS:                                                         *
C       NRMAX   Integer input scalar                                   *
C               Maximum number of rays and leading dimension of        *
C               matrices XRAY and ZRAY.                                *
C                                                                      *
C       NZMAX   Integer input scalar                                   *
C               Maximum number of Z's and leading dimension of         *
C               matrices VDSX, VDSZ, and VDT.                          *
C                                                                      *
C       NRAY    Integer input scalar                                   *
C               Number of rays                                         *
C                                                                      *
C       NTSKIP  Integer input scalar                                   *
C               Increment of time index (column index of XRAY and ZRAY)*
C                                                                      *
C       JT      Integer input scalar                                   *
C               Time index (column index of XRAY and ZRAY).            *
C                                                                      *
C       DXOVER  Real input scalar                                      *
C               Reciprocal of delta x.                                 *
C                                                                      *
C       DZOVER  Real input scalar                                      *
C               Reciprocal of delta z.                                 *
C                                                                      *
C       VDSX    Real input matrix of dimension NZMAX by NXMAX          *
C               VDSX = v * v * partial(d(1/v)/dx) * dt,                *
C               where v = interval velocity model                      *
C                                                                      *
C       VDSZ    Real input matrix of dimension NZMAX by NXMAX          *
C               VDSZ = v * v * partial(d(1/v)/dz) * dt,                *
C               where v = interval velocity model                      *
C                                                                      *
C       VDT     Real input matrix of dimension NZMAX by NXMAX          *
C               VDT = v * dt, where v = interval velocity model        *
C                                                                      *
C       CSTAB   Real input vector of implied length                    *
C               Packed cos/sin table as build by subroutine GCFTLP.    *
C                                                                      *
C       THETA   Real input/output vector of length NRAY                *
C               Incident angle on input.  Updated incident angle on    *
C               output.                                                *
C                                                                      *
C       WCOS    Real input/output vector of length NRAY                *
C               WCOS = cos( THETA ) on both input and output.          *
C                                                                      *
C       WSIN    Real input/output vector of length NRAY                *
C               WSIN = sin( THETA ) on both input and output.          *
C                                                                      *
C       WORK    Real output vector of length NRAY                      *
C               Gathered elements of VDT.                              *
C                                                                      *
C       XRAY    Real input/output matrix of dimension NRMAX by NTMAX   *
C               Wavefront x position at each dt.  Column JT is updated *
C               using column JT-NTSKIP.                                *
C                                                                      *
C       ZRAY    Real input/output matrix of dimension NRMAX by NTMAX   *
C               Wavefront z position at each dt.  Column JT is updated *
C               using column JT-NTSKIP.                                *
C                                                                      *
C  DESCRIPTION:                                                        *
C                                                                      *
C  SUBPROGRAMS CALLED:                                                 *
C       FORTRAN INTRINSICS: IFIX                                       *
C                                                                      *
C  ERROR CONDITIONS:                                                   *
C       None                                                           *
C                                                                      *
C--------------------------------------------------------------------- *
C                                                                      *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      AND     REAL -                                                  *
C      SHIFTL  REAL -                                                  *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      IFIX    INTEGER -                                               *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C*****************  ITEMS DELETED FROM DOCPACK  ************************
C  =============================== DATE: 92/07/22 ==================   *
C NAME: RKRAYD    RUNGE-KUTTA RAYTRACING               REL 2.0  FEB 92 *
C********************  END OF DELETED ITEMS  ***************************
C***********************************************************************
      subroutine  rkrayd (nrmax, nzmax, nray, ntskip, jt,
     &                   dxover, dzover, vdsx, vdsz, vdt, cstab,
     &                   theta, wcos, wsin, work, xray, zray)
      entry      xrkrayd (nrmax, nzmax, nray, ntskip, jt,
     &                   dxover, dzover, vdsx, vdsz, vdt, cstab,
     &                   theta, wcos, wsin, work, xray, zray)
      entry      crkrayd (nrmax, nzmax, nray, ntskip, jt,
     &                   dxover, dzover, vdsx, vdsz, vdt, cstab,
     &                   theta, wcos, wsin, work, xray, zray)
c
      integer nrmax, nzmax, nray, ntskip, jt
      real    dxover, dzover, vdsx(*), vdsz(*), vdt(*),
     &        xray(*), zray(*), theta(*),
     &        wcos(*), wsin(*), work(*)
#ifdef CRAY
      integer mask
      real    cstab(*), tabi
c
      parameter (mask = x'FFFFFFFF00000000')
#else
      real*8  cstab(*), tabi
      real    tabi2(2)
      equivalence (tabi, tabi2)
#endif
c
c-----------------------------------------------------------------------
c
      scale  = cstab(2)
      offset = cstab(3)
c
      j1 = (jt - 1 ) * nrmax
      j2 = j1 - ntskip * nrmax
      do 200 j = 1, nray
         j1 = j1 + 1
         j2 = j2 + 1
c
c        find temp node indices
c
         jx1 = ifix( xray(j2)*dxover + 0.5 )
         jz1 = ifix( zray(j2)*dzover + 1.5 )
         jv1 = jx1 * nzmax + jz1
c
c        compute thetmp
c
         thetmp  = theta(j) + wcos(j)*vdsx(jv1) - wsin(j)*vdsz(jv1)
         itab    = ifix( scale * thetmp + offset )
         tabi    = cstab(itab)
#ifdef CRAY
         tcos    = and( tabi, mask )
         tsin    = shiftl( tabi, 32 )
#else
         tcos    = tabi2(1)
         tsin    = tabi2(2)
#endif

c
c        find temp node indices
c
         jx2 = ifix( (xray(j2) + tsin*vdt(jv1) )* dxover + 0.5 )
         jz2 = ifix( (zray(j2) + tcos*vdt(jv1) )* dzover + 1.5 )
         jv2 = jx2 * nzmax + jz2
c
c        compute Runge-Kutta update of theta
c
         theta(j) = 0.5 * ( theta(j) + thetmp
     &                        + tcos*vdsx(jv2) - tsin*vdsz(jv2) )
         itab     = ifix( scale * theta(j) + offset )
         tabi     = cstab(itab)
#ifdef CRAY
         wcos(j)  = and( tabi, mask )
         wsin(j)  = shiftl( tabi, 32 )
#else
         wcos(j)  = tabi2(1)
         wsin(j)  = tabi2(2)
#endif
c
c        update wavefront 
c
C
         vdt_avg  = 2.*vdt(jv1)*vdt(jv2)/(vdt(jv1)+vdt(jv2)+.000001)  
         xray(j1) = xray(j2) + wsin(j) * vdt_avg
         zray(j1) = zray(j2) + wcos(j) * vdt_avg
         work(j)  = vdt_avg

  200 continue
c
      return
      end
