c***********************************************************************
c     
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c***********************************************************************
c     NAME: MKUSP
c     MAKE USP DATA FILE 
c     REV 3.0  MAY 97 
***********************************************************************
      
c     HISTORY:
      
c     APR 95          REV 1.0         J. Cooperstein, CETech Inc.
      
c     SEP 96          REV 2.0         J. Cooperstein, Axian Inc.
c     --------------- Modified to work with pipes
      
c     MAY 97          REV 3.0         J. Cooperstein, Axian Inc.
c     --------------- Change output
c     --------------- general improvements                 

c     FEB 98          REV 4.0         J. Cooperstein, Axian Inc.
c     --------------- improved error handling
      
c***********************************************************************

c     mkusp can create a usp file of several different types:

c     -COS (cosine signal)
c     -INT (integer pattern)
c     -SIG (shifted signal generator)
c     -RAN (random numbers)

c     usage: mkusp -O[otap] -nrec[nrec] -ntrc[ntrc] -nsmp[nsmp] -V
c     < -COS -t[time] > |
c     < -INT -t[time] -off[off] -rad[rad] > |
c     < -SIG -F[ffile] -nsi[nsi] -nxshft[nxshft] -nyshft[nyshft] -C > |
c     < -RAN -t[time] -lbnd[lbnd] -ubnd[ubnd] -seed[iseed] >

c     If -COS is specified, mkusp creates a usp file specified by:

c     val(it,ix) = cos( w*t )

c     where

c     w = 2*pi*ix,  and t = dt * (it-itbias),

c     and itbias is the shift from record to record

c     If -INT is specified, mkusp creates a usp file specified by:

c     val(it,ix,iy) = off + mod(it,rad) + rad * mod(ix,rad)
c     + rad**2 * mod(iy,rad)

c     If -SIG is specified, mkusp creates a usp file specified by:
c     
c     val(it,ix,iy) = signal( index ) 

c     where

c     index = mod( (ix-1)*nxshft +(iy-1)*nyshft + (it-1) , nsmp)

c     and the signal amplitudes are generated from the frequency
c     file specified by ffile.  This file must contain:

c     nfreq
c     frequency   amplitude  phase  ( nfreq entries )

c     If the option "-C" is specified, the output file will
c     be complex with imaginary components zeroed.

c     If -RAN is specified, mkusp creates a usp file in which
c     the samples are filled with random numbers uniformly dis-
c     tributed from parameters lbnd to ubnd, seeded with iseed.

c***********************************************************************
c***********************************************************************
      program mkusp
c***********************************************************************

      implicit none
      
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
      
c     parameters:
      
      integer   IFORM           !
      integer    MAXFREQ
      parameter ( IFORM = 3, MAXFREQ = 100 )
      
c     variables:
      
      character ffile*128       ! frequency file name
      character otap*128        ! name of output dataset
      character ppname*5        ! program name
      character title*66        ! title for printout banner
      character version*4       ! version number

      logical   verbos          ! verbose output flag

      integer   icase           ! case index
      integer   icmplx          ! 1 if real, 2 if complex samples
      integer   ier             ! error flag
      integer   ierr            ! error flag
      integer   iseed           ! random number generator seed
      integer   jf              ! frequency index
      integer   jrec            ! record index of current record
      integer   luout           ! logical unit of output dataset
      integer   nfreq           ! number of frequencies
      integer   nrec            ! number of records in output dataset
      integer   nsi             ! sample interval (msecs or microsecs)
      integer   nsmp            ! number of samples per trace
      integer   ntrc            ! number of traces per record
      integer   nxshft          ! sample shift in x direction
      integer   nyshft          ! sample shift in y direction
      integer   off             ! offset for icase=2
      integer   rad             ! radix for icase=2

      real      dt              ! delta t
      real      lbnd            ! lower bound for random samples
      real      time            ! time per trace (sec)
      real      ubnd            ! upper bound for random samples
      
c     dynamically allocated arrays:
      
      real      amp(1)          ! input amplitude for icase=3
      real      freq(1)         ! input frequency for icase=3
      real      phase(1)        ! input phase for icase=3
      real      rec(1)          ! scratch space for output record
      real      sig(1)          ! generated signal
      real      trace(1)        ! scratch space, trace (including header)

c     variables for length (in words) of dynamically allocated arrays:
      integer   lrec, ltrace, lsig

c     functions:

      integer   argis
      logical errchk            ! check if error, print warning message

c     pointers:

      pointer   ( pamp  , amp )
      pointer   ( pfreq , freq )
      pointer   ( pphase, phase )
      pointer   ( prec  , rec )
      pointer   ( psig  , sig )
      pointer   ( ptrace, trace )

c     data initialization:

      data ier, ierr   / 0, 0 /
      data luout  / 1 /
      data ppname / 'MKUSP' /
      data title   /'                      MKUSP -- MAKE USP DATA FILE'/
      data verbos  /.false./
      data version /'4.0 ' /

c***********************************************************************
 922  format( /' ', 'Output file name = ', A100 )
 923  format( /' ', 'Line header values after default check'/
     &     ' ', '   Number of records        =', i5/
     &     ' ', '   Number of traces/record  =', i5/
     &     ' ', '   Number of samples/trace  =', i5/
     &     ' ', '   Sample interval          =', i5/
     &     ' ', '   Data format              =', i5/)
 924  format(/' Dynamically allocated arrays: '/
     &     '    ltrace   =', i10/
     &     '    lrec     =', i10)
 925  format( '    lsig     =', i10)
 9010 format(/ '       time = ', f16.6/
     &     '       dt   = ', f16.10)
 9020 format(/ '       time = ', f16.6/
     &     '       dt   = ', f16.10/
     &     '       off  = ', i10/
     &     '       rad  = ', i10)
 9030 format( /' ', 'Frequency file name = ', (a)/
     &     ' ', 'Other parameters:'/
     &     ' ', '   X sample shift           =', i5/ 
     &     ' ', '   Y sample shift           =', i5/ 
     &     ' ', '   Number of frequencies    =', i5)
 9031 format( /' ', '   i   freq(i)    amp(i)  phase(i)' )
 9032 format(  ' ', i4, 3f10.2 )
 9040 format(/ '       time = ', f16.6/
     &     ' ', '   Random sample lower bound =', f11.5/
     7     ' ', '   Random sample upper bound =', f11.5/
     8     ' ', '   Random number seed        =', i11/)
 9998 format( /' ', '***** NORMAL COMPLETION *****'/ )
 9999 format( /' ', '***** ABNORMAL COMPLETION CODE = ', i4, ' *****'/ )
c***********************************************************************

c     check for help (quits if help desired)

      if( argis( '-h' ) .gt. 0 .or. argis( '-?' ) .gt. 0 ) call help

c     open printout file

      call openpr( LUPPRT, LUPRT, ppname, ierr)

      if( errchk( ierr.ne.0, 'failure in OPENPR', ier ) ) go to 800

c     create print banner

c     #include <mbsdate.h> ! screws up pipes!

      call gamoco( title, 1, LUPRT )

c     read program parameters from command line

      call mkgcmdln( otap, nrec, ntrc, nsmp, time, off, rad
     &     , icmplx, ffile, nsi, nxshft, nyshft, lbnd, ubnd, iseed
     &     , verbos, icase )

c     compute some parameters

      if ( icase .eq. 1 .or. icase. eq. 2  .or. icase .eq. 4 ) then

         dt    = time / float( nsmp )

         if( dt .lt. 0.032 ) then
            nsi = nint( dt * 1.0e6 )
         else
            nsi = nint( dt * 1.0e3 )
         endif

      elseif( icase .eq. 3 ) then

         dt    = 0.001 * float( nsi )

c     read frequency file, allocate space for freq. parameters

         open( unit=LUCARD, file=ffile, status='old' )

         read( LUCARD, * ) nfreq
         if( errchk( nfreq.le.0, 'nfreq<=0', ier ) ) go to 800

         if( errchk( nfreq.gt.MAXFREQ, 'nfreq>MAXFREQ', ier )) go to 800

         call galloc( pfreq,  nfreq*ISZBYT, ierr, 'ABORT' )
         call galloc( pamp,   nfreq*ISZBYT, ierr, 'ABORT' )
         call galloc( pphase, nfreq*ISZBYT, ierr, 'ABORT' )

         do jf = 1, nfreq
            read ( LUCARD, *, end=790 ) freq(jf), amp(jf), phase(jf)
            if( freq(jf) .eq. 0.0 ) phase(jf) = 0.0
            if( errchk(freq(jf).lt.0.,'freq must be>=0',ier)) go to 800
            if( errchk(amp(jf).le.0.,'amp must be>=0',ier)) go to 800
         enddo
      endif

c     open output file (if not pipe)

      if( otap .ne. ' ') call lbopen( luout, otap, 'w' )

c     write line header

      call mkwrlhdr( luout, ppname, nrec, ntrc, icmplx*nsmp, nsi
     &     , IFORM )

c     allocate space for dynamic arrays

      ltrace = ITRWRD + nsmp
      lrec = nsmp * ntrc * icmplx

      call galloc( ptrace, ltrace*ISZBYT, ierr, 'ABORT' )
      call galloc( prec,   lrec*ISZBYT,   ierr, 'ABORT' )

      if ( icase .eq. 3 ) then

c     generate base signal

         lsig = ( nsmp + 1 )
         call galloc( psig, lsig*ISZBYT, ierr, 'ABORT' )
         call siggen( nfreq, nsmp, dt, amp, freq, phase, sig )
         sig(nsmp+1) = sig(1)

      endif

c     verbose output

      if( verbos ) then
         write( LUPRT, 922 ) otap
         write( LUPRT, 923 ) nrec, ntrc, nsmp, nsi, IFORM
         write( LUPRT, '(''     icase = '',i5)') icase
         if ( icase .eq. 1 ) then
            write( LUPRT, 9010 ) time, dt
         elseif ( icase .eq. 2 ) then
            write( LUPRT, 9020 ) time, dt, off, rad
         elseif ( icase .eq. 3 ) then
            write( LUPRT, 9030 ) ffile, nxshft, nyshft, nfreq
            write( LUPRT, 9031 )
            write( LUPRT, 9032 ) ( jf, freq(jf), amp(jf), phase(jf)
     &           , jf = 1, nfreq )
         elseif ( icase .eq. 4 ) then
            write( LUPRT, 9040 ) time, lbnd, ubnd, iseed
         endif
         write( LUPRT, 924 ) ltrace, lrec
         if( icase .eq. 3 ) write( LUPRT, 925 ) lsig
      endif

c***********************************************************************

c     loop over records

      do jrec = 1,nrec
         write(0,*) ' CREATING RECORD = ', jrec
         write( LUPRT, * ) ' CREATING RECORD = ', jrec

c     make usp output record

         call vclr( rec, 1, lrec )

         if( icase .eq. 1 ) then
            call mkrec1( jrec, nrec, ntrc, nsmp, dt, rec )

         elseif( icase .eq. 2 ) then
            call mkrec2( jrec, ntrc, nsmp, off, rad, rec )

         elseif( icase .eq. 3 ) then
            call mkrec3( jrec, ntrc, nsmp, icmplx, nxshft, nyshft
     &           , sig, rec )

         elseif( icase .eq. 4 ) then
            call mkrec4( iseed, lbnd, ubnd, ntrc, nsmp, rec )

         endif

c     write output record

         call wrrecnh( luout, LER, jrec, icmplx*nsmp, 1, ntrc
     &        , icmplx*nsmp, trace, rec, ierr )
         if( errchk(ierr.ne.0,'writing record',ier)) go to 800
      enddo

      go to 800

c***********************************************************************
 790  continue
      write(0,*) '***** ERROR: reached end of frequency file ****'
      ierr = 0

c     close files, clean-up, & exit

 800  continue

      call lbclos(luout)
c     
      if( ierr .lt. 0 .or. ier .lt. 0 ) then
         if( LUPRT .gt. 0) write( LUPRT, 9999 ) ier
         if( LUPRT .gt. 0) write( LUPRT, 9999 ) ierr
      else
         if( LUPRT .gt. 0) write( LUPRT, 9998 )
      endif
c***********************************************************************
      stop
      end
c***********************************************************************
c***********************************************************************
      logical function errchk( condition, string, ierr )
c***********************************************************************
      logical condition
      character *(*) string
      integer ierr
c***********************************************************************
      errchk = condition
      ierr = 0
      if( condition ) then
         write(0,'(''***** MKUSP FATAL ERROR: '',a,'' ******'')')string
         ierr = -1 
      endif
c***********************************************************************
      return
      end
c***********************************************************************

