/*********************************************************************
**
** Program name: nlx2lx - resample nonlinear x data into linear x data
**
** Language:     C
**
** Author:       David Nelson, Amoco Production Research
**
** Date written: Jan 14, 1993
**
**
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
**
** Abstract:
**
** Modification history: Jan 14, 1993 - Initial release - V: 1.0
**
************************************************************************/

#include "nlx2lx.h"
 
int dbx;
 
main(argc,argv)
unsigned int argc;
char *argv[];
{
  int i,error,irec=0,tsamps,one=1,lc,verbose,help;
  int nl_nrec,nlmap_nrec;
  int nl_ntr,nlmap_ntr,l_ntr;
  int nl_nsamp,nlmap_nsamp;
  int nl_lhlen,nlmap_lhlen;
  int nl_luin,nlmap_luin,l_luout;

  float xmin,xmax,dx,fill;
  float nl_lhead[6000],nlmap_lhead[6000];
  float *nl_data,*nlmap_data,*l_data;
  float *nl_thead,*nlmap_thead,*l_thead;
  float *warptransform;

  char nl_filename[256],nlmap_filename[256],l_filename[256];

  /* handle the command line args */
  if (help=is_arg("-h",&argc,argv)) prhelp();
  strcpy(nl_filename,get_arg("-N","",&argc,argv));
  strcpy(nlmap_filename,get_arg("-NLMAP","",&argc,argv));
  strcpy(l_filename,get_arg("-O","",&argc,argv));
  sscanf(get_arg("-xmin","-1.0",&argc,argv),"%f",&xmin);
  sscanf(get_arg("-xmax","-1.0",&argc,argv),"%f",&xmax);
  sscanf(get_arg("-dx","-1.0",&argc,argv),"%f",&dx);
  sscanf(get_arg("-fill","1.0",&argc,argv),"%f",&fill);
  verbose=is_arg("-V",&argc,argv);

  if(!strlen(nl_filename)) {
    help=1;
    fprintf(stderr,"input file -N must be specified\n");
  }
  if(!strlen(nlmap_filename)) {
    help=1;
    fprintf(stderr,"mapping file -NLMAP must be specified\n");
  }
  if(!strlen(l_filename)) {
    help=1;
    fprintf(stderr,"output file -O must be specified\n");
  }

  /* user needs help */
  if (help) prhelp();

  /* open the data in non-linear x */
  if (error=USPOpenAndReadLineHeader(nl_filename,&nl_luin,
                                     nl_lhead,&nl_lhlen)) exit(-1);

  /* open the non-linear x mapping function */
  if (error=USPOpenAndReadLineHeader(nlmap_filename,&nlmap_luin,
                                     nlmap_lhead,&nlmap_lhlen)) exit(-1);

  /* fetch crucial line header stuff */
  C_SAVER(nl_lhead,"NumRec",&nl_nrec, LINHED);
  C_SAVER(nl_lhead,"NumTrc",&nl_ntr,  LINHED);
  C_SAVER(nl_lhead,"NumSmp",&nl_nsamp,LINHED);

  C_SAVER(nlmap_lhead,"NumRec",&nlmap_nrec, LINHED);
  C_SAVER(nlmap_lhead,"NumTrc",&nlmap_ntr,  LINHED);
  C_SAVER(nlmap_lhead,"NumSmp",&nlmap_nsamp,LINHED);

  /* the non-linear x data and non-linear x  mapping function must conform */
  if (nl_ntr != nlmap_ntr || nl_nsamp != nlmap_nsamp) {
    fprintf(stderr,"Number of traces/record and number of samples in\n%s\n does not match number of traces/record and number of samples in\n%s\n",
      nl_filename,nlmap_filename);
    exit(-1);
  }

  /* allocate input buffers */
  nl_data=(float *)malloc(nl_ntr*nl_nsamp*sizeof(float));
  nl_thead=(float *)malloc(nl_ntr*ITRWRD*sizeof(float));
  nlmap_data=(float *)malloc(nl_ntr*nl_nsamp*sizeof(float));
  nlmap_thead=(float *)malloc(nl_ntr*ITRWRD*sizeof(float));
  if (nl_data==NULL || nl_thead==NULL ||
      nlmap_data==NULL || nlmap_thead==NULL) {
    fprintf(stderr,"Not enough memory - Adios Amigo\n");
    exit(-1);
  }

  /* read the non-linear x mapping function */
  if (error=USPReadDataByRecord(&nlmap_luin,nlmap_thead,nlmap_data,
                               &nl_ntr,&nl_nsamp)) exit(-1);

  /* if xmin and and xmax were not specified by user, extract them */
  tsamps=nl_ntr*nl_nsamp;
  if (xmin==-1.0)
    MINV(nlmap_data,&one,&xmin,&lc,&tsamps);
  if (xmax==-1.0)
    MAXV(nlmap_data,&one,&xmax,&lc,&tsamps);

  /* dx determines number of output traces if specified, otherwise
     output traces is set equal to input traces */
  if (dx==-1.0)
    l_ntr=nl_ntr;
  else
    l_ntr=(int)((xmax-xmin)/dx)+1;

  /* allocate output buffers */
  l_data=(float *)malloc(l_ntr*nl_nsamp*sizeof(float));
  l_thead=(float *)malloc(l_ntr*ITRWRD*sizeof(float));
  warptransform=(float *)malloc(l_ntr*nl_nsamp*sizeof(float));
  if (l_data==NULL || l_thead==NULL || warptransform==NULL) {
    fprintf(stderr,"Not enough memory - Adios Amigo\n");
    exit(-1);
  }

  /* create the non-linear x to linear x transform lookup table */
  ComputeWarpTransform(nlmap_data,warptransform,
                       &xmin,&xmax,&nl_nsamp,&nl_ntr,&l_ntr);

  /* open the output data in linear x */
  C_SAVEW(nl_lhead,"NumTrc",l_ntr,LINHED);
  if (error=USPOpenAndWriteLineHeader(l_filename,&l_luout,
                                      nl_lhead,&nl_lhlen)) exit(-1);

  /* read nonlinear data, transform, write linear data */
  for (i=0; i<nl_nrec; i++) {
    if (error=USPReadDataByRecord(&nl_luin,nl_thead,nl_data,
                                  &nl_ntr,&nl_nsamp)) exit(-1);
    ApplyWarpTransform(nl_data,l_data,warptransform,&fill,&nl_nsamp,&l_ntr);
    if (error=USPWriteDataByRecord(&l_luout,nl_thead,l_data,&irec,
                                   &l_ntr,&nl_nsamp)) exit(-1);
  }

  /* close datasets */
  lbclos(nl_luin);
  lbclos(nlmap_luin);
  lbclos(l_luout);

  exit(0);
}
