C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
c Read a pressure gradient field, convert to pressure, filter pressures
c so that no pressure regressions exist in the dataset. For intervals found
c to contain pressure regressions use the fluid pressure gradient supplied
c on the command line to calculate a new pore pressure gradient.
c Output the filtered pressure gradient field in pounds/gallon (drilling mudweight)
c ppfilt.F
c
c Martin Albertin                            September 18. 1995
c
c program framework by:
c Mary Ann Thornton           v: 1.0         September 29, 1994
c************************************************************************
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
      parameter   (convm = 3.280833, ppgal = 0.05195)
      integer     lhed (SZLNHD)
      integer * 2 itr  (SZLNHD)
      real        head (SZLNHD)
      real        ppg  (SZLNHD)
      real        nhg, dz

      integer     nsamp, ntrc, nrec, iform, obytes
      integer     luin, luout, lbytes, nbytes, lbyout

      character   ntap  * 100, otap * 100, name*5, version*4
      logical     hlp, query, met
      integer     argis
      equivalence (itr( 1), lhed (1), head(1))
      data lbytes/ 0 /, nbytes/ 0 /, name/'PPFILT'/, version/' 1.0'/ 
 
c-----
c     read program parameters from command line card image file
      query = ( argis ( '-?' ) .gt. 0 )
      hlp = ( argis ( '-h' ) .gt. 0 )
      if ( query .or. hlp )then
            call help()
            stop
      endif

c-----
c     open printout
#include <f77/mbsopen.h>

c-----
c     read command line arguments
      call gcmdln(ntap,otap,nhg,dz,met)

c     print* , 'ntap= ',ntap,' otap= ',otap,' nhg= ',nhg,' dz= ',dz

c     set depth conversion
      print* , ' dz= ',dz
      if ( met )then
            dz=dz*convm
      endif
      print* , ' dz= ',dz

c     open input and output files
      call getln(luin , ntap ,'r', 0)
      call getln(luout, otap ,'w', 1)

c     read line header from pressure gradient dataset
      call rtape(luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'ppfilt: no line header read from unit ',luin
         write(LOT,*)' ( pressure gradient dataset )'
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         stop
      endif

c  save values from line header
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)

c     print line header into printer listing
      call hlhprt(itr, lbytes, name, 5, LERR)

c     determine obytes as the number of bytes in output trace
      obytes = SZTRHD + nsamp * SZSMPD

c     save command line arguments in the historical part of line header
      call savhlh(itr,lbytes,lbyout)

c     write the output line header
      call wrtape(luout, itr, lbyout )
 
c--------------------------------------------------
c     BEGIN PROCESSING
c--------------------------------------------------

c-----LOOP ON TRACES
      do 1001  mm = 1, nrec
      do 1003  nn = 1, ntrc
c        read the ppg traces into itr then move to ppg
         nbytes = 0
         call rtape(luin, itr, nbytes)
         if(nbytes .eq. 0) then
            write(LERR,*)'End of file on pressure gradient input:',luin
            write(LERR,*)'  trace= ',nn
            go to 999
         endif
         call vmov(lhed(ITHWP1),1,ppg,1,nsamp)

c--------LOOP ON SAMPLES
c-       compute pressure for each sample (convert from pounds/gallon)
         do 1002 kk = 1,nsamp-1
            z1=(kk-1)*dz
            z2=kk*dz
            pp1=ppg(kk)*ppgal*z1
            pp2=ppg(kk+1)*ppgal*z2

c           pressure must increase with depth, at a rate faster than fluid pressure gradient
            minpp2=pp1+(nhg*dz)
            if (pp2 .lt. minpp2) then
               pp2=minpp2
               ppg(kk+1)=pp2/(ppgal*z2)
            endif
 1002    continue
c----       
         call vmov (ppg, 1, lhed(ITHWP1), 1, nsamp)
         call wrtape (luout, itr, obytes)
 1003 continue
 1001 continue

c--------------------------------------------------
c--------------------------------------------------
c-----
c     END PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------
 
  999 continue
 
c-----
c     close data files
      call lbclos(luin)
      call lbclos(luout)
c-----
      write(LERR,*)'end of ppetn, processed ',nrec,' record(s)',
     &               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
c...5....0....5....0....5....0....5....0....5....0....5....0....5....0..
      write(LER,*)'                                                    '
      write(LER,*)'PPFILT converts pressure gradient (PPG) to pressure '
      write(LER,*)'(PSI), applys a hydrostatic filter, and converts the'
      write(LER,*)'results back to pressure gradient (PPG).The filter  '
      write(LER,*)'rule is: pressure must increase with depth at a rate'
      write(LER,*)'greater than or equal to the fluid gradient given on'
      write(LER,*)'the command line.                                   '
      write(LER,*)'                                                    '
      write(LER,*)'Parameters [default values] :                       '
      write(LER,*)' -N[]        req   input pressure gradient dataset  '
      write(LER,*)' -O[]        req   output filtered gradient data    '
      write(LER,*)' -nhg[0.465] req   fluid gradient                   '
      write(LER,*)' -met        opt   metric depth units               '
      write(LER,*)' -dz[]       req   depth sample interval            '
      write(LER,*)'                                                    '
      write(LER,*)'Usage:                                              '
      write(LER,*)'ppfilt -N[] -O[] -nhg[] -dz[] [-met]                '
      write(LER,*)'                                                    '
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,nhg,dz,met)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name pressure gradient
c     otap  - C*100    output file name
c     nhg   - real     fluid pressure gradient for minimum pressure calculation
c                      default gradient is for USA Gulf Coast area .465
c     dz    - real     depth sample interval
c     met   - L        flag indicating metric units
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      logical     met
      integer     argis
      real        nhg,dz
 
c-------
c     last 2 arguments are values used when:
c     (1) if ONLY the key is present (no value attached to it)
c     (2) if NO key & no value are present
c-------
            call argstr('-N', ntap , ' ', ' ')
            call argstr('-O', otap , ' ', ' ')
            call argr4 ('-nhg', nhg ,.465 ,.465)
            call argr4 ('-dz', dz , ' ' , ' ')
            met    =   (argis('-met') .gt. 0)
            if(ntap.eq.' ')then
               write(LOT,*)' The pressure gradient field may not',
     &                     ' be a pipe.'
               write(LOT,*)' You must enter a filename for the',
     &                     ' pressure gradient'
               write(LOT,*)' dataset.'
               write(LOT,*)' JOB TERMINATED ABNORMALLY'
               stop
            endif
c-------
      return
      end
