C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
c ppg2psi.F
c Read a pressure gradient field (PPG), and convert to pressure (PSI),
c or pressure above normal (PSI). The output is technically guage
c pressure, since pressure at depth=0 is 0 (not atmospheric).
c
c Martin Albertin                            September 20. 1995
c
c program framework by:
c Mary Ann Thornton           v: 1.0         September 29, 1994
c************************************************************************
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
      parameter   (convm = 3.280833, ppgal = 0.05195)
      integer     lhed (SZLNHD)
      integer * 2 itr  (SZLNHD)
      real        head (SZLNHD)
      real        ppg  (SZLNHD)
      real        dz, nhg

      integer     nsamp, ntrc, nrec, iform, obytes, mode
      integer     luin, luout, lbytes, nbytes, lbyout

      character   ntap  * 100, otap * 100, name*5, version*4
      logical     hlp, query, met
      integer     argis
      equivalence (itr( 1), lhed (1), head(1))
      data lbytes/ 0 /, nbytes/ 0 /, name/'PPG2PP'/, version/' 1.0'/ 
 
c read program parameters from command line card image file
      query = ( argis ( '-?' ) .gt. 0 )
      hlp = ( argis ( '-h' ) .gt. 0 )
      if ( query .or. hlp )then
            call help()
            stop
      endif

c open printout
#include <f77/mbsopen.h>

c read command line arguments
      call gcmdln(ntap,otap,dz,mode,nhg,met)

c set depth conversion
      if ( met )then
            dz=dz*convm
            print* , ' dz (ft)= ',dz
      endif

c open input and output files
      call getln(luin , ntap ,'r', 0)
      call getln(luout, otap ,'w', 1)

c read line header from pressure gradient dataset
      call rtape(luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'ppg2psi: no line header read from unit ',luin
         write(LOT,*)' ( pressure gradient dataset )'
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         stop
      endif

c  save values from line header
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)

c     print line header into printer listing
      call hlhprt(itr, lbytes, name, 5, LERR)

c     determine obytes as the number of bytes in output trace
      obytes = SZTRHD + nsamp * SZSMPD

c     save command line arguments in the historical part of line header
      call savhlh(itr,lbytes,lbyout)

c     write the output line header
      call wrtape(luout, itr, lbyout )
 
c--------------------------------------------------
c     BEGIN PROCESSING
c--------------------------------------------------

c-----LOOP ON TRACES
      do 1001  mm = 1, nrec
      do 1003  nn = 1, ntrc
c        read the ppg traces into itr then move to ppg
         nbytes = 0
         call rtape(luin, itr, nbytes)
         if(nbytes .eq. 0) then
            write(LERR,*)'End of file on pressure gradient input:',luin
            write(LERR,*)'  trace= ',nn
            go to 999
         endif
         call vmov(lhed(ITHWP1),1,ppg,1,nsamp)

c--------LOOP ON SAMPLES
c-       compute pressure for each sample (convert from pounds/gallon)
         do 1002 kk = 1,nsamp
            z1=(kk-1)*dz
            if (ppg(kk) .le. 0) then
               ppg(kk)=nhg/ppgal
            endif
            pp1=ppg(kk)*ppgal*z1
            if (mode .eq. 2) then
               pp1=pp1-(z1*nhg)
            endif
            ppg(kk)=pp1
 1002    continue
c----       
         call vmov (ppg, 1, lhed(ITHWP1), 1, nsamp)
         call wrtape (luout, itr, obytes)
 1003 continue
 1001 continue

c--------------------------------------------------
c--------------------------------------------------
c-----
c     END PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------
 
  999 continue
 
c-----
c     close data files
      call lbclos(luin)
      call lbclos(luout)
c-----
      write(LERR,*)'end of ppg2psi, processed ',nrec,' record(s)',
     &               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
c...5....0....5....0....5....0....5....0....5....0....5....0....5....0..
      write(LER,*)'                                                    '
      write(LER,*)'PPG2PSI converts pressure gradient data (PPG units) '
      write(LER,*)'to pressure or pressure above normal (PSI units).   '
      write(LER,*)'The output pressure is, technically "guage" pressure'
      write(LER,*)'since pressure at depth=0 is 0 PSI, not atmospheric.'
      write(LER,*)'                                                    '
      write(LER,*)'Parameters [default values] :                       '
      write(LER,*)' -N[]       req  input pressure gradient dataset    '
      write(LER,*)' -O[]       req  ouput pressure dataset             '
      write(LER,*)' -dz[]      req  depth sample interval              '
      write(LER,*)' -mode[1]   req  pressure conversion mode           '
      write(LER,*)'               (1=pressure, 2=pressure above normal)'
      write(LER,*)' -nhg[0.46] req  normal hydrostatic gradient(psi/ft)'
      write(LER,*)'               (required for mode=2)                '
      write(LER,*)' -met       opt  metric depth units (default=ft)    '
      write(LER,*)'                                                    '
      write(LER,*)'Usage:                                              '
      write(LER,*)'                                                    '
      write(LER,*)'ppg2psi -N[] -O[] -dz[] -mode[] -nhg[] [-met]       '
      write(LER,*)'                                                    '
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,dz,mode,nhg,met)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name pressure gradient
c     otap  - C*100    output file name
c     dz    - real     depth sample interval
c     mode  - integer  pressure calculation mode: 1= psig
c                                                 2= pressure above normal
c     nhg   - real     normal pressure gradient (psi/ft), required if mode=2
c     met   - L        flag indicating metric units
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      logical     met
      integer     argis, mode
      real        dz, nhg
 
c-------
            call argstr('-N', ntap , ' ', ' ')
            call argstr('-O', otap , ' ', ' ')
            call argr4 ('-dz', dz , ' ' , ' ')
            call argr4 ('-nhg', nhg , 0.46 , 0.46)
            call argi4 ('-mode',mode, 1 , 1 )
            met    =   (argis('-met') .gt. 0)
            if(ntap.eq.' ')then
               write(LOT,*)' The pressure gradient field may not',
     &                     ' be a pipe.'
               write(LOT,*)' You must enter a filename for the',
     &                     ' pressure gradient'
               write(LOT,*)' dataset.'
               write(LOT,*)' JOB TERMINATED ABNORMALLY'
               stop
            endif
c-------
      return
      end
