C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
c Read an overburden gradient field, and some 
c  other parameters and compute a velocity trendline to be used in 
c  pore pressure prediction programs
c Output a velocity trendline field
c
c Reference:  
c   GEOPHYSICS, VOL. 54, NO. 1 (January 1989), P. 82-89, 4 Figs., 3 Tables
c   Empirical relationships among seismic velocity, effective pressure,
c    porosity, and clay content in sandstone.
c   D. Eberhart-Phillips, D-H. Han, and M. D. Zoback
c
c   The formula in kilometers and kilobars:
c   Vp = 5.77 - 6.94phi   - 1.73*C**.5  + 0.446(Pe - e**-16.7Pe)
c
c   The formula in feet and psi:
c   Vp = 18930 - 22769phi - 5676*C**.5 + .10089(Pe - 1463e**Pe/868)
c
c pptrnehz.F
c
c N. D. Whitmore, Jr. & Mary Ann Thornton   V: 1.0       September 30, 1994
c Modifications:
c   2/11/97 ... M. Albertin, modified code to compute trendlines using
c trace variable trendline parameters, revised online help
c
c************************************************************************
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
      parameter   (convm = 3.280833, convg = .433, ppgal = 0.052)
      integer     lhed (SZLNHD)
      integer * 2 itr  (SZLNHD)
      real        head (SZLNHD)
      real        vn (SZLNHD), obg(SZLNHD)
      real        nhg, acomp(100000), asphi(100000), avclay(100000)
      real        comp1, sphi1, vclay1, comp2, sphi2, vclay2
      real        dcomp, dsphi, dvclay

      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin, luout, lbytes, nbytes, lbyout
      integer     luflat, length, trace1, trace2

      character   ntap  * 100, otap * 100, name*8, version*4
      character   ftap * 100

      logical     verbos, hlp, query, met, eng, fflag
      integer     argis
      equivalence (itr( 1), lhed (1), head(1))
      data lbytes/ 0 /, nbytes/ 0 /, name/'PPTRNEHZ'/, version/' 1.0'/ 
 
      met = .false.
      eng = .false.
c-----
c     read program parameters from command line card image file
      query = ( argis ( '-?' ) .gt. 0 )
      hlp = ( argis ( '-h' ) .gt. 0 )
      if ( query .or. hlp )then
            call help()
            stop
      endif
c-----
c     open printout
#include <f77/open.h>
c-----
c     read command line arguments
      call gcmdln(ntap,otap,ftap,nhg, comp, sphi, vclay,
     &            met,eng,verbos)
c-----
c     open input and output files
      call getln(luin , ntap ,'r', 0)
      call getln(luout, otap ,'w', 1)
c 
c open parameter file if present:
       
      if (ftap .ne. ' ') then
         call alloclun ( luflat )
         length = lenth(ftap)
         open ( luflat, file=ftap(1:length), status='old', err=990 )
      endif

c-----
      lbytes = 0
      call rtape(luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'pptrnehz: no line header read from unit ',luin
         write(LOT,*)' ( velocity field )'
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         stop
      endif
c------
c     save values from velocity line header
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'Dx1000', idx,   LINHED)
      call saver(itr, 'Dz1000', idz,   LINHED)
      dx = idx/1000.
      dz = idz/1000.
      if(dz.le.0.0)then
         write(LOT,*)' Dz1000 in the line header cannot be zero.'
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         stop
      endif
c------
      if(met)dz = dz * convm
c------
c     print line header into printer listing
      call hlhprt(itr, lbytes, name, 5, LERR)
c-----
c     modify line header to reflect actual number of traces output
c----------------------
c     determine obytes as the number of bytes in output trace
      obytes = SZTRHD + nsamp * SZSMPD
c----------------------
c     save command line arguments in the historical part of line header
      call savhlh(itr,lbytes,lbyout)
c----------------------
c     write the output line header
      call wrtape(luout, itr, lbyout )
 
c-----
      if( verbos ) then
         call verbal(nsamp, nsi, ntrc, nrec, iform,
     &               ntap,otap,ftap
     &               nhg,comp,sphi,vclay,met,eng,verbos)
      endif

c-----
c--------------------------------------------------
c--------------------------------------------------
c-----
c     BEGIN PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------

c-----met=metric velocity field
c     eng=English velocity points for trend_line
c     convm=conversion factor for metric to english
c     convg=conversion factor for converting to grams/cubic_centimeter
c     read the overburden gradient trace into itr for the trace header
c     then move the above calculated trendline into the data
c     portion of the trace and write the trace out so the 
c     velocity trendline dataset will be the same size as the velocity
c     dataset for use in subsequent pore pressure programs
c    
c     PeV = effective stress
c     rat = horizontal/vertical ratio
c     PeM = mean stress

c
c     first fill trendline parameter array

      print*, 'ftap= ',ftap
      if (ftap .ne. ' ') then

         fflag= .true.
 10      read(luflat, *, end=777, err=992 ) trace2, sphi2, vclay2, comp2

            if(fflag) then
               print*,' first parameter line is ',trace2,' ',
     1                 sphi2,' ',vclay2,' ',comp2
               trace1 = 1
               sphi1=sphi2
               vclay1=vclay2
               comp1=comp2
               fflag=.false.
            endif
   
            dtr=trace2-trace1

            if(dtr .gt. 0) then
               dsphi=(sphi2-sphi1)/dtr
               dvclay=(vclay2-vclay1)/dtr
               dcomp=(comp2-comp1)/dtr
 
               do i = trace1, trace2
                  print*, i
                  asphi(i)=sphi1+ ((i-trace1)*dsphi)
                  avclay(i)=vclay1 + ((i-trace1)*dvclay)
                  acomp(i)=comp1+ ((i-trace1)*dcomp)
                  write(LER,*)' ',i,' ',asphi(i),' ',avclay(i),' ',
     1                acomp(i)
               enddo
 
               trace1=trace2
               sphi1=sphi2
               vclay1=vclay2
               comp1=comp2
            endif

         goto 10

777      continue

         if (trace2 .lt. ntrc) then
            trace2 = ntrc
            dtr=trace2-trace1

            do i = trace1, trace2
                asphi(i)=sphi1
                avclay(i)=vclay1
                acomp(i)=comp1
             enddo

          endif

      else 

          do i=1,ntrc
             asphi(i)=sphi
             avclay(i)=vclay
             acomp(i)=comp
          enddo

      endif

c-----RECORD LOOP
      do 1000 jj = 1,nrec
c--------TRACE LOOP
         do 1003  nn = 1,ntrc
            nbytes = 0
            call rtape(luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on velocity input:',luin
               write(LERR,*)'  rec= ',jj,'  trace= ',nn
               go to 999
            endif
            call vmov(lhed(ITHWP1),1,obg,1,nsamp) 
c-----------LOOP ON SAMPLES
            do 1001 kk = 1,nsamp
               z   = (kk-1)*dz
               PeV = (obg(kk)-nhg)*z
               rat = .039 * z**.33
               PeM = PeV*(1.0 + rat + rat)/3.
               phi_n  = asphi(nn) * exp( (-PeM)/acomp(nn))
               vn(kk) = 18930. - 22769. * phi_n
     &                  -5676. * sqrt(avclay(nn))
     &                  +.10089*PeM -1463.* exp( (-PeM)/868.) 

cmat     &                  +1463. * ( (PeM/14500.) - exp( (-PeM)/868.) )

 1001       continue
            if(met)then
               do 1002 kk = 1, nsamp
                  vn(kk) = vn(kk) / convm 
 1002          continue
            endif
            call vmov (vn, 1, lhed(ITHWP1), 1, nsamp)
            call wrtape (luout, itr, obytes)
 1003    continue
c-----
 1000 continue

c--------------------------------------------------
c--------------------------------------------------
c-----
c     END PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------
 
  999 continue
 
c-----
c     close data files
      call lbclos(luin)
      call lbclos(luout)
c-----
      write(LERR,*)'end of pptrnehz, processed ',nrec,' record(s)',
     &               ' with ',ntrc, ' traces'
      stop

  990 write(LERR,*) ' error opening parameter file: check spelling'
      stop
  992 write(LERR,*) ' error reading parameter file'
      stop

      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
c...5....0....5....0....5....0....5....0....5....0....5....0....5....0..
      write(LER,*)'                                                    '
      write(LER,*)'PPTRNEHZ computes a normal compaction velocity trend'
      write(LER,*)'using the EHZ empirical equation. The input data is '
      write(LER,*)'overburden pressure gradient (psi/ft). The three    '
      write(LER,*)'important program variables are:                    '
      write(LER,*)'                                                    '
      write(LER,*)'   P0  (-sphi) surface porosity                     '
      write(LER,*)'   VCL (-vclay) volume (percentage) of clay         '
      write(LER,*)'   C   (-comp) the compaction factor                '
      write(LER,*)'                                                    '
      write(LER,*)'If these parameters do not vary across a dataset,   '
      write(LER,*)'they can be input on the command line. For datasets '
      write(LER,*)'requiring trace variable trendlines, the program can'
      write(LER,*)'read an ascii file containing the trendline values  '
      write(LER,*)'at key control points (trace P0 VCL C).             '
      write(LER,*)'                                                    '
      write(LER,*)'Parameters [default values] :                       '
      write(LER,*)' -OBG[]      req    overburden gradient data set    '
      write(LER,*)' -VTR[]      req    output velocity trendline file  '
      write(LER,*)' -P          opt    trendline parameter file        '
      write(LER,*)' -nhg[0.465] req    normal hydrostatic gradient     '
      write(LER,*)' -comp[5000] req    compaction factor               '
      write(LER,*)' -sphi[0.4]  req    surface porosity                '
      write(LER,*)' -vclay[0.4] req    volume of clay                  '
      write(LER,*)' -met        opt    output velocity in meters/sec   '
      write(LER,*)' -eng        opt    output velocity in feet/sec     '
      write(LER,*)' -V          opt    verbose printout                '
      write(LER,*)'                                                    '
      write(LER,*)'Usage:                                              '
      write(LER,*)'pptrnehz -OBG[] -VTR[] -nhg[] -comp[] -sphi[] \     '
      write(LER,*)' -vclay[] -met -V                                   '
      write(LER,*)'                                                    '
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,ftap,nhg,comp,sphi,vclay,
     1                  met,eng,verbos)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name velocity
c     otap  - C*100    output file name
c     ftap  - C*100    parameter filename
c     nhg   - real     normal hydrostatic gradient
c     comp  - real     compaction factor
c     sphi  - real     surface phi
c     vclay - real     volume of clay
c     met   - L        flag indicating metric units
c     eng   - L        flag indicating English units
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*), ftap*(*)
      logical     verbos, met, eng
      integer     argis
      real        nhg,comp,sphi,vclay
 
c-------
c     last 2 arguments are values used when:
c     (1) if ONLY the key is present (no value attached to it)
c     (2) if NO key & no value are present
c-------
            call argstr('-OBG',  ntap,  ' ', ' ')
            call argstr('-VTR',  otap,  ' ', ' ')
            call argstr('-P', ftap,   ' ', ' ')
            call argr4 ('-nhg', nhg ,.465 ,.465)
            call argr4 ('-comp', comp ,5000.  ,5000.)
            call argr4 ('-sphi', sphi ,.4  ,.4)
            call argr4 ('-vclay',vclay, .4, .4)
            verbos =   (argis('-V') .gt. 0)
            met    =   (argis('-met') .gt. 0)
            eng    =   (argis('-eng') .gt. 0)

            if(.not.met .and. .not.eng)then
              write(LOT,*)' The Units must be specified'
              write(LOT,*)' JOB TERMINATED ABNORMALLY'
              stop
            endif
c-------
      return
      end
 
C***********************************************************************
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     &               ntap,otap,ftap
     &               nhg,comp,sphi,vclay,met,eng,verbos)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ntap  - C*100   input file name velocity
c     otap  - C*100   output file name
c     ftap  - C*100   input parameter filename
c     nhg   - real     normal hydrostatic gradient
c     comp  - real     compaction factor
c     sphi  - real     surface phi
c     vclay - real     volume of clay
c     met   - L       flag indicating metric units
c     eng   - L       flag indicating English units
c-----
#include <f77/iounit.h>
 
      integer     nsamp, nsi, ntrc, nrec
      real        nhg,comp,sphi,vclay
      logical     met,eng
      character   ntap*100, otap*100, ftap*100
 
      write(LERR,*)' '
      write(LERR,*)' line header values after default check '
      write(LERR,*)' # of samples/trace          =  ', nsamp
      write(LERR,*)' sample interval             =  ', nsi
      write(LERR,*)' traces per record           =  ', ntrc
      write(LERR,*)' records per line            =  ', nrec
      write(LERR,*)' format of data              =  ', iform
      write(LERR,*)' input data set name velocity=  ', ntap
      write(LERR,*)' output data set name        =  ', otap
      write(LERR,*)' parameter file name         =  ', ftap
      write(LERR,*)' normal hydrostatic gradient =  ', nhg
      write(LERR,*)' compaction factor           =  ', comp
      write(LERR,*)' surface porosity            =  ', sphi
      write(LERR,*)' volume of clay              =  ', vclay
      if(met)then
         write(LERR,*)' input velocities are in meters/sec '
      endif
      if(eng)then
         write(LERR,*)' input velocities are in feet/sec '
      endif
      write(LERR,*)' '
 
      return
      end
