C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C*********************************************************************
C NAME: PWMVZN   PLANE WAVE MIGRATION - V OF Z   REV 8.1     JAN 91   *
C*********************************************************************
C
C  PURPOSE:
C       PWMVZN performs a V of Z plane wave migration.
C
C  USAGE:
C       pwmvzn [-Nintape] [-Oouttape] [-Ccard] [-Sstretch] [-V] [-h]
C
C       -Nintape
C               Specifies 'intape' as the input SIS data set.  If
C               omitted, standard input is used.
C
C       -Oouttape
C               Specifies 'outtape' as the output SIS data set.  If
C               omitted, standard output is used.
C
C       -Ccard
C               Specifies 'card' as the input card file.  If omitted,
C               pwmvzn.crd is used.
C
C       -VCvcards
C               Specifies 'vcards' as the input velocity file.
C
C       -Sstretch
C               Specifies 'stretch' as a multiplier for the depths and
C               velocities.
C
C       -V
C               Causes the program to generate verbose print output.
C
C       -h
C               Causes the program to write help information to the
C               standard output and then halts.
C
C  FILES:
C       intape
C               Input standard SIS data set.
C
C       outtape
C               Output standard SIS data set.
C
C       card
C               Input card file.
C
C       vcards
C               Input velocity cards.
C
C       pfile
C               Output print file.  This file contains informational
C               output and error messages.  The print file name is
C               PW.xxxxx.yyyyy, where xxxxx and yyyyy are 5 digit
C               numbers that are generated from the process ID.
C
C  EXAMPLE:
C       The following command line will execute pwmvzn with an input SIS
C       data set named pwmvzn.inp, an output SIS data set named
C       pwmvzn.out, an input card file named pwmvzn.crd, a velocity card
C       file called velcards and with the verbose output option set:
C
C    pwmvzn -Npwmvzn.inp -Opwmvzn.out -Cpwmvzn.crd -VCvelcards -S1.2 -V
C
C       The following is an example of a card file:
C
C          1111111111222222222233333333334444444444555555555566666666667
C 1234567890123456789012345678901234567890123456789012345678901234567890
C
C TIME       ENDTIMEMS PADTIMEMS BEGTIMEMS      DTMS
C TIME            2000       200       100
C FREQUENCY  MINFREQHZ  F2FREQHZ  F3FREQHZ MAXFREQHZ
C FREQUENCY         20        25        55        60
C DEPTH      MAX DEPTH   DELTA Z
C DEPTH          10000        50
C WIDTH         DELTAX   BEG REC   END REC   REC INC  BEGTRACE  ENDTRACE
C WIDTH             80
C MODEL                REF VELOC BEAMWIDTH PADTRACE
C MODEL                     8000        10
C   # ANGLES     START INCREMENT
C          3        -8         8
C
C       The following is an example of a velocity card file:
C
C          1111111111222222222233333333334444444444555555555566666666667
C 1234567890123456789012345678901234567890123456789012345678901234567890
C
C NOVEL     # VELOCITY
C NOVEL              4
C VELOCITIES REFLECTED  INCIDENT END DEPTH
C          1      6000      6000      2000
C          2      5000      5100      4000
C          3      7000      7000      5000
C          4      8000      8000      8000
C
C  HISTORY:
C       DEC 87          REL 1.0         R.D. Coleman, QTC
C
C       JAN 88          REL 1.1         M.A. Thornton
C               Added GAMOCO and COF TAPEIO
C
C       FEB 88          REL 1.2         M.A. Thornton
C               If INC.VEL = 0., then INC.VEL = REF.VEL
C
C       APR 88          REL 1.3         M.A. Thornton
C               (1) Changed RFFTM to RFFTMR.  Ron Coleman changed the
C               names because MATHADV came out with a different RFFTM
C               (it has a different call list).  RFFTMR is in DANLIB.
C               (2) Added call to HLHPRT.
C
C       MAY 88          REL 2.0         R.D. Coleman, QTC
C               Replaced FFT calls in PWREAD with SCILIB looping mixed
C               radix FFT'S (RFFTMLT and CFFTMLT).  Replaced call to
C               NRFFTM with call to NRFFT5.  Increased MAXNX to 1027
C               to increase the size of depth for FFT work space and
C               decreased the MAXNW tO 512 (=MAXNT/2).
C
C       JUN 88          REL 3.0         R.D. Coleman, QTC
C               Replaced XFSMP with CCUINT (CAL-coded cubic interpo-
C               lation routine) for performing the resampling.
C               CCUINT is in DANLIB.
C
C       JUL 88          REL 4.0         R.D. Coleman, QTC
C               Replaced XVZPKL with XVZPK2 and replaced IPWR2 with
C               NRFFT5.  XVZPK2 and NRFFT5 are in DANLIB.  This change
C               allows NK to have values other than a power of two.
C               Specifically, NK will be set to the smallest even
C               number greater than or equal to NX that is of the form
C               2**I * 3**J * 5**K.  (NX is the number of traces.)
C
C       AUG 88          REL 5.0         R.D. Coleman, QTC
C               Major restructuring and clean-up of the code.
C               (1) Eliminated one COMMON block.
C               (2) Replaced COMMON's with INCLUDE statements.
C               (3) Replaced subroutines PWCRD1, PWCRD2, PWCRD3, PWCRD4,
C               PWCRD5, PWCRD6, and PWCRD7 with PWRCRD.  PWRCRD calls
C               the general input routines RDCARD, RDVMOD, and RDANGL.
C               These routines are in DANLIB.
C               (4) Restructured error messages.
C               (5) Doubled the maximum number of samples per trace.
C
C       SEP 88          REL 5.1         T.G. Mattson, QTC
C               Added the capability to pad the number of traces.
C
C       APR 89          REL 5.2         M.A. THORNTON
C               DOUBLE NO. TRACES, INCREASE NO. OUTPUT DEPTH SAMPLES
C
C       AUG 89          REL 5.3         M.A. THORNTON
C               Added stretch parameter to adjust velocities and depths
C
C       SEP 89          REL 5.4         M.A. THORNTON
C               REWROTE PWRCRD TO DO SIMPLE READING OF CARDS
C               CHANGED THE NAME TO PWMVZNSS, AND EVENTUALLY ADDED THE
C               SORT AND SUM ROUTINES TO THIS PROGRAM
C               PROGRAM NO LONGER CALLS RDCARD
C
C       JAN 90          REL 6.0         M.A. THORNTON
C               Changed pwmvzss to read velocity cards from separate
C               file program will be called pwmvzn, pwmvzss may or
C               may not be completed at another time.
C
C       FEB 90          REL 6.1         M.A. THORNTON
C               Removed # velocities from the model card and introduced
C               a new card (NOVEL) to contain the number of velocities.
C               Added .01 to the nz computation in pwrcrd
C
C       MAY 90          REL 6.2         M.A. THORNTON
C               Corrected internal name of the crd file to pwmvzn.crd
C
C       MAY 90          REL 7.0         R.D. Coleman, QTC
C               ADDED SUBROUTINE PWITAB AND CHANGED CALL TO XVZPK2 TO
C               CALL TO XVZPK3.  PHASE SHIFT VECTORS ARE NOW CALCULATED
C               BY TABLE LOOKUP.
C
C       NOV 90          REL 7.1         R.D. Coleman, QTC
C               (1) Did general clean up and minor restructuring.
C               (2) Changed name of common file to pwmvzn.h
C               (3) Fixed PWITAB so that the complex function table
C                   requires less space
C               (4) Split the functionality of RDVMOD between two new
C                   routines - PWRDVM and PWVMOD.
C               (5) Calculate the two way travel time (TT2WAY) and use
C                   it as the default ENDTIME.
C               (6) Stop migrating whenever an angle becomes critical.
C
C       NOV 90          REL 8.0         R.D. Coleman, QTC
C               Wave field and image data is now maintained in natural
C               order as opposed to FFT order.  Subroutine XVZPK3
C               replaced by XVZPK4.
C
C       JAN 91          REL 8.1         R.D. Coleman, QTC
C               Corrected K limits calculation.
C
C-----------------------------------------------------------------------
C
      PROGRAM PWMVZN
C
      INCLUDE 'pwmvzn.h'
C
C-----------------------------------------------------------------------
C
C===  INITIALIZE, OPEN FILES, ETC.
C
      CALL PWINIT
C
C===  READ LINE HEADER & JOB PARAMETERS, WRITE OUTPUT LINE HEADER
C
      CALL PWPARM
C
C===  INITIALIZE COMPLEX FUNCTION TABLE
C
      CALL PWITAB
C
      NBYTES = 4 * NSMP + 2 * NTRHDR
C
C===  LOOP OVER ANGLE (RECORD)
C
      NXREC = IREC1
      KREC  = 0
      DO 120 JREC = 1, IREC2
C
         IF (JREC .EQ. NXREC) THEN
            IF (VERBOS) WRITE (LUPRT, *) ' PROCESSING RECORD ', JREC
            NXREC = NXREC + INCREC
            KREC  = KREC  + 1
C
C======     READ RECORD DATA
C
            CALL PWREAD (JREC)
C
C======     DOWNWARD CONTINUE AND IMAGE
C
            CALL PWDCIM (JREC)
C
C======     RESAMPLE
C
            CALL PWRSMP
C
C======     WRITE RECORD
C
            CALL PWWRIT (KREC)
C
         ELSE
C
C======     SKIP RECORD
C
            IF (VERBOS) WRITE (LUPRT, *) ' SKIPPING RECORD ', JREC
            CALL SKIPT (LUINP, NTRC, NBYTES)
C
         ENDIF
C
  120 CONTINUE
C
C===  CLOSE FILES, CLEAN-UP, & EXIT
C
      CALL PWEXIT (0)
C
      END
C*********************************************************************
C NAME: PWINIT                                                       *
C*********************************************************************
C
C  PURPOSE:
C      PWINIT INITIALIZES THE PROGRAM INCLUDING OPENING ALL REQUIRED
C      FILES.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWINIT
C
      INCLUDE 'pwmvzn.h'
C
      CHARACTER*2   PNAME
      CHARACTER*1   PARR(66)
      CHARACTER*128 NTAPE, OTAPE, CARDS, VCARDS
      INTEGER       ARGIS
C
      DATA PNAME/'PZ'/
      DATA PARR/ ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ',
     1           ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ',
     2           'V', '(', 'Z', ')', ' ',
     3           'M', 'I', 'G', 'R', 'A', 'T', 'I', 'O', 'N', ' ',
     4           ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ',
     5           ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ',
     6           ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' '/
C
C-----------------------------------------------------------------------
C
  901 FORMAT (/' ', 'PWMVZN COMMAND LINE ARGUMENTS',
     &        /' ',
     &        /' ', 'INPUTS:',
     &        /' ', '  -N[ntape]   - INPUT  DATASET NAME',
     &        /' ', '  -O[otape]   - OUTPUT DATASET NAME',
     &        /' ', '  -C[cards]   - CARD FILE NAME',
     &        /' ', '  -VC[cards]  - VELOCITY CARD FILE NAME',
     &        /' ', '  -S[stretch] - STRETCH DEPTHS, VELOCITIES',
     &        /' ', '  -V          - VERBOSE PRINTOUT',
     &        /' ',
     &        /' ', 'USAGE:',
     &        /' ', '  pwmvzn -N[] -O[] -C[] -VC[] -S[] -V'/)
  902 FORMAT (/' ', ' INPUT DATASET  = ' / A128,
     &        /' ', ' OUTPUT DATASET = ' / A128)
  903 FORMAT (/' ', ' NO VELOCITY CARD FILE NAME ENTERED'/)
C
C-----------------------------------------------------------------------
C
C  IF HELP, WRITE INSTRUCTIONS TO SCREEN
C
      LUTRM = 0
      IF (ARGIS( '-h' ) .GT. 0) THEN
         WRITE (LUTRM, 901)
         CALL PWEXIT (-1)
      ENDIF
C
C  CREATE UNIQUE PRINT FILE NAME FROM THE PROCESS ID & OPEN PRINT FILE
C
      CALL OPENPR (LUSUR, LUPRT, PNAME, JERR)
      IF (JERR .NE. 0) CALL PWEXIT (-2)
C
      NLIN = 1
      CALL GAMOCO (PARR, NLIN, LUPRT)
C
      VERBOS = ARGIS( '-V' ) .GT. 0
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWINIT'
C
C  GET REMAINING COMMAND LINE ARGUMENTS
C
      CALL ARGSTR ('-N', NTAPE , ' ', ' ')
      CALL ARGSTR ('-O', OTAPE , ' ', ' ')
      CALL ARGSTR ('-C', CARDS , ' ', ' ')
      CALL ARGSTR ('-VC',VCARDS, ' ', ' ')
      CALL ARGR4  ('-S', STRCH, 1.0, 1.0)
C
C  SET LUTRM
C
      IF (NTAPE .EQ. ' ') LUTRM = 2
C
C  DEFAULT FOR PIPES
C
      LUINP = 0
      LUOUT = 1
C
C  OPEN I/O FILES
C
      IF (NTAPE .NE. ' ') CALL LBOPEN (LUINP, NTAPE, 'r')
      IF (OTAPE .NE. ' ') CALL LBOPEN (LUOUT, OTAPE, 'w')
C
      IF (CARDS .NE. ' ') THEN
         OPEN (UNIT=LUCRD, FILE=CARDS, STATUS='OLD')
      ELSE
         II = ICOPEN ('-pwmvzn.crd', LUCRD)
      ENDIF
C
C  OPEN VELOCITY CARD FILE
C
      IF (VCARDS .NE. ' ') THEN
         OPEN (UNIT=LUVEL, FILE=VCARDS, STATUS='OLD')
      ELSE
         WRITE(LUPRT, 903)
         CALL PWEXIT(2)
      ENDIF
C
      WRITE (LUPRT, 902) NTAPE, OTAPE
C
      RETURN
      END
C*********************************************************************
C NAME: PWPARM                                                       *
C*********************************************************************
C
C  PURPOSE:
C      PWPARM READS THE INPUT LINE HEADER, JOB PARAMETERS, AND WRITES
C      THE OUTPUT LINE HEADER.  IN ADDITION, PWPARM CALCULATES SOME JOB
C      PARAMETERS.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWPARM
C
      INCLUDE 'pwmvzn.h'
C
      INTEGER IHEAD(1500)
      CHARACTER*6  PNAME
C
      DATA PNAME / 'PWMVZN' /
C
C-----------------------------------------------------------------------
C
  901 FORMAT (/' ', 'INPUT LINE HEADER PARAMETERS:')
  902 FORMAT (/' ', 'OUTPUT LINE HEADER PARAMETERS:')
  903 FORMAT (/' ', '   NUMBER OF SEISMIC RECORDS IN THIS JOB =', I5,
     2        /' ', '   NUMBER OF TRACES PER SEISMIC RECORD   =', I5,
     3        /' ', '   NUMBER OF DATA SAMPLES PER TRACE      =', I5,
     4        /' ', '   SAMPLE INTERVAL (MILLISECONDS)        =', I5)
  991 FORMAT (/' ', '***** ERROR - LINE HEADER READ ERROR *****')
  992 FORMAT (/' ', '***** ERROR - DIMENSIONS EXCEED ARRAY SIZES *****')
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWPARM'
C
C=======================================================================
C                       PROCESS LINE HEADER
C=======================================================================
C
      NBYTES = 0
      CALL RTAPE (LUINP, IHEAD, NBYTES)
      IF (NBYTES .EQ. 0) THEN
         WRITE (LUPRT, 991)
         CALL PWEXIT (1)
      ENDIF
C
C  GET PARAMETERS FROM LINE HEADER
C
CMAT
      LINHED = 0
      CALL SAVER(IHEAD, 'NumTrc', NTRC, LINHED)
      CALL SAVER(IHEAD, 'NumRec', NREC, LINHED)
      CALL SAVER(IHEAD, 'SmpInt', IDT0, LINHED)
      CALL SAVER(IHEAD, 'NumSmp', NSMP, LINHED)
CMAT  NTRC = IHEAD(11)
CMAT  NREC = IHEAD(12)
CMAT  IDT0 = IHEAD(13)
CMAT  NSMP = IHEAD(14)
C%%%
      IF (VERBOS) THEN
         WRITE (LUPRT, 901)
         WRITE (LUPRT, 903) NREC, NTRC, NSMP, IDT0
      ENDIF
C%%%
      IF (NREC .GT. MAXREC .OR. NSMP .GT. MAXSMP) THEN
          WRITE (LUPRT, *) ' NREC, MAXREC, NSMP, MAXSMP = ', 
     &                       NREC, MAXREC, NSMP, MAXSMP
         WRITE (LUPRT, 992)
         CALL PWEXIT (1)
      ENDIF
C
C=======================================================================
C                 READ PARAMETERS FROM CARD FILE
C=======================================================================
C
      CALL PWRDVM
      CALL PWRCRD
      CALL PWVMOD
C
C=======================================================================
C                  UPDATE AND OUTPUT LINE HEADER
C=======================================================================
C
      LEN = 6
      CALL HLHPRT (IHEAD, NBYTES, PNAME, LEN, LUPRT)
C
CMAT
      CALL SAVEW(IHEAD, 'NumTrc', NX   , LINHED)
      CALL SAVEW(IHEAD, 'NumRec', NREC2, LINHED)
      CALL SAVEW(IHEAD, 'SmpInt', IDT  , LINHED)
      CALL SAVEW(IHEAD, 'NumSmp', NZ   , LINHED)
CMAT  IHEAD(11) = NX
CMAT  IHEAD(12) = NREC2
CMAT  IHEAD(13) = IDT
CMAT  IHEAD(14) = NZ
      CALL WRTAPE (LUOUT, IHEAD, NBYTES)
C%%%
      IF (VERBOS) THEN
         WRITE (LUPRT, 902)
         WRITE (LUPRT, 903) NREC2, NX, NZ, IDT
      ENDIF
C%%%
      IF (VERBOS) CALL PWWCOM
C%%%
      RETURN
      END
C********************************************************************C
C NAME: PWRDVM  READ VELOCITY MODEL                                  C
C********************************************************************C
C
C  PURPOSE:
C       PWRDVM reads a velocity model from a file.
C
C---------------------------------------------------------------------
C
      SUBROUTINE PWRDVM
C
      INCLUDE 'pwmvzn.h'
C
      CHARACTER*80 CARD
C
C-----------------------------------------------------------------------
C
  900 FORMAT (A80)
  901 FORMAT ( 10X, I10)
  902 FORMAT ( I10, 3F10.0)
  911 FORMAT (/' ', 'NOVEL     # VELOCITY')
  912 FORMAT (/' ', 'VELOCITIES REFLECTED  INCIDENT END DEPTH')
  921 FORMAT ( ' ', 'NOVEL     ', I10)
  922 FORMAT ( ' ', I10, 3F10.1)
  923 FORMAT (/' ', 'VELOCITIES AND DEPTHS ARE STRETCHED BY', F6.3)
  991 FORMAT (/' ', '***** PWRDVM ERROR - UNEXPECTED END-OF-FILE')
  992 FORMAT (/' ', '***** PWRDVM ERROR - DATA CONVERSION ERROR'/
     &         ' ', '      INPUT CARD: ', A80)
  993 FORMAT (/' ', '***** PWRDVM ERROR - INVALID DATA CARD ORDER')
  994 FORMAT (/' ', '***** PWRDVM ERROR - INVALID DATA VALUE')
  995 FORMAT (/' ', '***** PWRDVM ERROR - INVALID NUMBER OF VELOCITIES')
  996 FORMAT (/' ', '***** PWRDVM ERROR - TOO MANY VELOCITIES')
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWRDVM'
C
C  READ NOVEL (NUMBER OF VELOCITIES) CARD
C
  100 CONTINUE
         READ (LUVEL, 900, END=810) CARD
	 IF (CARD(1:5) .NE. 'NOVEL') GO TO 100
C
      WRITE (LUPRT, 911)
      READ  (LUVEL, 901, END=810) NZSEG
      WRITE (LUPRT, 921         ) NZSEG
C
      IF (NZSEG .LE.      0) GO TO 850
      IF (NZSEG .GT. MAXSEG) GO TO 860
C
C  READ VELOCITY CARDS
C
  200 CONTINUE
         READ (LUVEL, 900, END=810) CARD
	 IF (CARD(1:10) .NE. 'VELOCITIES') GO TO 200
C
      WRITE (LUPRT, 912)
      TT2WAY = 0.0
      ZLAST  = 0.0
      DO 210 IZSEG = 1, NZSEG
         READ  (LUVEL, 900, END=810) CARD
         READ  (CARD , 902, ERR=820) JZSEG, VREFL, VINCI, ZEND
C
	 IF (VINCI .LE. 0.0) VINCI = VREFL
C
	 IF (STRCH .NE. 0.0) THEN
	    VREFL = VREFL * STRCH
	    VINCI = VINCI * STRCH
	    ZEND  = ZEND  * STRCH
	 ENDIF
C
         WRITE (LUPRT, 922) JZSEG, VREFL, VINCI, ZEND
C
         IF (JZSEG .NE. IZSEG) GO TO 830
C
         ZSTEP = ZEND - ZLAST
         ZLAST = ZEND
         IF (VREFL .LE. 0.0 .OR. ZSTEP .LE. 0.0) GO TO 840
C
         TT2WAY = TT2WAY + ZSTEP / VREFL + ZSTEP / VINCI
C
         ZSSLOI(IZSEG) = VINCI
         ZSSLOR(IZSEG) = VREFL
         ZSDZ  (IZSEG) = ZSTEP
  210 CONTINUE
C
      WRITE (LUPRT, 923) STRCH
C
      RETURN
C
C  ERROR EXITS
C
  810 CONTINUE
      WRITE (LUPRT, 991)
      CALL PWEXIT (7)
C
  820 CONTINUE
      WRITE (LUPRT, 992) CARD
      CALL PWEXIT (7)
C
  830 CONTINUE
      WRITE (LUPRT, 993)
      CALL PWEXIT (7)
C
  840 CONTINUE
      WRITE (LUPRT, 994)
      CALL PWEXIT (7)
C
  850 CONTINUE
      WRITE (LUPRT, 995)
      CALL PWEXIT (7)
C
  860 CONTINUE
      WRITE (LUPRT, 996)
      CALL PWEXIT (7)
C
      END
C*********************************************************************
C NAME: PWRCRD                                                       *
C*********************************************************************
C
C  PURPOSE:
C      PWRCRD READS AND PROCESSES THE CONTENTS OF THE CARD FILE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWRCRD
C
      INCLUDE 'pwmvzn.h'
C
      INTEGER IPWR(4)
      CHARACTER*80 CARD
C
      DATA PI / 3.141592653589793 /
C
      IROUND(I,J) = J * IFIX( FLOAT( I ) / FLOAT( J ) + 0.5 )
C
C-----------------------------------------------------------------------
C
  900 FORMAT (A80)
  901 FORMAT (10X, 4I10) 
  902 FORMAT (10X, 4F10.0) 
  903 FORMAT (10X, 2F10.0) 
  904 FORMAT (10X,  F10.0, 5I10  ) 
  905 FORMAT (10X,   10X , 2F10.0, I10) 
  911 FORMAT (/' ', 'TIME       ENDTIMEMS PADTIMEMS BEGTIMEMS',
     &              '      DTMS')
  912 FORMAT (/' ', 'FREQUENCY  MINFREQHZ  F2FREQHZ  F3FREQHZ',
     &              ' MAXFREQHZ')
  913 FORMAT (/' ', 'DEPTH      MAX DEPTH   DELTA Z')
  914 FORMAT (/' ', 'WIDTH         DELTAX   BEG REC   END REC',
     &              '   REC INC  BEGTRACE  ENDTRACE')
  915 FORMAT (/' ', 'MODEL                REF VELOC BEAMWIDTH',
     &              '  PADTRACE')
  921 FORMAT ( ' ', 'TIME      ', 4I10  )
  922 FORMAT ( ' ', 'FREQUENCY ', 4F10.1)
  923 FORMAT ( ' ', 'DEPTH     ', 2F10.1)
  924 FORMAT ( ' ', 'WIDTH     ',  F10.1, 5I10  )
  925 FORMAT ( ' ', 'MODEL     ',   10X , 2F10.1, I10)
  991 FORMAT (/' ', '***** ERROR - UNEXPECTED END OF CARD FILE *****')
  992 FORMAT (/' ', '***** ERROR - INVALID PARAMETER VALUE *****')
  993 FORMAT (/' ', '***** ERROR - DIMENSIONS EXCEED ARRAY SIZES *****')
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWRCRD'
C
C=======================================================================
C                 READ SCALAR PARAMETERS FROM CARD FILE
C=======================================================================
C
C  READ TIME CARD
C
   10 CONTINUE
         READ (LUCRD, 900, ERR=810) CARD
         IF (CARD(1:4) .NE. 'TIME') GO TO 10
C
      READ (LUCRD, 901) ITEND, ITPAD, ITBEG, IDT
C
C  READ FREQUENCY CARD
C
   20 CONTINUE
         READ (LUCRD, 900, ERR=810) CARD
         IF (CARD(1:9) .NE. 'FREQUENCY') GO TO 20
C
      READ (LUCRD, 902) F1, F2, F3, F4
C
C  READ DEPTH CARD
C
   30 CONTINUE
         READ (LUCRD, 900, ERR=810) CARD
         IF (CARD(1:5) .NE. 'DEPTH') GO TO 30
C
      READ (LUCRD, 903) ZMAX, DZ
C
C  READ WIDTH CARD
C
   40 CONTINUE
         READ (LUCRD, 900, ERR=810) CARD
         IF (CARD(1:5) .NE. 'WIDTH') GO TO 40
C
      READ (LUCRD, 904) DX, IREC1, IREC2, INCREC, ITRC1, ITRC2
C
C  READ MODEL CARD
C
   50 CONTINUE
         READ (LUCRD, 900, ERR=810) CARD
         IF (CARD(1:5) .NE. 'MODEL') GO TO 50
C
      READ (LUCRD, 905) VELREF, BWIDTH, NXPAD
C
C=======================================================================
C                        PROCESS TIME PARAMETERS
C=======================================================================
C
      IF (IDT   .LE. 0) IDT   = IDT0
      IF (ITEND .LE. 0) ITEND = 1000.0 * TT2WAY + 0.5
      IF (ITPAD .LE. 0) ITPAD = 0
C
      ITBEG = IROUND( ITBEG, IDT )
      ITEND = IROUND( ITEND, IDT )
      ITPAD = IROUND( ITPAD, IDT )
C
C     CALCULATE:
C        IT1   = INDEX OF FIRST TIME SAMPLE TO USE
C        IT2   = INDEX OF LAST  TIME SAMPLE TO USE
C        NTOFF = NUMBER OF ZERO SAMPLES BEFORE FIRST SAMPLE
C        NTPAD = NUMBER OF ZERO SAMPLES AFTER LAST SAMPLE (PAD)
C
C        NOTE: FIRST SAMPLE IN TRACE CORRESPONDS TO TIME = 0
C
      IT2   = ITEND / IDT + 1
      NTPAD = ITPAD / IDT
      IF (ITBEG .GE. 0) THEN
         NTOFF = 0
         IT1   = ITBEG / IDT + 1
      ELSE
         NTOFF = - ITBEG / IDT
         IT1   = 1
      ENDIF
C
C     IF IT2 > NUMBER OF SAMPLES, ADJUST IT2 AND NTPAD
C
      IF (IT2 .GT. NSMP) THEN
         NTPAD = NTPAD + IT2 - NSMP
         IT2   = NSMP
      ENDIF
C
C     CALCULATE NT = NUMBER OF TIME SAMPLES FOR FFT, PAD AS NEEDED
C
      MT    = IT2 - IT1 + 1 + NTOFF + NTPAD
      CALL NRFFT (MT, 5, NT, IPWR)
      DT    = 0.001 * FLOAT( IDT )
      NTPAD = NTPAD + NT - MT
C
C     RECALCULATE ITPAD AND ITEND
C
      ITPAD = IDT * NTPAD
      ITEND = IDT * (IT2 - 1)
C
      WRITE (LUPRT, 911)
      WRITE (LUPRT, 921) ITEND, ITPAD, ITBEG, IDT
C
      IF (ITEND .LE. ITBEG) GO TO 820
      IF (NT    .GT. MAXNT) GO TO 830
C
C=======================================================================
C                     PROCESS FREQUENCY PARAMETERS
C=======================================================================
C
      IF (F2 .LE. 0) F2 = F1
      IF (F3 .LE. 0) F3 = F4
      WRITE (LUPRT, 912)
      WRITE (LUPRT, 922) F1, F2, F3, F4
C
      IF (F1.LT.0.0 .OR. F2.LT.F1 .OR. F3.LT.F2 .OR. F4.LT.F3) GO TO 820
C
C     CALCULATE REMAINING FREQUENCY PARAMETERS
C
      W1 = 2.0 * PI * F1
      W2 = 2.0 * PI * F2
      W3 = 2.0 * PI * F3
      W4 = 2.0 * PI * F4
C
      DW  = 2.0 * PI / (NT * DT)
      IW1 = IFIX( W1 / DW ) + 1
      IF (DW*(IW1-1) .LT. W1) IW1 = IW1 + 1
      IW2 = IFIX( W4 / DW ) + 1
      IF (W2 .GT. W1 .AND. DW*(IW1-1) .EQ. W1) IW1 = IW1 + 1
      IF (W4 .GT. W3 .AND. DW*(IW2-1) .EQ. W4) IW2 = IW2 - 1
      NW  = IW2 - IW1 + 1
C
      IF (NW .GT. MAXNW) GO TO 830
C
      DO 110 I = 1, NW
         OMEGA(I) = DW * (IW1 + I - 2)
         IF      (OMEGA(I) .LT. W2) THEN
            FILTR(I) = (OMEGA(I) - W1) / (W2 - W1)
         ELSE IF (OMEGA(I) .GT. W3) THEN
            FILTR(I) = (W4 - OMEGA(I)) / (W4 - W3)
         ELSE
            FILTR(I) = 1.0
         ENDIF
  110 CONTINUE
C
      FMIN = OMEGA( 1) / (2.0 * PI)
      FMAX = OMEGA(NW) / (2.0 * PI)
C
C=======================================================================
C                       PROCESS DEPTH PARAMETERS
C=======================================================================
C
      IF (STRCH .NE. 0.0) THEN
         ZMAX = ZMAX * STRCH
         DZ   = DZ   * STRCH
      ENDIF
C
      WRITE (LUPRT, 913)
      WRITE (LUPRT, 923) ZMAX, DZ
C
      IF (ZMAX .LE. DZ .OR. DZ .LE. 0.0) GO TO 820
C
      NZ = IFIX( ZMAX / DZ + .01)
C
      IF (NZ .GT. MAXNZ) GO TO 830
C
C=======================================================================
C                       PROCESS WIDTH PARAMETERS
C=======================================================================
C
      IF (IREC1  .LE. 0) IREC1  = 1
      IF (IREC2  .LE. 0 .OR. IREC2 .GT. NREC) IREC2  = NREC
      IF (INCREC .LE. 0) INCREC = 1
      IF (ITRC1  .LE. 0) ITRC1  = 1
      IF (ITRC2  .LE. 0) ITRC2  = NTRC
C
      NREC2 = (IREC2 - IREC1) / INCREC + 1
      IREC2 = IREC1 + (NREC2 - 1) * INCREC
C
      WRITE (LUPRT, 914)
      WRITE (LUPRT, 924) DX, IREC1, IREC2, INCREC, ITRC1, ITRC2
C
      IF (IREC1 .GT. IREC2 .OR. ITRC1 .GT. ITRC2) GO TO 820
C
C     CALCULATE REMAINING WIDTH PARAMETERS
C     NOTE: NK IS FORCED TO BE EVEN BY USING NRFFT INSTEAD OF NCFFT
C     BECAUSE THERE MAY STILL BE CODE LEFT FROM THE RADIX 2 FFT DAYS
C     THAT EXPECTS NK TO BE EVEN.
C
      NX    = ITRC2 - ITRC1 + 1
      CALL NRFFT (NX+NXPAD, 5, NK, IPWR)
      DK    = 2.0 * PI / (NK * DX)
      NXPAD = NK - NX
C
      IF (NX .GT. MAXNX .OR. NK .GT. MAXNK) GO TO 830
C
      KBIAS = (NK + 1) / 2
      DO 120 I = 1, NK
	 K = I - KBIAS
         AK(I) = K * DK
  120 CONTINUE
C
C=======================================================================
C                       PROCESS MODEL PARAMETERS
C=======================================================================
C
      WRITE (LUPRT, 915)
      WRITE (LUPRT, 925) VELREF, BWIDTH, NXPAD
C
      IF (VELREF .LE. 0.0 .OR. BWIDTH .LE. 0.0) GO TO 820
C
C=======================================================================
C                              READ ANGLES
C=======================================================================
C
      READ (LUCRD, 900) CARD
      CALL RDANGL (LUCRD, LUPRT, LUPRT, NREC, KREC, THETA, IERR)
      IF (IERR .NE. 0) CALL PWEXIT (2)
C
      RETURN
C
C=======================================================================
C                              ERROR EXITS
C=======================================================================
C
  810 CONTINUE
      WRITE (LUPRT, 991)
      CALL PWEXIT (2)
C
  820 CONTINUE
      WRITE (LUPRT, 992)
      CALL PWEXIT (2)
C
  830 CONTINUE
      WRITE (LUPRT, 993)
      CALL PWEXIT (2)
C
      END
C********************************************************************C
C NAME: PWVMOD  COMPUTE VELOCITY MODEL                               C
C********************************************************************C
C
C  PURPOSE:
C       PWVMOD computes a velocity model from velocities and Z steps.
C
C---------------------------------------------------------------------
C
      SUBROUTINE PWVMOD
C
      INCLUDE 'pwmvzn.h'
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWVMOD'
C
      NZT = 0
      DO 110 IZSEG = 1, NZSEG
         VREFL = ZSSLOR(IZSEG)
         VINCI = ZSSLOI(IZSEG)
         ZSTEP = ZSDZ  (IZSEG)
C
         SLOR  = 1.0 / VREFL
         SLOI  = 1.0 / VINCI
	 AVEL  = 2.0 / (SLOR + SLOI)
         DZMAX = 0.25 * AVEL / FMAX
         MZ    = IFIX( ZSTEP / DZMAX )
         IF (MZ * DZMAX .LT. ZSTEP) MZ = MZ + 1
C
         IZSNZ (IZSEG) = MZ
         ZSDZ  (IZSEG) = ZSTEP / FLOAT( MZ )
         ZSSLOR(IZSEG) = SLOR
         ZSSLOI(IZSEG) = SLOI
C
         NZT = NZT + MZ
  110 CONTINUE
C
      IF (NZT .GT. MAXNZT) THEN
         WRITE (LUPRT, *) '***** ERROR: NZT IS TOO LARGE'
         CALL PWEXIT (8)
      ENDIF
C
      RETURN
      END
C*********************************************************************
C NAME: PWWCOM                                                       *
C*********************************************************************
C
C  PURPOSE:
C      PWWCOM WRITES SELECTED COMMOM BLOCK VARIABLES TO LUPRT.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWWCOM
C
      INCLUDE 'pwmvzn.h'
C
C-----------------------------------------------------------------------
C
  900 FORMAT (/' ', 'COMMON BLOCK PARAMETERS:')
  901 FORMAT (/' ', 'NUMBER OF INPUT RECORDS     =', I5,
     2        /' ', 'NUMBER OF TRACES PER RECORD =', I5,
     3        /' ', 'NUMBER OF SAMPLES PER TRACE =', I5,
     4        /' ', 'FIRST RECORD                =', I5,
     5        /' ', 'LAST  RECORD                =', I5,
     6        /' ', 'RECORD INCREMENT            =', I5,
     7        /' ', 'NUMBER OF OUTPUT RECORDS    =', I5,
     8        /' ', 'START TRACE IN EACH RECORD  =', I5,
     9        /' ', 'END   TRACE IN EACH RECORD  =', I5,
     &        /' ', 'INPUT  SAMPLE RATE (MS)     =', I5,
     1        /' ', 'OUTPUT SAMPLE RATE (MS)     =', I5)
  902 FORMAT (/' ', 'NUMBER OF T-S, DELTA T      =', I5, E14.5,
     2        /' ', 'NUMBER OF W-S, DELTA W      =', I5, E14.5,
     3        /' ', 'NUMBER OF X-S, DELTA X      =', I5, E14.5,
     4        /' ', 'NUMBER OF K-S, DELTA K      =', I5, E14.5,
     5        /' ', 'NUMBER OF Z-S, DELTA Z      =', I5, E14.5)
  903 FORMAT (/' ', 'INDEX OF FIRST SAMPLE       =', I5,
     2        /' ', 'INDEX OF LAST  SAMPLE       =', I5,
     3        /' ', 'LEADING  ZERO PAD           =', I5,
     4        /' ', 'TRAILING ZERO PAD           =', I5)
  904 FORMAT (/' ', 'INDEX OF FIRST OMEGA        =', I5,
     2        /' ', 'INDEX OF LAST  OMEGA        =', I5)
  905 FORMAT (/' ', 'MINIMUM FREQUENCY           =', F9.3,
     2        /' ', 'MAXIMUM FREQUENCY           =', F9.3,
     3        /' ', 'REFERENCE VELOCITY          =', F9.3,
     4        /' ', 'BEAMWIDTH                   =', F9.3)
  906 FORMAT (/' ', 'NUMBER OF Z SEGMENTS        =', I5,
     2        /' ', 'TOTAL NUMBER OF Z STEPS     =', I5)
  907 FORMAT (/' ', '    I   NZ        DZ   VEL INC   VEL REF'/)
  908 FORMAT ( ' ', 2I5, 3F10.3)
  909 FORMAT (/' ', '    I     OMEGA    FILTER'/)
  910 FORMAT ( ' ', I5, 2F10.3)
  911 FORMAT (/' ', '    I     THETA'/)
  912 FORMAT ( ' ', I5, F10.3)
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWWCOM'
C
      WRITE (LUPRT, 900)
      WRITE (LUPRT, 901) NREC, NTRC, NSMP, IREC1, IREC2, INCREC, NREC2,
     &                   ITRC1, ITRC2, IDT0, IDT
      WRITE (LUPRT, 902) NT, DT, NW, DW, NX, DX, NK, DK, NZ, DZ
      WRITE (LUPRT, 903) IT1, IT2, NTOFF, NTPAD
      WRITE (LUPRT, 904) IW1, IW2
      WRITE (LUPRT, 905) FMIN, FMAX, VELREF, BWIDTH
      WRITE (LUPRT, 906) NZSEG, NZT
C
      WRITE (LUPRT, 907)
      DO 210 I = 1, NZSEG
         VELI = 1.0 / ZSSLOI(I)
         VELR = 1.0 / ZSSLOR(I)
         WRITE (LUPRT, 908) I, IZSNZ(I), ZSDZ(I), VELI, VELR
  210 CONTINUE
C
      WRITE (LUPRT, 909)
      DO 220 I = 1, NW
         WRITE (LUPRT, 910) I, OMEGA(I), FILTR(I)
  220 CONTINUE
C
      WRITE (LUPRT, 911)
      DO 230 I = 1, NREC
         WRITE (LUPRT, 912) I, THETA(I)
  230 CONTINUE
C
      RETURN
      END
C***********************************************************************
C NAME: PWITAB                                                         *
C***********************************************************************
C
C  PURPOSE:
C	PWITAB INITIALIZES THE INTERPOLATION TABLE CXFTAB FOR LATER USE
C	IN THE PHASE SHIFT VECTOR CALCULATION.  IT DETERMINES THE
C	ARGUMENT RANGE AND THE NUMBER OF TABLE ENTRIES AND, THEN, CALLS
C	GCFTLU TO BUILD THE TABLE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWITAB
C
      PARAMETER (NTSCAL = 10000, IFFLG = 2)
C
      INCLUDE 'pwmvzn.h'
C
C-----------------------------------------------------------------------
C
  901 FORMAT (//'***** WARNING: CXFTAB REQUIRED SIZE EXCEEDS MAXTAB'/
     &          '               NTAB   = ', I7/
     &          '               MAXTAB = ', I7/
     &          '               NTAB SET TO MAXTAB'/)
  902 FORMAT (/'***** ERROR: FATAL ERROR REPORTED IN GCFTLU'/)
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWITAB'
C
C     NOTE: XMAX = 2.5 > (pi/2)**2
C
      XMIN = 0.0
      XMAX = 2.5
      NTAB = NTSCAL * XMAX + 1
C
      IF (NTAB .GT. MAXTAB) THEN
	 WRITE (LUPRT, 901) NTAB, MAXTAB
	 NTAB = MAXTAB
      ENDIF
C
C  INITIALIZE CXFTAB
C
      CALL GCFTLU (JUNK, JUNK, 0, NTAB, XMIN, XMAX, IFFLG, CXFTAB, IERR)
C
      IF (IERR .NE. 0) THEN
	 WRITE (LUPRT, 902)
	 CALL PWEXIT (5)
      ENDIF
C
      RETURN
      END
C*********************************************************************
C NAME: PWREAD                                                       *
C*********************************************************************
C
C  PURPOSE:
C      PWREAD READS ALL OF THE TRACES FOR AN ENTIRE RECORD FROM THE
C      INPUT DATA SET.  THE DATA IS TRANSFORMED FROM THE T-X DOMAIN
C      TO THE F-K DOMAIN AND TRANSPOSED TO FORM AN NK BY NW COMPLEX
C      MATRIX DATA.  THE TRACE HEADERS ARE STORED IN THE ARRAY HEADER.
C      NOTE: THE ARRAY RIMAGE IS USED FOR TEMPORARY WORKSPACE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWREAD (JREC)
C
      INCLUDE 'pwmvzn.h'
C
CMAT  INTEGER TRACE(NTRHDR+MAXSMP), THEAD(NTRHDR)
      INTEGER TRACE(NTRHDR+MAXSMP)
      INTEGER * 2 THEAD(NTRHDR)
      INTEGER ITABR(MAXNT+34), ITABC(2*MAXNK+34)
      REAL    TDATA(MAXSMP)
      REAL    RTABR(3*MAXNT/2+13), RTABC(2*MAXNK+13)
      LOGICAL FIRST
C
CMAT  EQUIVALENCE (THEAD, TRACE(1)), (TDATA, TRACE(1+NTRHDR))
      EQUIVALENCE (THEAD, TRACE(1)), (TDATA, TRACE(1+NTRHD2))
C
      DATA FIRST / .TRUE. /
C
      SAVE FIRST, ITABR, RTABR, ITABC, RTABC
C
C-----------------------------------------------------------------------
C
  991 FORMAT (/' ', '***** TAPE READ ERROR - RECORD', I5, ' TRACE', I5,
     &              ' *****')
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWREAD'
C
      MT  = IT2 - IT1 + 1
      NT3 = NT + 3
CMAT  CALL VCLR (RIMAGE, 1, NT3*NK)
      CALL VCLR (RIMAGE, 1, 2*MAXNK1*MAXNZ)
C
C===  IF ITRC1 > 1, SKIP THE FIRST ITRC1-1 TRACES
C
      MTRC = ITRC1 - 1
      IF (MTRC .GT. 0) THEN
         NBYTES = 4 * NSMP + 2 * NTRHDR
         CALL SKIPT (LUINP, MTRC, NBYTES)
      ENDIF
C
C===  READ A TRACE, SAVE THE HEADER AND MOVE SAMPLES IT1:IT2 TO RIMAGE.
C
      JD = 1 + NTOFF
      DO 110 JX = 1, NX
         NBYTES = 0
         CALL RTAPE (LUINP, TRACE, NBYTES)
         IF (NBYTES .EQ. 0) THEN
            JTRC = JX + ITRC1 - 1
            WRITE (LUPRT, 991) JREC, JTRC
            CALL PWEXIT (3)
         ENDIF
C
         IF (THEAD(125) .EQ. 30000) CALL VCLR (TDATA, 1, NSMP)
C
CMAT     CALL VMOV   (THEAD, 1, HEADER(1,JX), 1, NTRHDR)
         CALL VMOV   (THEAD, 1, HEADER(1,JX), 1, NTRHD2)
         CALL VMOV   (TDATA(IT1), 1, RIMAGE(JD), 1, MT)
         JD = JD + NT3
  110 CONTINUE
C
C===  IF ITRC2 < NTRC, SKIP THE LAST NTRC-ITRC2 TRACES
C
      MTRC = NTRC - ITRC2
      IF (MTRC .GT. 0) THEN
         NBYTES = 4 * NSMP + 2 * NTRHDR
         CALL SKIPT (LUINP, MTRC, NBYTES)
      ENDIF
C
C===  IF FIRST, INITIALIZE FFT TABLES
C
      IF (FIRST) THEN
         CALL RFFTM (RIMAGE, NT, 0, 1, ITABR, RTABR, KERR)
         CALL CFFTM (RIMAGE, NT3, NK, 0, 1, ITABC, RTABC, KERR)
         FIRST = .FALSE.
      ENDIF
C
C===  PERFORM 2D FFT
C
      JR = 2 * IW1 - 1
      KR = 1
      DO 120 JX = 1, NX
         CALL RFFTM(RIMAGE(KR), NT, 1, 0, ITABR, RTABR, KERR)
         CALL RFFTSC(RIMAGE(KR), NT, 3, 0)
         KR = KR + NT3
120   CONTINUE
      KR = JR
      DO 130 JW = 1, NW
         CALL CFFTM(RIMAGE(KR),NT3,NK,1,0,ITABC,RTABC,KERR)
         KR = KR + 2
130   CONTINUE
C
C===  SCALE, FILTER, TRANSPOSE, AND PUT IN NATUAL ORDER
C
      R2NKNT = 1.0 / (2.0*FLOAT(NK*NT))
      NK1 = NK / 2 + 1
      NK2 = NK - NK1
      JR1 = JR
      JR2 = JR1 + NK1 * NT3
      JD2 = 1
      JD1 = JD2 + NK2
      DO 210 JW = 1, NW
         SCALE = R2NKNT * FILTR(JW)
         CALL CVSMUL (RIMAGE(JR1), NT3, SCALE, DATA(JD1), 2, NK1)
         CALL CVSMUL (RIMAGE(JR2), NT3, SCALE, DATA(JD2), 2, NK2)
         JR1 = JR1 + 2
         JR2 = JR2 + 2
         JD1 = JD1 + NK
         JD2 = JD2 + NK
  210 CONTINUE
C
      RETURN
      END
C*********************************************************************
C NAME: PWDCIM                                                       *
C*********************************************************************
C
C  PURPOSE:
C      PWDCIM DOWNWARD CONTINUES THE WAVE FIELD AND IMAGES.
C      NOTE: THE ARRAY DEPTH IS USED FOR WORK SPACE.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWDCIM (JREC)
C
      INCLUDE 'pwmvzn.h'
C
      INTEGER KLIM(2)
C
      DATA PI / 3.141592653589793 /
C
C-----------------------------------------------------------------------
C
  901 FORMAT (/'***** ERROR: FATAL ERROR REPORTED BY FVZPK4'/)
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWDCIM'
C
      PIOVR2 = PI / 2.0
      RADIAN = PI / 180.0
      BW     = ABS( BWIDTH ) + 10.0
C
      A  =  THETA(JREC)       * RADIAN
      A1 = (THETA(JREC) - BW) * RADIAN
      A2 = (THETA(JREC) + BW) * RADIAN
C
      IF (A1 .LT. -PIOVR2) A1 = -PIOVR2
      IF (A2 .LT. -PIOVR2) A2 = -PIOVR2
      IF (A1 .GT.  PIOVR2) A1 =  PIOVR2
      IF (A2 .GT.  PIOVR2) A2 =  PIOVR2
C
      P  = SIN( A  ) / VELREF
      P1 = SIN( A1 ) / VELREF
      P2 = SIN( A2 ) / VELREF
C
C  LIMIT DEPTH OF MIGRATION IF ANGLE BECOMES CRITICAL
C
      PABS = ABS( P )
      DO 110 IZSEG = 1, NZSEG
         IF (PABS .GT. ZSSLOI(IZSEG)) THEN
            MZSEG = IZSEG - 1
            IF (MZSEG .GT. 0) GO TO 200
            CALL VCLR (RIMAGE, 1, NZT*NX)
	    GO TO 800
         ENDIF
  110 CONTINUE
      MZSEG = NZSEG
C
C  LOOP OVER FREQUENCY
C
  200 CONTINUE
      JD = 1
      DO 210 JW = 1, NW
         KLIM(1) = NINT( 2.0 * P1 * OMEGA(JW) / DK )
         KLIM(2) = NINT( 2.0 * P2 * OMEGA(JW) / DK )
         CALL FVZPK4 (NK, NK, 1, MZSEG, NZT, NX, NW, JW, OMEGA(JW), AK,
     &                P, KLIM, IZSNZ, ZSDZ, ZSSLOI, ZSSLOR, DATA(JD),
     &                DEPTH, RIMAGE, CXFTAB, IERR)
C
         IF (IERR .NE. 0) THEN
            WRITE (LUPRT, 901)
            CALL PWEXIT (6)
         ENDIF
C
         JD = JD + NK
  210 CONTINUE
C
C  EXIT SUBROUTINE
C
  800 CONTINUE
C     IF (VERBOS) WRITE (LUPRT, *) 'THETA, NZSEG, MZSEG = ',
C    &                              THETA(JREC), NZSEG, MZSEG
      RETURN
      END
C*********************************************************************
C NAME: PWRSMP                                                       *
C*********************************************************************
C
C  PURPOSE:
C      PWRSMP RESAMPLES AND TRANSPOSES THE NX BY NZ IMAGE MATRIX TO FORM
C      NZ BY NX DEPTH SECTION MATRIX.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWRSMP
C
      INCLUDE 'pwmvzn.h'
C
      INTEGER IZ(MAXNZ)
      REAL    Z1(MAXNZT), V1(MAXNZT), Z2(MAXNZ), ZZ(4*MAXNZ)
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWRSMP'
C
C===  CALCULATE Z'S (Z1) CORRESPONDING TO VALUES IN RIMAGE
C
      KZ = 0
      Z  = 0
      DO 120 JZSEG = 1, NZSEG
         DLTZ = ZSDZ(JZSEG)
         DO 110 JZ = 1, IZSNZ(JZSEG)
            KZ = KZ + 1
            Z  = Z  + DLTZ
            Z1(KZ) = Z
  110    CONTINUE
  120 CONTINUE
C
C===  CALCULATE Z'S (Z2) CORRESPONDING TO OUTPUT DEPTH SECTION
C
      Z = 0.0
      DO 130 JZ = 1, NZ
         Z = Z + DZ
         Z2(JZ) = Z
  130 CONTINUE
C
C===  FOR EACH X, MOVE ROW JX OF RIMAGE TO WORK VECTOR V1, RESAMPLE
C===  (FFT) V1 WITH RESULTS STORED IN COLUMN JX OF DEPTH
C
      INIT = 1
      KX   = 1
      DO 210 JX = 1, NX
         CALL VMOV   (RIMAGE(JX), NX, V1, 1, NZT)
         CALL VCINT (Z1, V1, NZT, Z2, DEPTH(KX), NZ, IZ, ZZ, INIT)
         INIT = 0
         KX   = KX + NZ
  210 CONTINUE
C
      RETURN
      END
C*********************************************************************
C NAME: PWWRIT                                                       *
C*********************************************************************
C
C  PURPOSE:
C      PWWRIT WRITES ALL OF THE TRACES FOR AN ENTIRE RECORD (I.E., A
C      DEPTH SECTION) TO THE OUTPUT DATA SET.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWWRIT (KREC)
C
      INCLUDE 'pwmvzn.h'
C
CMAT  INTEGER TRACE(NTRHDR+MAXNZ), THEAD(NTRHDR)
      INTEGER TRACE(NTRHDR+MAXNZ)
      INTEGER * 2 THEAD(NTRHDR)
      REAL    TDATA(MAXNZ)
C
CMAT  EQUIVALENCE (THEAD, TRACE(1)), (TDATA, TRACE(1+NTRHDR))
      EQUIVALENCE (THEAD, TRACE(1)), (TDATA, TRACE(1+NTRHD2))
C
C-----------------------------------------------------------------------
C
  991 FORMAT (/' ', '***** TAPE WRITE ERROR - RECORD', I5, ' TRACE', I5,
     &              ' *****')
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWWRIT'
C
CMAT  NBYTES = 8 * (NZ + NTRHDR)
      NBYTES = 4 * NZ + 2 * NTRHDR
C
      KX = 1
      DO 110 JX = 1, NX
CMAT     CALL VMOV   (HEADER(1,JX), 1, THEAD, 1, NTRHDR)
         CALL VMOV   (HEADER(1,JX), 1, THEAD, 1, NTRHD2)
         CALL VMOV   (DEPTH(KX), 1, TDATA, 1, NZ)
         THEAD(106) = KREC
         THEAD(107) = JX
C
         CALL WRTAPE (LUOUT, TRACE, NBYTES)
C
         IF (NBYTES .EQ. 0) THEN
            WRITE (LUPRT, 991) KREC, JX
            CALL PWEXIT (4)
         ENDIF
C
         KX = KX + NZ
  110 CONTINUE
C
      RETURN
      END
C*********************************************************************
C NAME: PWEXIT                                                       *
C*********************************************************************
C
C  PURPOSE:
C      PWEXIT PRINTS COMPLETION MESSAGE, CLOSES THE SIS FILES AND EXITS
C      THE PROGRAM.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PWEXIT (ICODE)
C
      INCLUDE 'pwmvzn.h'
      PARAMETER (NSUBR = 8)
C
      CHARACTER*6 SUBR(NSUBR)
C
      DATA SUBR / 'PWPARM', 'PWRCRD', 'PWREAD', 'PWWRIT', 'PWITAB',
     &            'PWDCIM', 'PWRDVM', 'PWVMOD' /
C
C-----------------------------------------------------------------------
C
  901 FORMAT (/' ', '***** NORMAL COMPLETION *****')
  902 FORMAT (/' ', '***** ABNORMAL COMPLETION CODE =', I3, ' *****')
  903 FORMAT (/' ', '***** FATAL ERROR IN SUBROUTINE ', A6, ' *****')
C
C-----------------------------------------------------------------------
C
      IF (VERBOS) WRITE (LUPRT, *) ' ENTER SUBROUTINE PWEXIT'
C
      IF      (ICODE .EQ. 0) THEN
         WRITE (LUPRT, 901)
      ELSE IF (ICODE .GT. 0) THEN
         WRITE (LUPRT, 902) ICODE
         IF (ICODE .LE. NSUBR) WRITE (LUPRT, 903) SUBR(ICODE)
      ENDIF
C
      IF (ICODE .GE. 0) THEN
         CALL LBCLOS (LUINP)
         CALL LBCLOS (LUOUT)
      ENDIF
C
      IF (ICODE .EQ. 0) THEN
         STOP 0
      ELSE
         STOP 1
      ENDIF
C
      END
C********************************************************************C
C NAME: RDANGL  READ ANGLES                     REV 1.0     AUG 88   C
C********************************************************************C
C
C  PURPOSE:
C       RDANGL reads angles from a file.
C
C  CALLING FORMAT:
C       SUBROUTINE RDANGL (LUINP, LUPRT, LUERR, INCNT, NANG, THETA,
C      &                   IERR)
C
C  PARAMETERS:
C       LUINP   Integer input scalar
C               Logical unit number of input file.
C
C       LUPRT   Integer input scalar
C               Logical unit number of print file.  If LUPRT <=  0,
C               then print output is suppressed.
C
C       LUERR   Integer input scalar
C               Logical unit number of error file.  If LUERR <= 0,
C               then output error messages are suppressed.
C
C       INCNT   Integer input scalar
C               Input angle count.  If INCNT <= 0, then input continues
C               until an end-of-file.
C
C       NANG    Integer output scalar
C               Total number of angles (SUM[ NANGLE(i), i = 1, nrec ]).
C
C       THETA   Real output vector of length NANG
C               Angle vector
C
C       IERR    Integer output scalar
C               Completion code:
C                  IERR = 0 - Normal completion
C                  IERR = 1 - Unexpected end of file
C                  IERR = 2 - Data conversion error
C                  IERR = 3 - Invalid data value
C                  IERR = 4 - Too many input values
C
C  DESCRIPTION:
C       RDANGL reads a angles from a file that contains one record per
C       set of angles.  Each record contains the following:
C
C          Columns   Format   Contents
C
C            1-10     I10     NANGLE(I) - Number of angles in set
C           11-20     F10.0   ANG(I)    - Starting angle in degrees
C           21-30     F10.0   DANG(I)   - Delta angle in degrees
C
C       As each record is read, the values of the angles are computed
C       and inserted sequentially in the output vector, THETA.
C
C       If LUPRT > 0, then the input data is written to logical
C       unit LUPRT in tabular form with column headers.
C
C       The routine continues to read angle set records until either
C       normal completion occurs or an error is detected.  Normal
C       completion occurs if INCNT > 0 and INCNT angles have been
C       inserted or INCNT <= 0 and an end-of-file is encountered after
C       at least one record has been read.  On return, the value of
C       NANG will equal the number of angles successfully inserted in
C       THETA (even if an error has been detected).
C
C  ERROR CONDITIONS:
C       Any of the following conditions will cause an error condition:
C
C       1. An unexpected end-of-file occured; i.e., an end-of-file was
C          encountered and either INCNT > 0 or no records had been read.
C
C       2. A error occured while attempting to convert the record under
C          (I10, 2F10.0) format.
C
C       3. A data value is invalid; i.e., NANGLE(I) <= 0.
C
C       4. INCNT > 0 and inserting a set of angles would cause NANG
C          to exceed INCNT.
C
C       If an error condition occurs, the completion code is set to
C       the appropriate value and the routine is aborted.  In
C       addition, if LUERR > 0, then an error message is written
C       to logical unit LUERR.
C
C  EXAMPLE:
C
C       Inputs:
C
C         LUINP = 5
C         LUPRT = 6
C         LUERR = 6
C         INCNT = 9
C
C       Input file:
C
C                         111111111122222222223
C       Column   123456789012345678901234567890
C
C                         3        -8         8
C                         2        10         5
C                         4         1        -2
C                (more records may follow)
C
C       Outputs:
C
C         NANG = 9
C         IERR = 0
C
C         I    THETA(I)
C
C         1     -8.0
C         2      0.0
C         3      8.0
C         4     10.0
C         5     15.0
C         6      1.0
C         7     -1.0
C         8     -3.0
C         9     -5.0
C
C  SUBPROGRAMS CALLED:
C       None
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       Original                Aug 88          R.D. Coleman, QTC
C
C---------------------------------------------------------------------
C
      SUBROUTINE RDANGL (LUINP, LUPRT, LUERR, INCNT, NANG, THETA, IERR)
C
      CHARACTER*80 BUF
      INTEGER      LUINP, LUPRT, LUERR, INCNT, NANG
      REAL         THETA(*)
C
C-----------------------------------------------------------------------
C
  900 FORMAT (A80)
  901 FORMAT (/' ', '  # ANGLES     START INCREMENT')
  902 FORMAT (      I10, 2F10.0)
  903 FORMAT ( ' ', I10, 2F10.1)
  911 FORMAT (/' ', '***** RDANGL ERROR - UNEXPECTED END-OF-FILE')
  912 FORMAT (/' ', '***** RDANGL ERROR - DATA CONVERSION ERROR'/
     &         ' ', '      INPUT CARD: ', A80)
  913 FORMAT (/' ', '***** RDANGL ERROR - INVALID DATA VALUE')
  914 FORMAT (/' ', '***** RDANGL ERROR - TOO MANY INPUT VALUES')
C
C-----------------------------------------------------------------------
C
      WRITE (LUPRT, 901)
C
      LPCNT  = INCNT
      NANG   = 0
  100 CONTINUE
         READ  (LUINP, 900, END=810) BUF
         READ  (BUF  , 902, ERR=820) NANGLE, ANG, DANG
         WRITE (LUPRT, 903         ) NANGLE, ANG, DANG
C
         IF (NANGLE .LE. 0) GO TO 830
         IF (LPCNT  .GT. 0 .AND. NANGLE .GT. LPCNT) GO TO 840
C
         DO 110 I = 1, NANGLE
            NANG = NANG + 1
            THETA(NANG) = ANG
            ANG = ANG + DANG
  110    CONTINUE
C
         LPCNT = LPCNT - NANGLE
         IF (LPCNT .NE. 0) GO TO 100
C
C  SUBROUTINE EXITS
C
      IERR = 0
      RETURN
C
  810 CONTINUE
      IF (INCNT .LE. 0 .AND. NANG .GT. 0) THEN
         IERR = 0
         RETURN
      ELSE
         IF (LUERR .GT. 0) WRITE (LUERR, 911)
         IERR = 1
         RETURN
      ENDIF
C
  820 CONTINUE
      IF (LUERR .GT. 0) WRITE (LUERR, 912) BUF
      IERR = 2
      RETURN
C
  830 CONTINUE
      IF (LUERR .GT. 0) WRITE (LUERR, 913)
      IERR = 3
      RETURN
C
  840 CONTINUE
      IF (LUERR .GT. 0) WRITE (LUERR, 914)
      IERR = 4
      RETURN
C
      END
C********************************************************************C
C NAME: FVZPK4  IMAGE V OF Z PLANE - K LIMITS   REV 1.0     NOV 90   C
C********************************************************************C
C
C  PURPOSE:
C       IMAGES A K-Z PLANE WHERE VELOCITY IS A FUNCTION OF Z ONLY.
C       DOWNWARD CONTINUATION IS PERFORMED FOR A LIMITED RANGE OF
C       K VALUES WHICH ARE A FUNCTION OF THE ANGLE, P.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       ORIGINAL                APR 90          R.D. COLEMAN, QTC
C               MODIFICATION OF XVZPK3 WITH THE DATA IN PSI AND
C               PLANE IN NATURAL ORDER AS OPPOSED TO FFT ORDER.
C               ALSO THE ORDER OF THE LOOPS HAS BEEN REVERSED AND
C               NEW K LIMITS ARE CALCULATED IN THE Z LOOP
C               IN ORDER TO ELIMINATE UNNECESSARY (ZERO) ELEMENTS
C               IN THE PHASE SHIFT VECTOR.
C
C                                DEC 90          L.TARVESTAD, QTC
C               CHANGE WORKSPACE ALLOCATION FOR FRCFT2.
C
C  CALLING FORMAT:
C       CALL FVZPK4 (MK, NK, NP, NZS, NZT, NX, NW, JW, OMEGA, AK, P,
C      &             KLIM, IZSNZ, ZSDZ, ZSSLOI, ZSSLOR, PSI, WORK,
C      &             PLANE, CXFTAB, IERR)
C
C  PARAMETERS:
C
C       MK      INTEGER INPUT SCALAR
C               ROW DIMENSION OF MATRIX PSI.  MK MUST BE >= NK.
C
C       NK      INTEGER INPUT SCALAR
C               NUMBER OF K'S (WAVE NUMBERS).  NK MUST BE AN EVEN
C               INTEGER OF THE FORM 2**I * 3**J * 5**K WHERE I, J,
C               AND K ARE NON-NEGATIVE INTEGERS.
C
C       NP      INTEGER INPUT SCALAR
C               NUMBER OF P'S (ANGLES).
C
C       NZS     INTEGER INPUT SCALAR
C               NUMBER OF Z SEGMENTS.
C
C       NZT     INTEGER INPUT SCALAR
C               TOTAL NUMBER OF Z'S (NZT = SUM(IZSNZ(I), I = 1, NZS).
C
C       NX      INTEGER INPUT SCALAR
C               NUMBER OF X'S.  NX MUST BE <= NK.
C
C       NW      INTEGER INPUT SCALAR
C               NUMBER OF W'S (FREQUENCIES).
C
C       JW      INTEGER INPUT SCALAR
C               FREQUENCY INDEX.
C
C       OMEGA   REAL INPUT SCALAR
C               ANGULAR FREQUENCY.
C
C       AK      REAL INPUT VECTOR OF LENGTH NK
C               K (WAVE NUMBER) VECTOR.
C
C       P       REAL INPUT VECTOR OF LENGTH NP
C               P (ANGLE) VECTOR.
C
C       KLIM    INTEGER INPUT ARRAY OF DIMENSION 2 X NP
C               KLIM(1,J) AND KLIM(2,J) CONTAIN THE LOWER AND UPPER
C               K LIMITS FOR ANGLE P(J), RESPECTIVELY.
C
C       IZSNZ   INTEGER INPUT VECTOR OF LENGTH NZS
C               IZSNZ(I) = NUMBER OF Z'S FOR THE I-TH Z SEGMENT.
C
C       ZSDZ    REAL INPUT VECTOR OF LENGTH NZS
C               ZSDZ(I) = DELTA Z FOR THE I-TH Z SEGMENT.
C
C       ZSSLOI  REAL INPUT VECTOR OF LENGTH NZS
C               ZSSLOI(I) = INDICENT  SLOWNESS FOR THE I-TH Z SEGMENT.
C
C       ZSSLOR  REAL INPUT VECTOR OF LENGTH NZS
C               ZSSLOR(I) = REFLECTED SLOWNESS FOR THE I-TH Z SEGMENT.
C
C       PSI     COMPLEX INPUT MATRIX OF DIMENSION MK X NP
C               WAVE FIELD AT THE TOP (Z=0).  ON OUTPUT PSI CONTAINS
C               THE WAVE FIELD AT THE BOTTOM.  THE DATA IS CONTAINED
C               IN THE FIRST NK ROWS OF PSI.  THE REMAINING ROWS (IF
C               ANY) ARE NEVER ACCESSED.  PSI IS IN NATURAL ORDER.
C
C       WORK    REAL SCRATCH VECTOR OF LENGTH NWRK
C               NWRK = MAX( 6*NK+47, 7*NK, 7*NZS )
C
C       PLANE   REAL INPUT/OUTPUT MATRIX OF DIMENSION (NK+1) * NZT * 2
C               IF JW < NW, THEN PLANE CONTAINS COMPLEX IMAGE IN THE
C               K-Z PLANE WHICH MUST BE INPUT ON THE NEXT CALL.
C               IF JW = NW, THEN PLANE CONTAINS THE REAL IMAGE IN THE
C               X-Z PLANE IN THE FIRST NX*NZT WORDS.  THE COMPLEX DATA
C               IS IN NATURAL ORDER.
C
C	CXFTAB	REAL INPUT VECTOR OF IMPLIED LENGTH
C		TABLE FOR FCXF2D AS GENERATED BY GCFTLU.
C
C	IERR	INTEGER OUTPUT SCALAR
C		THE LOGICAL OR OF ALL THE FCXF2D COMPLETION CODES.
C		IF NOT ZERO, THEN THE RANGE OF CXFTAB IS INSUFFICIENT.
C
C  DESCRIPTION:
C       THE ACTION OF FVZPK4 IS DESCRIBED BY THE FOLLOWING PSEUDO
C       CODE:
C
C       IF JW = 1, THEN PLANE(1:NK,1:NZT) = 0.0
C
C       IZS = 0
C       DO JZS = 1, NZS
C          DO JP = 1, NP
C             COMPUTE K1 AND K2 (ACTUAL K INDEX RANGE) FOR P(JP)
C             COMPUTE PHAS(K1:K2) (PHASE-SHIFT OPERATOR) FOR P(JP)
C             DO JZ = 1, IZSNZ(JZS)
C                IZ = IZS + JZ
C                PSI(K1:K2,JP) = PHAS(K1:K2) * PSI(K1:K2,JP)
C                PLANE(K1:K2,IZ) = PLANE(K1:K2,IZ) + PSI(K1:K2,JP)
C             ENDDO
C          ENDDO
C          IZS = IZS + IZSNZ(JZS)
C       ENDDO
C
C       IF (JW = NW) THEN
C          CONVERT PLANE TO FFT ORDER
C          CALL FRCFT2 (PLANE, NK, NX, NZT, 1, WORK, IERR)
C       ENDIF
C
C  SUBPROGRAMS CALLED:
C       FCXF2D, FVZDCI, FRCFT2
C
C  ERROR CONDITIONS:
C       IF AN ERROR OCCURS IN SUBROUTINE FCXF2D, THEN THE APPROPRIATE
C	ERROR CODE IS SET AND PROCESSING CONTINUES.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE FVZPK4 (MK, NK, NP, NZS, NZT, NX, NW, JW, OMEGA, AK, P,
     &                   KLIM, IZSNZ, ZSDZ, ZSSLOI, ZSSLOR, PSI, WORK,
     &                   PLANE, CXFTAB, IERR)
C
      PARAMETER (BI = 1.0, BR = 1.0, ICXFLG = 1)
C
      INTEGER MK, NK, NP, NZS, NZT, NX, NW, JW, KLIM(2,NP), IZSNZ(NZS),
     &        IERR, IORERR(0:3,0:3)
      REAL    OMEGA, AK(NK), P(NP), ZSDZ(NZS), ZSSLOI(NZS), ZSSLOR(NZS),
     &        WORK(*), PLANE(*), CXFTAB(*)
      COMPLEX PSI(MK,NP), CA
C
      DATA IORERR / 0, 1, 2, 3, 1, 1, 3, 3, 2, 3, 2, 3, 3, 3, 3, 3 /
C
C-----------------------------------------------------------------------
C
C     ALLOCATE WORK SPACE
C
      INC = MAX0( NK, NZS )
      I1  = 1
      I2  = I1 + 2*NZS
      I3  = I2 + 2*NK
      I4  = I3 + INC
      I5  = I4 + INC
C
C     IF JW = 1, CLEAR PLANE
C
      NKP1 = NK + 1
      IF (JW .EQ. 1) CALL VCLR (PLANE, 1, 2*NKP1*NZT)
C
C     INITIALIZE SOME STUFF
C
      IERR  = 0
      IPR   = 1
      IPI   = IPR + NKP1 * NZT
      KBIAS = (NK + 1) / 2
      DK    = AK(1+KBIAS)
C
C     LOOP OVER ANGLE
C
      DO 130 JP = 1, NP
	 IK1 = KLIM(1,JP) + KBIAS
	 IK2 = KLIM(2,JP) + KBIAS
C
	 IF (IK1 .GT. NK .OR. IK2 .LT. 1 .OR. IK1 .GT. IK2) THEN
            CALL VCLR (PSI(1,JP), 1, 2*NK)
	    GO TO 130
	 ENDIF

         IF (IK1 .LT.  1) IK1 = 1
         IF (IK2 .GT. NK) IK2 = NK
	 LK = IK2 - IK1 + 1
C
C        CLEAR PSI OUTSIDE K LIMITS
C
         IF (IK1 .GT.  1) CALL VCLR (PSI(1    ,JP), 1, 2*(IK1-1))
         IF (IK2 .LT. NK) CALL VCLR (PSI(IK2+1,JP), 1, 2*(NK-IK2))
C
         CALL FCXF2D (0.0, ZSSLOI, OMEGA,  P(JP), ZSDZ, BI, WORK(I1),
     &                  1,    NZS,     1,      1, ICXFLG,
     &                WORK(I3), WORK(I4), WORK(I5), CXFTAB, JERR)
	 IERR = IORERR( IERR, JERR )
C
C        LOOP OVER Z SEGMENTS
C
         JPR = IPR - 1
         JPI = IPI - 1
	 J1  = I1
         DO 110 JZS = 1, NZS
	    CA = CMPLX( WORK(J1), WORK(J1+1) )
	    IF (CA .EQ. (0.0,0.0)) THEN
               CALL VCLR (PSI(IK1,JP), 1, 2*(IK2-IK1+1))
	       GO TO 120
	    ENDIF
C
            NZ   = IZSNZ(JZS)
            DZ   = ZSDZ(JZS)
            SLOI = ZSSLOI(JZS)
            SLOR = ZSSLOR(JZS)
C
C           COMPUTE K LIMITS
C
	    AKMIN = OMEGA * (P(JP) - SLOR)
	    AKMAX = OMEGA * (P(JP) + SLOR)
            JK1   = IFIX( AKMIN / DK ) + KBIAS
            JK2   = IFIX( AKMAX / DK ) + KBIAS
C
C           CHECK LOWER K LIMIT
C
	    IF (JK1 .GT. IK1) THEN
	       IF (JK1 .LE. IK2) THEN
                  CALL VCLR (PSI(IK1,JP), 1, 2*(JK1-IK1))
	          IK1 = JK1
	          LK  = IK2 - IK1 + 1
	       ELSE
                  CALL VCLR (PSI(IK1,JP), 1, 2*(IK2-IK1+1))
		  GO TO 120
	       ENDIF
	    ENDIF
C
C           CHECK UPPER K LIMIT
C
	    IF (JK2 .LT. IK2) THEN
	       IF (JK2 .GE. IK1) THEN
                  CALL VCLR (PSI(JK2+1,JP), 1, 2*(IK2-JK2))
	          IK2 = JK2
	          LK  = IK2 - IK1 + 1
	       ELSE
                  CALL VCLR (PSI(IK1,JP), 1, 2*(IK2-IK1+1))
		  GO TO 120
	       ENDIF
	    ENDIF
C
C           COMPUTE PHASE-SHIFT OPERATOR THEN DOWNWARD CONTINUE AND
C           IMAGE FOR THE FIRST K INDEX RANGE FOR P(IP)
C
            CALL FCXF2D (AK(IK1), SLOR, OMEGA, P(JP), DZ, BR, WORK(I2),
     &                        LK,    1,     1,     1, -1,
     &                   WORK(I3), WORK(I4), WORK(I5), CXFTAB, JERR)
	    IERR = IORERR( IERR, JERR )
            CALL FVZDCI (CA, WORK(I2), PSI(IK1,JP), 
     &                   PLANE(IK1+JPR), PLANE(IK1+JPI), NKP1, LK, NZ)
C
            JPR = JPR + NKP1 * NZ
            JPI = JPI + NKP1 * NZ
	    J1  = J1 + 2
  110    CONTINUE
  120    CONTINUE
  130 CONTINUE
C
C  IF LAST FREQUENCY, THEN EXTRACT REAL IMAGE, ELSE RETURN
C
      IF (JW .NE. NW) RETURN
C
C  PUT DATA IN FFT ORDER
C
      NK1 = NK / 2 + 1
      NK2 = NK - NK1
      JPR = IPR
      JPI = IPI
      DO 210 JZ = 1, NZT
         CALL VMOV (PLANE(JPR    ), 1, WORK          , 1, NK2)
         CALL VMOV (PLANE(JPR+NK2), 1, PLANE(JPR)    , 1, NK1)
         CALL VMOV (WORK          , 1, PLANE(JPR+NK1), 1, NK2)
         CALL VMOV (PLANE(JPI    ), 1, WORK          , 1, NK2)
         CALL VMOV (PLANE(JPI+NK2), 1, PLANE(JPI)    , 1, NK1)
         CALL VMOV (WORK          , 1, PLANE(JPI+NK1), 1, NK2)
         JPR = JPR + NKP1
         JPI = JPI + NKP1
  210 CONTINUE
C
C  FFT AND KEEP THE REAL COMPONENTS
C
      I1 = 4 * NK + 48
      CALL FRCFT2 (PLANE(IPR), PLANE(IPI), NKP1, NK, NX, NZT, 1,
     &             WORK, WORK(I1), JERR)
C
      RETURN
      END
C********************************************************************C  
C NAME: FCXF2D  2D CMPLX EXP. FUNCTION (TABLE)  REV 1.0     DEC 90   C  
C********************************************************************C  
C
C  PURPOSE:
C       COMPUTES THE 2D COMPLEX EXPONENTIAL PHASE-SHIFT FUNCTION BY
C       MEANS OF A PACKED TABLE LOOKUP.  THIS ROUTINE DIFFERS IN TWO
C       WAYS FROM XCXFTP: (1) ITS FOR 2D ONLY; I.E., THERE IS ONLY
C       ONE K VECTOR ANS ONE P VECTOR, AND (2) IT IS FOR V OF Z ONLY;
C       I.E., DZ IS ASSUMED TO BE A FUNCTION OF SLOWNESS: THUS, THE
C       VECTORS S AND DZ MUST BE THE SAME LENGTH AND ARE INDEXED BY
C       THE SAME INDEX.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       ORIGINAL                DEC 90          R.D. COLEMAN, QTC
C
C  CALLING FORMAT:
C       CALL FCXF2D ( K,  S,  W,  P, DZ, B, C,
C      &             NK, NS, NW, NP, ISW, W1, W2, IW3, TABLE, IERR)
C
C  PARAMETERS:
C       K       REAL INPUT VECTOR OF LENGTH NK
C               X WAVE NUMBERS
C
C       S       REAL INPUT VECTOR OF LENGTH NS
C               SLOWNESS
C
C       W       REAL INPUT VECTOR OF LENGTH NW
C               ANGULAR FREQUENCY
C
C       P       REAL INPUT VECTOR OF LENGTH NP
C               X ANGLES
C
C       DZ      REAL INPUT VECTOR OF LENGTH NS
C               DELTA Z
C
C       B       REAL INPUT SCALAR
C               MULTIPLIER (NORMALLY = 1.0 OR -1.0 TO SELECT SIGN
C               OF THE EXPONENT).
C
C       C       COMPLEX OUTPUT VECTOR OF LENGTH NC
C               PHASE-SHIFT FUNCTION.
C               NC = NK * NS * NW * NP
C
C       NK      INTEGER INPUT SCALAR
C               ELEMENT COUNT FOR VECTOR K (MUST BE >= 1).
C
C       NS      INTEGER INPUT SCALAR
C               ELEMENT COUNT FOR VECTORS S AND DZ (MUST BE >= 1).
C
C       NW      INTEGER INPUT SCALAR
C               ELEMENT COUNT FOR VECTOR W (MUST BE >= 1).
C
C       NP      INTEGER INPUT SCALAR
C               ELEMENT COUNT FOR VECTOR P (MUST BE >= 1).
C
C       ISW     INTEGER INPUT SCALAR
C               FUNCTION SELECTOR SWITCH.
C               IF ISW = 0 & Y < 0.0, THEN C(I) = EXP( -B * SQRT(-Y) )
C               IF ISW > 0 & Y < 0.0, THEN C(I) = 0.0
C               IF ISW < 0 & Y < 0.0, THEN C(I) = 1.0
C
C       W1      REAL SCRATCH VECTOR OF LENGTH NC
C
C       W2      REAL SCRATCH VECTOR OF LENGTH NC
C
C       IW3     INTEGER SCRATCH VECTOR OF LENGTH NC
C
C       TABLE   REAL INPUT VECTOR OF IMPLIED LENGTH
C               PACKED, COMPLEX FUNCTION TABLE AS GENERATED BY GCFTLU
C               WITH IFFLG = 2 AND XMIN = 0.0.
C
C       IERR    INTEGER OUTPUT SCALAR
C               IERR = IERRHI + IERRLO, WHERE
C                  IERRLO = 1, IF TABLE UNDERFLOW OCCURRED AND THE
C                              TABLE XMIN != 0.0.
C                           0, OTHERWISE
C                  IERRHI = 2, IF TABLE OVERFLOW OCCURRED.
C                         = 0, OTHERWISE
C
C  DESCRIPTION:
C       THE COMPUTATION OF THE COMPLEX OUTPUT VECTOR C FROM THE REAL
C       INPUT VECTORS K, S, W, P, AND DZ AND THE SCALAR B IS
C       IS DESCRIBED BY THE FOLLOWING FORTRAN EQUIVALENT.
C
C       REAL    K(*), S(*), W(*), P(*), DZ(*), B, X, Y
C       COMPLEX C(*)
C
C       I = 0
C       DO 10 IP = 1, NP
C       DO 10 IW = 1, NW
C       DO 10 IS = 1, NS
C       DO 10 IK = 1, NK
C          I = I + 1
C
C          Y = ( ( W(IW) * S(IS) ) ** 2
C      &       - ( K(IK) - W(IW) * P(IP) ) ** 2 ) * DZ(IS)**2
C
C          X = B * SQRT( ABS( Y ) )
C
C          IF (Y .GE. 0.0) THEN
C             C(I) = ( COS(X), SIN(X) )
C          ELSE IF (ISW .EQ. 0) THEN
C             C(I) = (EXP(-X), 0.0 )
C          ELSE IF (ISW .GT. 0) THEN
C             C(I) = ( 0.0, 0.0 )
C          ELSE
C             C(I) = ( 1.0, 0.0 )
C          ENDIF
C
C    10 CONTINUE
C       RETURN
C       END
C
C  SUBPROGRAMS CALLED:
C       FORTRAN INTRINSICS: SQRT, EXP
C       OTHER             : FCFTLU, WHENFLT, GATHER, SCATTER
C
C  ERROR CONDITIONS:
C       IF A TABLE LOOKUP ERROR OCCURS, THEN THE APPROPRIATE ERROR
C       CODE IS SET (SEE ABOVE) AND PROCESSING CONTINUES.
C
C---------------------------------------------------------------------
C
      SUBROUTINE FCXF2D ( K,  S,  W,  P, DZ, B, C,
     &                   NK, NS, NW, NP, ISW, W1, W2, IW3, TABLE, IERR)
C
      INTEGER NK, NS, NW, NP, N, ISW, IW3(*), IERR
      REAL    K(*), S(*), W(*), DZ(*), P(*),
     &        B, C(*), W1(*), W2(*), TABLE(*), KMWP, KMWPSQ
C
C---------------------------------------------------------------------
C
      N = NK * NS * NW * NP
      IF (NK .EQ. 1) GO TO 200
C
      I = 0
      DO 140 IP = 1, NP
         XP = P(IP)
         DO 130 IW = 1, NW
            XW = W(IW)
            WP = XW * XP
            DO 120 IS = 1, NS
	       WS    = XW * S(IS)
	       BDZ   = B * DZ(IS)
               WSSQ  = WS  * WS
               BDZSQ = BDZ * BDZ
               DO 110 IK = 1, NK
                  I     = I + 1
		  KMWP  = K(IK) - WP
                  W1(I) = (WSSQ - KMWP * KMWP) * BDZSQ
  110          CONTINUE
  120       CONTINUE
  130    CONTINUE
  140 CONTINUE
      GO TO 700
C
  200 CONTINUE
      I = 0
      DO 240 IP = 1, NP
         XP = P(IP)
         DO 230 IW = 1, NW
            XW     = W(IW)
	    KMWP   = K(1) - XW * XP
	    KMWPSQ = KMWP * KMWP
            DO 220 IS = 1, NS
               I     = I + 1
	       WS    = XW * S(IS)
	       BDZ   = B * DZ(IS)
               WSSQ  = WS  * WS
               BDZSQ = BDZ * BDZ
               W1(I) = (WSSQ - KMWPSQ) * BDZSQ
  220       CONTINUE
  230    CONTINUE
  240 CONTINUE
C
  700 CONTINUE
      IF (B .GE. 0.0) THEN
	 ICFLG =  0
	 SGN   = -1.0
      ELSE
	 ICFLG =  1
	 SGN   =  1.0
      ENDIF
C
      CALL FCFTLU (TABLE, W1, 1, C(1), C(2), 2, N, ICFLG, IERR)
C
      IERR0 = MOD( IERR, 2 )
      IF (IERR0 .EQ. 0 .OR. TABLE(4) .NE. 0.0) RETURN
C
C  RESET BIT 0 OF IERR
C
      IERR = IERR - IERR0
C
      IF (ISW .LT. 0) RETURN
C
C  GET INDICES OF ARGUMENTS LESS THAN ZERO
C
      CALL WHENFLT (N, W1, 1, 0.0, IW3, M)
C
C  CALCULATE NEW REAL COMPONENTS FOR THOSE ELEMENTS
C  (NOTE: THE IMAGINARY COMPONENTS WILL ALREADY BE ZERO)
C
      IF (ISW .EQ. 0) THEN
         CALL GATHER  (M, W2, W1, IW3)
         DO 710 I = 1, M
            W2(I) = EXP( SGN * SQRT( - W2(I) ) )
  710    CONTINUE
      ELSE
	 DO 720 I = 1, M
	    W2(I) = 0.0
  720    CONTINUE
      ENDIF
C
C  CALCULATE THE INDICES OF THE REAL COMPONENTS
C
      DO 730 I = 1, M
	 IW3(I) = IW3(I) + IW3(I) - 1
  730 CONTINUE
C
C  SCATTER THE NEW REAL COMPONENTS
C
      CALL SCATTER (M, C, IW3, W2)
C
      RETURN
      END
C********************************************************************C
C NAME: FVZDCI  V OF Z DOWNWARD CONT. & IMAGE   REV 2.0     SEP 87   C
C********************************************************************C
C
C  PURPOSE:
C       DOWNWARD CONTINUES AND IMAGES IN THE K-Z PLANE WHERE VELOCITY
C       IS A FUNCTION OF Z ONLY.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       ORIGINAL                JUN 87          R.D. COLEMAN, QTC
C       REVISION 2.0            SEP 87          R.D. COLEMAN, QTC
C                               SPLIT PLANE INTO COMPLEX COMPONENTS
C
C  CALLING FORMAT:
C       CALL FVZDCI (CA, PHAS, PSI, PLNR, PLNI, MK, NK, NZ)
C
C  PARAMETERS:
C       CA      COMPLEX INPUT SCALAR
C               PHASE-SHIFT SCALE FACTOR.
C
C       PHAS    COMPLEX INPUT VECTOR OF LENGTH NK
C               PHASE-SHIFT OPERATOR.  CONTENTS OF PHAS ARE MODIFIED
C               BY THIS ROUTINE.
C
C       PSI     COMPLEX INPUT/OUTPUT VECTOR OF LENGTH NK
C               ON INPUT PSI CONTAINS THE WAVE FIELD AT THE TOP (Z=0).
C               ON OUTPUT PSI CONTAINS THE WAVE FIELD AT THE BOTTOM.
C
C       PLNR    REAL INPUT/OUTPUT MATRIX OF DIMENSION MK X NZ
C               PLANE CONTAINS THE REAL COMPONENTS OF THE IMAGE IN
C               THE K-Z PLANE IN THE FIRST NK ROWS.
C
C       PLNI    REAL INPUT/OUTPUT MATRIX OF DIMENSION MK X NZ
C               PLANE CONTAINS THE IMAGINARY COMPONENTS OF THE IMAGE IN
C               THE K-Z PLANE IN THE FIRST NK ROWS.
C
C       MK      INTEGER INPUT SCALAR
C               TOTAL NUMBER OF K'S (WAVE NUMBERS); I.E., ROW DIMENSION
C               OF PLNR AND PLNI.  MK MUST BE >= NK.
C
C       NK      INTEGER INPUT SCALAR
C               NUMBER OF K'S (WAVE NUMBERS).
C
C       NZ      INTEGER INPUT SCALAR
C               NUMBER OF Z'S.
C
C
C  DESCRIPTION:
C       THE ACTION OF FVZDCI IS DESCRIBED BY THE FOLLOWING PSEUDO
C       CODE:
C
C       PHAS(1:NK) = CA * PHAS(1:NK)
C       DO J = 1, NZ
C          PSI(1:NK)    = PHAS(1:NK) * PSI(1:NK)
C          PLNR(1:NK,J) = PLNR(1:NK,J) + REAL( PSI(1:NK) )
C          PLNI(1:NK,J) = PLNI(1:NK,J) + IMAG( PSI(1:NK) )
C       ENDDO
C
C  SUBPROGRAMS CALLED:
C       NONE
C
C  ERROR CONDITIONS:
C       NONE
C
C-----------------------------------------------------------------------
C
      SUBROUTINE FVZDCI (CA, PHAS, PSI, PLNR, PLNI, MK, NK, NZ)
C
      COMPLEX CA, PHAS(*), PSI(*)
      REAL    PLNR(*), PLNI(*)
C
      DO 100 I = 1, NK
         PHAS(I) = CA * PHAS(I)
  100 CONTINUE
C
      JZ = 1
      DO 130 J = 1, NZ
         IZ = JZ
         DO 120 I = 1, NK
            PSI(I)   = PHAS(I) * PSI(I)
            PLNR(IZ) = PLNR(IZ) + REAL ( PSI(I) )
            PLNI(IZ) = PLNI(IZ) + AIMAG( PSI(I) )
            IZ = IZ + 1
  120    CONTINUE
         JZ = JZ + MK
  130 CONTINUE
C
      RETURN
      END
C********************************************************************C
C NAME: FRCFT2  REALS OF COMPLEX FFT (LOOPING)  REV 2.0     JUL 88   C
C********************************************************************C
C
C  PURPOSE:
C       COMPUTES N3 COMPLEX TO COMPLEX INVERSE FFT'S AND SAVES ONLY
C       THE REAL COMPONENTS OF FIRST N2 ELEMENTS.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       ORIGINAL                JUL 88          R.D. COLEMAN, QTC
C
C                               DEC 90          L.TARVESTAD, QTC
C                               REPLACE CFTFAX/CFFTMLT WITH CFFTM.
C
C  CALLING FORMAT:
C       CALL FRCFT2 (XR, XI, M1, N1, N2, N3, INIT, TAB, WRK, IERR)
C
C  PARAMETERS:
C       XR      REAL INPUT/OUTPUT MATRIX OF VARIABLE DIMENSION
C               ON INPUT, XR CONTAINS AN N1 BY N3 REAL SUBMATRIX THAT IS
C               THE REAL COMPONENTS OF N3 COMPLEX TRACES EACH OF LENGTH
C               N1.  ON OUTPUT, XR CONTAINS AN N2 BY N3 REAL MATRIX THAT
C               IS THE REAL COMPONENTS OF THE FIRST N2 ELEMENTS OF THE
C               INVERSE COMPLEX-TO-COMPLEX FFT OF EACH TRACE.
C
C       XI      REAL INPUT MATRIX OF DIMENSION M1 BY N3
C               ON INPUT, XI CONTAINS AN N1 BY N3 REAL SUBMATRIX THAT IS
C               THE IMAGINARY COMPONENTS OF N3 COMPLEX TRACES EACH OF
C               LENGTH N1.
C
C       M1      INTEGER INPUT SCALAR.
C               ROW DIMENSION OF XR AND XI ON INPUT.
C
C       N1      INTEGER INPUT SCALAR.
C               NUMBER OF COMPLEX SAMPLES PER TRACE ON INPUT.
C               N1 MUST BE <= M1 AND OF THE FORM 2**I * 3**J * 5**K
C               WHERE I, J, AND K ARE NON-NEGATIVE INTEGERS.
C
C       N2      INTEGER INPUT SCALAR.
C               NUMBER OF REAL SAMPLES PER TRACE ON OUTPUT (ROW
C               DIMENSION OF XR ON OUTPUT).  N2 MUST BE <= N1.
C
C       N3      INTEGER INPUT SCALAR.
C               NUMBER OF TRACES (COLUMN DIMENSION OF XR AND XI).
C
C       INIT    INTEGER INPUT SCALAR.
C               INITIALIZATION FLAG.  IF INIT <> 0, THEN PERFORM
C               INITIALIZATION OF TAB.
C
C       TAB     REAL INPUT/OUTPUT VECTOR OF LENGTH 4 * N1 + 38.
C               FFT INITIALIZATION TABLES.  INPUT IF INIT = 0; OTHERWISE,
C               COMPUTED WITHIN THIS ROUTINE.
C
C       WRK     REAL SCRATCH VECTOR OF LENGTH 2 * N1.
C
C       IERR    INTEGER OUTPUT SCALAR.
C               COMPLETION CODE.  IERR = 0 FOR NORMAL COMPLETION;
C               OTHERWISE, IT IS SET TO THE SUM OF THE CONDITION CODES
C               THAT APPLY.  THE CONDITION CODE VALUES ARE:
C                   1 - THE VALUE OF M1 IS INVALID
C                   2 - THE VALUE OF N1 IS INVALID
C                   4 - THE VALUE OF N2 IS INVALID
C                   8 - THE VALUE OF N3 IS INVALID
C
C  DESCRIPTION:
C       FRCFT2 PERFORMS A COMPLEX-TO-COMPLEX INVERSE FFT ON N3 TRACES
C       EACH OF LENGTH N1.  THE REAL COMPONENTS OF THE N3 TRACES ARE
C       STORED AS COLUMNS OF AN N1 BY N3 REAL SUBMATRIX OF THE M1 BY N3
C       REAL MATRIX XR.  THE IMAGINARY COMPONENTS ARE SIMILARLY STORED
C       STORED IN THE N1 BY M3 REAL SUBMATRIX SUBMATRIX XI.  ONLY THE
C       REAL COMPONENTS OF THE FIRST N2 ELEMENTS OF EACH RESULTING TRACE
C       ARE SAVED.  THE SELECTED RESULTS ARE COMPRESSED TO FORM AN N2 BY
C       N3 REAL MATRIX IN THE FIRST N2*N3 WORDS OF XR.  THE REMAINER OF
C       XR IS NOT USED ON OUTPUT.
C
C  SUBPROGRAMS CALLED:
C
C  ERROR CONDITIONS:
C       IF ANY OF THE PARAMETERS M1, N1, N2, OR N3 HAS A VALUE OUTSIDE
C       ITS VALID RANGE (SEE ABOVE), THEN THE APPROPRIATE COMPLETION
C       CODE IS SET (SEE ABOVE) AND THE PROCESS IS ABORTED.
C
C---------------------------------------------------------------------
C
      SUBROUTINE FRCFT2 (XR, XI, M1, N1, N2, N3, INIT, TAB, WRK, IERR)
C
      REAL    XR(*), XI(*), TAB(*), WRK(*)
      INTEGER M1, N1, N2, N3, INIT, IERR, IPWR(4)
C
C---------------------------------------------------------------------
C
C  CHECK PARAMETERS FOR VALIDITY
C
      CALL NCFFT (N1, 5, K1, IPWR)
C
      IERR = 0
      IF (M1 .LE.  0                ) IERR = IERR + 1
      IF (N1 .NE. K1 .OR. N1 .GT. M1) IERR = IERR + 2
      IF (N2 .LE.  0 .OR. N2 .GT. N1) IERR = IERR + 4
      IF (N3 .LE.  0                ) IERR = IERR + 8
C
      IF (IERR .NE. 0) GO TO 800
C
C  ALLOCATE WORKSPACE
      KI = 1
      KR = KI + 2*N1+34
C
C  IF INIT <> 0, INITIALIZE TABLES
C
      IF (INIT .NE. 0) THEN
        CALL CFFTM(WRK, 2, N1, 0, 1, TAB(KI), TAB(KR), KERR)
      ENDIF
C
C  MOVE TRACE TO WORKSPACE, PERFORM FFT, COMPRESS SELECTED RESULTS
C
      IX = 1
      JX = 1
      DO 210 JT = 1, N3
      CALL CVCOMB(XR(IX),1,XI(IX),1,WRK,2,N1)
      CALL CFFTM(WRK, 2, N1, -1, 0, TAB(KI), TAB(KR), KERR)
      CALL VMOV (WRK, 2, XR(JX), 1, N2)
      IX = IX + M1
      JX = JX + N2
  210 CONTINUE
C
  800 CONTINUE
      RETURN
      END
C********************************************************************C
C NAME: GCFTLU  GENERATE COMPLEX FUNCTION TABLE REV 1.0     NOV 90   C
C********************************************************************C
C
C  PURPOSE:
C       GENERATES A COMPLEX VALUED FUNCTION LOOK-UP TABLE FOR USE WITH
C       FCFTLU.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       ORIGINAL        NOV 90          R.D. COLEMAN, QTC
C
C  CALLING FORMAT:
C       CALL GCFTLU (ZR, ZI, IZ, N, XMIN, XMAX, IFFLG, TABLE, IERR)
C
C  PARAMETERS:
C       ZR      REAL INPUT VECTOR OF LENGTH N
C               REAL COMPONENT OF FUNCTION IF IFFLG = 0.
C               NOT USED IF IFFLG != 0.
C
C       ZI      REAL INPUT VECTOR OF LENGTH N
C               IMAGINARY COMPONENT OF FUNCTION IF IFFLG = 0.
C               NOT USED IF IFFLG != 0.
C
C       IZ      INTEGER INPUT SCALAR
C               STRIDE OF VECTORS ZR AND ZI
C
C       N       INTEGER INPUT SCALAR
C               NUMBER OF FUNCTION VALUES (LENGTH OF VECTORS ZR AND ZI)
C               N MUST BE >= 2.
C
C       XMIN    REAL INPUT SCALAR
C               MINIMUM VALUE OF REAL FUNCTION ARGUMENT.
C
C       XMAX    REAL INPUT SCALAR
C               MAXIMUM VALUE OF REAL FUNCTION ARGUMENT.
C               XMAX MUST BE > XMIN
C
C       IFFLG   INTEGER INPUT SCALAR
C               FUNCTION FLAG, VALUES ARE:
C                  0 - FUNCTION VALUES CONTAINED IN ZR AND ZI
C                  1 - Z = CEXP( i*X) = COS(X) + i*SIN(X)
C                 -1 - Z = CEXP(-i*X) = COS(X) - i*SIN(X)
C                  2 - Z = CEXP( i*SQRT(X))
C                 -2 - Z = CEXP(-i*SQRT(X))
C
C       TABLE   REAL OUTPUT VECTOR OF LENGTH NTAB = 2*N+5
C               TABLE OF VALUES AND PARAMETERS:
C                 WORD  CONTENTS
C                      1  N      - NUMBER OF FUNCTION VALUES IN TABLE
C                      2  SCALE  - INDEX SCALE FACTOR
C                      3  OFFSET - INDEX OFFSET VALUE
C                      4  XMIN   - MINIMUM VALID ARGUMENT
C                      5  XMAX   - MAXIMUM VALID ARGUMENT
C                  6:N+5  REAL COMPONENTS OF FUNCTION VALUES
C               N+6:NTAB  REAL COMPONENTS OF FUNCTION VALUES
C
C       IERR    INTEGER OUTPUT SCALAR
C               COMPLETION CODE.  VALUES ARE:
C                  0 - NORMAL COMPLETION
C                  1 - INVALID VALUE OF N OR XMAX
C                  2 - INVALID VALUE OF IFFLG
C                 21 - IFFLG = 2 OR -2 AND XMIN < 0.0
C
C  DESCRIPTION:
C       THIS ROUTINE GENERATES A COMPLEX VALUED FUNCTION LOOK-UP TABLE
C       IN THE FORMAT REQUIRED BY THE ROUTINE FCFTLU.
C       THE FUNCTION IS OF THE FORM Z = F(X), WHERE X IS REAL AND Z IS
C       COMPLEX.  THE TABLE GENERATED CONTAINS N VALUES OF THE FUNCTION
C       CORRESPONDING TO N EQUALLY SPACED VALUES OF X BETWEEN XMIN AND
C       XMAX INCLUSELY.  IF IFFLG = 0, THEN THE REAL AND IMAGINARY
C       COMPONENTS OF THE FUNCTION, Z, ARE CONTAINED IN ZR AND ZI,
C       RESPECTIVELY, ON INPUT.  IF IFFLG != 0, THEN IFFLG SELECTS ONE
C       OF SEVERAL PREDETERMINED FUNCTIONS (SEE ABOVE) TO BE COMPUTED
C       BY THIS ROUTINE.  IN THIS CASE, ZR, ZI, AND IZ ARE IGNORED AND
C       NEVER ACCESSED. THE FUNCTION CORRESPONDING TO IFFLG = -K IS THE
C       COMPLEX CONJUGATE OF THE FUNCTION CORRESPONDING TO IFFLG = K.
C
C  SUBPROGRAMS CALLED:
C       NONE
C
C  ERROR CONDITIONS:
C       IF ANY PARAMETER VALUE OR COMBINATION OF PARAMETER VALUES IS
C       INVALID (SEE ABOVE), THEN THE APPROPRIATE COMPLETION CODE IS
C       SET (SEE ABOVE) AND THE ROUTINE IS ABORTED.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE GCFTLU (ZR, ZI, IZ, N, XMIN, XMAX, IFFLG, TABLE, IERR)
C
C     IMPLICIT NONE
C
C  PARAMETERS:
C
      INTEGER IZ, N, IFFLG, IERR
      REAL    ZR(*), ZI(*), XMIN, XMAX, TABLE(*)
C
C  LOCAL VARIABLES:
C
      INTEGER J, JT, JZ, ISW, MAXFUN
      REAL    DELTA, SCALE, OFFSET, SGN, XJ
C
      PARAMETER (MAXFUN = 2)
C
C-----------------------------------------------------------------------
C
      IF (N .LE. 1 .OR. XMIN .GE. XMAX) THEN
         IERR = 1
         RETURN
      ELSE IF (IFFLG .GT. MAXFUN .OR. IFFLG .LT. -MAXFUN) THEN
         IERR = 2
         RETURN
      ELSE IF (IABS( IFFLG ) .EQ. 2 .AND. XMIN .LT. 0.0) THEN
         IERR = 21
         RETURN
      ENDIF
C
      IERR = 0
C
      DELTA  = (XMAX - XMIN) / FLOAT( N - 1 )
      SCALE  = 1.0 / DELTA
      OFFSET = 6.5 - SCALE * XMIN
C
      TABLE(1) = FLOAT( N )
      TABLE(2) = SCALE
      TABLE(3) = OFFSET
      TABLE(4) = XMIN
      TABLE(5) = XMAX
      TABLE(6) = 0.0
C
      IF (IFFLG .GE. 0) THEN
         SGN = 1.0
         ISW = IFFLG
      ELSE
         SGN = -1.0
         ISW = -IFFLG
      ENDIF
C
      GO TO (200, 300), ISW
C
      JT = 6
      JZ = 1
      DO 110 J = 1, N
         TABLE(JT  ) = ZR(JZ)
         TABLE(JT+N) = ZI(JZ)
         JT = JT + 1
         JZ = JZ + IZ
  110 CONTINUE
      RETURN
C
  200 CONTINUE
      JT = 6
      DO 210 J = 0, N-1
         XJ  = XMIN + DELTA * FLOAT( J )
         TABLE(JT  ) = COS( XJ )
         TABLE(JT+N) = SIN( XJ ) * SGN
         JT = JT + 1
  210 CONTINUE
      RETURN
C
  300 CONTINUE
      JT = 6
      DO 310 J = 0, N-1
         XJ  = SQRT( XMIN + DELTA * FLOAT( J ) )
         TABLE(JT  ) = COS( XJ )
         TABLE(JT+N) = SIN( XJ ) * SGN
         JT = JT + 1
  310 CONTINUE
      RETURN
C
      END
C********************************************************************C
C NAME: FCFTLU	COMPLEX FUNCTION TABLE LOOK-UP	REV 1.0	    NOV 90   C
C********************************************************************C
C
C  PURPOSE:
C	FINDS A VECTOR OF VALUES OF A COMPLEX VALUED FUNCTION OF A
C	REAL VARIABLE BY MEANS OF A TABLE LOOK-UP.
C
C  LANGUAGE:
C	FORTRAN 77
C
C  HISTORY:
C	ORIGINAL	NOV 90		R.D. COLEMAN, QTC
C
C  CALLING FORMAT:
C       CALL FCFTLU (TABLE, X, IX, ZR, ZI, IZ, N, ICFLG, IERR)
C
C  PARAMETERS:
C	TABLE	REAL INPUT VECTOR OF IMPLIED LENGTH
C		TABLE OF VALUES AND PARAMETERS:
C		  WORD  CONTENTS
C		        1  M      - NUMBER OF TABLE ENTRIES
C		        2  SCALE  - INDEX SCALE FACTOR
C		        3  OFFSET - INDEX OFFSET VALUE
C		        4  XMIN   - MINIMUM VALID ARGUMENT
C		        5  XMAX   - MAXIMUM VALID ARGUMENT
C		    6:M+5  REAL COMPONENTS OF FUNCTION VALUES
C		M+6:2*M+5  IMAG COMPONENTS OF FUNCTION VALUES
C
C	X	REAL INPUT VECTOR OF LENGTH N
C		SOURCE VECTOR
C
C	IX	INTEGER INPUT SCALAR
C		STRIDE OF VECTOR X
C
C	ZR	REAL OUTPUT VECTOR OF LENGTH N
C		REAL COMPONENT OF RESULT VECTOR
C
C	ZI	REAL OUTPUT VECTOR OF LENGTH N
C		IMAGINARY COMPONENT OF RESULT VECTOR
C
C	IZ	INTEGER INPUT SCALAR
C		STRIDE OF VECTORS ZR AND ZI
C
C	N 	INTEGER INPUT SCALAR
C		LENGTH OF VECTORS X, ZR, AND ZI.
C
C	ICFLG	INTEGER INPUT SCALAR
C		CONJUGATE FLAG:
C		   = 0  FUNCTION VALUE RETURNED
C		  != 0  CONJUGATE OF FUNCTION VALUE RETURNED
C
C	IERR	INTEGER OUTPUT SCALAR
C		COMPLETION CODE.  VALUES ARE:
C		   0 - NORMAL COMPLETION
C		   1 - ONE OR MORE INPUT VALUES ARE LESS THAN XMIN
C		   2 - ONE OR MORE INPUT VALUES ARE GREATER THAN XMAX
C		   3 - BOTH CONDITIONS 1 AND 2 OCCURRED.
C
C  DESCRIPTION:
C	THIS ROUTINE FINDS A VECTOR OF VALUES OF A COMPLEX VALUED
C	FUNCTION OF A REAL VARIABLE BY MEANS OF A TABLE LOOK-UP.
C       THE FUNCTION IS OF THE FORM Z = F(X), WHERE X IS REAL AND Z IS
C	COMPLEX.  THE TABLE CONTAINS M VALUES OF THE FUNCTION
C	CORRESPONDING TO M EQUALLY SPACED VALUES OF X BETWEEN XMIN AND
C	XMAX, INCLUSELY.  THE REAL COMPONENTS OF THE VALUES ARE STORED
C       IN TABLE LOCATIONS 6 THROUGH NTAB = M+5 AND THE IMAGINARY
C       COMPONENTS ARE STORE IN TABLE LOCATIONS NTAB+1 THROUGH NTAB+M.
C
C	FOR EACH ELEMENT OF THE SOURCE VECTOR, X(J), A TABLE INDEX IS
C	CALCULTED: I = IFIX( SCALE * X(J) + OFFSET ).  	IF X(J) < XMIN,
C	THEN I IS SET TO THE FIRST TABLE ENTRY (6) AND BIT 0 OF IERR IS
C	SET TO 1.  IF X(J) > XMAX, THEN I IS SET TO THE LAST TABLE ENTRY
C	(NTAB) AND BIT 1 OF IERR IS SET TO 1.  THE REAL COMPONENT IS
C       THEN FETCHED FROM TABLE(I) AND THE IMAGINARY COMPONENT FROM
C       TABLE(I+M).
C
C  SUBPROGRAMS CALLED:
C	NONE
C
C  ERROR CONDITIONS:
C	IF ANY ELEMENT OF THE SOURCE VECTOR HAS A VALUE OUTSIDE THE
C	VALID RANGE, THEN THE APPROPRIATE COMPLETION CODE IS SET (SEE
C	ABOVE), A DEFAULT RESULT IS SUPPLIED (SEE ABOVE), AND PROCESSING
C	CONTINUES.
C
C-----------------------------------------------------------------------
C
      SUBROUTINE FCFTLU (TABLE, X, IX, ZR, ZI, IZ, N, ICFLG, IERR)
C
C     IMPLICIT NONE
C
C  PARAMETERS:
C
      INTEGER IX, IZ, N, ICFLG, IERR
      REAL    TABLE(*), X(*), ZR(*), ZI(*)
C
C  LOCAL VARIABLES:
C
      INTEGER M, NTAB, I, J, JX, JZ
      REAL    SCALE, OFFSET, XMIN, XMAX, SGN
      LOGICAL ERRLO, ERRHI
C
C-----------------------------------------------------------------------
C
      IERR = 0
      IF (N .LE. 0) RETURN
C
      M      = TABLE(1)
      SCALE  = TABLE(2)
      OFFSET = TABLE(3)
      XMIN   = TABLE(4)
      XMAX   = TABLE(5)
      NTAB   = M + 5
C
      IF (ICFLG .EQ. 0) THEN
	 SGN = 1.0
      ELSE
	 SGN = -1.0
      ENDIF
C
      ERRLO = .FALSE.
      ERRHI = .FALSE.
      JX    = 1
      JZ    = 1
      DO 110 J = 1, N
         IF      (X(JX) .LT. XMIN) THEN
	    ERRLO = .TRUE.
	    I     = 6
         ELSE IF (X(JX) .GT. XMAX) THEN
	    ERRHI = .TRUE.
	    I     = NTAB
	 ELSE
	    I = IFIX( SCALE * X(JX) + OFFSET )
	 ENDIF
C
	 ZR(JZ) = TABLE(I  )
	 ZI(JZ) = TABLE(I+M) * SGN
C
	 JX = JX + IX
	 JZ = JZ + IZ
  110 CONTINUE
C
      IF (ERRLO) IERR = IERR + 1
      IF (ERRHI) IERR = IERR + 2
      RETURN
      END
C********************************************************************C
C NAME: VCINT   CUBIC INTERPOLATION             REV 1.0     JUN 88   C
C********************************************************************C
C
C  CALL FORMAT
C
C       CALL VCINT (X1, Y1, N1, X2, Y2, N2, IZ, ZZ, INIT)
C
C       where,
C
C       X1      Real input vector of length N1
C               Source vector containing X coordinates corresponding
C               to Y1.
C
C       Y1      Real input vector of length N1
C               Source vector containing Y coordinates.
C
C       N1      Integer input scalar
C               Element count for X1 and Y1.  N1 MUST BE >= 4.
C
C       X2      Real input vector of length N2
C               Source vector containing X coordinates corresponding
C               to Y2.
C
C       Y2      Real output vector of length N2
C               Result vector.
C
C       N2      Integer input scalar
C               Element count for X2 and Y2.
C
C       IZ      Integer input/output vector of length N2
C               Intermediate index vector calculated if INIT <> 0.
C
C       ZZ      Real input/output vector of length 4*N2
C               Intermediate coefficient vector calculated if INIT <> 0.
C
C       INIT    Integer input scalar
C               Initialization flag. If INIT <> 0, then the inter-
C               mediate vectors IZ and ZZ are calculated; otherwise
C               they are assumed to have been calculated in a previous
C               call.
C
C
C  DESCRIPTION
C
C       Given a set of (X,Y) coordinates (X1,Y1), VCINT performs a
C       cubic interpolation to obtain an output set of (X,Y)
C       coordinates (X2,Y2).  The values of both X1 and X2 must be in
C       ascending order and may have arbitrary spacing.  If X2(I) <
C       X1(1) for some I, then Y2(I) = Y1(1).  Similarly, if X2(I) >
C       X1(N1), then Y2(I) = Y1(N1).
C
C       The intermediate result vectors IZ and ZZ are calculated
C       if the initialization flag INIT <> 0.  These vectors are
C       dependent on N1, N2, X1, and X2; i.e., they are dependent on
C       all input arguments except Y1.  When a sequence of calls are
C       made in which only Y1 changes, it is most efficient to set
C       INIT to 1 for the first call then set it to 0 for the
C       subsequent calls.
C
C
C  ERROR CONDITIONS
C
C       If N1 < 4 or N2 < 1, the routine is aborted.
C
C
C  EXAMPLE
C
C       CALL VCINT (X1, Y1, N1, X2, Y2, N2, IZ, ZZ, INIT)
C
C       Input Operands:
C
C       X1   = 0.013     Y1 = 1.542
C              0.028          2.138
C              0.039          1.414
C              0.055         -0.749
C              0.075         -2.451
C              0.090         -1.473
C              0.107          1.098
C              0.119          2.508
C              0.137          2.046
C              0.155         -0.928
C              0.172         -2.818
C              0.187         -2.142
C              0.205          0.920
C              0.222          2.934
C              0.239          1.951
C              0.254         -0.741
C              0.271         -2.900
C              0.287         -2.229
C              0.300         -0.051
C              0.319          2.694
C              0.334          2.368
C              0.352         -0.348
C              0.368         -2.474
C              0.388         -1.867
C              0.407          1.022
C              0.421          2.397
C              0.431          2.240
C              0.450          0.068
C              0.460         -1.360
C              0.474         -2.159
C       N1   = 30
C       X2   = 0.013
C              0.042
C              0.071
C              0.100
C              0.128
C              0.157
C              0.186
C              0.215
C              0.244
C              0.272
C              0.301
C              0.330
C              0.359
C              0.388
C              0.416
C              0.445
C              0.474
C       N2   = 17
C       INIT = 1
C
C       Output Operands:
C
C       Y2 = 1.542
C            1.073
C           -2.311
C           -0.040
C            2.586
C           -1.234
C           -2.229
C            2.333
C            1.145
C           -2.934
C            0.226
C            2.674
C           -1.423
C           -1.849
C            2.129
C            0.668
C           -2.159
C
C
C  HISTORY
C
C       1) Jun 88     R.D. Coleman         Original
C       2) Oct 90     M. Frommer           MathAdvantage version
C
C*********************************************************************
C
      SUBROUTINE VCINT (X1, Y1, N1, X2, Y2, N2, IZ, ZZ, INIT)
C
      INTEGER N1, N2, IZ(N2), INIT
      REAL    X1(N1), Y1(N1), X2(N2), Y2(N2), ZZ(N2,4), XX
C
C-----------------------------------------------------------------------
C
      IF (N1 .LT. 4 .OR. N2 .LT. 1) GO TO 800
C
      IF (INIT .EQ. 0) GO TO 200
C
C     ----------------------
C     PERFORM INITIALIZATION
C     ----------------------
C
      X1LO = X1( 1)
      X1HI = X1(N1)
C
      J  = 3
      DO 140 I = 1, N2
C
C     DO FIX UP FOR OUT-OF-RANGE VALUES OF X2
C
         IF      (X2(I) .LT. X1LO) THEN
            XX = X1LO
         ELSE IF (X2(I) .GE. X1HI) THEN
            XX = X1HI
         ELSE
            XX = X2(I)
         ENDIF
C
C     UPDATE J
C
  120    CONTINUE
         IF (XX .GT. X1(J) .AND. J .LT. N1-1) THEN
            J = J + 1
            GO TO 120
         ENDIF
C
C     CALCULATE IZ AND ZZ
C
         J1 = J  - 2
         J2 = J1 + 1
         J3 = J2 + 1
         J4 = J3 + 1
C
         DX1 = XX     - X1(J1)
         DX2 = XX     - X1(J2)
         DX3 = XX     - X1(J3)
         DX4 = XX     - X1(J4)
         D12 = X1(J1) - X1(J2)
         D13 = X1(J1) - X1(J3)
         D14 = X1(J1) - X1(J4)
         D23 = X1(J2) - X1(J3)
         D34 = X1(J3) - X1(J4)
         D42 = X1(J4) - X1(J2)
C
	 IZ(I)   = J1
         ZZ(I,1) = DX2 * DX3 * DX4 / (D12 * D13 * D14)
         ZZ(I,2) = DX1 * DX3 * DX4 / (D12 * D23 * D42)
         ZZ(I,3) = DX1 * DX2 * DX4 / (D13 * D23 * D34)
         ZZ(I,4) = DX1 * DX2 * DX3 / (D14 * D42 * D34)
  140 CONTINUE
C
C     ---------------------
C     PERFORM INTERPOLATION
C     ---------------------
C
  200 CONTINUE
      DO 210 I = 1, N2
         J  = IZ(I)
         Y2(I) = ZZ(I,1) * Y1(J) + ZZ(I,2) * Y1(J+1) + ZZ(I,3) * Y1(J+2)
     &         + ZZ(I,4) * Y1(J+3)
  210 CONTINUE
C
C     ------------
C     EXIT ROUTINE
C     ------------
C
  800 CONTINUE
      RETURN
      END
C********************************************************************C  
C NAME: WHENFLT GATHER INDICES OF ELEMENTS LESS THAN TARGET JAN 89   C
C********************************************************************C  
C
C     INPUTS:
C       
C      N        number of elements to be searched
C
C      ARRAY    first element of real array to be searched
C
C      INC      increment between elements of searched array
C
C      TARGET   real value searched for in the array
C      
C     OUTPUTS:
C
C      INDEX  integer array containing index of the found target in 
C             the array
C
C      NVAL   number of values put in the index array
C      
C *******************************************************************
C
      SUBROUTINE WHENFLT (N, ARRAY, INC, TARGET, INDEX, NVAL)
C
      INTEGER N, INC, INDEX(*), NVAL
      REAL    ARRAY(*), TARGET
      INTEGER I, IA
C
      IA   = 1
      NVAL = 0
      DO 100 I = 1, N
	 IF (ARRAY(IA) .LT. TARGET) THEN
	    NVAL = NVAL + 1
	    INDEX(NVAL) = I
         ENDIF
	 IA = IA + INC
100   CONTINUE
C
      RETURN
      END
C********************************************************************C  
C NAME: GATHER  GATHER A VECTOR FROM A SOURCE VECTOR        JAN 89   C
C********************************************************************C  
C
C     INPUTS:
C       
C      N        Integer number of elements in A and INDEX (not b)
C      
C      B        Real source vector
C      
C      INDEX    Integer vector of indices 
C      
C     OUTPUTS:
C
C      A        Real vector resulting from the gather
C      
C *******************************************************************
C
      SUBROUTINE GATHER (N, A, B, INDEX)
      INTEGER N, INDEX(*)
      REAL    A(*), B(*)
      INTEGER I
C
      DO 100 I = 1, N
         A(I) = B(INDEX(I))
100   CONTINUE
C
      RETURN
      END
C********************************************************************C  
C NAME: SCATTER SCATTER A VECTOR INTO ANOTHER VECTOR        JAN 89   C
C********************************************************************C  
C
C     INPUTS:
C       
C      N        number of elements to be scattered
C
C      INDEX    Integer vector of indices
C      
C      B        Real source vector
C      
C     OUTPUTS:
C
C      A        Real vector resulting from the scatter
C      
C *******************************************************************
C
      SUBROUTINE SCATTER (N, A, INDEX, B)
C
      INTEGER N, INDEX(*)
      REAL    A(*), B(*)
      INTEGER I
C
      DO 100 I = 1, N
         A(INDEX(I)) = B(I)
100   CONTINUE

      RETURN
      END
