C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  time2depth.ray.F reads an xsd depth pick file and a velocity field and  *
C   writes an xsd arrival time pick file                               *
C   aand also writes an xsd file of rays                               *
C  Mary Ann Thornton        Version 1.0              January 6, 1995   *
C    Modified by D. Vasicek to trace rays starting at a picked layer.  *
C      10 January, 1995                                                *
C    Modified for dynamic memory allocation aug 1, 1996                *
C***********************************************************************
      Program time2depth_ray
      Implicit none
#include <localsys.h>
c    see trcgp/include for localsys.h
#include <f77/lhdrsz.h>
#include <f77/iounit.h>
#include <f77/sisdef.h>
c  see usp/include/f77 for sisdef.h iounit.h lhdrsz.h
cccc#include <f77/lpick.h>
c see ~mbs/include/f77 for f77/lpick.h
C  lpick.h contains some pointers necessary for allocating the arrays to
c  contain the record,trace,sample numbers which make up the xsd picks file
C***********************************************************************
C
C         1         2         3         4         5         6         7
C232456789012345678901234567890123456789012345678901234567890123456789012
      integer lpck, lpout, n_bytes, ijk
      parameter (lpck=27, lpout=28)
      integer   lhed (SZLNHD)
      integer*2 itr  (SZLNHD)
      real      head (SZLNHD)
      real      signz
      Integer  Nsteps
      real      vel(1)
c     integer ptr_vel
      pointer(ptr_vel,vel)
      
      real      velsmo(1)
      real      slow(1)
      real      x0(1)
      real      z0(1)
      real      slowness(1,1)
      real      slow0(1)
      real      slowx(1)
      real      slowz(1)
      real      dSds(1)
      real      Tau(1,1)
      real      tx(1)
      real      tz(1)
      real      tzprev(1)
      real      rx(1)
      real      rz(1)
      Integer   ix(1)
      Integer   iz(1)
      real      x(1,1)
      real      z(1,1)
      real      rray(1,1,1)
      real      normx(1,1)
      real      normz(1,1)
      INTEGER   Lray(1)
      Integer   inray(1)
      logical   saveray(1)
      Integer   errcode, errcode2
      Integer   icolor(1)
      real      rec(1), trac(1), samp(1)

c
c   set up pointers for dynamic allocation 
c
c     Integer    prec, ptrac, psamp, pcolor
c     Integer    Ptr_velsmo  
c     Integer    Ptr_slow
c     Integer    Ptr_x0
c     Integer    Ptr_z0
c     Integer    Ptr_slowness
c     Integer    Ptr_slow0
c     Integer    Ptr_slowx
c     Integer    Ptr_slowz
c     Integer    Ptr_dSds
c     Integer    Ptr_Tau
c     Integer    Ptr_tx
c     Integer    Ptr_tz
c     Integer    Ptr_tzprev
c     Integer    Ptr_rx
c     Integer    Ptr_rz
c     Integer    Ptr_ix
c     Integer    Ptr_iz
c     Integer    Ptr_x
c     Integer    Ptr_z
c     Integer    Ptr_rray
c     Integer    Ptr_inray
c     Integer    Ptr_saveray
c     Integer    Ptr_normx
c     Integer    Ptr_normz
c     INTEGER    Ptr_Lray
      Integer   nray, nrays

      pointer(ptr_velsmo,   velsmo)
      pointer(ptr_slow,     slow)
      pointer(ptr_x0,       x0)
      pointer(ptr_z0,       z0)
      pointer(ptr_slowness, slowness)
      pointer(ptr_slow0,    slow0)
      pointer(ptr_slowx,    slowx)
      pointer(ptr_slowz,    slowz)
      pointer(ptr_dSds,     dSds)
      pointer(ptr_Tau,      Tau)
      pointer(ptr_tx,       tx)
      pointer(ptr_tz,       tz)
      pointer(ptr_tzprev,   tzprev)
      pointer(ptr_rx,       rx)
      pointer(ptr_rz,       rz)
      pointer(ptr_ix,       ix)
      pointer(ptr_iz,       iz)
      pointer(ptr_x,        x)
      pointer(ptr_z,        z)
      pointer(ptr_rray,     rray)
      pointer(ptr_inray,    inray)
      pointer(ptr_saveray,  saveray)
      pointer(ptr_normx,    normx)
      pointer(ptr_normz,    normz)
      pointer(ptr_Lray,     Lray)
      pointer(pcolor,      icolor)
      pointer(psamp,        samp)
      Pointer(ptrac,        trac)
      Pointer(prec,          rec)

      integer     nsamp, nsi, ntrc, nrec, iform, Current, Next
      integer     luin , lbytes, nbytes
      integer     irs,ire,ns,ne, nseg

      character   ntap*128, picks*128, pout*128, name*10
      character   junkc, uword*6
      character   version * 4
      logical     verbos, hlp, raypath
      integer     argis
      Integer     kk,nn,jseg,maxpik, ntrac, i, n_picks, n1rays, j, L
      Integer     jerr, idx, idz, nz, nx, jj
      Integer     ismv, ismh
      Real         dx, dz, dt
      equivalence (itr( 1), lhed (1), head(1))

      real units(3),offset(3)
C     units(1) = units for the record number from the 'Units' card
C     units(2) = units for the trace number from the 'Units' card
C     units(3) = units for the sample number from the 'Units' card
C     offset(1)= offset for the record number from the 'Units' card
C     offset(2)= offset for the trace number from the 'Units' card
C     offset(3)= offset for the sample number from the 'Units' card

      Integer max_segs    
      parameter (Max_segs = 2000)
c     real rec(max_segs),trac(max_segs),samp(max_segs)
C     rec(1)     = array containing the record numbers
C     trac(1)    = array containing the trace numbers
C     samp(1)    = array containing the sample numbers

C     icolor(1)  = array containing the segment color
C                  the length of icolor array is jseg (the no. of segments)

      character*20 segname(max_segs), segname_dummy

      integer npts(max_segs), npts_dummy, icolor_dummy
      data lbytes/ 0 /, nbytes/ 0 /
      data  name/'time2depth_ray'/, version/' 1.0'/
C     npts(1) = the number of points in segment 1
C     npts(2) = the number of points in segment 2 
c     This program limits the number of segments to max_segs
c-----
      hlp  = (argis( '-h' ).gt.0) .or. (argis( '-?').gt.0)
      if( hlp ) then
          call help
          stop
      endif
      do i = 1,max_segs
       segname(i)= 'NO_PICK_NAME_HERE'
      enddo
c-----
c     open printout
#include <f77/mbsopen.h>
c-----
c     read command line arguments
      call gcmdln(ntap,picks,pout,ns,ne, raypath,
     1            irs,ire,Nsteps,dt,verbos, nseg)
c-----
c     open input and output files
      call getln(luin , ntap,'r', 0)
c-----
c     read line header
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'time2depth_ray: no line header read ', 
     1               'from unit ',luin
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         go to 999
      endif
c------
c     save values from line header
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'Dx1000', idx,   LINHED)
      call saver(itr, 'Dz1000', idz,   LINHED)
      dx = idx/1000.
      dz = idz/1000.
      nz = nsamp
      nx = ntrc
      If (verbos) then
        Write(LERR,*) 'Velocity model Description'
        Write(LERR,*) 'nsamp =',nsamp
        Write(LERR,*) 'nsi   =',nsi
        Write(LERR,*) 'ntrc  =',ntrc
        Write(LERR,*) 'nrec  =',nrec
        Write(LERR,*) 'iform =',iform
        Write(LERR,*) 'Dx1000=',idx
        Write(LERR,*) 'Dz1000=',idz
      Endif
      if(dz.le.0.0)then
         write(LOT,*)' Dz1000 in the line header cannot be zero.'
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         go to 999
      endif
c------
c     print line header into printer listing
      call hlhprt(itr, lbytes, name, 8, LERR)
c-----
c     check validity of these arguments
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
c-----
c-----
      if( verbos ) then
         call verbal(nsamp, nsi, ntrc, nrec, iform, raypath,
     &               ntap,picks,pout,ns,ne,irs,ire,dt,verbos)
      endif
c 
c------Allocate space for the velocity model----------------
c
      if (verbos) then
        write(LERR,*) 'Allocating ',nsamp*ntrc,' locations for velocity'
c       call galloc_verify()
      endif
      call galloc(ptr_vel,nsamp*ntrc*SZSMPD, errcode, errcode2)

c-----
c--------------------------------------------------
c     Read the velocity field
c--------------------------------------------------
c-----skip unwanted records
      call recskp(1,irs-1,luin,ntrc,itr)

c-----RECORD LOOP
      do 1000 jj = irs, ire
         call trcskp(jj,1,ns-1,luin,ntrc,itr)
         kk = 1
c--------TRACE LOOP
         do 1003  nn = ns, ne
c           read the traces into itr then move the data to vel
            nbytes = 0
            call rtape( luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on input:'
               write(LERR,*)'  rec= ',jj,'  trace= ',nn
               go to 999
            endif
            call vmov(lhed(ITHWP1),1,vel(kk),1,nsamp)
            kk = kk + nsamp
 1003    continue
c-----
c        skip to end of record
         call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
 1000 continue
c     call galloc_verify()
c--------------------------------------------------
c     Open the xsd input and output files       
c--------------------------------------------------
      if(picks.ne.' ')then
         open(unit=lpck,file=picks,status='old',iostat=jerr) 
         if(jerr.ne.0)then
            write(LERR,*)'  Error opening xsd picks file'
            go to 999
         endif
      else
         write(LERR,*)' Picks filename must be supplied'
         write(LER,*)' Job terminated abnormally'
         go to 999
      endif

      if(pout.ne.' ')then
         open(unit=lpout,file=pout,iostat=jerr)
         if(jerr.ne.0)then
            write(LERR,*)'  Error opening output xsd picks file'
            go to 999
         endif
      else
         write(LERR,*)' Output picks filename must be supplied'
         write(LER,*)' Job Terminated abnormally'
         go to 999
      endif
      write(LERR,10)
c     call galloc_verify()
   10 format (//, 27X, 'program parameters',//)
      write(LERR,37)picks
      write(LERR,38)pout
   37 format(' input picks dataset = '/,A128)
   38 format(' output picks dataset = '/,A128)
c-----
c--------------------------------------------------
C  read the picks file
c--------------------------------------------------
C  rdpick parameters:
C    cardl      Character array input/output (defined in lpick.h)      *
C    cardd      Character array input/output (defined in lpick.h)      *
C    cardx      Character array input/output (defined in lpick.h)      *
C    prec       Pointer to real array of x points (defined in lpick.h) *
C               This array will be returned as array 'rec'             *
C    ptrac      Pointer to real array of y points (defined in lpick.h) *
C               This array will be returned as array 'trac'            *
C    psamp      Pointer to real array of z points (defined in lpick.h) *
C               This array will be returned as array 'samp'            *
C    pcolor     Pointer to integer array of segment colors             *
C               (defined in lpick.h)                                   *
C               This array will be returned as array 'icolor'          *
C    segname    Character*20 array input/output array of length max_seg*
C               This must be dimensioned 1000 in the calling routine   *
C    npts       Integer input/output array of length 1000              *
C               This must be dimensioned 1000 in the calling routine   *
C               npicks(1) = the number of points in segment one        *
C               npicks(2) = the number of points in segment two, etc.  *
C    units      Real input/output array of length 3                    *
C               This must be dimensioned 3 in the calling routine      *
C               Units measure for the x,y,z points read from file      *
C               units(1) = Units measure for the x points              *
C               units(2) = Units measure for the y points              *
C               units(3) = Units measure for the z points              *
C    offset     Real input/output array of length 3                    *
C               This must be dimensioned 3 in the calling routine      *
C               Offsets measure of x,y,z points read from file         *
C               offset(1) = offset measure for the x points            *
C               offset(2) = offset measure for the y points            *
C               offset(3) = offset measure for the z points            *
C    jseg       Integer output scalar                                  *
C               Number of segments in the picks file                   *
C    maxpik     Integer output scalar                                  *
C               Maximum number of picks in any one segment             *
C    SZSMPD     Integer input scalar (defined in lhdrsz.h)             *
C               Size of data sample                                    *
C    nrec       Integer output scalar
C               No. records in original 'picked' dataset               *
C    ntrac      Integer output scalar                                  *
C               No. records in original 'picked' dataset               *
C    nsamp      Integer output scalar                                  *
C               No. records in original 'picked' dataset               *
C    LERR       Integer input scalar                                   *
C               Logical unit of printout                               *
C    lpck       Integer input scalar                                   *
C               Logical unit of pick file                              *
C    jerr       Integer input/output scalar                            *
C               Error flag                                             *
c     call rdpick(cardl,cardd,cardx,prec,ptrac,psamp,pcolor,
c    &segname,npts,units,offset,jseg,maxpik,SZSMPD,nrec,ntrac,nsamp,
c    &LERR,lpck,jerr)

       read(lpck,50) uword,
     1    units, nrec, ntrac, nsamp, Offset,jseg, maxpik
       If(verbos) then
       write(LERR,50)    units, nrec, ntrac, nsamp, Offset,jseg, maxpik
       write(LERR,*) ' No. of segments=',jseg
c      call galloc_verify()
       endif

c50    Format('Units',3F13.6,3I6,' Offset',3F13.6, ' Count',2I6)
50     Format(a6,3(f12.6,1x),3(i5,1x),6x,3(f12.6,1x),7x,i5,1x,i5)
        n_picks= 0
        do i=1,jseg
         READ(lpck,20) Ijk, SEGNAME_dummy, ICOLOR_dummy, NPTS_dummy           
20    FORMAT('Segment = ',I5,' Name ',A20,2X,'color = ',   
     1        I5,' picks =',I6)
        n_picks=N_picks + Npts_dummy 
        do j=1,Npts_dummy
          read(lpck,'(A)') junkc
        Enddo
        Enddo

        If (verbos) then
        write(LERR,*) 'Allocating ', n_picks,' locations for '
        write(LERR,*) 'rec(.), trac(.), samp(.), tx, & tz.'
c       call galloc_verify()
        endif

        call galloc(prec  , n_picks*SZSMPD, errcode, errcode2)
        call galloc(ptrac , n_picks*SZSMPD, errcode, errcode2)
        call galloc(psamp , n_picks*SZSMPD, errcode, errcode2)
        call galloc(Ptr_tx, n_picks*SZSMPD, errcode, errcode2)
        call galloc(Ptr_tz, n_picks*SZSMPD, errcode, errcode2)
        
        If (verbos) then
        write(LERR,*) 'Allocating', jseg, ' locations for seg color' 
c       call galloc_verify()
        endif

        call galloc(pcolor,  jseg*SZSMPD, errcode, errcode2)

        
        rewind (lpck)
        if (verbos) then
        write(LERR,*) 'reading from file ', picks, lpck
c       call galloc_verify()
        endif
        Call rxsdpicks(rec,trac,samp,icolor,segname,npts,
     1            units,offset, n_picks,
     1            nrec,ntrac,nsamp,jseg,LERR,lpck,jerr,
     1            tx, tz, max_segs)

      if(jerr.ne.0)then
         write(LERR,*)' Error reading xsd file  '
         write(LER,*) ' Error reading xsd file  '
         go to 999
      endif

      If(verbos) then
      Write(LERR,*) 'Picked Horizons description:'
      Write(LERR,*) '# of segments=',jseg
      Write(LERR,*) 'npts(:)=',(npts(i),i=1,jseg)
c     call galloc_verify()
      endif
c
c---------allocate storage for saving ray paths
c
      
      if(raypath) then
      n_bytes=5*npts(nseg)*nz*nsteps*SZSMPD
      If (verbos) then
      write(LERR,*) 'Keeping ray paths for',npts(nseg),' rays.' 
      write(LERR,*) 'This requires ',n_bytes,' bytes.'
      endif
      nray= nz
      call galloc(Ptr_rray,4*npts(nseg)*nz*SZSMPD,
     1 errcode, errcode2)
      call galloc(Ptr_saveray,n_picks*SZSMPD,errcode, errcode2)
      call galloc(Ptr_lray   ,n_picks*SZSMPD,errcode, errcode2)
      Do i=1,npts(nseg)
        saveray(i)=.false.
      enddo
      endif

      n_picks= 0
      Do i = 1,jseg

        if(i .eq. nseg .and. raypath) then
        n1rays = 0
        do j =n_picks+1, N_picks+npts(i)
         n1rays = n1rays + 1
         saveray(j) = .true.
         Lray(j) = n1rays
        enddo
        endif
        If (verbos) then
        write(LERR,*) 'seg#=',i
        write(LERR,*) 'trace values='
        write(LERR,*) (trac(j),j=n_picks+1, n_picks+npts(i))
        write(LERR,*) 'sample values='
        write(LERR,*) (samp(j),j=n_picks+1, n_picks+npts(i))
        endif
        n_picks = n_picks+npts(i)
      Enddo
      nrays = n1rays 
      If (verbos) then
      Write(LERR,*) 'Total picks submitted for raytracing=', n_picks
c     call galloc_verify()
      endif
c-----
c--------------------------------------------------
c--------------------------------------------------
c--------------------------------------------------
c-----
c     BEGIN PROCESSING 
c-----
c--------------------------------------------------
c----
c       At this point the velocity field and the xsd file have been read
c       and are ready to be used.
c----
c
c--------------allocate storage for tracing the rays----------------
c
       
      call galloc(Ptr_slow,    nx*SZSMPD  ,errcode, errcode2)
      call galloc_msg(errcode, errcode2,1,LERR)
      call galloc(Ptr_slowness,nx*2*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,2,LERR)
      call galloc(Ptr_x0,     n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,3,LERR)
      call galloc(Ptr_z0,     n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,4,LERR)
      call galloc(Ptr_slow0,  n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,5,LERR)
      call galloc(Ptr_slowx,  n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,6,LERR)
      call galloc(Ptr_slowz,  n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,7,LERR)
      call galloc(Ptr_dSds ,  n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,8,LERR)
      call galloc(Ptr_tzprev, n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,9,LERR)
      call galloc(Ptr_rx   ,  n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,10,LERR)
      call galloc(Ptr_rz   ,  n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,11,LERR)
      call galloc(Ptr_iz   ,  n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,12,LERR)
      call galloc(Ptr_ix   ,  n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,13,LERR)
      call galloc(Ptr_normx,  n_picks*SZSMPD*2,errcode, errcode2)
      call galloc_msg(errcode, errcode2,14,LERR)
      call galloc(Ptr_normz,  n_picks*SZSMPD*2,errcode, errcode2)
      call galloc_msg(errcode, errcode2,15,LERR)
      call galloc(Ptr_inray,  n_picks*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,16,LERR)
      call galloc(Ptr_Tau  ,  n_picks*2*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,17,LERR)
      call galloc(Ptr_x    ,  n_picks*2*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,18,LERR)
      call galloc(Ptr_z    ,  n_picks*2*SZSMPD,errcode, errcode2)
      call galloc_msg(errcode, errcode2,19,LERR)


C-----we need to account for the scaling of the original data.
C     If the "Units" fields of the xsd data were set then 
C     the data may have some spurious factors built into it
C     that we need to allow for:
       L = 0
       Do i = 1, n_picks
         x0(i) = dx*trac(i)/units(2)
         z0(i) = dt*samp(i)*.0005/units(3)
         if (abs(tz(i)) .gt. .001) then
            L = L +1
         endif
       enddo
       If (verbos) then
          write(LERR,*) '# Nonzero directions=',L
          write(LERR,*) 'Number of picks =',n_picks
       Endif


c--------smooth the velocity data if requested-------------------------------
      If (ismv .gt. 0 .or. ismh .gt. 0) then
        call galloc(Ptr_velsmo,nx*nz*SZSMPD,errcode, errcode2)
        call smoothslow(vel,velsmo,nx,nz, ismv, ismh)
        call gfree(ptr_velsmo)
      endif

c-----compute normal directions---------------------------------------------
      If (L.lt. n_picks/10) then 
       If (verbos) then
       Write(LERR,*) 'Creating normals because less than 10% '
       write(LERR,*) 'of the initial directions are non-zero.'
       write(LERR,*) 'Only ',L,' are nonzero of ',n_picks
       Endif
      call lsq(z0, vel, Nx, Nz, x0,dx,
     1    n_picks, jseg, npts, tx, tz)
c     call djv(z0, vel, Nx, Nz, x0,dx,
c    1    n_picks, jseg, npts, tx, tz)
c     identify the rays to be traced:
      
       else
        If (verbos) then
        write(LERR,*) 'Using predefined intitial directions.'
        write(LERR,*) 'Selecting downward going ray directions.'
        Endif
        Do i = 1,n_picks
           signz = sign(1.,tz(i))
           tx(i) = tx(I)*signz
           tz(i) = tz(I)*signz
        enddo
       endif


c--------Propagate rays from the surface---------------------------------------

       
       If (verbos) then 
       write(LERR,*) 'units(:)=', Units
       Endif
C a second order Runga-Kutta version
c     write (*,*) 'Checking the ray2d interface'
c     j = 0
c     Do i = 1,n_picks
c       j = j+1
c       x0(j)= j
c     enddo
c     Do i = 1,n_picks
c       j = j+1
c       z0(j)= j
c     enddo
c     Do i = 1,n_picks
c       j = j+1
c       tx(j)= j
c     enddo
c     Do i = 1,n_picks
c       j = j+1
c       tz(j)= j
c     enddo
c      
      call ray2d( offset(2), offset(3),
     1   x0,  z0,
     1   tx,  tz,
     1   dx,  dz,  n_picks,
     1   nx,  nz,  Vel ,
C   ---------------Arrays provided for storage inside of ray2ddown-----
     1       Slow,
C             slowness at one level
     1       Slowness,
C              Slowness at previous, current, and next level
     1       Slow0,
C              Slowness interpolated to ray locations
     1       Slowx,
C              Slowness gradient interpolated to ray locations
     1       Slowz,
     1       dSds,
C              Derivative of slowness in the direction of the rays
     1       tau,
C              Travel time along the rays
     1       normx,
C              Cosine of ray direction in the x direction times the slowness
     1       normz,
C              Cosine of the ray direction in the Z direction * slowness
     1       X ,
C              X-location of the rays
     1       z ,
C              z-location of the rays
     1       ix, iz,
C              Location of the rays within ther grid (grid index of rays)
     1       rx, rz, Current, Next, Nsteps, rray, nrays, nray, inray
C              Location of the ray within a grid block
     1        ,saveray, icolor, segname, Lray, tzprev, raypath,verbos)


c-----
c     END PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------
c--------------------------------------------------
c-----Write the xsd file 
c
      If (verbos) then
      write(LERR,*) 'Finished ray tracing, output final positions and '
      write(LERR,*) 'times.  dx =', dx,' n_picks=',n_picks
      write(LERR,*) 'Current, Next=',Current, Next
      write(LERR,*) 'x0     , Time'
      do i = 1,n_picks
        write(LERR,*) x0(i), z0(i)
      enddo
      Endif

      jerr = 0
      Do i = 1,n_picks
        trac(i)= x0(i)
        samp(i)= z0(i)
      enddo
      units(1)=1.
      units(2)=dx
      units(3)=dz

c write ray paths for viewing with xsd:
      if(raypath) then
      call write_ray_paths(inray, saveray, units, nrec, ntrac
     1  ,nsamp, Offset, rray, nrays, nray, N_picks,
     1  lray )
      endif

c Remove dead rays from the pick file:::::::
c       call xsdclean(rec,trac,samp,icolor,segname,npts,
c    1            units,offset, jseg,
c    1            nrec,ntrac,nsamp,jseg,maxpik,LERR,lpout,jerr,
c    1            normx(1,Current), normz(1,Current))
c     call wrpick(rec,trac,samp,icolor,segname,npts,units,offset,
c    1            nrec,ntrac,nsamp,jseg,maxpik,LERR,lpout,jerr)
      units(1)=1.
      units(2)=dx
      units(3)=dz
        call wxsdpicks(rec,trac,samp,icolor,segname,npts,
     1            units,offset, n_picks, max_segs,
     1            nrec,ntrac,nsamp,jseg,maxpik,LERR,lpout,jerr,
     1            tx, tz, pout)
c    1            normx(1,Current), normz(1,Current), pout)

      if(jerr.ne.0)then
         write(LERR,*)' Error writing xsd file  '
         write(LER,*)' Error writing xsd file  '
         go to 999
      endif
  999 continue
c-----
c     close data files
      call lbclos(luin)
c-----
      stop
      end
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine gcmdln(ntap,picks,pout,ns,ne, raypath,
     1                  irs,ire,Nsteps,dt,verbos,nseg)
c-----
c     get command arguments
c
c     ntap  - C*128    input velocity file name
c     picks - C*128    input xsd file name
c     pout  - C*128    output xsd file name
c     ns    - int      starting trace index
c     ne    - int      ending trace index
c     irs   - int      starting record index
c     ire   - int      ending record index
c     dt    - real     sample interval in milliseconds
c     verbos- logical  verbose output or not
c     raypath- Logical Keep raypaths or not
c     Nsteps - int     Number of steps per level of the velocity model
c     nseg  - int      Index of the segment from which to trace rays
c-----
#include <f77/iounit.h>
      character   ntap*(*), picks*(*), pout*(*)
      integer     ns, ne, irs, ire
      logical     verbos, raypath
      integer     argis, nseg
      real        dt

c-------
c Notice that the longest strings should come first so that shorter 
c  substrings contained in them will not cause the longer string to 
c  be parsed as the shorter string followed by a contiguous value.
c     last 2 arguments are values used when:
c     (1) if ONLY the key is present (no value attached to it)
c     (2) if NO key & no value are present
c-------
            call argi4 ('-Nsteps', Nsteps ,  1  ,  1)
            call argstr('-VEL',ntap , ' ', ' ')
            call argi4 ('-seg', nseg ,  1  ,  1)
            call argr4 ('-dt', dt ,1.0  ,1.0)
            call argi4 ('-ns', ns ,   1  ,  1)
            call argi4 ('-ne', ne ,   0  ,  0)
            call argi4 ('-rs', irs ,  1  ,  1)
            call argi4 ('-re', ire ,  0  ,  0)
            call argi4 ('-ismv',  ismv ,    0  ,  0)
            call argi4 ('-ismh',  ismh ,    0  ,  0)
            call argstr('-N',  picks, ' ', ' ')
            call argstr('-O',  pout , ' ', ' ')
            verbos =   (argis('-V') .gt. 0)
            raypath =  (argis('-raypath') .gt. 0)
      If (verbos) then
        write(LERR,*) '-VEL ',ntap
        write(LERR,*) '-N   ',picks
        write(LERR,*) '-O   ',pout
        write(LERR,*) '-seg ',nseg
        write(LERR,*) '-ns  ',ns
        write(LERR,*) '-ne  ',ne
        write(LERR,*) '-rs  ',irs
        write(LERR,*) '-re  ',ire
        write(LERR,*) '-ismv',ismv
        write(LERR,*) '-ismh',ismh
        write(LERR,*) '-dt  ',dt  
        write(LERR,*) '-V present'
        write(LERR,*) '-raypath ',raypath
      Endif
c-------
      return
      end
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)'time2depth_ray reads an xsd file and a ',
     &              'velocity field'
        write(LER,*)'and outputs an xsd file'
#ifdef CRAYSYSTEM
        write(LER,*)'See manual pages by typing:   tman time2depth_ray '
#else
        write(LER,*)'See manual pages by typing:   man time2depth_ray ',
     &              'or:   mman time2depth_ray'
#endif
        write(LER,*)'Execute time2depth_ray by typing time2depth_ray',
     &              'and the program parameters.'
        write(LER,*)'Note that each parameter is proceeded by -a ',
     &              'where "a" is a character(s) corresponding ',
     &              'to some parameter.'
        write(LER,*)'Users enter the following parameters, or use ',
     &              'the default values.'
        write(LER,*)' '
        write(LER,*)'-N[picks] : input xsd segment file (with trac'
        write(LER,*)'     coord = 2 way travel time in milliseconds.)'
        write(LER,*)'-VEL[ntap]: input velocity field'
        write(LER,*)'-O[pout]  : output xsd file'
        write(LER,*)'-ns[ns]   : start trace number      (default=1)'
        write(LER,*)'-ne[ne]   : end trace number        (default=all)'
        write(LER,*)'-rs[irs]  : start record number     (default=1)'
        write(LER,*)'-re[ire]  : end record number       (default=all)'
        write(LER,*)'-dt[dt]   : input sample rate in ms (default=1)'
        write(LER,*)'-seg[segno] : input segment number  (default=1)'
        write(LER,*)'    (Ray paths will be kept from this segment.)'
        write(LER,*)'-ismv[ismv]: input number of times to apply a 3  '
        write(LER,*)'       point smoother to the velocity in vertical'
        write(LER,*)'       direction.                   (default = 0)'
        write(LER,*)'-ismh[ismh]: input number of times to apply a 3  '
        write(LER,*)'       point smoother to the velocity in '
        write(LER,*)'       horizontal direction.        (default = 0)'
        write(LER,*)' '
        write(LER,*)'-raypath  output raypaths in xsd format '
        write(LER,*)'          (default is do not output raypaths). '
        write(LER,*)' '
        write(LER,*)'-V  include on command line for verbose printout'
        write(LER,*)' '
        write(LER,*)'usage:  '
        write(LER,*)'time2depth_ray -N[picks] -VEL[ntap] -O[pout]'
        write(LER,*)'        [-ns[ns]] [-ne[ne]] [-rs[irs]] [-re[ire]]'
        write(LER,*)'         [-dt[dt]][-seg[segno]] [-V] '
        write(LER,*)' '
        write(LER,*)
     :'***************************************************************'
      return
      end
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform, raypath,
     &               ntap,picks,pout,ns,ne,irs,ire,dt,verbos)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in velocity trace
c     nsi   - I*4     sample interval in line header
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ntap  - C*128   input velocity file name
c     picks - C*128   input xsd file name
c     pout  - C*128   output xsd file name
c     ns    - Int     starting record
c     ne    - Int     ending record
c     irs   - Int     starting trace
c     ire   - Int     ending trace
c     dt    - real    sample rate in milliseconds
c-----
#include <f77/iounit.h>

      integer     nsamp, nsi, ntrc, nrec,ns,ne,irs,ire
      real        dt
      character   ntap*128, picks*128, pout*128
      logical verbos, raypath

      write(LERR,*)' '
      write(LERR,*)' line header values after default check '
      write(LERR,*)' # of samples/trace          =  ', nsamp
      write(LERR,*)' sample interval in line hdr =  ', nsi
      write(LERR,*)' traces per record           =  ', ntrc
      write(LERR,*)' records per line            =  ', nrec
      write(LERR,*)' format of data              =  ', iform
      write(LERR,*)' input velocity dataset name =  ', ntap
      write(LERR,*)' input xsd file name         =  ', picks
      write(LERR,*)' output xsd file name        =  ', pout
      write(LERR,*)' starting record             =  ', irs
      write(LERR,*)' ending record               =  ', ire
      write(LERR,*)' starting trace              =  ', ns
      write(LERR,*)' ending trace                =  ', ne
      write(LERR,*)' sample rate in ms           =  ', dt
      write(LERR,*)' '
      if(verbos) write(LERR,*)' Verbose output is ON'
      if(raypath) write(LERR,*) 'Raypaths will be output.'
      return
      end
 

      Subroutine djv(z0, vel, Nx, Nz, x0,dx,
     1    npicks, jseg, npts, tx, tz)
c Use this subroutine to compute initial ray directions from picked 
c surface locations.
      Real z0(npicks),  Vel(Nx*Nz)
      Real   x0(npicks),   tx(npicks), tz(npicks)
      Integer npts(jseg),  i, j
c-----compute normal directions---------------------------------------------
       i   = 0
       Do j = 1,jseg
c---------use slope at the midpoint for intervals---------
         do ii = 1, npts(j)-1
            i = i + 1
            tx(i) = z0(i+1) - z0(i)
            tz(i) = x0(i  ) - x0(i+1)
         Enddo
C--------fixup end values with rough guess for directions--------------------
         i = i+1
         tx(i) = z0(i) - z0(i-1)
         tz(i) = x0(i-1)-x0(i)
       Enddo
       if (npicks .ne. i) then
         write(*,*) 'Error in initial slope computation'
         write(*,*) 'npicks is ',npicks,' But we found ',i,' picks.'
         stop
       endif
C--------Normalize the ray directions----------------------------------------
C-------And fix bad directions, (downward is bad)----------------------------
      i = 0
      do j = 1, jseg
        tx1 = 0.0
        tz1 = -1.0
        do ii = 1, npts(j)
        i = i + 1
         IF (tz(i) .lt. 0.) then
           s     = 1.0  /sqrt(tx(i)**2 +tz(i)**2)
           tx(i) = tx(i)*s
           tz(i) = tz(i)*s
           tx1   = tx(i)
           tz1   = tz(i)
         Else
           tx(i) = tx1
           tz(i) = tz1
         Endif
       Enddo
      Enddo
c ------interpolate depth to the midpoint of intervals..................
      i = 0
      do j = 1, jseg
        do ii = 1, npts(j)-1
        i = i + 1
        x0(i) = (x0(i)+x0(i+1))*.5
        z0(i) = (z0(i)+z0(i+1))*.5
        enddo
        i = i+1
      enddo

      return
      end
      Subroutine lsq(z0, vel, Nx, Nz, x0,dx,
     1    npicks, jseg, npts, tx, tz)
c Use this subroutine to compute initial ray directions from picked 
c surface locations using a least square fit to 3 points about
c the point of interest.
C Formulae used:  Square Error = Sum(b*xi + a -zi)**2
C b = (N*sum(xizi)-(sum(xi))**2)/(n*sum(xi**2)-sum(xi)*sum(zi))
C 
      Real z0(npicks),  Vel(Nx*Nz)
      Real   x0(npicks),   tx(npicks), tz(npicks)
      Integer npts(jseg),  i, j
c-----compute normal directions---------------------------------------------
c First we define some usefuls statement function:
       dx0(i) = x0(i+1)-x0(i)
       v(i) = vel(1+Nz*ifix((x0(i+1)+x0(i))/(2*dx)))
       dt(i)= (z0(i+1)-z0(i))*dx0(i)/v(i)
       dx2(i)= dx0(i)**2/v(i)**2
       cosz(xx) = sqrt(1.-xx**2) 
c Next we write some diagnostic information:
       i   = 0
       Do j = 1,jseg
c---------use a rough guess for the first interval
         i     = i + 1
         tx(i) = sinz((z0(i+1)-z0(i))*v(i)/dx0(i))
         tz(i) = cosz(tx(i))
c--------use a least square method for most of the intervals
c-------We can save some operations by using runniing sums here.
         do ii = 2, npts(j)-1
            i = i + 1
            sx2   = dx2(i-1) + dx2(i)
            stx   = dx0(i-1)*dt(i-1)/v(i-1) + dx0(i)*dt(i)/v(i)
            tx(i) = sinz(stx/sx2) 
            tz(i) = cosz(tx(i))
         Enddo
C--------fixup end values with rough guess for directions--------------------
         i = i+1
         tx(i) =sinz(dx0(i-1)*dt(i-1)/(v(i-1)*dx2(i-1)))
         tz(i) = cosz(tx(i))
       Enddo
       if (npicks .ne. i) then
         write(*,*) 'Error in initial slope computation'
         write(*,*) 'npicks is ',npicks,' But we found ',i,' picks.'
         stop
       endif
C--------Normalize the ray directions----------------------------------------

      return
      end
       Function sinz(tx)
       Real t0
       Save t0
       Data t0 /0./
        if (abs(tx) .lt. 1. ) then
            sinz = tx 
            t0   = tx
         else
            sinz = t0 
         endif
         return
         end

        Subroutine smoothslow(vel, slow, nx, nz, ismv, ismh)
      Real vel(nx,nz), slow(nx,nz), scale
      Parameter (scale = 3.)
      Integer nx,nz,ismv, ismh, ismhs, ismvs
      if (ismh .gt. 0) then
        Do ismhs = 1,ismh
          Do i=2,nx-1
            Do j = 1,nz
              slow(i,j) =1./vel(i-1,j) +1./vel(i,j)+1./vel(i+1,j)
            Enddo
          Enddo
          Do i=2,nx-1
            Do j = 1,nz
              vel(i,j) = scale/slow(i,j)
            Enddo
          Enddo
        Enddo
      Endif
      If (ismv .gt. 0) then
        Do ismvs = 1,ismv
          Do i=1,nx
            Do j = 2,nz-1
              slow(i,j) =1./vel(i,j-1) +1./vel(i,j)+1./vel(i,j+1)
            Enddo
          Enddo
          Do i=1,nx
            Do j = 2,nz-1
              vel(i,j) = scale/slow(i,j)
            Enddo
          Enddo
        Enddo
      Endif
      Return
      End
      subroutine galloc_msg(errcode, errcode2,n,LERR)
      Integer errcode, errcode2, n
      if(errcode.ne.0) then
        Write(LERR,*) 'Galloc error ',errcode
        Write(LERR,*) 'Error location ',n
        Stop
      Endif
      Return
      End
      subroutine write_ray_paths(inray, saveray, units, nrec, ntrac
     1  ,nsamp, Offset, rray, nrays, nray, N_picks,
     1  lray )
      Implicit none
      Integer N_picks, nrays, nray, ntrac, nrec, nsamp, n_rays
      Integer max_ray, i, j, k, l
      real rray(4,nrays,nray), units(3), Offset(3)
      real one
      Parameter (one=1.)
      Logical saveray(N_picks)
      Character*20 segname
      integer inray(N_picks), lray(N_picks)
C-------write rays to a ray file------------------------
       Open(25,file='ray.path')
       n_rays  = 0
       max_ray = 0
       Do L = 1,n_picks
         If (saveray(L)) then
         n_rays = n_rays+1
         max_ray = max(max_ray, inray(L))
         Endif
       enddo
       Write(25,11) units, nrec, ntrac, nsamp, Offset,
     1   n_rays, max_ray

      do L = 1,N_Picks
      if (saveray(L)) then
        I = lray(L)
        segname= 'NO_PICK_NAME_HERE' 
        write(25,20) I, segname, mod(i,10), inray(L)


      Do j = 1,inray(L)
      write(25,'(F12.6,4F13.6)') one, (rray(k,I,j),k=1,4)
      enddo
      endif
      Enddo
      close (25)
      Return
11    Format('Units',3F13.6,3I6,' Offset',3F13.6, ' Count',2I6)
20    FORMAT('Segment = ',I5,' Name ',A20,2X,'color = ',
     1        I5,' picks =',I6)
      End
