C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C
C     PROGRAM MODULE xwmig
C
C**********************************************************************C
C
C xwmig ...
c input data is one or more seismic traces from surface-surface,
c vsp or crosswell data set.  Migration is constant velocity,
c and requires header words (sx,gx,sy,gy) to be set.
c (sy,gy) are interpreted as source and receiver DEPTHS.
C xwmig performs a migration in the t-x domain  
C and writes the results to otap.
C
C REQUIRED trace header fields:
C
C  Information                Halfword number      Header nemonic 
C  ___________                _______________      __________________
C
C  x-coordinate of source        23-24                 SrPtXC 
C  z-coordinate of source        25-26                 SrPtEl
C  x-coordinate of receiver      27-28                 RcPtXC
C  z-coordinate of receiver      29-30                 GrpElv 
C
C  z-coordinate of source        25-26                 SrPtYC -- old
C  z-coordinate of receiver      29-30                 RcPtYC -- old
C
C REQUIRED command line arguments:
C
C  -xmin[] -xmax[] -dx[]
C
C
C
C
C SUBROUTINE CALLS: RTAPE, HLH, WRTAPE, SAVE
C
C**********************************************************************C
C
C     DECLARE VARIABLES
C

#include <save_defs.h>

#include <localsys.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      INTEGER * 2 ITR ( SZLNHD )
      INTEGER     LHED( SZLNHD )
      INTEGER     NSAMP, NSI, NTRC, NREC, IFORM
      INTEGER     LUIN , LBYTES, NBYTES,obytes
      integer     jrec, jtr
      integer     argis

      integer     nsampo, ns, ne, irs, ire

      real        chop, delay, dt, dx, dz, xs, xg, zs, zg
      integer     oper
      real        xmin, xmax, zmin, zmax, v1, v2, dv, xint
      integer     nq, nt, nx, nz, nv

      real        tabl1 (SZLNHD), tabl2(SZLNHD)
      real        zz(4*SZLNHD)
      integer     iz(SZLNHD)

c     .. wkadr1 = indata [real] = indata(nt)
c     .. wkadr2 = outdata[real] = outdata(nz,nx)
c     .. wkadr3 = tsin   [real] = tsin(nq)
c     .. wkadr4 = tcos   [real] = tsin(nq)

      REAL        xtr(SZLNHD), work(SZLNHD) 
      real        indata, outdata, tsin, tcos 
      pointer     (wkadr1, indata(1) )
      pointer     (wkadr2, outdata(1))
      pointer     (wkadr3, tsin(1)   )
      pointer     (wkadr4, tcos(1)   )

      CHARACTER   NAME * 29, ntap * 100, otap * 100
#include <f77/pid.h>
      logical     verbos,query,heap,first,depth
 
c     EQUIVALENCE ( ITR(129), xtr (1) )
      EQUIVALENCE ( ITR(  1), LHED(1) )
      DATA NAME     /'XWMIG: cross-well migration'/
      DATA LUIN / 1 /, LBYTES / 0 /, NBYTES / 0 /
      DATA first/.true./


c---------------------------------
c  get online help if necessary
c---------------------------------
      query = (argis('-?').gt.0 .or. argis('-h').gt.0)
      if ( query ) then
           call help ()
           stop
      endif

c------------------------------------
c  open printout file
c------------------------------------
#include <f77/open.h>

C**********************************************************************C
C     READ PROGRAM PARAMETERS FROM COMMAND LINE ARGUMENT STRING
C**********************************************************************C
      call cmdln (ntap, otap, verbos, xint, depth, nv,
     :            v1, v2, xmin, xmax, dx, zmin, zmax, dz, 
     :            chop, delay, oper, nq)


c  .. now available: v,xmin,xmax,dx,zmin,zmax,dz,chop,delay,oper, nq

c     ... output data image is outdata(nz,nx)
      nx = (xmax - xmin) / dx + 1
      nx = abs(nx)
      nz = (zmax - zmin) / dz + 1
      nz = abs(nz)
      if (nv .gt. 1) then
         dv = (v2 - v1) / float (nv-1)
      else
         dv = 0.
      endif

c     ... echo vital info to user
      write(LERR,*)'xmin,xmax = ', xmin, xmax
      write(LERR,*)'nx,dx = ',     nx, dx
      write(LERR,*)'zmin,zmax = ', zmin, zmax
      write(LERR,*)'nz,dz = ',     nz, dz

C**********************************************************************C
C     READ AND UPDATE LINE HEADER,
C     WRITE LINE HEADER, SAVE KEY PARAMETERS.
C**********************************************************************C
      call getln ( luin, ntap, 'r', 0 )
      call getln (luout, otap, 'w', 1 )

      lbytes = 0
      call rtape ( luin, itr, lbytes )
      if(lbytes .eq. 0) then
         write(LERR,*)'XWMIG: no header read on unit ',ntap
         write(LERR,*)'FATAL'
         write(LERR,*)'Check existence of file & rerun'
         stop
      endif
      CALL HLHprt    ( ITR , LBYTES, NAME, 29, LERR        )

      call savelu('SrPtXC',ifmt_SrPtXC,l_SrPtXC,ln_SrPtXC,TRACEHEADER)
      call savelu('SrPtEl',ifmt_SrPtEl,l_SrPtEl,ln_SrPtEl,TRACEHEADER)
      call savelu('RcPtXC',ifmt_RcPtXC,l_RcPtXC,ln_RcPtXC,TRACEHEADER)
      call savelu('GrpElv',ifmt_GrpElv,l_GrpElv,ln_GrpElv,TRACEHEADER)

c---------------------------------
c  save key header values
#include <f77/saveh.h>
      if(nsamp .gt. SZLNHD) nsamp=SZLNHD
c----------------------------------


c-----
c     ensure that command line values are compatible with data set
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

          dt = float(nsi) * unitsc

      fac = float(nz) / float(nsamp)
      do  300  j = 1, nz
         tabl1(j) = float( j )
300   continue
 
      do  301  j = 1, nsamp
         tabl2(j) = float( j )  * fac
301   continue


c------------------------------------------------------
c  save headers: exchange # traces/rec & # samples/rec
       if (depth) then
          nsampo = nz
       else
          nsampo = nsamp
       endif
       ntro   = nx
       ntr    = ntro
       nrecc  = 1
       call savew( itr, 'NumSmp', nsampo, LINHED)
       call savew( itr, 'NumTrc', ntro  , LINHED)
       call savew( itr, 'NumRec', nrecc , LINHED)

c----------------------------------------------------------------
c  change output bytes to reflect change from time to # traces
      obytes = SZTRHD + SZSMPD * nsampo

c---------------------------------
c  verbos printout
        write(LERR,*)
        write(LERR,*)' Values read from input data set line header'
        write(LERR,*)
        write(LERR,*) ' # of Samples/Trace =  ', nsamp
        write(LERR,*) ' Sample Interval    =  ', nsi  
        write(LERR,*) ' Traces per Record  =  ', ntrc
        write(LERR,*) ' Records per Line   =  ', nrec
        write(LERR,*) ' Format of Data     =  ', iform
        write(LERR,*) ' Output # samples   =  ', nsampo
        write(LERR,*) ' Output # traces    =  ', ntr
        write(LERR,*) ' Output records     =  ', nrecc
        write(LERR,*) ' starting v         =  ', v1
        write(LERR,*) ' ending v           =  ', v2
        write(LERR,*) ' delta v            =  ', dv
        write(LERR,*) ' output xmin        =  ', xmin
        write(LERR,*) ' output xmax        =  ', xmax
        write(LERR,*) ' output dx          =  ', dx
        write(LERR,*) ' output zmin        =  ', zmin
        write(LERR,*) ' output zmax        =  ', zmax
        write(LERR,*) ' output dz          =  ', dz
        write(LERR,*) ' chop               =  ', chop
        write(LERR,*) ' delay              =  ', delay
        write(LERR,*) ' oper              =  ', oper 
        write(LERR,*) ' fractional interp =  ', xint

        write(LERR,*)' '
        write(LERR,*)'interpolation table (input)'
        write(LERR,*)(tabl1(ii),ii=1,nz)
        write(LERR,*)' '
        write(LERR,*)'interpolation table (output)'
        write(LERR,*)(tabl2(ii),ii=1,nsamp)
        write(LERR,*)' '
        write(LERR,*)'Velocities in image:'
        do  j = 1, nv
            v = v1 + (j-1) * dv
            write(LERR,*)j,v
        enddo
        write(LERR,*)' '


c-----------------------------------------------
c  adjust historical line header & write header
      call savhlh ( itr, lbytes, lbyout )
 
      call wrtape(luout,itr,lbyout)

c------------------------------------------------

c---------------------------------------------------
c  malloc only space we're going to use
      nt = nsamp
      heap = .true.

      call galloc (wkadr1, nt*SZSMPD, errcod, abort)
       if (errcod .ne. 0) heap = .false.
       if (.not. heap) write(LERR,*)'Unable to allocate wkadr1'
      call galloc (wkadr2, nz*nx*SZSMPD, errcod, abort)
       if (errcod .ne. 0) heap = .false.
       if (.not. heap) write(LERR,*)'Unable to allocate wkadr2'
      call galloc (wkadr3, nq*SZSMPD, errcod, abort)
       if (errcod .ne. 0) heap = .false.
       if (.not. heap) write(LERR,*)'Unable to allocate wkadr3'
      call galloc (wkadr4, nq*SZSMPD, errcod, abort)
       if (errcod .ne. 0) heap = .false.
       if (.not. heap) write(LERR,*)'Unable to allocate wkadr4'

      if (.not. heap) then
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) nt*SZSMPD,'  bytes'
         write(LERR,*) nz*nx*SZSMPD,'  bytes'
         write(LERR,*) nq*SZSMPD,'  bytes'
         write(LERR,*) nq*SZSMPD,'  bytes'
         go to 999
      else
         write(LERR,*)'Allocated workspace:'
         write(LERR,*) nt*SZSMPD,'  bytes'
         write(LERR,*) nz*nx*SZSMPD,'  bytes'
         write(LERR,*) nq*SZSMPD,'  bytes'
         write(LERR,*) nq*SZSMPD,'  bytes'
      endif
c---------------------------------------------------

c     ... trigtable frequency vector w(iw)
      call trigtable ( tsin, tcos, nq )
 
C**********************************************************************C
C
C     READ traces, DO XWMIG, WRITE OUTPUT RECORD
C
C**********************************************************************C

c-------------------------------
c     loop over records
c-------------------------------
      do 100 jrec = 1, nrec 

c-------------------------------
c     loop over input traces within this record
c-------------------------------
           DO 99 jtr = 1, ntrc 

                 nbytes = 0
                 CALL RTAPE  ( LUIN , ITR, NBYTES         )
                 if(nbytes .eq. 0) then
                    write(LERR,*)'WARNING'
                    write(LERR,*)'End of file on input:'
                    write(LERR,*)'  rec= ',jrec,'  trace= ',jtr
                    go to 59
                 endif
                 call vmov (lhed(ITHWP1), 1, xtr, 1, nsamp)

c                ****** get xs,zs,xg,zg from header *********
c
c                ... source  and receiver (x,z) coordinates

                  call saver2(lhed,ifmt_SrPtXC,l_SrPtXC, ln_SrPtXC,
     1                        ixs , TRACEHEADER)
                  call saver2(lhed,ifmt_SrPtEl,l_SrPtEl, ln_SrPtEl,
     1                        izs , TRACEHEADER)


                  call saver2(lhed,ifmt_RcPtXC,l_RcPtXC, ln_RcPtXC,
     1                        ixg , TRACEHEADER)
                  call saver2(lhed,ifmt_GrpElv,l_GrpElv, ln_GrpElv,
     1                        izg , TRACEHEADER)

c                 ... convert integers returned by saver2 to real
                  xs = ixs
                  zs = izs
                  xg = ixg
                  zg = izg

                  if (verbos .or. nv .gt. 1)
     1            write(LER,*)'migrating trc= ',jtr,' src x,z= ',
     2                        ixs,izs,' rcvr x,z= ',ixg,izg


c                ... check header words were set
                 if (xs .eq. 0. .and. xg .eq. 0. .and.
     :               zs. eq. 0. .and. zg .eq. 0.) then
                    write(LERR,*)'XWMIG: xs=xg=zs=zg=0 '
                    write(LERR,*)'FATAL'
                    write(LERR,*)'Header sx,sy,gx,gy must be set.'
                    write(LERR,*)'Header sy,gy are src/rec depth.'
                    stop
                 endif

c                ... move trace to indata vector
                 call vmov (xtr(1),1, indata(1),1,nt)
c		 ... xwmig of one trace, 
c                ... returned 'outdata' is the image 

c                ... allow for negative x's and z's
                 xs = abs(xs)
                 zs = abs(zs)
                 xg = abs(xg)
                 zg = abs(zg)
                 xmin = abs(xmin)
                 dx = abs(dx)
                 zmin = abs(zmin)
                 dz = abs(dz)

                 do  iv = 1, nv
                     v = v1 + (iv-1) * dv
      	             call xwmig (indata, outdata, tsin, tcos, 
     :               v, chop, xs, zs, xg, zg, xmin, dx, zmin, dz, dt,
     :               nq, nt, nx, nz, delay,oper,xint)
                 enddo

                 ic = jtr - ns + 1
 
c          ... end trace loop
   99      continue 

c          ... ntrk: traces processed from this record
   59      ntrk = ic

           if (ntrk .ne. ntrc) then
               write(LERR,*)'WARNING:'
               write(LERR,*)'read ',ntrk,' traces from record ',jrec
               write(LERR,*)'instead of ',ntrc
               write(LERR,*)'processing continuing'
           endif

c        ... end record loop
  100    continue

c------------------------------------------------
c  extract output
c  data from vector
           icinit = 1
           DO 199 jtr = 1, nx
                 istrc = (jtr-1) * nz
                 call vmov (outdata(istrc+1),1,work,1,nz)

                 if (depth) then
                    call vmov (work, 1, lhed(ITHWP1), 1, nz)
                 else
                    call vclr   (xtr, 1, nsamp)
                    call fcuint (tabl1, work, nz,
     1                           tabl2, xtr , nsamp, iz, zz, icinit)
                    icinit = 0
                    call vmov (xtr ,1, lhed(ITHWP1), 1, nsamp)
                 endif

                 CALL WRTAPE  ( LUOUT , ITR, OBYTES         )
  199      CONTINUE
c------------------------------------------------



  999 continue
       call lbclos(luin)
       call lbclos(luout)
      END

c------------------------------
c  online help section
c------------------------------
      subroutine  help
#include <f77/iounit.h>

        write(LER,*)' '
        write(LER,*)'Command Line Arguments for XWMIG: xwell migration'
        write(LER,*)' '
        write(LER,*)'Input....................................... (def)'
        write(LER,*)' '
        write(LER,*)'-N[ntap]   -- input data set'
        write(LER,*)'-O[otap]   -- output data set'
        write(LER,*)' '
        write(LER,*)'Mode 1:'
        write(LER,*)'-vmigc[v]  -- const. migration velocity     (none)'
        write(LER,*)'Mode 2:'
        write(LER,*)'-vmigs[vs] -- start migration velocity      (1500)'
        write(LER,*)'-vmige[ve] -- end migration velocity        (6000)'
        write(LER,*)'-nv[nv]    -- number of migration vels        (50)'
        write(LER,*)' '
        write(LER,*)'-xmin[xmin]-- min x in mig image            (NONE)'
        write(LER,*)'-xmax[xmax]-- max x in mig image            (NONE)'
        write(LER,*)'-dx[dx]    -- x-step in mig image           (NONE)'
        write(LER,*)'-zmin[zmin]-- min z in mig image               (0)'
        write(LER,*)'-zmax[zmax]-- max z in mig image  (abs(xmax-xmin))'
        write(LER,*)'-dz[dz]    -- z-step in mig image             (dx)'
        write(LER,*)'-chop[chop]-- smallest abs(amp) to process     (0)'
        write(LER,*)'-delay[]  -- processing delay after 1st arrival(0)'
        write(LER,*)'-oper[]    -- single or double sign operator(2)'
        write(LER,*)'-nq[nq]   -- points in trig table on [0,2Pi](2000)'
        write(LER,*)'-f[f]      -- fractional interpolation ampl.  (.5)'
        write(LER,*)' '
        write(LER,*)'-D         -- depth output'
        write(LER,*)'-V         -- verbos printout'
        write(LER,*)' '
        write(LER,*)'Usage:'
        write(LER,*)'     xwmig -N[] -O[] [ -D -V ]'
        write(LER,*)'           [ -vmigc[]'
        write(LER,*)'             -vmigs[] vmige[] -nv[] ]'
        write(LER,*)'           -xmin[] -xmax[] -dx[]'
        write(LER,*)'           -zmin[] -zmax[] -dz[]'
        write(LER,*)'           -chop[] -delay[] -oper[] -nq[] -f[]'
        write(LER,*)' '

      return
      end

c-----
c     get command arguments
c
c     ntap  - C*100  input file name
c     otap  - C*100  output file name
c  verbos   - L      verbose output or not
c       v   - R      velocity
c    xmin   - R      min x in migrated image 
c    xmax   - R      max x in migrated image  
c      dx   - R      x-step in migrated image 
c    zmin   - R      min z in migrated image 
c    zmax   - R      max z in migrated image  
c      dz   - R      z-step in migrated image 
c    chop   - R      smallest abs(amp) to process
c   delay   - R       
c   oper    - I       
c-----
      subroutine cmdln (ntap, otap, verbos, xint, depth, nv,
     :            v1, v2, xmin, xmax, dx, zmin, zmax, dz, 
     :            chop, delay, oper, nq)
#include <f77/iounit.h>
      character  ntap*(*), otap*(*)
      integer    argis
      logical    verbos, depth
      real       v1,v2,xmin,xmax,dx,zmin,zmax,dz,chop,delay,xint
      integer    oper, nq

      call argstr('-N',ntap,' ',' ') 
      call argstr('-O',otap,' ',' ') 
      depth  = (argis('-D') .gt. 0)
      verbos = (argis('-V') .gt. 0)

      call argr4('-vmigc',v,0.,0.)
      if (v .ne. 0) then
         v1 = v
         v2 = v
         nv = 1
      else
         call argr4('-vmigs',v1,1500.,1500.)
         call argr4('-vmige',v2,6000.,6000.)
         call argi4('-nv',nv,50,50)
      endif

      call argr4('-xmin',xmin,-999.,-999.)
      call argr4('-xmax',xmax,-999.,-999.)
      call argr4('-dx',dx,-999.,-999.)
      call argr4('-zmin',zmin,-999.,-999.)
      call argr4('-zmax',zmax,-999.,-999.)
      call argr4('-dz',dz,-999.,-999.)
      call argr4('-chop',chop,0.,0.)
      call argr4('-delay',delay,0.,0.)
      call argr4('-f',xint,0.5,0.5)
      call argi4('-oper',oper,2,2)
      call argi4('-nq',nq,2000,2000)

      if (xmin .eq. -999) then
        write(LERR,*) 'XWMIG: Must enter xmin - FATAL.'
        stop
      endif

      if (xmax .eq. -999) then
        write(LERR,*) 'XWMIG: Must enter xmax - FATAL.'
        stop
      endif

      if (dx .eq. -999) then
        write(LERR,*) 'XWMIG: Must enter dx - FATAL.'
        stop
      endif

      if (zmin .eq. -999) then
        zmin = 0
      endif

      if (zmax .eq. -999) then
        zmax = abs(xmax - xmin)
      endif
        zmax = abs(zmax)
        zmin = abs(zmin)

      if (dz .eq. -999) then
        dz = dx
      endif

      if (zmin .gt. zmax ) then
       tmp  = zmax
       zmax = zmin
       zmin = tmp
      endif

      if (xmin .gt. xmax ) then
       tmp  = xmax
       xmax = xmin
       xmin = tmp
      endif

c     ... try to avoid stupid mistakes
      if (chop .lt. 0.) then chop = abs(chop) 
      if (delay .lt. 0.) then delay = abs(delay) 
      if (oper .ne. 1 .and. oper .ne. 2) then oper = 2 
      if (nq .lt. 0.) then nq = abs(nq) 

      return
      end
