C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c grot reads 2-component interleaved seismic trace data from an input 
c file, performs a global rotation on pairs of traces in the data 
c stream, and writes the results to an output file
c 
c    get machine dependent parameters

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

c declare standard USP variables
 
      integer     itr ( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout, luflat
      integer     irs, ire, ns, ne
      integer     argis, Nangles
      integer     Records
      pointer     (wkRecords, Records(1))

      character   ntap * 512, otap * 512, name*4, flattap*512
      character   hdrwrd * 6

      logical     verbos, query, flt, horrot, live1, live2, EW

      real        Angles
      pointer     (wkAngles, Angles(1))
      real        RotationAngle, rad
      real        xtrv ( SZLNHD ), xtrh ( SZLNHD )
      real        otrv ( SZLNHD ), otrh ( SZLNHD )

      integer     RecNum, StaCor, ierr, ierrs, iabort
 
      data lbytes / 0 /, nbytes / 0 /, name/'GROT'/
      data iabort/0/


c read program parameters from command line card image file

      query = ( argis ( '-?' ) .gt. 0 .or. argis ( '-h' ) .gt. 0)
      if ( query )then
         call help()
         stop
      endif

c initialize variables 

      rad = 3.14159265/180.
      
c open printout files

#include <f77/open.h>
 
      call gcmdln ( ntap, otap, flattap, ns, ne, irs, ire, 
     :     RotationAngle, hdrwrd, verbos, flt, horrot, EW)

c convert angle to sin/cosine

      if (.not. horrot) then
        cs = cos (RotationAngle * rad)
        sn = sin (RotationAngle * rad)
        write(LERR,*)'sin/cos factors =  ',sn,cs
      endif

c get logical unit numbers for input and output of seismic data

      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c get logical unit for input flatfile if used and determine the number of lines
c in file

      if ( flattap .ne. ' ' ) then
         leflat = lenth(flattap)
         call alloclun ( luflat )
         open ( unit = luflat, file = flattap(1:leflat), status = 'old',
     :        err = 990 )
         Nangles = 0
         do while (1.eq.1)
            read (luflat, *, err=111, end = 101) irec, iang
            Nangles = Nangles + 1
         enddo
         go to 101
111      continue
         write(LERR,*)'FATAL ERROR in grot:'
         write(LERR,*)'Error reading flat file ',flattap(1:leflat)
         write(LER ,*)'FATAL ERROR in grot:'
         write(LER ,*)'Error reading flat file ',flattap(1:leflat)
         call ccexit (666)
101      continue
         rewind luflat
      endif

      item = Nangles * SZSMPD
      ierrs = 0
      call galloc (wkRecords, item, ierr, iabort)
      ierrs = ierrs + ierr
      call galloc (wkAngles , item, ierr, iabort)
      ierrs = ierrs + ierr
      if (ierrs .ne. 0) then
         write(LERR,*)'FATAL ERROR in grot:'
         write(LERR,*)'Cannot allocate memory for ',Nangles,' angles'
         write(LER ,*)'FATAL ERROR in grot:'
         write(LER ,*)'Cannot allocate memory for ',Nangles,' angles'
         call ccexit (666)
      endif

c read line header of input

      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'GROT: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif

c load standard variables from lineheader

      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif


      call hlhprt (itr, lbytes, name, 4, LERR)

      if (mod(ntrc,2) .ne. 0) then
         write(LERR,*)'Data set contains odd number traces/rec'
         write(LERR,*)'Is this a 2-component data set?'
         write(LERR,*)'use program mergn with -m flag to multiplex'
         write(LERR,*)'2 data sets into an interleaved stream'
         write(LERR,*)'          --- FATAL END --- '
         stop
      endif

c ensure that dataset is compatible with command line options requested

      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

c modify line header to reflect actual number of traces output

      nrecc=ire - irs+1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr=ne-ns+1
      call savew(itr, 'NumTrc', jtr  , LINHED)

c number output bytes

      obytes = SZTRHD + nsamp * SZSMPD

c build pointers for later access of trace header entries

      call savelu ( 'RecNum', fmt_RecNum, l_RecNum, ln_RecNum, 
     :     TRACEHEADER )
      call savelu ( 'StaCor', fmt_StaCor, l_StaCor, ln_StaCor, 
     :     TRACEHEADER )
      call savelu ( 'SrPtXC', fmt_SrPtXC, l_SrPtXC, ln_SrPtXC, 
     :     TRACEHEADER )
      call savelu ( 'SrPtYC', fmt_SrPtYC, l_SrPtYC, ln_SrPtYC, 
     :     TRACEHEADER )
      call savelu ( 'RcPtXC', fmt_RcPtXC, l_RcPtXC, ln_RcPtXC, 
     :     TRACEHEADER )
      call savelu ( 'RcPtYC', fmt_RcPtYC, l_RcPtYC, ln_RcPtYC, 
     :     TRACEHEADER )
      if (hdrwrd(1:) .ne. ' ')
     :call savelu ( hdrwrd, fmt_hdrwrd, l_hdrwrd, ln_hdrwrd, 
     :     TRACEHEADER )

c output line header

      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout  )

c verbose output of all pertinent information before processing begins

c     if( verbos ) then
         call verbal ( nsamp, nsi, ntrc, nrec, iform, RotationAngle, 
     :        ntap, otap, flattap, Nangles, hdrwrd, horrot, EW )
c     endif

c compute sample interval in secs take care of micro secs if necessary

      dt = real (nsi) * unitsc

c read flatfile and load global rotation parameters to memory if required

      if ( flattap .ne. ' ') then
         do i = 1, Nangles
            Records(i) = 0
            Angles(i) = 0.0
         enddo
         call GetFlat(luflat, Angles, Records, Nangles )
      endif


c BEGIN PROCESSING

c skip unwanted records

      call recskp(1,irs-1,luin,ntrc,itr)

c process desired records

      do 1000 jj = irs, ire
 
c skip to start trace

         call trcskp(jj,1,ns-1,luin,ntrc,itr)
         ic = 0
         do 1001  kk = ns, ne-1, 2

            nbytes = 0

c read in component "v"

            call rtape( luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on input:'
               write(LERR,*)'  rec= ',jj,'  trace= ',kk
               go to 999
            endif

c put data into work vector

            call saver2 ( itr, fmt_RecNum, l_RecNum, ln_RecNum, RecNum, 
     :           TRACEHEADER )
            call saver2 ( itr, fmt_StaCor, l_StaCor, ln_StaCor, StaCor, 
     :           TRACEHEADER )

            if (StaCor .ne. 30000) then
               call vmov ( itr(ITHWP1), 1, xtrv(1), 1, nsamp )
               live1 = .true.
            else
               call vclr ( xtrv, 1, nsamp )
               live1 = .false.
            endif

c read in other component "h"
            
            call rtape( luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on input:'
               write(LERR,*)'  rec= ',jj,'  trace= ',kk
               go to 999
            endif

c----
c  Note that for -H [horizontal mode] where both components are horizontal
c  the first trace of the pair is assumed associated with the X-axis (east)
c  of a cartesian system. Given the angle defined by the R/S vector (made
c  w.r.t. the X-axis and given equal amplitudes on both components at a
c  particular time then a rotation by that angle will result in all the
c  amplitude being put onto the inline component and zero ampl on the
c  component perpendicular to the R/S vector, at least for phones that 
c  are oriented NS/EW. If the phones are not aligned with the cardinal
c  directions then another rotation angle must be supplied describing the
c  rotation of the phones away from this alignment. This angle may be input
c  as a global value, in a flat file, or most properly in a trace header word.
c  Hopefully the misalignment will be supplied by the contractor using say
c  first break analysis.
c----
            
c put data into work vector

            call saver2 ( itr, fmt_StaCor, l_StaCor, ln_StaCor, StaCor, 
     :           TRACEHEADER )
            if (StaCor .ne. 30000) then
               call vmov ( itr(ITHWP1), 1, xtrh(1), 1, nsamp )
               live2 = .true.
            else
               call vclr (xtrh,1,nsamp)
               live2 = .false.
            endif

c interpolate rotation angle for this record if required

            if ( flattap .ne. ' ' ) then
               call InterpAngle ( Angles, Records, Nangles, RecNum, 
     :              RotationAngle ) 

c convert angle to sin/cosine

               if (.not. horrot) then
                 cs = cos (RotationAngle * rad)
                 sn = sin (RotationAngle * rad)
                 if (verbos)
     :           write(LERR,*)'Record = ',RecNum,
     :                ' sin/cos factors =  ',sn,cs
               endif
            elseif ( hdrwrd .ne. ' ' ) then

              if (flt) then
               call saver2 ( itr, fmt_hdrwrd, l_hdrwrd, ln_hdrwrd,
     :                       xhdrwrd, TRACEHEADER )
               RotationAngle = xhdrwrd
              else
               call saver2 ( itr, fmt_hdrwrd, l_hdrwrd, ln_hdrwrd,
     :                       ihdrwrd, TRACEHEADER )
               RotationAngle = ihdrwrd
cdb            write(0,*)'ihdrwrd= ',ihdrwrd
              endif
               if (.not. horrot) then
                 cs = cos (RotationAngle * rad)
                 sn = sin (RotationAngle * rad)
                 if (verbos)
     :           write(LERR,777)RecNum,RotationAngle,sn,cs
777   format('Record = ',i6,' Angle= ',f5.1,' sin/cos ',2f10.4)
               endif

            endif

c----
c   rotate XY phones:  read in the src and rcvr XYs and compute what
c   the S/R angle (measured from the east-pointing direction counter-
c   clockwise being positive).  Then use a global rotation angle or one
c   read from the trace headers to rotate the phones by the difference
c   between these 2 angles. Angles are in degrees.
c----
            if (horrot) then
               call saver2 ( itr, fmt_SrPtXC, l_SrPtXC, ln_SrPtXC,
     :                       isrcx, TRACEHEADER )
               call saver2 ( itr, fmt_SrPtYC, l_SrPtYC, ln_SrPtYC,
     :                       isrcy, TRACEHEADER )
               call saver2 ( itr, fmt_RcPtXC, l_RcPtXC, ln_RcPtXC,
     :                       ircvx, TRACEHEADER )
               call saver2 ( itr, fmt_RcPtYC, l_RcPtYC, ln_RcPtYC,
     :                       ircvy, TRACEHEADER )
               deltax = isrcx - ircvx
               deltay = isrcy - ircvy
               theta = atan2 (deltay, deltax) / rad
               if (EW) RotationAngle = 90 - RotationAngle
               phi = (theta - RotationAngle)
               cs = cos (phi * rad)
               sn = sin (phi * rad)
cdb            write(0,*)'kk= ',kk,isrcx,isrcy,ircvx,ircvy
cdb            write(0,*)deltax,deltay,theta,RotationAngle,phi
               if (verbos)
     :         write(LERR,777)RecNum,phi,sn,cs
 
            endif

c rotate the data

            do  50  i = 1, nsamp

               otrv(i) =  cs * xtrv(i) + sn * xtrh(i)
               otrh(i) = -sn * xtrv(i) + cs * xtrh(i)

 50         continue

            call vmov ( otrv, 1, itr(ITHWP1), 1, nsamp )
            call wrtape (luout, itr, obytes)

            call vmov ( otrh, 1, itr(ITHWP1), 1, nsamp )
            call wrtape ( luout, itr, obytes )

 1001    continue
 

c skip to end of record

            call trcskp ( jj, ne+1, ntrc, luin, ntrc, itr )

c           if(verbos)write(LERR,*)'GROT:  RecNum =  ',RecNum
 
1000  continue

c close data files

 999  continue

      call lbclos ( luin )
      call lbclos ( luout )
      if ( flattap .ne. ' ' ) close(luflat)

      write(LERR,*)'end of grot, processed',nrecc,' record(s)',
     :     ' with ',jtr, ' traces'
      write(LER,*)'grot - normal completion'
      stop

 990  continue
      write(LERR,*)' '
      write(LERR,*)'GROT: Error opening flatfile ',flattap(1:leflat)
      write(LERR,*)'      Check existence/permissions and try again.'
      write(LER,*)' '
      write(LER,*)'GROT: Error opening flatfile ',flattap(1:leflat)
      write(LER,*)'      Check existence/permissions and try again.'
      write(LER,*)' '
      stop
      end
 
      subroutine help

#include <f77/iounit.h>
         write(LER,*)
     :'----------------------------------------------------------------'
        write(LER,*) ' '
        write(LER,*)
     :'grot  - apply rotation to 2-component seismic data'
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute grot by typing grot and the of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)         : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)         : output data file name'
        write(LER,*)
     :' -F [otap]    (not used)           : Record, Angle Flat file'
        write(LER,*)
     :' -hw[hdrwrd]  (not used)           : get angle from hdr word'
        write(LER,*)
     :' -flt  (for hdr word option) angle stored in floating point'
        write(LER,*)
     :'       word, e.g Horz01, else stored in integer word'
        write(LER,*)
     :' -ns[ns]      (default = first)    : start trace number'
        write(LER,*)
     :' -ne[ne]      (default = last)     : end trace number'
        write(LER,*)
     :' -rs[irs]     (default = first)    : start record number'
        write(LER,*)
     :' -ne[ire]     (default = last)     : end record number'
        write(LER,*) ' '
        write(LER,*)
     :' -a[ang] (default = 0) :  rotation angle from vertical (deg)'
        write(LER,*) ' '
        write(LER,*)
     :' -H  rotate XY phones by some angle to align with R/S vector'
        write(LER,*)
     :' -NS (for -H option) hdr word angle is measured from east'
        write(LER,*)
     :'     pointing axis (pos counterclockwise), else from north'
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :'usage:  grot -N[ntap] -O[otap] -F[ftap] -ns[ns] -ne[ne] -rs[irs]'
        write(LER,*)
     :'                 -re[ire] -a[ang] -hw[] -flt -H -NS -V'
        write(LER,*) ' '
        write(LER,*)
     :'----------------------------------------------------------------'
      return
      end
 
      subroutine gcmdln ( ntap, otap, flattap, ns, ne, irs, ire, 
     :     RotationAngle, hdrwrd, verbos , flt, horrot, EW)

#include <f77/iounit.h>

      integer     ns, ne, irs, ire
      integer     argis

      real        RotationAngle

      character   ntap*(*), otap*(*), flattap*(*), hdrwrd *6

      logical     verbos, flt, horrot, EW

      EW     =   (argis('-NS') .gt. 0)

      call argstr( '-N', ntap, ' ', ' ' )
      call argstr( '-O', otap, ' ', ' ' )
      call argstr( '-F', flattap, ' ', ' ' )
      call argstr( '-hw', hdrwrd, ' ', ' ' )
      call argi4 ( '-ns', ns ,   0  ,  0    )
      call argi4 ( '-ne', ne ,   0  ,  0    )
      call argi4 ( '-rs', irs ,   0  ,  0    )
      call argi4 ( '-re', ire ,   0  ,  0    )
      call argr4 ( '-a', RotationAngle, 0.0, 0.0)
      horrot =   (argis('-H') .gt. 0)
      flt    =   (argis('-flt') .gt. 0)
      verbos =   (argis('-V') .gt. 0)

      if (RotationAngle .ne. 0.0) then
         flattap = ' '
         hdrwrd  = ' '
      else
         if (flattap(1:) .ne. ' ') hdrwrd = ' '
      endif

      if (hdrwrd(1:1) .ne. ' ' .AND. .not. flt) then
          if (hdrwrd(1:4) .eq. 'Horz') then
              flt = .true.
              write(LERR,*)'WARNING:'
              write(LERR,*)'You asked for a float hdr word ',hdrwrd
              write(LERR,*)'Will set flt = ',flt
          endif
      endif

      if (horrot .AND. hdrwrd(1:1) .eq. ' ') then
         write(LERR,*)'WARNING in grot XY rotation option:'
         write(LERR,*)'Will use a global rotation angle -a[]'
      endif
 
      return
      end
 
      subroutine verbal ( nsamp, nsi, ntrc, nrec, iform, RotationAngle, 
     :     ntap, otap, flattap, Nangles, hdrwrd, horrot, EW ) 

#include <f77/iounit.h>

      integer     nsamp, nsi, ntrc, nrec, Nangles
      real        RotationAngle
      character   ntap*(*), otap*(*), flattap*(*), hdrwrd*6
      logical horrot, EW
 
      write(LERR,*)' '
      write(LERR,*)' line header values after default check '
      write(LERR,*) ' # of samples/trace   =  ', nsamp
      write(LERR,*) ' sample interval      =  ', nsi
      write(LERR,*) ' traces per record    =  ', ntrc
      write(LERR,*) ' records per line     =  ', nrec
      write(LERR,*) ' format of data       =  ', iform
      write(LERR,*) ' rotation angle       =  ', RotationAngle
      write(LERR,*) ' input data set name  =  ', ntap
      write(LERR,*) ' output data set name =  ', otap
      if ( flattap .ne. ' ' ) then
        write(LERR,*) ' input rotation angle flat file = ',flattap
        write(LERR,*) ' number of angles in file =  ',Nangles
      endif
      if ( hdrwrd .ne. ' ' ) then
        write(LERR,*) ' input rotation angle from hdr word ',hdrwrd
      endif
      if (horrot) then
        write(LERR,*) ' mode is horizontal XY phone rotation'
        if (EW) then
        write(LERR,*) ' hdr word angle measured clockwise from east'
        else
        write(LERR,*) ' hdr word angle measured clockwise from north'
        endif
      endif
      write(LERR,*)' '
  
      return
      end
 
