C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
c performs some arcane geophysical process
c writes the results to an output file
 
c It is assumed that the development location has afp - Amoco Fortran
c Preprocessor
c
c
c**********************************************************************c
c
c     declare variables
c
c-----
c    get machine dependent parameters
c    these includes are picked up and expanded into the code by afp
c    and are found in ~usp/include/f77
 
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
 
c-----
c    SZLNHD is a value obtained from lhdrsz.h
c    The 3 vectors below are equivalenced and are
c    to access the trace header entries (whatever
c    they may be)
c-----
      integer     lhed ( SZLNHD )
      integer     nhed ( SZLNHD )
      integer     itr  ( SZLNHD )
      integer     ntr  ( SZLNHD )
      real        head ( SZLNHD )

      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne
      real        vref, rad, deg
 
c------
c  static memory allocation
c     real        bigar1(SZSPRD*SZSMPM)
c     real        bigar2(SZSPRD*SZSMPM)
c------
c  dynamic memory allocation for big arrays, eg whole records
      integer     itrhdr
      real        bigar1, bigar2
      pointer     (wkadri, itrhdr(1))
      pointer     (wkadr1, bigar1(1))
      pointer     (wkadr2, bigar2(1))
c------
 
c------
c    this is necessary for buliding printout files
#include <f77/pid.h>
c------
      integer     recnum, trcnum
      integer     dstsgn, dstusg, stacor
c-----
c    SZSMPM is a value obtained from lhdrsz.h
c-----
      real        tri ( SZLNHD ), vel ( SZLNHD )
      real        p ( SZLNHD ), ang0 ( SZLNHD )
      character   ntap * 256, otap * 256, vtap * 256, name*8
      character   key1 * 1, key3 * 1
      logical     verbos, query, heap1, heap2, heapi
      integer     argis, pipe
 
c-----
c    we access the header values which can be shot or long integers
c    or real values.  The actual trace values start at position
c    ITRWRD1  (position 65 in the old SIS format).  This value is
c    set in lhdrsz.h but eventually could come in thru the line header
c    making the trace header format variable
c-----
      equivalence ( itr( 1), lhed (1), head(1) )
      equivalence ( ntr( 1), nhed (1) )

      data lbytes / 0 /, nbytes / 0 /, name/'TAUPSAMP'/
      data pipe /3/
 
c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 .or. argis ( '-h' ) .gt. 0)
      if ( query )then
            call help()
            stop
      endif
 
      rad = 3.14159265/180.
      deg = 1./rad

c-----
c     open printout files
c     this consists of a 2-character identifier, the process id number,
c     and the child process id number:
c     XY.nnnnn.mmmmm
c     this should be unique even for multiple occurences of the same
c     process in a pipeline
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otap,vtap,ns,ne,irs,ire,
     1             verbos)
 
c-----
c     get logical unit numbers for input and output of seismic data
 
c     input values are strings ntap & otap (i.e. names of files), and the
c     read 'r' or write 'w' designations of these files, and the default
c     logical unit numbers if ntap & otap are blank strings, i.e. ntap = " "
c     in which case:
c     0 = default stdin
c     1 = default stdout
c     Note: default values other that 0 & 1 may be used
 
c     output values are the logical unit numbers accessing these disk files. if
c     these values are less than 0 it means there was a fatal error in trying to
c     open these files
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

      ikp = in_ikp()
      if (vtap .ne. ' ') then
          call getln(luvel, vtap, 'r',-1)
      else
          if (ikp .eq. 1) then
             write(LERR,*)'taupsamp assumed to be running inside IKP'
             call sisfdfit (luvel, pipe)
          else
             write(lerr,*)'taupsamp error: cannot find velocity file'
             stop
          endif
      endif

      if(luvel .lt. 0)   then
         write(lerr,*)'taupsamp error: velocity file -v not accessible'
         stop
      endif
 
c-----
c     read line header of inputa DSN (rtape reads data into vector "itr"
c     lbytes is the number of bytes actually read
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'TAUPSAMP: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif
 
c------
c     save certain pace header rameters
 
c     To get and/or put LINE header values please use saver/savew
c     since that way code will always be portable to whatever machine
c     we maintain usp on
c     saver/w refer to header words using mnemonics rather than position
c     LINHED is a value in the include file <f77/sisdef.h> that refers to the
c     lineheader
c
c     see saver/w manual pages
c     (LINHED = 0  - just like LINEHEADER)
c------
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that 
c     refers to the trace header (LINEHEADER = 0; TRACEHEADER = 1)

      call savelu('MutVel',ifmt_MutVel,l_MutVel,ln_MutVel, LINEHEADER)
      call savelu('WatVel',ifmt_WatVel,l_WatVel,ln_WatVel, LINEHEADER)

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

c-----------
c format values are:

c     integer    = 1
c     float      = 2
c     character  = 3
c     2-byte int = 4

c the mnemonic definitions are found in the man pages for program scan
c-----------
c
c ----- velocity data -----
c
 
      vflag=1
      lvbytes=0
 
      call rtape(luvel,ntr,lvbytes)
 
      if(lvbytes .eq. 0) then
         write(LERR,*)'FATAL: no lineheader read on',vtap
         write(LERR,*)' '
         stop
      endif
 
      call saver(ntr, 'NumSmp', nsampv, LINHED)
      call saver(ntr, 'SmpInt', nsiv  , LINHED)
      call saver(ntr, 'NumTrc', ntrcv , LINHED)
      call saver(ntr, 'NumRec', nrecv , LINHED)
      call saver(ntr, 'Format', iformv, LINHED)
 
c
c ----- vflag=0 single velocity function -----
c       vflag=1 multiple velocity function
 
      if(nrecv.le.1) vflag=0

      if (vflag .eq. 0) then
 
         nvbytes=0
         call rtape(luvel,ntr,nvbytes)

         if(nvbytes .eq. 0) then
            write(LERR,*)' '
            write(LERR,*)'FATAL ERROR in taupsamp:'
            write(LERR,*)' No velocity data found after '
            write(LERR,*)' lineheader in ',vtap,' check '
            write(LERR,*)' velin run thoroughly.'
            write(LER ,*)' '
            write(LER ,*)'FATAL ERROR in taupsamp:'
            write(LER ,*)' No velocity data found after '
            write(LER ,*)' lineheader in ',vtap,' check '
            write(LER ,*)' velin run thoroughly.'
            stop
         endif
 
         call vmov (nhed(ITHWP1), 1, vel, 1, nsampv)
 
         if(verbos) then
            write(LERR,*)'velocity'
            write(LERR,777)(vel(i),i=1,nsampv)
 777        format(10f8.2)
         endif
c
c ----- check for valid velocities -----
c
 
         call dotpr (vel,1,vel,1,vdot,nsampv)
 
         if (abs(vdot) .lt. 1.e-06) then
            write(LERR,*)' '
            write(LERR,*)'FATAL ERROR in taupsamp:'
            write(LERR,*)'Velocity trace contains zeros'
            write(LERR,*)'---    FATAL  ---'
            write(LERR,*)'Check the velin step thoroughly'
            write(LER ,*)' '
            write(LER ,*)'FATAL ERROR in taupsamp:'
            write(LER ,*)'Velocity trace contains zeros'
            write(LER ,*)'---    FATAL  ---'
            write(LER ,*)'Check the velin step thoroughly'
            go to 999
         endif
 
      endif



c------
c     hlhprt prints out the historical line header of length lbytes AND
 
c     hlhprt takes "name", in this case 4 characters long and stuffs this
c     into the modified historical line header and returns the NEW length
c     of the line header in lbytes
c------
      call hlhprt (itr, lbytes, name, 8, LERR)
 
c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records).  This guards against
c     zero start values or values that are greater than those specified
c     in the line header
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
 
c---------------------------------------------------
c  malloc only space we're going to use
      heapi = .true.
      heap1 = .true.
      heap2 = .true.
 
c--------------------------
c  note: these don't
c  have to be the same size
 
      itemi = ntrc * ITRWRD * SZSMPD
      item1 = ntrc * nsamp  * SZSMPD
      item2 = ntrc * nsamp  * SZSMPD
 
c  note also SZSMPD is the native
c  size of a float or int in bytes
c--------------------------
 
c--------
c  galloc - general allocation (machine independent since it uses C
c  malloc internally
c  inputs to galloc are pointer, number of bytes to allocate
c  outputs are error codes:
c     errcod = 0  (allocation succeeded)
c     errcod = 1  (allocation failed)
c--------
 
      call galloc (wkadri, itemi, errcdi, aborti)
      call galloc (wkadr1, item1, errcd1, abort1)
      call galloc (wkadr2, item2, errcd2, abort2)
 
      if (errcdi .ne. 0.) heapi = .false.
      if (errcd1 .ne. 0.) heap1 = .false.
      if (errcd2 .ne. 0.) heap2 = .false.
 
      if (.not. heap1 .or. .not. heap2 .or. .not. heapi) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) item2,'  bytes'
         write(LERR,*)' '
         write(LER ,*)' '
         write(LER ,*)'Unable to allocate workspace:'
         write(LER ,*) itemi,'  bytes'
         write(LER ,*) item1,'  bytes'
         write(LER ,*) item2,'  bytes'
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) item2,'  bytes'
         write(LERR,*)' '
      endif
c---------------------------------------------------
 
c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD
 
 
c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
c     if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  vtap,ntap,otap)
c     end if
 
c--------------------------------------------------
c  compute sample interval in secs
c  take care of micro secs if necessary
 
         dt = real (nsi) * unitsc

         call saver (itr, 'Crew01' , key1  , LINHED)
         call saver (itr, 'Crew03' , key3  , LINHED)

         if (key1 .ne. 't' .AND. key3 .ne. 'L') then
            write(LERR,*)' '
            write(LERR,*)'FATAL ERROR in taupsamp:'
            write(LERR,*)'Input data not from radonf ... -L'
            write(LER ,*)' '
            write(LER ,*)'FATAL ERROR in taupsamp:'
            write(LER ,*)'Input data not from radonf ... -L'
            go to 999
         endif

         call saver (itr, 'MxRSEL' , ixmax , LINHED)
         call saver (itr, 'MxGrEl' , itmax , LINHED)
         call saver (itr, 'MnGrEl' , itmin , LINHED)
         xmax = ixmax
         tmin = itmin
         tmax = itmax
         tmxx = amax1 (float(itmax), float(iabs(itmin)))
         if (tmxx .eq. 0.) then
            vref = 0.
         else
            vref = 1000. * xmax / tmxx
         endif
         write(LERR,*)' '
         write(LERR,*)'radonf input:'
         write(LERR,*)'Max distance      =  ',xmax
         write(LERR,*)'Max time (ms)     =  ',tmax
         write(LERR,*)'Min time (ms)     =  ',tmin
         write(LERR,*)'Reference velocity=  ',vref
         write(LERR,*)' '

         if (vref .eq. 0.) then
            write(LERR,*)' '
            write(LERR,*)'FATAL ERROR in taupsamp:'
            write(LERR,*)'reference velocity is zero. Something'
            write(LERR,*)'wrong with radonf run (xmax, mmin, mmax)'
            write(LER ,*)' '
            write(LER ,*)'FATAL ERROR in taupsamp:'
            write(LER ,*)'reference velocity is zero. Something'
            write(LER ,*)'wrong with radonf run (xmax, mmin, mmax)'
            go to 999
         endif
         
         call savew (itr, 'Crew05' , 'A'   , LINHED)

         write(LERR,*)' '
         write(LERR,*)'radonf input:'
         write(LERR,*)'Max distance      =  ',xmax
         write(LERR,*)'Max time (ms)     =  ',tmax
         write(LERR,*)'Reference velocity=  ',vref
         write(LERR,*)' '

c----------------------
c  inject command line into
c  historical LH:
c  inputs are current line header in vector "itr" and length lbytes (bytes)
c  outputs are line header modified by insertion of the command line into
c  the historical LH and the modified length of the new LH (lbyout)
 
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout  )
c----------------------
 
 
c--------------------------------------------------
c     BEGIN PROCESSING
c     read trace, do terrible things to data, write to output file
c-----

c-----
c     process desired trace records
c-----
      DO  JJ = 1, nrec
 
            ic = 0

            do  kk = 1, ntrc
 
                  nbytes = 0
                  call rtape( luin, itr, nbytes)
c------
c     if end of data encountered (nbytes=0) then bail out
c     Note:  if you're processing records you might really want
c     to branch to the processing part rather than bailing out
c------
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)
c------
c     read next velocity if we have it
c     don't stop the program if we don't
c------
                  if (vflag.eq.1 .and. nvbytes.ne.0) then
                     nvbytes=0
                     call rtape (luvel,ntr,nvbytes)
                     call vmov (nhed(ITHWP1), 1, vel, 1, nsampv)
                  endif
 
c------
c     use previously derived pointers to trace header values
                  call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        recnum , TRACEHEADER)
                  call saver2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        trcnum , TRACEHEADER)
                  call saver2(lhed,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        dstsgn , TRACEHEADER)
                  call saver2(lhed,ifmt_DstUsg,l_DstUsg, ln_DstUsg,
     1                        dstusg , TRACEHEADER)
                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        stacor , TRACEHEADER)

                  IF (stacor .eq. 30000) THEN
                     call vclr (tri,1,nsamp)
                  ELSE

                     p0  = float (dstusg) / 10000000.
                     vp0 = vref * p0
                     if (abs(vp0) .le. .999999) then
                        ic = ic + 1
                        p (ic) = p0
                        ang0 (ic) = deg * asin (vp0)
c----------------------
c  pack data into array
                        istrc = (ic-1) * nsamp
                        ishdr = (ic-1) * ITRWRD
                        call vmov (tri,  1, bigar1(istrc+1), 1, nsamp)
                        call vmov (lhed, 1, itrhdr(ishdr+1), 1, ITRWRD)
                     endif

                  ENDIF
 
            enddo
            jtr = ic

            call minmgv (ang0, 1, amin, loca, jtr)
 
c-----------------------
c  here's the meat...
c  do something to data
 
            IF (loca .eq. 1) THEN

               call subs  (jtr, nsamp, deg, p, ang0, vel,
     1                    bigar1, bigar2)

            ELSEIF (loca .eq. jtr) THEN
 
               call subsr (jtr, nsamp, deg, p, ang0, vel,
     1                    bigar1, bigar2)

            ELSEIF (loca .gt. 1 .AND. loca .lt. jtr) THEN
 
               if     (ang0(loca).lt.0. .AND. ang0(loca+1).ge.0.) then
                   jtr1 = loca
                   jtr2 = jtr - jtr1
                   iflg = 0
               elseif (ang0(loca).lt.0. .AND. ang0(loca+1).le.0.) then
                   jtr1 = loca - 1
                   jtr2 = jtr - jtr1
                   iflg = 1
               elseif (ang0(loca).ge.0. .AND. ang0(loca+1).lt.0.) then
                   jtr1 = loca
                   jtr2 = jtr - jtr1
                   iflg = 1
               elseif (ang0(loca).ge.0. .AND. ang0(loca+1).gt.0.) then
                   jtr1 = loca - 1
                   jtr2 = jtr - jtr1
                   iflg = 0
               endif
               call subs2 (jtr, nsamp, deg, jtr1, jtr2, loca,
     1                     iflg, p, ang0, vel, bigar1, bigar2)
            ENDIF

c  (USER: insert your subroutine above)
c-----------------------
 
c---------------------
c  extract traces from
c  output array and
c  write output data
            do  kk = 1, ntrc
 
                ishdr = (kk-1) * ITRWRD
                istrc = (kk-1) * nsamp
                call vmov (bigar2(istrc+1),1,lhed(ITHWP1),1, nsamp)
                if (kk .gt. jtr) then
                   call vclr (lhed(ITHWP1),1, nsamp)
                endif
                call vmov (itrhdr(ishdr+1),1,lhed,1,ITRWRD)
                call wrtape (luout, itr, obytes)
 
 
            enddo
 
 
      ENDDO
 
  999 continue
 
c-----
c     close data files
c     flush data left
c     in output buffer
c     NOTE: if the output buffer is not closed you can sometimes end up
c     with missing data
c-----
      call lbclos ( luin )
      call lbclos ( luout )
 
      write(LERR,*)'end of taupsamp, processed',nrec,' record(s)',
     :     ' with ',ntrc, ' traces'
      write(LER,*)'end of taupsamp, processed',nrec,' record(s)',
     :     ' with ',ntrc, ' traces'
      stop      
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'taupsamp laterally resamples NMO corrected tau-p data from'
        write(LER,*)
     :'constant p traces to constant angle traces'
        write(LER,*)' '
        write(LER,*)
     :'execute by typing taupsamp and the of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)         : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)         : output data file name'
        write(LER,*) ' '
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :'usage:   taupsamp -N[ntap] -O[otap] -v[vtap]'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,vtap,ns,ne,irs,ire,
     1                  verbos)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name
c     otap  - C*100    output file name
c     vel   - R*4      design velocity
c     ns    - I*4      starting trace index
c     ne    - I*4      ending trace index
c     irs   - I*4      starting record index
c     ire   - I*4      ending record index
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*), vtap*(*)
      integer     ns, ne, irs, ire
      logical     verbos
      integer     argis
 
c-------
c     import values from the command line using keys, e.g. -N
c     to which are immediately attached the users values.
 
c     For example program taupsamp might be invoked in the following way:
 
c     taupsamp  -Nxyz -Oabc
 
c     in which case xyz is a string (the name of the input data set)
c     which will be imported into taupsamp and associated with the variable
c     "ntap"
 
c     see manual pages on the argument handler routines
c     for the meanings of these functions.  Briefly though
c     we can import from the command line strings, integers, reals,
c     double precision reals, and check the existence of command line
c     keys.  The last 2 fields are the value to be assigned the variable
c     (1) if ONLY the key is present (no value attached to it), or
c     (2) if NO key & no value are present
c-------
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argstr( '-v', vtap, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            ns = 0
            ne = 0
            irs = 0
            ire = 0
            verbos =   (argis('-V') .gt. 0)
 
c-------
c     Note:  argis() = 1 if the key is present
c            argis() = 0 if it isn't
 
      return
      end
 
C***********************************************************************
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  vtap,ntap,otap)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     vel   - R*4     design velocity
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ntap  - C*100   input file name
c     otap  - C*100   output file name
c-----
#include <f77/iounit.h>
 
      integer     nsamp, nsi, ntrc, nrec
      character   ntap*(*), otap*(*), vtap*(*)
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' design window velocity =  ', vel
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' input velocity dsn  =  ', vtap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' '
            write(LERR,*)' '

      return
      end
 
