C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C**********************************************************************C
C**********************************************************************C
C
C      PROGRAM MODULE TAUPSPEC: tau-p spectra ( as sis files )
C
C**********************************************************************C
c      B.V. NGUYEN and R.B. HERRMANN                     APRIL 1986 SLU
C**********************************************************************C
C          THE MAIN PROGRAM CALL SUBROUTINE TVEL TO PERFORM tau-p-
C      TRAJECTORY SCAN SEARCH FOR OPTIMUM STACKING VELOCITY (RMS VEL.)
C      AS A FUNCTION OF TWO-WAY ZERO-OFFSET TIME VIA A COHERENCY MEASURE
C      CALLED SEMBLANCE IN A CMP GATHER OF TRACES.
C      THE SURFACE REPRESENTING THE COHERENCY MEASURE AS A FUNCTION OF
C      TWO-WAY ZERO-OFFSET TIME AND TEST STACKING VELOCITY DEFINES A
C      VELOCITY SPECTRUM.
C      FOR SEMBLANCE REFERENCE:
C      TANER, M.T. and F. KOEHLER (1969) GEOPHYSICS,34,p.859-881.
C      FOR ENERGY-NORMALIZED CROSS-CORRELATION SUM REFERENCE:
C      HUBRAL, P. and T. KREY (LARNER, K. L., Ed.) (1980) SOCIETY OF
C      EXPLORATION GEOPHYSICISTS, 203pp.
C**********************************************************************C

#include <localsys.h>
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      INTEGER     ITR ( SZLNHD )
      INTEGER     STR ( SZLNHD )
      INTEGER     LUIN, NBYTES, OBYTES
      INTEGER     NSAMP, NSI, NTRC, NREC, IFORM, LBYTES
      INTEGER     ISTOP, ISTART, NPT
      REAL        START,STOP
      real        vref, angle
      REAL        TRI ( SZLNHD )
      CHARACTER   NAME * 8, ntap * 256, otap * 256

      logical     verbos
      logical     logsc
      logical     abslut
      logical     slowness
      logical     query, heap, slnt, radon, nowt
      integer     argis

      real        traces, vels
      pointer     (wkaddr, traces(1))
      pointer     (wkvels, vels(1))
      real        tmul, dmul
      real        dis(SZLNHD), weight( SZLNHD )
      real        snum ( SZLNHD ), sden ( SZLNHD ), cn   ( SZLNHD )
      integer     nvel, non_zero_samples(SZLNHD)
      integer     ns,ne
      real        vmin,vmax
      integer     igt


      DATA NAME /'TAUPSPEC'/, NBYTES / 0 /, LBYTES / 0 /

      data pi/3.14159265/

C*********************************************************************C
c  get online help if necessary
C*********************************************************************C
      query = ( argis ('-?').gt.0 .or. argis('-h').gt.0 )
      if ( query ) then
           call help ()
           stop
      endif

C*********************************************************************C
c  open printout file
C*********************************************************************C
#include <f77/open.h>

C**********************************************************************C
C     READ PROGRAM PARAMETERS FROM COMMAND LINE
C**********************************************************************C
      call cmdln (ntap,otap,start,stop,dmul,tmul,igt,
     &               ns,ne,irs,ire,nvel,vmin,vmax,abslut,verbos,slnt,
     &               radon,logsc,slowness,nowt)

C**********************************************************************C
C     READ AND UPDATE LINE HEADER,
C     WRITE LINE HEADER, SAVE KEY PARAMETERS.
C**********************************************************************C
      call getln ( luin, ntap, 'r', 0 )
      call getln (luout, otap, 'w', 1 )

      lbytes = 0
      CALL RTAPE  ( LUIN, itr, LBYTES           )
      if(lbytes .eq. 0) then
         write(LERR,*)'TAUPSPEC: no header read on unit ',ntap
         write(LERR,*)'FATAL'
         write(LERR,*)'Check existence of file & rerun'
         stop
      endif
      CALL HLHprt ( ITR , LBYTES, name, 7, LERR         )

c-----------------------------------------
c  save key header values
#include <f77/saveh.h>

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('SGRNum',ifmt_SGRNum,l_SGRNum,ln_SGRNum,TRACEHEADER)
 

         write(LERR,*)' '
         write(LERR,*)' Values read from input data set lineheader'
         write(LERR,*)' # of Samples/Trace =  ',nsamp
         write(LERR,*)' Sample Interval    =  ',nsi
         write(LERR,*)' Traces per Record  =  ',ntrc
         write(LERR,*)' Records per Line   =  ',nrec
         write(LERR,*)' Format of Data     =  ',iform
         write(LERR,*)' Slowness**2 spectra?  ',slowness
         write(LERR,*)' Velocity spectra?     ',.not. slowness
         write(LERR,*)' Log vel/slow scale =  ',logsc
         write(LERR,*)' '

C**********************************************************************C
C     CHECK DEFAULT VALUES AND SET PARAMETERS
C**********************************************************************C
      if(START.le.0.0)START = 0.0
      ttmax = (NSAMP-1)*NSI
      if(STOP.le.0.0 .or. STOP.ge. ttmax)STOP=ttmax
      dt=real(NSI)
      dt=dt*tmul
      ISTART =   INT   ( START / dt ) + 1
      ISTOP  =   INT   ( STOP  / dt ) + 1
        IF ( ISTART . LT. 1     .OR. ISTART.GT. NSAMP ) THEN
           ISTART = 1
        ENDIF
        IF ( ISTOP . LE. ISTART .OR. ISTOP .GT. NSAMP ) THEN
           ISTOP  = NSAMP
        ENDIF
      NPT    =       ISTOP - ISTART + 1
      if(ire.lt.1.or.ire.lt.irs.or.ire.gt.nrec)ire = nrec
      if(ns.le.0 )ns = 1
      if(ne.le.0 .or. ne.gt.ntrc)ne=ntrc

      igt0 = 2 * igt + 1
      call vfill (1.0, weight, 1, igt0)
      if (igt0 .gt. 1 .AND. .not.nowt) then
         igt2 = igt / 2
         do  i = 1, igt2
             ang = 3.14159265 * float(i-1)/float(igt2)
             weight (i) = .5 * (1. - cos ( ang ) )
             weight (igt0-i+1) = weight (i)
         enddo
      endif


C**********************************************************************C
         write(LERR,*)' Values read from file command line'
         write(LERR,*)' Start time in ms (default = 0 ms)    =  ',START
         write(LERR,*)' Stop time in ms (default= full trace) =  ',STOP
         write(LERR,*)' Number of velocity inputs             =  ',nvel
         write(LERR,*)' Minimum velocity for plots            =  ',vmin
         write(LERR,*)' Maximum velocity for plots            =  ',vmax
         write(LERR,*)' Time gate integer                     =  ',igt
         write(LERR,*)' Turn off cosine weighting?            =  ',nowt
         write(LERR,*)' Distance multiplied by a factor       =  ',dmul
         write(LERR,*)' Time multiplied by a factor           =  ',tmul
         write(LERR,*)' First record to process               =  ',irs
         write(LERR,*)' Last record to process                =  ',ire
         write(LERR,*)' Starting trace number                 =  ',ns
         write(LERR,*)' Ending   trace number                 =  ',ne
         if(slnt)
     1   write(LERR,*)' Input data comes from program slnt'
         if(logsc)
     1   write(LERR,*)' Slowness or velocity scale is logarthmic'
         write(LERR,*)' '
         write(LERR,*)' Weight applied to time gate:'
         write(LERR,*)(weight(ii),ii=1,igt0)
         write(LERR,*)' '

C**********************************************************************C


c---------------------------------------------------
c  malloc only space we're going to use
      heap = .true.

      items = ntrc * (nsamp)
      itemv = nvel * (nsamp)
      call galloc (wkaddr, items*SZSMPD, errcod, abort)
      if (errcod .ne. 0) heap = .false.
      call galloc (wkvels, itemv*SZSMPD, errcod, abort)
      if (errcod .ne. 0) heap = .false.

      if (.not. heap) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) items*SZSMPD,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) items*SZHFWD,'  bytes'
         write(LERR,*)' '
      endif
c---------------------------------------------------
c
c--------------------------------------------------------
c  adjust line header, historical l.h., & output header
       ivmin = vmin
       ivmax = vmax
       dvel  = (vmax - vmin)/float(nvel-1)
 
       call savew( itr, 'MnLnIn',  4   , LINHED)
       call savew( itr, 'MxLnIn', nvel , LINHED)
       call savew( itr, 'ILClIn',  vmin, LINHED)
       call savew( itr, 'CLClIn',  dvel, LINHED)
       call savew( itr, 'MinVel', ivmin, LINHED)
       call savew( itr, 'MaxVel', ivmax, LINHED)

       call savew( itr, 'NumTrc', nvel , LINHED)
       call savew( itr, 'NumSmp', istop, LINHED)
       obytes = SZTRHD + SZSMPD * istop
       call savhlh ( itr, lbytes, lbyout )
       call wrtape(luout, itr, lbyout )

c---------------------------------------
c  skip to start record
      call recskp(1,irs-1,luin,ntrc,itr)

C**********************************************************************C
C
C      READ TRACES, COMPUTE SPECTRA, WRITE RESULTS
C
C**********************************************************************C
        ist=0
        DO 5001 JJ = irs, ire

c-----------------------------------------------
c  skip to start trace of current record
            call trcskp(jj,1,ns-1,luin,ntrc,itr)

           ist=ist+1
           k=0

c----------------------------
c  read selected data
           DO 5002 KK = ns, ne

                 NBYTES = 0
                 CALL RTAPE  ( LUIN, ITR, NBYTES )
                 if(nbytes .eq. 0) then
                    write(LERR,*)'End of file on input:'
                    write(LERR,*)'  rec= ',jj,'  trace= ',kk
                    go to 999
                 endif
                 call vmov (itr(ITHWP1), 1, tri, 1, nsamp)

                 call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                       istatic, TRACEHEADER)

                 IF (istatic .ne. 30000) THEN
                     if (kk .eq. ns)
     1               call vmov (itr, 1, str, 1, ITRWRD)
                     call saver2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                           irec , TRACEHEADER)
                     call saver2(itr,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                           idi  , TRACEHEADER)
                     k=k+1
                     IF (slnt) THEN

                        call saver2(itr,ifmt_DstUsg,l_DstUsg,ln_DstUsg,
     1                               ivref , TRACEHEADER)
                        vref = float( ivref )

                        call saver2(itr,ifmt_DstSgn,l_DstSgn,ln_DstSgn,
     1                              irayp , TRACEHEADER)
                        angle = pi * float( irayp )/180.
                        dis(k) = sin( angle )/vref

                     ELSEIF (radon) THEN

                        call saver2(itr,ifmt_DstUsg,l_DstUsg,ln_DstUsg,
     1                              irayp , TRACEHEADER)
                        dis(k) = dmul * float(irayp)/10000000.

                     ELSE

                        call saver2(itr,ifmt_DstUsg,l_DstUsg,ln_DstUsg,
     1                              irayp , TRACEHEADER)
                        dis(k) = dmul * float(irayp)/10000000.

                     ENDIF

                     istrc  = (k-1) * nsamp

                     do 5004 LL= 1, nsamp
                           traces(istrc + LL) = tri(LL)
 5004                continue

                 ENDIF

 5002      CONTINUE

c----------------------------------
c  do spectra
      call tvel (traces,vels,str,tri,dt,nvel,k,dis,
     1          vmin,vmax,luout,obytes,slowness,irec,logsc,
     2          ntap,JJ,ns,ne,NSI,NSAMP,ISTART,ISTOP,weight,
     3          START,STOP,igt,abslut,irs,ire,ist,idi,
     4          l_RecNum,l_TrcNum,l_DstSgn,l_StaCor,
     5          ln_RecNum,ln_TrcNum,ln_DstSgn,ln_StaCor,
     6          ifmt_RecNum,ifmt_TrcNum,ifmt_DstSgn,ifmt_StaCor,
     7          ifmt_DphInd,l_DphInd,ln_DphInd,
     8          snum,sden,cn,ITRWRD,ITHWP1,SZLNHD,unitsc,
     9          non_zero_samples )

      write(LER,*)'Completed Record= ',JJ

 5001 CONTINUE

  999 continue

      call lbclos( luin)
      call lbclos(luout)

      END
C**********************************************************************C

C**********************************************************************C
      subroutine tvel (traces,vels,itr,tri,dt,nvel,k,dis,
     1                vmin,vmax,luout,obytes,slowness,irec,logsc,
     2                ntap,nrec,ktrc1,ktrc2,nsi,nsamp,ISTART,ISTOP,
     3                wt,START,STOP,igt,abslut,irs,ire,ist,idi,
     4                l_RecNum,l_TrcNum,l_DstSgn,l_StaCor,
     5                ln_RecNum,ln_TrcNum,ln_DstSgn,ln_StaCor,
     6                ifmt_RecNum,ifmt_TrcNum,ifmt_DstSgn,ifmt_StaCor,
     7                ifmt_DphInd,l_DphInd,ln_DphInd,
     8                snum,sden,cn,ITRWRD,ITHWP1,SZLNHD,unitsc,
     9                non_zero_samples )

C**********************************************************************C
c      SUBROUTINE ARGUMENTS
c
c      traces    R*4      Two dimensional array of spectra of trace
c                         traces(LL,LR) for trace LR and time LL
c      obytes    I*4      output number of bytes
c      itr       I*2      output data vector equivalenced to...
c...   tri       R*4      ...output real data vector
c      luout     I*4      output data unit number
c      istart    I*4      start time in samples
c      start     R*4      start time in ms
c      istop     I*4      stop time in samples
c      stop      R*4      stop time in ms
c      dt        R*4      Sample interval in MILLISECONDS
c      nvel      I*4      Number of velocity inputs
c      k         I*4      Number of traces
c      dis       R*4      Array of source-receiver distances
c      vmin      R*4      Minimum velocity  (ft/sec)for plot
c      vmax      R*4      Maximum velocity  (ft/sec)for plot
c      ntap      C*120    File name
c      nrec      I*4      Record number
c      ktrc1     I*4      Starting trace number
c      ktrc2     I*4      Ending trace number
c      nsi       I*4      Sample interval in MILLISECONDS
c      NSAMP     I*4      Number of samples in original time series
c      IGT       I*4      integer for time gate centered at zero-offset time
c                         semblance varies between 0 and 1
c      abslut    L        contour absolute semblance rather than relative
C**********************************************************************C
c

#include <f77/iounit.h>
#include <f77/sisdef.h>

      integer SZLNHD
      integer non_zero_samples(SZLNHD)
      integer   itr(*)
      integer   IGT,NVEL,IO

      real      traces(nsamp,k), vels(nsamp,nvel), tri(*)
      real      dis(*), wt(*)
      real      cnnum,cnden,xtrc,amp
      real      vel
      real      snum(*),sden(*)
      real      cn(*)

      logical   abslut,slowness,logsc

      xtrc = k
      dt1 = dt * unitsc
C**********************************************************************C
C**********************************************************************C
c      perform scan of trajectories to determine vrms
C**********************************************************************C
      recmax = 0.0
      dvel = (vmax-vmin)/(NVEL-1)
      if (logsc) then
         v1 = log (vmin)
         v2 = log (vmax)
         dvel = (v2 - v1)/float(nvel-1)
      else
         v1 = vmin
         v2 = vmax
         dvel = (vmax-vmin)/(NVEL-1)
      endif

      if (nrec .eq. irs) then
         write(LERR,*)'Velocity increment = ',dvel
         write(LER ,*)'Velocity increment = ',dvel
      endif

C**********************************************************************C
      if(slowness) then
         if (logsc) then
            smin2 = log((1./vmax)**2)
            smax2 = log((1./vmin)**2)
            ds2   = (smax2-smin2)/(nvel-1)
         else
            smin2 = (1./vmax)**2
            smax2 = (1./vmin)**2
            ds2   = (smax2-smin2)/(nvel-1)
         endif
      endif

C**********************************************************************C
c       for each velocity,
C**********************************************************************C
      DO 100 IR = 1,NVEL

            do 45 LL=1,ISTOP
               cn(LL)=0.
   45       continue

            if(slowness) then
               if (logsc) then
                  s2  = exp(smin2+(ir-1)*ds2)
                  vel = sqrt(1./s2)
               else
                  s2  = smin2+(ir-1)*ds2
                  vel = sqrt(1./s2)
               endif
            else
               if (logsc) then
                  vel = exp (v1 + float(ir-1)*dvel)
               else
                  vel = v1+(ir-1)*dvel
               endif
            endif

            DO LL=ISTART,ISTOP
               snum(LL) = 0.
               sden(LL) = 0.
               non_zero_samples(LL) = 0
            ENDDO
C**********************************************************************C
      DO 102 LR = 1,k
            sinei = dis(LR) * vel
            sqr   = abs (1. - sinei*sinei)
C**********************************************************************C
      DO 102 LL = ISTART,ISTOP

            tt = float(LL-1) * sqrt( sqr )
            it = tt + 1
            if(it.lt.1)then
                  amp = 0.0
            elseif(it.ge.nsamp)then
                  amp = 0.0
            else
               c1 = float(it) - tt
               c2 = 1. - c1
               amp = c1 * traces(it-1,LR) + c2 * traces(it,LR)

c keep track of number of non zero samples contributing to the semblance

               if ( abs(amp) .gt. 1.e-32 ) 
     :              non_zero_samples(LL) = 
     :              non_zero_samples(LL) + 1

            endif

            snum(LL)=snum(LL) + amp
            sden(LL)=sden(LL) + amp*amp

102       continue
C**********************************************************************C
c      square numerator term
C**********************************************************************C

            DO 103  LL = ISTART,ISTOP

               snum(LL) = snum(LL)**2

103         continue

C**********************************************************************C
c      decide for a time gate
C**********************************************************************C
      IGTP = ISTART + IGT
      IGTM = ISTOP  - IGT
      IO   = IGTM - IGTP + 1
C**********************************************************************C
c      begin to compute semblance for a given velocity
C**********************************************************************C

      DO 200  LL = IGTP,IGTM

            cnnum = 0.
            cnden = 0.
C**********************************************************************C
            IGL = LL - IGT
            if (IGL.le.0) IGL = 1
            IGH = LL + IGT
            if(IGH.gt.ISTOP) IGH = ISTOP
C**********************************************************************C
            ig = 0
            DO 305 I = IGL,IGH
              ig = ig + 1
              wti = wt (ig)
              cnnum = cnnum + snum(I) * wti
              cnden = cnden + sden(I)
305         continue
C**********************************************************************C
c
c   put correlation z-values into big matrix

            if(cnden.ne.0.0) then
               cn(LL) = cnnum / (xtrc*cnden)
               cn(LL) = cn(LL) * ( 2.0 - (float(non_zero_samples(LL))
     :              / xtrc))
            endif

C**********************************************************************C

200       continue

C**********************************************************************C
c
c   if absolute is true do not normalize correlation in time
      call vclr ( tri,1,istop )
            do 230  LL  = ISTART,ISTOP
                tri(LL) = cn(LL)
230         continue
C**********************************************************************C
          call vmov (tri, 1, vels(1,IR), 1, nsamp)
100       continue
C**********************************************************************C
      do  1000  J = 1, nvel
          call vmov (vels(1,J), 1, tri, 1, nsamp)
          if (logsc) then
             vel = exp (v1 + float(J-1)*dvel)
          else
             vel = v1+(J-1)*dvel
          endif
          ivel = vel
                call savew2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                      ivel  , TRACEHEADER)
                call savew2(itr,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                      idi   , TRACEHEADER)
                call savew2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        J   , TRACEHEADER)
                call savew2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                       0   , TRACEHEADER)
                call savew2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                      irec , TRACEHEADER)
          call vmov (tri, 1, itr(ITHWP1), 1, nsamp)
          call wrtape(luout, itr, obytes)
1000  continue
c
      return
      end
C**********************************************************************C
C**********************************************************************C
      function rinter(y1,y2,dx)
C**********************************************************************C
c      interpolate between y1 and y2 for 0 < dx < 1
C**********************************************************************C
      rinter = (1.0 - dx)*y1 + dx*y2
      return
      end

c-----------------------------------------
c  online help section
c-----------------------------------------
       subroutine  help
#include <f77/iounit.h>

         write(LER,*)
     :'***************************************************************'
      write(LER,*)' '
      write(LER,*)
     :'              taupspec - velocity spectra from tau-p data'
      write(LER,*)' '
      write(LER,*)
     :'Execute program by typing taupspec and the following arguments'
      write(LER,*)
     :' -N [ntap]     (default stdin)    : Input data file name'
      write(LER,*)
     :' -O [otap]     (default stdout)   : Output data file'
      write(LER,*)
     :' -tmin [start] (default = 0 ms) start time of spectra (ms)'
      write(LER,*)
     :' -tmax [stop]  (default = full trace) stop time of spectra (ms)'
      write(LER,*)
     :' -rs [irs],    (default=1)        : beginning record number'
      write(LER,*)
     :' -re [ire],    (default=last)     : ending record number'
      write(LER,*)
     :' -ns[ns],      (default=1)        : starting trace number'
      write(LER,*)
     :' -ne[ne]       (default=last)     : ending trace number'
      write(LER,*)
     :' -nvel[nvel]   (default =100)     : number of velocities'
      write(LER,*)
     :' -igt[igt](default = 8)   : an integer for time gate(= 2*igt*dt)'
      write(LER,*)
     :' -dmul [dmul]  (default =1.0)     : distance multiplier'
      write(LER,*)
     :' -tmul [tmul]  (default=1.0)      : sampling interval multiplier'
      write(LER,*)
     :' -vmin[vmin](default 1000) : minimum velocity (ft/sec) for plot'
      write(LER,*)
     :' -vmax[vmax](default 10000) : maximum velocity(ft/sec)for plot'
      write(LER,*) ' '
      write(LER,*)
     :' -V            (default = false)              : verbose output'
      write(LER,*)
     :' -radon  ( default false )   : input data from program radonf'
      write(LER,*)
     :' -slnt   ( default false )   : input data from program slnt'
      write(LER,*)
     :' -S      ( default false )  : horizontal axis is slowness**2'
      write(LER,*)
     :' -log    ( default false )  : velocity (or slowness scale) is'
      write(LER,*)
     :'                              logarithmic'
      write(LER,*)
     :' -W      ( default cosine ) : no cosine weighting'
      write(LER,*) ' '
      write(LER,*)
     :'USAGE: taupspec -N[ntap] -re[re] -rs[rs] -ns[ns] -ne[ne] '
      write(LER,*)
     :'                -vmin[vmin] -vmax[vmax]'
      write(LER,*)
     :'                -nvel[nvel] -tmin[start] -tmax[stop] -igt[igt] '
      write(LER,*)
     :'                -tmul[tmul] -dmul[dmul]  [-W -radon -slnt -V]'
         write(LER,*)
     :'***************************************************************'

      return
      end

c-----
c     get command arguments
c
c     ntap  - C*100  input file name
c     otap  - C*100  output file name
c      ns   - I      start trace
c      ne   - I      stop end trace
c     irs   - I      start record
c     ire   - I      stop end record
c   start   - R      start time of analysis
c    stop   - R      stop time
c    dmul   - R      distance multiplier
c    tmul   - R      sample interval multiplier
c    nvel   - I      number velocities
c    vmin   - R      min velocity
c    vmax   - R      max velocity
c    abslut - L      use absolute semblance
c    slnt   - L      input data from program slnt
c    radon  - L      input data from program radon
c    verbos - L      verbose output or not
c-----
      subroutine cmdln (ntap,otap,start,stop,dmul,tmul,igt,
     &               ns,ne,irs,ire,nvel,vmin,vmax,abslut,verbos,slnt,
     &               radon,logsc,slowness,nowt)
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
      character  ntap*(*), otap*(*)
      integer    argis,ns,ne,irs,ire,nvel,igt,start,stop
      real       dmul,tmul,vmin,vmax
      logical    abslut, verbos, slnt, radon,logsc,slowness,nowt

         call argstr('-N', ntap, ' ', ' ' )
         call argstr('-O', otap, ' ', ' ' )
         call argr4 ('-tmin', START, 0., 0.)
         call argr4 ('-tmax', STOP , 0., 0.)
         call argr4('-dmul',dmul,1.0,1.0)
         call argr4('-tmul',tmul,1.0,1.0)
         call argi4('-igt',IGT,8,8)
         call argi4 ('-rs',irs,1,1)
         call argi4 ('-re',ire,0,0)
         call argi4 ('-ns',ns,1,1)
         call argi4 ('-ne',ne,0,0)
         call argi4 ('-nvel',nvel,101,101)
         call argr4('-vmin',vmin,1000.,1000.)
         call argr4('-vmax',vmax,10000.0,10000.0)

         nowt     = ( argis ('-W') .gt. 0 )
         verbos   = ( argis ('-V') .gt. 0 )
         abslut   = ( argis ('-A') .eq. 0 )
         radon    = ( argis ('-radon') .gt. 0 )
         slnt     = ( argis ('-slnt') .gt. 0 )
         logsc    = ( argis ('-log') .gt. 0 )
         slowness = ( argis ('-S') .gt. 0 )
         if(nvel.gt.SZSPRD) then
            write(LERR,*)'Number velocities too high'
            write(LERR,*)'Must be less than ',SZSPRD
            stop
         endif

      return
      end
