C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c-------------------------------------------------------------------------c
c                                                                         c
c     PROGRAM phzpik 						          c
c									  c
c     AUTHOR  Stephen F. Elston, Princeton University, 1991               c
c									  c
c									  c
c	This program preforms residual statics estimation and             c
c	correction on a set of Normal Moveout corrected CMP gathers       c
c									  c
c	This program is long and complex, but here is a brief summary     c
c       of what is happening:						  c
c									  c
c									  c
c       1)  The command line parameters are read in by the subroutine     c
c               CMDLN.
c	2)  The input file is copied to the output file by the            c
c		subroutine DCOPY0. DCOPY0 also scales the traces so       c
c		that they have normalized RMS amplitudes in the test      c
c               stack window (the window in which crosscorrelations       c
c		are performed).						  c
c       3)  The test stack is made and storted in the test stack file     c
c  		by the subroutine STACK                                   c
c	4)  Memory is dynamically allocated for large floating point      c
c		arrays.							  c
c	5)  An outter loop is run once for each iteration required.       c
c	        This loop has an inner loop that estimated and            c
c               applies the statics corrections to randomly chosen        c
c		shot and receiver points. The subroutine SISORT is        c
c		used to read in shot or receiver gathers at random        c
c		A more complete description of the processing in these    c
c	6)  The files are closed and the program ends                     c
c									  c
c									  c
c     Variable list:                                                      c
c                 (integers -- all i*4)                                   c
c           LUIN  Logical unit number of input                            c
c          LUOUT  Logical unit number of output                           c
c         NBYTES  Number of bytes in trace record                         c
c         LBYTES  Number of bytes in lineheader record                    c
c          NSAMP  Number of samples in each trace                         c
c            NSI  Sampling interval                                       c
c           NTRC  Number of traces per record                             c
c           NREC  Number of records per line                              c
c            IRS  First record to process                                 c
c            IRE  Last record to process                                  c
c             NS  Starting trace number                                   c
c             NE  Ending trace number                                     c
c          NSKTR  Number of traces to skip                                c
c          IFORM  Input data format                                       c
c         NREOUT  Number of output records                                c
c         NTROUT  Number of output traces                                 c
c	  NSTRT   Starting index for input traces		          c
c	  NTLEN   Number of samples used from input trace  		  c
c	 NSHORT	  Number of bytes that output record is shortned	  c
c	  maxsp   Largest shot point index                                c
c	  minsp   Smallest receiver point index     			  c
c	  maxrp   Largest receiver point index                            c
c	  minrp   Smallest shot point index     			  c
c        maxtrc   Maximum number of traces in a gather                    c
c          ndat   Number of entries in sort table                         c
c         item3   Number of bytes allocated to a dynamic array            c
c         item4   Number of bytes allocated to a dynamic array            c
c         item5   Number of bytes allocated to a dynamic array            c
c         item7   Number of bytes allocated to a dynamic array            c
c          iend   Number of last common shot or receiver gather           c
c                 record read.                                            c
c        istart   Number of last common shot or receiver gather           c
c                 record read.                                            c
c                 NOTE: in this program iend = istart always              c
c        ntrace   Number of traces in common shot or receiver gather      c
c	   ised   The seed for the random number generator                c
c           nit   The number of iterations that the static shifts         c
c                 are to be applied                                       c
c          numsp  Number of shotpoints                                    c
c          numrp  Number of reciever points                               c
c	  numtrc  The number of traces in a record on the cmp gather      c
c                 file        						  c
c         obytes  The number of bytes in a output trace                   c
c		  (real)						  c
c	      DT  Time sample interval					  c
c	    TLEN  Trace length in seconds				  c
c          nstrt  The starting sample of the of the stack trace           c
c          damp1  The shot static spatial smoothing constraint parameter  c
c	   damp2  The receiver static spatial smoothing constraint        c
c                 parameter
c	    ssmb  The stack semblance					  c
c          derv2  The second spatial derivative of the static shifts      c
c                 (logical)                                               c
c         VERBOS  If true, print verbos messages                          c
c          QUERY  If true, print query loop and end program               c
c          heap3  True if dynamic memory allocation worked                c
c          heap4  True if dynamic memory allocation worked                c
c          heap5  True if dynamic memory allocation worked                c
c          heap7  True if dynamic memory allocation worked                c
c              R  True if common receiver gather is to be operated on     c
c	       S  True if common shot gather is to be operated on         c
c                 (character)                                             c
c           NTAP  chr*120  Input file name from command line flag         c
c          NTAP2  chr*120  Output file name from command line flag        c
c           OTAP  chr*120  Test stack file name from command line flag    c
c           NAME  chr*4    File name passed to HLH subroutine             c
c                                                                         c
c     Array list:                                                         c
c     NOTE: arrays of dimention (*) are dynamically allocated             c
c          ITR (8320)  i*2    Input data character stream                 c
c         LHED (1500)  i*4    Line header information                     c
c          TRI (4096)  r*4    Data trace                                  c
c       reccnt (4)     i*4    Number of gathers for different sorts       c
c	trccnt (4)     i*4    Number of traces in gathers                 c
c       itrace (2,1500)i*4    CMP and trace index of traces in common     c
c        xlags (1500)  r*4    Array holds the cross correlation lags      c
c                             between trace and reference trace           c
c          aux (4096)  r*4    An auxilliary array used to time shift      c
c                             traces                                      c
c        ihead (*)     i*2    Holds the headers of a single shot or       c
c                             or receiver gather.                         c
c        shead (*)     i*2    Holds the headers of the coresponding       c
c                             cmp stack traces                            c
c       gathin (*)     r*4    Holds the trace samples of a shot or        c
c                             reciver gather                              c
c       strace (*)     r*4    Holds the trace samples of the coresponding c
c                             stack traces                                c
c       reftrc (*)     r*4    This array holds the reference traces       c
c                             (stack trace minus trace being              c
c                              crosscorrelated)				  c
c       ranvec (*)     i*4    Vector of [0,1] uniformaly distributed      c
c                             random deviates, used to compute ranidx     c
c         indx (*)     i*4    Array of scatter/gather indexes used to     c
c                             time shift traces                           c
c       ranidx (*)     i*4    Array of randomly generated shot and        c
c      			      receiver indexes                            c
c        stats (*)     r*4    Array of static shifts applied to shot      c
c                             and reciever points.                        c
c		NOTE: In the arrays ranidx and  stats the first numsp     c
c                     elements corespond to an ordered list of the        c
c                     shot points and the next numrp elements             c
c                     correspond to an ordered list of the reciever       c
c		      points						  c
c                                                                         c
c     Calls:                                                              c
c         (subroutines)                                                   c
c         RTAPE, LBOPEN, HLH, WRTAPE, SAVE, ARGSTR, ARGI4, ERROR          c
c                                                                         c
c         (functions)                                                     c
c         ARGIS                                                           c
c                                                                         c
c-------------------------------------------------------------------------c
c     program ampcor 
c---------------------------------
c        declare variables
c---------------------------------
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
      integer     ITR (SZLNHD), itr0 (SZLNHD)
      integer     ARGIS
#include <f77/pid.h>
      integer     LHED(SZLNHD)
      integer     itrace
      pointer     (wkitrace, itrace(1))
      integer     LUIN, LUOUT, NBYTES, LBYTES, NSAMP, NSI, NTRC, NREC
      integer     IGATH, LUNS
      integer     maxtrc, ndat, maxsp, minsp, maxrp, minrp
      integer     item3, item4, item5
      integer     item7
      integer     itemi
      integer     gimin, gidel
      integer     numsp, numrp, ised, nit
      integer     iend, istart, ntrace,nstrt, iswd
      integer     numtrc, obytes
      
      integer     reccnt(4), trccnt(4)
      real        xgain(1500)

      real        damp1, damp2, ssmb, derv2
 
      real        TRI   (SZLNHD)
      real        aux   (SZLNHD)
      real        workr (SZLNHD)
      real        workg (SZLNHD)
      real        stri  (SZLNHD)
      real        xliv  (SZLNHD)
 
      character   name*6, ntap*100,  mtap*100, otap*100, tnam*100
 
      logical     verbos, query, dbug, med
      logical     heap3, heap4, heap5, heapi
      logical     heap7
      logical     R, S
      logical     aveamp, rmsamp

c---------------------------------------------------------
c  declare dynamically allocated arrays
c--------------------------------------------------------
      integer   ihead, shead
      pointer   (wkshead, shead(1))
      pointer   (wkihead, ihead(1))
      integer   ranidx(30000)
      integer   indx(30000)
      real      stats(30000)
      real      gathin, strace, reftrc, ranvec
      pointer( wkadr3, gathin(1))
      pointer( wkadr4, strace(1))
      pointer( wkadr5, reftrc(1))
      pointer( wkadr7, ranvec(1))

      integer   groups,shots,grprec,grptrc,shtrec,shttrc
      pointer( wkgrp , groups(1))
      pointer( wksht , shots (1))
      pointer( wkgrec, grprec(1))
      pointer( wkgtrc, grptrc(1))
      pointer( wksrec, shtrec(1))
      pointer( wkstrc, shttrc(1))
 
c     equivalence (itr (129), tri (1))
      equivalence (itr (1), lhed (1))
      data  itr0/SZLNHD*0/, name/'AMPCOR'/, nbytes/0/, lbytes/0/
	verbos = .false.
	dbug = .false.

c-------------------------------------------------------------------
c        If '-?' flag is used in command line, execute query
c        loop and end program.
c-------------------------------------------------------------------
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query ) then
         call help()
         stop
      end if

c-------------------------------------------------------------------
c        open printout file
c-------------------------------------------------------------------
#include <f77/open.h>

c---------------------------------------------------------------------------
c        Read program input parameters from command line argument flags
c---------------------------------------------------------------------------
      call cmdln(ntap,mtap,otap,tnam,tstr,tstp,damp1,damp2,med,
     2		       ised,nit,verbos,isrcscl,dbug,aveamp,rmsamp)

c---------------------------------------------------------------------------
c        print out parameters
c---------------------------------------------------------------------------
         write(LERR,*) ' Values read from command line'
	 write(LERR,*) ' Stack trace start time                =',tstr
	 write(LERR,*) ' Stack trace end time                  =',tstp
	 write(LERR,*) ' Shot spatial smoothness parameter     =',damp1
	 write(LERR,*) ' Receiver spatial smoothness parameter =',damp2
	 write(LERR,*) ' The trace header word for phase rot   =',iswd
         write(LERR,*) ' The random number generator seed      =',ised
         write(LERR,*) ' The number of iterations              =',nit
         write(LERR,*) ' Scale factor applied src pt #s in presort tbl=  
     1',                 isrcscl
         write(LERR,*) ' verbose printout = ',verbos
         write(LERR,*) ' debug printout   = ',dbug

c-------------------------------------------
c        Open input and output files
c-------------------------------------------
      call getln( luin1, ntap, 'r', 0)
      call getln( luin,  mtap, 'w+', -2)
      call getln( luout, otap, 'w+', -3)
      if(luin1.lt.0.or.luin.le.0.or.luout.le.0) then
	  write(LERR,*) '  A file name must be specified for each'
          write(LERR,*) '  of the file names for input, output'
          write(LERR,*) '  and test stack'
          stop
      endif

c-----------------------------------------------------------------
c	  Open the sort table file
c-----------------------------------------------------------------
       if(tnam(1:1).eq.' ')then
              luns = LIN
       else
              luns = 22
              open(luns,file=tnam,status='old',
     1                     form='formatted',access='sequential')
              rewind luns
       endif
       if(luin.eq.0)then
              write(LER,*)'DATA FILE must be given as -Nntap'
              write(LER,*)'FATAL ERROR'
              stop
       endif

c------------------------------------------
c  check for old table & abort if necessary
c------------------------------------------
      read(luns,*,end=7) itype
      if(itype .ne. -1) then
          write(LERR,*)'Sort table is old: run presort step again'
          stop
      endif
      go to 9

c-------------------------------------
c  if new table
c-------------------------------------
    7 continue
        write(LERR,*)'Premature end of file on sort table'
        write(LERR,*)'check presort run & repeat if necessary'
        stop
    9 continue

c-------------------------------------------------------------
c  read presort table header
c  set the values of the maximum number of traces in a gather
c  and the maximum number of shot and receiver points.
c-------------------------------------------------------------
      read(luns,*)ndat
      write(LERR,*)'number entries in sort table= ',ndat
      read(luns,*)(reccnt(i),trccnt(i),i=1,4)
      write(LERR,*)(reccnt(i),trccnt(i),i=1,4)
      if(trccnt(1).gt.trccnt(2)) then
        maxtrc = trccnt(1)
      else
        maxtrc = trccnt(2)
      endif

      numrp = reccnt(1)
      numsp = reccnt(2)
      numtot = numsp + numrp
      write(LERR,*)'numrp, numsp= ',numrp, numsp

c-----------------------------------------------------------
c       Allocate storage for receiver & source indices
c-----------------------------------------------------------

        heapi = .true.
        ierrt = 0
        itemi = ndat * SZSMPD
        itemt = 2 * maxtrc * SZSMPD
        item  = maxtrc * ITRWRD* SZSMPD

        call galloc ( wkihead  , item , ierr, iabort )
        ierrt = ierrt + ierr
        call galloc ( wkshead  , item , ierr, iabort )
        ierrt = ierrt + ierr
        call galloc ( wkitrace , itemt, ierr, iabort )
        ierrt = ierrt + ierr
        call galloc ( wkgrp , itemi, ierr, iabort )
        ierrt = ierrt + ierr
        call galloc ( wksht , itemi, ierr, iabort )
        ierrt = ierrt + ierr
        call galloc ( wkgrec, itemi, ierr, iabort )
        ierrt = ierrt + ierr
        call galloc ( wkgtrc, itemi, ierr, iabort )
        ierrt = ierrt + ierr
        call galloc ( wksrec, itemi, ierr, iabort )
        ierrt = ierrt + ierr
        call galloc ( wkstrc, itemi, ierr, iabort )
        ierrt = ierrt + ierr
        if ( ierrt .ne. 0) heapi = .false.

        if (.not. heapi) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemi,'  bytes'
         go to 999
        else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemi,'  bytes'
        endif
         write(LERR,*)' '

c-----------------------------------------------------------
c	Now we find the maximum and minimum number of the
c	shot and receiver point indexes. This requires an 
c	exaustive search of the sort table since the numbering
c	can start and end at arbitrary points
c-----------------------------------------------------------

	maxrp = 0
	minrp = 99999
	maxsp = 0
	minsp = 9999
        read(luns,*) gimin, gidel

	do 95 j= 1, ndat
           read(luns,*,end=9998)i1,i2,i3,i4,i5,i6,i7,i8,i9,i10,i11,i12
           
c------------
c store away
c rec * sht
c indices
           groups(j) = i1
           shots (j) = i2
           grprec(j) = i4
           grptrc(j) = i5
           shtrec(j) = i6
           shttrc(j) = i7
c------------

	   i2 = i2 /10
	   if(i1 .gt. maxrp) maxrp = i1
	   if(i1 .lt. minrp) minrp = i1
	   if(i2 .gt. maxsp) maxsp = i2
           if(i2 .lt. minsp) minsp = i2

  95	continue
9998	continue

 	numsp0 = maxsp - minsp + 1
 	numrp0 = maxrp - minrp + 1

        write(LERR,*)'numrp0, numsp0= ',numrp0, numsp0
        write(LERR,*)'maxrp, minrp, maxsp, minsp= ',
     1                maxrp, minrp, maxsp, minsp

c-------------------------------------------------------------
c  	Prepare the luout (test stack) and the luin (gather)
c	files for random access.
c------------------------------------------------------------
      call sislgbuf( luout, 'off')
      call sislgbuf( luin, 'off')

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

c---------------------------------------------------------------
c	 Copy the input file to the read/write output file were
c	 it will be operated on (ie statics applied)
c        Then close the input file. It is not needed any more.
c------------------------------------------------------------
c        Read line header and save a few important parameters
c------------------------------------------------------------
      lbytes=0
      call RTAPE (luin1, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LERR,*)'ampcor : no header read on unit ',luin
         write(LERR,*)'FATAL'
         goto 999
      endif
      call HLHprt(itr, lbytes, name, 6, LERR)
      call saver(itr, 'NumSmp', nsamp , 0)
      call saver(itr, 'SmpInt', nsi   , 0)
      call saver(itr, 'NumTrc', ntrc  , 0)
      call saver(itr, 'NumRec', nrec  , 0)
      call saver(itr, 'Format', iform , 0)
      call saver(itr, 'UnitSc', unitsc, 0)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, 0)
      endif
c----------------------------------------------------
c        print out header parameters
c----------------------------------------------------
         write(LERR,*) ' Values read from input data set lineheader'
         write(LERR,*) ' Number of Samples/Trace =  ', nsamp
         write(LERR,*) ' Sample Interval         =  ', nsi
         write(LERR,*) ' Traces per Record       =  ', ntrc
         write(LERR,*) ' Records per Line        =  ', nrec
         write(LERR,*) ' Format of Data          =  ', iform
         write(LERR,*) ' Traces/gather           =  ', igath
 
c-------------------------------------------------------------
c        save key parameters in line header; save command
c        line in historical line header. The stack trace length is
c        set to its shorter value.
c-------------------------------------------------------------
      call savhlh( itr, lbytes, lbyout)
      call WRTAPE (luin, itr, lbyout)
      DO  JJ = 1, nrec
          DO  KK = 1, ntrc
            nbytes = 0
            call RTAPE (luin1, itr, nbytes)
               if(nbytes .eq. 0) then
                  write(LERR,*)'COPY:'
                  write(LERR,*)'End of file on input:'
                  write(LERR,*)'  rec= ',i,'  trace= ',j
                  go to 799
               endif
               call WRTAPE (luin, itr, nbytes)
          ENDDO
      ENDDO
799   continue

      numtrc = ntrc
      call LBCLOS ( luin1 )

          nstrt = tstr / nsi + 1
          if (nstrt .le. 0) nstrt = 1
          nend  = tstp / nsi + 1
          if (nend  .gt. nsamp) nend = nsamp
          ntlen = nend - nstrt + 1
          if(ntlen + nstrt .gt. nsamp) ntlen = nsamp - nstrt
          ist = nstrt

c-----------------------------------------------------------------
c	Call the subroutine stack to make the initial test stack
c-----------------------------------------------------------------
      ierflg = 0
      call stack(luin,luout,igath,ntlen,ierror,ierflg,itr,tri,
     1           ssmb,ntrc,nrec,ist,ifmt_StaCor,l_StaCor,stri,
     2           ln_StaCor,xliv)

      ierflg = 1
      if(ierror.ne.0) goto 999
	write(LERR,*) '     '
	write(LERR,*) 'For the intial stack the semblance=',ssmb

c---------------------------------------------------
c  malloc only space we're going to use
c-------------------------------------------------
      heap3 = .true.
      heap4 = .true.
      heap5 = .true.
      heap7 = .true.


      item3 = maxtrc * nsamp * SZSMPD
      item4 = maxtrc * ntlen * SZSMPD
      item5 = maxtrc * ntlen * SZSMPD
      item7 = (numsp + numrp + 1) * SZSMPD * 10

      call galloc( wkadr3, item3, IERR3, 0 )
        if( IERR3 .ne. 0 ) heap3 = .false.
      call galloc( wkadr4, item4, IERR4, 0 )
        if( IERR4 .ne. 0 ) heap4 = .false.
      call galloc( wkadr5, item5, IERR5, 0 )
        if( IERR5 .ne. 0 ) heap5 = .false.
      call galloc( wkadr7, item7, IERR7, 0 )
        if( IERR7 .ne. 0 ) heap7 = .false.

      if ( .not. heap3 .or. .not. heap4 .or. .not. heap5
     2    .or. .not. heap7) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) item3,'  bytes'
         write(LERR,*) item4,'  bytes'
         write(LERR,*) item5,'  bytes'
         write(LERR,*) item7,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) item3,'  bytes'
         write(LERR,*) item4,'  bytes'
         write(LERR,*) item5,'  bytes'
         write(LERR,*) item7,'  bytes'
         write(LERR,*)' '
      endif


c-------------------------------------------------------------------
c	set the array that holds th total acumulated shifts to
c	zero.
c-------------------------------------------------------------------

	do 36 i = 1, numsp + numrp
            stats(i) = 1.
  36	continue

c---------------------------------------------------------------------
c	The main processing loop of the program
c
c	The operations done in this loop are as follows:
c	1) The outer loop is run once for each iteration of 
c          the algorithm.
c       2) A vector containing a random order of surface points
c          is computed.
c       3) An inner loop executes once for each shot or receiver 
c          point. The order in which surface points are operated
c          on is random. 
c          Within this loop the following is done:
c	   A) The shot or receiver gather is read in by the subroutine
c             sisort.
c	   B) The corresponding stack traces are read in.
c          C) A set of reference traces are computed by subtracting
c             the traces in the gather from the corresponding
c	      stack traces.
c          D) The crosscorrelations between the reference traces
c             and the traces in the common surface point gather
c             are computed and the lag of the maximum saved.
c          E) The median lag value is computed.
c	   F) The second derivative spatial smoothness constraint is
c             applied to the time shift.
c          G) The time shift is checked to see that it is in the 
c             bounds of +/-imshft.
c          H) The traces in the gather are time shifted.
c          I) Stack traces are recomputed by adding the time shifted
c             traces in the gather to the reference traces.
c	   J) Operations B-I are only preformed if SISORT finds a
c             nonzero number of traces in the gather.
c 	4) The stack is rebuilt after each iteration to limit round-
c	   off error.
c       5) A new seed for the random number generator is computed.
c
c	The iteration loop.
c---------------------------------------------------------------------
      do 100  it = 1, nit

          write(LER,*)'Starting iteration= ',it,' of ',nit

c-----------------------------------------------------------------
c	Generate the vector of random numbers
c	First get a vector of [0,1] uniform random deviates then
c       convert this to shot and reciever gather indexes
c       Indexes are computed untill one has been generate 
c       for every shot and receiver point
c	If the indexes are not filled in with in 10 iterations
c       They are filled in deterministically.
c-----------------------------------------------------------------
	 itemp1 = numsp + numrp
	 rtemp1 = real(itemp1)
	 jj = 1
	 icount = 0
	 do 200 j=1,itemp1
	    indx(j) = 0
 200	 continue
 210     continue
         icount = icount + 1
         if( icount .gt. 10 ) goto 240
         call vrand(ised, ranvec, 1, 10*itemp1)
         do 220 j = 1, 10*itemp1
	   itemp2 = int( rtemp1 * ranvec(j)) +1
           if(itemp2 .le. itemp1) then
	     if(indx(itemp2) .eq. 0) then
	        ranidx(jj) = itemp2
                indx(itemp2) = 1
                jj = jj + 1
                if( jj .gt. itemp1) goto 230
             endif
           endif
 220	 continue
         ised = ised + 11
	 if( jj .lt. itemp1) goto 210
	 goto 230
 240     continue
         jjj = jj
         kk = 0
	 do 250 j = 1,itemp1 - jj
	   ranidx(jjj) = jj + kk
	   jjj = jjj + 1
           kk = kk + 1
 250	 continue
 230     continue

c-------------------------------------------------------------------
c	Convert the random indexes generated to numbers in the
c	range of the shot and receiver point indexes.
c-------------------------------------------------------------------

	 do 260 j = 1, numsp + numrp
	    if( ranidx(j) .le. numsp) then
	       ranidx(j) = ranidx(j) + minsp - 1
               if (ranidx(j) .gt. 60000) then
                  write(LERR,*)'sht index ',ranidx(j),' too large'
                  write(LERR,*)'stopping at J= ',j
                  stop
               endif
	    else
	       ranidx(j) = ranidx(j) + minrp - 1 + 60000
	    endif
 260	 continue


c-----------------------------------------------------------
c	The inner loop runs once for each source or receiver 
c       point operated on during an iteration.
c-----------------------------------------------------------
	 do 110 j=1,numsp + numrp

      call sislgbuf( luout, 'off')
      call sislgbuf( luin, 'off')



c-------------------------------------------------------------
c	Find the index of the source or receiver point given
c       by the index and set the logicals S and R accordingly
c-------------------------------------------------------------
	  IF(ranidx(j) .gt. 0) then
	    if( ranidx(j) .lt. 60000 ) then
               istart = ranidx(j)
	       iend =   ranidx(j)
	       indxx = ranidx(j) - minsp + 1
               S = .true.
               R = .false.
            else
               istart = ranidx(j) - 60000 - numsp
	       iend =   ranidx(j) - 60000 - numsp
	       indxx = ranidx(j) - 60000 - minrp + 1
               S = .false.
               R = .true.
            endif

             
c--------------------------------------------
c	Read in the surface point gather
c--------------------------------------------
            call sisort(luin,S,R,istart,iend,gathin,ihead,
     1                  itrace,ntrace,verbos,ierror,isrcscl,
     2                  reccnt,trccnt,ndat,groups,shots,grprec,
     3                  grptrc,shtrec,shttrc,gimin,gidel,dbug,
     4                  curtrc,itr,tri,maxtrc,
     5                  l_RecInd,ifmt_RecInd,ln_RecInd,l_SrcLoc,
     6                  ifmt_SrcLoc,ln_SrcLoc,l_DphInd,ifmt_DphInd,
     7                  ln_DphInd,l_RecNum,ifmt_RecNum,ln_RecNum,
     8                  l_TrcNum,ifmt_TrcNum,ln_TrcNum,l_StaCor,
     9                  ifmt_StaCor,ln_StaCor)

c------------------------------------------------------
c	Only execute the estimation and shifting loops 
c	if there are a nonzero number of traces in the
c	gather
c------------------------------------------------------

	    IF(ntrace .ne. 0) then

c---------------------------------------------------
c	clear the trace and header arrays
c--------------------------------------------------
	    call vclr(strace,1,ntlen*maxtrc)
            do 35 kkk = 1, maxtrc*ITRWRD
                shead(kkk) = 0
  35        continue

c------------------------------------------------------------------
c	read in the stack traces corresponding to the traces in 
c	the gather. then save the header and sample values
c
c	In the second part of the loop compute the reference
c	trace, compute the crosscorrelation, and save the
c	maximum lag value.
c------------------------------------------------------------------
	    do 120 jj = 1, ntrace
	       ioff = itrace(jj)
	       call sisseek(luout, ioff)
	       nbytes = 0
               call RTAPE (luout, itr, nbytes)
                 if(nbytes .eq. 0) then
		    write(LERR,*)'Test stack file'
                    write(LERR,*)'End of file on input:'
                    write(LERR,*)'  rec= ',i,'  trace= ',j
                    stop
                 endif
                 call vmov (itr(ITHWP1), 1, tri, 1, ntlen)

c-------------------------------------------------
c  save the values in the trace header.
c-------------------------------------------------
               kk = (jj - 1) * ITRWRD
	       do 130 kkk = 1, ITRWRD
	          shead(kk + kkk) = itr(kkk)
 130	       continue

c----------------------------------------------
c  copy the trace sample values
c---------------------------------------------
              kk = ((jj - 1) * ntlen) +1
              call vmov(tri, 1, strace(kk), 1, ntlen)

c----------------------------------------------------
c	Compute the reference trace.
c----------------------------------------------------
	      kk = (jj - 1) * ntlen
	      kkk = (jj - 1) * nsamp + nstrt - 1
	      do 140 jk = 1, ntlen
		 reftrc(kk + jk) = strace(kk + jk) - gathin(kkk + jk)
 140	      continue


c-------------------------------------------------------------
c	Compute the cross correlation between the trace in the
c       gather and the refernce traces
c
c	NOTE: We use the Math Advantage subroutine CONV rather
c	      than CCOR, since the former computes positive and
c             negative lags.
c-------------------------------------------------------------

	      kk = (jj - 1) * ntlen + 1
 	      kkk = (jj - 1) * nsamp + nstrt

              call vmov (reftrc(kk) , 1, workr, 1, ntlen)
              call vmov (gathin(kkk), 1, workg, 1, ntlen)

              ismp = 0
              do  700  ii = 1, ntlen
                  if (workg(ii) .ne.  0.0) then
                      ismp = ismp + 1
                  endif
700           continue
c             call maxmgv (workg, 1, trcmax, locs, ntlen)
c             call dotpr  (workg, 1, workg, 1, trcmax, ntlen)
c             xgain(jj) = sqrt ( trcmax/float(ntrace) )
              call svemg   ( workg, 1, trcmax, ntlen )
              if (ismp .ne. 0)
     1        xgain(jj) = trcmax / float(ismp)

c-------------------------------------------------------------
c	Save the maximum lag value of the crosscorelation
c------------------------------------------------------------

 120	    continue

c---------------------------------------------------------------
c	Find the gain factor for this gather:
c                default  = max abs
c                ave abs
c                rms 
c---------------------------------------------------------------
	
            if     (aveamp) then
                   call svemg  (xgain, 1, recmax, ntrace)
                   recmax = recmax / float(ntrace)
            elseif (rmsamp) then
                   call svesq  (xgain, 1, recmax, ntrace)
                   recmax = sqrt ( recmax / float(ntrace) )
            elseif ( med  ) then
                   call medmad (xgain, ntrace, recmax)
            else
                   call maxmgv (xgain, 1, recmax, locs, ntrace)
            endif

             if (recmax .ne. 0.0) then
                 gmax = 2047. / recmax
             else
                 gmax = 1.0
             endif

c--------------------------------------------------------------
c	If this is a receiver point and it is not on the line
c	ends then apply a second derivative constraint to 
c	the shift move.
c--------------------------------------------------------------
          IF (damp2 .ne. 0.0) THEN

 	     if(R .and. (iend-1) .ge. minrp .and. 
     1 		(iend+1) .le. maxrp) then
 	        itemp1 = iend - minrp + numsp
 	        itemp2 = itemp1 + 1
 	        itemp3 = itemp2 + 1
 	         temp4 =   stats(itemp2)
 	        derv2 = 0.5 *     (   stats(itemp1) - 2 *  temp4 
     1                              +   stats(itemp3))
 	        rtemp1 = gmax + damp2 * derv2
 	     endif

          ENDIF
c--------------------------------------------------------------
c	If this is a shot point and it is not on the line
c	ends then apply a second derivative constraint to 
c	the shift move.
c--------------------------------------------------------------
          IF (damp1 .ne. 0.0) THEN

 	     if(S .and. (iend-1) .ge. minsp .and. 
     1 		(iend+1) .le. maxsp) then
 	        itemp1 = iend - minsp
 	        itemp2 = itemp1 + 1
 	        itemp3 = itemp2 + 1
 	         temp4 =  stats(itemp2)
 	        derv2 = 0.5 *     (   stats(itemp1) - 2 *  temp4 
     1                              +   stats(itemp3))
 	        rtemp1 = gmax + damp1 * derv2
 	     endif
	
          ENDIF

c------------------------------------------------------------
c	Make sure that the time shift does not exceed the 
c	maximum allowed.
c-----------------------------------------------------------

	    stats( indxx ) =  gmax * stats( indxx )

c------------------------------------------------------------
c	The time has come to apply the shift to the traces in 
c	the gather.
c
c	This is done in the following loop. The if statement
c       controlls if the traces are shifted up or down.
c	In either case a trace is copied to an auxiliary
c       array and then it is copied back with the offset
c       equal to the desired time shift. If the shift
c       is zero we do nothing.
c
c	The stack trace is also recomputed at the end
c       of the loop.
c-----------------------------------------------------------

	    do 160 jj = 1, ntrace
		kk = (jj - 1) * nsamp +1

 		   call vclr (aux, 1, nsamp)
 		   call vmov( gathin(kk), 1, aux, 1, nsamp)
                   do 161 ii = 1, nsamp
                          aux(ii) = aux(ii) * gmax
161                continue
		   call vmov (aux, 1, gathin(kk), 1, nsamp)
 

		kk = kk + nstrt -1
		kkk = (jj - 1)  * ntlen + 1
		call vadd( reftrc(kkk), 1, gathin(kk), 1,
     1                     strace(kkk), 1, ntlen)

 160	    continue

c------------------------------------------------------------------
c	Now we are ready to output the shifted gather traces and
c	the updated stack traces. The loop runs one for each
c       trace in the gather.
c----------------------------------------------------------------

	    do 170 jj = 1, ntrace

c----------------------------------
c	First output a stack trace.
c----------------------------------

	       kkk = (jj - 1) * ITRWRD
	       do 300 kk = 1, ITRWRD 
		  itr(kk) = shead( kk + kkk )
 300	       continue
	       kkk = (jj - 1) * ntlen + 1
	       call vmov( strace(kkk), 1, itr(ITHWP1), 1, ntlen)

	       obytes = SZTRHD + ntlen * SZSMPD
	       ioff = itrace(jj)
	       call sisseek(luout, ioff)
	       call wrtape( luout, itr, obytes)

c---------------------------------
c	Output a gather trace
c-------------------------------

	       kkk = (jj - 1) * ITRWRD
	       do 310 kk = 1, ITRWRD
	          itr(kk) = ihead( kk + kkk )
 310	       continue

c---------------------------------------------------------
c	If this is the last iteration uspdate the static
c---------------------------------------------------------

	       kkk = (jj - 1) * nsamp + 1
	       call vmov( gathin(kkk), 1, itr(ITHWP1), 1, nsamp)

	       obytes = SZTRHD + nsamp * SZSMPD
               jst = maxtrc
	       ioff = (itrace(jj) - 1) * numtrc + itrace(jst+jj)
	       call sisseek(luin, ioff)
	       call wrtape( luin, itr, obytes)
	
 170	     continue
		
	     ENDIF
	   ENDIF


 110	 continue

c-----------------------------------------------
c  	Update the random number generator seed
c-----------------------------------------------
	 ised = ised + 10

c-------------------------------------------------------------
c	rebuild the stack to limit the roundoff errors caused
c       by computing reference traces and updating stack
c       traces
c
c	The input and output files are closed and reopened to 
c	flush the buffers.
c--------------------------------------------------------------
      call LBCLOS ( luin )
      call LBCLOS ( luout )
      call getln( luin,  mtap, 'r+', -2)
      call getln( luout, otap, 'w+', -3)
      if(luin.le.0.or.luout.le.0) then
	  write(LERR,*) '  A file name must be specified for each'
          write(LERR,*) '  of the file names for input, output'
          write(LERR,*) '  and test stack'
          stop
      endif
      call sislgbuf( luout, 'off')
      call sislgbuf( luin, 'off')

      call stack(luin,luout,igath,ntlen,ierror,ierflg,itr,tri,
     1           ssmb,ntrc,nrec,ist,ifmt_StaCor,l_StaCor,stri,
     2           ln_StaCor,xliv)

      if(ierror.ne.0) goto 999
	write(LERR,*) '   '
	write(LERR,*) 'For iteration ',i,'  the stack semblance=',ssmb

 100  continue

	write(LERR,*) '   '
	write(LERR,*) 'The source point gains:'
	do 350 jkl=1,numsp
	    WRITE(LERR,*) jkl,'   ', stats(jkl)
 350	continue
	write(LERR,*) '   '
	write(LERR,*) '   '
	write(LERR,*) 'The receiver point gains:'
	do 351 jkl = numsp + 1 , numsp + numrp
	    write(LERR,*) jkl-numsp,'   ', stats(jkl)
 351	continue


c-------------------------------------------
c        Close files and end program
c-------------------------------------------
  999 continue
      call LBCLOS ( luin )
      call LBCLOS ( luout )
      write(LERR,*)'ampcor : normal end'
      END

c--------------------------------------------------------------------------c
c
c
c     subroutine module sisort
c
c**********************************************************************c
c
c This subroutine is modifed from the usp program sisort. The 
c subroutine will return one or more gathers of either common
c shot or reciver configuration. The gathers are stored in the 
c array gathin. The array itrace tells what the coresponding cdp
c numebr is for each of the traces in the gather. The number
c of traces in the gather is given by ntrace.
c
c
c The subroutine formal parameters are given as:
c luin   -  i*4      The logical unit number of the input data array
c S      -  logical  If true sort to shot gather
c R      -  logical  If true sort to receiver gather
c istart -  i*4      The index of the first gather 
c iend   -  i*4      The index of the last gather
c                    Note: ordinarily istart = iend to read
c                          one gather
c gathin -  real     The array holding the samples of the traces 
c		     in the gather(s)
c ihead  -  i*2      The array holding the headers of the gather.
c itrace -  i*4      Array holding cmp and trace number of 
c		     traces in gathin. The first row are the
c                    cmp or record numbers the second row are
c                    the trace numbers.
c ntrace -  i*4      Number of traces in gathin
c verbos -  logical  If true produce verbose output file.
c ierror -  i*4      Normal completion if ierror = 0
c
c subroutine calls: rtape, hlh, wrtape, save, dagc, odd
c
c**********************************************************************c
c
c
      subroutine sisort(luin,S,R,istart,iend,gathin,ihead,
     1                  itrace,ntrace,verbos,ierror,isrcscl,
     2                  reccnt,trccnt,ndat,groups,shots,grprec,
     3                  grptrc,shtrec,shttrc,gimin,gidel,dbug,
     4                  curtrc,itr,tri,maxtrc,
     5                  l_RecInd,ifmt_RecInd,ln_RecInd,l_SrcLoc,
     6                  ifmt_SrcLoc,ln_SrcLoc,l_DphInd,ifmt_DphInd,
     7                  ln_DphInd,l_RecNum,ifmt_RecNum,ln_RecNum,
     8                  l_TrcNum,ifmt_TrcNum,ln_TrcNum,l_StaCor,
     9                  ifmt_StaCor,ln_StaCor)
c
c     declare variables
c
#include <localsys.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c
      integer     itr ( * ),  ihead(*)
      integer     nsamp, nsi, ntrc, nrec, iform, rind, tind
      integer     luin , lbytes, nbytes
      integer     istart, iend, ntrace
#include <f77/pid.h>
      integer     reccnt(4),trccnt(4)
      integer     trc,rec
      integer     curtrc
      integer     gimin, gidel,recind
      integer     itrace ( maxtrc, 2 )
      real        tri ( * ), gathin(*)
      integer     groups(*),shots(*),grprec(*),grptrc(*)
      integer     shtrec(*),shttrc(*)
      logical     verbos, S, R, dbug

c     equivalence ( itr(129), tri (1) )
      data lbytes / 0 /, nbytes / 0 /
c     SAVE

c--------------------------------------------------------
c     Check that only one of S and R is set
c--------------------------------------------------------

      if(S.and.R) then
          write(LERR,*) '  '
          write(LERR,*) 'Subroutine sisort was called with'
          write(LERR,*) 'both,  S=',S,'   and R=',R
          goto 9999
      endif

c-----------------------------------------------------------
c     Set the state of the input device for random access.
c------------------------------------------------------------
      call sislgbuf( luin, 'off' )

c-------------------------------
c     rewind the input file.
c     read line header of input
c     save certain parameters
c-------------------------------
      call rwd ( luin )
      call rtape ( luin, itr, lbytes            )
      if(lbytes .eq. 0) then
             write(LERR,*)'subroutine sisort: no header read from unit'
             write(LERR,*)luin,'          FATAL'
             goto 9999
      endif

      call saver(itr, 'NumSmp', nsamp , 0)
      call saver(itr, 'SmpInt', nsi   , 0)
      call saver(itr, 'NumTrc', ntrc  , 0)
      call saver(itr, 'NumRec', nrec  , 0)
      call saver(itr, 'Format', iform , 0)
      call saver(itr, 'UnitSc', unitsc, 0)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, 0)
      endif

      if(S)then
            nrecc = reccnt(2)
            jtr   = trccnt(2)
      elseif(R)then
            nrecc = reccnt(1)
            jtr   = trccnt(1)
      endif
      if(iend .ne. 0) then
         nrecc = iend - istart + 1
      else
         iend = 1000000
      endif


      if(verbos .and. dbug) then
          write(LERR,*) '   '
          write(LERR,*) 'In subroutine SISORT'
          write(LERR,*)'number entries in sort table= ',ndat
          write(LERR,*)(reccnt(i),trccnt(i),i=1,4)
      endif


c------------------------------------------------------
c  Zero the book keeping itrace array
c-----------------------------------------------------
      ntrace = jtr
      do  jjj=1,2
          do  jj=1,ntrace
             itrace(jj,jjj) = 0
          enddo
      enddo


c---------------------------------------------------
c	clear the trace and header arrays
c--------------------------------------------------
c - fixed typo - j.m.wade 8/24/92
c	call vclr(gathin,1,nsmap*ntrace)
	call vclr(gathin,1,nsamp*ntrace)
        do 35 kkk = 1, ntrace*ITRWRD
              ihead(kkk) = 0
  35    continue

c-----
c     BEGIN PROCESSING
c     process desired trace records
c-----

      curtrc = 1
      if(verbos .and. dbug)write(LERR,*)'# recs= ',nrecc, 
     1           ' # trc/rec= ',jtr,
     2           ' istart= ',istart,' iend= ',iend

c--------------------------------------------------------
c	Take care of the strange fact that souce indexes
c       are multiplied by isrcscl in presort table
c-------------------------------------------------------
	if(S) then
	    istart = isrcscl * istart
	    iend = isrcscl * iend
	endif

c----------
c  read each line of presort table
c  pick off appropriate index and its associated rec & trc #'s
c  these #'s are then used to pick off the correct sorted trace
c  from somewhere in the input disk data set
c----------

      DO  1000 jjj = 1, ndat

            if(R)then
                  jj  = groups(jjj)
                  rec = grprec(jjj)
                  trc = grptrc(jjj)
            elseif(S)then
                  jj  = shots (jjj)
                  rec = shtrec(jjj)
                  trc = shttrc(jjj)
            endif

c-------------
c  is the sort index within the
c  specified range?
c-------------
         IF (jj .ge. istart .and. jj .le. iend) THEN



c------------------------------------------------------------
c   Set the record pointer and read a trace.
c----------------------------------------------------------

            ioff = (rec-1)*ntrc + trc
            call sisseek (luin, ioff)

            nbytes = 0
            call rtape( luin, itr, nbytes)

            call saver2(itr,ifmt_SrcLoc,l_SrcLoc,
     1                  ln_SrcLoc, isi, 1)
            call saver2(itr,ifmt_DphInd,l_DphInd,
     1                  ln_DphInd, idi, 1)
            call saver2(itr,ifmt_RecInd,l_RecInd,
     1                  ln_RecInd, igi, 1)

 
c           if( dbug ) then
c             call maxmgv (tri, 1, wrkmax, locs, nsamp)
c             write(LER,*)'r/t= ',rec,trc,ic,igi,isi,idi,' g= ',
c    1                    wrkmax,nsamp,ntrace,jjj,S,R,istart,itr(125)
c           endif

            if(nbytes .eq. 0) then
              write(LERR,*)'FATAL ERROR'
              write(LERR,*) 'In subroutine SISORT'
              write(LERR,*)' '
              write(LERR,*)'read failed for rec, trc= ',rec,trc
              write(LERR,*)' '
              write(LERR,*)'presort table line...'
              write(LERR,*)i1,i2,i3,i4,i5,i6,i7,i8,i9,i10,i11,i12
              write(LERR,*)' '
              write(LERR,*)'Try rerunning presort on your data to build'
              write(LERR,*)'a new table.  Has your data changed since'
              write(LERR,*)'sort table was built?'
              goto 9999
            endif

            call saver2(itr,ifmt_RecInd,l_RecInd,
     1                  ln_RecInd, recind, 1)
            call saver2(itr,ifmt_RecNum,l_RecNum,
     1                  ln_RecNum, rind  , 1)
            call saver2(itr,ifmt_TrcNum,l_TrcNum,
     1                  ln_TrcNum, tind  , 1)
            call saver2(itr,ifmt_StaCor,l_StaCor,
     1                  ln_StaCor, istat , 1)



c-----------------------------------------------
c	If this is not a dummy trace then save it
c-----------------------------------------------

	    IF(istat .ne. 30000) then

c----------------------------------------------------------------
c	Save the record (CMP) and trace index
c----------------------------------------------------------------
	       itrace(curtrc,1) = rec
               itrace(curtrc,2) = trc


c-------------------------------------------------
c  save the values in the trace header.
c-------------------------------------------------
               kk = (curtrc - 1) * ITRWRD
	       do 30 kkk = 1, ITRWRD
                  ihead(kk + kkk) = itr(kkk)
   30	       continue

c----------------------------------------------
c  copy the trace sample values
c---------------------------------------------
              kk = ((curtrc - 1) * nsamp) +1
              call vmov(itr(ITHWP1), 1, gathin(kk), 1, nsamp)


c--------------------------------------------
c  Update the trace count
c-------------------------------------------
               curtrc = curtrc + 1
	    ENDIF

c----------------------------------------------------------------
c  If we have found the full complement of traces leave the loop
c---------------------------------------------------------------

           if(curtrc.gt.ntrace*nrecc) goto 2000


         ENDIF

 1000    CONTINUE
 2000	 CONTINUE

c---------------------------------
c    Update ntrace
c--------------------------------
	 ntrace = curtrc - 1

 
c-----------------------------------------------------------
c  set completion status and get out.
c---------------------------------------------------------

      ierror = 0
      goto 9000
 9999 continue
      ierror = 1
 9000 continue
      return
      end
c
c--------------------------------------------------------------------------c
c									   c
c	Subroutine stack.						   c
c									   c
c	This subroutine is based on the sis program STACK.	           c
c        								   c
c        This subroutine only makes linear stacks, so that all of the      c
c	 features of the program stack that make diversity or              c
c        weighted stacks have been removed                                 c
c                                                                          c

      subroutine stack(luin,luout,igath,ntlen,ierror,ierflg,itr,tri,
     1                 ssmb,ntrc,nrec,ist,ifmt_StaCor,l_StaCor,stri,
     2                 ln_StaCor,xliv)

#include <localsys.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
      integer     ITR (*)
#include <f77/pid.h>
      integer     LUIN, LUOUT, NBYTES, LBYTES, NTRC, NREC
      integer     obytes
      real        ssmb, powin, powout
 
      real        TRI (*), STRI (*), XLIV (*)
 
c---------------------------------------------
c       Set the input and output power to zero
c---------------------------------------------
 
       powin = 0.0
       powout = 0.0
c----------------------------------------
c         rewind the input file
c--------------------------------------
      call rwd(luin)
 
c----------------------------------------------------
c        Read line header and save parameters
c----------------------------------------------------
      lbytes=0
      call RTAPE (luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LERR,*)'phzcor : no header read on unit ',luin
         write(LERR,*)'FATAL'
         goto 999
      endif
      call saver(itr, 'NumSmp', nsampi, 0)
      call saver(itr, 'SmpInt', nsii  , 0)
      call saver(itr, 'NumTrc', ntrci , 0)
      call saver(itr, 'NumRec', nreci , 0)
      call saver(itr, 'Format', iformi, 0)
      call saver(itr, 'UnitSc', unitsc, 0)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, 0)
      endif
 
      obytes = (ITRWRD + ntlen) * SZSMPD
c-------------------------------------------------------------
c        save key parameters in line header; save command
c        line in historical line header. The stack trace length is
c        set to its shorter value.
c-------------------------------------------------------------
       call savew( itr, 'NumTrc', 1    , 0)
       call savew( itr, 'NumRec', nrec , 0)
       call savew( itr, 'NumSmp', ntlen, 0)
      call savhlh( itr, lbytes, lbyout)
      call WRTAPE (luout, itr, lbyout)
 
c----------------------------------------------------------------------
c                    Main processing loop
c
c        From the first selected record (irs) to last selected
c           record (ire) do the following:
c           (1)  Zero the storage array (stri).
c           (2)  Sum from trace (ns) to trace (ne) all selected
c                traces in each record.
c           (3)  Write the summed trace of that record to the
c                output file.
c----------------------------------------------------------------------
      do 200 i = 1, nrec
 
c        ------------------
c        Zero storage array
c        ------------------
         call vclr (stri,1,ntlen)
         call vclr (xliv,1,ntlen)
 
c        -------------------
c        Sum selected traces
c        -------------------
         inorm = 0
         do 400 K = 1, ntrc
            nbytes = 0
            call RTAPE (luin, itr, nbytes)
               if(nbytes .eq. 0) then
                  write(LERR,*)'STACK:'
                  write(LERR,*)'End of file on input:'
                  write(LERR,*)'  rec= ',i,'  trace= ',k
                  go to 400
               endif
              call vmov (itr(ITRWRD+ist), 1, tri, 1, ntlen)
              call saver2(itr,ifmt_StaCor,l_StaCor,
     1                    ln_StaCor, istatic, 1)
 
 
            if(istatic .ne. 30000) then
c------------------------------
c       Sum the traces.
c------------------------------
 
                  do 500 L = 1, ntlen
                     if (tri(L) .ne. 0.0) xliv(L) = xliv(L) + 1.0
                     stri (L) = tri(L) + stri (L)
                     powin  = tri (L) * tri (L) + powin
  500             continue
 
c--------------------------------
 
            endif
 
  400    continue
 
c        ----------------------------------
c        Write stacked trace to output file
c        ----------------------------------
 
              do 601 M = 1, ntlen
                  amp = 1.
                  if (xliv(M) .ne. 0.0) amp = xliv(M)
                  tri (M) = stri (M) / amp
                  powout = stri (M) * stri (M) + powout
601           continue
 
c------------------------------------------------------------
c        Output the shortned trace
c-----------------------------------------------------------
         call vmov (tri, 1, itr(ITHWP1), 1, ntlen)
         call savew2(itr,ifmt_StaCor,l_StaCor,
     1               ln_StaCor, 0, 1)
         call WRTAPE (luout, itr, obytes)
 
  200 continue
 
c-----------------------------------------------
c       Compute the stack semblance
c----------------------------------------------
 
      if( ntrc .gt. 0 .and. powin .gt. 0.0 ) then
          ssmb = sqrt( powout / ( real( ntrc * ntrc) * powin ) )
      else
          ssmb = 0.0
      endif
 
      ierror = 0
      goto 1000
 999  continue
      ierror = 1
 1000 continue

      return
      end


C
C*********************************************************************
C*********************************************************************
C
C	THIS SUBROUTINE COMPUTES THE MEDIAN
C	INPUT DATA
C	VECTOR.
C	
C	THE PROCESSING SEQUENCE IS AS FOLLOWS;
C	1)	THE INPUT VECTOR IS SORTED
C	2)	IF THE NUMBER OF ELEMENTS IN THE INPUT VECTOR IS
C		ODD THE MEDIAN IS TAKEN AS THE MIDDLE ELEMENT OF THE
C		SORTED VECTOR. IF THERE ARE AN EVEN NUMBER OF ELEMENTS
C		THE MEDIAN IS TAKEN AS THE ARITHMATIC MEAN OF THE
C		TWO MIDDLE ELEMENTS.
C	THE SORTING IS DONE USING THE SORT SUBROUTINE SHELL.
C
C
	SUBROUTINE MEDMAD(INDAT,N,MEDOUT)
C
C	THE FORMAL PARAMETERS OF THE SUBROUTINE ARE DEFINED.
C
C	INDAT-THE INPUT DATA VECTOR OF FLOATING POINT NUMBERS OF
C		LENGTH N.
C	N-THE NUMBER OF ELEMENTS IN THE INPUT ARRAY.
C	MEDOUT-THE VALUE OF THE MEDIAN OF THE INPUT ARRAY.
C
C
	REAL*4 INDAT(1)
	REAL*4 MEDOUT
	INTEGER*4 N
C
C	THE INTERNAL VARIABLES OF THE SUBROUTINE ARE DECLARED.
C
C	INDX-IS THE INDEX OF THE MIDDLE VALUE OF THE SORTED ARRAYS
C
	INTEGER*4 INDX
C
C	THE WORK ARRAY IS SORTED
C
	CALL SHELL(N,INDAT)
C
C	NOW THE MEDIAN VALUE IS FOUND.
C
	IF(MOD(N,2).EQ.1) THEN
C
C	THE NUMBER OF ELEMENTS IS ODD.
C
	  INDX=N/2+1
	  MEDOUT=INDAT(INDX)
	ELSE
C
C	THE NUMBER OF ELEMENTS IS EVEN.
C
	 INDX=N/2
	 MEDOUT=(INDAT(INDX)+INDAT(INDX+1))/2.0
	ENDIF
	RETURN
	END
C
C	THIS SUBROUTINE SORTS THE INPUT ARRAY INTO ASSCENDING
C	NUMERICAL ORDER. THE ROUTINE USED SHELL'S METHOD
C	OF SORTING. THIS METHOD IS USEFULL IF THE NUMBER
C	OF ELEMENTS BEING SORTED IS GREATER THAN 50 BUT LESS
C	THAN A FEW THOUSAND.
C
C
	SUBROUTINE SHELL(N,IN)
	REAL*4 IN(1)
	INTEGER*4 N
	REAL*4 ALN2I,TINY
	REAL*4 T
	INTEGER*4 NN,K,I,J,L,M,LOGNB2
	PARAMETER(ALN2I=1.0/0.69314718, TINY=1.0E-6)
C
	LOGNB2=INT(ALOG(FLOAT(N))*ALN2I+TINY)
	M=N
	DO 10 NN=1,LOGNB2
	  M=M/2
	  K=N-M
	  DO 20 J=1,K
	    I=J
  30	    CONTINUE
	    L=I+M
	    IF(IN(L).LT.IN(I)) THEN
	      T=IN(I)
	      IN(I)=IN(L)
	      IN(L)=T
	      I=I-M
	      IF(I.GE.1) GO TO 30
	    ENDIF
  20	  CONTINUE
  10	CONTINUE
	RETURN
	END




c-------------------------------------------
c        online help
c-------------------------------------------
      subroutine help
#include <f77/iounit.h>

         write(LER,*)
     1 '***************************************************************'
         write(LER,*)' '
         write(LER,*)
     1 'ampcor: do surface consistent gain correction'
         write(LER,*)
     1 'Run this program by typing: ampcor and the following arguments'
         write(LER,*)
     1 ' -N[ntap]    (stdin     )      : Input data file name'
         write(LER,*)
     1 ' -O[mtap]    (no default)      : Output data file name'
         write(LER,*)
     1 ' -T[otap]    (no default)      : Test stack data file name'
         write(LER,*)
     1 ' -S[tnam]    (no default)      : Input data sort file name'
         write(LER,*)' '
         write(LER,*)
     1 ' -tstr[tstr] stack trace start time (ms) (default = none)'
         write(LER,*)
     1 ' -tstp[tstp] stack trace start end time (ms) (default = none)'
         write(LER,*)
     1 ' -dsp[damp1] shot static smoothness parameter',
     2       ' (default = 0.0)'
         write(LER,*)
     1 ' -drp[damp2] receiver static smoothness parameter',
     2      ' (default = 0.0)'
         write(LER,*)
     1 ' -ised[ised] seed for random number generator',
     2      ' (default = 1256)'
         write(LER,*)
     1 ' -nit[nit] number of statics estimation iterations',
     2          ' (default = 2)'
         write(LER,*)
     1 ' -scl[srcscl] scale factor applied to src pt #s in persort tbl',
     2          ' (default = 10)'
         write(LER,*)' '
         write(LER,*)
     1 ' -M  gain correction based on Median Amplitude, or'
         write(LER,*)
     1 ' -A  gain correction based on Average Amplitude, or'
         write(LER,*)
     1 ' -R  gain correction based on RMS Amplitude, or'
         write(LER,*)
     1 ' (default) gain correction based on Maximum Amplitude'
         write(LER,*)
     1 ' For most data Median (-M) recommended'
         write(LER,*)
     1 ' -V  Verbose mode.  All command line and lineheader parameters'
         write(LER,*)
     1 '                    printed to standard error output'
         write(LER,*)' '
         write(LER,*)
     : 'USAGE:  ' 
         write(LER,*)
     : 'ampcor -N[ntap] -O[mtap] -T[otap] -S[tnam] -tstr[tstr]'
         write(LER,*)
     : '       -tstp[tstp] -dsp[] -dsr[] -ised[] -nit[] -scl[]'
         write(LER,*)
     : '       [ -M -A -R -V -?]'
         write(LER,*)
     1 '***************************************************************'

      return
      end

      subroutine cmdln(ntap,mtap,otap,tnam,tstr,tstp,damp1,damp2,med,
     2		       ised,nit,verbos,isrcscl,dbug,aveamp,rmsamp)
c-----
c     get command arguments
c
c     ntap  - C*100  input file name
c     mtap  - C*100  output file name
c     otap  - C*100  test stack file name
c     tnam  - C*100  name of the sort file
c     tstr  - R      start time of stack trace
c     tstp  - R      end time of stack trace
c    damp1  - R      shot static spatial constraint parameter
c    damp2  - R      receiver static spatial constraint parameter
c     ised  - I      The seed for the random number generator used
c                    to compute the order surface points are visited
c   isrcscl - I      scaler to apply to source istart/iend in sisort
c       nit - I      Number of static correction iterations.
c      norm - L      normalize stacked trace by # live traces
c      semb - L      output semblance
c     semwt - L      output semblance weghted stack
c    verbos - L      verbose output or not
c-----
      character  ntap*100,  mtap*100, otap*100, tnam*100
      integer    ised, nit
      integer    argis
      real       tstr,tstp,damp1,damp2
      logical    verbos, dbug, aveamp, rmsamp, med

    	verbos = .false.
    	dbug   = .false.

c----------------------------------------------------------------------------
c        ARGXXX has parameters
c             ( flag, variable name, default value, format error value )
c-----------------------------------------------------------------------------
      call ARGSTR ('-N', ntap, ' ', ' ' )
      call ARGSTR ('-O', mtap, ' ', ' ' )
      call ARGSTR ('-T', otap, ' ', ' ' )
      call ARGSTR ('-S', tnam, ' ', ' ' )
      call ARGR4  ('-tstr', tstr, 0.0, 0.0)
      call ARGR4  ('-tstp', tstp, 99999.,  99999.)
      call ARGR4  ('-dsp', damp1,  0.0,  0.0)
      call ARGR4  ('-drp', damp2,  0.0,  0.0)
      call ARGI4  ('-ised', ised, 1256, 1256)
      call ARGI4  ('-nit', nit, 2, 2)
      call ARGI4  ('-scl', isrcscl, 10, 10)
 
      med    = ( argis ('-M') .gt. 0 )
      aveamp = ( argis ('-A') .gt. 0 )
      rmsamp = ( argis ('-R') .gt. 0 )
      dbug   = ( argis ('-dbug') .gt. 0 )
      verbos = ( argis ('-V') .gt. 0 )

      return
      end
