C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c*****************************************************************
c                                                                 
c     SUBROUTINE - WeightamAB
c                                                               
c                AMOCO PRODUCTION CO. PROPRIETARY
c                 TO BE MAINTAINED IN CONFIDENCE
c                                                        
c     ABSTRACT - Compute weighted stack equivalents for B0 and B1
c                using least squares solution for truncated
c                Aki and Richards eqn for input angle matrix
c           R()=B0 + B1*tan^2(theta)+(-B1+B0/(c+1))tan^2*sin^2
c                     or
c           R()=B0(1.+1./(1+c)*tan^2*sin^2)+B1*sin^2
c                                                            
c     USAGE                                                 
c                                                          
c       call WeightamAB(nsamp,ntr, data, stacka, 
c                     stackb, angles, astrt,aend,c,ier)
c                                                       
c            nsamp    =  Number of samples per trace (rows)
c              ntr    =  Number of traces per record (columns)
c              data   =  Matrix of trace data
c             stacka  =  Returned weighted stack for B0
c             stackb  =  Returned weighted stack for B1
c             angles  =  Matrix of incident angles from ray tracing
c             astrt   =  Start angle for sum for angle-limited stack
c             aend    =  End angle for sum for angle-limited stack
c              C      =  User-supplied constant from Gardner relation
c                        for Vp-density (density=K*V^C)
c             IERR    =  Error flag 
c                                                      
c     ERROR/RETURN CODES - ier                 
c                          1 = singular matrix
c                                              
c***************************************************************
      subroutine WeightamAB(nsamp,ntr, data, 
     :  stacka, stackb, angles,astrt,aend,c, vp,ier)
#include <localsys.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c                                                            
      real data(*), stacka(nsamp), stackb(nsamp)
      real angles(*),astrt,aend,c,vp(*)
      integer ier
      real Sxx1, Sxx2, Sx1y, Sx2y

c +===========================+
c | Build the matrix elements |
c | Get the solutions         |
c +===========================+
      add = 1./(1.+c)
      vp1 = vp(1)
      do i=1,nsamp
       Sxx1  = 0.
       Sxx2  = 0.
       Sx1y  = 0.
       Sx2y  = 0.
       Sx1x2 = 0.
       n     = 0
       vp2 = vp(i)
       vc = vp1/vp2
       do j=1,ntr
        indx=(j-1)*nsamp+i
        theta=angles(indx)
        y = data(indx)
        sine = sin(theta)
        tang = tan(theta)
*       if(sine.gt.vc)y=0.
        if(theta.ge.astrt.and.theta.le.aend.and.y.ne.0.0)then
         sine = sine*sine
         tang = tang*tang
         ax1 = (1.0 + add*tang*sine)
         ax2 = sine
         if(c.eq.0.0)ax2=tang
         Sxx1 = Sxx1 + ax1*ax1
         Sxx2 = Sxx2 + ax2*ax2
         Sx1x2 = Sx1x2 + ax1*ax2
         Sx1y  = Sx1y + ax1*y
         Sx2y  = Sx2y + ax2*y
         n = n + 1
        endif
       end do
       if(n.ge.3)then
        D = Sx1x2*Sx1x2 - Sxx1*Sxx2
        x1 = Sx2y*Sx1x2 - Sx1y*Sxx2
        x2 = Sx1x2*Sx1y - Sx2y*Sxx1
        if(D.ne.0)then
         stacka(i)=x1/D
         stackb(i)=x2/D
        else
         stacka(i)=0.
         stackb(i)=0.
        endif
       else
        stacka(i)=0.
        stackb(i)=0.
       endif
       vp1 = vp2
      end do
      return
      end
c******************************************************************
c                                                                 
c     SUBROUTINE - WeightamIJ 
c                                                               
c                AMOCO PRODUCTION CO. PROPRIETARY
c                 TO BE MAINTAINED IN CONFIDENCE
c                                                        
c     ABSTRACT - Compute weighted stack equivalents for delVp/Vp
c                and delVs/Vs using Smith and Gidlow (1987) solution
c                for A and B in the equation
c                   R()=A*delVp/Vp + B*delVs/Vs
c                       A=0.5*(1+tan^2+c-cKsin^2)
c                       B=-K*sin^2, where 
c                       K = (2*Vs/Vp)^2
c                 for an input angle matrix
c                                                            
c     USAGE                                                 
c                                                          
c        call WeightamIJ(nsamp,ntr, data, stacka, 
c          stackb, ratio, angles, astrt,aend,c,ier)
c                                                       
c            nsamp    =  Number of samples per trace (rows)
c              ntr    =  Number of traces per record (columns)
c              data   =  Matrix of trace data
c             ratio   =  Vs/Vp ratio squared vector
c             stacka  =  Returned weighted stack for delVp/Vp
c             stackb  =  Returned weighted stack for delVs/Vs
c             angles  =  Matrix of incident angles from ray tracing
c             astrt   =  Start angle for sum for angle-limited stack
c             aend    =  End angle for sum for angle-limited stack
c               c     =  Gardner coefficient (rho = K*Vp^c)
c             IERR    =  Error flag to be returned by crvray(creray)
c                                                      
c     ERROR/RETURN CODES - ier                 
c                          .ne. 0 = memory allocation error
c                                              
c***************************************************************
      subroutine WeightamIJ(nsamp,ntr, data,
     *  stacka, stackb, ratio,angles,astrt,aend,c,vp, ier)
#include <localsys.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c                                                            
      real data(*), stacka(nsamp), stackb(nsamp),ratio(*)
      real angles(*),astrt,aend,c,vp(*)
      integer ier
      real Sxx1, Sxx2, Sx1y, Sx2y
      real K

c +===========================+
c | Build the matrix elements |
c | Get the solutions         |
c +===========================+
      vp1 = vp(1)
      do i=1,nsamp
       Sxx1  = 0.
       Sxx2  = 0.
       Sx1y  = 0.
       Sx2y  = 0.
       Sx1x2 = 0.
       n     = 0
       K = ratio(i)*4.
       vp2 = vp(i)
       vc = vp1/vp2
       do j=1,ntr
        indx=(j-1)*nsamp+i
        theta=angles(indx)
        y = data(indx)
        sine = sin(theta)
        tang = tan(theta)
*       if(sine.gt.vc)y=0.
        if(theta.ge.astrt.and.theta.le.aend.and.y.ne.0.0)then
         sine = sine*sine
         tang = tang*tang
         ax1 = 0.5*(1.0 + tang + c *(1. - K*sine))
         ax2 = -sine*K
         Sxx1 = Sxx1 + ax1*ax1
         Sxx2 = Sxx2 + ax2*ax2
         Sx1x2 = Sx1x2 + ax1*ax2
         Sx1y  = Sx1y + ax1*y
         Sx2y  = Sx2y + ax2*y
         n = n + 1
        endif
       end do
       if(n.ge.3)then
        D = Sx1x2*Sx1x2 - Sxx1*Sxx2
        x1 = Sx2y*Sx1x2 - Sx1y*Sxx2
        x2 = Sx1x2*Sx1y - Sx2y*Sxx1
        if(D.ne.0)then
         stacka(i)=x1/D
         stackb(i)=x2/D
        else
         stacka(i)=0.
         stackb(i)=0.
        endif
       else
        stacka(i)=0.
        stackb(i)=0.
       endif
       vp1 = vp2
      end do
      return
      end
c*****************************************************************
c                                                                 
c     SUBROUTINE - WeightamB0
c                                                               
c                AMOCO PRODUCTION CO. PROPRIETARY
c                 TO BE MAINTAINED IN CONFIDENCE
c                                                        
c     ABSTRACT - Compute weighted stack equivalents for B0, B1, and
c                B2 using least squares solution for Aki and Richards
c                eqn             
c                   R()=A*B0 + B*B1 + C*B2
c                     A = 1.0
c                     B = tan^2(theta)
c***************      B = sin^2(theta)
c                     C = tan^2(theta) * sin^2(theta)
c                                                            
c     USAGE                                                 
c                                                          
c       call WeightamB0(nsamp,ntr, data, stacka, 
c                     stackb, stackc, angles, astrt,aend,ier)
c                                                       
c            nsamp    =  Number of samples per trace (rows)
c              ntr    =  Number of traces per record (columns)
c              data   =  Matrix of trace data
c             stacka  =  Returned weighted stack for B0
c             stackb  =  Returned weighted stack for B1
c             stackc  =  Returned weighted stack for B2
c             angles  =  Matrix of incident angles from ray tracing
c             astrt   =  Start angle for sum for angle-limited stack
c             aend    =  End angle for sum for angle-limited stack
c             IERR    =  Error flag 
c                                                      
c     ERROR/RETURN CODES - ier                 
c                          .ne. 0 = memory allocation error
c                                              
c***************************************************************
      subroutine WeightamB0(nsamp,ntr, data,
     :  stacka, stackb, stackc, ratio, angles,astrt,aend, vp,
     :  ier)
#include <localsys.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c                                                            
      real data(*), stacka(nsamp), stackb(nsamp),vp(*)
      real ratio(*),angles(*),astrt,aend, stackc(nsamp)
      real*8  Sxx1,Sxx2,Sx1y,Sx2y,Sx1x2,Sxx3,Sx1x3,Sx2x3
      real*8  ax1, ax2, ax3, D
      integer ier
c +===========================+
c | Build the matrix elements |
c | Get the solutions         |
c +===========================+
      vp1 = vp(1)
      do i=1,nsamp
       Sxx1 = 0.
       Sxx2 = 0.
       Sxx3  = 0.
       Sx1x2 = 0.
       Sx1x3 = 0.
       Sx2x3 = 0.
       Sx1y  = 0.
       Sx2y  = 0.
       Sx3y  = 0.
       n     = 0
       vp2 = vp(i)
       vc = vp1/vp2
       do j=1,ntr
        indx=(j-1)*nsamp+i
        theta=angles(indx)
        y = data(indx)
        sine = sin(theta)
        tang = tan(theta)
        sine = sine*sine
        tang = tang*tang
*       if(sine.gt.vc)y = 0.
        if(theta.ge.astrt.and.theta.le.aend.and.y.ne.0.0)then
         ax1 = 1.
         ax2 = tang
*        ax2 = sine
         ax3 = sine*tang
         Sxx1  = Sxx1  + ax1*ax1
         Sxx2  = Sxx2  + ax2*ax2
         Sxx3  = Sxx3  + ax3*ax3
         Sx1x2 = Sx1x2 + ax1*ax2
         Sx1x3 = Sx1x3 + ax1*ax3
         Sx2x3 = Sx2x3 + ax2*ax3
         Sx1y  = Sx1y + ax1*y
         Sx2y  = Sx2y + ax2*y
         Sx3y  = Sx3y + ax3*y
         n = n+1
        endif
       end do
       if(n.ge.4)then
        D = Sxx1*Sxx2*Sxx3-Sxx1*Sx2x3*Sx2x3-Sx1x2*Sx1x2*Sxx3
        D = D + Sx1x2*Sx2x3*Sx1x3
        D = D + Sx1x3*Sx1x2*Sx2x3 - Sx1x3*Sx1x3*Sxx2
        x1 = Sx1y*(Sxx2*Sxx3-Sx2x3*Sx2x3)
        x1 = x1 - Sx1x2*(Sx2y*Sxx3 - Sx3y*Sx2x3)
        x1 = x1 + Sx1x3*(Sx2y*Sx2x3 - Sx3y*Sxx2)
        x2 = Sxx1*(Sx2y*Sxx3 - Sx3y*Sx2x3)
        x2 = x2 - Sx1y*(Sx1x2*Sxx3 - Sx1x3*Sx2x3)
        x2 = x2 + Sx1x3*(Sx1x2*Sx3y - Sx2y*Sx1x3)
        x3 = Sxx1*(Sxx2*Sx3y - Sx2y*Sx2x3)
        x3 = x3 - Sx1x2*(Sx1x2*Sx3y - Sx1x3*Sx2y)
        x3 = x3 + Sx1y*(Sx1x2*Sx2x3 - Sx1x3*Sxx2)
        if(D.ne.0)then
         stacka(i)=x1/D
         stackb(i)=x2/D
         stackc(i)=x3/D
        else
         stacka(i)=0.
         stackb(i)=0.
         stackc(i)=0.
        endif
       else
        stacka(i)=0.
        stackb(i)=0.
        stackc(i)=0.
       endif
       vp1 = vp2
      end do
      return
      end
c*****************************************************************
c                                                                 
c     SUBROUTINE - WeightamVrho
c                                                               
c                AMOCO PRODUCTION CO. PROPRIETARY
c                 TO BE MAINTAINED IN CONFIDENCE
c                                                        
c     ABSTRACT - Compute weighted stack equivalents for delVp/Vp,
c                delVs/Vs, and delrho/rho using least squares solution
c                for Aki and Richards eqn for input angle matrix
c           R()=A*delVp/Vp + B*delVs/Vs + C * delrho/rho
c             A = 0.5*(1.+ tan^2(theta))
c             B = -K*sin^2(theta)
c             C = +0.5 * (1.0 - K*sin^2(theta))
c             K = 4*(Vs/Vp)^2
c                                                            
c     USAGE                                                 
c                                                          
c       call WeightamVrho(nsamp,ntr, data, stacka, 
c                     stackb, stackc, ratio, angles, astrt,aend,ier)
c                                                       
c            nsamp    =  Number of samples per trace (rows)
c              ntr    =  Number of traces per record (columns)
c              data   =  Matrix of trace data
c             ratio   =  Vs/Vp ratio squared vector
c             stacka  =  Returned weighted stack for delVp/Vp
c             stackb  =  Returned weighted stack for delVs/Vs
c             stackc  =  Returned weighted stack for delrho/rho
c             angles  =  Matrix of incident angles from ray tracing
c             astrt   =  Start angle for sum for angle-limited stack
c             aend    =  End angle for sum for angle-limited stack
c             IERR    =  Error flag 
c                                                      
c     ERROR/RETURN CODES - ier                 
c                          .ne. 0 = memory allocation error
c                                              
c***************************************************************
      subroutine WeightamVrho(nsamp,ntr, data,
     :  stacka, stackb, stackc, ratio, angles,astrt,aend, 
     :  ier)
#include <localsys.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c                                                            
      real data(*), stacka(nsamp), stackb(nsamp),stackc(nsamp)
      real ratio(*),angles(*),astrt,aend
      real K
      real *8 Sxx1, Sxx2,Sxx3,Sx1x2,Sx1x3,Sx2x3,Sx1y,Sx2y,Sx3y
      real *8 ax1, ax2, ax3, D
      integer ier
c +===========================+
c | Build the matrix elements |
c | Get the solutions         |
c +===========================+
      do i=1,nsamp
       Sxx1  = 0.
       Sxx2  = 0.
       Sxx3  = 0.
       Sx1x2 = 0.
       Sx1x3 = 0.
       Sx2x3 = 0.
       Sx1y  = 0.
       Sx2y  = 0.
       Sx3y  = 0.
       n = 0
       do j=1,ntr
        indx=(j-1)*nsamp+i
        theta=angles(indx)
        y = data(indx)
        if(theta.ge.astrt.and.theta.le.aend.and.y.ne.0.0)then
         K = -ratio(i)*4.
         sine = sin(theta)
         sine = sine*sine
         tang = tan(theta)
         tang = tang*tang
         ax1 = 0.5*(1.0 + tang)
         ax2 = -K*sine
         ax3 = 0.5*(1.0 - K*sine)
         Sxx1 = Sxx1 + ax1*ax1
         Sxx2 = Sxx2 + ax2*ax2
         Sxx3 = Sxx3 + ax3*ax3
         Sx1x2 = Sx1x2 + ax1*ax2
         Sx1x3 = Sx1x3 + ax1*ax3
         Sx2x3 = Sx2x3 + ax2*ax3
         Sx1y  = Sx1y + ax1*y
         Sx2y  = Sx2y + ax2*y
         Sx3y  = Sx3y + ax3*y
         n = n+1
        endif
       end do
       if(n.ge.4)then
        D = Sxx1*Sxx2*Sxx3-Sxx1*Sx2x3*Sx2x3-Sx1x2*Sx1x2*Sxx3
        D = D + Sx1x2*Sx2x3*Sx1x3
        D = D + Sx1x3*Sx1x2*Sx2x3 - Sx1x3*Sx1x3*Sxx2
        x1 = Sx1y*(Sxx2*Sxx3-Sx2x3*Sx2x3)
        x1 = x1 - Sx1x2*(Sx2y*Sxx3 - Sx3y*Sx2x3)
        x1 = x1 + Sx1x3*(Sx2y*Sx2x3 - Sx3y*Sxx2)
        x2 = Sxx1*(Sx2y*Sxx3 - Sx3y*Sx2x3)
        x2 = x2 - Sx1y*(Sx1x2*Sxx3 - Sx1x3*Sx2x3)
        x2 = x2 + Sx1x3*(Sx1x2*Sx3y - Sx2y*Sx1x3)
        x3 = Sxx1*(Sxx2*Sx3y - Sx2y*Sx2x3)
        x3 = x3 - Sx1x2*(Sx1x2*Sx3y - Sx1x3*Sx2y)
        x3 = x3 + Sx1y*(Sx1x2*Sx2x3 - Sx1x3*Sxx2)
        if(D.ne.0)then
         stacka(i)=x1/D
         stackb(i)=x2/D
         stackc(i)=x3/D
        else
         stacka(i)=0.
         stackb(i)=0.
         stackc(i)=0.
        endif
       else
        stacka(i)=0.
        stackb(i)=0.
        stackc(i)=0.
       endif
      end do
      return
      end
      subroutine rmsint (v,t,n,vi)
C ********************************************************************
C |                                                
C |    Subroutine to compute interval velocity from RMS 
C |    velocity using Dix's equation.                   
C |                                                    
C |    Input:                                         
C |    V   - R*4() - Vector of RMS velocities, in ft/m per sec.
C |    t   - R*4() - Vector of times for the V vector, in sec.
C |    n   - I*4   - Length of the V vector.        
C |                                               
C |    OUTPUT:                                      
C |    vi  - R*4() - Vector of inteval velocities.     
C |                                                  
C ********************************************************************
      DIMENSION V(*),T(*),vi(*)

      vi(1)=v(1)
      DO I=2,N
         T2=T(I)
         T1=T(I-1)
         TDEL=T2-T1
         vi(I)=V(I)*V(I)*T2-V(I-1)*V(I-1)*T1
         vi(I)=vi(I)/TDEL
         if(vi(i).le.0.0)then
           vi(i)=vi(i-1)
         else
          vi(I)=SQRT(vi(I))
         endif
      end do
      RETURN
      END
