C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c     program module asig
c
c**********************************************************************c
c
c asig reads seismic trace data from an input file,
c computes an analytical signal attribute,  and
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      integer     itr ( SZLNHD )
      integer     lhed( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform,  nbyte
      integer     luin , luout, lbytes, nbytes, lbyout,lbyte
      integer     irs,ire,ns,ne, static, recnum, trcnum
#include <f77/pid.h>
      real        tri ( SZLNHD ), q(SZLNHD ), data(SZLNHD)
      character   ntap * 256, otap * 256, name*4
      logical     verbos, query
      integer     argis

c     equivalence ( itr(129), tri (1) )
      equivalence ( itr(  1), lhed(1) )
      data lbytes / 0 /, nbytes / 0 /, name/'ASIG'/

c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif

c-----
c     open printout files
c-----
#include <f77/open.h>

      call gcmdln(ntap,otap,ns,ne,irs,ire,nop,thresh,verbos)
c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)
c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'ASIG: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

      samp = float(nsi) * unitsc
c-----
c     ensure that command line values are compatible with data set
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
c-----
c     modify line header to reflect actual number of traces output
c-----
      call hlhprt (itr, lbytes, name, 4, LERR)
      nrecc=ire - irs+1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr=ne-ns+1
      call savew(itr, 'NumTrc', jtr  , LINHED)
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout                 )
c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
      if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  nop,thresh,ntap,otap)
      end if
c-----
c     BEGIN PROCESSING
c     read trace, compute trace attribute, write to output file
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin,ntrc,itr)
c-----
c     process desired trace records
c-----
      do 1000 jj = irs, ire
            call trcskp(jj,1,ns-1,luin,ntrc,itr)
            do 1001 kk=ns,ne
                  nbytes = 0
                  call rtape ( luin, itr, nbytes)
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)
                  call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        recnum , TRACEHEADER)
                  call saver2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        trcnum , TRACEHEADER)
                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        static , TRACEHEADER)

c*
c - put in dotpr call to check for non-flagged dead tracess
c					- joe m. wade 4/22/96
c*
                  if(static .ne. 30000)then
			call dotpr(tri, 1, tri, 1, xdot, nsamp)
			if (xdot .eq. 0) then
                          call vclr(tri,1,nsamp)
			  call savew2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1  			30000, TRACEHEADER)
			else
                          call asig3(tri,q,samp,nsamp,nop,data,thresh)
                          call vmov(data,1,tri,1,nsamp)
			endif
                  else
                        call vclr(tri,1,nsamp)
                  endif
                  call vmov (tri, 1, lhed(ITHWP1), 1, nsamp)
                  call wrtape( luout, itr, nbytes)
            if(verbos .and. kk.eq.ns) write(LERR,*)'ri ',recnum
 1001             continue
            call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
 1000       continue
c-----
c     close data files
c-----
  999 continue
      call lbclos ( luin )
      call lbclos ( luout )
            write(LERR,*)'end of asig, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
            write(LER ,*)'end of asig, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end

      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
      write(LER,*)
     :'execute asig by typing asig and a list of program parameters.'
      write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
      write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)
     :' -N [ntap]    (no default)      : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)      : output data file name'
       write(LER,*)
     :' -ns[ns]      (default = first)         : start trace number'
       write(LER,*)
     :' -ne[ne]      (default = last)         : end trace number'
       write(LER,*)
     :' -rs[irs]      (default = first)         : start record number'
       write(LER,*)
     :' -ne[ire]      (default = last)         : end record number'
        write(LER,*)
     :' -nop[nop]  (default = 1    ) : specific trace attribute'
		write(LER,*) ' NOP = 1 CARRIER'
		write(LER,*) ' NOP = 2 QUADRATURE'
		write(LER,*) ' NOP = 3 ENVELOPE'
		write(LER,*) ' NOP = 4 INST. PHASE'
		write(LER,*) ' NOP = 5 RESPONSE PHASE'
		write(LER,*) ' NOP = 6 INST. FREQUENCY'
		write(LER,*) ' NOP = 7 RESPONSE FREQUENCY'
		write(LER,*) ' NOP = 8 0-PHASE DECOMPOSITION'
		write(LER,*) ' NOP = 9 90-PHASE DECOMPOSITION'
		write(LER,*) ' NOP = 10 RESPONSE AMPLITUDE'
		write(LER,*) ' NOP = 11 RESPONSE LENGTH'
		write(LER,*) ' NOP = 12 ENVELOPE SKEWNESS'
		write(LER,*) ' NOP = 13 ENVELOPE RISE TIME'
		write(LER,*) ' NOP = 14 INSTANTANEOUS BANDWIDTH'
		write(LER,*) ' NOP = 15 Apparent polarity'
	write(LER,*)
     :' -t[thresh]  (default = 0.15) : % of max env peak to pick'
         write(LER,*)
     :'usage:   asig -N[ntap] -O[otap] -nop[nop] -t[thresh]',
     :' -ns[ns] -ne[ne] -rs[irs] -re[ire] -V'
         write(LER,*)
     :'***************************************************************'
      return
      end

      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,nop,thresh,verbos)
c-----
c     get command arguments
c
c     ntap  - c*120     input file name
c     otap  - c*120     output file name
c     ns    - i*4 starting trace index
c     ne    - i*4 ending trace index
c     irs   - i*4 starting record index
c     ire   - i*4 ending record index
c     nop   - i*4 option for trace attribute
c     verbos      - l   verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      integer     ns, ne, irs, ire, nop  
      logical    verbos
      integer    argis

            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argi4 ( '-nop', nop , 1, 1 )
	    call argr4 ( '-t',thresh, 0.15, 0.15)
            verbos = ( argis( '-V' ) .gt. 0 )

      return
      end

      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1            nop,thresh,ntap,otap)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4 number of samples in trace
c     nsi   - I*4 sample interval in ms
c     ntrc  - I*4 traces per record
c     nrec  - I*4 number of records per line
c     iform - I*4 format of data
c     nop   - I*4 option for trace attribute
c     thresh- r*4 threshold to pick peaks
c     ntap  - C*120     input file name
c     otap  - C*120     output file name
c-----
#include <f77/iounit.h>
      integer     nsamp, nsi, ntrc, nrec, iform
      character   ntap*(*), otap*(*)

            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' nop (tr attribute) =  ', nop
		write(LERR,*) ' NOP = DESIRED ATTRIBUTE'
		write(LERR,*) ' NOP = 1 CARRIER'
		write(LERR,*) ' NOP = 2 QUADRATURE'
		write(LERR,*) ' NOP = 3 ENVELOPE'
		write(LERR,*) ' NOP = 4 INST. PHASE'
		write(LERR,*) ' NOP = 5 RESPONSE PHASE'
		write(LERR,*) ' NOP = 6 INST. FREQUENCY'
		write(LERR,*) ' NOP = 7 RESPONSE FREQUENCY'
		write(LERR,*) ' NOP = 8 0-PHASE DECOMPOSITION'
		write(LERR,*) ' NOP = 9 90-PHASE DECOMPOSITION'
		write(LERR,*) ' NOP = 10 RESPONSE AMPLITUDE'
		write(LERR,*) ' NOP = 11 RESPONSE LENGTH'
		write(LERR,*) ' NOP = 12 ENVELOPE SKEWNESS'
		write(LERR,*) ' NOP = 13 ENVELOPE RISE TIME'
		write(LERR,*) ' NOP = 14 Instantaneous Bandwidth'
		write(LERR,*) ' NOP = 15 Apparent polarity'
	    write(LERR,*) ' input threshold     =  ', thresh
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' '
            write(LERR,*)' '

      return
      end

       SUBROUTINE ASIG3(R, Q, SAMP, NSAMP, NOP, DATA, thresh)

#include <f77/lhdrsz.h>

C
C John Bodine originally authored "ASIG" in July of 1981. This is
C a rewrite of Johns work to remove errors, and to speed up the
C code by removing some unnecessary calculations. 
C (Dennis Frampton, March 11th, 1992).
C Carrier, threshold, and env picker added by Don Wagner, April 29, 1992)
C
C  RETURNS DESIRED COMPLEX TRACE ATTRIBUTE
C        R = INPUT SEISMIC TRACE
C     SAMP = SAMPLE INTERVAL
C    NSAMP = NO. SAMPLES PER TRACE
C      NOP = DESIRED ATTRIBUTE
C          = 1 Carrier
C          = 2 QUADRATURE
C          = 3 ENVELOPE
C          = 4 INST. PHASE
C          = 5 RESPONSE PHASE
C          = 6 INST. FREQUENCY
C          = 7 RESPONSE FREQUENCY
C          = 8 0-PHASE DECOMPOSITION
C          = 9 90-PHASE DECOMPOSITION
C          = 10 RESPONSE AMPLITUDE
C          = 11 RESPONSE LENGTH
C          = 12 ENVELOPE SKEWNESS
C          = 13 ENVELOPE RISE TIME
C          = 14 Instantaneous Bandwidth
C          = 15 Apparent polarity
C
C     DATA = OUTPUT ATTRIBUTE
C
      REAL    R(*),Q(*),DUM(SZLNHD), DATA(*), VP(SZLNHD), SAMP, PI
      REAL    F(SZLNHD),A(SZLNHD),FREQ(SZLNHD),PH(SZLNHD), RADDEG, WC
      INTEGER IP(SZLNHD)
      INTEGER NSAMP, NOP, NF, NZ
C
      PI     = 3.14159265
      RADDEG = 360./(2.*PI)
      NF     = 101
      WC     = 3
      NZ     = NF + NSAMP

      call vclr (data, 1, nsamp)
C
C Calculate Quadrature 
C
      CALL HILBRT(NZ, NF, WC, DUM, SAMP, NSAMP, R, Q)
C
C Carrier
C
      IF (NOP .EQ. 1) THEN
         DO 10 I = 1, NSAMP
              A(I) = 0.0
              DATA(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
                   DATA(I) = R(I)/A(I)
              ENDIF
   10    CONTINUE
         GO TO 9999
      ENDIF
C
C Quadrature.
C
      IF (NOP .EQ. 2) THEN
         DO 20 I = 1, NSAMP
              DATA(I) = Q(I)
   20    CONTINUE
         GO TO 9999
      ENDIF
C
C Envelope.
C
      IF (NOP .EQ. 3) THEN
         DO 30 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
   30    CONTINUE
         GO TO 9999
      ENDIF
C
C Inst. Phase.
C
      IF (NOP .EQ. 4) THEN
         DO 40 I = 1, NSAMP
              DATA(I) = PHAS(R(I), Q(I)) * RADDEG
              IF (DATA(I) .GT. 180.0) DATA(I) = DATA(I) - 360.0
   40    CONTINUE
         GO TO 9999
      ENDIF
C
C Response Phase.
C
      IF (NOP .EQ. 5) THEN
         DO 50 I = 1, NSAMP
              PH(I) = PHAS(R(I), Q(I)) * RADDEG
              IF (PH(I) .GT. 180.0) THEN
                   PH(I) = PH(I) - 360.0
              ENDIF
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
   50    CONTINUE
         PH(1) = PH(2)
         CALL RPHS(PH, DATA, NSAMP)
         GO TO 9999
      ENDIF
C
C Inst. Frequency.
C
      IF (NOP .EQ. 6) THEN
         DO 60 I = 1, NSAMP
              FREQ(I) = XFREQ(R, Q, SAMP, NSAMP, I) / (2.0 * PI)
              DATA(I) = -1.0 * FREQ(I)
   60    CONTINUE
         GO TO 9999
      ENDIF
C
C Response Frequency.
C
      IF (NOP .EQ. 7) THEN
         DO 70 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
              FREQ(I) = XFREQ(R, Q, SAMP, NSAMP, I) / (2.0 * PI)
              FREQ(I) = -1.0 * FREQ(I)
   70    CONTINUE
         call findpk1(freq,data,nsamp,f   ,thresh)
385      call vmov(f   ,1,data,1,nsamp)
c        CALL PKDT(FREQ, DATA, NSAMP, thresh)
         GO TO 9999
      ENDIF
C
C Calculate 0 or 90(which ever asked for) Phase Decomposed Signal.
C
      IF (NOP .EQ. 8 .OR. NOP .EQ. 9) THEN
         DO 80 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
              PH(I) = PHAS(R(I), Q(I)) * RADDEG
              IF (PH(I) .GT. 180.0) THEN
                   PH(I) = PH(I) - 360.0
              ENDIF
   80    CONTINUE
         PH(1) = PH(2)
         CALL RPHS(PH, DATA, NSAMP)
         CALL QCON(R, Q, DATA, NSAMP)
         IF (NOP .EQ. 8) THEN
              DO 90 I = 1, NSAMP
                   DATA(I) = R(I)
   90         CONTINUE
         ENDIF
         IF (NOP .EQ. 9) THEN
              DO 100 I = 1, NSAMP
                   DATA(I) = Q(I)
  100         CONTINUE
         ENDIF
         GO TO 9999
      ENDIF
C
C Calculate Response Amplitude.
C
      IF (NOP .EQ. 10) THEN
         DO 110 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
  110    CONTINUE
c        call findpk1(a   ,data,nsamp,f   ,thresh)
c        call vmov(f   ,1,data,1,nsamp)
         CALL PKDT(A, DATA, NSAMP, thresh)
         GO TO 9999
      ENDIF
C
C Calculate length of each envelope lobe.
C
      IF (NOP .EQ. 11) THEN
         DO 120 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
  120    CONTINUE
         CALL ALEN(DATA, NSAMP)
         GO TO 9999
      ENDIF
C
C Calculate skewness of envelope lobe.
C
      IF (NOP .EQ. 12) THEN
         DO 130 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
  130    CONTINUE
         CALL ASKW(DATA, NSAMP)
         GO TO 9999
      ENDIF
C
C Calculate max slope of envelope lobe.
C
      IF (NOP .EQ. 13) THEN
         DO 140 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
              DATA(I) = A(I)
  140    CONTINUE
         CALL ASLP(DATA, NSAMP)
         GO TO 9999
      ENDIF
C
C Calculate Instantaneous Bandwidth(after Cohen&Lee,Proc. IEEE
c		Int Conf. Acoust. Speech, Signal Processing,2451)
C
      IF (NOP .EQ. 14) THEN
         DO 150 I = 2, NSAMP - 1
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) 
		   rslope = ( r(i+1) - r(i-1) ) / (2.0 * samp)
		   qslope = ( q(i+1) - q(i-1) ) / (2.0 * samp)
		   result = abs(r(i)*rslope + q(i)*qslope)/( a(i)*2.*PI)
		   a(i)   = result
              ENDIF
              DATA(I) = A(I)
  150    CONTINUE
	 Data(1) = data(2)
	 Data(nsamp) = data(nsamp-1)
         GO TO 9999
      ENDIF
C
C Apparent polarity
C
      IF (NOP .EQ. 15) THEN
         DO 160 I = 1, NSAMP
              A(I) = 0.0
              IF (R(I) .NE. 0.0 .AND. Q(I) .NE. 0.0) THEN
                   A(I) = (R(I)**2 + Q(I)**2) ** .5
              ENDIF
  160     CONTINUE
          call maxmgv ( A, 1, amax, loca, nsamp )
          afloor = thresh * amax
          do  i = 1, nsamp
              amp = A (i)
              if ( abs(amp) .lt. afloor ) amp = 0.0
              A (i) = amp
          enddo
          call peak (A, nsamp, 0, vp, ip, np)
          do  i = 1, np
              data ( ip(i) ) = sign ( 1.0, R(ip(i)) )
          enddo
             
         GO TO 9999
      ENDIF

C
C
 9999 CONTINUE
C
      RETURN
      END

      SUBROUTINE HILBRT(NZ,NF,WC,F,DX,NX,X,Z)
C
C  PROGRAM TO PERFORM A HILBERT TRANSFORM ON A VECTOR X
C  OF LENGTH NX.
C
C  Z SERVES FIRST AS A DUMMY CONVOLUTION ARRAY
C  AND MUST BE DIMENSIONED NZ=NX+NF-1
C
C  TRANSFORM IS RETURNED AS Z OF LENGTH NX
C
C  NOTE: HILF OPERATOR RESULTS IN COS --> -SIN.....
C  BY JH BODINE   7/27/81
C
C
C      F = DUMMY ARRAY FOR THE HILBERT OPERATOR
C     NF = NO.POINTS IN HILBERT TRANSFER FUNCTION (ODD)
C     DX = SAMPLE INTERVAL
C   NX,X = NO. POINTS IN VECTOR X WHICH IS TO BE TRANSFORMED
C
      DIMENSION X(1),F(2000),Z(1)
C     DIMENSION X(1),F(NF),Z(NZ)
C
      CALL HILF(DX,WC,NF,F)
C
      CALL FOLD(NF,F,NX,X,NZ,Z)
C
      DO 10 I=1,NX
C     WRITE(*,*) Z(I),I
   10 Z(I)= Z((NF-1)/2+I)*DX
C
      RETURN
      END
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       RPHS                                                 *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  TO CALCULATE RESPONSE PHASE.                              *
C  ENTRY POINTS:                                                       *
C      RPHS  (PHASE,A,NPT)
C  ARGUMENTS:                                                          *
C      PHASE   REAL         U*  (1) - WORKING ARRAY CONTAINS INST PHAS *
C      A       REAL         U*  (1) - COMPLEX ENVELOPE ARRAY ON INPUT  *
C                                   - RESPONSE PHASE ON OUTPUT         *
C      NPT     INTEGER      I*      - NUMBER OF SAMPLE POINTS          *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   J. BODINE                          ORIGIN DATE: 81/09/02  *
C  LANGUAGE: FORTRAN 77/IV               DATE LAST COMPILED: 83/06/03  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      MNMX         - TO FIND RELATIVE MAXIMA/MINIMA IN AMPLITUDE ARRAY*
C      UNWRAP       - TO UNWRAP PHASE                                  *
C      FPEAK   REAL - TO FIND MAXIMUM VALUE ARRAY POSITION             *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      FLOAT   REAL -                                                  *
C  FILES:                                                              *
C      6  ( OUTPUT SEQUENTIAL ) - DEBUG OUTPUT                         *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  NONE                                               *
C  GENERAL DESCRIPTION:                                                *
C       ROUTINE TO SET ANALYTIC SIGNAL ENVELOPE LOBES TO VALUE OF      *
C       THE INSTANTANEOUS PHASE AT THE POINT WHERE THE ENVELOPE        *
C       IS A MAXIMUM.                                                  *
C       THE VALUE OF THE ENVELOPE IS SET TO THAT OF THE INSTANTANEOUS P*
C       FROM TROUGH TO TROUGH. THE RESULTING "ENVELOPE" MAY BE         *
C       CONSIDERED TO BE A MEASURE OF THE PHASE OF AN ASSUMED NON-DISPE*
C       SEISMIC RESPONSE CONTAINED WITHIN THE ENVELOPE.                *
C       SEE TANER ET AL., 1979, GEOPHYSICS, V.44, #6,P.1041.           *
C                                                                      *
C       PROGRAM ASSUMES THE FIRST POINT IN THE SIGNAL IS A TROUGH      *
C       AND THE FIRST POINT OF ZERO SLOPE IS A PEAK. IF THIS POINT     *
C       IS A TROUGH IT WILL BE SKIPPED IN MNMX.                        *
C                                                                      *
C       NOTE FOR USERS:                                                *
C     THIS ROUTINE WILL FIND EVERY ENVELOPE PEAK REGARDLESS            *
C     OF RELATIVE AMPLITUDE. DISPLAY OF RESPONSE PHASE OF NOISY        *
C     DATA IS THUS MOST EFFECTIVE WITH A COLOR PLOT WHERE COLOR        *
C     INTENSITY IS AMPLITUDE MODULATED AND A COLOR WHEEL IS USED       *
C     FOR PHASE (0-360 DEGREES).                                       *
C                                                                      *
C  REVISED BY:  J. NORRIS                     REVISION DATE: 83/06/01  *
C       MODIFY MIN/MAX LOCATOR LOGIC TO BE SUBROUTINE CALL.            *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NON-STANDARD FEATURES:  NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      SUBROUTINE RPHS (PHASE,A,NPT)
      DIMENSION A(1), PHASE(1), ITRF(2000), IPEAK(2000), P(3)
C***********************************************************************
C       FIND ARRAY POSITIONS OF MAXIMA AND MINIMA. THEN
C       SET IPEAK VALUES TO INTEGER VALUE OF INSTANTANEOUS
C       PHASE AT IPEAK(I) + FRACTION ARRAY POSITIONS.
C***********************************************************************
      CALL MNMX (A,NPT,IPEAK,JP,ITRF,JT)
C
      DO 100 I=1,JP
         K = IPEAK(I)
         DO 50 M=1,3
         P(M) = PHASE(K-2+M)
         IF( ABS(P(M)) .GT. 1.0E20 ) P(M)=0.0
	 if (abs(p(m)) .lt. 1.e-10 ) then
		p(m) = 0.0
		go to 50
	 endif
         IF( 1/ABS(P(M)) .GT. 1.0E20 ) P(M)=0.0
C  50    P(M) = PHASE(K-2+M)
CWC50    WRITE(*,*)M,P(M)
   50    CONTINUE
         CALL UNWRAP(P)
         FP = FPEAK(A,K)
         TK = FLOAT(K)
         IF(FP.LT.TK)IPEAK(I) = P(2)+(TK-FP)*(P(1)-P(2))
         IF(FP.GE.TK)IPEAK(I) = P(2)+(TK-FP)*(P(2)-P(3))
         IF(IPEAK(I).GT.180)IPEAK(I) = IPEAK(I)-360
C        WRITE (6,*) K,IPEAK(I),ITRF(I),PHASE(K)
  100 CONTINUE
      if (ITRF(JP) .ne. NPT) ITRF(JP) = NPT
C***********************************************************************
C       SET ENVELOPE LOBES TO IPEAK VALUES
C***********************************************************************
      J = 1
      ISTRT = 1
  150 IEND = ITRF(J)
      PEAK = FLOAT(IPEAK(J))
      DO 200 I=ISTRT,IEND
		a(i) = peak
  200 continue
C
      J = J+1
      IF (IEND.EQ.NPT) GO TO 250
      ISTRT = IEND
      GO TO 150
C
  250 CONTINUE
C
      RETURN
C
      END
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       FPEAK                                                *
C  ROUTINE TYPE:  FUNCTION  REAL                                       *
C  PURPOSE: ROUTINE TO FIND INFERRED FRACTIONAL ARRAY POSITION         *
C         (TK +/- FRACTION) OF A PEAK OF AN ARRAY.                     *
C  ENTRY POINTS:                                                       *
C      FPEAK  REAL  (A,K)                                              *
C  ARGUMENTS:                                                          *
C      A       REAL         I*  (1) - INPUT ARRAY OF VALUES            *
C      K       INTEGER      O*      - INDEX OF LOCAL MAXIMUM IN A      *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   J.H. BODINE                        ORIGIN DATE: 82/10/17  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 83/06/03  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      FLOAT   REAL    -                                               *
C      ABS     GENERIC -                                               *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  NONE                                               *
C  GENERAL DESCRIPTION:  LATERAL SYMMETRY ABOUT PEAK IS ASSUMED.       *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NON-STANDARD FEATURES:  NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      FUNCTION FPEAK (A,K)
      DIMENSION A(1)
C
      TK = FLOAT(K)
      DAL = ABS(A(K)-A(K-1))
      DAH = ABS(A(K)-A(K+1))
C
      TEST = DAL-DAH
      IF(TEST.GT.0.)FPEAK = TK+(1.-DAH/DAL)*.5
      IF(TEST.LT.0.)FPEAK = TK-(1.-DAL/DAH)*.5
      IF(TEST.EQ.0.)FPEAK = TK
C
      RETURN
      END
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       UNWRAP                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C      ROUTINE TESTS FOR CONDITIONS REQUIRING UNWRAPPING OF PHASE      *
C      TO PERMIT INTERPOLATION ACROSS DISCONTINUITIES AT +/-180 DEG.   *
C  ENTRY POINTS:                                                       *
C      UNWRAP  (P)                                                     *
C  ARGUMENTS:                                                          *
C      P       REAL      U*  (1) - VALUES TO UNWRAP                    *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   J.H. BODINE                        ORIGIN DATE: 82/10/17  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 83/06/03  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      ABS     GENERIC -                                               *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  NONE                                               *
C  GENERAL DESCRIPTION:  NONE                                          *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NON-STANDARD FEATURES:  NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      SUBROUTINE UNWRAP(P)
      DIMENSION P(3)
C
      DIS = 0.
C
      DIFL = ABS(P(1)-P(2))
      DIFH = ABS(P(2)-P(3))
C
      IF (DIFL.LT.180..AND.DIFH.LT.180.) RETURN
C
      DO 50 I=1,3
   50 IF(P(I).LT.0.)P(I) = P(I)+360.
C
      RETURN
      END
       SUBROUTINE QCON(R,Q,A,N)
C
C  JH BODINE  10/19/82
C
C  ROUTINE TO OBTAIN COMPLIMENTARY FUNCTIONS FROM REAL
C  SEISMIC TRACE USING THE RESPONSE PHASE (DESCRIBED IN PROGRAM
C  CSIG). ONE FUNCTION IS ASSOCIATED WITH ZERO PHASE AND THE OTHER
C  WITH PI/2 PHASE.
C
C      R = REAL SEISMIC TRACE INPUT
C          RETURNED AS ZERO PHASE COMPONENT
C      Q = QUADRATURE FOR ANALYTIC SIGNAL AS INPUT
C          RETURNED AS PI/2 PHASE COMPONENT
C      A = RESPONSE PHASE ARRAY
C
C      N = INTEGER LENGTH OF ALL ARRAYS
C
C  DERIVATION OF MATH BY KEN HANSON.
C
C
       DIMENSION R(1),Q(1),A(1)
C
       PI = 3.14159265
       DEGRAD = PI/180.
C
       DO 10 I=1,N
       ANG = A(I) * DEGRAD
       RP = (COS(ANG)**2*R(I)+COS(ANG)*SIN(ANG)*Q(I))
       Q(I) = R(I) - RP
   10  R(I) = RP
       RETURN
       END
      SUBROUTINE ALEN(A,NPT)
C
C  JH BODINE 9/2/81
C
C  ROUTINE TO SET ANALYTIC SIGNAL ENVELOPE LOBES TO VALUE OF
C  THE TIME WIDTH OF THE ENVELOPE LOBES.
C  THE VALUE OF THE ENVELOPE IS SET FROM TROUGH
C  TO TROUGH. THE RESULTING "ENVELOPE" MAY BE CONSIDERED TO
C  REPRESENT A MODIFICATION OF
C     A = COMPLEX ENVELOPE
C   NPT = NUMBER OF SAMPLE POINTS
C
C  PROGRAM ASSUMES THE FIRST POINT IN THE SIGNAL IS A TROUGH
C  AND THE FIRST MAXIMA IS A PEAK. IF THE FIRST MAXIMA IS A TROUGH,
C  IT WILL BE SKIPPED (STATEMENT #50).
C
      DIMENSION A(1),ITRF(1000)
C
      JT = 1
      JP = 1
      SLOP = 0.
C
C  FIND ARRAY POSITIONS OF MAXIMA AND MINIMA
C
      NPT1 = NPT - 1
      DO 100 I = 2,NPT1
      SLAST = SLOP
      SLOP = A(I+1) - A(I)
      CHNG = SLAST * SLOP
      IF(CHNG.GE.0.) GO TO 100
      IF(CHNG.LT.0. .AND. SLAST.LT.0.) GO TO 50
      JP = JP+1
      GO TO 100
C
   50 IF(JP.EQ.1) GO TO 100
      ITRF(JT) = I
      JT = JT+1
C
  100 CONTINUE
      ITRF(JT)=NPT
      JP = JP-1
      IF(JP.EQ.0) JP=1
C
C  SET ENVELOPE LOBES TO VALUE OF ENVELOPE LOBE WIDTH
C
      J = 1
      ISTRT = 1
  250 IEND = ITRF(J)
      VALUE = IEND - ISTRT
      DO 300 I = ISTRT,IEND
  300 A(I) = VALUE
C
      J = J+1
      IF(IEND.EQ.NPT) GO TO 350
      ISTRT = IEND
      GO TO 250
C
  350 CONTINUE
C
      RETURN
      END
      SUBROUTINE ASLP(A,NPT)
C
C  JH BODINE 9/2/81
C
C  ROUTINE TO SET ANALYTIC SIGNAL ENVELOPE LOBES TO A VALUE
C  PROPORTIONAL TO THE MAX SLOPE OF THE ENVELOPE LOBES.
C
C    SLOPE = MAX(A(I+1)-A(I))/MAX(A(I)) * 1000.
C
C  THE VALUE OF THE ENVELOPE IS SET FROM TROUGH
C  TO TROUGH. THE RESULTING "ENVELOPE" MAY BE CONSIDERED TO
C  REPRESENT A MODIFICATION OF
C     A = COMPLEX ENVELOPE
C   NPT = NUMBER OF SAMPLE POINTS
C
C  PROGRAM ASSUMES THE FIRST POINT IN THE SIGNAL IS A TROUGH
C  AND THE FIRST MAXIMA IS A PEAK. IF THE FIRST MAXIMA IS A TROUGH,
C  IT WILL BE SKIPPED (STATEMENT #50).
C
      DIMENSION A(1),ITRF(1000),SLOPMX(1000),ENVMAX(1000)
C
      JT = 1
      JP = 1
      SLOP = 0.
C
C  FIND ARRAY POSITIONS OF MAXIMA AND MINIMA
C
      NPT1 = NPT - 1
      SMAX = 0.
      AMAX = 0.
      DO 100 I = 2,NPT1
      SLAST = SLOP
      SLOP = A(I+1) - A(I)
      CHNG = SLAST * SLOP
      IF(ABS(SLOP).GT.ABS(SMAX)) SMAX = SLOP
      IF(ABS(A(I)).GT.ABS(AMAX)) AMAX = A(I)
      IF(CHNG.GE.0.) GO TO 100
      IF(CHNG.LT.0. .AND. SLAST.LT.0.) GO TO 50
      JP = JP+1
      GO TO 100
C
   50 IF(JP.EQ.1) GO TO 100
      ENVMAX(JT) = AMAX
      SLOPMX(JT) = SMAX
      SMAX = 0.
      ITRF(JT) = I
      JT = JT+1
C
  100 CONTINUE
      ITRF(JT)=NPT
      JP = JP-1
      IF(JP.EQ.0) JP=1
C
C  SET ENVELOPE LOBES TO VALUE OF ENVELOPE LOBE WIDTH
C
      J = 1
      ISTRT = 1
  250 IEND = ITRF(J)
      DO 300 I = ISTRT,IEND
      IF (ENVMAX(J).EQ.0.) ENVMAX(J) = 1.
  300 A(I) = SLOPMX(J)/ENVMAX(J)*1000.
C
      J = J+1
      IF(IEND.EQ.NPT) GO TO 350
      ISTRT = IEND
      GO TO 250
C
  350 CONTINUE
C
      RETURN
      END
      SUBROUTINE ASKW(A,NPT)
C
C  JH BODINE 9/2/81
C
C  ROUTINE TO CALCULATE THE ASYMMETRY OF THE ENVELOPE
C
C  ROUTINE TO SET ANALYTIC SIGNAL ENVELOPE LOBES TO VALUE OF
C  THE DIFFERENCE BETWEEN THE ARRAY POSITIONS OF THE ENVELOPE LOBE
C  PEAK AND THE FIRST MOMENT OF THE LOBE.
C  THE VALUE OF THE ENVELOPE IS SET TO THAT OF THIS DIFFERENCE * 100
C  FROM TROUGH TO TROUGH. THE RESULTING "ENVELOPE" MAY BE
C  CONSIDERED TO BE A MEASURE OF THE DEGREE OF ASYMMETRY OF EACH
C  ENVELOPE LOBE.
C
C  SEE TANER ET AL., 1979, GEOPHYSICS, V.44, #6,P.1041.
C     A = COMPLEX ENVELOPE
C   NPT = NUMBER OF SAMPLE POINTS
C
C  PROGRAM ASSUMES THE FIRST POINT IN THE SIGNAL IS A TROUGH
C  AND THE FIRST POINT OF ZERO SLOPE IS A PEAK. IF THIS POINT
C  IS A TROUGH IT WILL BE SKIPPED (STATEMENT #50).
C
C  NOTE FOR USER'S;
C     THIS ROUTINE WILL FIND EVERY ENVELOPE PEAK REGARDLESS
C     OF RELATIVE AMPLITUDE. DISPLAY OF RESULTS FOR NOISY
C     DATA IS THUS MOST EFFECTIVE WITH A COLOR PLOT WHERE COLOR
C     INTENSITY IS AMPLITUDE MODULATED.
C
C
      DIMENSION A(1),ITRF(1000),IPEAK(1000)
C
      JT = 1
      JP = 1
      SLOP = 0.
C
C  FIND ARRAY POSITIONS OF MAXIMA AND MINIMA
C
      NPT1 = NPT - 1
      DO 100 I = 2,NPT1
      SLAST = SLOP
      SLOP = A(I+1) - A(I)
      CHNG = SLAST * SLOP
      IF(CHNG.GE.0.) GO TO 100
      IF(CHNG.LT.0. .AND. SLAST.LT.0.) GO TO 50
      IPEAK(JP) = I
      JP = JP+1
      GO TO 100
C
   50 IF(JP.EQ.1) GO TO 100
      ITRF(JT) = I
      JT = JT+1
C
  100 CONTINUE
      ITRF(JT)=NPT
      JP = JP-1
      IF(JP.EQ.0) JP=1
C
C  SET ENVELOPE LOBES TO DIFFERENCE IN ARRAY POSITION OF ENVELOPE
C  PEAK AND FIRST MOMENT
C
      J = 1
      ISTRT = 1
  250 IEND = ITRF(J)
      SUMM = 0.
      ASUM = 0.
C  GET FRACTIONAL ARRAY POSITION OF ENVELOPE PEAK
      K = IPEAK(J)
      FP = FPEAK(A,K)
C  GET POSITION OF FIRST MOMENT
      DO 260 II = ISTRT,IEND
      SUMM = SUMM + (FP-II)*A(II)
      ASUM = ASUM + A(II)
  260 CONTINUE
      IF(ASUM.EQ.0.) ASUM = 1.
      AMOM = SUMM/ASUM
      IF(ASUM.EQ.0.) AMOM = 0.
C  SET ENVELOPE ARRAY TO DIFFERENCE OF PEAK AND 1ST MOM POSITIONS
      DO 300 I = ISTRT,IEND
  300 A(I) = AMOM *100.
C
      J = J+1
      IF(IEND.EQ.NPT) GO TO 350
      ISTRT = IEND
      GO TO 250
C
  350 CONTINUE
C
      RETURN
      END
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       PKDT                                                 *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  TO SET ANALYTIC SIGNAL ENVELOPE LOBES TO VALUE OF         *
C            THE INPUT DATA AT THE POINT WHERE THE ENVELOPE            *
C            IS A MAXIMUM. VALUE OF ENVELOPE SET FROM TROUGH TO TROUGH *
C  ENTRY POINTS:                                                       *
C      PKDT  (DATA,A,NPT,scale)
C  ARGUMENTS:                                                          *
C      DATA    REAL         I*  (1) - INSTANTANEOUS ATTRIBUTE ARRAY    *
C      A       REAL         U*  (1) - COMPLEX ENVELOPE ON INPUT        *
C                                   - RESPONSE ATTRIBUTE OUTPUT        *
C      NPT     INTEGER      I*      - NUMBER OF SAMPLES IN 'DATA'&'A'  *
C      scale   REAL         I*  (1) - % env to keep
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   J.H. BODINE                        ORIGIN DATE: 81/09/02  *
C  LANGUAGE: FORTRAN 77/IV               DATE LAST COMPILED: 83/06/03  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      MNMX - TO FIND RELATIVE MINIMA/RELATIVE MAXIMA OF AMPLITUDE FCN *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:                                                              *
C      6  ( OUTPUT SEQUENTIAL ) - DEBUG FILE                           *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  NONE                                               *
C  GENERAL DESCRIPTION:  PROGRAM ASSUMES FIRST POINT IN SIGNAL IS TROUG*
C      AND THE FIRST MAXIMA IS A PEAK. IF THE FIRST MAXIMA IS A TROUGH,*
C      IT WILL BE SKIPPED IN MNMX.                                     *
C  REVISED BY:  J. NORRIS (PKDTA)             REVISION DATE: 83/06/02  *
C       ADD CALL TO MNMX TO SCAN FOR RELATIVE MINIMA/MAXIMA.           *
C  REVISED BY:  Don Wagner 3/39/92
C       ADD scale factor to restrict envelop peaks
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NON-STANDARD FEATURES:  NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      SUBROUTINE PKDT(DATA,A,NPT,scale )
      DIMENSION A(1),DATA(1),ITRF(1000),IPEAK(1000),PEAK(1000)
C***********************************************************************
C       FIND ARRAY POSITIONS OF MAXIMA AND MINIMA. THEN SET PEAK VALUES
C        OF INPUT DATA AT DATA AT IPEAK(I) ARRAY POSITIONS
C***********************************************************************
      CALL MNMX(A,NPT,IPEAK,JP,ITRF,JT)

	call maxmgv (a,1,amax,index,npt)
	thresh = scale*amax
C
      DO 200 I=1,JP
         K = IPEAK(I)
         PEAK(I) = DATA(K)
  200 CONTINUE
C***********************************************************************
C  SET ENVELOPE LOBES TO IPEAK VALUES, if IPEAK > thresh
C***********************************************************************
      J = 1
      ISTRT = 1
  250 IEND = ITRF(J)
      DO 300 I = ISTRT,IEND
      if (abs(a(ipeak(j))) .gt. thresh) then
	a(i) = peak(j)
      else
	a(i) = 0.0
      endif
  300 continue
C
      J = J+1
      IF(IEND.EQ.NPT) GO TO 350
      ISTRT = IEND
      GO TO 250
C
  350 CONTINUE
C
      RETURN
      END
      FUNCTION XFREQ(R,Q,DX,NPT,M)
C
C  JH BODINE  7/28/81
C
C  CALCULATES INSTANTANEOUS FREQUENCY FOR ANALYTIC
C  TRACE ANALYSIS USING FINITE DIFFERENCE APPROXIMATIONS
C  (SEE TANER ET AL., 1979, GEOPHYSICS, V44 NO.6, P1041.).
C
C      R, Q = REAL AND QUADRATURE TRACES
C       NPT = NUMBER OF POINTS IN THE TRACE ARRAYS
C        DX = SAMPLE INTERVAL
C         M = CURRENT CENTRAL DIFFERENCE POSITION
C
      DIMENSION R(2),Q(2)
      PI = 3.14159265
C
      XFREQ = 0.
      IF(R(M).EQ.0. .AND. Q(M).EQ.0.) GO TO 10
C
      RLST = R(2)
      QLST = Q(2)
      RMST = R(NPT-1)
      QMST = Q(NPT-1)
C
      IF(M.LT.NPT) RMST = R(M+1)
      IF(M.LT.NPT) QMST = Q(M+1)
      IF(M.GT.1) RLST = R(M-1)
      IF(M.GT.1) QLST = Q(M-1)
C
      XFREQ = (R(M)*(QMST-QLST)/(2.*DX)
     1 - Q(M)*(RMST-RLST)/(2.*DX))
     2 /(R(M)**2 + Q(M)**2)
C
   10 RETURN
      END
      FUNCTION PHAS(FR,FI)
C
C  JH BODINE    7/28/81
C
C  PROGRAM CALCULATES PHASE IN RADIANS ZERO TO 2*PI
C  FROM ENTERED REAL AND IMAGINARY COMPONENTS.
C
      PHAS = 0.
      PI=3.14159265
      A = (FR**2 + FI**2)**.5
      IF(A.EQ.0.) GO TO 10
      ARG = ABS(FI)/A
      IF(ARG.GT.1.) ARG=1.
C
C  0 TO PI/2
      IF(FI.GT.0. .AND. FR.GE.0.) PHAS=ASIN(ARG)
C
C  PI/2 TO PI
      IF(FI.GE.0. .AND. FR.LT.0.) PHAS=PI - ASIN(ARG)
C
C  PI TO 3*PI/2
      IF(FI.LT.0. .AND. FR.LE.0.) PHAS=PI + ASIN(ARG)
C
C  3*PI/2 TO 2*PI
      IF(FI.LT.0. .AND. FR.GT.0.) PHAS=2*PI - ASIN(ARG)
C
   10 RETURN
      END
C
      SUBROUTINE ZERO(ARRAY,NSAMP)
      REAL ARRAY(NSAMP)
      DO 10 I=1,NSAMP
          ARRAY(I) = 0.0
   10 CONTINUE
      RETURN
      END
C
C
      SUBROUTINE HILF(TDEL,WC,LF,F)
C                            FORTRAN BY KEN PEACOCK  11-11-76
C      SUBROUTINE HILF CONSTRUCTS THE OPERATOR FOR HILBERT
C      TRANSFORMATION. WEIGHTING IS AVAILABLE ON OPTION.
C         INPUTS ARE;
C           TDEL = SAMPLE INCREMENT IN SECONDS
C             WC = 0 OUTPUT RAW OPERATOR
C                NE 0 USE A ROSS WEIGHT FUNCTION WITH EXPONENT WC
C             LF = LENGTH OF FILTER IN SAMPLES (MUST BE ODD)
C        OUTPUT IS;
C              F = THE LF-LENGTH OUTPUT ARRAY
C      CODED FOR THE IBM 370/158 COMPUTER
C      VERSION AS OF 11-11-76
C
      DIMENSION F(1)
      N = LF/2
      ISTA = N+2
      KFACT = 1-ISTA
      JFACT = LF+1
      DO 1 I=1,LF
    1 F(I) = 0.
      FACT = -2./(TDEL*3.14159265)
      DO 2 I=1,N,2
      J = N+1+I
      K = N+1-I
      F(J) = FACT/I
    2 F(K) = -F(J)
      IF(WC.EQ.0.) GO TO 4
      DO 3 I=ISTA,LF
      AK = I+KFACT
      F(I) = F(I)*((1.-(AK/N)**2)**WC)
      J = JFACT-I
    3 F(J) = -F(I)
    4 RETURN
      END
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       MNMX                                                 *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  TO PICK RELATIVE MINIMA AND MAXIMA FOR INPUT FUNCTION.    *
C  ENTRY POINTS:                                                       *
C      MNMX  (A,NPT,IPEAK,JP,ITRF,JT)                                  *
C  ARGUMENTS:                                                          *
C      A       REAL         I*  (1) - INPUT ARRAY TO SCAN FOR MIN/MAX  *
C      NPT     INTEGER      I*      - NUMBER OF SAMPLES IN ARRAY 'A'   *
C      IPEAK   INTEGER      O*  (1) - INDEX  ARRAY OF RELATIVE PEAKS   *
C      JP      INTEGER      O*      - NUMBER OF RELATIVE PEAKS IN IPEAK*
C      ITRF    INTEGER      O*  (1) - INDEX  ARRAY OF RELATIVE TROUGHS *
C      JT      INTEGER      O*      - NUMBER OF REL TROUGHS IN ITRF    *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   J. NORRIS                          ORIGIN DATE: 83/06/03  *
C  LANGUAGE: FORTRAN 77/IV               DATE LAST COMPILED: 83/06/03  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  IF NPT <=1, SUBROUTINE IS NOT EXECUTED.            *
C  GENERAL DESCRIPTION:  EVERY PEAK OR TROUGH IS FOUND REGARDLESS OF   *
C       AMPLITUDE.  THIS CODE ADAPTED FROM PKDT AND RPHS BY J.H.BODINE.*
C       +------------------------------------------------------+       *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NON-STANDARD FEATURES:  NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      SUBROUTINE MNMX(A,NPT,IPEAK,JP,ITRF,JT)
      DIMENSION A(1), ITRF(1), IPEAK(1)
C
      JT = 0
      JP = 0
      SLOP = 0.
      ILAST=0
C***********************************************************************
C ***** FIND ARRAY POSITIONS OF MAXIMA AND MINIMA AFTER CHECKING VALID
C ***** LOOP INDEX.
C***********************************************************************
      NPT1 = NPT-1
      IF(NPT1.LE.0) GO TO 150
      DO 100 I=1,NPT1
         SLAST = SLOP
         SLOP = A(I+1)-A(I)
         CHNG = SLAST*SLOP
         IF (CHNG.GT.0.) GO TO 100
         IF (SLOP.EQ.0..AND.SLAST.EQ.0.) GO TO 100
         IF (SLAST.LE.0..AND.SLOP.GE.0.) GO TO 50
C***********************************************************************
C ******* HERE FOR A PEAK.
C ******* CHECK FOR TWO CONSECUTIVE PEAKS. IGNORE SUBSEQUENT PEAKS UNTIL
C ******* A TROUGH IS FOUND.  ILAST=1 FOR PEAK.
C***********************************************************************
         IF(ILAST.EQ.1)GO TO 100
         JP = JP+1
         IPEAK(JP) = I
         ILAST=1
         GO TO 100
C***********************************************************************
C ******* HERE FOR A TROUGH. SEE IF PEAK ALREADY PICKED FIRST.
C ******* CALLING ROUTINES EXPECT FIRST POINT OF ZERO SLOPE TO BE PEAK.
C ******* CHECK FOR TWO CONSECUTIVE TROUGHS. IGNORE SUBSEQUENT TROUGHS
C ******* UNTIL A PEAK IS FOUND.ILAST=-1 FOR TROUGH.
C***********************************************************************
   50    IF (JP.EQ.0) GO TO 100
         IF(ILAST.EQ.-1)GO TO 100
         JT = JT+1
         ITRF(JT) = I
         ILAST=-1
  100 CONTINUE
C***********************************************************************
C ******* ASSIGN LAST POINT OF TROUGH ARRAY TO BE LAST SAMPLE. IF NO
C ******* PEAKS FOUND, DEFAULT SET JP TO ONE POINT.
C***********************************************************************
      ITRF(JT+1) = NPT
C
  150 CONTINUE
c*
c  modified this to also set the first element of IPEAK to first sample
c						- j.m.wade 4/23/96
c*
      IF(JP.EQ.0) then
	JP=1
	IPEAK(JP) = 1
      endif
      RETURN
      END
	subroutine findpk1(trifrq,rdata,nsamp,trires,factor)
#include <localsys.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
	dimension vp(SZSMPM), vv(SZSMPM), ip(SZSMPM), iv(SZSMPM)
	dimension imin(SZSMPM),imax(SZSMPM)
	dimension trifrq(*), trires(*), rdata(*)
	call maxv(rdata,1,amaxval,lc,nsamp)
	thresh =  factor *amaxval
	call pkval1(rdata ,trifrq,nsamp,0,vp,ip,np)
	call maxv(vp   ,1,amaxpk ,lc,np   )
	call pkval1(rdata ,trifrq,nsamp,1,vv,iv,nv)
	call maxv(vv   ,1,amaxpk ,lc,nv   )
	call vclr (trires,1,nsamp)
	do 50 i = 1,np
		imin(i) = 0
		imax(i) = 0
		do 55 j = 2, nv
			if(iv(j-1) .lt. ip(i) .and.
     :			   iv(j  ) .gt. ip(i)) then
				imin(i) = iv(j-1)
				imax(i) = iv(j  )
				go to 50
			endif
55		continue
50	continue
	do 60 i = 1,np
		if (imin(i) .eq. 0 .or. imin(i) .eq. 0 ) go to 60
		do 70 j = imin(i), imax(i)
	          if (rdata(ip(i)) .gt. thresh) trires(j) = vp(i)
70		continue
60	continue
30	format(7I10)
40	format(7f10.0)
500	format(7f10.3)
1000    format(7f10.3)
	return
	end
	subroutine pkval1(rdata,trifrq,nsamp,iflag,vp,ip,np)
	dimension vp(*),ip(*),trifrq(*),rdata(*)
	j = 0
	call vclr(vp,1,nsamp)
	call vclr(vv,1,nsamp)
	do 100 i = 2,nsamp-1
	if (iflag .eq. 0 ) then
		if (rdata  (i-1) .lt. rdata  (i)
     :		.and. rdata  (i) .gt. rdata  (i+1)) then
			j = j + 1
			vp(j) = trifrq(i)
			ip(j) = i
		endif
	else
		if (rdata  (i-1) .gt. rdata  (i)
     :			.and. rdata  (i) .lt. rdata  (i+1)) then
			j = j + 1
			vp(j) = trifrq(i)
			ip(j) = i
		elseif (rdata  (i-1) .ge. rdata  (i)
     :                  .and. rdata  (i) .lt. rdata  (i+1)) then
                         j = j + 1
                         vp(j) = trifrq(i)
                         ip(j) = i
	        elseif (rdata  (i-1) .gt. rdata  (i)
     :                  .and. rdata  (i) .le. rdata  (i+1)) then
                        j = j + 1
                        vp(j) = trifrq(i)
                        ip(j) = i
		endif
	endif
100	continue
	np = j
	return
	end
c----
c  pick peaks (or troughs)
c----

	subroutine peak (rdata,nsamp,iflag,vp,ip,np)

#include <f77/lhdrsz.h>

	real      vp(SZLNHD), rdata(SZLNHD)
	integer   ip(SZLNHD)

	j = 0
        do  i = 1, nsamp
            vp (i) = 0.
        enddo

	do 100 i = 2,nsamp-1

c----
c   search for all peaks
c   vp = vector of peak values
c   ip = vector of peak positions (in original data)
c----
	if (iflag .eq. 0 ) then
		if (rdata  (i-1) .lt. rdata  (i)
     :		.and. rdata  (i) .gt. rdata  (i+1)) then
			j = j + 1
			vp(j) = rdata(i)
			ip(j) = i
		endif
c----
c   search for all troughs
c----
	else
		if (rdata  (i-1) .gt. rdata  (i)
     :			.and. rdata  (i) .lt. rdata  (i+1)) then
			j = j + 1
			vp(j) = rdata(i)
			ip(j) = i
		elseif (rdata  (i-1) .ge. rdata  (i)
     :                  .and. rdata  (i) .lt. rdata  (i+1)) then
                         j = j + 1
                         vp(j) = rdata(i)
                         ip(j) = i
	        elseif (rdata  (i-1) .gt. rdata  (i)
     :                  .and. rdata  (i) .le. rdata  (i+1)) then
                        j = j + 1
                        vp(j) = rdata(i)
                        ip(j) = i
		endif
	endif
100	continue
	np = j
	return
	end

c----
c   fit parabola to 3 points
c----
      SUBROUTINE PARAB(C1,C2,C3,X,Y)

	x = 0.0
	y = c2
        A  =0.5*(C1+C3-2.*C2)
        B = 0.5*(C3-C1)
        C = C2
        IF(A .EQ. 0.0 ) return
	  X = -B/(2.*A)
          Y = A * X**2 + B * X + C
	if ( abs(x) .gt. 1 ) then
		x = 0.0
		y = c2
	endif
      RETURN
      END
