/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
#include <stdio.h>
#include <localsys.h>

#include "srtape.h"
#include "sisio.h"

/* Author:  Kelly D. Crawford
   Date:    05/10/95
   Purpose: Handle trace skipping automatically.  Many usp codes require
            options to allow skipping of records and traces.  This code
            will let you initialize a "smart" rtape with the necessary
            skip information, and have it do the skipping for you.  This
            will decrease the amount of complexity built into the usp
            codes (although overall readability might suffer a bit).

   init_srtape(ntrc,  - Number of traces per record
               lerr,  - Unit number for error messages
               rs,    - Start record (or line)
               re,    - End record (or line)
               ns,    - Start trace
               ne     - End trace
   ) - Sets up the parameters needed for trace skipping.

   srtape( Same arguments as rtape )
     - Reads the next trace in the range of [rs,re], [ns,ne].  Will
       automatically skip traces when needed according to the following:

          skip rs-1 records
          loop
             skip ns-1 traces
                read [ns,ne]
             skip ntrc-ne+1 traces
          end loop

    Note that init_srtape and srtape are Fortran-callable.  Check the header
    file (srtape.h) for information on how to call from C for any particular
    platform.

    An example use of these tools from Fortran is shown below.

      call argi4('-rs', rs, 1, 1)
      call argi4('-re', re, nrec_in, nrec_in)
      call argi4('-ns', ns, 1, 1)
      call argi4('-ne', ne, ntr_in, ntr_in)
      call init_srtape(ntrc, lerr, rs, re, ns, ne)
      do 20 record = rs, re
         do 10 trace = ns, ne
            call srtape(luin, buf, nbytes)
            ... processing ...
 10      continue
 20   continue

    Alternatively,

      call argi4('-rs', rs, 1, 1)
      call argi4('-re', re, nrec_in, nrec_in)
      call argi4('-ns', ns, 1, 1)
      call argi4('-ne', ne, ntr_in, ntr_in)
      call init_srtape(ntrc, lerr, rs, re, ns, ne)
      total = (ne - ns + 1) * (re - rs + 1)
      do 10 i = 1, total
         call srtape(luin, buf, nbytes)
            ... processing ...
 100  continue

   These tools assume that the file has already been opened for reading
   and the line header has already been removed (i.e., the file pointer
   is resting on the first trace of the first record).
*/

/* User input parameters */
static int ntrc;         /* Number of traces */
static int lerr;         /* Unit number for error information */
static int rs;           /* Start record (line) */
static int re;           /* End record (line) */
static int ns;           /* Start trace */
static int ne;           /* End trace */
static int skip_amount;  /* How many traces to skip between records */

/* These are used to keep track of where we are in the file */
static int rcount = 0;
static int ncount = 0;

void C_INIT_SRTAPE(luin, in_ntrc, in_lerr, in_rs, in_re, in_ns, in_ne)
int luin, in_ntrc, in_lerr, in_rs, in_re, in_ns, in_ne;
{
int nrecs;

/* Set the user input parameters */
ntrc = in_ntrc;
lerr = in_lerr;
rs = in_rs;
re = in_re;
ns = in_ns;
ne = in_ne;

/* Compute how many traces to skip between records */
skip_amount = ntrc - ne + ns - 1;

/* It is our first time in and we need to set everything up.  Skip to the
   proper location, and then set rcount and ncount so that when srtape is
   called it looks like we are just doing a simple increment and grabbing
   the next trace. */
rcount = rs;
ncount = ns - 1;

/* recskp(1, rcount-1, luin, ntrc, buf); */
/* trcskp(rcount, 1, ns-1, luin, ntrc, buf); */
nrecs = (rs-1)*ntrc + ns - 1;

/* printf("init_srtape: skip %d traces\n", nrecs); */
if (nrecs > 0)
   C_SKIPT(luin, nrecs);
}

void F_INIT_SRTAPE(luin, in_ntrc, in_lerr, in_rs, in_re, in_ns, in_ne)
int *luin, *in_ntrc, *in_lerr, *in_rs, *in_re, *in_ns, *in_ne;
{
C_INIT_SRTAPE(*luin, *in_ntrc, *in_lerr, *in_rs, *in_re, *in_ns, *in_ne);
}

void C_SRTAPE(luin, buf, nbytes)
int luin;
char *buf;
int *nbytes;
{
/* Move to the next trace */
ncount++;

/* If we are past the end of traces. */
if (ncount > ne) {
   ncount = ns;
   rcount++;

   if (rcount > re) {
      /* We should have never been called this time! */
      /* fprintf(lerr, "Out of sync error from srtape!\n"); */
      fprintf(stderr, "Out of sync error from srtape!\n");
      return;
   }

   /* Skip to the proper location.  Normally this is thought of as skipping
      the remaining traces in the record, and then skipping down to trace
      ne in the next record.  Since they are actually contiguous, just
      combine the skips for efficiency. */
   /* trcskp(rcount, ne+1, ntrc, luin, ntrc, buf); */
   /* trcskp(rcount, 1, ns-1, luin, ntrc, buf); */
   /* printf("srtape: skip %d traces\n", skip_amount); */
   if (skip_amount > 0)
      C_SKIPT(luin, skip_amount);
}

/* Read in the next trace and return */
/* printf("srtape: rcount(%d) ncount(%d)\n", rcount, ncount); */
C_RTAPE(luin, buf, nbytes);
}

void F_SRTAPE(luin, buf, nbytes)
int *luin;
char *buf;
int *nbytes;
{
C_SRTAPE(*luin, buf, nbytes);
}
