C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************


C **********************************************************************
C avoan - amplitude versus offset analysis
C
C
c PROGRAM CHANGES:
c 5 April 1996 by Steve Markley  - original version;
c 20 June 1996 by Dmitriy Repin  - the following options are added '-amp',
c        '-decay', '-comp'.  The option '-linamp' was changed to '-func' 
c        and the linear scaling became default. 
c
c 1 August 1996 by Dmitriy Repin - case of tmp = 0 when scaling the 
c        amplitudes is handled.  argi4 reverted to a subroutine (as defined
c        in Fortran source) instead of int function (as defined in C source
c        of the library)
c
c 16 Dec 1996 by Steve Markley
c         Added logic to process traces outside the analysis window.
c         Added command line arguments -a[], and -b[] for application 
c         only option.
c
c 16 Dec 1996 by Steve Markley
c         Changed to use DstSgn header at Richard's request.
c
c 7 June 1997 by Steve Markley
c         Changed the offset increment (xinc) and minimum offset (xmin)
c         to a floating point variables to handle 12.5 m group intervals.
c         Added some verbose print out.
c
c 10 June 1997 by Steve Markley
c         Added standard deviation evaluation.
c
c 11 June 1997  Version 3.3   by Steve Markley
c         Fixed bug that caused reel header to be read as a trace after rewind.
c
c 10 Dec 1997   Version 3.4    by Steve Markley
c         Speed enhancements.
c
c 21 Jan 1998   Version 3.5    by Steve Markley
c         Fixed standard deviation calculation (it was off by one sample).
C         Speed enhancements (now runs in two passes rather than three).  
C         Added std deviation to xgraph file.
c
c 8 October 1998 Version 3.6   by Steve Markley
c         Fixed argument mismatch in avoan_vsmul that caused the apply only
c         option to fail.   Added comments to avoan_vsmul.
c
c
c PROGRAM DESCRIPTION:
C Performs an amplitude analysis on the input dataset.  Using the user-
C supplied xmin, xinc, and noff, the program estimates amplitude versus 
C offset dependence.  The amplitude versus offset curve is generated once 
C for the whole dataset, i.e. job constant operation is always performed.
C The analysis results are written to an auxiliary file.  The auxiliary 
C file can be plotted on a workstation with the xgraph command.
C
C avoan also performs an offset variable amplitude scaling.  The scaling 
C is keyed on the DstSgn trace header and uses the following algorithm:
C - first, the least-square linear fit to the measured average absolute, 
C   rms, or geometric mean amplitudes (the type of the used amplitude is 
C   determined by the -amp option) is performed;
C - second, the desired amplitude curve is constructed by combining 
C   an intercept of the linear fit and a slope derived from the amplitude 
C   decay at the last offset supplied by a user (-decay option);
C - third, for each offset, a scalar is derived as a weighted sum (with 
C   the weighting coefficient taken from the -comp option) of two
C   scaling coefficients computed by a) dividing the desired amplitude by 
C   the amplitude of the least-square fit and b) dividing the desired 
C   amplitude by the average amplitude at this offset;
C - forth, the output amplitude is calculated by multiplying the input
C   amplitude by the scalar.
C The resulting output amplitude curve has the same intercept as the 
C input-amplitude fitting curve, a slope that provides the specified 
C amplitude decay at the last offset, and its maximum deviation from 
C the desired amplitude curve is controlled by the weighting coefficient. 
C
C Note that an exponential amplitude scaling is also available as the 
C -funcexp option.  With this option the output amplitude curve will also
C have an additional property.  The amplitude decay between any two 
C neighboring offsets will be the same percentage change.  
C
C **********************************************************************
        implicit none
C **********************************************************************

c get machine dependent parameters 
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

#define DEBUG .true.

c dimension standard USP variables 

      integer     itr ( SZLNHD )
     
      integer     nsamp, nsi, ntrc, nrec, iform
      integer     luin , luout, lbytes, nbytes, lbyout, obytes
      integer     irs, ire, ns, ne, argis, jerr, jj, kk
      integer     ludsk

      real        tri ( SZLNHD )

      character   ntap*255, otap*255, name*5

      logical     verbos

C **********************************************************************
C ***** Static Variables ***********************************************
C **********************************************************************

        real x_min      ! Minimum source to receiver offset
        real x_inc      ! Source to receiver offset increment
        integer n_x        ! Number of source to receiver values
        integer n_err      ! error count
        integer ix_x       ! Address of the ith offset
        integer mem_status ! status of a memory allocation
        integer start_samp ! Address offset to start sample of analysis window
        integer wind_len   ! Number of samples in the analysis window
        integer i_samp
        integer n_less     ! Count of traces with offsets significantly < x_min
        integer n_large    ! Count of traces with offsets significantly > x_max
        real min_offset ! Mimimum offset in the input dataset
        real max_offset ! Maximum offset in the input dataset
        real x_max      ! Maximum offset (= x_min + (n_x-1) * x_inc)
        integer nonz_x     ! Number of offsets with none zero amplitudes
        integer iostatus

        real amp      ! The amplitude of a seismic sample
        real sum      ! Sum of trace amplitudes
        real sum_sq   ! Sum of squared trace amplitudes
        real sum_ln   ! Sum of log of absolute trace amplitudes
        real s        ! Window start time
        real e        ! Window end time
        real weight   ! Sample count
        real a, b     ! Coefficients of the linear amp (amp = a*offset + b)
        real x1, y1   ! X, Y coordinate of constrained least squares solution
        real x        ! offset
        real compact  ! degree of "squeezing" of assigned amps (range ]0,1] )
        real decline  ! Decline of amplitude [0,1] at the last offset 
        real decay    ! Decay of the amplitude (%) at the last offset  
        real anew     ! Assigned coefficient of the linear amp fit
        real scale    ! Trace scalar

        character sfile*255    ! Filename of xgraph output
        character*10 amptype   ! Name of the ampl. type used in fitting
        character*10 fittype   ! Linear, hyperbolic  or exponential

        logical analysis   ! Analysis mode (no trace o/p)
        logical apply      ! Apply only flag
        logical pipe_input

        integer warnings

C **********************************************************************
C ***** Dynamic variables **********************************************
C **********************************************************************

        real abs_amp         ! Absolute amplitude
        real sq_amp          ! Squared amplitudes
        real ln_amp          ! Log of absolute amp for geometric mean
        real n_live          ! Number of live samples in each offset bin
        real sum_sq_error    ! Squared error sum for standard deviation comp
        real offsets         ! Table of actual offset values
        real y               ! Array of amplitudes to fit to y = a*x + b
        pointer (ixr_abs_amp, abs_amp(1))
        pointer (ixr_sq_amp, sq_amp(1))
        pointer (ixr_ln_amp, ln_amp(1))
        pointer (ixr_n_live, n_live(1))
        pointer (ixr_sum_sq_error, sum_sq_error(1))
        pointer (ixr_offsets, offsets(1))
        pointer (ixr_y, y(1))


c Program Specific _ dynamic memory variables

      integer TraceSize, errcd1, abort

      real    Trace_WorkSpace

      pointer (wkadr1, Trace_WorkSpace(200000))

c Program Specific _ static memory variables

      integer ifmt_StaCor,l_StaCor,ln_StaCor, StaCor
      integer ifmt_DstSgn, l_DstSgn, ln_DstSgn, DstSgn
      integer DstUsg

c External functions
      integer eliminate_0s ! Eliminates traces with 0 amplitudes

c Temp vars
      real    tmp
      integer error        ! Error code returned by functions
      real anumerator, denominator

c Initialize variables

      data abort/0/
      data name/"AVOAN"/
      warnings = 0

c give command line help if requested

      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis ( '-h' ) .gt. 0 .or. 
     :     argis ( '-help' ) .gt. 0 ) then
         call help()
         stop
      endif

c open printout file

#include <f77/open.h>

c get command line input parameters

        call cmdln (ntap, otap, sfile, ns, ne, irs, ire, s, e, 
     *     x_min, x_inc, n_x, name, analysis, verbos, fittype, 
     *     amptype, decay, compact, apply, a, b, n_err)

        if (n_err .gt. 0) goto 9999
C--------------------
C Translate from user specified units (%) to the internal program
C units
C--------------------
        decline = 1 + (decay / 100.)

C **********************************************************************
C ***** Open input and output files ************************************
C **********************************************************************

        call getln (luin , ntap,'r', 0)

c  read input line header and save certain parameters

        call rtape (luin, itr, lbytes)
        if (lbytes .eq. 0) then
            write (LERR, 5110) ntap
            write (LER, 5110) ntap
 5110       format ('AVOAN: no line header on input dataset' / 
     *             a100 / 'FATAL')
            call lbclos (luin)
            if (.not. analysis) call lbclos (luout)
            stop

        endif

c ----> Check input filename
        pipe_input = .false.

        if (ludsk(luin) .eq. 0) then
            pipe_input = .true.

            if (analysis) then
                write (LERR, 5120)
                write (LER, 5120)
 5120           format (/ ' Standard deviation calculation will ',
     *                         'not be performed on piped input')
            else
                write (LERR, 5000)
                write (LER, 5000)
 5000           format (/ ' Anlaysis mode (-anl) is required ',
     *                                           'for piped input')
                goto 999
            endif
        endif

c ----> Check output filename
        if (analysis) then
            if (otap(1:2) .ne. '  ') then
                write (LERR, 5100)
                write (LER, 5100)
 5100           format (/ 'Analysis mode, output filename ignored'/)
                otap(1:2) = '  '
            endif
        else
            call getln (luout, otap,'w', 1)

        endif

c ----> open xgraph file
        if (.not. apply) then
            open (lun, file = sfile, status = 'unknown',
     *                                             iostat = iostatus)
            if (iostatus .ne. 0) then
                write (LERR, 4100)
                write (LER, 4100)
 4100           format (/ 'Could not open file for amplitude plot - ',
     *                                                     'FATAL')
                goto 999
            endif
        endif

        if (n_err .gt. 0) goto 999

        call saver (itr, 'NumSmp', nsamp, LINHED)
        call saver (itr, 'SmpInt', nsi  , LINHED)
        call saver (itr, 'NumTrc', ntrc , LINHED)
        call saver (itr, 'NumRec', nrec , LINHED)
        call saver (itr, 'Format', iform, LINHED)

c define pointers to header words required by your routine

       call savelu ('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
       call savelu ('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)

c historical line header and print to printout file 

       call hlhprt (itr, lbytes, name, 5, LERR)

C **********************************************************************
C **********************************************************************
C **********************************************************************
c check user supplied boundary conditions and set defaults

      if (irs .eq. 0 )   irs = 1
      if (ire .eq. 0 )   ire = nrec
      if (irs .gt. nrec) then
         write(LER, *)'***ERROR: rs > number of records on input'
         write(LERR,*)'***ERROR: rs > number of records on input'
         goto 999
      endif
      if (ire .gt. nrec) then
         write(LER, *)'***WARNING: re > number of records on input,',
     &                ' assuming re = ',nrec
         write(LERR,*)'***WARNING: re > number of records on input,',
     &                ' assuming re = ',nrec
         ire = nrec
      endif

      if ( ns .eq. 0 ) ns = 1
      if ( ne .eq. 0 ) ne = ntrc
      if ( ns .gt. ntrc) then
         write(LER, *)'***ERROR: ns > number of traces in a record'
         write(LERR,*)'***ERROR: ns > number of traces in a record'
         goto 999
      endif
      if ( ne .gt. ntrc) then
         write(LER, *)'***WARNING: ne > number of traces in a record,',
     &                ' assuming ne = ', ntrc
         write(LERR,*)'***WARNING: re > number of traces in a record,',
     &                ' assuming ne = ', ntrc
         ne = ntrc
      endif

      if ( s .gt. real( nsi*(nsamp-1) )) then
C        e = real(nsi * (nsamp-1)) 
         write(LER, *)'***ERROR: s > trace length' 
         write(LERR,*)'***ERROR: s > trace length' 
         goto 999
      endif

      if (e .gt. real( nsi*(nsamp-1) )) then
         e = real(nsi * (nsamp-1)) 
         write(LER, *)'***WARNING: e > traces length, assuming e= ', e 
         write(LERR,*)'***WARNING: e > traces length, assuming e= ', e 
      endif

      start_samp = nint (real(s) / real(nsi))

      if (e .eq. 0.0) then
            e = real(nsi * (nsamp-1)) 
            wind_len = nsamp - start_samp + 1
      elseif (e .lt. s) then
            write (LERR, 4200) e, s
            write (LER, 4200) e, s
 4200       format (/ 'AVOAN ERROR: Specified window stop time,', f6.0, 
     *          '  is less than the specified start time,', f6.0)
            goto 999
      else
            wind_len = int ((e - s) / real (nsi)) + 1
      endif

c NOTE: In this case, the processing limits: irs, ire, ns, ne, ist, and
C       iend are used to limit analysis only.
C       All data traces and samples are actually passed.
 
      call savew (itr, 'NumRec', nrec, LINHED)
      call savew (itr, 'NumTrc', ntrc, LINHED)
 
c number output bytes

      obytes = SZTRHD + SZSMPD * nsamp 

c save out hlh and line header

      call savhlh (itr, lbytes, lbyout)
      if (.not. analysis) call wrtape (luout, itr, lbyout)

c verbose output of all pertinent information before processing begins

      call verbal (ntap, otap, sfile,
     *               nsamp, nsi, ntrc, nrec, iform, s, 
     *               e, irs, ire, ns, ne, analysis, verbos,
     *               amptype, decay, compact, fittype, 
     *               apply, a, b)

c dynamic memory allocation:  

      TraceSize = nsamp 
      call galloc (wkadr1, TraceSize * SZSMPD, errcd1, abort)
    
      if (errcd1 .ne. 0) then
         write (LERR, 1100) TraceSize * SZSMPD
         write (LER, 1100) TraceSize * SZSMPD
 1100    format (/'Unable to allocate workspace:', i8, ' bytes' /)
         goto 999

      else
         write(LERR, 1110) TraceSize * SZSMPD
         write(LER, 1110) TraceSize * SZSMPD
 1110    format (/ 'Allocating workspace:', i8, ' bytes' /)

      endif

c initialize memory
         
      call vclr (Trace_WorkSpace, 1, TraceSize)

C ----> Analysis buffers
        call galloc (ixr_abs_amp, n_x * SZSMPD, mem_status, abort)
        if (mem_status .ne. 0) then
            write (LERR, 4000) n_x * SZSMPD
            write (LER, 4000) n_x * SZSMPD
 4000       format (/ ' Unable to allocate workspace:', i8, ' bytes' /)
            goto 999

        endif
        call vclr (abs_amp, 1, n_x)

        call galloc (ixr_sq_amp, n_x * SZSMPD, mem_status, abort)
        if (mem_status .ne. 0) then
            write (LERR, 4000) n_x * SZSMPD
            write (LER, 4000) n_x * SZSMPD
            goto 999

        endif
        call vclr (sq_amp, 1, n_x)

        call galloc (ixr_ln_amp, n_x * SZSMPD, mem_status, abort)
        if (mem_status .ne. 0) then
            write (LERR, 4000) n_x * SZSMPD
            write (LER, 4000) n_x * SZSMPD
            goto 999

        endif
        call vclr (ln_amp, 1, n_x)

        call galloc (ixr_n_live, n_x * SZSMPD, mem_status, abort)
        if (mem_status .ne. 0) then
            write (LERR, 4000) n_x * SZSMPD
            write (LER, 4000) n_x * SZSMPD
            goto 999

        endif
        call vclr (n_live, 1, n_x)

        call galloc (ixr_sum_sq_error, n_x * SZSMPD, mem_status, abort)
        if (mem_status .ne. 0) then
            write (LERR, 4000) n_x * SZSMPD
            write (LER, 4000) n_x * SZSMPD
            goto 999

        endif
        call vclr (sum_sq_error, 1, n_x)

        call galloc (ixr_offsets, n_x * SZSMPD, mem_status, abort)
        if (mem_status .ne. 0) then
            write (LERR, 4000) n_x * SZSMPD
            write (LER, 4000) n_x * SZSMPD
            goto 999

        endif
        call vclr (offsets, 1, n_x)

        call galloc (ixr_y, n_x * SZSMPD, mem_status, abort)
        if (mem_status .ne. 0) then
            write (LERR, 4000) n_x * SZSMPD
            write (LER, 4000) n_x * SZSMPD
            goto 999

        endif
        call vclr (y, 1, n_x)


C **********************************************************************
C ***** For apply only option, skip analysis ***************************
C **********************************************************************
 
        if (apply) goto 9000
 

C **********************************************************************
C ***** First trace pass for amplitude analysis ************************
C **********************************************************************

c ----> setup error counting
        n_less = 0
        n_large = 0
        min_offset = 999999.
        max_offset = 0.

        call recskp (1, irs-1, luin, ntrc, itr)     ! skip unwanted input rec
        DO JJ = irs, ire
            call trcskp (JJ, 1, ns-1, luin, ntrc, itr)  ! skip to start trace

            DO KK = ns, ne
                nbytes = 0
                call rtape (luin, itr, nbytes)

                if (nbytes .eq. 0) then
                    write (LERR, *) 'Premature EOF on input at:'
                    write (LERR, *) '  rec= ', JJ, '  trace= ', KK
                    goto 999
                endif

c ------------> get required trace header information
                call saver2 (itr, ifmt_StaCor, l_StaCor, ln_StaCor, 
     *                   StaCor, TRACEHEADER)
   
                call saver2 (itr, ifmt_DstSgn, l_DstSgn, ln_DstSgn, 
     *                   DstSgn, TRACEHEADER)

c ------------> process only live traces
                if (StaCor .ne. 30000) then

                    call vmov (itr(ITHWP1), 1, tri, 1, nsamp)

                    DstUsg = abs(DstSgn)

                    ix_x = 1 + nint ((real(DstUsg) - x_min) / x_inc)

                    if (ix_x .lt. 1) then
                        n_less = n_less + 1
                        ix_x = 1
                        min_offset = min (min_offset, real(DstUsg))
                        if (verbos) then
                            write (LERR, 6000) JJ, KK, DstUsg
                            write (LER, 6000) JJ, KK, DstUsg
 6000                       format (' RecNum: ', i9, '    TrcNum: ',
     *                               i5, '   has an offset, ', i8,
     *                               ', less than the minimum')
                        endif

                    elseif (ix_x .gt. n_x) then
                        n_large = n_large + 1
                        ix_x = n_x
                        max_offset = max (max_offset, real(DstUsg))
                        if (verbos) then
                            write (LERR, 6100) JJ, KK, DstUsg
                            write (LER, 6100) JJ, KK, DstUsg
 6100                       format (' RecNum: ', i9, '    TrcNum: ',
     *                               i5, '   has an offset, ', i8,
     *                               ', greater than the maximum')
                        endif

                    else
                        sum = 0.0
                        sum_sq = 0.0
                        sum_ln = 0.0
                        weight = 0.0

                        do i_samp = start_samp + 1, 
     *                              start_samp + wind_len

                            amp = abs (tri(i_samp))
                            if (amp .ne. 0.0) then
                                sum = sum + amp
                                sum_sq = sum_sq + amp * amp
                                sum_ln = sum_ln + log (amp)
                                weight = weight + 1.0

                            endif
                        enddo

                        abs_amp(ix_x) = abs_amp(ix_x) + sum
                        sq_amp(ix_x) = sq_amp(ix_x) + sum_sq
                        ln_amp(ix_x) = ln_amp(ix_x) + sum_ln
                        n_live(ix_x) = n_live(ix_x) + weight

                    endif
                endif
            ENDDO
 
            call trcskp (JJ, ne+1, ntrc, luin, ntrc, itr) ! skip to end of rec

        ENDDO


C **********************************************************************

        if (n_less .gt. 1) then
            write (LERR, 2000) n_less, 
     *                         x_min - x_inc / 2., 
     *                         min_offset
            write (LER, 2000)  n_less, 
     *                         x_min - x_inc / 2., 
     *                         min_offset
 2000       format (/ 'WARNING - traces encountered with offsets ',
     *              'less than XMIN' // i6, ' traces ',
     *              'found with offsets less than ', f12.3 / 
     *              '    The minimum offset encountered was ', f12.3/)

        endif

        if (n_large .gt. 1) then
            write (LERR, 2100) n_large, 
     *                         x_min + (real (n_x) - .5) * x_inc, 
     *                         max_offset
            write (LER, 2100)  n_large, 
     *                         x_min + (real (n_x) - .5) * x_inc, 
     *                         max_offset
 2100       format (/ 'WARNING - traces encountered with offsets ',
     *              'greater than the' , 
     *              ' expected maximum offset', // i6, ' traces ',
     *              'found with offsets greater than ', f12.3 / 
     *              '    The maximum offset encountered was ', f12.3/)

        endif


C **********************************************************************
C ***** Perform analysis operations ************************************
C **********************************************************************
C ----> Compute mean values
        do ix_x = 1, n_x
            offsets(ix_x) = x_min + real (ix_x - 1) * x_inc
            if (n_live(ix_x) .gt. 1.) then
                sum_sq_error(ix_x) = abs_amp(ix_x)**2 / n_live(ix_x)
                sum_sq_error(ix_x) = sq_amp(ix_x) - sum_sq_error(ix_x)
                sum_sq_error(ix_x) = sqrt (sum_sq_error(ix_x) / 
     *                                     (n_live(ix_x) - 1.))
                abs_amp(ix_x) = abs_amp(ix_x) / n_live(ix_x)
                sq_amp(ix_x) = sqrt (sq_amp(ix_x) / n_live(ix_x))
                ln_amp(ix_x) = exp (ln_amp(ix_x) / n_live(ix_x))
            else
                sum_sq_error(ix_x) = 0.
                abs_amp(ix_x) = 0.0
                sq_amp(ix_x) = 0.0
                ln_amp(ix_x) = 0.0
            endif
        enddo

        if (verbos) then
            write (LERR, 6200) (ix_x, offsets(ix_x), nint(n_live(ix_x)),
     *                        abs_amp(ix_x), sq_amp(ix_x), ln_amp(ix_x),
     *                        sum_sq_error(ix_x),
     *                        ix_x = 1, n_x)

            write (LER, 6200) (ix_x, offsets(ix_x), nint(n_live(ix_x)),
     *                        abs_amp(ix_x), sq_amp(ix_x), ln_amp(ix_x),
     *                        sum_sq_error(ix_x),
     *                        ix_x = 1, n_x)

 6200       format (15x, 'offset     n live samples   mean abs amp',
     *                   '      RMS amp     geom mean amp ',
     *                   '      Std Dev' / 
     *             9999(i10, f10.1, i16, 3f16.3, f16.4/))

        endif

c ----> Write xgraph values
        write (lun, 887)
 887    format ('"Average abs amp')
        do ix_x = 1, n_x
            if (n_live(ix_x) .gt. 0)
     *                write (lun, 888) offsets(ix_x), abs_amp(ix_x)
 888        format (f12.4, f20.6)
        enddo
 
        write (lun, 890)
 890    format (/ '"RMS amplitude')
        do ix_x = 1, n_x
            if (n_live(ix_x) .gt. 0)
     *                write (lun, 888) offsets(ix_x), sq_amp(ix_x)
        enddo
 
        write (lun, 891)
 891    format (/ '"Geometric mean')
        do ix_x = 1, n_x
            if (n_live(ix_x) .gt. 0.0) 
     *                write (lun, 888) offsets(ix_x), ln_amp(ix_x)
        enddo
 
        write (lun, 896)
 896    format (/ '"Standard deviation')
        do ix_x = 1, n_x
            if (n_live(ix_x) .gt. 0.0) 
     *                write (lun, 888) offsets(ix_x), sum_sq_error(ix_x)
        enddo
 

C ***** Compute linear fit *********************************************

C----------------------------------------------------------------------
C Eleminate zeros and transform the data for the linear fit (for 
C the 'lin' fit y=amp, for 'hyp' fit y=1/amp, for 'exp' fit y=log(amp))
C----------------------------------------------------------------------
        if     ( amptype .eq. 'abs'  ) then
           error = eliminate_0s (n_x, x_min, x_inc, abs_amp(1),
     *                            fittype, nonz_x, offsets(1), y(1))
           if( error .ne. 0 ) goto 999
        elseif ( amptype .eq. 'rms' ) then
           error = eliminate_0s(n_x, x_min, x_inc, sq_amp(1),
     *                           fittype, nonz_x, offsets(1), y(1))
           if( error .ne. 0 ) goto 999
        elseif ( amptype .eq. 'geom'  ) then
           error = eliminate_0s(n_x, x_min, x_inc, ln_amp(1),
     *                           fittype, nonz_x, offsets(1), y(1))
           if( error .ne. 0 ) goto 999
        else
           write (LERR,*)'***Internal ERROR: unrecognized amptype'
           write (LER, *)'***Internal ERROR: unrecognized amptype'
           goto 999
       endif

c ----> Perform least mean squared error fit
        call abfit (offsets, y, nonz_x, a, b)      ! fit y = a*x + b

        x_max = x_min + real (n_x - 1) * x_inc

        if( (fittype .eq. 'lin') .or. (fittype .eq. 'hyp') ) then
           if (b .lt. 0.0) then    ! If the lmse fit had a polarity reversal
              x1 = 0.0
              y1 = 0.0
              call constrained_fit (offsets, y, nonz_x, x1, y1, a, b) 
           elseif ((a * x_max + b) .lt. 0.0) then
              x1 = x_max
              y1 = 0.0
              call constrained_fit (offsets, y, nonz_x, x1, y1, a, b) 
           endif
        elseif ( fittype .eq. 'exp' ) then        
           continue
        else
           write (LERR,*)'***Internal ERROR: unrecognized -func value'
           write (LER,*) '***Internal ERROR: unrecognized -func value'
           goto 999
        endif
C-----------------------------------------------------------------------
C Translate decline of the amplitude on the last sensor compared to the
C x-offset sensor (x=0) to the fit coefficient a and b for the different 
C types of fit:
C-----------------------------------------------------------------------
C Assign intercep f(x=0), anew, and tmp=f(x_max) where
C linear amplitudes: f(x) = anew * x + b
C hyperb amplitudes: f(x) = 1 / (anew * x + b)
C expon  amplitudes: f(x) = exp( anew * x + b )
C-----------------------------------------------------------------------
        x = 0
        if     ( fittype .eq. 'lin' ) then
                anew = (decline-1)*b / ( x_max - decline*x )
                tmp  = anew * x + b
        elseif ( fittype .eq. 'hyp' ) then        
                anew = (decline-1)*b / ( x - decline*x_max )
                tmp  = anew * x + b
                if( tmp .ne. 0 ) tmp = 1 / tmp
        elseif ( fittype .eq. 'exp' ) then        
                anew = alog(decline) / ( x_max - x )
                tmp  = exp( anew * x_max + b )
        endif

C--------
C Now check for the polarity change for assigned amplitudes
C--------
        if( tmp .lt. 0.0) then
               write (LERR, 3111) decay
               write (LER,  3111) decay
 3111          format ('AVOAN: ',f12.8,' specified in -decay option ',
     *                 'causes a polarity change.' / 
     *                 '       Try another value')
               goto 999
        endif

C------------------------------------------------------------------
C Write graphs of fitted, desired, and projected amplitudes to 
C the xgraph file
C------------------------------------------------------------------
C fitted amps
C------------
        if ( fittype .eq. 'hyp') then           
            write (lun, 893) amptype
 893        format (/ '"Fit to ',A4,' scalars')
            do ix_x = 0, n_x - 1
                x = x_min + real (ix_x) * x_inc
                tmp = a * x + b
                if( tmp .ne. 0 ) write (lun, 888) x, 1.0 / tmp
            enddo
        elseif( fittype .eq. 'lin' ) then      
            write (lun, 892) amptype
 892        format (/ '"Fit to ',A4,' amp')
            do ix_x = 0, n_x - 1
                x = x_min + real (ix_x) * x_inc
                write (lun, 888) x, a * x + b
            enddo
        elseif ( fittype .eq. 'exp' ) then        
            write (lun, 893) amptype
            do ix_x = 0, n_x - 1
                x = x_min + real (ix_x) * x_inc
                write (lun, 888) x, exp( a * x + b)
            enddo
        endif
C------------
C desired amps
C------------
        write (lun, 894)
 894    format (/ '"Desired fit')
        if( fittype .eq. 'lin' ) then      
            do ix_x = 0, n_x - 1
                x = x_min + real (ix_x) * x_inc
                write (lun, 888) x, anew * x + b
            enddo
        elseif( fittype .eq. 'hyp') then           
            do ix_x = 0, n_x - 1
                x = x_min + real (ix_x) * x_inc
                tmp = anew * x + b
                if( tmp .ne. 0 ) write (lun, 888) x, 1.0 / tmp
            enddo
        elseif ( fittype .eq. 'exp' ) then        
            do ix_x = 0, n_x - 1
                x = x_min + real (ix_x) * x_inc
                write (lun, 888) x, exp( anew * x + b )
            enddo
        endif
C------------
C Projected amps
C------------
          write (lun, 895) amptype
 895      format (/ '"Projected ',A4,' amp')
          do ix_x = 1, n_x
             if (n_live(ix_x) .ne. 0.0) then
                x   = x_min + real (ix_x - 1) * x_inc

                if     ( fittype .eq. 'hyp' ) then           
                        tmp = 1. / (a * x + b)
                        amp = 1. / (anew * x + b)
                elseif ( fittype .eq. 'lin' ) then       
                        tmp = a    * x + b
                        amp = anew * x + b
                elseif ( fittype .eq. 'exp' ) then        
                        tmp = exp( a    * x + b )
                        amp = exp( anew * x + b )
                endif

                if    (amptype .eq. 'abs') then
                        tmp = abs_amp(ix_x) * amp / tmp
                elseif(amptype .eq. 'rms') then 
                        tmp = sq_amp(ix_x) * amp / tmp
                elseif(amptype .eq. 'geom') then 
                        tmp = ln_amp(ix_x) * amp / tmp
                endif

                write(lun, 888) x, (1-compact)*amp + compact*tmp
            endif
          enddo

        close (lun)
C-------------
C Some debugging info...
C------------- 
        if (verbos) then
            if    ( fittype .eq. 'hyp') then        
                write (LERR, 4400) a, b
                write (LER,  4400) a, b
 4400           format (/ 'Scalars fitted with: '/
     *              ' scale = ', f12.8, ' * offset    + ', f10.5 /)

            elseif ( fittype .eq. 'lin' ) then       
                write (LERR, 4410) a, b
                write (LER,  4410) a, b
 4410           format (/ 'Data amplitudes fitted with: '/
     *              ' amplitude = ', f12.8, ' * offset    + ', f10.5 /)

            elseif ( fittype .eq. 'exp' ) then        
                write (LERR, 4420) a, b
                write (LER,  4420) a, b
 4420           format (/ 'Data amplitudes fitted with: '/
     *              ' amplitude = exp (', f11.7, ' * offset    + ', 
     *                                                    f12.7, ')' /)

            endif
        endif


C ***** Stop here if user requested analysis only **********************

        if (analysis) then
            call lbclos (luin)             ! close data files 
            close (lun)

            write (LERR, *) 'AVOAN: Normal Termination'
            write (LER,  *) 'AVOAN: Normal Termination'
            stop             ! If analysis only mode, terminate

        endif


C **********************************************************************
C ***** Second data pass for scalar application ************************
C **********************************************************************

        call rwd (luin)
        call rtape (luin, itr, lbytes)       ! read reel header

 9000   continue         ! Jump to here if apply only option invoked
 
 
C ***** Processing loop (all traces are processed) *********************
 
        DO JJ = 1, nrec
 
            DO KK = 1, ntrc
                nbytes = 0
                call rtape (luin, itr, nbytes)

                if (nbytes .eq. 0) then
                    write (LERR,*) 'Premature EOF on input at:'
                    write (LERR,*) '  rec= ',JJ,'  trace= ',KK
                    goto 999
                endif

c ------------> get required trace header information
                call saver2 (itr, ifmt_StaCor, l_StaCor, ln_StaCor, 
     *                 StaCor, TRACEHEADER )

                call saver2 (itr, ifmt_DstSgn, l_DstSgn, ln_DstSgn, 
     *                 DstSgn, TRACEHEADER )

C---------------------------------------------------------------------
C Process only live traces that are within a specified range of offsets
C and simply copy all other traces
C Pass the rest to the output untouched 
C---------------------------------------------------------------------
                DstUsg = abs (DstSgn)
                x = real (DstUsg)
 
                if (apply) then
                    if (fittype .eq. 'lin') then
                        scale = a * x + b
                        if (scale .ne. 0.) then
                            scale = 1. / scale
                        else
                            scale = 1.
                        endif
                    elseif (fittype .eq. 'exp') then
                        scale = exp (-b - a * x)
                    endif
                    call avoan_vsmul (itr(ITHWP1), scale, itr(ITHWP1), 
     *                                                            nsamp)

c                   call vmov (itr(ITHWP1), 1, tri, 1, nsamp)
c                   do i_samp = 1, nsamp
c                       tri(i_samp) = tri(i_samp) * scale
c                   enddo
c                   call vmov (tri, 1, itr(ITHWP1), 1, nsamp)
 
 
C---------------------------------------------------------------------
C If trace is marked as dead, just write it back.  Otherwise, scale it
C---------------------------------------------------------------------
                elseif (StaCor .ne. 30000) then

                   ix_x = 1 + nint ((real (DstUsg) - x_min) / x_inc)

                   if ( (ix_x .ge. 1) .and. (ix_x .le. n_x) ) then
                      if    (amptype .eq. 'abs') then
                         tmp = abs_amp(ix_x)
                      elseif(amptype .eq. 'rms') then
                         tmp = sq_amp(ix_x)
                      elseif(amptype .eq. 'geom') then
                         tmp = ln_amp(ix_x)
                      endif
                   else
                      tmp = 0.0
                   endif

                   if ( tmp .ne. 0.0 ) then
C------------------------------------------------------------------
C Estimate numerator and denominator of the scaling ratio for the
C output amplitudes and scale them as:
C    trace(out) = tmp * trace(in)
C The scaling coefficient is determined as:
C    tmp = Amp(desired)*( (1-compact)/Amp(real) + compact/Amp(from_fit) )
C------------------------------------------------------------------
                      if     ( fittype .eq. 'hyp' ) then
                         anumerator = (1-compact)/tmp + compact*(a*x+b)
                         denominator= anew * x + b
                      elseif ( fittype .eq. 'lin' ) then
                         anumerator = anew * x + b
                         denominator= (1-compact)/tmp + compact/(a*x+b)
                      elseif ( fittype .eq. 'exp' ) then
                         anumerator = exp( anew * x + b )
                         denominator= (1-compact)/tmp
                         denominator= denominator + compact/(exp(a*x+b))
                      endif
                      scale = anumerator * denominator
                   else
C-----------------------------------------------------------------------
C *** Average amp at this distance in the specified time window is 0 ***
C Thus we can do scaling only using extrapolated amplitude from fit:
C Output amplitudes than are:
C    trace(out) = scale * trace(in)
C The scaling coefficient is determined as:
C    scale = Amp(desired)/Amp(from_fit) )
C-----------------------------------------------------------------------
                      warnings = warnings + 1
                        if (verbos) then
                            write (LERR, 6410) JJ, KK, DstUsg
                            write (LER, 6410) JJ, KK, DstUsg
 6410                       format (' RecNum: ', i9, '    TrcNum: ',
     *                               i5, '   offset: ', i8,
     *                               '     is zero amplitude')
                        endif

                      if     ( fittype .eq. 'hyp' ) then
                         anumerator = a*x+b
                         denominator= anew * x + b
                      elseif ( fittype .eq. 'lin' ) then
                         anumerator = anew * x + b
                         denominator= 1. / (a*x+b)
                      elseif ( fittype .eq. 'exp' ) then
                         anumerator = exp( anew * x + b )
                         denominator= 1. / exp( a * x + b)
                      endif
                      scale = anumerator * denominator
                   endif

                   call vmov (itr(ITHWP1), 1, tri, 1, nsamp)
                   do i_samp = 1, nsamp
                      tri(i_samp) = tri(i_samp) * scale
                   enddo
                   call vmov (tri, 1, itr(ITHWP1), 1, nsamp)
 
                endif
 
                call wrtape (luout, itr, obytes)
 
            ENDDO
        ENDDO

C **********************************************************************
C **********************************************************************
C **********************************************************************
 
        call lbclos (luin)
        call lbclos (luout)

        IF (warnings .gt. 0) THEN
           WRITE(LER, 5555)
           WRITE(LERR,5555)
 5555      FORMAT('***WARNING: encountered nonzero trace(s) at ',
     *            'offset(s) with 0 average', /
     &            '            within the selected time window.' /
     &            '            -comp 1. was used for all such traces' )
        ENDIF
 
        write (LERR, *) 'AVOAN: Normal Termination'
        write (LER, *) 'AVOAN: Normal Termination'
        stop


C **********************************************************************
C ***** Abnormal termination *******************************************
C **********************************************************************

 999  continue
      call lbclos (luin)
      if (.not. analysis) call lbclos (luout)
      if (.not. apply) close (lun)

 9999 continue
      write (LERR, *) 'AVOAN: ABNORMAL Termination'
      write (LER, *) 'AVOAN: ABNORMAL Termination'
      stop

      end
C-----END-OF-FUNCTION-MAIN-----




      subroutine help()
      implicit none

c provide terse online help [detailed help goes in man page]

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for AVOAN: avo amplitude',
     *                                         ' analysis'
      write(LER,*)' '
      write(LER,*)' For a more detailed description of these parameters'
      write(LER,*)' see the online man page using uman, xman or xuspman'
      write(LER,*)' '
      write(LER,*)'Input........................................ (def)'
      write(LER,*)' '
      write(LER,*)'-N[]     -- input data set                  (stdin)'
      write(LER,*)'-O[]     -- output data set                (stdout)'
      write(LER,*)'-s[]     -- process start time (ms)             (1)'
      write(LER,*)'-e[]     -- process end time (ms)     (last sample)'
      write(LER,*)'-ns[]    -- start trace number                  (1)'
      write(LER,*)'-ne[]    -- end trace number           (last trace)'
      write(LER,*)'-rs[]    -- start record                        (1)'
      write(LER,*)'-re[]    -- end record                (last record)'
      write(LER,*)'-xmin[]  -- minimum unsigned offset          (none)'
      write(LER,*)'-xinc[]  -- offset increment                 (none)'
      write(LER,*)'-noff[]  -- number of offsets                (none)'
      write(LER,*)'-slope[] -- Apply linear slope, no analysis     (0)'
      write(LER,*)'-icept[] -- Apply linear intercept, no analysis (1)'
      write(LER,*)'-sfile[] -- xgraph file name           (xgraph.xgr)'
      write(LER,*)'-V       -- verbos printout'
      write(LER,*)'-anl     -- analysis only option (req for pipe i/p)'
      write(LER,*)'-func[]  -- function used in fit and scaling  (lin)'
      write(LER,*)'            range:   [ lin | exp ]                 '
      write(LER,*)'-amp[]   -- amplitude type used in fit,       (abs)'
      write(LER,*)'            range:   [ abs | rms | geom]           '
      write(LER,*)'-decay[] -- ampl decay (%) at the last offset (-5.)'
      write(LER,*)'            range: -100. < decay                   '
      write(LER,*)'-comp[]  -- weighting coef. for amplitude      (1.)'
      write(LER,*)'            scaling scalar:  0 <= comp <= 1        '
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'       avoan -N[] -O[] -s[] -e[] -ns[] -ne[] -V'
      write(LER,*)'             -rs[] -re[] -xmin[] -xinc[] -noff []'
      write(LER,*)'             -sfile[] -anl -func[] -amp[] -decay[]'
      write(LER,*)'             -slope[] -icept[] -comp[]'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end
C-----END-OF-FUNCTION-HELP-----



        subroutine cmdln (ntap, otap, sfile, ns, ne, irs, ire, s, e, 
     *                     x_min, x_inc, n_x, 
     *                     name, analysis, verbos, fittype, 
     *                     amptype, decay, compact, apply, a, b, n_err)
        implicit none

c pick up command line arguments 

#include <f77/iounit.h>

C ***** subroutine arguments *******************************************

        character ntap*(*)  ! Input dataset name (Output)
        character otap*(*)  ! Output dataset name (Output)
        integer ns         ! Start trace number (Output)
        integer ne         ! End trace number (Output)
        integer irs        ! Start record number (Output)
        integer ire        ! End record number (Output)
        real s             ! Analysis window start time (ms) (Output)
        real e             ! Analysis window end time (ms) (Output)
        real x_min      ! Minimum source to receiver offset (Output)
        real x_inc      ! Source to receiver offset increment (Output)
        integer n_x        ! Number of source to receiver values (Output)
        character name*(*) ! Program name (Input)
        logical analysis   ! Analysis only flag (Output)
        logical verbos     ! Verbose processing flag (Output)
        character*10 fittype    ! Linear, hyperbolic  or exponential
        character*10 amptype    ! Name of the ampl. type used in fitting
        real decay       ! Decay of the amplitude (%) at the last offset  
        real compact     ! degree of "squeezing" of assigned amps (range ]0,1] )
        logical apply      ! Apply only flag (Output)
        real a             ! Linear slope parameter (Output)
        real b             ! Linear intercept parameter (Output)
        integer n_err      ! error count
 
C ***** Local variables ************************************************
 
        integer argis

        character sfile*(*)  ! Filename for xgraph output


C ***** Initialize *****************************************************

        n_err = 0

C ***** Start **********************************************************

C ----> Amplitude type used in fit
        call argstr ('-amp', amptype, 'abs', 'abs')
        if (amptype .eq. 'ABS') amptype = 'abs'
        if (amptype .eq. 'RMS') amptype = 'rms'
        if (amptype .eq. 'GEOM') amptype = 'geom'

        if ( .not. ( 
     *     amptype.eq.'abs' .or. amptype.eq.'rms' .or. amptype.eq.'geom'
     *  ) ) then
            write (LERR, 1300)
            write (LER,  1300)
 1300       format (/ 'ERROR: option -amp has an illegal value', /
     *      '       the only allowed values are: abs rms geom',/)
            n_err = n_err + 1
        endif

C ----> Analysis only or analysis/apply mode
        analysis = (argis ('-anl') .gt. 0)

C-----> degree of "squeezing" of assigned amps (range [0,1] )
        call argr4 ( '-comp', compact, 1., 1.)
        if ( (compact .lt. 0.) .or. (compact .gt. 1) ) then
            write (LERR, 1002)
            write (LER,  1002)
 1002       format (/,'***ERROR: value of -comp must be within [0,1]',/)
            n_err = n_err + 1
        endif

C-----> Assign the decay of the amplitude 
        call argr4 ( '-decay', decay, -5., -5.)
        if (decay .le. -100.) then
            write (LERR, 1001)
            write (LER,  1001)
 1001       format (/,'***ERROR: decay must be greater than -100.%',/)
            n_err = n_err + 1
        endif

c ----> Analysis window end time
        call argr4 ('-e', e, 0., 0.)
        if (e .lt. 0.) then
            write (LERR, 1017)
            write (LER,  1017)
 1017       format (/ 'ERROR: e cannot be negative'/)
            n_err = n_err + 1
        endif

C ----> Linear or exponential amplitude switch
        call argstr ('-func', fittype, 'lin', 'lin')
        if (.not.((fittype .eq. 'lin') .or. (fittype .eq. 'exp')) ) then
            write (LERR, 1400)
            write (LER,  1400)
 1400       format (/ 'ERROR: option -func has an illegal value', /
     *      '       the only allowed values are: lin exp',/)
            n_err = n_err + 1
        endif

C ----> Prespecified application parameter: intercept term
        call argr4 ('-icept', b, 1., -99999.)

C ----> Number of unique source to receiver offsets
        call argi4 ('-noff', n_x, -99, -99)

c ----> Start and end trace numbers
        call argi4 ( '-ne', ne, 0, 0 )
        call argi4 ( '-ns', ns, 0, 0 )
        if ( ns .gt. ne ) then
            write (LERR, 1004)
            write (LER,  1004)
 1004       format (/ 'ERROR: ne must be larger or equal to ns'/)
            n_err = n_err + 1
        endif

C ----> Input and output trace datasets
        call argstr ( '-N', ntap, ' ', ' ' ) 
        call argstr ( '-O', otap, ' ', ' ' ) 

C ----> Start and end record numbers
        call argi4 ( '-re', ire, 0, 0 )
        call argi4 ( '-rs', irs, 0, 0 )
        if (irs .gt. ire) then
            write (LERR, 1006)
            write (LER,  1006)
 1006       format (/ 'ERROR: re must be larger than rs'/)
            n_err = n_err + 1
        endif

c ----> xgraph file anme
        call argstr ('-sfile', sfile, 'xgraph.xgr', 'xgraph.xgr')

C ----> Prespecified application parameter: slope term
        call argr4 ('-slope', a, 0., -99999.)
        apply = .false.
        if (a .ne. -99999.) then
            apply = .true.
            if (b .eq. -99999.) b = 1.
        endif
 
        if (b .ne. -99999.) then
            apply = .true.
            if (a .eq. -99999.) a = 0.
        endif
 
        if ((a .eq. 0.) .and. (b .eq. 0.)) then
            write (LERR, 1500)
            write (LER, 1500)
 1500       format (/' ERROR: Both the slope and intercept ',
     *                   'terms cannot be zero.')
            n_err = n_err + 1
        endif
 
        if (apply .and. analysis) then
            write (LERR, 1600)
            write (LER, 1600)
 1600       format (/' ERROR: Analysis option is incompatible with',
     *             ' specification of the -slope or -icept arguments.')
            n_err = n_err + 1
        endif

        if (n_x .le. 0) then
            if (.not. apply) then
                write (LERR, 1200)
                write (LER, 1200)
 1200           format (/ 'ERROR: positive noff must be supplied.'/)
                n_err = n_err + 1
            else
                n_x = 1
            endif
        endif
 
c ----> Analysis window start time 
        call argr4 ('-s', s, 0., 0.)
        if (s .lt. 0.) then
            write (LERR, 1007)
            write (LER,  1007)
 1007       format (/ 'ERROR: s cannot be negative'/)
            n_err = n_err + 1
        endif

        if (s .gt. e) then
            write (LERR, 1008)
            write (LER,  1008)
 1008       format (/ 'ERROR: e must be larger or equal to s'/)
            n_err = n_err + 1
        endif

C ----> Verbose output flag
        verbos = (argis ('-V') .gt. 0)

C ----> Source to receiver offset increment (assume units same as DstSgn hdr)
        call argr4 ('-xinc', x_inc, -99., -99.)
        if (x_inc .le. 0.) then
            if (.not. apply) then
                write (LERR, 1100)
                write (LER, 1100)
 1100           format (/ 'ERROR: positive xinc must be supplied.'/)
                n_err = n_err + 1
            else
                x_inc = 999999.
            endif
        endif

C ----> Minimum source to receiver offset (assume units same as DstSgn hdr)
        call argr4 ('-xmin', x_min, -99., -99.)
        if (x_min .lt. 0.) then
            if (.not. apply) then
                write (LERR, 1000)
                write (LER, 1000)
 1000           format (/ 'ERROR: xmin must be supplied ',
     *                                         '( xmin >= 0 ).'/)
                n_err = n_err + 1
            else
                x_min = 0.
            endif
        endif

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

        call xtrarg ( name, ler, .FALSE., .FALSE. )
        call xtrarg ( name, lerr, .FALSE., .TRUE. )
           
        return
        end
C-----END-OF-FUNCTION-CMDLN-----



      subroutine verbal (ntap, otap, sfile, 
     *                   nsamp, nsi, ntrc, nrec, iform, 
     *                   s, e, irs, ire, ns, ne, analysis, verbos,
     *                   amptype, decay, compact, fittype, 
     *                   apply, a, b)
      implicit none

c verbal printout of pertinent program particulars

#include <f77/iounit.h>

      integer nsamp, ntrc
      integer nrec, iform, irs, ire, ns, ne, nsi
      real s, e
      real decay, compact
      character*10 amptype
      logical apply      ! Apply only flag (Input)
      real a             ! Linear slope parameter (Input)
      real b             ! Linear intercept parameter (Input)

      character  ntap*(*), otap*(*), sfile*(*), fittype*(*) 
  

      logical    verbos
      logical    analysis

      write(LERR,*)' '
      write(LERR,*)' Input Line Header Parameters'
      write(LERR,*)' '
      write(LERR,*) ' input data set name   =  ', ntap(1:len(ntap))
      write(LERR,*) ' samples per trace     =  ', nsamp
      write(LERR,*) ' traces per record     =  ', ntrc
      write(LERR,*) ' number of records     =  ', nrec
      write(LERR,*) ' data format           =  ', iform
      write(LERR,*) ' sample interval       =  ', nsi
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Command Line Parameters '
      write(LERR,*)' '
      write(LERR,*) ' output data set name  =  ', otap(1:len(otap))
      write(LERR,*) ' start record          =  ', irs 
      write(LERR,*) ' end record            =  ', ire 
      write(LERR,*) ' start trace           =  ', ns
      write(LERR,*) ' end trace             =  ', ne
      write(LERR,*) ' processing start time =  ', s
      write(LERR,*) ' processing end time   =  ', e
      write(LERR,*) ' xgraph file name      =  ', sfile(1:len(sfile))
      write(LERR,*) ' decay of ampl         =  ', decay 
      write(LERR,*) ' ampl scaling weighting=  ', compact
      write(LERR,*) ' ampl type used in fit =  ', amptype
      write(LERR,*) ' used fit type is      =  ', fittype
      if ( verbos ) write(LERR,*) ' verbose printout requested'
      if (analysis) write(LERR,*) ' analysis only option invoked'
      if (apply)  then
          write(LERR,*) ' Apply only option invoked:'
          write(LERR,*) '     linear slope value = ', a
          write(LERR,*) '     linear intercept value = ', b
      endif
      write(LERR,*)' '
      write(LERR,*)'========================================== '
      write(LERR,*)' '

      return
      end
C-----END-OF-FUNCTION-VERBAL-----




         subroutine  abfit (x, y, ndata, a, b)
         implicit none

#include <f77/iounit.h>

c  routine to linearly fit data with  y = a*x + b

c  input:
c          x,y  -  ndata input points

c output:
c          a,b  -  coefficients of straight line

C -----> Subroutine arguments
         real x(*)        ! X, independent variable (Input)
         real y(*)        ! Y, dependent variable (Input)
         integer ndata    ! Number of data points (X-Ys)
         real a           ! Slope of the linear fit (Output)
         real b           ! Intercept of the linear fit (Output)

C -----> Local variables
         real sx      ! Sum of the of Xs
         real sy      ! Sum of the of Ys
         real sxoss   ! Mean of the Xs
         real ss      ! Number of data points
         real t       ! Difference between the ith X and the mean X
         real st2     ! Sum of the differences squared

         integer i

C **********************************************************************

         sx = 0.               !  initialize sums
         sy = 0.
         st2= 0.
         a  = 0.

         do i = 1, ndata
             sx = sx + x(i)
             sy = sy + y(i)
         enddo

         ss = real (ndata)

         sxoss = sx/ss        ! Mean x

         do i = 1, ndata
             t = x(i) - sxoss
             st2 = st2 + t*t
             a = a + t * y(i)
         enddo

         a = a / st2           !  solve for a & b
         b = (sy - sx * a) / ss

         return
         end
C-----END-OF-FUNCTION-ABFIT-----
      


         subroutine  constrained_fit (x, y, ndata, x1, y1, a, b)
         implicit none

#include <f77/iounit.h>

c  routine to linearly fit data with  y = a*x + b  with the constraint
c  the solution must pass through x1, y1.

C -----> Subroutine arguments
         real x(*)        ! X, independent variable (Input)
         real y(*)        ! Y, dependent variable (Input)
         integer ndata    ! Number of data points (X-Ys)
         real x1          ! X coordinate of constraining point (Input)
         real y1          ! Y coordinate of constraining point (Input)
         real a           ! Slope of the linear fit (Output)
         real b           ! Intercept of the linear fit (Output)

C -----> Local variables
         real xval  ! x value in shifted coordinate system
         real yval  ! y value in shifted coordinate system
         real sxy   ! Sum of the product of x*y  i.e. the dot product
         real sxx   ! Sum of the product of x*x  i.e. the 2nd moment norm

         integer i

C **********************************************************************

         sxy = 0.               !  initialize sums
         sxx = 0

         do i = 1, ndata
             xval = x(i) - x1
             yval = y(i) - y1
             sxy = sxy + yval * xval
             sxx = sxx + xval * xval
         enddo

         a = sxy / sxx
         b = y1 - a * x1

         return
         end
C-----END-OF-FUNCTION-CONSTRAINED_FIT-----


     
      integer function eliminate_0s (n_x, x_min, x_inc, amps, fittype, 
     *                         nonz_x, offsets, y)
      implicit none
C ============================================
C Eliminates zeros and transform data for linear fit:
C INPUT:     n_x, x_min, x_inc, amp(*), fittype
C OUTPUT:    nonz_x, offsets(*), y(*),
C RETURN:    eliminate_0s
C=============================================

#include <f77/iounit.h>

        real x_min      ! Minimum source to receiver offset
        real x_inc      ! Source to receiver offset increment
        integer n_x        ! Number of source to receiver values
        integer ix_x       ! Address of the ith offset
        integer jx_x       ! Address of the jth offset
        integer nonz_x     ! Number of offsets with none zero amplitudes

        character*10 fittype   ! Linear, hyperbolic  or exponential
 
        real amps(*)       ! Absolute/Squared/Log of absolute amplitude
        real offsets(*)    ! Table of actual offset values
        real y(*)          ! Array of amplitudes to fit to y = a*x + b

c       integer eliminate_0s  ! =0 normal completition, >0 error occured
C------------------------------BODY-----------------------------------
        eliminate_0s = 0
 
        jx_x = 1
        if   ( fittype .eq. 'hyp' ) then       ! hyperb amplitude fit
            do ix_x = 1, n_x
                if (amps(ix_x) .gt. 0.0) then
                    offsets(jx_x) = x_min + real (ix_x - 1) * x_inc
                    y(jx_x) = 1.0 / amps(ix_x)
                    jx_x = jx_x + 1
  
                endif
            enddo
        elseif( fittype .eq. 'exp' ) then      ! exp amplitude fit
            do ix_x = 1, n_x
                if (amps(ix_x) .gt. 0.0) then
                    offsets(jx_x) = x_min + real (ix_x - 1) * x_inc
                    y(jx_x) = log( amps(ix_x) )
                    jx_x = jx_x + 1

                endif
            enddo
        elseif( fittype .eq. 'lin' ) then     ! linear amplitude fit
            do ix_x = 1, n_x
                if (amps(ix_x) .gt. 0.0) then
                    offsets(jx_x) = x_min + real (ix_x - 1) * x_inc
                    y(jx_x) = amps(ix_x)
                    jx_x = jx_x + 1
                endif
            enddo
        else
            eliminate_0s = 1
            write (LERR,*)'***Internal ERROR: unrecognized -func value'
            write (LER,*) '***Internal ERROR: unrecognized -func value'
            return
        endif

        nonz_x = jx_x - 1

        if (nonz_x .le. 0) then
            write (LERR, 3100)
            write (LER, 3100)
 3100       format ('AVOAN: Degenerate data.' /
     *              ' No live traces encountered in the',
     *              ' specified range of offsets')
            eliminate_0s = 1
        elseif (nonz_x .le. 1) then
            write (LERR, 3120) y (1)
            write (LER, 3120) y (1)
 3120       format ('AVOAN: Degenerate data.' /
     *              ' Only one live offset.  Check DstSgn header' /
     *              ' The average amplitude was: ', f12.3)
            eliminate_0s = 1
        endif

        return
        end








      SUBROUTINE AVOAN_VSMUL (A, B, C, N)

c-arguments:
      REAL A(1)     ! Input vector
      REAL B        ! input scalar
      REAL C(1)     ! Output vector
      INTEGER N     ! Number of elements is A and C (Input)

c-local variables:
      INTEGER M     ! Local loop index

      DO M = 1, N
          C(M) = A(M) * B
      ENDDO

      RETURN

      END
