C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c dghost reads seismic record data from an input file,
c performs pont source dghosting on the input record and
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c
c-----
c    get machine dependent parameters

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
 
      integer     itr ( SZLNHD )
      integer     lhed( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne

c------
c  static memory allocation
c     complex     data(1024*1024),cop(1024*1024)
c------
c  dynamic memory allocation for big arrays, eg whole records
      integer     head
      complex     data, cop
      pointer     (wkhead, head(1))
      pointer     (datadr, data(1))
      pointer     (copadr, cop (1))
c------
      complex      czero

c------
c    this is necessary for buliding printout files
#include <f77/pid.h>
c------
      integer     recnum, trcnum, static
      real        tri ( SZLNHD ), tpi(SZLNHD), tps(SZLNHD)
      character   ntap * 256, otap * 256, name*80
      logical     verbos, query, heap
      integer     argis
 
c     equivalence ( itr(129), tri (1) )
      equivalence ( itr(  1), lhed(1) )
      data lbytes / 0 /, nbytes / 0 /, name/'DGHOST'/
 
      czero = cmplx (0.,0.)

c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif
 
c-----
c     open printout files
c     this consists of a 2-character identifier, the process id number,
c     and the child process id number:
c     XY.nnnnn.mmmmm
c     this should be unique even for multiple occurences of the same
c     process in a pipeline
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otap,ns,ne,irs,ire,
     1            fh, vel,zd,epsilon,pct,dx,verbos)

c-----
c     get logical unit numbers for input and output of seismic data
c     0 = default stdin
c     1 = default stdout
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'DGHOST: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif

c------
c     to get and/or put LINE header values please use saver/savew
c     since that way code will always be portable to whatever machine
c     we maintain usp on
c     saver/w refer to header words using mnemonics rather than position
c
c     see saver/w manual pages
c------
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

c------
c     save certain parameters
 
c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
 
c-----------
c format values are:
 
c     integer    = 1
c     float      = 2
c     character  = 3
c     2-byte int = 4
c-----------

      call hlhprt (itr, lbytes, name, 80, LERR)

      write(LERR,*)' '
      write(LERR,*)'Global Line Parameters'
      write(LERR,*)' '
      write(LERR,*)'Number samples/trace        = ',nsamp
      write(LERR,*)'Sample interval (ms)        = ',nsi
      write(LERR,*)'Number traces/record        = ',ntrc
      write(LERR,*)'Number records/line         = ',nrec
      write(LERR,*)' '

c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records)
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

c--------------------------------------------------
c  compute sample interval in secs
c  take care of micro secs if necessary

         dt = real (nsi) * unitsc

c--------------------------------------------------
c-----
c     initialize dimensions, etc
c-----
      call dginit (dt,dx,vel,zd,pct,epsilon,ntrc,nsamp,
     1             nsmp4,ntrc4,nsmpe,ntrce,ntro,ift,ifx,d4,t4)


c---------------------------------------------------
c  malloc only space we're going to use
      heap = .true.

c--------------------------
c  note: these don't
c  have to be the same size

      itemi = ntrc * ITRWRD
      itemd = 2 * nsmp4 * ntrc4
      itemc = 2 * nsmpe * ntrce
      maxdim = max(ntrce,nsmpe)
      itemw = 2 * maxdim
      write(LERR,*)'items: ',itemd,itemc,itemw

c  note also SZSMPD is the 
c  size of an item in bytes
c--------------------------

      call galloc (wkhead, itemi*SZSMPD, errcd, abort)
      if (errcd .ne. 0.) heap = .false.
      call galloc (datadr, itemd*SZSMPD, errcd, abort)
      if (errcd .ne. 0.) heap = .false.
      call galloc (copadr, itemc*SZSMPD, errcd, abort)
      if (errcd .ne. 0.) heap = .false.

      if (.not. heap) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemi*SZSMPD,'  bytes'
         write(LERR,*) itemd*SZSMPD,'  bytes'
         write(LERR,*) itemc*SZSMPD,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemi*SZSMPD,'  bytes'
         write(LERR,*) itemd*SZSMPD,'  bytes'
         write(LERR,*) itemc*SZSMPD,'  bytes'
         write(LERR,*)' '
      endif
c---------------------------------------------------


c-----
c     modify line header to reflect actual number of traces output
c-----
      nrecc=ire - irs+1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr=ne-ns+1
      call savew(itr, 'NumTrc', jtr  , LINHED)

c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD
c----------------------
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout                 )

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
      if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  fh,vel,zd,epsilon,pct,dx,ntap,otap)
      end if

c-----
c     initialize dimensions, etc
c-----
c     call cvfill (czero, cop, 2, nsmpe*ntrce)
      call vclr   (cop, 1, itemc)
      call tapers (dt,dx,vel,zd,pct,epsilon,ntrc,nsamp,
     1             nsmp4,ntrc4,nsmpe,ntrce,ntro,ift,ifx,d4,t4,
     2             fh,maxdim,fac,tpi,tps,cop)

      fac = 1./fac


c-----
c     BEGIN PROCESSING
c     read trace, do terrible things to data, write to output file
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin,ntrc,itr)

c-----
c     process desired trace records
c-----
      DO 1000 jj = irs, ire
 
            ic = 0
            call vclr (data,1, itemd)

            do 1001  kk = ns, ne

                  nbytes = 0
                  call rtape( luin, itr, nbytes)
c------
c     if end of data encountered (nbytes=0) then bail out
c     Note:  if you're processing records you might really want
c     to branch to the processing part rather than bailing out
c------
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)
c------
c     for trace header values you can use the I*2 vector
c     rather than saver/savew (which you can use)
c     this is faster in general than saver/w
c------
                  call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        recnum , TRACEHEADER)
                  call saver2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        trcnum , TRACEHEADER)
                  call saver2(lhed,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        dstsgn , TRACEHEADER)
                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        static , TRACEHEADER)

                  if (static .eq. 30000) then
                     call vclr (tri,1,nsamp)
                  endif

c----------------------
c  time/space taper &
c  pack data into array
                  ic = ic + 1
                  ishdr = (ic-1) * ITRWRD
                  call vmov (lhed,1, head(ishdr+1),1,ITRWRD)
                  istrc = (ic-1+ntro) * nsmp4
                  call vmul   (tri,1, tpi,1, tri,1, nsamp)
                  call vsmul  (tri,1, tps(kk), tri,1, nsamp)
                  call cvreal (tri,1, data(istrc+1),2, nsamp)

1001        continue

c-----------------------
c  here's the meat...
c  2-d fft
c  complex multiply deghost
c  operator with data

                  call cfft2d (data, nsmp4, ntrc4, +1)
                  call cvmul  (data, 2, cop, 2, data, 2, ntrc4*nsmp4, 1)
                  call cvsmul (data, 2, fac, data, 2, ntrc4*nsmp4)
                  call cfft2d (data, nsmp4, ntrc4, -1)

c-----------------------

c---------------------
c  extract traces from
c  output array and
c  write output data
            do 1002 kk = 1, jtr

                  ishdr = (kk-1) * ITRWRD
                  call vmov (head(ishdr+1),1,lhed,1,ITRWRD)

                  istrc = (kk-1+ntro) * nsmp4
                  call vmov (data(istrc+1),2, lhed(ITHWP1),1, nsamp)

                  call wrtape (luout, itr, obytes)


 1002             continue
 
                  if(verbos)write(LERR,*)'processed rec ',recnum
 
1000        CONTINUE

c-----
c     close data files
c-----
  999 continue

      call lbclos ( luin )
      call lbclos ( luout )

            write(LERR,*)'end of dghost, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      stop
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'dghost removes the ghosting effect caused by src/rcvr depth'
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute dghost by typing dghost and the of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)         : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)         : output data file name'
        write(LER,*)
     :' -rs[irs]     (default = first)    : start record number'
        write(LER,*)
     :' -re[ire]     (default = last)     : end record number'
        write(LER,*) ' '
        write(LER,*)
     :' -v[vel] (default = none) :  source/receiver velocity (ft,m/s)'
        write(LER,*)
     :' -h[zd]  (default = none) :  source/receiver depth (ft,m)'
        write(LER,*)
     :' -dx[dx] (default = none) :  input trace spacing (ft,m)'
c       write(LER,*)
c    :' -fh[fh] (def = 1/2 Nyq)  :  hi-cut freq (fraction of nyquist)'
        write(LER,*)
     :' -t[pct] (default = 5%)   :  time/space taper'
        write(LER,*) ' '
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :'usage:   dghost -N[ntap] -O[otap] -rs[ns] -re[ne]'
        write(LER,*)
     :'                 -v[vel] -h[zd] -dx[dx] -t[pct] -fh[fh] -V'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,
     1                  fh,vel,zd,epsilon,pct,dx,verbos)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name
c     otap  - C*100    output file name
c     vel   - R*4      src/rcvr velocity
c     zd    - R*4      src/rcvr depth
c  espsilon - R*4      surface rel coef
c     pct   - R*4      taper %
c     dx    - R*4      grp int
c     fh    - R*4      hi-cut freq
c     ns    - I*4      starting trace index
c     ne    - I*4      ending trace index
c     irs   - I*4      starting record index
c     ire   - I*4      ending record index
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      integer     ns, ne, irs, ire
      real        vel, zd, epsilon, pct, dx, fh
      logical     verbos
      integer     argis
 
c-------
c     see manual pages on the argument handler routines
c     for the meanings of these functions
c-------
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argr4 ( '-v', vel, 0.0, 0.0)
            call argr4 ( '-dx', dx, 0.0, 0.0)
            call argr4 ( '-h', zd, 0.0, 0.0)
            call argr4 ( '-r', epsilon, 0.10, 0.10)
            call argr4 ( '-t', pct, 5., 5.)
            call argr4 ( '-fh', fh, 0., 0.)
            verbos =   (argis('-V') .gt. 0)

      if (pct .gt. 50.0) then
         write(LERR,*)'Cannot have % taper > 50: will reset to 50'
         pct = 50.
      endif
      if (vel .eq. 0.0) then
         write(LERR,*)'Must enter surface velocity -- FATAL'
         stop
      endif
      if (dx .eq. 0.0) then
         write(LERR,*)'Using group interval from line header'
      endif
      if (zd .eq. 0.0) then
         write(LERR,*)'Must enter source/receiver depth -- FATAL'
         stop
      endif
 
      return
      end
 
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  fh,vel,zd,epsilon,pct,dx,ntap,otap)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     vel   - R*4      src/rcvr velocity
c     zd    - R*4      src/rcvr depth
c  espsilon - R*4      surface rel coef
c     pct   - R*4      taper %
c     dx    - R*4      grp int
c     fh    - R*4      hi-cut freq
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ntap  - C*100   input file name
c     otap  - C*100   output file name
c-----
#include <f77/iounit.h>

      integer     nsamp, nsi, ntrc, nrec
      character   ntap*(*), otap*(*)
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' source/rcvr velocity  = ',vel
            write(LERR,*) ' source/rcvr depth     = ',zd
            write(LERR,*) ' group interval        = ',dx
            write(LERR,*) ' high cut frequency    = ',fh
            write(LERR,*) ' % time/space taper    = ',pct
            write(LERR,*) ' epsilon               = ',epsilon
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
