C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c dipfk reads seismic record data from an input file (output of fft2da)
c performs pont source dghosting on the input record and
c writes the results to an output file
c the result must be inverse transformed using fft2da -R
c
c
c**********************************************************************c
c
c     declare variables
c
c-----
c    get machine dependent parameters

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
 
      integer     itr ( SZLNHD )
      integer     lhed( SZLNHD )
      real        head( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne

c------
c  static memory allocation
c     complex     data(1024*1024),cop(1024*1024)
c------
c  dynamic memory allocation for big arrays, eg whole records
      real        data
      pointer     (datadr, data(1))
      real        mask, work
      pointer     (maskadr, mask (1))
      pointer     (workadr, work (1))
      integer     itrhdr
      pointer     (iaddrh, itrhdr(1))
c------

c------
c    this is necessary for buliding printout files
#include <f77/pid.h>
c------
      integer     recnum, trcnum, static, mutes(SZLNHD)
      real        tri ( SZLNHD )
      character   ntap * 256, otap * 256, name*80
      character   domain * 2
      logical     verbos, query, heap, pass, kstrip
      integer     argis
 
c     equivalence ( itr(129), tri (1) )
      equivalence ( itr(  1), lhed(1), head(1) )
      data lbytes / 0 /, nbytes / 0 /, name/'DIPFK'/

c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif
 
c-----
c     open printout files
c     this consists of a 2-character identifier, the process id number,
c     and the child process id number:
c     XY.nnnnn.mmmmm
c     this should be unique even for multiple occurences of the same
c     process in a pipeline
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otap,ns,ne,irs,ire,
     1            vel1,vel2, pct,ipow,dx,verbos,pass,
     2            kstrip,kcen,kwid)

c-----
c     get logical unit numbers for input and output of seismic data
c     0 = default stdin
c     1 = default stdout
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'DIPFK: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif

c------
c     to get and/or put LINE header values please use saver/savew
c     since that way code will always be portable to whatever machine
c     we maintain usp on
c     saver/w refer to header words using mnemonics rather than position
c
c     see saver/w manual pages
c------
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      call saver(itr, 'DgTrkS',domain, LINHED)
      if (domain .ne. 'fk' .AND. domain .ne. 'kk') then
         write(LERR,*)'Input data not result of fft2da or fk or fftxy:'
         write(LERR,*)'(the 2-d fft transforms).  This is a fatal error'
         stop
      endif

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('SoPtNm',ifmt_SoPtNm,l_SoPtNm,ln_SoPtNm,TRACEHEADER)

      call hlhprt (itr, lbytes, name, 80, LERR)

      write(LERR,*)' '
      write(LERR,*)'Global Line Parameters'
      write(LERR,*)' '
      write(LERR,*)'Number samples/trace        = ',nsamp
      write(LERR,*)'Sample interval (ms)        = ',nsi
      write(LERR,*)'Number traces/record        = ',ntrc
      write(LERR,*)'Number records/line         = ',nrec
      write(LERR,*)' '

c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records)
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

c--------------------------------------------------
c  compute sample interval in secs
c  take care of micro secs if necessary

         dt = real (nsi) * unitsc

c--------------------------------------------------
c-----
c     initialize dimensions, etc
c-----
      call dginit (dt,dx,ntrc,nsamp,nf,nk)
      nk2 = 2 * nk


c---------------------------------------------------
c  malloc only space we're going to use
      heap = .true.

c--------------------------
c  note: these don't
c  have to be the same size

      itemh = ntrc * ITRWRD
      itemd =     nsamp * ntrc
      itemm =     nsamp * ntrc / 2 + 1
      write(LERR,*)'items: ',itemd

c  note also SZSMPD is the 
c  size of an item in bytes
c--------------------------

      call galloc (iaddrh, itemh*SZSMPD, errcd, abort)
      if (errcd .ne. 0.) heap = .false.
      call galloc (maskadr, itemm*SZSMPD, errcd, abort)
      if (errcd .ne. 0.) heap = .false.
      call galloc (workadr, itemm*SZSMPD, errcd, abort)
      if (errcd .ne. 0.) heap = .false.

      if (.not. heap) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemh*SZSMPD,'  bytes'
         write(LERR,*) itemm*SZSMPD,'  bytes'
         write(LERR,*) itemm*SZSMPD,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemh*SZSMPD,'  bytes'
         write(LERR,*) itemm*SZSMPD,'  bytes'
         write(LERR,*) itemm*SZSMPD,'  bytes'
         write(LERR,*)' '
      endif
c---------------------------------------------------

c-----
c     build mask for f-k domain
c-----
      if (domain .eq. 'fk') then

         call maskit   (dt,dx,vel1,vel2,pct,ipow,pass,
     1                  ntrc,nsamp, nf, nk, nk2, mask, work,
     2                  kstrip, kcen)
      elseif (domain .eq. 'kk') then

         call mask2d   (dt,dx,vel1,vel2,pct,ipow,pass,
     1                  ntrc,nsamp, nf, nk, nk2, mask, work,
     2                  kstrip, kcen)
      endif

      call gfree (workadr)

c---------------------------------------------------
c  malloc only space we're going to use for data
      heap = .true.

      call galloc (datadr, itemd*SZSMPD, errcd, abort)
      if (errcd .ne. 0.) heap = .false.

      if (.not. heap) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemd*SZSMPD,'  bytes'
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemd*SZSMPD,'  bytes'
      endif

c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD
c----------------------
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout                 )

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
            call verbal(nsamp, nsi, ntrc, nrec, iform,pass,
     1                  vel1,vel2,pct,ipow,dx,ntap,otap,
     2                  kstrip,kcen,kwid)

c-----
c     BEGIN PROCESSING
c     read trace, do terrible things to data, write to output file
c-----
c-----
c     pass unwanted records
c-----
      nbytes = obytes
      call recrw (1,irs-1,luin,ntrc,itr,luout, nbytes)
      if (nbytes  .eq. 0) go to 999

c-----
c     process desired trace records
c-----
      DO 1000 jj = irs, ire
 
            ic = 0
            call vclr (data,1, ntrc*nsamp)
            do 1001  kk = 1, ntrc

                  nbytes = 0
                  call rtape( luin, itr, nbytes)
c------
c     if end of data encountered (nbytes=0) then bail out
c     Note:  if you're processing records you might really want
c     to branch to the processing part rather than bailing out
c------
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (itr(ITHWP1), 1, tri, 1, nsamp)
                  call detmut (tri, mutes(kk), nsamp)

c                 dist   = iabs (itr(l_DstSgn))
c                 static = itr(l_StaCor)
c                 recnum = itr(l_RecNum)
c                 trcnum = itr(l_TrcNum)

                  call saver2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        idist  , TRACEHEADER)
                  call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        static , TRACEHEADER)
                  call saver2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        recnum , TRACEHEADER)
                  call saver2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        trcnum , TRACEHEADER)
                  dist = iabs (idist)

                  if (static .eq. 30000) then
                     call vclr (tri,1,nsamp)
                  endif

c----------------------
c  time/space taper &
c  pack data into array
                  ic = ic + 1
                  istrc = (ic-1) * nsamp
                  ishdr = (ic-1) * ITRWRD
                  call vmov (itr,1, itrhdr(ishdr+1),1,ITRWRD)
                  call vmov (tri,1, data(istrc+1),1, nsamp)

1001        continue

c-----------------------
c  here's the meat...
c  2-d fft
c  complex multiply deghost
c  operator with data

                  call notch  (nsamp, ntrc, nf, data, mask)

c-----------------------

c---------------------
c  extract traces from
c  output array and
c  write output data
            do 1002 kk = 1, ntrc

                  ishdr = (kk-1) * ITRWRD
                  istrc = (kk-1) * nsamp
                  call vmov (data(istrc+1),1, tri, 1, nsamp)
                  call resmut (tri, mutes(kk), nsamp)
                  call vmov (tri, 1, itr(ITHWP1),1, nsamp)
                  call vmov (itrhdr(ishdr+1),1,itr,1,ITRWRD)

                  call wrtape (luout, itr, obytes)


 1002             continue
 
                  if(verbos)write(LERR,*)'processed rec ',recnum
 
1000        CONTINUE

c------------------------
c  pass remainder of recs
      nbytes = obytes
      call recrw (ire+1, nrec, luin, ntrc, itr, luout, nbytes)
      if (nbytes .eq. 0) go to 999

c-----
c     close data files
c-----
  999 continue

      call lbclos ( luin )
      call lbclos ( luout )

            write(LERR,*)'end of DIPFK, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      stop
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'dipfk removes the ghosting effect caused by src/rcvr depth'
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute by typing dipfk and the of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)         : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)         : output data file name'
        write(LER,*)
     :' -rs[irs]     (default = first)    : start record number'
        write(LER,*)
     :' -re[ire]     (default = last)     : end record number'
        write(LER,*) ' '
        write(LER,*)
     :' -dx[dx] (default = none) :  input trace spacing (ft,m)'
        write(LER,*)
     :' -p[ipow] (default = 1)   :  filter sides steepness factor'
        write(LER,*)
     :' Fan FIlter Option:'
        write(LER,*)
     :' -vs[vel1] (def = none)   :  start velocity (ft,m/s)'
        write(LER,*)
     :' -ve[vel2] (def = none)   :  end velocity (ft,m/s)'
        write(LER,*)
     :' You can input a single velocity (vel1 or vel2) and -t[] below'
        write(LER,*)
     :' k Strip Option:'
        write(LER,*)
     :' -K  notch out k-values for all temporal frequencies'
        write(LER,*)
     :' -kc[kcen] (def = none)   :  % k-nyquist of notch center'
        write(LER,*)
     :' -t[pct] (default = 25%)  :  velocity notch width % vel1 or vel2'
        write(LER,*)
     :' or      (default = 5%)   :  or % k-nyquist k-notch width'
        write(LER,*) ' '
        write(LER,*)
     :' -P  include on command line to pass within fan (else reject)'
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :'usage:   dipfk -N[ntap] -O[otap] -rs[ns] -re[ne] [-vs[vel1]'
        write(LER,*)
     :'               -ve[vel2] -kc[kcen]] -dx[dx] -t[] -p[] [-P -K -V]'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,
     1                  vel1,vel2,pct,ipow,dx,verbos,pass,
     2                  kstrip,kcen,kwid)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name
c     otap  - C*100    output file name
c     vel   - R*4      src/rcvr velocity
c     pct   - R*4      taper %
c     dx    - R*4      grp int
c     fh    - R*4      hi-cut freq
c     ns    - I*4      starting trace index
c     ne    - I*4      ending trace index
c     irs   - I*4      starting record index
c     ire   - I*4      ending record index
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      integer     ns, ne, irs, ire, ipow
      real        vel1,vel2, pct, dx, kcen, kwid
      logical     verbos, pass, kstrip
      integer     argis
 
c-------
c     see manual pages on the argument handler routines
c     for the meanings of these functions
c-------
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argr4 ( '-vs', vel1, 0.0, 0.0)
            call argr4 ( '-ve', vel2, 0.0, 0.0)
            call argr4 ( '-dx', dx, 0.0, 0.0)
            call argr4 ( '-t', pct, 0.0, 0.0)
            call argi4 ( '-p', ipow, 1, 1)
            call argr4 ( '-kc', kcen, 0.0, 0.0)
            call argr4 ( '-kw', kwid, 0.1, 0.1)
            kstrip =   (argis('-K') .gt. 0)
            pass   =   (argis('-P') .gt. 0)
            verbos =   (argis('-V') .gt. 0)

      ns = 0
      ne = 0

      IF (.not. kstrip) THEN
         if (pct .eq. 0.0) then
            pct = 50.
            write(LERR,*)'Velocity fan filter option:'
            write(LERR,*)'Notch width % defaults to ',pct
         endif
         if (vel1 .eq. 0.0 .AND. vel2 .eq. 0.0) then
            write(LERR,*)'Must enter fan velocities -- FATAL'
            stop
         endif
         if (vel1 .eq. 0.0 .OR. vel2 .eq. 0.0) then
            vel = amax1 (vel1, vel2)
            v1 = vel - pct * vel / 100.
            v2 = vel + pct * vel / 100.
            if     (vel .gt. 0.) then
                   if (v1 .lt. 0.) v1 = +10.0
            elseif (vel .lt. 0.) then
                   if (v1 .gt. 0.) v1 = -10.0
            else
                   write(LERR,*)'Center velocity is zero -- FATAL'
                   stop
            endif
            vel1 = amin1 (v1,v2)
            vel2 = amax1 (v1,v2)
            if (vel1 .lt. 0.0) vel1 = 10.0
         endif
         if (vel1 .eq. 0.0) then
            write(LERR,*)'Lower fan velocity cannot be 0 -- FATAL'
            stop
         endif
         if (vel2 .eq. 0.0) then
            write(LERR,*)'Upper fan velocity cannot be 0 -- FATAL'
            stop
         endif
      ELSE
         if (pct .eq. 0.0) then
            pct = 5.
            write(LERR,*)'k-strip option:'
            write(LERR,*)'Notch width % defaults to ',pct
         endif
      ENDIF

      if (dx .eq. 0.0) then
         write(LERR,*)'Must enter trace spacing ( -dx[] )'
         stop 666
      endif
 
      return
      end
 
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,pass,
     1                  vel1,vel2,pct,ipow,dx,ntap,otap,
     2                  kstrip,kcen,kwid)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     vel   - R*4      src/rcvr velocity
c     pct   - R*4      taper %
c     dx    - R*4      grp int
c     fh    - R*4      hi-cut freq
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ntap  - C*100   input file name
c     otap  - C*100   output file name
c-----
#include <f77/iounit.h>

      integer     nsamp, nsi, ntrc, nrec,ipow
      real        vel1,vel2,dx,pct, kcen, kwid
      character   ntap*(*), otap*(*)
      logical     pass, kstrip
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
      if (kstrip) then
            write(LERR,*) ' % k-nyquist strip    =  ',kcen
      else
            write(LERR,*) ' start velocity filt  = ',vel1
            write(LERR,*) ' end velocity filt    = ',vel2
      endif
            write(LERR,*) ' trace spacing      = ',dx
            write(LERR,*) ' % notch width      = ',pct
            write(LERR,*) ' notch power        = ',ipow
            write(LERR,*) ' apply PASS filter  =  ',pass
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
