C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C**********************************************************************C
C
C     PROGRAM MODULE FILT
C
C**********************************************************************C
C
C FILT READS SEISMIC TRACE DATA FROM AN INPUT FILE,
C APPLIES A USER-SPECIFIED BUTTERWORTH, Ormsby or Bessel  FILTER, AND
C WRITES THE RESULTS TO AN OUTPUT FILE
C
C**********************************************************************C
C
c     Changes:
c
c     November 5, 2001
c         converted all arrays to dynamic memory allocation so that 
c         arbitrarily large number of samples may be filtered.  The Ormsby 
c         option has a limitation of 16384 [2**14] samples due to MathAdv
c         shotcommings [The qtc008 routine was only ever written to fold to
c         2**14 samples...believe it or not].  For long traces it automatically
c         switches over to a convolutional application and warns the user.
c         Also added implicit none and declared all undeclared variables.
c    Garossino
c      
C     DECLARE VARIABLES
C

      implicit none

#include     <save_defs.h>
#include     <f77/iounit.h>
#include     <f77/lhdrsz.h>
#include     <f77/sisdef.h>

c standard USP variables

      integer LUIN, LUOUT, NBYTES, LBYTES
      INTEGER NSAMP, NSI, NTRC, NREC, IFORM, msamp, deci, obytes
      integer argis, jerr, ns ,ne, irs, ire, lbyout, JJ, KK

      real UnitSc

      CHARACTER NAME * 4
      character ntap * 512, otap * 512

      logical verbos

c variables used in dynamic memory allocation

      integer itr, itr_size, abort
      integer errcd1, errcd2, errcd3, errcd4, errcd5, errcd6, errcd7
      integer errcd8, errcd9, errcd10

      real tri, fwt, work, phi, G, tmp, sf, amp, phz

      pointer ( mem_itr, itr(2) )
      pointer ( mem_tri, tri(2) )
      pointer ( mem_fwt, fwt(2) )
      pointer ( mem_work, work(2) )
      pointer ( mem_phi, phi(2) )
      pointer ( mem_G, G(2) )
      pointer ( mem_tmp, tmp(2) )
      pointer ( mem_sf, sf(2) )
      pointer ( mem_amp, amp(2) )
      pointer ( mem_phz, phz(2) )

c local variables

      integer imtop, imbottom, init, lfmax, norder, lsf, nrecc, jtr
      integer nfold, ii, ly, irec, istatic, i, nt, nf, lag, lg, ift
      integer ordfft, nu
      integer ifmt_TrcNum, l_TrcNum, ln_TrcNum
      integer ifmt_RecNum, l_RecNum, ln_RecNum
      integer ifmt_SrcLoc, l_SrcLoc, ln_SrcLoc
      integer ifmt_RecInd, l_RecInd, ln_RecInd
      integer ifmt_DphInd, l_DphInd, ln_DphInd
      integer ifmt_DstSgn, l_DstSgn, ln_DstSgn
      integer ifmt_DstUsg, l_DstUsg, ln_DstUsg
      integer ifmt_SoPtNm, l_SoPtNm, ln_SoPtNm
      integer ifmt_StaCor, l_StaCor, ln_StaCor

      real Fl, Fh, SAMP
      real tmul, fll, flh, fhl, fhh, ex, pow, db, band, wt, prew, dt, df
      real aint, xnorm, sum
      real coefs(2,64), wrk1(3), wrk2(384)

      logical zerop,mute,orms,bias,butt,bess,conv

c initialize variables
 
      data name/'FILT'/
      data lfmax/501/
      data NBYTES/0/
      data LBYTES/0/
      data abort/0/

c-------------------------------------
c  get online help if necessary

      if ( argis('-?') .gt. 0 .or. 
     :     argis('-h') .gt. 0 .or. 
     :     argis('-help') .gt. 0 ) then
         call help()
         stop
      endif
c-------------------------------
c  open printout file
#include  <f77/open.h>
c-------------------------------

C**********************************************************************C
C     READ PROGRAM PARAMETERS FROM COMMAND LINE
C**********************************************************************C

      call cmdln (ntap,otap,tmul,fl,fh,ns,ne,orms,fll,flh,fhl,fhh,
     &     irs,ire,norder,deci,zerop,verbos,mute,bias,
     &     bess,butt,aint,ex,lsf,pow,conv,db,band,wt,prew)

C**********************************************************************C
C     ABORT IF ANY UNUSED ARGUMENTS WERE PRESENT
C**********************************************************************C
      
      call xtrarg(name,ler,.FALSE.,.FALSE.)
      call xtrarg(name,lerr,.FALSE.,.TRUE.)

C**********************************************************************C
C     READ AND UPDATE LINE HEADER,
C     WRITE LINE HEADER, SAVE KEY PARAMETERS.
C**********************************************************************C

      call getln ( luin, ntap, 'r', 0 )
      call getln (luout, otap, 'w', 1 )

C**********************************************************************C
C     initial dynamic allocation of ITR to allow for line header 
c     interaction
C**********************************************************************C

      itr_size = SZLNHD * SZSMPD
      errcd1 = 0
      call galloc ( mem_itr, itr_size, errcd1, abort )
      if (errcd1 .ne. 0) then
         write(LERR,*) 'ERROR: Unable to allocate workspace '
         write(LERR,*) '       ',itr_size,' bytes requested '
         write(LERR,*) 'FATAL'
         write(LER,*) ' '
         write(LER,*) 'FILT:'
         write(LER,*) ' Unable to allocate workspace '
         write(LER,*) '  ',itr_size,' bytes requested '
         write(LER,*) 'FATAL'
         write(LER,*) ' '
         goto 999
      else
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itr_size,' bytes requested '
      endif

      call vclr ( itr, 1, SZLNHD )

C**********************************************************************C
C     read input line header
C**********************************************************************C

      lbytes = 0
      CALL RTAPE ( LUIN, ITR, LBYTES )
      if(lbytes .eq. 0) then
         write(LERR,*)'FILT: no header read on unit ',ntap
         write(LERR,*)'FATAL'
         write(LERR,*)'Check existence of file & rerun'
         stop
      endif

C**********************************************************************C
C     echo processing history to printout
C**********************************************************************C

      CALL HLHprt( ITR, LBYTES, NAME, 4, LERR)

C**********************************************************************C
C     capture global parameters
C**********************************************************************C

      call saver(itr, 'NumSmp', nsamp , LINHED)
      call saver(itr, 'SmpInt', nsi   , LINHED)
      call saver(itr, 'NumTrc', ntrc  , LINHED)
      call saver(itr, 'NumRec', nrec  , LINHED)
      call saver(itr, 'Format', iform , LINHED)
      call saver(itr, 'UnitSc', UnitSc, LINHED)
      if (UnitSc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',UnitSc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          UnitSc = .001
          call savew(itr, 'UnitSc', UnitSc, LINHED)
      endif

C**********************************************************************C
c     create pointers to trace header entries to be used in filt
C**********************************************************************C
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('SoPtNm',ifmt_SoPtNm,l_SoPtNm,ln_SoPtNm,TRACEHEADER)

      msamp = nsamp
      samp = real (nsi) * UnitSc

      if(deci .ne. 1) then
        if ( fh .gt. 5./(samp*deci) ) fh = 4.5/(samp*deci)
      endif

      msamp = ( nsamp + deci - 1 ) / deci

      SAMP = SAMP * tmul
      dt   = SAMP

c ---
c determine time and frequency requirements for fft logic.  For now 
c MathAdv fft routines have an upper limit of 2**14 or 16384 samples
c hard coded into the routine logic.  The folding algorithm only 
c goes far enough to allow this size dataset.  If nt gets bigger than
c 16384 then bail gracefully.
c ---

      nu = ordfft (nsamp)
      nt = 2 ** nu
      nf = nt / 2 + 1

c policeman

      if ( orms .and. nt .gt. 16384 .and. .not. conv ) then

         write(LERR,*)' '
         write(LERR,*)' number of samples in fft: ',nt
         write(LERR,*)' exceeds maximum allowed of 16384'
         write(LERR,*)' a convolutional application  of the ormsby '
         write(LERR,*)' coefficients is being forced'
         write(LERR,*)'WARNING '
         write(LER,*)' '
         write(LER,*)'FILT: '
         write(LER,*)' number of samples in fft: ',nt
         write(LER,*)' exceeds maximum allowed of 16384'
         write(LER,*)' a convolutional application  of the ormsby '
         write(LER,*)' coefficients is being forced'
         write(LER,*)'WARNING '
         write(LER,*)' '
         conv = .true.
      endif
      
c------------------------------
c  check defaults
      call cmdchk (ns,ne,irs,ire,ntrc,nrec)
c------------------------------
      nrecc = ire - irs + 1
      jtr   = ne - ns + 1
      obytes = SZTRHD + SZSMPD * msamp

c---------------------------------------------------
c  adjust line header & write header
      call savew( itr, 'NumTrc', jtr    , LINHED)
      call savew( itr, 'NumRec', nrecc  , LINHED)
      call savew( itr, 'NumSmp', msamp  , LINHED)
      call savew( itr, 'SmpInt',nsi*deci, LINHED)

      call savhlh ( itr, lbytes, lbyout )
      CALL WRTAPE ( LUOUT, ITR, LBYOUT )

c----------------------------
c   printout

      write(LERR,*) ' Values read from command line '
      write(LERR,*) 'Input data set =  ',ntap
      write(LERR,*) 'Output data set = ',otap
      if (orms) then
         write(LERR,*) ' F1 =  ', fll, '  Hz.'
         write(LERR,*) ' F2 =  ', flh, '  Hz.'
         write(LERR,*) ' F3 =  ', fhl, '  Hz.'
         write(LERR,*) ' F4 =  ', fhh, '  Hz.'
         write(LERR,*) ' Filter exponent = ', pow
         if (conv) then
            write(LERR,*) ' Convolutional filter used'
            if (.not.zerop) then
               write(LERR,*) ' Minimum delay filter used'
               write(LERR,*) ' Prewitening     =  ',prew,' (%)'
            endif
         else
            write(LERR,*) ' Frequency domain used'
         endif
      elseif (butt) then
         write(LERR,*) ' FL =  ', fl, '  Hz.'
         write(LERR,*) ' FH =  ', fh, '  Hz.'
         write(LERR,*) ' BWF order = ', norder
         if (zerop)
     1        write(LERR,*) ' Causal filter used'
      elseif (bess) then
         write(LERR,*) ' Bessel intercept=  ',aint
         write(LERR,*) ' Filter length   =  ',lsf
         write(LERR,*) ' Ross weight     =  ',ex
      elseif (bias) then
         write(LERR,*) ' Remove DC bias only from data'
      endif
      write(LERR,*) ' tmul = ',tmul
      write(LERR,*) ' Decimation = ',deci
      write(LERR,*) ' Output samples = ',msamp
      write(LERR,*) ' Traces per record = ',ntrc
      write(LERR,*) ' Records per line =',nrec
      write(LERR,*)' '
      write(LERR,*) 'Values read from input data set line header'
      write(LERR,*) ' # of Samples/Trace =  ', nsamp
      write(LERR,*) ' Sample Interval    =  ', nsi,' (input)'
      write(LERR,*) ' Traces per Record  =  ', ntrc
      write(LERR,*) ' Records per Line   =  ', nrec
      write(LERR,*) ' Format of Data     =  ', iform
      write(LERR,*) ' Unit Scale of Data =  ', UnitSc
      write(LERR,*) ' Do not restore early mute =  ', mute
      
      prew = prew / 100

C**********************************************************************C
C     dynamic memory allocation
C**********************************************************************C


      itr_size = ITRWRD + nsamp
      errcd1 = 0
      call grealloc ( mem_itr, itr_size * SZSMPD, errcd1, abort )

      errcd2 = 0
      errcd3 = 0
      errcd4 = 0
      errcd5 = 0
      errcd6 = 0
      errcd7 = 0
      errcd8 = 0
      errcd9 = 0
      errcd10 = 0
      call galloc ( mem_tri, nt * SZSMPD, errcd2, abort )
      call galloc ( mem_fwt, nt * SZSMPD, errcd3, abort )
      call galloc ( mem_work, 2 * nt * SZSMPD, errcd4, abort )
      call galloc ( mem_phi, nt * SZSMPD, errcd5, abort )
      call galloc ( mem_G, nt * SZSMPD, errcd6, abort )
      call galloc ( mem_tmp, 2 * nt * SZSMPD, errcd7, abort )
      call galloc ( mem_sf, nt * SZSMPD, errcd8, abort )
      call galloc ( mem_amp, nt * SZSMPD, errcd9, abort )
      call galloc ( mem_phz, nt * SZSMPD, errcd10, abort )

      if ( errcd1 .ne. 0 .or. 
     :     errcd2 .ne. 0 .or. 
     :     errcd3 .ne. 0 .or. 
     :     errcd4 .ne. 0 .or. 
     :     errcd5 .ne. 0 .or. 
     :     errcd6 .ne. 0 .or. 
     :     errcd7 .ne. 0 .or. 
     :     errcd8 .ne. 0 .or. 
     :     errcd9 .ne. 0 .or. 
     :     errcd10 .ne. 0 ) then
         write(LERR,*) 'ERROR: Unable to allocate workspace '
         write(LERR,*) '       ',itr_size*SZSMPD,' bytes requested '
         write(LERR,*) '       ',11*nt*SZSMPD,' bytes requested '
         write(LERR,*) '       FATAL'
         write(LER,*) 'DISORT: '
         write(LER,*) ' Unable to allocate workspace '
         write(LER,*) '       ',itr_size*SZSMPD,' bytes requested '
         write(LER,*) '       ',11*nt*SZSMPD,' bytes requested '
         write(LER,*) 'FATAL'
         goto 999
      else
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) '       ',itr_size*SZSMPD,' bytes requested '
         write(LERR,*) '       ',11*nt*SZSMPD,' bytes requested '
      endif

c initialize memory

      call move (0, itr, 0, itr_size*SZSMPD )
      call move (0, tri, 0, nt * SZSMPD )
      call move (0, fwt, 0, nt * SZSMPD )
      call move (0, work, 0, 2 * nt * SZSMPD )
      call move (0, phi, 0, nt * SZSMPD )
      call move (0, G, 0, nt * SZSMPD )
      call move (0, tmp, 0, 2 * nt * SZSMPD )
      call move (0, sf, 0, nt * SZSMPD )
      call move (0, amp, 0, nt * SZSMPD )
      call move (0, phz, 0, nt * SZSMPD )

c-----------------------------------
c  skip to start record
      call recskp(1,irs-1,luin,ntrc,itr)
c-----------------------------------

C**********************************************************************C
C
C     COMPUTE DESIRED FILTER
C
C**********************************************************************C

      if (orms) then

         call fweight ( nsamp, nt, nf, dt, df, fll, flh, fhl,
     1        fhh, fwt, work, tri, phi, G, zerop, verbos, tmp,
     2        samp, LG, conv, db, band, wt, lfmax, lag, prew )

      elseif (butt) then

         call bwcoef ( fl, fh, dt, coefs, xnorm, norder, ift)

      elseif (bess) then

         call vnwt(ex, aint, lsf, sf)

         write(LERR,*)'Filter Weights are'
         write(LERR,111) (sf(i),i=1,lsf)
111      format(5e12.4)

         lag = lsf/2 + 1

      endif

C**********************************************************************C
C
C     READ TRACE, FILTER, WRITE TO OUTPUT FILE
C
C**********************************************************************C

      DO 5001 JJ = irs, ire

c-------------------------------------------------
c  skip to start trace of this record
            call trcskp(jj,1,ns-1,luin,ntrc,itr)
c-------------------------------------------------

           DO 5002 KK = ns, ne

               nbytes = 0
               CALL RTAPE  ( LUIN, ITR, NBYTES           )
               if(nbytes .eq. 0) then
                  write(LERR,*)'Premature EOF on input:'
                  write(LERR,*)'  rec= ',JJ,'  trace= ',KK
                  go to 999
               endif

               call vmov (itr(ITHWP1), 1, tri, 1, nsamp)
               call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                     istatic, TRACEHEADER)
               call saver2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                     irec   , TRACEHEADER)

               IF ( istatic .ne. 30000 ) THEN

c detect mute zones top and bottom ala angle stack data unless not wanted

                  if (.not. mute) then
                     call bd_detmut ( tri, imtop, nsamp,0)
                     call bd_detmut ( tri, imbottom, nsamp,1)

                     imtop = (imtop+deci-1) / deci
                     imbottom = (imbottom+deci-1) / deci
                  endif

                  if (orms .AND. conv) then

                      call fold (LG, G, nsamp, tri, LY, work)

                      do  ii = 1, nsamp
                          tri (ii) = dt * work (ii+lag)
                      enddo

                  elseif (orms .AND. .not.conv) then

                      call trapf (nsamp, nt, nf, dt, df, tri, fwt,
     1                            work, amp, phz, phi, zerop, pow)

                  elseif (butt) then
                     
c set initialization of bworth filter on for each trace.  This is required
c for pathological datasets like mud weights and interval velocity and doesn't
c hurt regular seismic data - pgg

                     init = 1

                     call bwfilt ( tri, amp, wrk1, wrk2, coefs,
     1                            xnorm, norder, nsamp, init, 0)

                     init = 0
                     if(zerop)then
                        call vrvrs (amp,  1, nsamp)
                        call bwfilt ( amp, tri, wrk1, wrk2, coefs,
     1                       xnorm, norder, nsamp, init, 0)
                        call vrvrs (tri,  1, nsamp)
                     else
                        call vmov  (amp(1), 1, tri, 1, nsamp)
                     endif

c apply bessel filter

                  elseif (bess) then

                     call fold (lsf, sf, nsamp, tri, nfold, work)
                     call vmov (work(lag), 1, tri, 1, nsamp)

c remove DC bias only

                  elseif (bias) then

                     sum = 0.
                     do  i = 1, nsamp
                         sum = sum + tri (i)
                     enddo
                     sum = sum / float (nsamp)
                     do  i = 1, nsamp
                         tri (i) = tri (i) - sum
                     enddo

                  endif

                  call decim(tri, nsamp, deci)

c reapply mute zones top and bottom unless otherwise requested

                  if (.not. mute)then
                     call bd_resmut (tri, imtop, msamp, 0)
                     call bd_resmut (tri, imbottom, msamp, 1)
                  endif

               ENDIF

               call vmov (tri, 1, itr(ITHWP1), 1, nsamp)
               CALL WRTAPE ( LUOUT, ITR, OBYTES          )
 5002      CONTINUE

c-------------------------------------------------
c  skip to end trace of this record
            call trcskp(JJ,ne+1,ntrc,luin,ntrc,itr)
c-------------------------------------------------
           if( verbos ) write(LERR,*)'Filtered record ',irec

 5001 CONTINUE

      call lbclos ( luin )
      call lbclos ( luout)
      write(LERR,*)' Normal Termination'
      write(LERR,*)'filt: Normal Termination'
      stop

  999 continue

      call lbclos ( luin )
      call lbclos ( luout)
      write(LERR,*)' Abnormal Termination'
      write(LERR,*)'filt: Abnormal Termination'
      stop
      END
c---------------------
c-----
      subroutine decim(x,n,ndec)
      real      x(*)
      integer   n,ndec
c-----
c      decimate original time series
c      outputing only each ndec'th point
c-----
      j=0
      do 100  i = 1, n, ndec
             j=j+1
             x (j) = x (i)
100   continue

      return
      end

c---------------------------------
c  online help section
c---------------------------------
      subroutine  help
#include  <f77/iounit.h>

      write(LER,*)
     :'**************************************************************'
      write(LER,*)
     :'Run this program by typing filt and the following arguments'
      write(LER,*)
     :' -N[ntap]     (default stdin)         : Input data file'
      write(LER,*)
     :' -O[otap]     (default stdout)        : Output data file'
      write(LER,*)
     :' -ns[ns]      (default = first)       : start trace number'
      write(LER,*)
     :' -ne[ne]      (default = last)        : end trace number'
      write(LER,*)
     :' -rs[irs]     (default = first)       : start record number'
      write(LER,*)
     :' -re[ire]     (default = last)        : end record number'
      write(LER,*)' '
      write(LER,*)
     :'                 Input Filter Frequencies'
      write(LER,*)
     :'Butterworth:'
      write(LER,*)
     :' -fl[flow]    (default =10 Hz)        : Lowcut corner'
      write(LER,*)
     :' -fh[fhigh]   (default =40Hz)         : Highcut corner'
      write(LER,*)
     :' -nor[norder] (default = 2)           : Filter order'
      write(LER,*)
     :'Ormsby:'
      write(LER,*)
     :' -ormsby      Do trapezoidal (4-corner) filter'
      write(LER,*)
     :' -f1[flolo]  (default = 0 Hz)          : Low Lowcut corner'
      write(LER,*)
     :' -f2[flohi]  (default = 0 Hz)          : High Lowcut corner'
      write(LER,*)
     :' -f3[fhilo]  (default = Nyquist Hz)    : Low Highcut corner'
      write(LER,*)
     :' -f4[fhihi]  (default = Nyquist Hz)    : High Highcut corner'
      write(LER,*)
     :' -pow[pow]   (default = 1)             : filter exponent'
      write(LER,*)
     :' -conv       Use convolutional ormsby filter'
      write(LER,*)
     :' -db[db]     (default = 40)            : db reject'
      write(LER,*)
     :' -wt[wt]     (default = 1)             : bessel weight'
      write(LER,*)
     :' -pw[pw]     (default = 1)             : % prewitening (-conv)'
      write(LER,*)
     :'Bessel Filter:'
      write(LER,*)
     :' -i[aint]    (default = .05)           : intercept (notch width)'
      write(LER,*)
     :' -w[ex]    (default = 0)               : ross weight'
      write(LER,*)
     :' -l[lsf]    (default = 31)             : filter length'
      write(LER,*)' '
      write(LER,*)
     :' -C           For minimum phase output, else zero phase output'
      write(LER,*)
     :' '
      write(LER,*)
     :' -bias        Remove DC Bias Only'
      write(LER,*)
     :' -tmul[tmul]  (default = 1)           : time multiplier'
      write(LER,*)
     :' -d[idec]     (default = 1)           : Decimation factor'
      write(LER,*)
     :' -M                     Do not restore early mute'
      write(LER,*)' '
      write(LER,*)
     :'Usage:'
      write(LER,*)
     :' filt -N[] -O[] -ns[] -ne[] -rs[] -re[] -tmul[] [-C -M -V]'
      write(LER,*)
     :'      [ -d[] -fl[] -fh[] -nor[] ]'
      write(LER,*)
     :'      [ -f1[] -f2[] -f3[] -f4[] -ormsby -conv -db[] -wt[] -pw[]]'
      write(LER,*)
     :'      [ -i[] -l[] -w[] -bess ]'
      write(LER,*)
     :'      [ -bias ]'

      return
      end

c-----
c     get command arguments
c
c     ntap  - C*512  input file name
c     otap  - C*512  output file name
c    tmul   - R      time multiplier
c    fl,h   - R      corner frqs
c     ns    - I      start trace
c     ne    - I      stop end trace
c    irs    - I      start record
c    ire    - I      stop end record
c  norder   - I      butterworth order
c    deci   - I      decimation factor
c   zerop   - L      causal/acausal
c    verbos - L      verbose output or not
c-----

      subroutine cmdln ( ntap, otap, tmul, fl, fh, ns, ne, orms, fll, 
     :     flh, fhl ,fhh, irs, ire, norder, deci, zerop, verbos, mute, 
     :     bias, bess, butt, aint, ex, lsf, pow, conv, db, band, wt, 
     :     prew)

      implicit none

#include    <f77/iounit.h>

c declare variables passed from calling routine

      integer argis, ns, ne, irs, ire, norder, deci, lsf

      character  ntap*(*), otap*(*)

      real fl, fh, tmul, prew, fll, flh, fhl, fhh, db, band, wt, aint
      real ex, pow

      logical zerop, verbos, mute, orms, bias, butt, bess, conv

c parse command line

      bias   = ( argis( '-bias' ) .gt. 0   )
      bess   = ( argis( '-B' ) .gt. 0   )

      conv   = ( argis( '-conv' ) .gt. 0   )
      if(argis('-C').gt.0)then
         zerop = .false.
      else
         zerop = .true.
      endif

      call argr4 ( '-db', db , 40.0 , 40.0  )
      call argi4 ( '-d',deci,1,1)

      call argr4 ( '-fh', fh , 40.0 , 40.0  )
      call argr4 ( '-fl', fl , 10.0 , 10.0  )
      call argr4 ( '-f1', fll , 0.0 , 0.0  )
      call argr4 ( '-f2', flh , 0.0 , 0.0  )
      call argr4 ( '-f3', fhl , 0.0 , 0.0  )
      call argr4 ( '-f4', fhh , 0.0 , 0.0  )

      call argr4( '-i', aint, .05, .05 )

      call argi4( '-l', lsf, 3, 3 )

      mute   = ( argis( '-M' ) .gt. 0   )

      call argi4 ( '-nor', norder, 2 , 2 )
      call argi4 ( '-ne', ne ,   0  ,  0    )
      call argi4 ( '-ns', ns ,   0  ,  0    )
      call argstr( '-N', ntap, ' ', ' ' )

      orms   = ( argis( '-ormsby' ) .gt. 0   )
      call argstr( '-O', otap, ' ', ' ' )

      call argr4 ( '-pow', pow, 1.0, 1.0)
      call argr4 ( '-pw', prew , 5.0 , 5.0  )

      call argi4 ( '-re', ire ,   0  ,  0    )
      call argi4 ( '-rs', irs ,   0  ,  0    )
      call argr4 ( '-r', band , 10.0 , 10.0  )

      call argr4( '-tmul',tmul,1.0,1.0)

      verbos = ( argis( '-V' ) .gt. 0   )

      call argr4 ( '-wt', wt , 1.0 , 1.0  )
      call argr4( '-w', ex, 0., 0. )

      if (.not. bias .AND. .not. orms .AND. .not. bess) then
         butt = .true.
      else
         butt = .false.
      endif

c policemen         

      if (bias .and. (deci.ne.1)) then
         write(LERR,*)' '
         write(LERR,*)'WARNING:'
         write(LERR,*)'Cannot decimate data and remove DC bias. You'
         write(LERR,*)'must specify bandpass filter stops for this'
         write(LERR,*)'option (either butterworth or ormsby).'
         write(LERR,*)' '
      endif

      if (butt .AND. mod(norder,2) .ne. 0) then
         write(LERR,*)'WARNING:'
         write(LERR,*)'Butterworth order must be a factor of 2'
         norder = norder + 1
         write(LERR,*)'Resetting to ',norder
      endif

      if (butt .AND. norder .gt. 16) then
         write(LERR,*)'WARNING:'
         write(LERR,*)'Butterworth order must be < 16'
         norder = 16
      endif
      
      if (bias) mute = .true.
      
      return
      end
