C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c     program module fitter
c
c**********************************************************************c
c
c fitter reads seismic trace data from an input file,
c and fits a user specified function to the data
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c
#include     <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      parameter   (mmax = 2500)

      integer     itr ( SZLNHD )
      integer     lhed( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform
      integer     luin , luout, lbytes, nbytes, lbyout,lbyte
      integer     irs,ire,ns,ne
#include <f77/pid.h>
      integer     recnum, trcnum, static, poly
      real        tri ( SZLNHD )
      real        sig ( SZLNHD ), time ( SZLNHD )
      real        covar (mmax), A(mmax)
      integer     lista(mmax)
      character   ntap * 256, otap * 256, name*6
      logical     verbos, query
      integer     argis, itype

c     equivalence ( itr(129), tri (1) )
      equivalence ( itr(  1), lhed(1) )
      data lbytes / 0 /, nbytes / 0 /, name/'FITTER'/

c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif

c-----
c     open printout files
c-----
#include <f77/open.h>

      call gcmdln(ntap,otap,ns,ne,irs,ire,verbos,
     1            poly, itype)

      if (poly*poly .gt. mmax) then
         write(LERR,*)'Order of polynomial ',poly,' too high'
         amax = mmax
         imax = sqrt(amax)
         write(LERR,*)'Cannot be greater than ',imax
         stop
      endif
c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)
c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape   ( luin, itr, lbyte)
      if(lbyte .eq. 0) then
         write(LERR,*)'FITTER: no header read from unit ',luin
         write(LERR,*)'FATAL'
         stop
      endif
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
c-----
c     ensure that command line values are compatible with data set
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
c-----
c     modify line header to reflect actual number of traces output
c-----
      call hlhprt (itr, lbyte, name, 6, LERR)
      nrecc=ire - irs+1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr=ne-ns+1
      call savew(itr, 'NumTrc', jtr  , LINHED)
      call savhlh(itr,lbyte,lbyout)
      call wrtape ( luout, itr, lbyout                 )

c-----
c     generate vector of 2-way times
c-----
      dt = real (nsi) * unitsc

      do  10  i = 1, nsamp

          time (i) = float ( i ) * dt
          sig (i) = 1.0
10    continue

c-----
c     find limits
c-----
      call maxmgv (time, 1, xmax, indx, nsamp)
      call minmgv (time, 1, xmin, indx, nsamp)

c-----
c     generate list of parameters
c-----
      do  11  i = 1, poly

          lista (i) = i
11    continue

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
      if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  ntap,otap,poly,itype)
      endif
c-----
c     BEGIN PROCESSING
c     read trace, do agc, write to output file
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin,ntrc,itr)
c-----
c     process desired trace records
c-----
      DO  1000  JJ = irs, ire

c----------------
c  skip to start
c  of record
            call trcskp(jj,1,ns-1,luin,ntrc,itr)
c----------------

            do  1001  KK = ns, ne

                  nbytes = 0
                  call rtape ( luin, itr, nbytes)
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (itr(ITHWP1), 1, tri, 1, nsamp)

c                 static = itr(125)
c                 recnum = itr(106)
c                 trcnum = itr(107)
                  call saver2(itr,ifmt_StaCor,l_StaCor,
     1                        ln_StaCor, static    , TRACEHEADER)
                  call saver2(itr,ifmt_RecNum,l_RecNum,
     1                        ln_RecNum, recnum    , TRACEHEADER)
                  call saver2(itr,ifmt_TrcNum,l_TrcNum,
     1                        ln_TrcNum, trcnum    , TRACEHEADER)

                  if(static .ne. 30000)then

                        if (poly .ne. 0) then
                           call lfit (time, tri, sig, nsamp, A, poly,
     1                                lista, poly, covar, poly, chisq,
     2                                xmin, xmax, itype)
                        endif
                  else
                        call vclr(tri,1,nsamp)
                  endif

                  call vmov  (tri, 1, itr(ITHWP1), 1, nsamp)
                  call wrtape( luout, itr, nbytes)

 1001             continue

c----------------
c  skip to end of
c  current record
            call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
c----------------
            if(verbos) write(LERR,*)'fitter:  ri ',recnum

 1000       CONTINUE
c-----
c     close data files
c-----
  999 continue
      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'end of fitter, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end

      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
      write(LER,*)
     :'execute fitter by typing fitter and a list of program parameters'
      write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
      write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
      write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)       : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)       : output data file name'
      write(LER,*)' '
        write(LER,*)
     :' -ns[ns]      (default = first)  : start trace number'
        write(LER,*)
     :' -ne[ne]      (default = last)   : end trace number'
        write(LER,*)
     :' -rs[irs]      (default = first) : start record number'
        write(LER,*)
     :' -ne[ire]      (default = last)  : end record number'
      write(LER,*)' '
        write(LER,*)
     :' -p [poly]    (default = 0)      : fit nth order poly to data'
        write(LER,*)
     :' -f [itype]   (default = 0)      : type of fitting function'
        write(LER,*)
     :'     itype = 0 : polynomial in X (a0 + a1*X a2*X**2 + ...)'
        write(LER,*)
     :'     itype = 1 : exponentials (exp(X), exp(2*X), ...)'
        write(LER,*)
     :'     itype = 2 : gaussians (exp(X**2), exp(2*X**2), ...)'
        write(LER,*)
     :'     itype = 3 : sine (sin[pi*X/xmx], xmx = max X value'
        write(LER,*)
     :'     itype = 4 : cosine (cos[pi*X/xmx], xmx = max X value'
        write(LER,*)
     :'     itype = 5 : sinc (sin[pi*X/X]'
        write(LER,*)
     :'     itype = 6 : natural logs (log(x), log(X)**2), ...'
        write(LER,*)
     :'     itype = 7 : poly in 1/[xmx-X], xmx = max X value'
        write(LER,*)
     :'     itype = 8 : poly in 1/[X-xmn], xmn = min X value'
        write(LER,*)
     :'     itype = 9 : negative exps (exp(-X), exp(-2*X), ...)'
      write(LER,*)' '
        write(LER,*)
     :' -V                              : verbose printout'
      write(LER,*)' '
         write(LER,*)
     :'usage:   fitter -N[ntap] -O[otap] -p[poly] -f[itype]'
         write(LER,*)
     :'               -ns[ns] -ne[ne] -rs[irs] -re[ire] -V'
         write(LER,*)
     :'***************************************************************'
      return
      end

      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,verbos,
     1            poly, itype)
c-----
c     get command arguments
c
c     ntap  - c*100     input file name
c     otap  - c*100     output file name
c     ns    - i*4 starting trace index
c     ne    - i*4 ending trace index
c     irs   - i*4 starting record index
c     ire   - i*4 ending record index
c     poly    i*4 fit nth order polynomial to gain curve
c     itype   i*4 type of function to fit
c     verbos      - L   verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      integer     ns, ne, irs, ire
      logical     verbos
      integer     argis, itype

            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argi4( '-p', poly, 0, 0 )
            call argi4( '-f', itype, 0, 0 )
            verbos = ( argis( '-V' ) .gt. 0 )

      return
      end

      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1            ntap,otap,poly,itype)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4 number of samples in trace
c     nsi   - I*4 sample interval in ms
c     ntrc  - I*4 traces per record
c     nrec  - I*4 number of records per line
c     iform - I*4 format of data
c     poly    i*4 fit nth order polynomial to gain curve
c     itype   i*4 type of function to fit
c     ntap  - C*100     input file name
c     otap  - C*100     output file name
c-----
#include <f77/iounit.h>
      integer     nsamp, nsi, ntrc, nrec, iform, poly
      integer     itype
      character   ntap*(*), otap*(*)

            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' Order of polynomial to fit to data= ',poly
            write(LERR,*)' Function type= ', itype
            write(LERR,*)' '
            write(LERR,*)' '

      return
      end

