C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************
c FKMOD - 2D reflectivity code - USP version of DELPHI KXMOD 
c         fk amp/phase output compatible with USP FFT2DA
c**********************************************************************
#include <localsys.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h>

	character namin*255,namout*255,name*5
	logical verbos
	integer err,abort,smul,imul,elastic,argis
	integer lh(SZLNHD),buf(1)
	real tr(1),pv(1),sv(1),dns(1),thk(1)
      complex dat(1),p(1),p0tmp(1),ptmp(1)
      complex ru(1),rl(1),tu(1),td(1),wu(1)

	pointer (pbuf,buf)
	pointer (ptr,tr),(ppv,pv),(psv,sv),(pdns,dns),(pthk,thk)
	pointer (pdat,dat),(pp,p),(pp0tmp,p0tmp),(pptmp,ptmp)
	pointer (pru,ru),(prl,rl),(ptu,tu),(ptd,td),(pwu,wu)

	data verbos/.false./				!initialize variables
	data abort/1/
	data name/"FKMOD"/

      if ( argis ( '-?' ) .gt. 0 .or.
     :     argis ( '-h' ) .gt. 0 .or.		!command line help
     :     argis ( '-help' ) .gt. 0 ) then
       call help()
       stop
      endif

	call cmdln(namin,namout,verbos)		!get i/o filenames

	if (namin .ne. ' ') then
	  lui=1
	  open(lui,file=namin)				!open model input
	else
	  lui = LIN
	endif
	read(lui,*)nlyr,nt,dt,nx,dx,alpha,cmin,smul,imul

      pi=3.141592654
      nfq=nt/2
      df=1/(nt*dt)
	alp=alpha*df
	nkx=nx/2+1
	dkx=(2*pi)/(nx*dx)
	nly1=nlyr-1
	elastic=0
	
	nbyt=nlyr*SZSMPD				!model memory allocation
	call galloc(ppv,nbyt,err,abort)
	call galloc(psv,nbyt,err,abort)
	call galloc(pdns,nbyt,err,abort)
	nbyt=(nly1)*SZSMPD
	call galloc(pthk,nbyt,err,abort)

	do i=1,nly1					!get model
	 read(lui,*)pv(i),sv(i),dns(i),thk(i)
	 if(sv(i) .ne. 0)elastic=1
	enddo
	read(lui,*)pv(nlyr),sv(nlyr),dns(nlyr)
	 if(sv(i) .ne. 0)elastic=1

#include <f77/open.h>
	if(verbos)then 
	 write(LERR,*)'input model filename',namin
	 write(LERR,*)'output data filename',namout
	 write(LERR,*)'number of layers',nlyr
	 write(LERR,*)'number of samples',nt
	 write(LERR,*)'time increment',dt
	 write(LERR,*)'number of traces',nx
	 write(LERR,*)'trace increment',dx
	 write(LERR,*)'anti-wrap exponential time taper',alpha
	 write(LERR,*)'min. velocity to evaluate fk spectrum',cmin
	 write(LERR,*)'surface multiples (0=no)',smul
	 write(LERR,*)'internal multiples (0=no)',imul
	 write(LERR,*)'elastic response (0=no)',elastic
	 write(LERR,*)' '
	 write(LERR,*)'number of frequencies',nfq
	 write(LERR,*)'frequency increment',df
	 write(LERR,*)'number of wavenumbers',nkx
	 write(LERR,*)'wavenumber increment',dkx
	 write(LERR,*)'exp gain for ttothen',alp
	 write(LERR,*)' '
	 write(LERR,*)'model:'
	 write(LERR,*)' '
	 do i=1,nly1
	  write(LERR,*)i,pv(i),sv(i),dns(i),thk(i)
	 enddo
	 write(LERR,*)nlyr,pv(nlyr),sv(nlyr),dns(nlyr),'    inf.'
	endif

      if(elastic.eq.0)then				!# output records
       ncomp=1
      else
       ncomp=4
      endif

	nbyt=nt*SZSMPD+SZTRHD			!fkmod memory allocation
	call galloc(pbuf,nbyt,err,abort)
	nbyt=nt*SZSMPD
	call galloc(ptr,nbyt,err,abort)
	nbyt=2*ncomp*nkx*SZSMPD
	call galloc(pp,nbyt,err,abort)
	call galloc(pp0tmp,nbyt,err,abort)
	call galloc(pptmp,nbyt,err,abort)
	call galloc(pru,nbyt,err,abort)
	call galloc(prl,nbyt,err,abort)
	call galloc(ptu,nbyt,err,abort)
	call galloc(ptd,nbyt,err,abort)
	call galloc(pwu,nbyt,err,abort)
	nbyt=2*ncomp*nx*nfq*SZSMPD
	call galloc(pdat,nbyt,err,abort)

	call savew(lh,'NumSmp',nt,0)		!define lineheader entries
	call savew(lh,'NumTrc',nx,0)
	call savew(lh,'NumRec',ncomp,0)
	call savew(lh,'IndAdj',nx,0)		!for fft2da
	call savew(lh,'OrNSMP',nt,0)		!    "
	call savew(lh,'DgTrkS','fk',0)		!    "
	call savew(lh,'SmpInt',1,0)
	call savew(lh,'UnitSc',dt,0)
	call savew(lh,'Format',3,0)
	call savew(lh,'HlhEnt',0,0)
	call savew(lh,'HlhByt',2*SZHFWD,0)
	call savhlh(lh,HSTOFF,lbyout)
	call getln(luo,namout,'w',1)		!open output
	call wrtape(luo,lh,lbyout)			!write lineheader


c**********************************************************************
c     calculation of response in kx-w domain
c***********************************************************************

	write(LER,*)'modelling'
c modeling for each frequency
      do 1100 ifq=1,nfq
       freq=(ifq-1)*df

	 if (cmin .gt. 0.)then			!limits evaluation of fk
	  ikxm0=min(nint(freq*nx*dx/cmin)+1,nkx)	!spectrum to ap. vel. > cmin
	 else	
	 ikxm0=nkx
	 endif		

c initialize response to 0
       do i=1,ncomp*nkx
	  p(i)=0.
       enddo

c loop for all depth layers
       do 1000 ilyr=nly1,1,-1

c define velocities and densities at this interface
        cpup=pv(ilyr)
        csup=sv(ilyr)
        rhoup=dns(ilyr)
        deltaz=thk(ilyr)
        cplow=pv(ilyr+1)
        cslow=sv(ilyr+1)
        rholow=dns(ilyr+1)
	  ikxm=ikxm0

c generate operators at current interface
        if(elastic.eq.0)then
         call kxacoper(cpup,cplow,rhoup,rholow,deltaz,freq,dkx,nkx,
     &                                     alp,ikxm,ru,rl,td,tu,wu)
        else
         call kxeloper(cpup,cplow,csup,cslow,rhoup,rholow,deltaz,
     &                       freq,dkx,nkx,alp,ikxm,ru,rl,td,tu,wu)
        endif

c generate multiples at current interface: p=p/(1-rl*p)
        if(imul.gt.0 .and. ilyr.lt.nly1)then
         if(elastic.eq.0)then
          do ikx=1,ikxm
           p(ikx)=p(ikx)/(1-rl(ikx)*p(ikx))
          enddo
         else
	    fac=-1.
          call cmatcopy2by2(p,nkx,p0tmp,nkx,ikxm)
          call cmatmul2by2(rl,nkx,p,nkx,ptmp,nkx,ikxm)
          call cmataffn2by2(ptmp,nkx,ikxm,fac)
          call cmatinv2by2(ptmp,nkx,ikxm)
          call cmatmul2by2(p0tmp,nkx,ptmp,nkx,p,nkx,ikxm)
         endif
        endif

c model primary response: p=wu[tu*p*td+ru]wu
        if(elastic.eq.0)then
         do ikx=1,ikxm
          p(ikx)=wu(ikx)*(tu(ikx)*p(ikx)*td(ikx)+ru(ikx))*wu(ikx)
         enddo
         do ikx=ikxm+1,nkx
          p(ikx)=0.
         enddo
        else 
         call cmatmul2by2(p,nkx,td,nkx,ptmp,nkx,ikxm)
         call cmatmul2by2(tu,nkx,ptmp,nkx,p,nkx,ikxm)
         call cmatadd2by2(p,nkx,ru,nkx,p,nkx,ikxm)
         call cmatmul2by2(p,nkx,wu,nkx,ptmp,nkx,ikxm)
         call cmatmul2by2(wu,nkx,ptmp,nkx,p,nkx,ikxm)
         call cmatzero2by2(p,nkx,ikxm) 
        endif

1000   continue

c generate multiples at free surface: p=p/(1+p)
       if(smul.gt.0)then
        if(elastic.eq.0)then
         do ikx=1,ikxm
          p(ikx)=p(ikx)/(1+p(ikx))
         enddo
        else
	   fac=1.
         call cmatcopy2by2(p,nkx,p0tmp,nkx,ikxm)
         call cmatcopy2by2(p,nkx,ptmp,nkx,ikxm)
         call cmataffn2by2(ptmp,nkx,ikxm,fac)
         call cmatinv2by2(ptmp,nkx,ikxm)
         call cmatmul2by2(p0tmp,nkx,ptmp,nkx,p,nkx,ikxm)
        endif
       endif

c store results for this frequency
	call stordat(p,dat,nfq,nkx,ncomp,ifq)

1100  continue

	write(LER,*)'output'
c output fk amplitude/phase in format compatible with USP FFT2DA
	call out_usp(dat,tr,buf,ncomp,nx,nfq,luo)
	call lbclos(luo)

c***********************************************************************
c end of program fkmod
c***********************************************************************
9999  end


C++++++++++++++++++NEW SUBROUTINES++++++++++++++++++++++++++++++++++++++

      subroutine cmatzero2by2(mat,ld,n)

      integer ld,n
      complex mat(ld,2,2)

      integer i,j,l

      do j=1,2
         do i=1,2
            do l=n+1,ld
               mat(l,i,j)=0.
            enddo
         enddo
      enddo

	return
	end

c***********************************************************************
	subroutine stordat(p,dat,nfq,nkx,ncomp,i)
 
	complex dat(nfq,2*nkx-2,ncomp),p(nkx,ncomp)

	do k=1,ncomp
	 isym=1
	 if(k .eq. 2 .or. k .eq. 3)isym=-1
	 do j=1,nkx-1
 	  dat(i,j,k)=isym*p(nkx+1-j,k)
 	  dat(i,nkx-1+j,k)=p(j,k)
	 enddo
	enddo

	return
	end

c***********************************************************************
	subroutine out_usp(dat,tr,buf,ncomp,nx,nfq,luo)

c output fk data in form compatible with USP FFT2DA

#include <f77/lhdrsz.h>

	complex dat(nfq,nx,*)
	real tr(*)
	integer buf(*)

	nt=2*nfq
	nbyt=SZTRHD+SZSMPD*nt
	do k=1,ncomp
	 ish=-1				!x-shift by nx*dx/2
	 do j=1,nx
	  ia=nfq+1				!amplitudes in first nfq samples
	  ip=nt+1				!phases in last nfq samples
	  ish=-ish
	  do i=1,nfq
	   ia=ia-1				!low freqs at large samples
	   ip=ip-1
	   dat(i,j,k)=ish*dat(i,j,k)
	   tr(ia)=cabs(dat(i,j,k))
         if(tr(ia) .eq. 0.) then
	    tr(ip)=0.
         else
	    tr(ip)=atan2(aimag(dat(i,j,k)),real(dat(i,j,k)))
         endif
	  enddo
	  call savew(buf,'TrcNum',j,1)
	  call savew(buf,'RecNum',k,1)
	  call vmov(tr(1),1,buf(ITHWP1),1,nt)	
	  call wrtape(luo,buf,nbyt)
	 enddo
	enddo

	return
	end
	
c***********************************************************************
      subroutine help()				!online help 

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)'Command Line Arguments for: FKMOD'
      write(LER,*)' '
      write(LER,*)' Input..................................... (def)'
      write(LER,*)' '
      write(LER,*)' -N[]    -- 1D model                      (stdin)'
      write(LER,*)' -O[]    -- shot record                  (stdout)'
      write(LER,*)' -V      -- verbose                          (no)'
      write(LER,*)' '
	write(LER,*)' fkmod -N[] -O[] -V'
      write(LER,*)' '
      write(LER,*)'input file format:'
      write(LER,*)' '
	write(LER,*)' nlyr,nt,dt,nx,dx,alpha,cmin,smul,imul'
	write(LER,*)' Vp(1),Vs(1),rho(1),thick(1)'
	write(LER,*)' .   .   .   .   .   .   .   .'
	write(LER,*)' .   .   .   .   .   .   .   .'
	write(LER,*)' Vp(i),Vs(i),rho(i),thick(i)'
	write(LER,*)' .   .   .   .   .   .   .   .'
	write(LER,*)' .   .   .   .   .   .   .   .'
	write(LER,*)' Vp(nlyr),Vs(nlyr),rho(nlyr)'
      write(LER,*)' '
	write(LER,*)'where:'
      write(LER,*)' '
      write(LER,*)' nlyr = #layers incl. basal halfspace'
      write(LER,*)' nt = number of time samples (pwr. of 2)'
      write(LER,*)' dt = time sample interval'
      write(LER,*)' nx = number of space samples (pwr. of 2)'
      write(LER,*)' dx = space sample interval'
      write(LER,*)' alpha = anti-wrap exp. time taper'
      write(LER,*)' cmin = min. velocity to evaluate fk spectrum'
      write(LER,*)' smul = 0 for no free surface multiples'
      write(LER,*)' imul = 0 for no internal multiples'
      write(LER,*)' '
	write(LER,*)' note: if all Vs() = 0, do acoustic calculation'
      write(LER,*)' '
      write(LER,*)'For a more detailed description of these parameters'
      write(LER,*)'see the online man page using uman, xman or xuspman'
      write(LER,*)' '
      write(LER,*)'===================================================='

      return
      end

c***********************************************************************
	subroutine cmdln(namin,namout,verbos)

#include <f77/iounit.h>

	character  namin*(*),namout*(*)
	logical verbos
	integer argis

	call argstr ( '-N', namin, ' ', ' ' )
	call argstr ( '-O', namout, ' ', ' ' )
	verbos=(argis( '-V' ) .gt. 0)
	return
	end

C+++++++++++++++DELPHI SUBROUTINES (modified)+++++++++++++++++++++++++++

Cdoc********************************************************************
C* NAME:   kxacoper
C*
C* DESCRIPTION:
C*         Calculation of acoustic operators for refl/trans/propagation
C*         in kx domain for a certain frequency
C*
C* USAGE:  call kxacoper(cup,clow,rhoup,rholow,deltaz,freq,dkx,nkx,
C*        &                                alp,ikxm,ru,rl,td,tu,wu)
C*
C* INPUT:  variable type description
C*         -------- ---- -----------
C*         cup       R   velocity above reflector
C*         clow      R   velocity below reflector
C*         rhoup     R   density above reflector
C*         rholow    R   density below reflector
C*         deltaz    R   thickness of layer above reflector
C*         freq      R   frequency of operators to be calculated
C*         dkx       R   kx-sampling of operators
C*         nkx       I   number of kx values
C*         alp       R   laplace parameter for frequency
C*         ikxm      I   max. wavenumber for operator evaluation
C*                       (based on specified min. velocity)
C*
C* OUTPUT: variable type description
C*         -------- ---- -----------
C*         ikxm      I   maximum wavenumber of operator evaluation
C*                       (based on evanescence or min. velocity criteria)
C*         ru        C   reflectivity operator in kx domain from above
C*         rl        C   reflectivity operator in kx domain from below
C*         td        C   downward transmission operator in kx domain
C*         tu        C   upward transmission operator in kx domain
C*         wu        C   phase-shift operator in kx domain upper layer
C*
C* AUTHOR: Eric Verschuur (eric@delphi.tn.tudelft.nl)
C*
C*         eric@delphi.tn.tudelft.nl
C*         Delft University of Technology
C*         Laboratory of Seismics and Acoustics
C*         P.O. Box 5046
C*         2600 GA Delft
C*         the Netherlands
C*
C* REVISION HISTORY:
C*         Version Date     Author   Comment
C*         1.0     94/07/20 Eric V.  Initial version
C*                 98/05    Nekut
C*
Cdoc********************************************************************

      subroutine kxacoper(cup,clow,rhoup,rholow,deltaz,freq,dkx,nkx,
     &                                      alp,ikxm,ru,rl,td,tu,wu)

      implicit none

c***********************************************************************
c     subroutine variables
c***********************************************************************

      integer nkx,ikxm
      real    cup,clow,rhoup,rholow,deltaz,freq,dkx,alp
      complex ru(*),rl(*),td(*),tu(*),wu(*)

c***********************************************************************
c     local variables
c***********************************************************************

      integer ikx
      real eps1,eps2,kx,pi
      complex w,kup,klow,kzup,kzupp,kzlow,kzloww,j

c***********************************************************************
c     define value of pi
c     define value of eps1 : stabilizes calculation of reflection coef.
c     define value of eps2 : sets upper limit for evanescent wavenumbers
c     define value of j
c***********************************************************************

      pi=3.141592654
      eps1=1.e-3*freq*(rholow/cup+rhoup/clow)
	eps2=1.e-6
      j=cmplx(0.,1.)

c***********************************************************************
c     w     - complex angular frequency (laplace domain)
c     dkx   - sampling distance in kx domain (delta kx)
c     kup   - k-value upper medium
c     klow  - k-value lower medium
c***********************************************************************

      w=(2.*pi*freq)+j*alp
      kup=w/cup
      klow=w/clow

c***********************************************************************
c     calculate operators for all kx-values
c     below evanescent limit
c***********************************************************************

	ikx=0

c***********************************************************************
c        calculate value of kx
c        calculate value of kz for upper and lower layer
c***********************************************************************

100	ikx=ikx+1
      kx=float(ikx-1)*dkx
      kzupp=csqrt(kup*kup-kx*kx)
      kzup=real(kzupp)-j*abs(aimag(kzupp))
      kzloww=csqrt(klow*klow-kx*kx)
      kzlow=real(kzloww)-j*abs(aimag(kzloww))
c***********************************************************************
c        calculate reflection and transmission variables
c***********************************************************************

      ru(ikx)=(rholow*kzup-rhoup*kzlow)/(eps1+rholow*kzup+rhoup*kzlow)
      rl(ikx)=-ru(ikx)
      td(ikx)=1.0+ru(ikx)
      tu(ikx)=1.0-ru(ikx)
      wu(ikx)=cexp(-j*kzup*deltaz)
      if(cabs(wu(ikx)).gt.eps2 .and.
     &           ikx .lt. ikxm .and.
     &           ikx.lt.nkx)goto 100
	ikxm=ikx

c***********************************************************************
c     end of subroutine kxacoper
c***********************************************************************
      return
      end

Cdoc********************************************************************
C* NAME:   kxeloper
C*
C* DESCRIPTION:
C*         Calculation of elastic operators for refl/trans/propagation
C*         in kx domain for a certain frequency
C*
C* USAGE:  call kxeloper(cpup,cplow,csup,cslow,rhoup,rholow,deltaz,
C*        &                     freq,dkx,nkx,alp,ikxm,ru,rl,td,tu,wu)
C*
C* INPUT:  variable type description
C*         -------- ---- -----------
C*         cpup      R   P-wave velocity above reflector
C*         cplow     R   P-wave velocity below reflector
C*         csup      R   S-wave velocity above reflector
C*         cslow     R   S-wave velocity below reflector
C*         rhoup     R   density above reflector
C*         rholow    R   density below reflector
C*         deltaz    R   thickness of layer above reflector
C*         freq      R   frequency of operators to be calculated
C*         dkx       R   kx-sampling of operators
C*         nkx       I   number of kx values
C*         alp       R   laplace parameter for frequency
C*         ikxm      I   max. wavenumber for operator evaluation
C*                       (based on specified min. velocity)
C*
C* OUTPUT: variable type description
C*         -------- ---- -----------
C*         ikxm      I   maximum wavenumber of operator evaluation
C*                       (based on evanescence or min. velocity criteria)
C*         ru        R   reflectivity matrix in kx domain from above
C*         rl        R   reflectivity matrix in kx domain from below
C*         td        R   downward transmission matrix in kx domain
C*         tu        R   upward transmission matrix in kx domain
C*         wu        R   phase-shift matrix in kx domain upper layer
C*
C* NOTES:
C*         the output matrices ru, rl, td, tu and wu contain 
C*         4 operators of nkx points concatinated
C*         the matrix wu contains 2 zero operators (no. 2 and 3).
C*
C* AUTHOR: Eric Verschuur (eric@delphi.tn.tudelft.nl)
C*
C*         eric@delphi.tn.tudelft.nl
C*         Delft University of Technology
C*         Laboratory of Seismics and Acoustics
C*         P.O. Box 5046
C*         2600 GA Delft
C*         the Netherlands
C*
C* REVISION HISTORY:
C*         Version Date     Author   Comment
C*         1.0     94/07/22 Eric V.  Initial version
C*                 98/05    Nekut                    
C*
Cdoc********************************************************************

      subroutine kxeloper(cpup,cplow,csup,cslow,rhoup,rholow,deltaz,
     &                       freq,dkx,nkx,alp,ikxm,ru,rl,td,tu,wu)

c***********************************************************************
c     subroutine variables
c***********************************************************************

      integer nkx,ikxm
      real    cpup,cplow,csup,cslow,rhoup,rholow,deltaz,freq,dkx,alp
      complex ru(nkx,2,2),rl(nkx,2,2),td(nkx,2,2),tu(nkx,2,2)
      complex wu(nkx,2,2)

c***********************************************************************
c     local variables
c***********************************************************************

      integer ikx
      real kx,pi,kx2,eps1,eps2,m1,m2
      complex w,j
      complex kpup,kplow,kzpup,kzpupp,kzplow,kzploww,kzpup2,kzplow2
      complex ksup,kslow,kzsup,kzsupp,kzslow,kzsloww,kzsup2,kzslow2
      complex a, b, c, d, e, f, g, h, det, ksup2, kslow2

c***********************************************************************
c     define value of pi
c     define value of eps1 : stabilizes calculation of reflection coef.
c     define value of eps2 : sets upper limit for evanescent wavenumbers
c     define value of j
c***********************************************************************

      pi=3.141592654
	eps1=1.e-6*freq**2*(rholow/cpup+rhoup/cplow)*
     &(rholow/csup+rhoup/cslow)
	eps2=1.e-6
      j=cmplx(0.,1.)

c***********************************************************************
c     w     - complex angular frequency (laplace domain)
c     dkx   - sampling distance in kx domain (delta kx)
c     kpup  - k-value upper medium P-wave velocity
c     ksup  - k-value upper medium S-wave velocity
c     kplow - k-value lower medium P-wave velocity
c     kslow - k-value lower medium S-wave velocity
c***********************************************************************

      w=(2.*pi*freq)+j*alp
      kpup=w/cpup
      ksup=w/csup
      ksup2=ksup*ksup
      kplow=w/cplow
      kslow=w/cslow
      kslow2=kslow*kslow

c***********************************************************************
c     calculate operators for all kx-values
c     below evanescent limit
c***********************************************************************

	ikx=0

c***********************************************************************
c        calculate value of kx
c***********************************************************************

100	ikx=ikx+1
        kx=float(ikx-1)*dkx
        kx2=kx*kx

c***********************************************************************
c        calculate value of kz for upper and lower layer for P-velocity
c***********************************************************************

        kzpup2=kpup*kpup-kx*kx
        kzpupp=csqrt(kzpup2)
        kzpup=real(kzpupp)-j*abs(aimag(kzpupp))
        kzplow2=kplow*kplow-kx*kx
        kzploww=csqrt(kzplow2)
        kzplow=real(kzploww)-j*abs(aimag(kzploww))

c***********************************************************************
c        calculate value of kz for upper and lower layer for S-velocity
c***********************************************************************

        kzsup2=ksup*ksup-kx*kx
        kzsupp=csqrt(kzsup2)
        kzsup=real(kzsupp)-j*abs(aimag(kzsupp))
        kzslow2=kslow*kslow-kx*kx
        kzsloww=csqrt(kzslow2)
        kzslow=real(kzsloww)-j*abs(aimag(kzsloww))

c***********************************************************************
c        intermediate variables
c***********************************************************************

        a = rhoup*(1.-2.*kx2/ksup2) - rholow*(1.-2.*kx2/kslow2)
        b = rhoup*(1.-2.*kx2/ksup2) + 2.*rholow*kx2/kslow2
        c = rholow*(1.-2.*kx2/kslow2) + 2.*rhoup*kx2/ksup2
        d = 2.*(rhoup/ksup2-rholow/kslow2)
        e = kzplow*b+kzpup*c
        f = kzslow*b+kzsup*c
        g = a-d*kzplow*kzsup
        h = a-d*kzpup*kzslow

c***********************************************************************
c        define/stabilize determinant 
c***********************************************************************

        det = e*f+g*h*kx2
        det=det+eps1

c***********************************************************************
c***********************************************************************
c        calculate reflection and transmission variables
c***********************************************************************
c***********************************************************************

c***********************************************************************
c           Rpp,Rsp,Rps,Rss for reflection from upper level
c***********************************************************************

        ru(ikx,1,1) =
     &         ((-kzplow*b+kzpup*c)*f-(a+kzpup*kzslow*d)*g*kx2)/det
        ru(ikx,2,1) =
     &          2.*kzpup*kx*(a*c+b*d*kzplow*kzslow)/det
        ru(ikx,1,2) =
     &         -2.*kzsup*kx*(a*c+b*d*kzslow*kzplow)/det
        ru(ikx,2,2) =
     &         ((-kzslow*b+kzsup*c)*e-kx2*(kzsup*kzplow*d+a)*h)/det

c***********************************************************************
c           Rpp,Rsp,Rps,Rss for reflection from lower level
c***********************************************************************

        rl(ikx,1,1) =
     &         ((-kzpup*c+kzplow*b)*f-kx2*(a+kzsup*kzplow*d)*h)/det
        rl(ikx,2,1) =
     &          2.*kx*kzplow*(a*b+c*d*kzsup*kzpup)/det
        rl(ikx,1,2) =
     &         -2.*kx*kzslow*(a*b+c*d*kzsup*kzpup)/det
        rl(ikx,2,2) =
     &         ((kzslow*b-kzsup*c)*e-kx2*(a+kzslow*kzpup*d)*h)/det

c***********************************************************************
c           Tpp,Tsp,Tps,Tss for downward transmission
c***********************************************************************

        td(ikx,1,1) =  2.*rholow*kzpup*f/det
        td(ikx,2,1) =  2.*rholow*kx*kzpup*g/det
        td(ikx,1,2) = -2.*rholow*kx*kzsup*h/det
        td(ikx,2,2) =  2.*rholow*kzsup*e/det

c***********************************************************************
c           Tpp,Tsp,Tps,Tss for upward transmission
c***********************************************************************

        tu(ikx,1,1) =  2.*rhoup*kzplow*f/det
        tu(ikx,2,1) =  2.*rhoup*kx*kzplow*h/det
        tu(ikx,1,2) = -2.*rhoup*kx*kzslow*g/det
        tu(ikx,2,2) =  2.*rhoup*kzslow*e/det

c***********************************************************************
c        calculate P- and S- propagation operators
c***********************************************************************

        wu(ikx,1,1)=cexp(-j*kzpup*deltaz)
        wu(ikx,2,1)=0.
        wu(ikx,1,2)=0.
        wu(ikx,2,2)=cexp(-j*kzsup*deltaz)
        m1=cabs(wu(ikx,1,1))
        m2=cabs(wu(ikx,2,2))

        if( (m1.gt.eps2 .or. m2.gt.eps2) .and.
     &                     (ikx.lt.ikxm) .and.
     &                      (ikx.lt.nkx) )goto 100
        ikxm=ikx

c***********************************************************************
c     end of subroutine kxeloper
c***********************************************************************
      return
      end

c***********************************************************************
c***********************************************************************
c     Subroutine to do 2*2 matrix multiplications on operators
c***********************************************************************
c***********************************************************************

      subroutine cmatmul2by2(mat1,ld1,mat2,ld2,matout,ldout,n)

      implicit none

      integer ld1,ld2,ldout,n
      complex mat1(ld1,2,2),mat2(ld2,2,2),matout(ldout,2,2)

      integer i,j,k,l

      do j=1,2
         do i=1,2
            do l=1,n
               matout(l,i,j)=0.0
            enddo
            do k=1,2
               do l=1,n
                  matout(l,i,j)=matout(l,i,j)+mat1(l,i,k)*mat2(l,k,j)
               enddo
            enddo
         enddo
      enddo

c***********************************************************************
c     end of subroutine cmatmul2by2
c***********************************************************************

      return
      end

c***********************************************************************
c***********************************************************************
c     Subroutine to do 2*2 matrix additions on operators
c***********************************************************************
c***********************************************************************

      subroutine cmatadd2by2(mat1,ld1,mat2,ld2,matout,ldout,n)

      implicit none

      integer ld1,ld2,ldout,n
      complex mat1(ld1,2,2),mat2(ld2,2,2),matout(ldout,2,2)

      integer i,j,l

      do j=1,2
         do i=1,2
            do l=1,n
               matout(l,i,j)=mat1(l,i,j)+mat2(l,i,j)
            enddo
         enddo
      enddo

c***********************************************************************
c     end of subroutine cmatadd2by2
c***********************************************************************

      return
      end

c***********************************************************************
c***********************************************************************
c     Subroutine to do 2*2 matrix inversion on operators
c***********************************************************************
c***********************************************************************

      subroutine cmatinv2by2(mat,ld,n)

      implicit none

      integer ld,n
      complex mat(ld,2,2)

      integer l
      complex det,a,b,c,d

      do l=1,n
         a=mat(l,1,1)
         b=mat(l,1,2)
         c=mat(l,2,1)
         d=mat(l,2,2)
         
         det=1/(a*d-b*c)

         mat(l,1,1)=det*d
         mat(l,1,2)=-det*b
         mat(l,2,1)=-det*c
         mat(l,2,2)=det*a
         
      enddo

c***********************************************************************
c     end of subroutine cmatinv2by2
c***********************************************************************

      return
      end

c***********************************************************************
c***********************************************************************
c     Subroutine to copy 2*2 matrix array
c***********************************************************************
c***********************************************************************

      subroutine cmatcopy2by2(mat1,ld1,matout,ldout,n)

      implicit none

      integer ld1,ldout,n
      complex mat1(ld1,2,2),matout(ldout,2,2)

      integer i,j,l

      do j=1,2
         do i=1,2
            do l=1,n
               matout(l,i,j)=mat1(l,i,j)
            enddo
         enddo
      enddo

c***********************************************************************
c     end of subroutine cmatcopy2by2
c***********************************************************************

      return
      end

c***********************************************************************
c***********************************************************************
c     Subroutine to add unit matrix to (weighted) 2*2 matrix 
c***********************************************************************
c***********************************************************************

      subroutine cmataffn2by2(mat,ld,n,fac)

      implicit none

	real fac
      integer ld,n
      complex mat(ld,2,2)

      integer i,j,l

      do j=1,2
         do i=1,2
            do l=1,n
               mat(l,i,j)=fac*mat(l,i,j)
               if(i.eq.j)mat(l,i,j)=mat(l,i,j)+1.0
            enddo
         enddo
      enddo

c***********************************************************************
c     end of subroutine cmataffn2by2
c***********************************************************************

      return
      end
