C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C.....................................................................
C  Program to perform Biot-Gassmann modelling on lps files
C  using the same technique as in Geolog BIOTCALC loglan routines
c  Input format is LPS format ie 'ilis' format.
C  
C  Richard Margesson 
C  
C  INPUT
C  ----- 
C  Key logs are VP, Vs, Rhob and Lithcoa
C 
C Additional inputs are:
C
C   - Apparent fluid density as seen by logs. This will often be very
C       close to 1.0, but will vary in the presence of hydrocarbons,
C       especially gas.
C   - Brine density in gm/cc.
C   - Hydrocarbon density in gm/cc.
C   - Matrix (grain) density - eg 2.65 for many sandstones.
C   - Average water saturation for the reservoir.
C   - Bulk modulus of Hydrocarbon - obtainable from engineers,
C     FLUIDPVT, etc    
C   - depths of top and bottom of reservoir interval where the fluid
C       replacement is required
C
C  
C  OUTPUT
C  ------
C   Output logs are Vp; Vs; Vp/Vs ratio; for the hydrocarbon case;
C   Apparent bulk density (RHOB_APH), for hydrocarbon;
C   Density porosity overall (DPHI) and for sandstones only (PHI);



C INTERNAL VARIABLES
C ------------------
C  Fluid density of the hydrocarbon / brine mixture (rhofl)
C  Bulk modulus of the brine filled rock - matrix and fluid (Kbw)
C  Bulk modulus of the hydrocarbon /brine filled rock - matrix and fluid (Kbh)
C  Bulk modulus of the rock matrix (Kg)
C  Shear modulus of the rock matrix (Gw)
C  Bulk modulus of the empty frame (K*)
C  Bulk modulus of the fluid mixture hydrocarbon /brine (Kf)


C*******************************************************************************
C NOTE on units: All moduli are assumed to be in Mpsi, 
C                all velocities in ft/sec,               
C                all densities in gm/cc. 
C If other units are used, a different conversion factor,      
C  'MODCON', would be needed.
C******************************************************************************** 

C************************************************************

C THIS VERSION:
C ASSUMES THAT INPUT LOGS ARE WATER IE NOT HYDROCARBON AFFECTED.
C CALCULATES HYDROCARBON LOGS

C*************************************************************

C
C This version will work with time sampled data eg CDTLPS files which do
C not require going back to GETCOR. It does NOT read and write the Tops in the 
C Comments field


C Input is from command line in the same form as usp instructions.

C lpsbiotta -N208191g.CDTLPS01 -O208191g.testf -sdp7900 -edp8100 -rhg2.65
C -rhf1.0 -rhw1.0 -rhh0.2 -kw0.36 -khy0.015 -sw0.4

C where:

C -N    - input file name
C -O    - output file name
C -sdp  - start depth for replacement
C -edp  - end depth for replacement
C -rhg  - grain density, gm/cm3
C -rhf  - fluid density, gm/cm3
C -rhw  - brine density, gm/cm3
C -rhh  - hydrocarbon density, gm/cm3
C -kw   - brine bulk modulus, Mpsi
C -khy  - hydrocarbon bulk modulus, Mpsi
C -sw   - average water saturation, as decimal ie 0.4


C  
C 
c....  input: lps file    output : lps file
c.....  data format : 12f9.3
C     implicit real*8(b-h,o-z)

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h>
#include <f77/pid.h>

      character*40 ipfnam,opfnam,name*7
      character card1*10,card2*70,card3*40
      double precision modcon
      real depth,vp,vs1,lithcoa,kw,khy,kf,kbw,kbh,kst,kg,gr
      integer argis,iprnt,chkfle,luin,luout
      logical H
      data iprnt/0/,name/'FLUIREP'/

      H = (argis('-?').gt.0).or.(argis('-h').gt.0).or.(argis('-H').gt.0)
      if(H) then
       call help(LER)
       stop
      endif
C.....   next 6 lines are for reading in the input and output file 
C.....     input is on 1 
C.....     output is on 2 
c..... 

C     define modulus scaling factor to ensure moduli in Mpsi

      modcon = 6.895 * 1000000.0 / (0.3048 * 0.3048)
      luin = 1
      luout = 2
#include <f77/open.h>

C     Get input and output file names

      call argstr('-N',ipfnam,' ',' ')

      if(ipfnam.eq.' ') then
        write(LER,*)' No input file name supplied'
        write(LER,*)'Supply the -N parameter and try again'
        stop
      endif

      if(chkfle(ipfnam).ne.0)then
       write(LER,*)'Input file ',ipfnam,' not found.  Try again'
       stop
      else
       open(luin,file=ipfnam,status = 'old')
      endif
      call argstr('-O',opfnam,' ',' ')

      if(opfnam.eq.' ') then
       write(LER,*)' Output file name not supplied. Supply one and',
     : ' try again'
       stop
      endif

c +===================+
c | Open output files |
c +===================+
      open(2,file=opfnam)

C  .....Starting Depth for fluid replacement
      call argr4('-sdp',sdep,1.0,1.0)
C  .....Ending Depth for fluid replacement
      call argr4('-edp',edep,1000.0,1000.0)
c  .....Grain (Matrix) Density, (gm/cm3)
      call argr4('-rhg',rhog,2.65,2.65)
c  .....Apparent Fluid Density, (gm/cm3)
      call argr4('-rhf',rhofa,1.0,1.0)
c  .....Brine Fluid Density, (gm/cm3)
      call argr4('-rhw',rhow,1.0,1.0)
c  .....Hydrocarbon Fluid Density, (gm/cm3)
      call argr4('-rhh',rhohy,0.2,0.2)
c  .....Brine Bulk Modulus (Mpsi)
      call argr4('-kw',kw,0.36,0.36)
c  .....Hydrocarbon Bulk modulus (Mpsi)
      call argr4('-khy',khy,0.01,0.01)
c  .....Average Water Saturation, eg 0.4 
      call argr4('-sw',swa,0.50,0.50)
c  .....Porosity cutoff, %
      call argr4('-pc',pc,3.,3.)
      pc = pc*.01

C +==========================================================+
C | next 4 lines are used to read and write the header cards |
C +==========================================================+
      do j = 1,24
       read(luin,102) card1,card2
       write(luout,102) card1,card2
      end do     

102   format(a10,a70)
c103   format(12f9.3,a)

c +==========================+
c | the main processing loop |
c +==========================+
      i = 1
      do while (i.eq.1)
       read(luin,5011,end=5000)
     :   depth,time,rhog_log,rhob,phit,phi,vp,vs1,lithcoa,gr,
     :   vpvs,swb,card3

       if (depth.ge. sdep. and . depth.le.edep) then
c +==============================+
c |   calculate density porosity |
c +==============================+
        if(rhog.eq.0.0)then
         dphi = (rhog_log - rhob) / (rhog_log - rhofa)
        else
         dphi = (rhog - rhob) / (rhog - rhofa)
        endif

 6001  format(10f9.3)

       if (lithcoa.eq.4.or.lithcoa.eq.3) then 
        phi = dphi
        sw = swa
        if(swa.eq.0)sw = swb
        if(sw.gt.1)sw = sw*.01
c +=================================+
C | calculate apparent bulk density |
c +=================================+

c +----------------------+
C |  first fluid density |
c +----------------------+
        rhofl   = sw * rhow + (1 - sw) * rhohy
c +--------------------+
C |  then bulk density |
c +--------------------+
C      .....hydrocarbon 
        if(rhog.eq.0.0)then
         rhob_aph = (1 - dphi) * rhog_log + dphi * rhofl
        else
         rhob_aph = (1 - dphi) * rhog + dphi * rhofl
        endif
C      .....water
        if(rhog.eq.0.0)then
         rhob_apw = (1 - dphi) * rhog_log + phi * rhow
        else
         rhob_apw = (1 - dphi) * rhog + phi * rhow
        endif
c +================================================+
c |  calculate shear moduli and new shear velocity |
c +================================================+
        gw = rhob_apw * (vs1** 2) / modcon
        vsh = ((gw / rhob_aph) * modcon)** 0.5
c +=======================================+
c | calculate bulk modulus from vp and vs |
c +=======================================+
        kbw = (rhob_apw * ((vp** 2) - (vs1** 2)* 4/3)) / modcon
c +===============================+
C |  calculate kf from kw and khy |
c +===============================+
        kf = 1 / (sw / kw  + (1-sw) / khy )

c +=================================================================+
c |  kg                                                             |
c |  from ROCKMOD/GEM, use 2 different kg (matrix bulk modulus) for |
C |  porosity < or > 15 %                                           |
C |  kg will vary depending on your rock type.                      |
c +=================================================================+
        if (dphi.lt. 0.15)  then
         kg  =  5.85
        elseif (dphi.ge. 0.15) then
         kg  =  5.69
        endif
c +========================================================+
c |    kstar = k*                                          |
c |    calculate k* via the Biot Gassmann equation inverse |
c +========================================================+
        kst = (kbw * (dphi + (1 - dphi) * kw/kg) - kw) /
     :   ((kbw/kg - dphi - 1) * kw/kg + dphi)

c +===================================================+
C | calculate bulk moduli. check on factors and units |
c +===================================================+
c +------------------------------------------------+
C |    calculate kb via the Biot Gassmann equation |
c +------------------------------------------------+
        kbh = kst + (kf * (1 - kst / kg)**2) /
     :   (dphi +  ((1 - kst / kg) - dphi)* kf /kg)
c +----------------------------------------------+
c |    calculate new Vp for 2nd hydrocarbon case |
c +----------------------------------------------+
        vphx = ((kbh + gw * 4/3) / rhob_aph) * modcon
        if(vphx.gt.0)then
         vph = sqrt(vphx)
        else
         vph = 0.
        endif
       else
        sw = 1.0
       endif
c +========================================================+
c | if k* is negative, ignore and do not replace           |
c | if phi (porosity) is < 5%, ignore and do not replace.  |
c | this could be made a variable and input parameter      |
c +========================================================+
       if (kst.le. 0. or. phi. lt. pc) then
        if(iprnt.eq.0)then
         iprnt = 1
         write(LERR,*)' K* is negative or Porosity less than cutoff at'
         write(LERR,*)' '
         write(LERR,*)'   depth      kst      kbh      kbw      gw     r
     :hob      vp       vs1       phi'
        endif
        vsh = vs1
        vph = vp
        rhob_aph = rhob
        write(LERR,6001) depth,kst,kbh,kbw,gw,rhob,vp,vs1,phi
       elseif (lithcoa.eq.4.or.lithcoa.eq.3) then
        vsh = vsh
        vph = vph
        rhob_aph = rhob_aph
        lithcoa =  -lithcoa
       else
        vsh = vs1
        vph = vp
        rhob_aph = rhob
       endif
      else
       vsh = vs1
       vph =vp
       rhob_aph = rhob
      endif
      vpvs = vph / vsh
c +=========================+
C | compute Poisson's Ratio |
c +=========================+
      pr = 0.5 * (((vpvs)**2 - 2) / ((vpvs)**2 - 1))
      if(rhog.eq.0.0)then
      write(luout,5011)depth,time,rhog_log,rhob_aph,phit,phi,vph,vsh,
     :   lithcoa,gr,vpvs,sw,card3(1:20)
      else
        write(luout,5011)depth,time,rhog,rhob_aph,phit,phi,vph,vsh,
     :   lithcoa,gr,vpvs,sw,card3(1:20)
       endif
      end do
5000  continue
5011  format(12f9.3,a20) 
      close(1)
      close(2)
      write(LERR,*) 'Program completed'
      write(LER, *)'Program completed'
      stop
      end
      subroutine help(ler)
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'fluirep performs fluid replacement modelling on well log data '
        write(LER,*)
     :'input in LPS format (must have 24 line header!!).  Output'
        write(LER,*)
     :'is LPS formatted file in which the Vp, Vs, and density values'
        write(LER,*)
     :'have been replaced by the Biot-Gassman modelled values and in'
        write(LER,*)
     :'which the 2nd aux column contains the new Vp-Vs ratio.       '
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute fluirep by typing fluirep and the program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N []    (No default!)        : input LPS file name'
        write(LER,*)
     :' -O []    (No default!)        : output LPS file name'
        write(LER,*)
     :' -sdp[]     (default = 1.0)    : start depth for modelling'
        write(LER,*)
     :' -edp[]     (default = 1000.)  : end depth for modelling'
        write(LER,*)
     :' -rhg[]     (default = 2.65)   : grain density '
        write(LER,*)
     :' -rhf[]     (default = 1.0)    : apparent fluid density'
        write(LER,*)
     :' -rhw[]     (default = 1.0)    : brind fluid density'
        write(LER,*)
     :' -rhh[]     (default = 0.2)    : hydrocarbon fluid density'
        write(LER,*)
     :' -kw[]      (default = 0.36)   : brine bulk modulus      '
        write(LER,*)
     :' -khy[]     (default = 0.01)   : hydrocarbon bulk modulus'    
        write(LER,*)
     :' -sw[]      (default = aux3)   : average Sw.  If not present,'
        write(LER,*)
     :'                                 read from file as aux3'      
        write(LER,*)
     :' -pc[]      (default = 3% )    : minimum porosity for fluid repl'
        write(LER,*) ' '
        write(LER,*)
     : ' Usage: fluirep -N[CDTLPS] -O[] -sdp[] -edp[] -rhg[] ' 
        write(LER,*)
     : '  -rhf[] -rhw[] -rhh[] -kw[] -khy[] -sw[] -pc[]'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
      integer function chkfle(name)
      character*(*) name
      logical there
      inquire(FILE=name, EXIST=there)
      if(.not.there)THEN
        chkfle=1
      else
       chkfle=0
      endif
      return
      end
