/*
 *	Copyright (c) 1990 by Columbia University.
 */

/*
   Each Net has a private Pipeslot array used for the storage
   of pipe file descriptors which are opened during the process
   of forking that Net.

   Note that if the number of free file descriptors that may
   be opened during a net_fork() is FREEFDS, the size of the
   Pipeslot array will also be FREEFDS. Since each array slot
   holds the two descriptors associated with a pipe, we cannot
   have any more than half of the descriptors in the array open
   at any one time. We may have one end of each pipe open at the
   same time, however, which is why the array is sized at
   FREEFDS rather than FREEFDS/2.
*/

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/file.h>

#include <localsys.h>
#include "ikp_defines.h"
#include "ikp_externs.h"
#include "ikp_remote.h"
#include "globalE.h"

Pipeslot *pst_allocblk(int nslots)
{
  Pipeslot *ps;
  int i;
  
  if ((ps= (Pipeslot *) calloc((unsigned) nslots, sizeof(Pipeslot))) == (Pipeslot *) NULL)
    ikp_err("Memory allocation error.", FATAL);
  for (i= 0; i < nslots; i++)
    ps[i].pst_flags= 0;
  return(ps);
}

int pst_freefds(Net *nt)
{
  int i, free_fds;
  Pipeslot *ps;
  
  if (nt == (Net *) NULL)
    return(0);
  if ((ps= nt->net_pst) == (Pipeslot *) NULL)
    return(0);
  for (i= 0, free_fds= nt->net_npipes; i < nt->net_npipes; i++, ps++) {
    if (ps->pst_flags & PST_RDACTIVE)
      free_fds--;
    if (ps->pst_flags & PST_WRACTIVE)
      free_fds--;
  }
  return(free_fds);
}

int pst_openpipe(Net *nt,Connector *ct)
{
  Pipeslot *ps;
  Process *p, *q;
  int i;
  int sd;
  
  if (nt == (Net *) NULL) {
    ikp_err("Null net.", NONFATAL);
    return(IKP_NULLNET);
  }
  if ((ps= nt->net_pst) == (Pipeslot *) NULL) {
    ikp_err("Null pipeslot array.", NONFATAL);
    return(IKP_NULLPST);
  }
  if (ct == (Connector *) NULL) {
    ikp_err("Null connector.", NONFATAL);
    return(IKP_NULLCNT);
  }
  if (!(ct->cnt_flags & CNT_FILE) && (ct->cnt_conn == (Connector *) NULL)) {
    ikp_err("Null connector.", NONFATAL);
    return(IKP_NULLCNT);
  }
  
  p= (Process *) ct->cnt_process;
  if (ct->cnt_flags & CNT_FILE) {
    
				/* files opened by remote processes are
				   opened by the daemon on the remote host */
    if (p->prc_flags & PRC_REMOTE)
      return(IKP_SUCCESS);
  }
  else
    q= (Process *) ct->cnt_conn->cnt_process;
  
  /* locate an unused slot */
  for (i= 0; i < nt->net_npipes; i++, ps++) {
    if (!(ps->pst_flags & PST_RDACTIVE) && !(ps->pst_flags & PST_WRACTIVE))
      break;
  }
  if (i == nt->net_npipes) {
    ikp_err("No free pipeslots.", NONFATAL);
    return(IKP_NOSLOTS);
  }
  
  
  /* if both processes are remote,
     we do not need any local descriptors */
  if ((p->prc_flags & PRC_REMOTE) && (q->prc_flags & PRC_REMOTE)) {
    if (rp_remconn(ct) != IKP_SUCCESS)
      return(IKP_FAILURE);
    ps->pst_fd[0]= ps->pst_fd[1]= NULL_FD;
    ps->pst_flags= PST_RDACTIVE | PST_WRACTIVE | PST_SOCKET;
    ct->cnt_pipe= ct->cnt_conn->cnt_pipe= ps;
    return(IKP_SUCCESS);
  }
  
  /* local file (remote file already handled above) */
  if (ct->cnt_flags & CNT_FILE) {
    if (ct->cnt_iodir == CNT_INPUT) {
#ifdef CONVEXSYSTEM
      if ((ps->pst_fd[0]= open(ct->cnt_filename, O_RDONLY | O_LARGEFILE)) == -1) {
#else
      if ((ps->pst_fd[0]= open(ct->cnt_filename, O_RDONLY)) == -1) {
#endif
	sprintf(ikp_errbuf, "Cannot open file '%s'.", ct->cnt_filename);
	ikp_err(ikp_errbuf, NONFATAL);
	return(IKP_OPEN);
      }
      ps->pst_flags= PST_RDACTIVE;
      if (verbose >= 5)
	fprintf(stderr, "pst open %d (%s)\n", ps->pst_fd[0], ct->cnt_filename);
/*
	fprintf(stderr, "pst open %d (file)\n", ps->pst_fd[0]);
*/
    }
    else {
      if (ct->cnt_flags & CNT_APPEND) {
#ifdef CONVEXSYSTEM
	if ((ps->pst_fd[1]= open(ct->cnt_filename, O_WRONLY | O_CREAT | O_LARGEFILE, 0644)) == -1) {
#else
	if ((ps->pst_fd[1]= open(ct->cnt_filename, O_WRONLY | O_CREAT, 0644)) == -1) {
#endif
	  sprintf(ikp_errbuf, "Cannot open file '%s'.", ct->cnt_filename);
	  ikp_err(ikp_errbuf, NONFATAL);
	  return(IKP_OPEN);
	}
/*
	if (lseek(ps->pst_fd[1], 0L, L_XTND) < 0) {
*/
	if (lseek(ps->pst_fd[1], 0L, SEEK_END) < 0) {
	  sprintf(ikp_errbuf, "Cannot seek in file '%s'.", ct->cnt_filename);
	  ikp_err(ikp_errbuf, NONFATAL);
	  return(IKP_SEEK);
	}
	if (verbose >= 5)
	  fprintf(stderr, "pst open %d (file)\n", ps->pst_fd[1]);
      }
      else {
#ifdef CONVEXSYSTEM
	if ((ps->pst_fd[1]= open(ct->cnt_filename,
	   O_WRONLY | O_CREAT | O_TRUNC | O_LARGEFILE, 0644)) == -1) {
#else
	if ((ps->pst_fd[1]= open(ct->cnt_filename, O_WRONLY | O_CREAT | O_TRUNC, 0644)) == -1) {
#endif
	  sprintf(ikp_errbuf, "Cannot open file '%s'.", ct->cnt_filename);
	  ikp_err(ikp_errbuf, NONFATAL);
	  return(IKP_OPEN);
	}
	if (verbose >= 5)
	  fprintf(stderr, "pst open %d (file)\n", ps->pst_fd[1]);
      }
      ps->pst_flags= PST_WRACTIVE;
    }
    ct->cnt_pipe= ps;
  }
  
  /* one local and one remote process */
  else if ((p->prc_flags & PRC_REMOTE) || (q->prc_flags & PRC_REMOTE)) {
    if (rp_localconn(ct, &sd) != IKP_SUCCESS)
      return(IKP_FAILURE);
    ps->pst_fd[0]= ps->pst_fd[1]= sd;
    ps->pst_flags= PST_RDACTIVE | PST_WRACTIVE | PST_SOCKET | PST_SOCKACTIVE;
    ct->cnt_pipe= ct->cnt_conn->cnt_pipe= ps;
  }
  
  /* two local processes (two remote processes already handled above) */
  else {
    if (pipe(ps->pst_fd) == -1) {
      ikp_err("Pipe open error.", NONFATAL);
      return(IKP_PIPE);
    }
    ps->pst_flags= PST_RDACTIVE | PST_WRACTIVE;
    ct->cnt_pipe= ct->cnt_conn->cnt_pipe= ps;
    if (verbose >= 5)
      fprintf(stderr, "pst open %d -> %d (pipe)\n", ps->pst_fd[0], ps->pst_fd[1]);
  }
  
  return(IKP_SUCCESS);
}
