C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      program KFILT
c_______________________________________________________________________
c     2d Born/Kirchhoff Migration.        
c
c     a restructured version of Sam Gray's 2.5d Kirchhoff migration
c     algorithm, SMALLMIG/SmigA as it stood on 1/15/92.
c 
c     SMALLMIG was restructured for modularity and speed with the intent of   
c     becoming the kernal of a 2d velocity estimation via constrained
c     optimization algorithm. 
c     In addition, SMALLMIG has been separated into several routines, for 
c     more efficient memory management, software maintenance, and on the
c     Sparc workstations - for code performance.
c
c     KFILT performs the 45 degree phase shift, frequency filtering,
c     time origin shift and temporal interpolation needed for Kirchhoff
c     prestack migration.
c
c     since prestack migration is almost always run iteratively (in order
c     to determine the velocity model), it is more efficient to do the
c     filtering and phase shifting in KFILT once and only once. In this	
c     way the computational portion of SAMMIG is limited to imaging only,
c     halfing the time on the sparc w/s.
c
c     the following programs are functionally the same as SMALLMIG:
c
c     prepmig.................reads and processes headers from seismic
c                             common shot input gathers.
c     kfilt...................filters and resamples seismic data for 
c                             2D prestack Kirchhoff migration.
c     vsmooth.................smooths input velocity model
c     vsamp...................resamples smoothed velocity model at a
c                             given number of points per wavelength
c     kray....................calculates Kirchhoff 2D migration ray fans.
c     sammig..................images the common shot gathers onto an
c                             irregular depth grid of common reflection
c                             point gathers composed of 1 to the number
c                             of traces binned offsets.
c     zstretch................stretches/maps the data from irregular
c                             depth samples in zeta to equal depth samples
c                             in z.
c     tstretch................stretches/maps the data from irregular 
c                             depth samples in zeta to equal time samples 
c                             in t.
c
c
c                           with apologies, where necessary,
c                             and humble admiration to Sam Gray,
c
c                                 Kurt J. Marfurt
c                                   1/15/92
C***********************************************************************
#include <f77/lhdrsz.h>
#include <f77/iounit.h>
#include <f77/pid.h>
#include <save_defs.h>
c
      parameter (maxs=0 000 002)
      parameter (maxp=1000)       
      dimension s(maxs)
      pointer   (pntrs,s)
C
      integer   hbegin
      integer   argis
C
      dimension cputim(30) , waltim(30)
C
      integer   sheader(SZLNHD)
      integer   ipwr(4)
C
      integer   stdin,stdout,stderr
C
      logical   query
      logical   verbose
C
      character*5  name
c
c     parameter (i2fact=8/szsmpd,lenhed=lntrhd/i2fact,hbegin=1-lenhed)
C
      data          stdin/5/,stdout/6/,stderr/0/
      data name     /'KFILT'/
c____________________________________________________________________
c     get online help if necessary
c____________________________________________________________________
      query = (argis('-?').gt.0 .or. argis('-h').gt.0)
      if ( query ) then
         call help()
         stop 0
      endif
      call timstr(vtot,wtot)
      call vclr(cputim,1,30)
      call vclr(waltim,1,30)
c------------------------------------
c  open printout file
c------------------------------------
#include <f77/open.h>
c_______________________________________________________________________
c     read in command line file arguments:
c_______________________________________________________________________
      call cmdlin(luin,luout,verbose,lerr) 
c_______________________________________________________________________
c     read in seismic input (time domain) shot gather tape header
c_______________________________________________________________________
      lensh=0
      call rtape(luin,sheader,lensh)
      call hlhprt(sheader,lensh,name,4,lerr)
c_______________________________________________________________________
C     pull input data parameters off the line header.
C     these will serve as default input parameters.
c_______________________________________________________________________
      call saver(sheader,'NumTrc',ntr,LINEHEADER)
      call saver(sheader,'NumSmp',nt,LINEHEADER)
      call saver(sheader,'NumRec',nshot,LINEHEADER)
      call saver(sheader,'SmpInt',nsi,LINEHEADER)
      call saver(sheader, 'UnitSc', unitsc, LINEHEADER)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(sheader, 'UnitSc', unitsc, LINEHEADER)
      endif
      lenhed = ITRWRD
      hbegin= 1-lenhed

c     dtmsec=nsi
c     dt=.001*dtmsec
      dt = float(nsi) * unitsc
      dtmsec = 1000 * dt
c________________________________________________________________________
C     override input data parameters by command line arguements.
c________________________________________________________________________
      call rdparm(nt,f1,f2,f3,f4,dtmsec,jt0,t0,tmax,
     1            mutemin,mutemax,lerr)
c________________________________________________________________________
c     nt.......input trace length in samples
c     ntnew....length of array just greater than nt that is a factor of
c              2,3 and 5. i.e. ntnew=(2**m2)*(3**m3)*(5**m5).
c     ntnew4...length of temporal arrays sampled 4 times finer than ntnew.
c     ntcol....dimension of stored temporal arrays. this value is one 
c              larger than ntnew4. by initializing travel times outside
c              the raytrace fans to be ntcol and by augmenting the input 
c              migration trace with zeroes between ntcol and 2*ntcol,
c              we are able to treat or 'address' undefined data in the 
c              same way as the input data, allowing for clean loops and 
c              vectorization. 	
c     nf.......number of frequencies.
c     dt.......temporal increment
c     df.......frequency increment
c________________________________________________________________________
      call nrfft(nt,5,ntnew,ipwr)
      write(ler,*) 'power of 2 override!'
      call nrfft(nt,2,ntnew,ipwr)
      nf=ntnew/2
      ntnew4=ntnew*4
      ntcol=ntnew4+1
      df=1./(ntnew*dt)
      nfcut=f4/df
      if(verbose) then
         write(lerr,'(a20,a20)') 'variable','value'
         write(lerr,'(a20,i20)') 'nt',nt,'ntnew',ntnew,'ntnew4',ntnew4,
     1                        'ntcol',ntcol,'nf',nf,'nfcut',nfcut,
     2                        'jt0',jt0,'ntr',ntr
        write(lerr,'(a20,f20.4)')
     1                        'f1',f1,'f2',f2,'f3',f3,'f4',f4,
     2                        't0',t0,'dt',dt,'df',df
      endif
c______________________________________________________________________
c
c     calculate buffer requirements for nested mixed radix ffts.               
c     cray scilib routine rfftmlt runs twice as fast as mathadvantage
c     routine rnfftm.
c
c     buffer   mathadvantage rnfftm        cray scilib rfftmlt
c
c     itab      nt+34                      2*nt
c     rtab      3(nt/2)+13                 2*nt
c     work      18*ntr                     4*nt*ntr
c______________________________________________________________________
      if(szsmpd .eq. 4) then 
         maxitabin=ntnew+34
         maxitabout=ntnew4+34
         maxrtabin=3*(ntnew/2)+13
         maxrtabout=3*(ntnew4/2)+13
         nwork=18*ntr
      else
         maxitabin=max(2*ntnew,ntnew+34)
         maxitabout=max(2*ntnew4,ntnew4+34)
         maxrtabin=max(2*ntnew,3*(ntnew/2)+13)
         maxrtabout=max(2*ntnew4,3*(ntnew4/2)+13)
         nwork=max(4*ntr*ntnew4,18*ntr)
      endif
c______________________________________________________________________
c     calculate memory requirements
c______________________________________________________________________
      write(lerr,'(///,80a1)') ('_',i=1,80)
      write(lerr,'(A)')'storage map of major arrays'
      write(lerr,'(80a1)') ('_',i=1,80)
c
      l_free=1
c____________________________________________________________________
      write(lerr,'(///,80a1)') ('_',i=1,80)
c____________________________________________________________________
      write(lerr,'(a20,3a10)')'variable name','begin','end','length'
      call mapmem('tracebuf',l_tracebuf,l_free,lenhed+ntnew4,lerr)
      call mapmem('trheader',l_trheader,l_free,(ntr*lenhed),lerr)
      call mapmem('fwgt',l_fwgt,l_free,nf,lerr)
      call mapmem('fwgtr',l_fwgtr,l_free,nf,lerr)
      call mapmem('fwgti',l_fwgti,l_free,nf,lerr)
      call mapmem('twgt',l_twgt,l_free,mutemax-mutemin+1,lerr)
      call mapmem('urtemp',l_urtemp,l_free,ntr,lerr)
      call mapmem('uitemp',l_uitemp,l_free,ntr,lerr)
      call mapmem('itabin',l_itabin,l_free,maxitabin,lerr)
      call mapmem('rtabin',l_rtabin,l_free,maxrtabin,lerr)
      call mapmem('itabout',l_itabout,l_free,maxitabout,lerr)
      call mapmem('rtabout',l_rtabout,l_free,maxrtabout,lerr)
      call mapmem('work',l_work,l_free,nwork,lerr)
      call mapmem('utime',l_utime,l_free,ntr*(ntcol+1),lerr)
C_______________________________________________________________________
C     allocate dynamic memory.                      
C_______________________________________________________________________
      lens=l_free-1
      write(lerr,'(a20,10x,i10)') 'total vector length',lens       
      write(lerr,'(//,a)') 'allocate dynamic memory for KFILT: '
      write(lerr,*) 1.e-6*lens,' Mwords'
      write(lerr,*) 1.e-6*lens*szsmpd,' Mbytes'
      call galloc(pntrs,lens*szsmpd,ierrcd,0)                   
      if(ierrcd .ne. 0 ) then
         write(lerr,*)'galloc memory allocation error from main'   
         write(lerr,*)'ierrcd = ',ierrcd
         write(lerr,*)
         write(lerr,*)'probable cause: too much memory requested!' 
         write(lerr,*)
         write(lerr,*)'program KFILT aborted'
         stop 101
      endif
C_______________________________________________________________________
C     write out line header for filtered and resampled input
c     shot records
C_______________________________________________________________________
      nsi=dt*1000./4
      call savew(sheader,'NumTrc',ntr,LINEHEADER)
      call savew(sheader,'NumRec',nshot,LINEHEADER)
      call savew(sheader,'SmpInt',nsi,LINEHEADER)
      call savew(sheader,'NumSmp',ntnew4,LINEHEADER)
      call savhlh(sheader,lensh,lbyout)
      call wrtape(luout,sheader,lbyout)
      nbytes_out=(lenhed+ntnew4)*szsmpd
c_______________________________________________________________________
c     calculate tapers and filter coefficients.         
c_______________________________________________________________________
      call filtinit(mutemin,mutemax,nt,dt,
     1              f1,f2,f3,f4,df,nf,lerr,verbose,
     2              s(l_twgt),s(l_fwgt),s(l_fwgtr),s(l_fwgti))
c_______________________________________________________________________
c     determine location of dead trace indicator.       
c_______________________________________________________________________
c     call savelu('StaCor',ifmt,l_StaCor,length,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
c_______________________________________________________________________
c     loop over commons shot records.
c     filter/interpolate the data.                                             
c_______________________________________________________________________
      initin=1
      initout=1
      nfcut=nint(f4/df)
      do 50000 ishot=1,nshot
      call filtsub(s(l_utime),s(l_tracebuf),s(l_tracebuf),s(l_trheader),
     1              s(l_itabin),s(l_rtabin),s(l_itabout),s(l_rtabout),
     2              initin,initout,lenhed,
     3              nfcut,nf,lerr,luin,luout,l_stacor,ifmt_StaCor,
     4              jt0,dt,ishot,nbytes_out,ln_StaCor,
     5              nt,ntnew,ntnew4,ntcol,ntr,
     6              s(l_urtemp),s(l_uitemp),s(l_fwgtr),s(l_fwgti),
     7              s(l_work),s(l_twgt),mutemin,mutemax,cputim,waltim)
50000 continue

      call lbclos(luin)
      call lbclos(luout)

      call timend(cputim(20),vtot,v2,waltim(20),wtot,w2)
      write(lerr,'(A30,2A15,/)') 'routine','cpu time','wall time'
      write(lerr,'(A30,2f15.3)')
     1         'read',cputim(1),waltim(1),
     2         'filter/interpolate',cputim(2),waltim(2),
     3         'write',cputim(3),waltim(3) 
c
      write(lerr,*)'normal completion of KFILT'
      close(lerr)
C
      stop 0
      end
      subroutine  help
#include <f77/iounit.h>
      write(ler,*)' '
      write(ler,*)'Command Line Arguments for kfilt: '
      write(ler,*)'Filter/interpolation for prestack'
     1           //' Kirchhoff depth migration'
      write(ler,*)' '
      write(ler,*)'Input....................................... (def)'
      write(ler,*)' '
      write(ler,*)'-N[file_sg] (stdin)        :'
      write(ler,*)'             Input common shot time gathers'
      write(ler,*)'-O[file_filt] (stdout)     :'
      write(ler,*)'             Output filtered/interpolated gathers'
      write(ler,*)'-t0       -- zero time of source wavelet in msec'
      write(ler,*)'-f1       -- first frequency of 4 point'//
     1                          'cosine tapered filter (3. Hz)'
      write(ler,*)'-f2       -- second frequency of 4 point'//
     1                          'cosine tapered filter (8. Hz)'
      write(ler,*)'-f3       -- third frequency of 4 point'//
     1                          'cosine tapered filter (50. Hz)'
      write(ler,*)'-f4       -- fourth frequency of 4 point'//
     1                          'cosine tapered filter (60. Hz)'
      write(ler,*)'-V        -- verbos printout'
      write(ler,*)' '
      write(ler,*)'Usage:'
      write(ler,*)'   kfilt  -N[] -O[] -t0[]'                 
      write(ler,*)'          -f1[] -f2[] -f3[] -f4[] -V '
      write(ler,*)' '

      return
      end
