C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c *********************************************************************
c
c  SUBROUTINE ATT3P3D (amn, amx, numangles, attrnum, dtsec, nsamp,
c                       rtrpt, arrayout, err)
c
c    This subroutine processes attributes for 3 parameter angst data
c
c    "amn" is the minimum angle data array as used in ANGST3D
c    "amx" is the maximum angle data array as used in ANGST3D
c    "numangles" is the number of angle buckets input from ANGST3D
c    "attrnum" is the attribute (by number) to be calculated
c    "dtsec" is the sample interval in seconds
c    "nsamp" is the number of samples in each angle stack
c    "rtrpt" is the array of data traces input from ANGST3D
c    "arrayout" is the attribute data calculated from the three
c               input angle data stack traces 
c    "err" is an error flag
c
c**********************************************************************
 
      subroutine att3p3d (amn, amx, numangles, attrnum, dtsec,
     1     nsamp, rtrpt, arrayout, err)
#include <f77/lhdrsz.h>
  
      integer   numangles
      integer   attrnum, nsamp, err
      real      amn(*), amx(*)
      real      rtod, dtor, small, dtsec
      real      quadp, quadm, quada, quadb
      real      min1, max1, min2, max2, min3, max3
      real      c11,c12,c13,c21,c22,c23,c31,c32,c33
      real      ci11,ci12,ci13,ci21,ci22,ci23,ci31,ci32,ci33
      real      b0 (SZLNHD)
      real      b1 (SZLNHD)
      real      b2 (SZLNHD)
      real      Q (SZLNHD)
      real      wkarray1 (SZLNHD)
      real      wkarray2 (SZLNHD)
      real      rtrpt (SZLNHD,63)
      real      arrayout (nsamp)
 
c load "small" value
c
      small = 1.e-15
c
c load degree/radian conversion factors
c
      dtor = .0174533
      rtod = 57.29578

c
c    Clear output array to zeros
c
      call vclr (arrayout, 1, nsamp)
      err = 0

c-----
c
c    Convert min1, max1, min2, max2, min3, max3, to radians
c
        min1 = dtor * amn(1)
        max1 = dtor * amx(1)
	min2 = dtor * amn(2)
	max2 = dtor * amx(2)
	min3 = dtor * amn(3)
	max3 = dtor * amx(3)

c-----
c
c    Calculate b0, b1, b2
c
 
         c11=1.0
         c12=(tan(max1)-tan(min1)+(min1-max1))/(max1-min1)
         c13=(sin(max1)*cos(max1)**2.*cos(min1)-cos(max1)*(sin(min1)*
     :    cos(min1)**2.+3.*(max1-min1)*cos(min1)+2.*sin(min1))+2.
     :    *sin(max1)*cos(min1))/(2.*(max1-min1)*cos(max1)*cos(min1))
 
         c21=1.0
         c22=(tan(max2)-tan(min2)+(min2-max2))/(max2-min2)
         c23=(sin(max2)*cos(max2)**2.*cos(min2)-cos(max2)*(sin(min2)*
     :    cos(min2)**2.+3.*(max2-min2)*cos(min2)+2.*sin(min2))+2.
     :    *sin(max2)*cos(min2))/(2.*(max2-min2)*cos(max2)*cos(min2))
 
         c31=1.0
         c32=(tan(max3)-tan(min3)+(min3-max3))/(max3-min3)
         c33=(sin(max3)*cos(max3)**2.*cos(min3)-cos(max3)*(sin(min3)*
     :    cos(min3)**2.+3.*(max3-min3)*cos(min3)+2.*sin(min3))+2.
     :    *sin(max3)*cos(min3))/(2.*(max3-min3)*cos(max3)*cos(min3))
 

         ci11 = ((c22*c33-c23*c32)/(c11*(c22*c33-c23*c32)+c12
     :       *(c23*c31-c21*c33)+c13*(c21*c32-c22*c31)))
         ci12 = (c13*c32-c12*c33)/(c11*(c22*c33-c23*c32)+c12
     :       *(c23*c31-c21*c33)+c13*(c21*c32-c22*c31))
         ci13 = (c12*c23-c13*c22)/(c11*(c22*c33-c23*c32)+c12
     :       *(c23*c31-c21*c33)+c13*(c21*c32-c22*c31))

         ci21 = ((c23*c31-c21*c33)/(c11*(c22*c33-c23*c32)+c12
     :       *(c23*c31-c21*c33)+c13*(c21*c32-c22*c31)))
         ci22 = (c11*c33-c13*c31)/(c11*(c22*c33-c23*c32)+c12
     :       *(c23*c31-c21*c33)+c13*(c21*c32-c22*c31))
         ci23 = (c13*c21-c11*c23)/(c11*(c22*c33-c23*c32)+c12
     :       *(c23*c31-c21*c33)+c13*(c21*c32-c22*c31))

         ci31 = ((c21*c32-c22*c31)/(c11*(c22*c33-c23*c32)+c12
     :       *(c23*c31-c21*c33)+c13*(c21*c32-c22*c31)))
         ci32 = (c12*c31-c11*c32)/(c11*(c22*c33-c23*c32)+c12
     :       *(c23*c31-c21*c33)+c13*(c21*c32-c22*c31))
         ci33 = (c11*c22-c12*c21)/(c11*(c22*c33-c23*c32)+c12
     :       *(c23*c31-c21*c33)+c13*(c21*c32-c22*c31))
 
c
c   Calculate b0s, b1s, and b2s
c
c     b0 = ci11*st1 + ci12*st2 + ci13*st3
c     b1 = ci21*st1 + ci22*st2 + ci23*st3
c     b2 = ci31*st1 + ci32*st2 + ci33*st3
c

	  do 100 j = 1, nsamp 
         	b0(j) = (ci11*rtrpt(j,1))+(ci12*rtrpt(j,2))
     :               +(ci13*rtrpt(j,3))
         	b1(j) = (ci21*rtrpt(j,1))+(ci22*rtrpt(j,2))
     :               +(ci23*rtrpt(j,3))
         	b2(j) = (ci31*rtrpt(j,1))+(ci32*rtrpt(j,2))
     :               +(ci33*rtrpt(j,3))
 100 	  continue
 
c
c   If attrnum = 1, output b0s only
c
	if(attrnum .eq. 1) then
      		call vmov (b0, 1, arrayout, 1, nsamp)
		return
	endif

c
c   If attrnum = 2, output b1s only
c
	if(attrnum .eq. 2) then
      		call vmov (b1, 1, arrayout, 1, nsamp)
		return
	endif

c
c   If attrnum = 3, output b2s only
c
	if(attrnum .eq. 3) then
      		call vmov (b2, 1, arrayout, 1, nsamp)
                return
        endif

c
c   If attrnum = 4, output Bz=SIGN(B0)*B1
c
	if(attrnum .eq. 4) then
	  do 400 j = 1, nsamp 
		if(b0(j).lt.0.0) then
			arrayout(j) = -1.0 * b1(j)
		else
			arrayout(j) = b1(j)
		endif
 400 	  continue
	  return
	endif

c
c   If attrnum = 5, output Bp=B0*B1
c
        if(attrnum .eq. 5) then
          do 500 j = 1, nsamp
		if (b0(j) .le. small .or. b1(j) .le. small) then
                        arrayout(j) = 0
                else
                        arrayout(j) = b0(j) * b1(j)
                endif
 500      continue
          return
        endif

c
c   If attrnum = 6, output Br=B1/B0
c
        if(attrnum .eq. 6) then
          do 600 j = 1, nsamp
                if (b0(j) .eq. 0 .or. b1(j) .eq. 0) then
                        arrayout(j) = 0
                else
                        arrayout(j) = b1(j) / b0(j)
                endif
 600      continue
          return
        endif

c
c   If attrnum = 7, output st2 - st1
c
        if(attrnum .eq. 7) then
          do 700 j = 1, nsamp
                arrayout(j) = rtrpt(j,2) - rtrpt(j,1)
 700      continue
          return
        endif
 
c
c   If attrnum = 8, output st3 - st1
c
        if(attrnum .eq. 8) then
          do 800 j = 1, nsamp
                arrayout(j) = rtrpt(j,3) - rtrpt(j,1)
 800      continue
          return
        endif
 
c
c   If attrnum = 9, output st3 - st2
c
        if(attrnum .eq. 9) then
          do 900 j = 1, nsamp
                arrayout(j) = rtrpt(j,3) - rtrpt(j,2)
 900      continue
          return
        endif
 
c
c   If attrnum = 10, output in degrees the smaller of 
c		(-B1+sqrt((B1*B1)-4*(B2+2*B1/3)*B0))/(2*(B2+2*B1/3))
c	    or  (-B1-sqrt((B1*B1)-4*(B2+2*B1/3)*B0))/(2*(B2+2*B1/3))
c
        if(attrnum .eq. 10) then
          do 1000 j = 1, nsamp
		if(b1(j).le.small .or. b2(j).le.small) then
		  arrayout(j) = 0.0
		else
		  quada = b2(j) + ((2 * b1(j))/3)
		  quadb = (b1(j) * b1(j)) - (4 * quada * b0(j))
		  if(quadb .lt. 0.0) then
			arrayout(j) = 0.0
		  else
		       quadp = (-b1(j)+sqrt(quadb))/(2*quada)
		       quadm = (-b1(j)-sqrt(quadb))/(2*quada)
		       arrayout(j) = 0.0
                       if (quadp .le. quadm) then
                          if (quadp .ge. 0) then
				arrayout(j) = rtod * quadp
                          elseif (quadm .ge. 0) then
				arrayout(j) = rtod * quadm
                          endif
                       elseif (quadm .lt. quadp) then
                          if (quadm .ge. 0) then
				arrayout(j) = rtod * quadm
                          elseif (quadp .ge. 0) then
				arrayout(j) = rtod * quadp
                          endif
                       endif
		  endif
		endif
 1000     continue
          return
        endif
 
c
c   If attrnum = 11, output Restricted Gradient
c                       EE(large)-EE(small)
c
        if(attrnum .eq. 11) then
          call asig2(rtrpt(1,3),Q,dtsec,nsamp,3,wkarray1)
          call asig2(rtrpt(1,1),Q,dtsec,nsamp,3,wkarray2)
          do 1100 j = 1, nsamp
                arrayout(j) = wkarray1(j) - wkarray2(j)
 1100     continue
	  return
        endif
 
c
c   If attrnum = 12, output dvp/vp = 2B1+2B2
c
        if(attrnum .eq. 12) then
          do 1200 j = 1, nsamp
                arrayout(j) = (2 * b1(j)) + (2 * b2(j))
 1200     continue
          return
        endif
 
c
c   If attrnum = 13, output dvs/vs = B1-B0+2B2
c
        if(attrnum .eq. 13) then
          do 1300 j = 1, nsamp
                arrayout(j) = b1(j) - b0(j) + (2 * b2(j))
 1300     continue
          return
        endif
 
c
c   If attrnum = 14, output drho/rho = 2[B0-B1-B2]
c
        if(attrnum .eq. 14) then
          do 1400 j = 1, nsamp
                arrayout(j) = 2 * (b0(j) - b1(j) - b2(j))
 1400     continue
          return
        endif
 
c
c   If attrnum = 15, output 2*B0
c
        if(attrnum .eq. 15) then
          do 1500 j = 1, nsamp
                arrayout(j) = 2*b0(j)
 1500     continue
          return
        endif
 
c
c   If attrnum = 16, output B0-B1
c
        if(attrnum .eq. 16) then
          do 1600 j = 1, nsamp
                arrayout(j) = b0(j) - b1(j)
 1600     continue
          return
        endif
 
c
c   If attrnum = 17, output B0+B1
c
        if(attrnum .eq. 17) then
          do 1700 j = 1, nsamp
                arrayout(j) = b0(j) + b1(j)
 1700     continue
          return
        endif
 
c
c   If attrnum = 18, output 0.58*B0(j)+1.42*B1(j)+1.84*B2
c
        if(attrnum .eq. 18) then
          do 1800 j = 1, nsamp
                arrayout(j) = 0.58*b0(j)+1.42*b1(j)+1.84*b2(j)
 1800     continue
          return
        endif
 
c
c   If attrnum = 19, output 100000*(1/B2)
c
        if(attrnum .eq. 19) then
          do 1900 j = 1, nsamp
                if (b2(j) .eq. 0) then
                        arrayout(j) = 0
                else
                	arrayout(j) = 100000*(1.0/b2(j))
		endif
 1900     continue
          return
        endif
 
c
c   If attrnum = 20, output EE(small angle)
c
        if(attrnum .eq. 20) then
          call asig2(rtrpt(1,1),Q,dtsec,nsamp,3,arrayout)
          return
        endif
 
c
c   If attrnum = 21, output EE(mid angle)
c
        if(attrnum .eq. 21) then
          call asig2(rtrpt(1,2),Q,dtsec,nsamp,3,arrayout)
          return
        endif
 
c
c   If attrnum = 22, output EE(large angle)
c
        if(attrnum .eq. 22) then
          call asig2(rtrpt(1,3),Q,dtsec,nsamp,3,arrayout)
          return
        endif
 
c
c   If attrnum = 23, output EE(mid angle)-EE(small angle)
c
        if(attrnum .eq. 23) then
          call asig2(rtrpt(1,2),Q,dtsec,nsamp,3,wkarray1)
          call asig2(rtrpt(1,1),Q,dtsec,nsamp,3,wkarray2)
          do 2300 j = 1, nsamp
                arrayout(j) = wkarray1(j) - wkarray2(j)
 2300     continue
          return
        endif
 
c
c   If attrnum = 24, output EE(large angle)-EE(small angle)
c
        if(attrnum .eq. 24) then
          call asig2(rtrpt(1,3),Q,dtsec,nsamp,3,wkarray1)
          call asig2(rtrpt(1,1),Q,dtsec,nsamp,3,wkarray2)
          do 2400 j = 1, nsamp
                arrayout(j) = wkarray1(j) - wkarray2(j)
 2400     continue
          return
        endif

c
c   If attrnum = 25, output Enhanced Restricted Gradient
c                       (EE(large)-EE(small))*EE(large)
c
        if(attrnum .eq. 25) then
          call asig2(rtrpt(1,3),Q,dtsec,nsamp,3,wkarray1)
          call asig2(rtrpt(1,1),Q,dtsec,nsamp,3,wkarray2)
          do 2500 j = 1, nsamp
                arrayout(j)=(wkarray1(j)-wkarray2(j))*wkarray1(j)
 2500     continue
          return
        endif
 
c
c   If attrnum < 1 or attrnum > 25, return error code = 1
c
        if(attrnum .lt. 1 .or. attrnum .gt. 25) then
                write(LERR,*)' '
                write(LERR,*) ' Requested attribute not available. '
                write(LERR,*)' '
		err = 1
		return
	endif
      write(LERR,*)' '
      write(LERR,*) ' Invalid attribute condition.'
      write(LERR,*)' '
      err = 1
      return
      end
